
function xyw=polygauss_2013(N,vertices,rotation,P,Q)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Cubature rule with ade equal to "N" over a polygon with vertices 
% "vertices".
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% n     : degree of the 1 dimensional Gauss-Legendre rule.
%
% vertices: if the polygon has "L" sides, "vertices" is a
%         variable containing its vertices, ordered counterclockwise.
%         The last row must have the components of the first vertex, that
%         is the first row and last row are equal.
%         "vertices" is a "(L+1) x 2" matrix.
%
%
% rotation*: 0: no rotation.
%           1: automatic.
%           2: preferred direction rotation, defined by points P, Q.
%
% P, Q*: direction that fixes the rotation.
%
% Note: the variables with "*" are not mandatory.
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% xyw: cubature rule where "nodes=xyw(:,1:2)" and weights "w=xyw(:,3)".
%--------------------------------------------------------------------------
% EXAMPLE 1 (NO ROTATION.)
%---------------------------
%
% >> xyw=polygauss_2013(2,[0 0; 1 0; 1 1; 0 1; 0 0],0)
%
% xyw =
%
%     0.2113    0.2113    0.2500
%     0.2113    0.7887    0.2500
%     0.7887    0.2113    0.2500
%     0.7887    0.7887    0.2500
%
% >>
%
%--------------------------------------------------------------------------
% EXAMPLE 2 (AUTO ROTATION.)
%-----------------------------
%
% >> xyw=polygauss_2013(2,[0 0; 1 0; 1 1; 0 1; 0 0])
%
% xyw =
%
%     0.0683    0.0444    0.0078
%     0.3028    0.1972    0.0556
%     0.5374    0.3499    0.0616
%     0.6501    0.4626    0.0616
%     0.8028    0.6972    0.0556
%     0.9556    0.9317    0.0078
%     0.9317    0.9556    0.0078
%     0.6972    0.8028    0.0556
%     0.4626    0.6501    0.0616
%     0.3499    0.5374    0.0616
%     0.1972    0.3028    0.0556
%     0.0444    0.0683    0.0078
%     0.1008    0.0119    0.0078
%     0.4472    0.0528    0.0556
%     0.7935    0.0938    0.0616
%     0.9062    0.2065    0.0616
%     0.9472    0.5528    0.0556
%     0.9881    0.8992    0.0078
%     0.8992    0.9881    0.0078
%     0.5528    0.9472    0.0556
%     0.2065    0.9062    0.0616
%     0.0938    0.7935    0.0616
%     0.0528    0.4472    0.0556
%     0.0119    0.1008    0.0078
%
% >>

%--------------------------------------------------------------------------
%% Copyright (C) 2007- Marco Vianello and Alvise Sommariva
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

%% Authors:
%% Marco Vianello    <marcov@euler.math.unipd.it>
%% Alvise Sommariva  <alvise@euler.math.unipd.it>
%% Date: 2007
%% First version: April 30, 2013.
%% Modified on: January 1, 2026.
%--------------------------------------------------------------------------

if norm(vertices(:,1)-vertices(:,end)) > 0
    vertices(end+1,:)=vertices(1,:);
end

if nargin < 3
    rotation=1;
end

cubature_type=4; % Gauss-Legendre rule as default

%----------------------------------------------------------------------
% BOUNDARY PTS.
%----------------------------------------------------------------------
x_bd=vertices(:,1);
y_bd=vertices(:,2);

%----------------------------------------------------------------------
% "MINIMUM" RECTANGLE CONTAINING POLYGON.
%----------------------------------------------------------------------
x_min=min(x_bd);
y_min=min(y_bd); y_max=max(y_bd);

%--------------------------------------------------------------------------
% POLYGON ROTATION (IF NECESSARY).
%--------------------------------------------------------------------------
switch rotation
    case 0 % no rotation
        rot_matrix=eye(2);
        axis_abscissa=[x_min y_max]-[x_min y_min];
    case 1 % automatic rotation
        [vertices,rot_matrix,~,axis_abscissa]=...
            auto_rotation(vertices,[],[]);
    case 2 % fixed rotation: PQ direction
        [vertices,rot_matrix,~,axis_abscissa]=...
                auto_rotation(vertices,P,Q);
end

% One dimensional cubature nodes.

% ORDER GAUSS PRIMITIVE: DEGREE "N".
[s_N,w_N]=cubature_rules_1D((N-1),cubature_type);
N_length=length(s_N);

% ORDER GAUSS INTEGRATION: DEGREE "M".
M=N+1;
[s_M,w_M]=cubature_rules_1D((M-1),cubature_type);

% L: NUMBER OF SIDES OF THE POLYGON.
L=length(vertices(:,1))-1;

a=axis_abscissa(1);

%----------------------------------------------------------------------
% COMPUTE 2D NODES (nodes_x,nodes_y) AND WEIGHTS "weights".
%----------------------------------------------------------------------

nodes_x=[];
nodes_y=[];
weights=[];

for index_side=1:L
    x1=vertices(index_side,1); x2=vertices(index_side+1,1);
    y1=vertices(index_side,2); y2=vertices(index_side+1,2);
    if ~(x1 == a && x2 == a)
        if (y2-y1) ~=0
            
            if (x2-x1) ~=0
                s_M_loc=s_M;
                w_M_loc=w_M;
            else
                s_M_loc=s_N;
                w_M_loc=w_N;
            end
            
            M_length=length(s_M_loc);
            
            half_pt_x=(x1+x2)/2; half_pt_y=(y1+y2)/2;
            half_length_x=(x2-x1)/2; half_length_y=(y2-y1)/2;
            
            
            % GAUSSIAN POINTS ON THE SIDE.
            x_gauss_side=half_pt_x+half_length_x*s_M_loc; %SIZE: (M_loc,1)
            y_gauss_side=half_pt_y+half_length_y*s_M_loc; %SIZE: (M_loc,1)
            
            scaling_fact_plus=(x_gauss_side+a)/2; %SIZE: (M_loc,1)
            scaling_fact_minus=(x_gauss_side-a)/2;%SIZE: (M_loc,1)
            
            local_weights=...
                (half_length_y*scaling_fact_minus).*w_M_loc;%SIZE:(M_loc,1)
            
            term_1=repmat(scaling_fact_plus,1,N_length); % SIZE: (M_loc,N)
            
            term_2=repmat(scaling_fact_minus,1,N_length); % SIZE: (M_loc,N)
            
            rep_s_N=repmat(s_N',M_length,1);
            
            % x, y ARE STORED IN MATRICES. A COUPLE WITH THE SAME INDEX
            % IS A POINT, i.e. "P_i=(x(k),y(k))" FOR SOME "k".
            x=term_1+term_2.*rep_s_N;
            y=repmat(y_gauss_side,1,N_length);
            
            number_rows=size(x,1);
            number_cols=size(x,2);
            
            x=x(:); x=x';
            y=y(:); y=y';
            
            rot_gauss_pts=rot_matrix'*[x;y]; % THE INVERSE OF A ROTATION
            % MATRIX IS ITS TRANSPOSE.
            
            x_rot=rot_gauss_pts(1,:); % GAUSS POINTS IN THE ORIGINAL SYSTEM.
            y_rot=rot_gauss_pts(2,:);
            
            x_rot=reshape(x_rot',number_rows,number_cols);
            y_rot=reshape(y_rot',number_rows,number_cols);
            
            nodes_x=[nodes_x; x_rot];
            nodes_y=[nodes_y; y_rot];
            weights=[weights; local_weights];
            
            
        end
    end
end

weights=weights*w_N';
weights=weights(:);

nodes_x=nodes_x(:);
nodes_y=nodes_y(:);
xyw=[nodes_x nodes_y weights];









%----------------------------------------------------------------------
% FUNCTIONS USED IN THE ALGORITHM.
%----------------------------------------------------------------------


%--------------------------------------------------------------------------
% auto_rotation.
%--------------------------------------------------------------------------

function [control_pts_rot,rot_matrix,rot_angle,axis_abscissa,...
    vertex_1,vertex_2]=auto_rotation(control_pts,vertex_1,vertex_2)

%--------------------------------------------------------------------------
% OBJECT:
%-----------
%
% AUTOMATIC ROTATION OF A CONVEX POLYGON SO THAT "GAUSSIAN POINTS" 
% ARE ALL (OR "MOST OF THEM" IF THE DEGREE OF THE BORDER IS BIGGER THAN 2) 
% CONTAINED IN THE CONVEX POLYGON. SEE THE PAPER FOR DETAILS.
%
%--------------------------------------------------------------------------
% INPUTS:
%----------
%
% control_pts : CONTROL POINTS OF THE CURVILEAR POLYGON.
%
% vertex_1, vertex_2: ROTATION VERTICES (IF ANY ARE CHOOSEN).
%
%----------
% OUTPUTS:
%----------
%
% control_pts_rot: CONTROL POINTS OF THE CURVILEAR POLYGON, 
%       AFTER THE ROTATION.
%
% rot_matrix: ROTATION MATRIX.
%
% rot_angle: ROTATION ANGLE.
%
% axis_abscissa: ABSCISSA OF THE REFERENCE AXIS.
%
% vertex_1, vertex_2: ROTATION VERTICES (IF ANY ARE CHOOSEN OR COMPUTED BY 
%       THE ROUTINE).
%
%--------------------------------------------------------------------------
% ADDITIONAL ROUTINES :
%----------------------
%
% 1. points2distances
%
%--------------------------------------------------------------------------

% FIND DIRECTION AND ROTATION ANGLE.
if isempty(vertex_1)
    % COMPUTING ALL THE DISTANCES BETWEEN POINTS.A LITTLE TIME CONSUMING AS
    % PROCEDURE.
    distances = points2distances(control_pts);  
    [max_distances,max_col_comp]=max(distances,[],2);
    [max_distance,max_row_comp]=max(max_distances,[],1);
    vertex_1=control_pts(max_col_comp(max_row_comp),:);
    vertex_2=control_pts(max_row_comp,:);
    direction_axis=(vertex_2-vertex_1)/max_distance;
else
    direction_axis=(vertex_2-vertex_1)/norm(vertex_2-vertex_1);
end

rot_angle_x=acos(direction_axis(1));
rot_angle_y=acos(direction_axis(2));

if rot_angle_y <= pi/2
    if rot_angle_x <= pi/2
        rot_angle=-rot_angle_y;
    else
        rot_angle=rot_angle_y;
    end
else
    if rot_angle_x <= pi/2
        rot_angle=pi-rot_angle_y;
    else
        rot_angle=rot_angle_y;
    end
end


% CLOCKWISE ROTATION.
rot_matrix=[cos(rot_angle) sin(rot_angle); 
    -sin(rot_angle) cos(rot_angle)];

% number_sides=size(control_pts,1)-1;

control_pts_rot=(rot_matrix*control_pts')';

axis_abscissa=rot_matrix*vertex_1';





%--------------------------------------------------------------------------
% points2distances.
%--------------------------------------------------------------------------

function distances = points2distances(points)  

% Create euclidean distance matrix from point matrix.

% Get dimensions.
[numpoints,~]=size(points);                    

% All inner products between points.
distances=points*points';                     

% Vector of squares of norms of points.
lsq=diag(distances);                            

% Distance matrix.
distances=sqrt(repmat(lsq,1,numpoints)+repmat(lsq,1,numpoints)'-2*distances);












%----------------------------------------------------------------------
% 2. "cubature_rules_1D"
%----------------------------------------------------------------------

function [nodes,weights]=cubature_rules_1D(n,cubature_type)

% SEE WALDVOGEL PAPER. ADDED NODES

% Weights of the Fejer2, Clenshaw-Curtis and Fejer1 quadrature by DFTs
% n>1. Nodes: x_k = cos(k*pi/n)

N=(1:2:n-1)'; l=length(N); m=n-l; K=(0:m-1)';

switch cubature_type
    
    case 1 % FEJER 1.
        v0=[2*exp(1i*pi*K/n)./(1-4*K.^2); zeros(l+1,1)];
        v1=v0(1:end-1)+conj(v0(end:-1:2));
        weights=ifft(v1);
        k=(1/2):(n-(1/2)); nodes=(cos(k*pi/n))';
        
    case 2 % FEJER 2.
        v0=[2./N./(N-2); 1/N(end); zeros(m,1)];
        v2=-v0(1:end-1)-v0(end:-1:2);
        wf2=ifft(v2); weights=[wf2;0];
        k=0:n; nodes=(cos(k*pi/n))';
        
    case 3 % CLENSHAW CURTIS.
        g0=-ones(n,1); g0(1+l)=g0(1+l)+n; g0(1+m)=g0(1+m)+n;
        g=g0/(n^2-1+mod(n,2));
        v0=[2./N./(N-2); 1/N(end); zeros(m,1)];
        v2=-v0(1:end-1)-v0(end:-1:2);
        wcc=ifft(v2+g); weights=[wcc;wcc(1,1)];
        k=0:n; nodes=(cos(k*pi/n))';
        
    case 4 % GAUSS LEGENDRE
        beta=0.5./sqrt(1-(2*(1:n)).^(-2));
        T=diag(beta,1)+diag(beta,-1);
        [V,D]=eig(T);
        x=diag(D); [x,index]=sort(x); x=x';
        w=2*V(1,index).^2;
        nodes=x';
        weights=w';
        
end














