
function demo_wam_polygon(example)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% In this demo we show how to determine WAM over polygonal regions, even 
% not connected.
%
% Some plots of pointsets, domain, triangulations, are given.
%--------------------------------------------------------------------------
% Important:
%--------------------------------------------------------------------------
% * the code requires the Matlab toolbox "polyshape"
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% example: 1: not simply connected and not connected domain. 
%          2: not simply connected domain.
%--------------------------------------------------------------------------
% Subroutines:
%--------------------------------------------------------------------------
% 1.   (external)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >> demo_wam_polygon(1)
% 
%  ..................... Summary .............................
%   Example    : 1 
%   Vertices   : 299    
%   Triangles  : 295    
% 
%   Degree WAM : 10 
%   Card. WAM  : 32745  
% 
% >> demo_wam_polygon(2)
% 
%  ..................... Summary .............................
%   Example    : 2 
%   Vertices   : 203    
%   Triangles  : 202    
% 
%   Degree WAM : 10 
%   Card. WAM  : 22422  
% 
% >>
%--------------------------------------------------------------------------
% Copyright
%--------------------------------------------------------------------------
%% Copyright (C) 2018- Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Date: December 07, 2018
%% Modified on: January 2, 2026
%--------------------------------------------------------------------------

if nargin < 1, example=1; end

deg=10; % WAM degree

%--------------------------
% defining domain
%--------------------------
switch example
    case 1 % domain not connected and not simply connected.
        Nsides=100;
        th=linspace(0,2*pi,Nsides); th=(th(1:end-1))';
        polygon1=[cos(th) sin(th)]; P1=polyshape(polygon1);   % 1st polygon
        polygon2=2+[cos(th) sin(th)]; P2=polyshape(polygon2); % 2nd polygon
        polygon3=0.5*[cos(th) sin(th)]; P3=polyshape(polygon3); % 3rd poly.
        
        P=subtract(P1,P3);
        pgon=union(P,P2);
        
    case 2 % domain not simply connected (optics)
        Nsides=100;
        y=[0         0   -0.1184   -0.1184   -0.3761];
        r=[1.0000    0.6120    0.5663    1.0761    1.2810];
        th=linspace(0,2*pi,Nsides); th=(th(1:end-1))';
        C1=[0 y(1)]; P1v=C1+r(1)*[cos(th) sin(th)]; P1=polyshape(P1v);
        C2=[0 y(2)]; P2v=C2+r(2)*[cos(th) sin(th)]; P2=polyshape(P2v);
        C3=[0 y(3)]; P3v=C3+r(3)*[cos(th) sin(th)]; P3=polyshape(P3v);
        C4=[0 y(4)]; P4v=C4+r(4)*[cos(th) sin(th)]; P4=polyshape(P4v);
        C5=[0 y(5)]; P5v=C5+r(5)*[cos(th) sin(th)]; P5=polyshape(P5v);
        Pout=intersect(P1,P4);
        Pout=intersect(Pout,P5);
        Pin=union(P2,P3);
        pgon=subtract(Pout,Pin);
        
end


% ............................. determining WAM ........................... 

wam_pts=wam_polygon(deg,pgon);
vertex_coords = pgon.Vertices;
tri = triangulation(pgon);

fprintf('\n ..................... Summary .............................');
fprintf('\n  Example    : %-2.0f',example);
fprintf('\n  Vertices   : %-7.0f',size(vertex_coords,1));
fprintf('\n  Triangles  : %-7.0f',size(tri,1));
fprintf('\n \n  Degree WAM : %-3.0f',deg);
fprintf('\n  Card. WAM  : %-7.0f',size(wam_pts,1));


% ............................... plots ...................................

clf;
close;

% plotting triangulation
tri = triangulation(pgon);

clf(figure(1));
figure(1)
hold on;
axis off
axis equal
titlestr1='Domain triangulation.';
titlestr2=strcat('Triangles: ',num2str(size(tri,1)),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
triplot(tri);
plot(pgon,'FaceColor',[0.5 0.5 0.5]);
hold off;




% plotting WAM

clf(figure(2));
figure(2)
hold on;
axis off;
axis equal;
titlestr1='Domain and WAM. ';
titlestr2=strcat('Card: ',num2str(size(wam_pts,1)),...
    '. deg: ',num2str(deg),'.');
title_caption=[titlestr1,titlestr2];
title(title_caption, 'FontSize', 9);
title(title_caption);
plot(pgon,'FaceColor',[0.5 0.5 0.5]);
plot(wam_pts(:,1),wam_pts(:,2),'ko','LineWidth',1,...
    'MarkerEdgeColor','k',...
    'MarkerFaceColor','g',...
    'MarkerSize',2)
hold off;

fprintf('\n \n');




