function [wam_pts,pgon]=wam_polygon(deg,vertices,iv)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% The following routine computes a WAM over a polygon, of degree "deg".
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% deg: WAM degree.
% vertices : abscissae of the vertices of the polygonal region; in case t
%      a) the polygonal region is described by a polyshape object "P" than 
%         set "vertices=P", and do not declare "iv";
%      b) otherwise set as "vertices" as the L x 2 matrix of points 
%         describing the vertices, setting as "iv" a vector describing the
%         components (as said below); in case the region is a polygon, set
%         iv=[];
% iv: index of the polygonal components. 
%     Example: if the domain is defined by 3 not connected polygonal
%     components respectively composed by 5, 10, 12 components, then set
%     "iv=[5,10,12]".
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% wam_pts: wam points at degree "deg", it is a matrix "N x 2", where 
%      "wam_pts(k,:)" is a vector containing the coordinates of the k-th 
%      point of the wam.
% pgon   : polyshape object representing the domain.
%--------------------------------------------------------------------------
% Note:
%--------------------------------------------------------------------------
% This routine requires the Matlab built-in "polyshape" environment.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on November 17, 2009, by F. Basaglia, A. Sommariva, M. Vianello.
% Modified on November 17, 2020, by A. Sommariva.
%--------------------------------------------------------------------------
%% Copyright (C) 2009- Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Date    : November 17, 2009
%% Modified: January 3, 2026
%--------------------------------------------------------------------------

if nargin < 3, iv=[]; end
if isempty(iv), iv=size(vertices,1); end

S=class(vertices);
class_polyshape = strcmp(S,'polyshape');

% TROUBLESHOOTING.

if not(class_polyshape) % building polyshape class from matrix.
    xL=vertices(1:iv(1),1); yL=vertices(1:iv(1),2);
    xvc=xL; yvc=yL;
    ii2=iv(1);
    for ii=2:length(iv)
        ii1=ii2+1; ii2=ii2+iv(ii);
        xL=vertices(ii1:ii2,1); yL=vertices(ii1:ii2,2);
        xvc=[xvc; NaN; xL]; yvc=[yvc; NaN; yL];
    end
    pgon = polyshape(xvc,yvc);
else
    pgon=vertices;
end


% Determine triangulation of polyshape object
tri = triangulation(pgon);
TCL=tri.ConnectivityList; N=size(TCL,1);
X = tri.Points(:,1); Y = tri.Points(:,2);

% Wam points in barycentric coordinates
[~,wam_pts_bar]=ref_wam_pts(deg);

% Wam points of the polygon
wam_pts=[];
for k=1:N
    
    % generating k-triangle of polygonal region triangulation
    TTloc=TCL(k,:);
    Xloc=X(TTloc); Yloc=Y(TTloc);
    
    % mapping reference points to triangle
    wam_pts_loc=wam_pts_bar*[Xloc Yloc];
    wam_pts=[wam_pts; wam_pts_loc];
end









%--------------------------------------------------------------------------
% Subroutines:
%-------------------------------------------------------------------------

function [wam_pts,wam_pts_bar]=ref_wam_pts(deg)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% This function determines a wam "wam" of degree "deg" on the reference 
% simplex whose vertices are (0,0),(1,0),(0,1).
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% M: WAM degree.
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% wam_pts: points on the reference triangle (0,0),(1,0),(0,1); it is a  
%      matrix "N x 2", where "wam_pts(k,:)" is a vector containing the  
%      cartesian coordinates of the k-th  point of the wam.
%
% wam_pts_bar: points on the reference triangle (0,0),(1,0),(0,1); it is a  
%      matrix "N x 3", where "wam_pts(k,:)" is a vector containing the  
%      baricentric coordinates of the k-th  point of the wam.
%--------------------------------------------------------------------------

triangle=[0 1; 1 0; 0 0];

n1=2*deg;
n2=n1;
j1=(0:1:deg-1);
j2=(0:1:deg);
[rho,theta]=meshgrid(cos(j1*pi/n1),j2*pi/n2);

% Mapping to the simplex.

B=[rho(:).*cos(theta(:)) rho(:).*sin(theta(:))];

meshS=[B(:,1).^2 B(:,2).^2];
meshT(:,1)=triangle(3,1)*(1-meshS(:,1)-meshS(:,2))+...
    triangle(2,1)*meshS(:,2)+triangle(1,1)*meshS(:,1);
meshT(:,2)=triangle(3,2)*(1-meshS(:,1)-meshS(:,2))+...
    triangle(2,2)*meshS(:,2)+triangle(1,2)*meshS(:,1);

nodes_x=meshT(:,1); nodes_y=meshT(:,2);

nodes_x=[nodes_x; 0]; nodes_y=[nodes_y; 0];

wam_pts=[nodes_x nodes_y];
wam_pts_bar=[wam_pts 1-sum(wam_pts,2)];


