
function [XYZW,XYZW_C]=cub_polyhedron_tri_26(vertices,ade,tri)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Algebraic cubature of degree "ade" on a polyhedron defined by "vertices"
% and "facets"
%
% The algorithm requires the tetrahedralization of the domain.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% vertices: The vertices are stored in "vertices" and is an "m x 3" matrix.
% The k-th vertex has coordinates "vertices(k,:)".
%
% ade: it is the total algebraic degree of precision of the rule.
%
% tri: triangulation of polyhedron in tetrahedrons. It is not
%   fundamental if the domain is starshaped with center on the barycenter.
%   The triangulation is a m x 4 matrix of indices of vertices, e.g.
%
%  tri =[5     1     4     2
%      8     5     4     2
%      8     6     5     2
%      8     3     6     2
%      8     4     3     2
%      8     7     6     3];
%
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% XYZW: The variable "XYZW" is a matrix with "m" rows (that is the
%  cardinality of the cubature pointset) and 4 columns.
%  The first 3 columns are the coordinates of the nodes of a
%  rule over a reference tetrahedron, while the 4th column represent the
%  corresponding weights.
%
% XYZW_C: The variable "XYZW_C" is a matrix with "s" rows (that is the
%  cardinality of the cubature pointset) and 4 columns.
%  The first 3 columns are the coordinates of the nodes of a
%  compressed rule, while the 4th column represent the
%  corresponding weights. The rule has nodes taken from the set XYZW.
%  The cardinality of the rule is at most "s <=(ade+1)*(ade+2)*(ade+3)/6"
%  and the weights are positive if a Lawson-Hanson type algorithm is
%  applied.
%
% Note:
% On facets vertices: last vertex must not be equal to the first one.
%--------------------------------------------------------------------------
% Subroutines:
%--------------------------------------------------------------------------
% Attached subroutines:
%
% 1. cubature_tetrahedron
% 2. rule_compression (and its subroutines)
% 3. map_baricentric_rules (attached below, and its subroutine
%    "volume_tetrahedron", also attached below).
%--------------------------------------------------------------------------
%  Copyright (C) 2021- Alvise Sommariva, Marco Vianello.
% 
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2 of the License, or
%  (at your option) any later version.
% 
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
% 
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software
%  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
% 
%  Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%           Marco Vianello   <marcov@euler.math.unipd.it>
% 
% Data    : February 16, 2021.
% Revised : January 25, 2026.
%--------------------------------------------------------------------------




% .............................. start up .................................

% determine reference rule on tetrahedron

XYZW_bar=cubature_tetrahedron(ade);


L=size(tri,1); % number of tetrahedrons of partition
XYZW=[];

% determine cubature rule
for k=1:L % polyhedron partitioning by triangulation
    
    % Vertices of the facet ordered counterclockwise w.r.t. the
    % outward normal.
    tetr_indices=tri(k,:);
    tetr_indices=tetr_indices';
    
    XV=vertices(tetr_indices,1);
    YV=vertices(tetr_indices,2);
    ZV=vertices(tetr_indices,3);
    
    XYZW_L=map_baricentric_rules(XYZW_bar,[XV YV ZV]);
    XYZW=[XYZW; XYZW_L];
    
end

% ........................... rule compression ............................

if nargout >= 2
    if size(XYZW,1) > (ade+1)*(ade+1)*(ade+3)/6
        % ... Compression is needed ...
        
        % determine bounding box
        bbox(1)=min(vertices(:,1)); bbox(2)=max(vertices(:,1));
        bbox(3)=min(vertices(:,2)); bbox(4)=max(vertices(:,2));
        bbox(5)=min(vertices(:,3)); bbox(6)=max(vertices(:,3));
        
        % perform compression (Total degree tens. Cheb. shifted basis)
        [XYZW_C,~,~]=rule_compression(XYZW,ade,bbox);
    else
        % ... Compression is not needed ...
        % fprintf('No compression is needed');
        XYZW_C=XYZW;
    end
end









function XYZW=map_baricentric_rules(XYZW_bar,vertices)

%--------------------------------------------------------------------------
% Object:
%
% Mapping barycentrical rules "XYZW_bar" on rules to a general tetrahedron
% with vertices as in "vertices".
%--------------------------------------------------------------------------
% Input:
%
% XYZW_bar: The variable "XYZW_bar" is a matrix with "m" rows (that is the
%  cardinality of the cubature pointset) and 5 columns.
%  The first four columns are the baricentric coordinates of the nodes of a
%  rule over a reference tetrahedron, while the fifth column represent the
%  corresponding weights.
%  Sum of the weights in XYZW_bar must be equal to 1.
%
% vertices: vertices of a general tetrahedron, stored as a 4 x 3 matrix.
%--------------------------------------------------------------------------
% Output:
%
% XYZW: The variable "XYZW" is a matrix with "m" rows (that is the
%  cardinality of the cubature pointset) and 5 columns.
%  The first four columns are the coordinates of the nodes of a
%  rule over a tetrahedron with vertices as in "vertices", while the fifth
%  column represent the corresponding weights.
%--------------------------------------------------------------------------

% Mapping nodes
XYZ_bar=XYZW_bar(:,1:4);

XYZ=XYZ_bar*vertices;

% Determining pertinent weights.
volume=volume_tetrahedron(vertices);
W=volume*XYZW_bar(:,5);

% Assembling rule.
XYZW=[XYZ W];










function [volume,orient]=volume_tetrahedron(vertices)

%--------------------------------------------------------------------------
% Object:
%
% Computation of the volume of a tetrahedron having vertices stored in the
% variable "vertices", that is a 4 x 3 matrix, i.e. each row represent the
% coordinate of a vertex.
%--------------------------------------------------------------------------
% Input:
%
% vertices: The vertices are stored in "vertices" and is an "4 x 3" matrix.
% The k-th vertex has coordinates "vertices(k,:)".
%--------------------------------------------------------------------------
% Output:
%
% volume: volume of the tetrahedron
%
% orient: sign of the volume
%
%--------------------------------------------------------------------------
% Note and copyright:
%
% Original routine by David Legland.
%--------------------------------------------------------------------------

volume = det(vertices(2:4,:)-vertices([1 1 1],:))/6;
orient=sign(volume);
volume=abs(volume);









function [XYZW_bar,XYZW]=cubature_tetrahedron(degree)

%--------------------------------------------------------------------------
% Object:
%
% These rules on tetrahedra are taken for degree <= 20 from the paper:
%
% Jan Jaskowiec  N. Sukumar, "High-order cubature rules for tetrahedra",
% Internation Journal for Numerical Methods in Enegineering",
%
% The output variable is "XYZW_bar", a matrix with "m" rows (that is the
% cardinality of the pointset) and 5 columns.
% The first four columns are the baricentric coordinates of the nodes,
% while the fifth column represent the corresponding weights.
%
% For achieving a rule on a general tetrahedron T, one must use the
% barycentric coordinates, and modify the weights in view of the volume of T.
% In the rules in barycentric coordinates the sum of the weights is 1.
%
% Use the function "map_baricentric_rules" to map them in an arbitrary
% tetrahedron.
% 
% Note: for higher degrees, the software is "simplexquad" due to 
%--------------------------------------------------------------------------
% Input:
%
% degree: algebraic degree of precision of the rule; acceptable values are
% 1,...,20.
%
% Output:
%
% XYZW_bar: it is a matrix of dimension m x 5.
% The first four columns are the baricentric coordinates of the nodes,
% while the fifth column represent the corresponding weights.
% For achieving a rule on a general tetrahedron T, one must use the
% barycentric coordinates, and modify the weights in view of the volume of T.
% In the rules in barycentric coordinates, the sum of the weights is 1.
% Use the function "map_baricentric_rules" to map them in an arbitrary
% tetrahedron.
%
% XYZW: it is a matrix of dimension m x 4.
% The first 3 columns are the baricentric coordinates of the nodes,
% while the 4th column represent the corresponding weights.
% The rules are referred to the unit tetrahedra, with vertices
% (0,0,1), (0,0,1), (1,0,0), (0,1,0).
%
%--------------------------------------------------------------------------

% note: 
% 1. TR_jaskowiec_sukumar_2020 provides W such that sum(W)=1;
% 2. simplexquad provides W such that sum(W)=1;
% 3. barycentric rules require sum(W)=1.

if degree <= 20
    [XYZW_bar,XYZW]=TR_jaskowiec_sukumar_2020(degree);
    XYZW(:,4)=XYZW(:,4)/6;
else
    [XYZ,W]=simplexquad(degree,3);
    XYZW=[XYZ W];
    X=XYZW(:,1); Y=XYZW(:,2); Z=XYZW(:,3); 
    W=6*XYZW(:,4); C=1-X-Y-Z;
    XYZW_bar=[X Y Z C W];
end




function [XYZW_bar,XYZW]=TR_jaskowiec_sukumar_2020(degree)

%--------------------------------------------------------------------------
% Object:
%
% These rules on tetrahedra are taken from the paper:
%
% Jan Jaskowiec  N. Sukumar, "High-order cubature rules for tetrahedra",
% Internation Journal for Numerical Methods in Enegineering",
%
% The output variable is "XYZW_bar", a matrix with "m" rows (that is the
% cardinality of the pointset) and 5 columns.
% The first four columns are the baricentric coordinates of the nodes,
% while the fifth column represent the corresponding weights.
%
% For achieving a rule on a general tetrahedron T, one must use the
% barycentric coordinates, and modify the weights in view of the volume of T.
% In the rules in barycentric coordinates the sum of the weights is 1.
%
% Use the function "map_baricentric_rules" to map them in an arbitrary
% tetrahedron.
%--------------------------------------------------------------------------
% Input:
%
% degree: algebraic degree of precision of the rule; acceptable values are
% 1,...,20.
%
% Output:
%
% XYZW_bar: it is a matrix of dimension m x 5.
% The first four columns are the baricentric coordinates of the nodes,
% while the fifth column represent the corresponding weights.
% For achieving a rule on a general tetrahedron T, one must use the
% barycentric coordinates, and modify the weights in view of the volume of T.
% In the rules in barycentric coordinates, the sum of the weights is 1.
% Use the function "map_baricentric_rules" to map them in an arbitrary
% tetrahedron.
%
% XYZW: it is a matrix of dimension m x 4.
% The first 3 columns are the baricentric coordinates of the nodes,
% while the 4th column represent the corresponding weights.
% The rules are referred to the unit tetrahedra, with vertices
% (0,0,1), (0,0,1), (1,0,0), (0,1,0).
%
%--------------------------------------------------------------------------
% Statistics:
%--------------------------------------------------------------------------
% | ADE| CARD | MAXERR  | MAXERR2 | RELERR  | RELERR2 |
% ------------------------------------------------------
% |  1 |    1 | 8.3e-17 | 6.3e-03 | 6.7e-16 | 6.0e-01 |
% |  2 |    4 | 8.3e-17 | 6.4e-04 | 5.0e-16 | 1.4e-01 |
% |  3 |    6 | 1.1e-16 | 2.5e-04 | 1.1e-15 | 2.0e-01 |
% |  4 |   11 | 8.3e-17 | 6.0e-05 | 9.1e-16 | 7.0e-02 |
% |  5 |   14 | 1.1e-16 | 1.4e-05 | 1.5e-15 | 4.9e-02 |
% |  6 |   23 | 5.6e-17 | 2.0e-06 | 2.3e-15 | 2.2e-02 |
% |  7 |   31 | 1.1e-16 | 5.9e-07 | 1.4e-15 | 1.3e-02 |
% |  8 |   44 | 2.8e-17 | 7.3e-08 | 1.1e-15 | 5.9e-03 |
% |  9 |   57 | 1.1e-16 | 3.4e-08 | 1.1e-15 | 3.7e-03 |
% | 10 |   74 | 1.1e-16 | 7.0e-09 | 1.9e-15 | 2.1e-03 |
% | 11 |   94 | 8.3e-17 | 1.0e-09 | 2.2e-15 | 1.1e-03 |
% | 12 |  117 | 8.3e-17 | 1.9e-10 | 1.5e-15 | 5.6e-04 |
% | 13 |  144 | 1.1e-16 | 8.6e-11 | 1.4e-15 | 1.8e-04 |
% | 14 |  175 | 1.1e-16 | 3.0e-11 | 1.9e-15 | 1.0e-04 |
% | 15 |  207 | 2.8e-17 | 4.9e-12 | 1.4e-15 | 5.9e-05 |
% | 16 |  247 | 8.3e-17 | 7.6e-13 | 1.5e-15 | 2.6e-05 |
% | 17 |  288 | 1.1e-16 | 2.8e-13 | 1.6e-15 | 1.5e-05 |
% | 18 |  338 | 1.7e-16 | 7.3e-14 | 2.2e-15 | 7.2e-06 |
% | 19 |  390 | 6.9e-18 | 1.7e-14 | 1.4e-15 | 3.8e-06 |
% | 20 |  448 | 2.8e-17 | 1.9e-15 | 2.1e-15 | 1.2e-06 |
% ------------------------------------------------------
% MAXERR: Max absolute error w.r.t. cubature on monomial basis with maximum
% total degree equal to "degree".
% MAXERR2: Max absolute error w.r.t. cubature on monomial basis with maximum
% total degree equal to "degree+1" (so to show that "ade" is the algebraic
% degree of precision).
% RELERR: Max relative error w.r.t. cubature on monomial basis with maximum
% total degree equal to "degree".
% RELERR2: Max relative error w.r.t. cubature on monomial basis with maximum
% total degree equal to "degree+1" (so to show that "ade" is the algebraic
% degree of precision).
%
% Note:
% In the paper, rules with ADE equal to 1 are not present and are taken from
% elsewhere (given for granted).
%--------------------------------------------------------------------------

switch degree
    case  1
        XYZW=[
            2.50000000000000000000e-01 2.50000000000000000000e-01 2.50000000000000000000e-01 1.00000000000000000000e+00
            ];
    case 2 % 4 points
        XYZW=[0.1285157070717654367454741104459494767402619743951754771664984926 0.1395716909679451068215139347070948009372495642970174062333004595 0.6217058655734218858308602288449462725624264675787461190925571035 0.2130740063197066357460721572833601168921882328264813964795771965
            0.6080544789917290353352026976056677921802082913005553415291934022 0.1373527300771633569107882610864618900519280751754135287492534816 0.1458986947266748553574428066516150609037050817091542553840849233 0.2255331922680785275710356483050177561885055662106374200994805939
            0.1617347277547459923615595186392622540961879347711045996392823248 0.1593401922468276001062132196061943110026068555182838354937325668 0.1606122963421696761669446114764495101877657843008427050038823641 0.3424848235779030092612650248825592500338594753218873241996872278
            0.1374481016885109345821280537976724404795934260972582838554398239 0.6153805112897389227931972863574431963718778923710965447030325334 0.1353005423383199276112897414236425035052085434846657439186420321 0.2189079778343118274216271695290628768854467256409938592212549816];
        
    case 3 % 6 points
        XYZW=[0.07010269736516834897736462029976430785011843298410622625050315869 0.1666606997304260297016092880404796118340890479476333677911203262 0.5965757414636450161961272288833909172047011020163722585507979648 0.1803964076327240283334414371862453732640354846299904834352014794
            0.1686953714777434194063881697766320840602583565917915720240463437 0.5846526266988473180276330034071724153303774694336167762347360022 0.1686952187085394908963706207115437603315019845879267042275601757 0.1933162942936617162042461769849470145063728552557915884599267316
            0.4181757892411656483176803182671606926460116703243191603687184150 0.08030372567072929843306858094418692085204089383480180581934490977 0.4181013481561183058236675015725756214640081556844194190279848046 0.1335947988703586858488407665869457449969922688735209123401974628
            0.5965762888488200451527951128512667270383572507169584331552272702 0.1666726351059825905390237222297222566195634855799369007855606371 0.07007860269782415401041675884388745351141240885236800372881631442 0.1803710791466976225500128521115315819241940927971300112888658184
            0.1648085473528479714735177092441034872397817913647892505622391356 0.08511347151861073562116206354661707875932467379499308066010573494 0.1648086847553500152229885987726456869244096967467589431790984663 0.2010167209445907825248413683318606316706884617229592357431595534
            0.07314756676686161148531657432390011208855848604897186370995059239 0.4403391369182129672139387601616732784428248743137658678096415149 0.07315706144977753729237788840752190766020382895975209785913938587 0.1113046991119671645386173987984696536377168367206077687326489541
            ];
    case 4 % 11 points
        XYZW=[0.1187280740805765614658639389170267085472544548912068661126491745 0.4881393122183348832959284457718380124283260963666127760618372132 0.3108120274441792079905347983695953683589963306637501188369796599 0.1418665725465847014948685038898466757366086905064170727503375133
            0.005415713071965579426862766019138515996498526630062101705897797222 0.02862010802441506326943107075001085576567014936938975330603766653 0.1144811451032983997411963251414568538128880931675687990694538388 0.02473208243107074619310608835429529601466618507947362541915840333
            0.4469931106692654477845362509669001332842685008635795026141285477 0.1201149311199702551543190768114253314239961275103111408069730883 0.3324881246726454351952954494667590423404265233026706059002822782 0.1529190224329338591905637483687222855585684009523460726563401310
            0.01268724525917333180172226604324943627793730543857702824612781274 0.1901505501887097238153367310560664684726737978189940520349739515 0.5852910754713261973064770216527621576048243938084895901174947039 0.05481814733153768903283920531124609871858804874964444155035853492
            0.04279583295055314052454812977960144517746363743124432297334367261 0.8193450524383626762104715269949310323059353565512507746886484201 0.02671935171415229235800182204152276603751106738335835036068917651 0.03169672337724006391736600290382916976334567602894547109382939627
            0.09191786683442173602129512412285988925260333974142827212944369053 0.3372182623043156654408594625086703909804839401143837322193079212 0.1126491331073949636463747409667054840581556135361708713660978657 0.1515632978457940561397783044970716022551875530784912175711279856
            0.3402164177011169403465418482955694703105352216790032298958498490 0.08655748467986596439619122583411920611631218967291972081566120883 0.07939187738707083682460293462966653785109799706329421108124558322 0.1169682461544246930646656514256423568008967350122483311821136612
            0.7509779430720097359299199889290449338923992991393724371062540151 0.07969252419357779339651450503667351419367665385151290589346157362 0.06329659411388904167191227868981808245280284930118207801685726505 0.05694567177765836933951634097774173135006883684099758624949362770
            0.1450477588516786028349056403417534991425132685478121330761576754 0.06120689333101743624977598380996843213550127154582266233939177055 0.4429974680874531993180029313529213717928315913515097735081109304 0.1171324626635184437009794289261547815785026328124359468387178781
            0.3940913370297614660850503183792239579827990046775630965962743152 0.4395788784711709315009781992837774174263097579764804086490582064 0.05774155056577569149305980156403274153999695643028063085682442527 0.1112955667737245230333137843765286469109973301301310630123212924
            0.1316834262016172706240647729033412361296198937782777905412054805 0.08522680266570491482014013895078729670321620649339994132872523735 0.7742840020908915685951289509575183035363967422243507367396688598 0.04006220666551285489300294096892135531256991080886917167620157592
            ];
    case 5 % 14 points
        XYZW=[0.3108859192633006097973457337634578329926174134397483791027517499 0.3108859192633006097973457337634578329926174134397483791027517499 0.3108859192633006097973457337634578329926174134397483791027517499 0.1126879257180158507991856523332863338104677940836629167501113137
            0.4544962958743503505081194737206605609343675783809439756985401565 0.4544962958743503505081194737206605609343675783809439756985401565 0.04550370412564964949188052627933943906563242161905602430145984340 0.04254602077708146643806942812025744177762727811712344868328099570
            0.04550370412564964949188052627933943906563242161905602430145984340 0.4544962958743503505081194737206605609343675783809439756985401565 0.4544962958743503505081194737206605609343675783809439756985401565 0.04254602077708146643806942812025744177762727811712344868328099570
            0.4544962958743503505081194737206605609343675783809439756985401565 0.04550370412564964949188052627933943906563242161905602430145984340 0.04550370412564964949188052627933943906563242161905602430145984340 0.04254602077708146643806942812025744177762727811712344868328099570
            0.09273525031089122640232391373703060524520335376054576302841333523 0.09273525031089122640232391373703060524520335376054576302841333523 0.7217942490673263207930282587889081842643899387183627109147599942 0.07349304311636194954371020548632750352309128874065191022496719265
            0.3108859192633006097973457337634578329926174134397483791027517499 0.3108859192633006097973457337634578329926174134397483791027517499 0.06734224221009817060796279870962650102214775968075486269174475018 0.1126879257180158507991856523332863338104677940836629167501113137
            0.09273525031089122640232391373703060524520335376054576302841333523 0.09273525031089122640232391373703060524520335376054576302841333523 0.09273525031089122640232391373703060524520335376054576302841333523 0.07349304311636194954371020548632750352309128874065191022496719265
            0.09273525031089122640232391373703060524520335376054576302841333523 0.7217942490673263207930282587889081842643899387183627109147599942 0.09273525031089122640232391373703060524520335376054576302841333523 0.07349304311636194954371020548632750352309128874065191022496719265
            0.3108859192633006097973457337634578329926174134397483791027517499 0.06734224221009817060796279870962650102214775968075486269174475018 0.3108859192633006097973457337634578329926174134397483791027517499 0.1126879257180158507991856523332863338104677940836629167501113137
            0.7217942490673263207930282587889081842643899387183627109147599942 0.09273525031089122640232391373703060524520335376054576302841333523 0.09273525031089122640232391373703060524520335376054576302841333523 0.07349304311636194954371020548632750352309128874065191022496719265
            0.06734224221009817060796279870962650102214775968075486269174475018 0.3108859192633006097973457337634578329926174134397483791027517499 0.3108859192633006097973457337634578329926174134397483791027517499 0.1126879257180158507991856523332863338104677940836629167501113137
            0.04550370412564964949188052627933943906563242161905602430145984340 0.04550370412564964949188052627933943906563242161905602430145984340 0.4544962958743503505081194737206605609343675783809439756985401565 0.04254602077708146643806942812025744177762727811712344868328099570
            0.04550370412564964949188052627933943906563242161905602430145984340 0.4544962958743503505081194737206605609343675783809439756985401565 0.04550370412564964949188052627933943906563242161905602430145984340 0.04254602077708146643806942812025744177762727811712344868328099570
            0.4544962958743503505081194737206605609343675783809439756985401565 0.04550370412564964949188052627933943906563242161905602430145984340 0.4544962958743503505081194737206605609343675783809439756985401565 0.04254602077708146643806942812025744177762727811712344868328099570
            ];
    case 6 % 23 points
        XYZW=[0.8746168885670683211515560342705094036625597522531710494830371001 0.001650241439687542348920482354188793210543315647551672559880280318 0.1042983037802942551421306327936264304288745676284915740037305275 0.007205133350337782940137339490138164678345058797715414925665471704
            0.1624377743364805917899983031327321669624189397722713675471474565 0.04865782566395184274524243214177301556975516465791953661594731704 0.5554865546922777798981468533075921508275740201734977139942936650 0.06472302348093830152718458471759595426848016405007927649421276305
            0.6255956733270350011839827995095490366958470642789806973707627437 0.2167840595851656861011251376166332396584702540865947887664052054 0.1335729717161918626754589676041274142086231096600579739857123887 0.04175612898142028926723733140190272135003126197846216764856809282
            0.04855903498549970698735616559058971298799201074164384504913671197 0.3152341342565913027955530634336081090607846868589394083529135238 0.07777994479067082753289110394651330623096865747261254344788048822 0.04688665919497774445372213168860159667195375523126050342891204037
            0.2123913908430850203242966168987365383019826059577368186669395016 0.5512046215526779823245171835573365708142360580078244335008637263 0.1911298253738990766024456523240259711727272906264402307376594813 0.06426265470498371451658160013026891691848064469921190457616590165
            0.08025024911809860240657982790002946957925913688064129781411137559 0.8735128127614880874360602828497898437729007648697375988908970404 0.02333982278796531609028058543735758489340996133478186731866805100 0.009710719914043639123721288037174395894533834997209642430039355032
            0.002045284371327750281966983976994857212527374015015020870916964711 0.06191857376022004358329341386809745860270045814756630123056300418 0.3821105213368406413368030364029761487162277850039952576543215209 0.02283065076321384934828300305836596837693684953135648907538393770
            0.7430454470768171867847475481340921785158339840892175634068081500 0.1116735876122876298259292321394102481391606430873028341820016639 0.001400971492713620452185227365020696765497862073586368164046200245 0.02039770409962432293073745686341474447066427331665856482581074957
            0.2912491656449773899276741507807177096311368003174316163092120777 0.2201903067498265652903049518178370314778846372585339020860242023 0.01567729286548079800976796615624429003183369672950291241960402308 0.03968001929848992052501991917522553358627793761537927749774785654
            0.5120512410177051661169687398320834512680193513432287154856725427 0.01140461112584902193069555064169925955231989674630231720527475523 0.4147866887697848493685877943468155752247775399630782993791980746 0.02133035504471112136665350081653808097646984058517880410827335423
            0.001195728927010588214179446967734119750747882412115799710449056715 0.6803017849056812688801808198974952034810246390699550875634996789 0.2350505385746095678806482975347333009336715387051331194310737076 0.02116339423803208968610322183233493952108125614974998316730377321
            0.4182556428001524980724747062820275357139523084593716525634188443 0.5029243848539261850073581233145207600070380954269878712880193374 0.001482349258806382118029746320525466697934167651916598940034440430 0.02367032154323294435505272392510705370581710798242764113181429569
            0.07564950577309438537482781255994015502454834361594065348861756780 0.3323211418716275455936010234022166701082300685232192327021713516 0.5403722579539822064299469850947385868995640971157034838536936128 0.04905329690183727516257115299110210637930162443257931315969635348
            0.5480249213620508489214294146278656474146199859372967622224250633 0.05966721644700009976969433431252907099955006265257295332734345863 0.1545963494454863390575505406683065458499038596962591573497946565 0.06907902313785016107071432258900567114894447860178241232418779610
            0.1410085046735627363768578203759566929989591415666733642947889341 0.05089373132544250283124628449526797730892208626130255153354913049 0.8062812451359968514507145834520217542496770814608626710075257459 0.01433110646771543700189670807549236554796760563666000675716477845
            0.2166008062191251411347784741587961059103369823478750773558625012 0.06860031149169896682308501855062138362028884219960659159043512230 0.2168670911641968115369299326816732950511530487666545537205769803 0.08365916971233740563858822692325458773320178394953561736997414801
            0.4246910368665688903852863129213182235506669994838484206231833532 0.008480121919598680578671903126927814010991038160694223798158123664 0.002614592869876169181527701311231280789795907522495676278695893735 0.01084648927559288981207903381976897454901249314313127320532089374
            0.3340496585496776964479068591872794546478087725688983817197146929 0.3098946335446950811334092941096793296044413618109673577350993199 0.1284572624659843743988158677977683345069632079799698041375549889 0.1025240624805954140108871310034691437960075695782700383460649286
            0.3337613494669106215532914533702865937003545576182918707421274354 0.1731672255755042396626892403608544650848316500845479826493806075 0.4173503478363065589560481565040762457278932363236494411556683279 0.08301291941491797035662615766313575302762390968726562060635034244
            0.07670397394343721220036492322717171932529887779260828346672031273 0.3017356805716037481323103616456862551493777109598758835117934753 0.3274922350236289851567823424593573489172558208865565310478874026 0.1017105458079781586389930630390023458854938235272702912996596639
            0.07187641262922304176222905703470437062320006138601001730699062084 0.06288888243458653410711604458147565213495740472397274040890038920 0.06706093747844392364644082983965292013119428121127964670693533182 0.02955279671645887553256869887964162121999600740458568611351032447
            0.006739627681279246598233187323114368141182559370144488473382397554 0.09047458959966058940660845162939320952292168490247792743059309366 0.7668964455753671441968107999114920662838656483074079802537356244 0.02007399654007994478750430415115075077118043135795843462299850329
            0.08409554458377318756266697277855057230233915003210555670426740187 0.6211204831496858994577294133198983857430449746121492043456906043 0.05551455307013085019413398501802894169834753728167772728193080595 0.05253982893063074794713709972830860952219828774627163688517467573];
        
    case 7 % 31 points
        XYZW=[0.3002416572973887656866992672519355286700525757977768998363441806 0.03439587640900977848044854484667970409182930383197853347148024985 0.03635743445583757695286062546210534048307203447815373746624143152 0.01746349349656299841086778811672928570465516006519633046880037335
            0.4562748748421011559492795637526454124362405277936776605195339404 0.1770949336338915326154957139418579027504222500966645550628678073 0.2023800436826119741637662358749509003060685290487436657437722744 0.06498228046837347635538555914198267721765906815561529441616310332
            0.04351567070579801181952788424515304623822014316841976383729883036 0.1907080463902385792165225155988818983512079846137490373055540623 0.5156119554737180579293958607822366531287567211735470875539248000 0.04329094194118220857838121515336273278501494861635374807914951390
            0.06747042686520559919148718746375322323157932479786551072078692093 0.6248052088506927347062442263724047514352234284050913767272602481 0.2864704911297047222201140241941160080496074197645225948220984939 0.01969661830969762822734938940106040252604797024288066343720645645
            0.03517579620044008515680725665028650044884492888383611417943268853 0.3009324536669736934027108173984487052549048817718068240800557535 0.6263899456052338372620043776822295855814948390131783401713560290 0.01803660452307751841493601872287886802749790623772031246819157806
            0.04655862318236976268201514248832519761501654025251599138731453801 0.2782694670147702911981473159589136215611766736857946431410225235 0.1904686063806941764876569796540923132578247425925528625311597234 0.04657897895007257547974218141341820092434846439907761466674682463
            0.02704000074580650329739276408467490138286545246434791535263787874 0.05925535137076793813978499976762919446439224152039282794899106104 0.3085538422115337493559918097413529452542205881967805652525507776 0.02082559198561092147580686487003423652987199887541534609653716428
            0.2646801005318440835575112386393895463508712167407997820372214220 0.1971493272516444871568259395228645422130637459780745828867772610 0.06497654228552173910377938949958679605113682882845684669487005044 0.05754252163356967296547213489065306099916198298820333686630330788
            0.05843088702805924173129736118824468388569827597313913160160400001 0.8178995621576657930359360906854489594733775463697882359918311530 0.06038923495562989096924364436391935988355480820541288511370458662 0.02119465165017259007139457229140243840830920105514261177748421750
            0.2777673159649886781636045958789863280716359898007711933948155675 0.04767365733981071112509810570381723423870731587382125846748400374 0.4793739967382065670868481971474885348283927965703528906213127193 0.04718838116371158418551891806956473712754130668398674309216241182
            0.05551463732861113609589576954610409802893437147533265684300059641 0.2869217065138451684371717686782282837463984980928962154402035270 0.02724948084801731479831425317911633153464971223502325476426553420 0.01965015918981211239008307024846592430899337884361241278318801301
            0.02037461645928369105177827792906589980609517782584956660235907104 0.6125583966450518735866262037438717093777917379126513670914173563 0.06230181632969301678432626375543965681412349211192225426197960501 0.01848208373447896788272569136959831008046530860072578749166983128
            0.1922244985520259378173352527254720031198239687789418571636353679 0.04302849383342601343526064502392125652679077706490414869759733072 0.2485713056693554494470769563260976838848269467790360009622383367 0.04274470110086474761441477035507362697278927103930378400736297045
            0.1991087918995903692938841224839894801804724819808213751592951030 0.2642325286400432477392993554985002122330222694485680866306470644 0.4718676792912370988337835998260627303283439153020014524022072953 0.05764072284578248473815219397480812100442840787978442870578914334
            0.6235525826354381888816910409226592069397231964052581604221064879 0.2894310610359156768646203878282718768470569212651879476763077419 0.05148311927580126367913777399584618199119202735048157678633332474 0.02235761391364179344971116049623471411799009979570133834678617656
            0.05942256900961527889070519643216435937141695429645645822223553504 0.05887963068014094138452135146910114304289468638303703977050788413 0.06378926589682203746162658174612263506048793619604931548190377123 0.02121913845260227263763644111171274866835087348190290240692331278
            0.5028328636834030614048624510389734038223954931144313843392684768 0.03114362120227596860937503685829822129649939485284406089170843617 0.1603144456632615508628949556668836492544993161300050641937292397 0.03387286857333518375214178562264970336926779034746802376425130737
            0.1646301258774454026955823306071147789630536769078953981275430925 0.5201765728647110749180942681421881401384955499760388858139145504 0.02287291427434434588177834489416018884005905697080773425290378248 0.02679493575809087234438749757092856111190297740482753247987082306
            0.2891312880999799223705909406672058009414217729530984232128801434 0.6241982800589735951062400408183700680769895014577523934799290820 0.03473184011806575732833739776949314761956811288384679162659577637 0.02210398035048985360419426618434389043426182499959589866768069418
            0.4579592178379015607956331254262215890609493492288368305890647629 0.3051966920830379520927077776982432184308643909505134204567754544 0.03801602948513748122762527378917368221313419951849818274821775804 0.03735595782259541920412262548479575795106981653738525658003538814
            0.5226788734283342929988377875471312977551674800054936237498124561 0.1641956191205870640638579371381424147003885718015191143577963256 0.2936439912332349367098453322490357376614844378798828653636783985 0.02517150534658316899885722667518524743522273392207692608763267987
            0.8177970251994465695208587297891591191387434438243797836899628092 0.05836065147429206950998031780706418605872405015512551717419403744 0.06354525938565314380561127122141572768756360306010386327421693424 0.02123516865835264192372623792187308763642352446224143787188855618
            0.3053135448784433060503422138244883208111707516660466727793331411 0.4603995934951201099496039244184516473620768501479896121135598225 0.1965711945189183044257549020141382053676146100644069005492430179 0.03723519909494987429420525926795980213585575044291500733971077660
            0.6106947282193217708014004939158705510093563109800818306551177464 0.02022148557997553456616481976654899997868754392361103699801751419 0.3055925317265655129895010914817928859704408296907068970140185947 0.01872554972841556670787041893669026395658095429810562882290708085
            0.05856964023548992432656269926475024405129924069493895136193596170 0.05987811122172502460189748915301092697670834377749827476398130376 0.8184045279916535126715765110962047533647337293245271842401383518 0.02104105715176330442109603360264744045711018002878621736758974020
            0.1947183776661725883021182791834612529547219913266494639172554428 0.1980390771723790386211150507092575555985575119215444485368576790 0.3015283379031818455114968562154812132239335538631806664128864033 0.06053153533562694097383387616179358029502902536946540473276806352
            0.2858333457907975336770204071855020409659393492987929376691602668 0.05471855733121778042734018902032464390307339460348888262295807296 0.6296884514679080189539979097287381488042511902661284904262894029 0.02046100323875207002919952979696991373084961206304785679351598752
            0.03040919400991044809756611492204499301469079812815907574032897595 0.5012486403424117830367920761919834592142574505460145754395323826 0.3042244651274234441504007185890354941081202558907298363363283746 0.03351092315697243842623627396414892983984369122687894867838006555
            0.05971708157604289460364452305206330253445013219174866706995444657 0.02595901540764639682925672397757890862323007265246579812064187364 0.6093786510419171533787234641962141069892539386703717555276146905 0.02034625732143072352641213919089956442071668156790676566409058302
            0.1765706521407181650667343490852300269646224681407968680108292508 0.4553614944072237859549715831356060506620979225976937055820973784 0.1676294047317003065909522594355222949747732065251159982880545222 0.06356544866906927125935015608007763611252775324191305379227259948
            0.6262307245732673361979731088984081913760273318246944493636663489 0.06805761811995238408657836379138677913411213203740740818468461441 0.01987916664128244372824290860716914245666808262912666155550539648 0.01915412643435911765678870391205653571021233712676338624694125569
            ];
    case 8 % 44 points
        XYZW=[0.4145770165127839523260563162137157432248014085591379301616948211 0.3830837692811989755046493889732289198933103301142227002417440202 0.1912186370119430237400841976263187920469515937241671639703486296 0.01816992319296444806598748256791491649876001290385496135265266837
            0.04756498669496588706112675145299493824919928312001304870349063457 0.03844137387025071653863590409984309752730560041315348351273098171 0.8871119506366457448286254056661856329696669222626337324334267776 0.005610582376754646253542447320811019454262931309910802129013252531
            0.3022392039475506093410605685632070637023565967458450814342373056 0.3096755189223818289352494834843800005487142134942805118082169552 0.01621645270453462382921118841152990076087056298505297515765765881 0.02461811359257071116457930863311443260078312189723503194732926317
            0.04244441849908791230693112165521358254650701856787000217986931633 0.04785486560640255940518380926121621141505152628006734788430477046 0.7264185551267094125129789345338852448770539248965389739549113475 0.01805401819601900387641424707630385933559993274282327984205796482
            0.4438540306537173117427021019142288839991577467206872210535733077 0.1593000599996238854279103070653453114780612500471254828923901962 0.3626667438035576362774253301206844319795419021525815671976315481 0.02675388072840435007213716433270220605320104247117260070261972558
            0.1349985069363986610905421049799078215706301544961755110529663741 0.4329948100296093850371765329189473642137794865442414269814686396 0.4065632160324652587420839967650948644435619339830641275379715713 0.02287988794382060043495552321985363942760182307857961625622777497
            0.2702491895036498066587238426284318620171222681874464778911191388 0.008432871254456106096331768590487914748391005172763338616843672462 0.2125111291650657491262296852500526649654935023857042247802058979 0.01979240000909301825498521608442359306129328438458426775950838122
            0.2141588954260017017102494769866173195603041983684647995466321403 0.05020385763562403336697559095902570706371580708288322223473456150 0.4936462128941918861601406245418568083453443928773447717544994968 0.04079990775085950930263895174082839316342324852046105216346237674
            0.1456793207565755050105102196886674074294509606953632207398803121 0.1420799277732693185114564257431534679521447526623928646166847203 0.2654961248080462445199903250239608300742798961170373034937732210 0.05604835500630743182010644461992327634136622722917127989666375895
            0.4800422639855878545337162897452538446859864357487744189692954800 0.02455159627903423994389148693796192002560463746554179533436575360 0.4555570662781256937770815193260864072585190662590777018554533184 0.01164453502630739190538011826222576207089385166804321958895527067
            0.03248920813688127819966726579803172013634147374575606919655855443 0.2126005989136902796609588612758417493147142158164204271568927142 0.7103849921091126047512532863060232861332581328299894776468150766 0.01394128156335584233691322028258941988106163254101589187607976256
            0.6991852897392833618972174748319206179214488793008553721504274485 0.009268153293509993251799932447052196755741149302784893432622987749 0.07326958716291519821578098856662130443314997530894497342138691247 0.01204346778275301877117813443344737971677205945588432534069131429
            0.6912707213620969769557231557164322166524088654127737405746858764 0.2590322769551171593083607042745117761962720469944330458031507056 0.03655068479224149425290442071827517567814296709284515052332976714 0.008058305251223210510055950738171070581391515917366116883928658870
            0.01110928261445524939677388705262486893621789160273289298776751433 0.5019172513337366495053434896270780980211508217778041097124499645 0.4337572382922951188785756739346361193506529513240288319294097578 0.01116402260722876327670033649002872167059205476522075074406698008
            0.1589177658747160214752027313304510549087145617292819135992530658 0.3525635612848826337967230653845339752176581470531995583435742526 0.1335110842049796110408807624752835905916634741537024447675625034 0.04890169949804244589243294779685234003644357147130217609746581417
            0.06361953532472796614785076875493478847025588839003558668360579604 0.2425694992116304220406128752875800377227546343458481206315251712 0.4950048247755332323544446287520426754253483254064475367350500157 0.04263459696404202165502918476882317421126034866619552986173019465
            0.7512107984274565353012916132693028970496997236985264786787219994 0.03613079173054786446588850609184719107336215084626466716261261395 0.1814260426523001673003484509613689272798481545535818791986128935 0.01104116292165817664145046568590974782528212949410557324200128494
            0.4416365362117750683299341139846905008698782431020430337141583157 0.4543312862588164296492543913781852658990302194536865388904177477 0.02412795208653873250597777124192506066716590421049044142972752033 0.01872970233228429185192360959270976709852326471238128086727231921
            0.4793998925970132339647537848105723015323869494589153389504885201 0.04121146360448955469884775120761877007515494861177775650785920981 0.2900348230690179699925667450105607335806383328893389118391640934 0.03326724373021123822668820649821672064820569705342629882474401884
            0.1526196910014446176320370029722023229794452450481882919036187300 0.01104942398276893024188018750282184698989865602510608363361819609 0.03947035584220393401745031036388426975770453172858316801442427937 0.006754871470699925047713340043111735514313113728347044357741209337
            0.8812579575152194077071948125450496061453135142443186630825350327 0.04935459249309241617739622162223410069360045600433329928228452087 0.02381618927809267813513256341169493124376761209767449834797258489 0.006242538545827603548039066440186178474642606591950010341665821771
            0.6348065288024368417490557079646890528307769137822988896760420942 0.1477269775427559021793835033306234474546254893067398918322935439 0.01432119099457404468546336920110003925379040783623216076658362976 0.01752586025912829408600466791606617391968214240620055127008635361
            0.01615377683620787392925198709409029381919061994192553406166767562 0.05331588633516811023430670677590391330741428656053713307032690280 0.02542799570554903353278545948267758333465598829972134756353872504 0.003488893720357850056106626073441343172256636708467731603081594506
            0.1571350802815834165925008666272940322423143087234768124456896849 0.6346254701849137141993252806292067551282698914927234054749125001 0.03929870015634977827382241305513598945277004909398388350011549831 0.02935166662665394874876182620800819022314788300021186897525517980
            0.2221434793473390094975271620704443691043656519490933897453018439 0.03739285310701666455181425819891334267155134573149156836287182321 0.6865530348199036059004953003708333503632875873251765202938376036 0.01674287823223452199822683849270221014476305401768053095928885817
            0.04741976612308610429075781416693129217523035465910981663611076861 0.8920202885533636649656981564374590545864670312451635420735945841 0.01355820160253030897260322617578140466359908073036203000076265536 0.004837963053622819456642838604704795006579451564709800738872456613
            0.1971257374399104337243786052730102035498422026888031971036199816 0.1263766406253084003367534143222088925460996137325873685906766863 0.05347737816301998482255537438368929431955752779716472064836615380 0.03000460309354623305435021780758228465667493299207238242418023227
            0.05579486121779924735981146215620478825122215329063774502535314297 0.4698411775898500962937994513698821915984500259659809488501083752 0.03096815675648862464240612477157248574845552501139441811034906584 0.01847005074510811842317869064443576941301147480222887468855612534
            0.6316261452724231343096228161048671514658303133720767739848223617 0.1564233918934593708569247439984739015905392318349254827801831079 0.1315762321268426686201296817949013044501862717937979418703920302 0.03149108504398388292698315400620301583138867423966540812533012747
            0.1941392655923862140215379675613514418288003274699574004836796593 0.5222754374782991846732384178822411096958233971332112607542100944 0.1993778513250609625889839311118421617490888991767784846515640307 0.03737969199078129055376243089635492861370782072945405559124625764
            0.05342664290776658854154756212247456117909928348835636195678109917 0.02049675335612217454115746328747384488178614727159053941214587861 0.4580407029017354211015919359591539633587187538124314501123684190 0.01459278455954343760345614167400927407905080180134943958941736225
            0.4151950751920222413190224270565479718305893267862007961544873957 0.1206348095358562424058187844537095272314276303818418477952644968 0.1356578205019857832271855757192394547494599345284822610732698684 0.04944187964742250949968520166011860069576324791604624701471494680
            0.3896011380561273344919767613892312566392770020234468775618625039 0.3332724856599186281438563830586026269388764940118740670328896683 0.1115717539936369081307292806643017961472197127392434121250949194 0.04255228787151546709918247859048744725816293366948312401238293239
            0.4405519657828008015011273678646562349271497558013920458516640470 0.04262372771846321807253430017580931230946081528011073308160570479 0.02988794306725831104226253366740079853584432535466412900438441703 0.01579711060544557088227331706021987058633751196121527917816573593
            0.1901087733754798970527322498160015730297900264622450028879040466 0.1815142502860739299098500452182412097973626912794655660951218082 0.5904777744426020773307886866398397666779282535385433939393703896 0.02414200245018548686740454560967733803606371011026542802473177581
            0.04452852561676643761543685401294553345856330036836590385374593295 0.05119555882859951472512212904601912039046064802354913947264831513 0.1838336361599616296925526426406552137091773112702799650179104442 0.01989585861926247536428483037379438444559531597185789553176444791
            0.008617799898794720961797790528074408046489675145755588416533084753 0.7022002301888654885670763773324305113213709930883443483459083537 0.06639890449480630645698701720943809432127551248276656310515021571 0.01037440256858263975534394572642041027203625119637190977644337737
            0.01160409006576667221091175693229381062987872474642505460703958671 0.1433209858915356446016268267319961413012184978591209425621834175 0.4412164737903684430921864576019589738368641926949610519051686529 0.01435845151360826941601156946627201540898142009285044848556444770
            0.2455987399683951671149717295926094359725777137414395194043375768 0.6929759446465587724416333564175388415682558435369013037515775137 0.05252859154932023514052254609858751594003904170963424598194442850 0.009882447251868656491492661303077815811027138628983914384497484214
            0.05006102729372312557337981718594779333151876422413576813150270718 0.2173800961200812981801587682885584580973561655809830731757459357 0.03959081503289849435352089182333378442129230509653544287994012493 0.01574647974943123466712488077238247788123776951939333331700478956
            0.2671415544232039321697519837973616156547462435419524586593688429 0.2376839659731869312533929626270367161590595734153733122973132701 0.3370228174476876520365694630492699301952219337044477531723793731 0.06188107670855173725354316503285864412141922200764431045136734033
            0.04805066034237290936043330172197807087119230819767254145731223778 0.4742858045797399153895590961949979318327720462906493792850516576 0.2518463964414198878897144382017686112531579103152402125287139636 0.03917960499575403379044249112997155922774604100291307974817692011
            0.01305281180096150091708221063615920011219388558429121782315557831 0.2822707494912109361621218896750212791905568510103005725303873511 0.1848591801453774458652010535085312772274369546754956156292552307 0.01882938169315409561722978882023364989766010691369951207187375580
            0.04754699961865440604214772988601477203834612445300266940930183060 0.7516869678701615458418566575205061234828950161347067601165316810 0.1600276135512940932511954696393140573156637695813313554882220296 0.01688304250982977747965632550283044763203298814420376396211968249
            ];
    case 9 % 57 points
        XYZW=[0.02287719212594691969238364620247207258528259582316138309600278667 0.3854277790362540678151657031943119844721132958322404576931514498 0.2037937874247621969093703779070501004228356253708853266457258475 0.01845410409506825747552158923764951654488501116216510709426748016
            0.1779013709110090783698179586470339428471656743238001514676290251 0.4091789152201282429804320045491010659914515607147578500728175861 0.3768320144226283933755278120800405043072297194789123531564741747 0.02636884413793378261083116173495702519015992801710721726113370953
            0.01533462885048548410054345552015955563029359561628269269667256583 0.4395414314627663569746050628746102086604990262418726002397815585 0.3856915922907349836903823771935318347517916280387174631893311089 0.01298436299973524535017227187178775018115649198317547914772544465
            0.04102661434874325521765876674738293054482453478177885950336207632 0.8702665707058525998201204605405254875886007866026818296635112125 0.05240075009433512776137537645575592243970903287847561133965714950 0.007536988747324099163001874321482595711070016871990476944966917759
            0.1221435077921257375841034381895619203527447402387988560279090252 0.1703632113240204921046545332380279174328455823662683938166909557 0.6931483893283681239893367414857034852261555593053659747092854819 0.01129021459574985239698280205589066858097006384774783934069297948
            0.00001300647882490821858074028500308796795007354856543662556421734243 0.1691795878401991586781424618235963636999231969766110555094026361 0.7840636802171264387964248926582205789231145940447825687362923560 0.004197459451012271919206226241059501117040877794310724650076769499
            0.6539546088038215416554093783651624981220317741976085525598146172 0.03580138983355016566574276553244419022801679564011497421338204373 0.1084429676794620788574122496517296056673835761787771953851319068 0.01952974812451395659734635683129758824135194316602354835307051084
            0.04237202744110467717009895828830805052962397367474981800382433039 0.04623303913344045457294016457981535111572607877959071921498872627 0.03694431376695004416930310662934406430384733396722585662177734568 0.007070128308920461183702701460450312306566264941866399194360858432
            0.1676767943964418593287954081105171875312166454305246720444304340 0.02761060565207890489085492816083817976962233942853648605039468555 0.2633783150667404184942795343886395360477802362316593879926959926 0.01822955861889333003615848884722344289517478955292040255569483187
            0.03004165456210961738866171392936439281750722810659089794951068804 0.1684488866191471171405307020580624027760019586586184572264156715 0.2257928860667317241271311788302879368684476762695458939294805981 0.01897424997994633829667303135029128280425118582688810123789976796
            0.2969101972787469040671988164301119480004799300287856313266829504 0.3463378689772197124038880107887937017617371882404099250684991268 0.02433878290908153355007756914601386577155089801018323123787557582 0.01247902990466096100008173775079267274636946603803668366763256150
            0.4389309398965947607704874132302208609598971324352922855178336124 0.02427526936031881731239037753350750617199894068194238444776132382 0.3599485361902332846156905292705616199822481081689818533626237202 0.01885900285298375157860967747772849374718534071756681485080452827
            0.5060894004176661426257474394704652684184044017859355250123721301 0.1129766995689154825357381293785989487617975966322309696081831365 0.2848034194058732546433932110120998531799467648962647962251715426 0.02374855088176500681108398358580408788917001389726167762966880679
            0.3093454158727241102557001090369193659045300835567190101600145607 0.3249822555407154213498266942307991982330143250412548124493054062 0.1774721052174033791814591828780090502277274590834340955316230739 0.04086653822830791132412369115441927621828049475932071213875162851
            0.2003288579260044391501746450880029820365831035559849926814609493 0.6356109460036826581523552318241045131617360428400657298046163728 0.1459507330961254944255377467772847294834541737437077849844316042 0.01227814341296510742443684601360331057343591478430550729901012140
            0.6409443148321043405099362073300363827645981113559250620809087384 0.1669655587902913971931455016975045161511666410323952056413603944 0.0003804307100489131020176535545877093185621789913539041534225702667 0.007080971736782364232964619963830307162376101506365644016362717036
            0.8684348906155640890521887854715105018634699171790064125014400423 0.04595241793043273715589842666237528588668119804485014352010466375 0.04194385793691838808664941821574073264299877064014830935621356954 0.008242683037014377131452469322908997116224170924417637077644276627
            0.03532441866247937998203036866897450073768944411612176946441313715 0.03426027817584332846474330798472025097272693323760863464078210023 0.1926456994684344593430597853572826793849616470481804421843737787 0.01019679557298388682729615751045730588911614298947743901273343932
            0.2082525047814187213102921204461012172936827902133477756391439068 0.007498900544573472929467514978076754825942862727517851841187258270 0.05874804056749174929887047561948539380112487772644648131655613900 0.007169073146995515175540440140260098045035632799440477564005094042
            0.04249793219020790316278089482338581842082039452760565900918515867 0.03852225238810312429880912160101485209808201577422968506627100379 0.7122430931384430996952353862470610564676246779935511345638509322 0.01405601645950155477305855973018967023183100870769722023267348958
            0.3992138743685673258256841956947415547754674076603725613896392766 0.1757543676917409351732487979086156807212392454782143614643246915 0.05095278262061837458228117135032173011711815610207625560891583778 0.02882311372705531748488963409177928632049001975927377255528728637
            0.1976857681788971034065484590866097953567832650617252196052842051 0.7343371859702916032174975089489255010051797714493882275871360731 0.01963027402815139235692320338558769712242279990562471690185325679 0.008907737555643785998193780168545086362616866808071456249792128009
            0.2099687083368246718958772828202027079536388790718099678357486490 0.1044992464136398899660086432077306741841199893349875239897031253 0.0000007540268307832739000490527994114715355022322622507213269166325885 0.006645719886780895720770095543575189364813366233605630455679216134
            0.07549279545611463642361737492497729626123897196089871163932790965 0.1955780460642323250078524379913790153259927106830749968695359239 0.5953204920779403869074362940064617744717745678619703281683748428 0.02990352654545476940578124731709465654818750739615716855546217869
            0.03482679719386750118953618526691836417727118735343317241646105935 0.6603573627255149318924507348507932067983708252462106905978644307 0.1200651834331449545151040693690683731082738961288188996186132000 0.01995997946254851648360681105686749195302141825675449602896989409
            0.02837870852627252838305609833200526639524963343686563282489681118 0.7846734722712416329781016648687578410916540711591481002786519164 0.0000004032714584772950642598528862272645433447723385919366052638689646 0.003290291235590296381745981527566426375171714105850531915631918640
            0.4599570448520207366531747899822016312928554008838060624924625532 0.03293952190549310109726483258848512228856178742792638531268803546 0.4785665294057564631406899300651647245297732866830923556104545750 0.01025264269584168956217456195328314635918888331559823070432910153
            0.3521798518257496463429445301656988106304451560541536456623301154 0.1751588071180923500461073470394681454868210127331596459859760201 0.4612374007141584901974832090631567134818570455614116014152715489 0.01247458978083449504413102536131433907450566852981332192158893062
            0.2171825728901714940873430038904189094828859115130011962868694134 0.04861598719816122535902554346323120965167736918745592640863236234 0.6083076242759426577054448030870185959085295108295539990542204434 0.01645360700693184619995865670532765598759118856824310987023048563
            0.1769835041486189086582368961723547541748306923368226518773379179 0.1221421889557460233245761679884343335668857539895836964365115147 0.1093851510436900935374063608240452055562170090448112147326186809 0.03432193539279222218308962069754660292594480366596844492247649899
            0.03856883963667623457956455949749211873604369311422860712593017477 0.6737565304587229398533507549726891560073357639583085514696614079 0.2508341644461082942428653353200059613462435425046066257336831758 0.01374549767395276735449158457884372955420537670896230134418017293
            0.5711357877921358977919428820593886764377771493028742611664405017 0.1727855794196116835572803487534801182620238976969652379609718366 0.1030022318532128168401569541792942118374452842955103363007170419 0.02908706017163081066389355976297528530948523788066788403991511338
            0.2646596673044796581000995302912689388917798169867570089708844623 0.1884660494181322153004960512382570314535080975536635936938683673 0.4291259275953451170854052769397940619829458893302041536136865840 0.03992564485418243001554674783787094952818418486823742831709345998
            0.1571292828045067034342380326561293621365986035857802788302452370 0.5805461406075157055151051482172391223748861151129195422744015662 0.03423984485521735494693338135088887703182955324734863659805213655 0.02038066428508292926069455883636752032933513243987284838790470316
            0.6635380829388825716487263799358323415790547160199677131085159256 0.1217610687738568918133590186435371941417099156559516679123771577 0.1969586385674539180287678782086689843076729387840089231863379082 0.01179556636392477317526513466570892852511758076877206585944018259
            0.2136593684439528785952133698756719854771201321821406224215578918 0.03084866148752992478634149542604248920311130474465811392925616908 0.7171530557196868115322669674573305481114385993831721092473731403 0.008939677834312218959357957255155106264740383921255535180018075791
            0.1566773917997997889565246658034250735546476385316503680041130411 0.3175181736034333157708729908112255595742473834773467022807687709 0.0000002460050490821037194339259693047477773851504555555437673830267454 0.007425779116095979510762951907219484100000496702976852048634691955
            0.1079976673827427725378112422225633658244887652498443036885897311 0.4017631489702297494844959717018967536845889053902173459267854812 0.2986812946355224628388099116259798366242528711344647179054676400 0.03848509972215682057923298500232381982827442080494575876588842957
            0.4409806880459452236267383180996902363960083488831381299464639959 0.4970119774390469726657415144495140426673685313572218455132866505 0.03995060269348636512292627105088858790016241557225248749878825088 0.009736618016837455800258034548292754438756802697393176840260585123
            0.4254477514318533062169750252232534814006967328941470368301829261 0.3401421454318485362918264262810309767774390084588146345215657534 0.1980448837883506699466162038523158832698618699608053729427335395 0.02834155035146342201443705820210098851362088756894371879951264078
            0.6946928384569798843177532988373419598770125551625359126217684852 0.2319317735985255826981150566846586802217543551048258058322001344 0.03885534337916970193675919998032905559173854178058815601935208410 0.01236701950190766460296158060944513986904733286670797038779098586
            0.03565058105480223901966563966751151036753186868827994078280225654 0.4036293430763501766771868669069939645988233539683939077689020330 0.5294685040698826341574655519873291881039802117141863818717601927 0.01166945044103011340951147666570272557829303066040385758522611868
            0.03348740065692610359191259557279981060480247421610216728200354674 0.03618609344397923964262633245002421996153713195853216952573393706 0.4581708108037080771699202744333371135149525515845858181649883577 0.01347927190919886995527023732285295917030180833008445749400164203
            0.04544303666479976360919363751378094281888379134362789697283936558 0.03359144007653598648353632725740283240924355208742223942337980526 0.8803952330519741652121133104439524014668735541183941718689566419 0.006025666941674677523296014416841145358592517093212364272454299129
            0.001044818554827728425733381917849069699044722940311838024070532423 0.1916915089545268700443476086536648540497101909025550129801953717 0.5074181318482486435695056783979891293449841072997410220576879497 0.009839388884278935950167250811526493491720406691423537542316580617
            0.3163270208687076989470820110153224087692097565729943133850030312 0.1363678337579023917895366846357822066954318437897063118807695504 0.2413790612332155545938737066189062228370665960544235366379276518 0.04284388868816257530776564811047102275591124205157830204084381818
            0.3896405492708102859371319646855476258652510714008297301070480974 0.02777424541933767083455622858015419488718878682549418826104433078 0.1878688449107907593031995968280361217716544106559656585761603135 0.01911771906547980133908825473337060085812959270189686637671646886
            0.7406364104471735007380845772966009510520313356569943057597255686 0.02716950256485472390391065292445749619217695049718157955565177190 0.001008844862560039514563048750300805255189559982208981209436658139 0.003348545129506310653425284914326813986145008130939391495761781440
            0.1477794601830333331833513955398795761667050266406926879610209222 0.3339861060406962997449603037394500010552518753015889026150669717 0.1171280597097285495146566680690343624143793697902467853323132369 0.04106124111911068701844111067639418416508990958472821354494029985
            0.1886404904119914590542820646628579634279616876830983173750251514 0.03217919103674414241509626945672890393619375948255415693293674648 0.4831348472084820258077710676129626904197211376536286878288304897 0.02190399852484975363216360795678404584855646038365839423771769893
            0.7281357192410779733524806447150022695978992615463674797898963737 0.01129681308709260416563728696774358069172109155550030811712034078 0.2175637092104645779629898370471374971818867001876474396042644280 0.006504429286287903288630664925147279955653419328097237921373867610
            0.03240512776127729634052768493598481236818287291488305186023938815 0.4924708008762203051838987777608125657610975143887647686253816027 0.03811805759138896749238908093337784430840525130078752869090457714 0.01328194826275863299918399875967828308720941356344197567789748499
            0.4108404244356825409196577685980341899939567025726683998222091555 0.4161327723147283844450703239073226190755583010478016636667024027 0.03415513292230986957455402889283355619130641073400551206625267621 0.02233058537624501303818928632669666620159766672967625964014352194
            0.1878404308389233689466999174564699648375122452189384436534464076 0.5631872407093852922814332180957975550864258998989723875897886405 0.1376501472148007048981640615058725373495000380613265894132604031 0.02513806369813400909182496326127527159499664418457994697274215210
            0.1089137999932238079191557680388725433818006670765448665199438869 0.1703763474658899296799896709290883032230100238686768096023353710 0.3671064610617264447161175785589201926107813256491421446037398937 0.04381406595976971757152456438488729018466426618799855846742395828
            0.4539162997518049832535880773627410118486189728733387098232303452 0.03827044807273425912532079922093343639451885462298651949704817933 0.03761018324157428708441308062491957024369531972566441487667804748 0.01431835725222905505434833981702660131224528460322810245537791925
            0.03504901730403313162671913108285819142688716816797127030436728768 0.2229861824768844882308917954718944638573624261069467276303425153 0.04655071414693403697803127368049004212305030046974902063278184402 0.01444759398720480645768104268573112572694319561889572185579977484
            ];
    case 10 % 74 points
        XYZW=[0.7108211811635610048907787386093626814632870578524399622361704347 0.1680938151486082589088476101833401404113529231106889300091756512 0.09104326150600829553905136326127164560007196439835150647683333786 0.01309410011922259118449103500179950627736450758271008115623556116
            0.5790076670015544963571349745444308270869078875225384713696707868 0.07541334190963965196275176592007768205283515103550145807645959184 0.1931731991654421260728802408570236975518811258358618917411811098 0.02323642684425444504594106318087387887631270010127462838877812488
            0.09789429528083000780036450047258502848731107217626026855073842819 0.5193266649058177318974764179649440871981256947171351438181409479 0.3520237533382442663330838283299329878594837362734770066314026218 0.01588797011449340787775765064013843242639039923219275338130553292
            0.07117427797431348857105362099422381071141712865979799020324657240 0.2990859942693232188941070410440221796150457102440760116813938613 0.5748644395468989169390372523455235519773461542616105206170837683 0.01970291399252915081919642718445777410220347723899807591513058063
            0.01788074908929377745053693203587802783201556636655978842113008770 0.4111564572188602778855327678491878854591070112324946349651735083 0.02710034287581230544902982384750505614085990633215231780972146886 0.006087015511552062998784367189235251113473142219856021594261258068
            0.003714135933259649135272265549641893519755323539524159177460474632 0.4870271497277897483155443994529463389280219187539316423556143309 0.5086671624419364346483579357197554157841436258181459246408611199 0.001681051109304387418325590675207784814203530555268112542237349115
            0.1653019287626687380486844507644914813627059567768955668211118184 0.03221425864831313119444039531234294207113789873179238349811602432 0.1856060969044481244036797814988102264309268344584854768900783904 0.01857581757955914389930604067528134328399774715138853093231420612
            0.1548345938765296421626430717860054414865726211234825033590467592 0.1665886064354638243444945971418996896075564917142865436787657694 0.1008521679205936815308281600991028117889284550676066192160401156 0.02586594850472569810081794745217338803665150359102113668370804551
            0.5053653467106303479274185383938458314087081268740825198360955051 0.3052749158008759082276318680344486546427923121839982362904871036 0.003529745262325717063031933492443228105081730690454729944976193963 0.007429891826602586819604538287904190304620979356371856411954744303
            0.3497214515955615496760346260588544849419325613534073828135080951 0.06504062919753306787677317529580353844529543256214254475830008194 0.3704875867283166915173870490030421172605273633434806814011896414 0.02798081057813970450688687699469248446036216517117740895668873100
            0.3621294571184972829227254059626640251849505864984984016471876449 0.2230479437288030695318829187613275567932461492283787546471256709 0.2111222297561496661636489164320209667624623609922302720983652959 0.03977431060746941449892295649157551539004122928223154427449973493
            0.001519081155224517413881337263526846225289709178603186197804712915 0.4112670915387675025713904599037289420172341135104371374715617656 0.4585708068459480919837753408576349186827242535112347020104559560 0.007578214743998405479671369392790279600407467663126426489386860297
            0.3676710158884832666862240529471356033220649183235653066999431911 0.04423043096931951643455516404963012760232540552524307665270714256 0.5653369031608720392473778334077022688050648878817626520143127311 0.009342931028929043498649049253837371048361532436111814255603123750
            0.04970977205470890704774369089430833006315745522374551462589200569 0.8388588477211602453615991092078635203033676123710590041681255267 0.1076048046045897609644176647400988676756390996948248604752960334 0.003106664715382946116743011214406689973018689934790145876449906430
            0.1325393365313179790323415654670833976623362347374161193007808555 0.08519629211601484425655651838638611662643929242596791643308089345 0.6044123655429240881367084164889662560965007499098151091272309454 0.02229603925830669855241396419348333278074182502788414341063586628
            0.01679305508897423900958677513496391875949491369650863297610756944 0.1467379830061631921189897077641740630999877659630696207099099709 0.7198942577861557185229197026126273392155627202645717367173457615 0.009326327493327437132019938547279493326814014904537696667131840470
            0.3846449765100088885536218347428621600777403154309860913547727615 0.07783085442813045253086678545486594726373424076787932646197102941 0.1338208423184902402643288265287302326913334367233458034434718690 0.03398259450507288027372331771230456844953309780270739733617696340
            0.6107193475414290620807891699836531662261837649395216207376298300 0.03149305422937896134578760931216398212105245370414538843992586543 0.02564592398677684482372400726799629507872792864516864249797641537 0.007740525383061873357124691244329740813505559821407192349387860625
            0.08000980918092662711176981513296914719667770377202656927565272010 0.9082336711181266954642969503163067887705884577415937218734504770 0.01060698511643333084900447863991731972592765765576784046188564875 0.001022280798055693618722235811596397038427548639648229260087200233
            0.1722115445736072315007491378429883276940690311309153800540971906 0.004171136945129776637992493806936352369870419061540409553698297913 0.7881404292621583424686802536771287076736295879517277191852997488 0.003158471264025583677188308497684918793922297387460509050509835234
            0.06973532690213797493656064126423378495263932249408880319306955038 0.4653246578925895999088767008531587262671447427369542214191927934 0.2797431303256196034724596185913337215025266436873970702646801525 0.03051758446204997428632515161164628273266466013500823704099662223
            0.6113701604897331833028745354140124450586004251648849463757807593 0.1367777657504910824098692069887049793789691953399165481014760290 0.05114018902837437891592415706936858155131111159471234586114814372 0.02084215839725578403935222273850578675491949191420554932849736044
            0.2211161924187932305698738294389044404475173550363053292969937503 0.5356748817939830288604508713004581523463572568050783757172897877 0.01081182557261097838787730396179074376534965514041835289536714441 0.01079183423508955945850467880539587764072410627057662252793763134
            0.5754435811494772631812024240187675069516845840766452066336532426 0.3828233507637828580933477006164800564878963535556624351412378871 0.01776436757307739255590378073349023824993648460478341317878012412 0.005320875178278764727049401180672702993409334535337123778879069639
            0.1158392818921408344671135210165120016812640538278932885550187169 0.7727886692399550266539963438569941152155120390565970022721197987 0.02421950794091996882173999125091325725253604219508205835326832307 0.008695668525163580026442600928709101100236127163091855003704264495
            0.7827988815314628696247071643319345902302983327369215096523096100 0.1619283845747259307575443793724809374685637814149834059039417947 0.001617373723301612067366734051985710632221833428454209527303400018 0.003559817806856251274445219238957098552169510350439306601919771027
            0.4121634149061806457064051183003741693807233572227461279041436246 0.1202901254531915091238958466991348645409107740153452472988131270 0.009527433995123112759536603565685981336056075834728834032523544897 0.008659661354720185184384606749088169007621516601805009947180267493
            0.1814261795878509974282293896170620749566399795994284776008115889 0.1096295391696643749840240712041142595230514888923778329952743282 0.3301733234164556862477745176062423262201038270984170297980115401 0.03328840404303977651784425513043105909313604109087905841514414454
            0.2928680693443875462833559957792428692030205578795126564077157823 0.6537012944960302943670239228143164683483272016724092162793476660 0.02914797842474067296235130482062923354222130712531211989242530283 0.006562279626228476020999019110886880233702266548860702460905804535
            0.4907127714152918576613722491398576398418519156455366483087025541 0.3114372654169192432706184770112295980956908244155986238497159607 0.1006675664676695487483131957105923392260704799373245533629892608 0.02450835255948747867285818667324368650113936713127929923913923167
            0.02932156161395127232716172260901167306050257454863768243624988132 0.04140233291734257330144453702371652857797652278580130820901784570 0.1095147189589663411120917581383432448400024719402667129551764266 0.007421307978774415798289711390929174806373305452841777906082587522
            0.3276033530938000454673270619384086362113263325840971101462939179 0.00007414948872850782430576539021247151305238987651031654843472961590 0.2954875351026062779485458377170862420287593430712101793637233832 0.008386126724249986598355368552963147701641773377405438048233857644
            0.3600178613386471691654159132544898412451299639847377849194691997 0.5012978987010694663072132926742328388975109892015213448859839098 0.04325082268064638049329732153926294642254513804908103267920834245 0.01408084707618404938703848472141372020265931242234681957996851754
            0.7755311879383773430270937981768526091299012687316665918848211661 0.03014850186656025566419630564152602781105851486899087586380915749 0.1481879111466897465513547485573206637507235503218163772022391886 0.008082108840172627530822498556907776130790666976396381570133715051
            0.2126225190120122088303919172508420852637098930036575182827969232 0.4825350362748294354019858845763429775763967906028251929832331477 0.1303547520984165604741314756520779259057180994627770003766483029 0.03231995784686943300668018151935763986024061880221885738165290374
            0.2443494082536043011269385470348641736278595792598849970050758802 0.01031838292211476240647934763465076886820417265424268946247794441 0.5903498037553033131974899841913245870484881620089713646308953745 0.009072479627205161133785298956988894175455413263730397995518846720
            0.3016505415459932273377266653694185200561547290803001080161095410 0.1430011264098975329593676184296898571222384542730603122912796668 0.4807196239735930076717974568489557185651445404531171144605448864 0.02292069116841404941476580323579998247779775940740888141482402369
            0.02620512767679740550920151626554527218637287656561855641766453929 0.1956862242710102687400982319002794479857905985381057345595190743 0.09816137173569048778737344652889735599968385109589288262734534347 0.01026815196089628418247582058894405743234614172997978886274194666
            0.04549979875706120564555798232973958967211154590174973009928881483 0.1988251256663947569459979994838224403929667541006550042099840789 0.4768646629265832671007863686005559528848282059447223757428895049 0.02571874145840721534208855493462094422945036898001294915436030409
            0.03143834939358190434599295636867728666582519354854827364855023290 0.03593263178382785912646229875851160768833988375163345021035187906 0.8977091528373697584153306877760535579172056979495019433631778779 0.003928181298541984121303450842708093173291277577465921312367907050
            0.02912228872651709321696190864596566922659606730572946279967306027 0.0005751516278394052786570188775245254820668253401198529771113826762 0.3188433055232422038531872610701494087045359660805909906368337873 0.003385704666090109158323220436376755652651364703205259885430968366
            0.2272222363946760462916301970717351834841380417909534595369421094 0.09020567255534082578104707037803688449334290088827528087362206415 0.02516122404142109618905055069081763432431946038131873522550933515 0.009885959023549995078768786544585491403119643805192473170317551936
            0.1773693932713253145188377281677045099270130548164876774612098672 0.2659421383325723369036143777738110183925941905701906537316634208 0.4053675049542561021390756081243557387571939452638377727352637988 0.03612749723951521034014764138366654493830294702012048276176423390
            0.6431484774270714162818404844479581606476886595283773150493257596 0.04282712606419561963883012869517879188866138789524161480270190370 0.3113671388220656732305970911755087972366150346187277364526172329 0.003783824796959272900701384668342004870203287851377230676013386351
            0.0002651387752094785789482438315429249109373470858282691344024455543 0.8943497560407246800853724556068632032046289923180418023405180308 0.03421723509518098966557759093140156553738240854283482332474193811 0.002228316340458555860161791729878416346266411998174679363657190926
            0.03436114786887454887988546594748726562237455953913118217904143418 0.6780058688832242725957075357391129099377125496089713062912615967 0.01514255136958273572205294783421610543862431375659449005530961166 0.006033323404683552695493527099885912395405080981673397845552642822
            0.4999879882641083497689310302232335155771817189382129424075769607 0.1742036398751961850827836525085048490323883211461041867847057303 0.2886008529574142066370523299422091267548750978074497884498649506 0.02321485984779925893985366301036299703316514619727143436187084033
            0.4444501765779667019726812946765990003858948134659725890404418167 0.4135053284871942811267394493320920387760980134460305595050983983 0.1419288711528731365671429580830130122472198182456738984573719305 0.007223921239921966824918849911292030947699184906385968963808871556
            0.09433857024181371489718188164950414334872224930817176433446091345 0.4872400695815088652837806902603627215011951239201840251474377341 0.07081379415052846074435494591673337239530296262620694183131182275 0.02167621981184194755337989187473109724007181850366282843645568123
            0.6011876812681009564267122106145483640446308841677109023072696142 0.0001683512016895687781052579611373274109256251012366616739736740623 0.1454413486084074501375823189028354653283215364408824652009009734 0.006521656161728592067097156914734256415492740486814123261916421785
            0.3249680995582529890844836549620813690485094407459109372572640411 0.2796271780202718883096770956502460010479835271509575678185561897 0.05572815291086467272437159721353356992975423670332826040312992210 0.03231935273860977553647466851820281938249630719612847761048580647
            0.1089238586456045864910775970128567955255948611173522216784451146 0.02316734606797215314206366217258701533847276595616860620350350717 0.4749489083627679835095007681246822369413666157161449711121154779 0.01443014739238000354363202967340773539631693624417138662202973683
            0.04537221677128411596688307400399527542999819089714476650964996967 0.1577014452371543104085766893164863270645970153032232454428203375 0.01340622550369934549398406407139509487809184296645869920287692414 0.005421548340346177339242589038691525694993702482311068759861155153
            0.02062784031489133902374788443058836740068973371365755072691573029 0.3071952605835924977836656534194110950666771369610105786052960590 0.2815097419109018484843873276855831099760996539239282068336646419 0.01354578978085626407614256756883564848509983381059321200613340225
            0.04166180450251433551451847312524864327095139207159942507217591714 0.3402925074117468804897049083626619404577885084899609681934412621 0.1419441168969618882642046294149346549853024653018550606900204449 0.01313807260634024565250623673458668483347559414842328114732717042
            0.5232246056946657511119846422051620030215195398673066411351548216 0.01719398350680940676782694483893913548605293865712096864694649401 0.3747867167288931105234432020767907066020973863833598689879509493 0.01028808323077808482245963960631960666693474169173718767753069018
            0.9333826965414616437584432597540820094590754172977209156018893306 0.03556652000863333284371234578656120736492667519761485118293304316 0.03071780497641933729347964919814110497770971063607100321246157878 0.001331733048713957456733301393575097845891555708303937886785738281
            0.03687945316825942414755726938570129753134119673924726533190771215 0.1068733463428279450335124681600270812378163158832433023206390603 0.2878274336814129436496156115640682710471661716706935646443718069 0.02007334955553700060796246770143968763418460426914080010259222821
            0.2390695873803809891539601409836727748638167686095300815215926371 0.2707750397116681464325928520694265577247405886026241826084129480 0.4900314794039396097706372230541548648617272310160272058500235118 0.008491714999674873150455137304415986953973774126962806130207726528
            0.009877454016776290207637402871878873766325164585613014787361773202 0.05159428160513717525629555462217522830778803305891756424840142833 0.5576518513346611325586749959199392082412096453759954671157469929 0.007028574995616469192538727994578739240941328502385041911360210766
            0.1468723669075991388290127659517392096918934333971096827408834187 0.3163196386441845539529070258123488474513037868249583533025172208 0.01409646558374430906858621509904449717428691000397546340782622923 0.01182998537284047778384635781461134500406604856516307700181188712
            0.8260093606523856299815408419435394854082320231639476578497096342 0.03061674957912464368631075231785162967559583948966514144048394715 0.03146370360263207016086236834906868110701833962755143907791438803 0.006514504411571351266102476961542801219657853453787178311863335937
            0.1380115817340293654428759534636634605624381815252417378042249504 0.1031336966538923851815520735133194419715382950941333342482718403 0.7246848432623155669711160816206424915136854774599984528349532677 0.01319895510304183555024615669805038979473320049331551872135911676
            0.2852351403960840420859809323829352188715493647264090868592368282 0.3927043788424667583262111857770940523056274397962051751688431643 0.2677783805703437262527167014997777778822029700172200837658665772 0.02806026462787013481759335460157361614979553026440303686068230374
            0.01948060332263198582015800498511692538386244166519052213823246064 0.6891648157458625307683187152075520760498588424557327355114680595 0.2406622508912331254884355844099381500129442281790339467359368571 0.007962236640442574249958724817039422255821957575876606578441714290
            0.1832473550535123868541572615408534316926491604802884277992329434 0.6410981463475119956110877429489976026845679806647903570494050013 0.1440710813942678423956915753632490137024647684983547225882157641 0.01485918160369193292914041698493308180432689123765610784858053239
            0.06054935118439677885877235246418889881467937035896948580846250702 0.7035992796137526950299053548652353043674217724891575343707717891 0.1064038967839555490536752502760149158037913386096566798081680899 0.01498960576932980041262570872829802918061930146600882940535945679
            0.0005913322473979361808517606312601242302119312190951270660885877178 0.5834193040821614335233173122900229363722914231615364607937396033 0.1419049463214869209565537148762376729209386583851411195944300446 0.006840660252544596595928344136978530002393471788914657762058134955
            0.01973822022058031458412591315549245252379765063426998907265084814 0.1983391941632586244477041125700103834422682142685818728489509212 0.7814887334628181388639187208745868177470412841275257028063638856 0.002166195430713587378795740182235014601642549157582760724298748664
            0.002021795121763506710144483213453321766940617084217928864742514575 0.02656068967740498700947827848417343268796237519915805084396928311 0.0002569174707227898888427748999601522473181436288759108203892691376 0.0004963334735625753645461228002763546323902138445988204944566379868
            0.1583679664182448762395640540406282325103453944311535946396743630 0.2740685756121609292101771933517220222937352990194467802373855161 0.2078803717179151919946260024929504954627109542359313996692217508 0.03644792147399488786024527517906004577874222624361893760939455763
            0.1164605173320014452424270761661283008364428730683154578938937611 0.02553741647329541149623184596774417953778399555525221853186282739 0.03354563172935472932049911359215042282013582102517126673763847563 0.005589227059271511908386571562929471233971160299208318667220492530
            0.04143329941798143064633972279542124900095007277236678866532255231 0.02055594062842863254836601212333924575382097332665954961714054360 0.7571094133732397067241018038123226330828463028086558062289920848 0.006940849367264468062944594401589747819298216989403664102910819918
            0.3572328788470244855750315052118116481488544076440767648014148636 0.006146477733479475827571060431927264378532950842507729271910078563 0.04657792557005045562149053403406249400535471712562002786111186250 0.006138894046560761422545980913750695435732899154973724827788504314
            ];
    case 11 % 94 points
        XYZW=[0.7711844778232402089131167963293973218562001991582469073450284311 0.03014947663577570913333355045890143507502485298933237308407539468 0.1740796203170159956289977557876166054665997730473551375754992070 0.005550636989257514341880936648225760351924480064326615189811110071
            0.5394395795547526087888845237557289772779670012493991158076567896 0.4044528775520811777960391106909819686378187140197579351871374244 0.01887451280506266908275355289703316395734357873393347204246411067 0.006147753214446438768490126833886794513633878746004918050157576898
            0.6457575204652854517430465513989876755273515048943400415757316777 0.002468748755846532657745223932759074102111352801448305271471219961 0.2083885256804204609287453810540953821663106039957412861131990663 0.005159095574993549908080693069037324523947743551078872484607057593
            0.03520686725091060582881964517514072276585758668630554898774344866 0.8146746953574893358876858462720267322438341713794664912528146461 0.02498033407592563481513770196511351153850769727526773756540593171 0.003962322477394990317114735984526397216574307206988701248703560900
            0.2021436675206495221991017739847728049575177210249310569893180008 0.02298055071534334684980789625189242834806662604786714528015418160 0.4569966367023357493697393302932969378245009044764015656441665507 0.01224160891071207419503458099970840301944043356167686916388132963
            0.02870946292682618297690463951821979547183391154477776576156349607 0.1655957107683522663805926168775148415659349761539697405599232855 0.8032733154805311276841932745944437758476765400663124719664915690 0.002084868434742420049665294716451498973045122541270355865774048397
            0.1419317987488828896909679632734381252842838160753819943444530253 0.03747437813374934257197651831611766340878901909614171348854120674 0.2623940600277828229867296610253611183698559557435630044416159863 0.01501178719923951013505161768288680027223018244248012678779630167
            0.2701223887431900102594489705314139830288152904537378311050514396 0.2732714193234510269633680548173556888595382370632011249817371458 0.2689257506935214260149542389947835739487029573405683635635638317 0.03115596639294223554922563849965833391159449411994718320842080529
            0.2510183910688732451608238355827184584171653778989635987816129371 0.07209439351005499289194519856145853695303778139705423503184595160 0.5326787755410077635421690601943593780966704896846613806952853265 0.01218545609694984672548417594612800749913440347190236395270280830
            0.008015049556647654378154130639620985605784657450743764440591737110 0.4517353131725011968848110155597975347813597994494571473175345611 0.2349483897045958529965141121340340521923558782481792475591896506 0.008082313515216951192503246472527171158747523975951046212177541688
            0.1256112322146619049228454868733374563652430529878122394105924402 0.7008259627499786132895417230774040793923301667868040688116392603 0.1365624266086917007133921734708898040985853859759826070206514105 0.01052463608961385158056841352382208383912055792041771047686750964
            0.3082991524611926477799885231433942601359866368774967501610125312 0.009474230997113106777818460899936754715523476641577711761737947325 0.2539019719671364470203128565919237242783622760376480511386623743 0.006993275844091770716292971849416544237623220472351423024440101093
            0.2894505362944093547132357067079105346728745525272469912681471276 0.1012872995794822845282599027035212809858551929938904683256331295 0.1680947349248090825940158947292238978134285879190727143496595484 0.02141884292013995786601061110370877143807426792588751554499179972
            0.4831492503929166856202713409413898224085532586242814530047990424 0.2792961278398116413299556165073268731130621043140727196302139219 0.1578304696242923342360469203762197582804218804601390915539242588 0.01984850887526023428272125541944722830453681221498348657735839525
            0.09392678151394223146652508602935770575081826834868920642376621277 0.2949752294653353454801933699831774399118601580409642854585142113 0.3553310294059755305329437676726544471333698105632513773812708229 0.02649222365730424218783900435637214571066097313735113270915020296
            0.2808816621594445584783806615932996101835043265715896545733159772 0.02351836034933950467706805365899933391925002177630277745186838165 0.09478877782578331527272192633542063304722137901429623522035179989 0.009222775411517665207348570961244621795909211149390626169704045803
            0.02343976120314664402863697860417742122781276571727183447153660442 0.03453508440335095888001872254146711065534608448843132140359998142 0.9161341706173096087332154943246389067359205098067138144448405120 0.001993361086264612301895626134850680398287403465683272126382127374
            0.5556450051969628780406728774139014972667497438886035193926191477 0.1636507721311736114589672734770932134960824988866688001819322631 0.2696041385038325102240796185115395507278893344151369011854581096 0.008170583812742500421884804157306852037473272649665118787021004139
            0.1773229935078376341166824784095084567290982879127787173529433335 0.4737847516190886242302283987952598145267811746591305400087350106 0.2264221203250912363022128793794788590423609188387838469186445596 0.02190035438658174717641484004406923096177857414330073034743177178
            0.1441759114924441861696459206881486284447395079758966747666485661 0.5220675824440285746446274275553420214640583315489420305743439988 0.02257229572275167530822699321116067694249553733496845177331836535 0.01197185865202230626948737986546152657029583641463056618163340063
            0.1152111775498098873297134299380863390264227947376924222415266073 0.1090127466489408759852645207976467946541426774293523521650098061 0.09872517909372756227741880887702277140616079301113628931950271636 0.01463135482378870341974550247910264981778654616169925065759335831
            0.4733178686519731818183475000870156361563487996906642862257507596 0.1517140158477598901806284746117010405462308643302150046122489113 0.1938136477537488334029109883641248196998045759106940420120199144 0.02171325171451392777353253649090273280123566548315397451603750990
            0.5122225271882200785089054582572471234359361967318609813928672012 0.02871956642755280328547075173508552619770979702505081037020679738 0.1135916422159246605957460493855065447216759538422971587135468509 0.01299378131808617404428305412231659280060504452064629896616017699
            0.09620467794522470723898091025074164570981129737649930393853186044 0.5698867145402618178669208025391540089950017528519692819473029586 0.3273915096219961072359505584153328968858010221851674399345950444 0.005524217950502176461001816456728117701231960859251868115893767798
            0.3076571603915160083530084669061299955391016900385836912501428899 0.02481655300669888071948150249538494338758262402035831044593396305 0.6355817032442831744058802858816346112899683804374952855921895774 0.006452899539006407324094976149873183656474385407715379757393520567
            0.08248094888860368330279454122751893561955300398827534676484252682 0.1103757607585965185763483018119259393591978803153364857826353753 0.3776039968142007832809485133640668206476848253284262185765880641 0.01877556868844735363932111104937177604876648750963755513826470863
            0.02231261617827788098926257064482967427939462185010274497127089232 0.3396375417746841399622781429566811692289228420253097615421238611 0.4751488623470270002790527924088299287094196342544641265571201492 0.01259868499819532026530517680817146017851432713440890461834151775
            0.1271943387335868010293949963153908537686511032583124805036129502 0.01709761747062155279718571477094336255015695290410379850808939537 0.8343069386606727964129922612756248764428489472180401241172956062 0.002535822373515776986068792917734755056099940046861082796527825475
            0.02358020972399388444174321856045378670994886090884324009161683569 0.2399541315955699250305759310765920475945218582544360626920947305 0.2936869631859191857072371488301912089357346707773780654223041611 0.01314888157401206764265742431513476014064258244618992201930232439
            0.1261328774854234993240979877654625898057273363205273490297165132 0.8277297969982348788355375683937794393274294218179245422330506889 0.03036627565244715303771555992087175336752313255417889771666285109 0.003092139090153155795668524997636442992776079094700384104550490463
            0.5433860470400686352424613881639325144827856290186171108673372740 0.07901872216423140918802952019200178724796852877865001198110609230 0.2889653882919527549088481863473551239583539029863156571595478780 0.01578256192003822098985928041380191820981971175551861232850371945
            0.3330983043307960739597017522707304748360950802627094940782770860 0.1270227445628321997278601004516118073301347588747313702359052498 0.4515957208853387918526248938730121724040266218326554991124397925 0.01776763673406845063323459428474639729739471334626635616826859442
            0.1428682273509877729140259455869622134478940753193157442527224872 0.02117435230394200724268904389765697281262430570578071970721616465 0.6897635213808653575498550754465617329014109061594445867378430085 0.009093208915888955324990455690514400560701946417560684483828181550
            0.01429367810244483641264374714257394517194644816575240812199777092 0.1637458793061360743242180257148358568670590595124049776900244486 0.03950025149360151157136589616843853370923150651970837319102295586 0.004315066142665269268461083430474670291628181340237839733266572526
            0.1682572149812816856933966934635593811767639297687370054053959217 0.5975401842305000279568762943858495298159647578563253792610580458 0.06970598016949057182922800917911184725506307856647310326323083860 0.008994064821018319167992924764084560125899864569842824870542911123
            0.1371966830657232314140399597119515285960221610475905473222471209 0.1085011480827594865501509739655744450173964291057347771148801711 0.7164719127569050512543906731829796325389188354210828973337077611 0.01179474361039443990488017632711192529930460491420179101017015014
            0.3434898926369193682309335982278224464930773363536337436657656807 0.4790587248716928598320403869460910707062462750794248458796722424 0.01725565007984547905345612940323648305258014233615662062618589794 0.01017451273496762246929331010941375837326655453610095717876337198
            0.3066913090673156000494274193847745544140514189588538761757175332 0.3822890858795550738314260512296334802352215072114853689385179035 0.2916660311203054850659259657222859572320735024368987048359035421 0.01030598923040418529713574573957255382311675260071365254285789609
            0.02880725046415603509371055237725974315187919837949471174337932961 0.3833122732338565275569242950472075556521302176369388122266877700 0.003557852602795268959135671887772093785732768526467536587581799626 0.003171680957597384748669201149338692813155005056812942060397173909
            0.2734461453976160841681336559183508190586589679203824949095022583 0.5513270326941551718398012373605391128035559203328347969691690464 0.1543243378159333760286500015637860602322388278605539173780058676 0.009053841925534956955451669838408132963227506367672371939608817011
            0.02600157795962618318094249181723357592316117797637018997576775112 0.5363659992766203962178005396587666005182954539203588638007116848 0.05810135650993010263694065550847110859015872379456570664528166221 0.009445082278709696944496636395118599202186621750115334853121760956
            0.03131045113380852790182245065681168018947142065785109192630819034 0.02850197944244143330325397272545538568651696718408266871104243595 0.8236488080856889702875520714245711656681276952754681562826303429 0.004031861453377401629365345385940567203819001785447929715475993202
            0.3550619345456816107196595475255506257958261801699284851628379383 0.2917074286572396583662900533527215610958330075838563817801880619 0.007813939611909787062963018361415410211670414191388900507047642994 0.008585227019103841011437568906471007289392358265057333200443615486
            0.6916054692669160122524926286045591988752504914889799392570924529 0.06819770169196687911186973146004022770093198118941195244721413881 0.08990641344212710859460061767388347802812088096657450519736359937 0.01534590600280093562745684003931040509083581911275130807739973918
            0.009405227228336286660385772485439483165679745951342958266414242383 0.04574695501737517035083122657925187652827274490860174592061709056 0.4140993090813226819891592142284965341525818151537480363429589228 0.004873442114530441237029600348361562699598477909656334775168688132
            0.02325513245390474891031068930576424873407084029436173126254987320 0.9150198280595015268437476637874703436181803327583797507112649253 0.03280694574517720318441480377108031786360625315998361560751794775 0.002139005846149799267807439138070031678591880034168810542488619209
            0.7024402649615887007601925695862371239134996396770986579924503955 0.009157891497218531872209559896681889941808054265456230358881044922 0.02311123403957317466441417653426424440857855864586702708828924593 0.002967915751666871867803683185060960109662114203056752202628553995
            0.7879831247050387469466224098675459493022389814390607759471408919 0.08976320241786773892745347698320346340246903735537644955102783761 0.002915510329983943380822271489251762290938007458499396563581316342 0.003518360883127706738984595844178926827200655386517259554247005333
            0.03505306296069043317157436399460448732567242179234036292193279135 0.03289341384411806188696053983308102077507645522495073231794910255 0.02195972794051755802201748590366685808259569350121253749580047281 0.002555050102184611162124307200258504768773154715662260576218282770
            0.4345305394862628451678261315626827968928522658157198438851370339 0.02529261520957148842296208572796435901919471065733895079980574112 0.02106505335686653185427002364317424317999576107519399198622824986 0.005251457700557674837070232347078242483597957719259041726025709665
            0.1468760245223108931111485948116706183507757404111920853493686795 0.3129930001468372293924200597958654149659636532064543383811335616 0.03632503173710676813427885931411755459826135206219555434370080633 0.01676017738953891119659334263103029233615904121225592203673110146
            0.9076548181283545032614806438897140296604884811567154031405034818 0.04383661807847673041979372425219617863052985160017972779288171983 0.03143250437664333378165635379625498328691110970202990838438375312 0.002324129461087942195889480647743631849696598275876390172201562160
            0.1149226026773446696038843555307107525727252767862320105057808768 0.01737256159552440950873941808485740583602126143530480022545786822 0.08819333505663815160391925701647907626147954344824942223894975007 0.005132517445510792445665226224773296687967555565328758161264939899
            0.1619426373062327408154220158303768722544469666305189060030333246 0.7201165760933549886682149628805971043325931045413619812007793180 0.01592658097672330833770148681304255786737332784018702715607553376 0.005904602908289963657946955156602326218283045384152825302454368577
            0.02523303638072876413870530041609228582848970000793104188506129974 0.1551497086465964021051265094377981835784296215520452039105578279 0.7242739744250221782240554430683892501072820862520240547235903537 0.009639573859123944356621784355991230793242217016972149721927245495
            0.5200529885440474128083867193797649465549796038547815902173165097 0.3636659889353487396496850827343205636366638263851027892972621433 0.1079611671376980229583902417742883466629572860242899581322720456 0.006352807885546679903434158750107641168955283679129224764004489953
            0.3485982841079376577939723977811040851610455668685691933003173344 0.1214672512056179641955448177194641856760907146984732673540300019 0.5234435660609304156401942146985125685193251676913500246655911479 0.005831270753073525413629303314627665478411243334216519809449907163
            0.8677294760242432668471466703283862316174366993194449333396580512 0.006252178054674931616065479330196456868826736202489062395138780534 0.04240948342757449191569080879154290368792624896431581429595896482 0.002143787009537775983407418453754742822446291005300626770288315401
            0.5537603813843986937532057502744425555423216061628810866611848832 0.1027764784202967731607344651437751499020317193751150771999827259 0.01953729165776202757941486655448701825752614882063335142442550365 0.009716562570060612544944728458932817134545231450087394421961498085
            0.1335828513618848190830550491627973126300005076721294334549097840 0.4249040259083643723775480342665087120882753953143776520595978846 0.3836085578463221734414528035524966070278463976481177176307224188 0.01523298730548854380877854408117598133540453659958928419226724196
            0.1054315111602953591581012581127514329643960483420669451103793948 0.6435348221542448359794088415376436152470697431159934922182431814 0.09870677476314302160290535134840885288105678991813839178602393258 0.009002001244582276130882322199841199336425461100250463745893304350
            0.02811728413371734356226237688091713522378406429900038080690351678 0.1108904210213228581398864405625308871023439091340866396262963907 0.2020610048684909300277386758812333121921607333660279351762651430 0.01086964687827923567148755062512123459992570644191844983481474624
            0.3869479248663309853037530163057107628167510875380126965127071120 0.01090916521886956572633981959112087808000768687830446512342486252 0.4618265112109715493019416209642784388094777781949375636732081851 0.007634593126404574534036082234176584920521526268277525305472481575
            0.1452594763567463043520525392108162546572748532010287554668913500 0.2349113614306094458380936191282037362953336032193938109394601373 0.5174259562858567670495402043212394268024555762128255789716326889 0.02250889240431539883206570575684115725132442201298920622879971034
            0.2395216689990708269016847105894824674169188816894974118525538419 0.1283549647759243201066746475661790758393650024349429735293070965 0.3463406420655392617539805944946946689497859486737903259875849175 0.02817454192589774939741613159768549011838438209774114305775432250
            0.2877907652677581875803305752409805361945155433156927676007159746 0.1367717521130778528194161990863472840847127031431168282661435464 0.03257507636713551708758610424554541139957078081660797174111594980 0.01553670649414102447311819221111680384186522446914464176493877265
            0.4222383818394575194775160242066146096880997982405531281829209623 0.04291723766667762184220439758944815487489075366980244717896485910 0.2849739419837088642932067359141540782692990085347965961276728207 0.01699234752848081984901047880894287760000031655850893645100347531
            0.7507722938938017155393138549785051834903312019781186333547587653 0.1995529007090345944222173051195704663367377210271664514003074043 0.02391264307376579244360190678672195832727098700688446318720612300 0.004510819570887507693497898526809844465699106480852185706404565805
            0.02429876776716154752771628780296816584542520904151159691862396945 0.7812922981934326580559953935377149150797677388920751863202939241 0.1643645210076598027494977524158147346768733132449665057910012580 0.005114557411551538706382020698821520926632106864203379956543286166
            0.3189097509011578124597552714787367029457360052670617033968309860 0.6237966775778837566684484880517257270031279612270921523729600445 0.03096943391304691327718300648127957973611964481979733751170902053 0.006855228583780265907860302758622733084594245803848294040646107476
            0.3383914130243770896231054551445662156438277354777685482159233113 0.4540500317550128385858270629448925705411344236975421719097167213 0.1032853795361590457797893777359521013383973269505526184152403767 0.01882433357209000829474322423619244425319638150415086668603177026
            0.4193930236460097562440573326946029356505668008376511334731733520 0.1678534563045752019724369231647127582863789518931526114817752227 0.09582658490179263297174254317286966736305932401446275235596312039 0.02016122993299703112495808323588796296174986709835041655234203592
            0.3431256019434479283682952436103737330849109043182652977828857273 0.2552381855213296999877328646021352736475183124227125379658555195 0.3507460874062558964841189836810769845106617403826312895470042190 0.01560586222591324497022203236325713718516095193979701617376356500
            0.009312024647538243369826863870214518126032991794926764981045908948 0.5757200057987701442178327552132685361569099454002999830671315977 0.3868686228740342810734667157005404092143672235624558224830757185 0.003366968831028588145257242398960535750922736034711755513911563767
            0.7105512292030012731577730165715388997970624409857334752939930424 0.1443495491508345781803775308588312308502903305593573303299602323 0.1077423251954225033606559740651202919192277394964292763967703560 0.01018929629412850263829125002123378987466396905366534643201680653
            0.03900792081747334432923813155276908868264842834927360789615259635 0.5568716912871344172531505911062481200083862213936721432810278670 0.2846497104868330095734405853216087117235423968330072600902371706 0.01530760210225528324436739161370034984982548317429929372879788757
            0.03441336638670823107659475565402790707185598277772938201513409389 0.3490653137444714060340777064140682911178512691362840913788586484 0.5837848319292742903281979625721552983907178022184213904931172473 0.008954105230737795355543356272685205152428013541800383578401918789
            0.03048293610357537791024253098572568004081010956716575126209063974 0.6893545798573442282419573746140386459281999083369784678491192323 0.005360741363266648993419229756955980290521192373643190372626730332 0.002416023117016776169514435900648726151017451802983858783264897432
            0.1493590787712792884721195092935641765987729158290212069215966063 0.2203449174658770181790611637863915575834722828241548975652931055 0.1785493595948735009607441845208704670548576249489738274029797438 0.02930962263686634334368145580565981517797674733837784367376442759
            0.03071642646880893700997934471221877456872029559639003419474316517 0.3039798426492484365525667099916208519334148420759502887191448207 0.1041008147288003957585803368166405734682349005099019711774140886 0.01395420979625161007500684647680062281277232479351760390785489408
            0.008511440408983542995050164946299476539665130723166483237214840063 0.1587110672393317930275617576052476729011930174522330570899215995 0.5309545427077874878258891705608235854026878673031881429056808685 0.007247126660799781883442509481142161030934089061042275337303657853
            0.06539962098351801939780831774667514525549639856058051864568203731 0.01449788853300232027182975996522728275567918439558698188110855941 0.5222269587387865771358549585033891207528076286893189206376735720 0.006234291478851291726204330945206109849802977217395803469356304566
            0.01982223338548014713534883400808298347097815823089778520004341180 0.03193260304594063039244032117483847992112781373898801249800596269 0.6806584142813803357936742200582498935634877937339055419269086442 0.004793693342196706519060028623595206208007521878019676662574539949
            0.2798652924603839726937283661590154789237430851063627412407535457 0.3373176094562046030056229953361666121428636464089318896423452576 0.09749534001611287116845170423513510646348846610937406589338698486 0.02900539087014204385413712207611011920182918495012093887940673547
            0.5497643401108277058062958801473279626691782744893745107584127477 0.02592941071381694682620227346738331348394500591318242331737856126 0.3988783705696772863139722305727953357754027778587401707393021145 0.006338978068872099580738465256369675032343203731419849823533706246
            0.5620256156092019078417259420438073709208482175259322817912948912 0.2494368180621700085677968251684367622525738238177885370211964999 0.03602837970946916636745118652339133299977045385803614951729190562 0.01765385282596371054677406655421737943656042932499515367038423454
            0.08921697996561641857633695653565914586546919857384081683883628508 0.1146384014370199112957583093734150196167048021699411840550672431 0.5721825540592999741153171059054077515599729135079733037880031749 0.01910695497814381215009374107454138583696942230855911293188435282
            0.1955494300661871393968465563468120051308350880864307056760728470 0.03254578801262212428549238770650175597111191847331679231868084580 0.007923794324134563559034099305925430941370354987767595071983744437 0.003301514457098113084032465715950887555460351866999143184501786714
            0.01416451992084521529298082147619049431066365609550890837757524592 0.03137041733266367912640680048734382064438733808760046368817771205 0.1233052889040826028136653294144772047397787214068579182435327720 0.002812761170989695415232791928191399517765210793811637728869871727
            0.02001946413880567299279587052657581150744447969928131986274697262 0.7001218163520293386598732730014148879051558780684527931759091251 0.1116533130144697559804094273487378041001020044843651221698215569 0.007858633577998087864197418418841544348292577634475781430050048701
            0.04383203373620403733320376148656042860722320270309571195415679808 0.004485252771218634152239824982200871351844268819722285905253149980 0.2681977199959751894050417973552585430585628274285568944588298669 0.003185024632215935141960043106983364810527244169707947991012096677
            0.1715297304385654691263517855082133499808058141345507103609778936 0.2718078178716385609692895672048277991550729603208449107320097149 0.5507613752642044346514483329127734159663359220098613449033449342 0.006548830813398040562411752692806540144230673394340585508354153924
            0.09789804517008209600077272326530325631499325788237375906696003451 0.1570521893868311466555774334537946151786333555861656596886718974 0.01526643147156637597996888967310770887983308099338396611844750470 0.005784493645876048434802736422402313881096744266675653129560949184
            0.09520374977005055160353831892213540574156926775398503598159516471 0.4410265784501197379263964575999483465364433465673985991629119081 0.1605655557330306740100831542295392970961899022106315988797491279 0.02505000022614809954444145129287178696664430187396508143973329669
            ];
    case 12 % 117 points
        XYZW=[0.007573319150277204412016551184167787486225723474386887773285023363 0.008102704337199682031348184302642883691234784168503894220404713561 0.8178475633492317341543742330353775157237613903246388193456057368 0.0009653614884470391513474125027723414792601351562222352409655816519
            0.3356390339050894186284526202240607842509778431567088277319403868 0.4018660308052447675228088855629864750121181892354442187094547211 0.1008052518090979717442039492837965191052811910180786474053125217 0.02101288737329522210782075134042420040038444130861236708736393592
            0.5046879605230297884303977046501337897438712306843936447872931717 0.2717916485821852844544871504445111446946724463654864357233855618 0.02661595661711024016249003262240259383217871119437276475461099227 0.01306435062250552752357176740784513866632122373206514579921625072
            0.2343486003779260965375082072202382377292447175177754096269366800 0.1191720672739488566483740405310783371598296052935746927734549397 0.08748438657091772685233735968619295806079597312407229332713752199 0.01503934777414299766102410446288149487985352393662749095104873962
            0.4867802030219838504804138808708431529555683649174420414301154400 0.1410023206082576900559717445576065530787653031040055490976471693 0.0004187664752551392197544654827513538275514693484237894876204644630 0.002640935693473757361722477822812388158451647708485146549105899476
            0.4392880948947025500676195969285921909287112326193386069867905782 0.1518941689092816914268848768358382039319272751126717010953443852 0.4085674360628064241514902272032102456868791652065188390642733394 0.003789736951139808724089699532307567058930129980554437293625165835
            0.9131465045672992302935313188340556760862075283946807230939801370 0.04135391003311856048914063549543523311651563716447264333309159525 0.0006325970093631133805954270075992189001430532702012121869627655641 0.001053695411479178230411242212974346925900945666258336957315467729
            0.4463100620422013219330101839878227576680812069625862554794246175 0.07081893785530934868938118577701526087645840128823205017482484144 0.03570638239050877483422514182172353173648722637175386930864427245 0.008509267419625564515252345962340691787021938412914875405279284696
            0.01947308412420471354906069126171258033598774790822724644546978517 0.8391776222800945083414717864149599623533678175500508632934632110 0.0002288590063723560042403498932154795107027549039624805687019144439 0.001010928663208124344767430186944723053709311264824318602784031725
            0.6046176966995361159229207400081266817906138374314477220644244966 0.01194286090614513102193143739205154815451516772320744302116545126 0.02161534604783446961244503176206777982180939103452895875933294021 0.002996905444674039018764565593123837893314134553825901939531963236
            0.1037512321464330872667334870838798732533827425757379947665185425 0.2518124336201633824444523448700307560542707874708547327668340879 0.09735506751090281012745892120766736256581021641211251718267481919 0.01659034509740838791206823785536948109003981238016431634115208482
            0.3971636905794573395570622222093016171730091469697235789612622005 0.009721853696853559522463470862365491193800646050338051491572979077 0.2825693088792311022987838216180213395464758355536447186109101631 0.007506075728462502483047853369924699537133016064400250343026826117
            0.02152972248795681366604819472615188348553490948756267317051441467 0.4478165274633722642369181026443586624206445705535763454239244018 0.08989375911387566418383343212479047228633935186407742022638651567 0.009532636118114756386745085199304040844597209365968031224592749994
            0.06269074667619766256700643653494925115775197128692372922013773226 0.07755222637175055755823851255276857250591862131998127291336218004 0.07131185646671626545590798879136715334000998993931974271935706875 0.006168636754972898243928940156980729113531220928080140131601183112
            0.01731794863553578106559391594267606772248876825048744447991491871 0.1541951808534945699038455917855643243017883735240572475179365329 0.4504665367679892401206259931058373864115274087954156655442949863 0.008697472735997106803862858088543255770716074775586734112370460322
            0.02186373873167973033687808968307063434371247781547406372627592435 0.5768445299230294157447282302621960191208049902858043945124852015 0.0006913977017133931377299626352407372637477251277498654921364872120 0.001830372084735771481186483138072669463491431114278805247972382385
            0.3202358608656336595108571909344459000316539632723089058170378245 0.6645567488633414065336327254968095542285868640871475199041289310 0.01288307116622971754573240601380244504900138114629177479198634994 0.001237044505268342881152404001199760556423318884281743889262828200
            0.06868004886032333368033472728616456359811380269349986398971519663 0.6390898459493830716211576312676299657524801402600998499535390092 0.2628573414222329487829477035517338020618089250716153637085678615 0.008820843790047528790484095338056120534052816218902443457724433759
            0.3188844015702966051314039922076953133616167792028466837083020803 0.2563279148827533539606939449759802426224740870151765919357221767 0.02205413159425847658767707730164550832357822394122571941902662568 0.01043159593479736440448738373588159521726478346177513928808905239
            0.02258957897651032405236866538380716466378865087669668589036497842 0.9377204060474156121716369034850055203741201579558987249982433011 0.01974346468974943905074407343538354583332455581110107982989230380 0.0009832537285086476102237461273261629370849253015342565828817835165
            0.07831105515288454193577020424667875569268638289574424157475685297 0.3074707858431254670349738221801779642579511403048304249371545687 0.3550956692902213567793363304971745700084115432275092863593245607 0.02094039926708678870868611810595414007864719762038045837374767037
            0.2716451688507565975328316658745807544178052611855045897647946004 0.6221891790537599447816269369525852314587762206319162781475743165 0.03140889184182092190199978366144343702075162832690757416801002535 0.009965004019572324194259362357527806573738171937628733055628175687
            0.7397074542388278993196030056489194582809414191142432742352156263 0.01227259345967834043553136609802716491806121691938648542451397578 0.2157413191478024137619864061605270788530476439512438584491176647 0.002799277577074749668022529660420028181102933382584699834827146068
            0.5339960387370612681297094947532042242347097901701342456725961524 0.02814573507833481268315178877940484561951245890789834809440781474 0.4132940105936008425985573660156462321539221406831882377130181619 0.005939738127679060464713563851667855600484015829311809140790042611
            0.0006824242549708069233567966675591291677397555272680738752624267695 0.1137640292747117709240797569056858910659276369197356977251704678 0.7166715323629420345921557605287674575703244871693655671553462879 0.003061424545739267900276484144941960692869778354387113231075696348
            0.5701915626602876137002505905087833146320172856303711408600220052 0.03066058830695637217652832212236262527784589017844501523262476139 0.1246885351764930108685417654160883617964027248936418069226711466 0.01160311315220582150299041388177618297974527251273481347419635445
            0.02995966614441044649662004405866685621430276138020813688917161863 0.2464499932207717716395816351542192036634955438211091120854705910 0.2356837957374230309584425830960276091574834997640546879244372887 0.01354121811112875387378661313534962549345646171856461351124682103
            0.2727223353627747673188195046113814849722157016116353892592399714 0.1044235761373312069110381098790806706438294788820709030761101394 0.6111929978180554328433647772938239100954581242499266611246164832 0.004274517994504303766284508671439334787714506275717441931028947263
            0.2528047498513282837638014413842408996985780600724048659451287557 0.2498259561981234438659393306389071670879123981761258736113087187 0.2488346162853460280855375219782690618325666972525182644895747080 0.02923665725823202701042926037412957875948455981727938627571912733
            0.6497798438391258814016741591732822110282459676345775418203197390 0.09622818785925554227600660302131699073514832529938610932498002892 0.02698437182572767023120438823233595243441262725676416507528956731 0.008162172019839207430701935642053213563987076250494615647664621850
            0.05699473423226969688885180374206649463869156806417946852884314539 0.4360573676985982751073539804022998162449028320864207209690526452 0.4870807031633788185911701191154849564091920914906887995613821154 0.006944182041937637215892907656647866313300104089883296355623304465
            0.1470080731838995899216201604644066921403229541020385986199559119 0.1091053066424883510421539270520493080736797647815710149028369368 0.6815889653149101762087063498465853419641722493965378469230565159 0.01276239494976923322172157319794498602597162335438182236592096922
            0.1407540134025785858709428941469676075523736418238581198113663662 0.007503496946606807423187494353435141605936833752252032015432256612 0.7893156528387471947019470252463828675191314792733139680286629555 0.002940624011088653898638176336014635994487431097923441783145396692
            0.2467424148439599166203887767081568746760739104891106179060464689 0.1251505020940540222299581302944422416343268828995583238804738532 0.01617064024117165050569087856667466778784498423936499337422601119 0.006247382051632931457497444009315160614065433188500176972841347062
            0.7702698273071400797379583229948698076588156961846895172739305050 0.0005873779377406700462688609386606338752086910940526967842035825486 0.09362275728027097835442550307755591371758226402871144017867606700 0.002104416649748736602437253985935231917860751045122070856470632072
            0.2220333288859927512388798233337555417953774628997626771814992691 0.06452224139777309122384885154289761446934066053229407271120698348 0.4067098130204821245364877662408318052863352974123047006359064336 0.01915481953123733166187524728507822203838723768408610365756039918
            0.02316476390369501665706378295396900505500145122732783364253625404 0.6781970453492861123902620961132292923926652752802139399334842548 0.05525249503797257529208151713754104463273723373009613717905397006 0.006715400634863321087705222636901329308846607236600156733085700794
            0.1153811621675991454296991750743125919701480660556463200381933534 0.6985845293762786514697148714556590891052352028221229987147943438 0.01822562132962529287723086927737764281546252034668288272194196887 0.006870744407837441022740319151551446095053987861560251156376343992
            0.3460362619520863006169316106075266169556036449235894730141102714 0.02004264513515800191709360952258572327874228506514021504035824480 0.01311264687451299392278153988569242009887437703936783554120210193 0.002804019587918524603564425395196053224523310616635723761384200640
            0.08825761266576439783860480335235565868037861023277242729881495698 0.2218907431948260607187704949252869844423303076430606258898518030 0.6834728563778517377605628429550215062979629849742500300038602300 0.004038654069631706484277447806377417928579229442735665455815930581
            0.02328415184160280046737082408005684781228783694788910670371038415 0.1131876532546426526389366715520268713728485096860145922783768769 0.8352511859305203506691452533040294911727081933980478885180526435 0.003534590512539963449992806961090063130154485793821070112295400813
            0.6346903922139635044082104159994778016475484014825551856226503059 0.2176445844200594579544208067304715772826850722757614946627320795 0.1312497605983451704034309677781796968825405470643375723319874614 0.006149854937007207912558892297317278684947665353749235387273377791
            0.7454549730142180101289789914511336009136812054271037700844762818 0.06459803113306077751121640019895127905610655674368384145455232285 0.1894979700153291757252918495559419661709697418107705468968735825 0.002071937798747306014899220043876939252487642650450358184264170011
            0.4182665806607064654770881076457709496569546913361545113537710131 0.1607029619518292849289282153429138842189262093296887068213774523 0.09650337090191119894061909264313118227654005645925926445210783088 0.01967059909592654195443289057518088063336382077227742983874745024
            0.4179086868664544976862224078413950399572407270783241321619862924 0.4067574654703031537091874917993863608440882652978289285610758273 0.1323210978870586391367868249251988607721734030183288285646475107 0.01141325362137926030985233902024319225335543969620476822154114074
            0.1974444466250253594429700283560642856262635809612429708681862516 0.4720712203528455189012060141991065920180084338158344243078682974 0.3143430689287406794974951148308611923318140738493664425722941846 0.009217525161029499298084541587990864410298872795921850962848425098
            0.09416658461669926938792108645247526575532087450175182916722040784 0.1576674107654040585019822080071292999400303819149364568474818944 0.02157725790188702575612274729457158754717294397520306665749033649 0.005869045228598484589512337850972373683282606750660590298699615410
            0.2848764897543504994845317026780232027275544396950761328827392464 0.4671964383569775843686730621658144160390690951221556755731853866 0.01372274093510227155689506680872275753006916407462408462403705318 0.009450194021304984233830753661920642671104999771603320194279724522
            0.7941787805907179559119529754672539765682687416733548983414843675 0.02617087792018037467424762849900020814472553717161301139886223362 0.02170108604597572142405301396053389464039869531623980699430407916 0.003159973409905152182281785794298350340040588840741149458785854274
            0.1955902916256252013836485532588539879056271577414300144345250611 0.02600247410900522075812121621997792865957984769666158718700033750 0.2355013025602733684775335167519263420354604885797492120883829279 0.01259927402323624845886438828012211693758329883881872046914245830
            0.01315505860244797154448378537924301948450137700580392146523441997 0.1995186514188924262565933586594404759907297806235760783594118248 0.1093275565845199134451111221068994860186364476663485842001011989 0.005831001253564495517299474775592776769924719181557396827173171678
            0.01163208358812014349164509891187924533315299013821900562758126159 0.1446609216224161802186210852456755833536214199458080291712758070 0.01663429160988098814812246650789081849307189475119055457649525968 0.001641941245776402553705027426131986272937605435613709101229384925
            0.0007710667847104975377913552581651432170415010150075938370548502563 0.4105659428666379320753341227000157058240708046310595161510731340 0.2749753955183103389073633133421221385527561306688089144268743537 0.005189411757444605879765158563289069700795494157997627234874518027
            0.1925909847140324323062153129409638280835493088395058745550645275 0.1879454180127334211634601454834941012498402591552170737343259908 0.4563220166156197114271757860499843040284230537791585477229724097 0.02521817333435903702790623038748327376636039747218841312330516910
            0.04590957741387623460248856029014351030671718981790327703695520950 0.2211138175930674569903932145029682574722231384753328451493019002 0.6385284550185903299623101256165205767544153512891734697797156610 0.01103869662066780244927764179332150498526774899900985191495471451
            0.2209716933436543171468280262278753778491021751916139858593811068 0.2656331286604849855187354996933207016908103471814569316512776078 0.4865291274357054941959662874083594866040219181640184233255502557 0.01270099691891489530977120049601315932224017903668645407598697737
            0.3574707778694012103715005353953612006113249927403728961944467911 0.2771418019798105692035527445439097395740947732116912548547786452 0.2947597002583992668735991321908523402053032527476923415851730022 0.02244616322101378284440066601367456756376507506190479836782309882
            0.1834661942572162027537466457145637076195347261772562701667459862 0.4354309443066392958023335333185272444256359081198433901577484272 0.2596147955284586988118893207019842289821983899687844447520661048 0.02492470897295827879019738207849804922677038634785903874237569854
            0.7786780894362586008676973364061969871670819761606004302920706014 0.06824556243916786445130041667453473676967514823095745782561234774 0.09029064384091519005623092334289850034571520791290096403316147146 0.008631912682878424582119515004244265030627820097208479216539437804
            0.08350368495438306165178529160542596761006492644819389444917000702 0.01302807471301202803364680828558845148953294178591734718779199646 0.6481074404001489492396803258806112457719090478286007991405801561 0.005280799823034181199657322099962057552480584299243859739751213795
            0.4111811646991075373146871315038328329411152642956756783150571105 0.01535890607400518970365988077789997912588497351544445500376921751 0.4382626475841270454226807344330319632223574713784593420128313757 0.007824712877444318668994159061934333670784125960893805008362924412
            0.6626595119522672366515299141311039089359329753215365069108253968 0.3078481341740694606890602784106631102702427109333788984700898310 0.01836036614640199918383225883987936781948734738879363170309957316 0.002232095189359336532092446103290188131953877927296396250209750648
            0.1363574870412026335823852628338947151742579251912398143560999828 0.03119358133951347968640749236742715373528469764825004784298958220 0.0001969136453310146632092735733960025511259536097869278110717783636 0.001568128252565593634428514774326837892544221285143611217773332691
            0.03189017059340543988638781630496828109598729006254731830537793058 0.8415309916463650158762894120595832128809442148258789755942786583 0.1263691242321863556343245421047402417047220776458503422127192844 0.001557665371032691639523815486946992943553801141914147837889816472
            0.09965105882674535878642847724671637272007296735476530742751811653 0.7020901884670813040193397537475955544175613463282626308194202925 0.1221274353454344474315290256564044021153637203595888558114292326 0.01126426457011302870593041112378047266597782476190406544734781475
            0.4012767547231727103439061199818486199374930491277356664208335635 0.1001360910035592549295685547001716281751332244575705885186262403 0.2991513048410059468806885109333490180786263515892791365410025273 0.02242171919832805211344428176326594337340568538752944447810505149
            0.1110348427231385295340527042594040085268905486941232426880017061 0.1442668515669939676650090051313559643398724528733103420289311380 0.3430318222425007204367502044417667600526108980897573748270481995 0.01850140896519679180928820445084102549160478068055473102434149358
            0.1132803365591721349110076407524931031368665979889643719680657392 0.4988523951725763475534442076083548341664468168728189992208049640 0.03310321391768801991100139051829594052170864229709272013522523516 0.01222483529961557914670778826392029564130982405605101258820609270
            0.1851251285337305736153882459395326804332082203989855476347106271 0.02525792567653101405773630348333666507410723658192134843695374210 0.07195998898968393723673207908315951292118828478832867691998861097 0.007470830135410525876123098859026726392799607311174045234890416794
            0.1681180841912250364053735040776430800640397554476349985517606872 0.7334197178616813483913704202894181800216792080053912923382171118 0.09707359427562630452860258644879918967600407251070177926140364926 0.002541739357824796321317423832991630919922169984630223730174332266
            0.03284832390018427420679292348298041219803732917602794311015161176 0.5872821684491243071114099952187834650719053551130475216209769719 0.1752584104734631969455113144606805416568730733992395452221259043 0.01114055745514471490202388021418714368476695280705214334180818863
            0.01902488932431112315950712300859970328890118689036458007189169051 0.2798401617940674785591174445414425372751600200089453322243377691 0.4941255477715584002694660493096874164487365465488556966944314560 0.008287094219034135014681694532144349652954741100257257630034415957
            0.2219031972233327747060293474753928189807149183133954516174536694 0.03336708697129783320190629366425722289671579134498451319518957127 0.5889748193187903215288010480019799288543772373070035542573803860 0.01235819093825746816499167987094478857906517503892568335546779676
            0.1225548818823874048115528228482420369743379266173852887555503076 0.04221589833631808581213415400052447873165443951355827122396812164 0.8332743833227930199856763926560714521476600296085402585175720919 0.001912427425028697324650155743490933854595787522684728470082402282
            0.3373458687303550535461848135736067934523765637579549742781605891 0.07956685331036832389357532643756373263115208501241230126824910116 0.1975008197245110540130005060879133660833700647160039959044595432 0.01719677900880949456263163068638808472739952814626441021092990700
            0.0004281152710074780652254071572196505469598417452390942952238171307 0.7339455993998903920323073406152610653941348101999970072671202092 0.1948635591947421787171736358278927493483136024877023706648340645 0.002661407412511803017294293027022429900462384806186361767935486688
            0.004161074458799275604505161575489843880800507725705838789974002599 0.3032221849371555321077229536969700028485679298286838696021133453 0.6668028244623116954318459557270468778213552790727326025425974129 0.002111852715192493265958153515154591607776985736525045954894446515
            0.05607045901332246177145050235394973290409073459380689059098472344 0.005594038297952845803297651871882239337896018392271112247197206433 0.2223406529843133767708995761418743663479850289534914679591729382 0.003231081404682448012557836119262047852991389426951085428986509477
            0.02313314451309500341131614326320037449110875143614640214827162184 0.01775232131798975385673759167865484759832602594314766362219922516 0.9307215353220839072722647163353914641990610469456987764805386509 0.001287254652510155007652761518810393025864053977439925852114286475
            0.9160929177005886631598824312123288029069337679031216356759943288 0.009566613286067694471618801189162737106839189510230935302260698949 0.05509528495879926304002181886391982500961440587634154463657873554 0.001199207188133607576409774129225901449132903068869035548676460305
            0.3765542243935160914332964990166457609395490878519616535913031486 0.01564160092176653265160852294119507078521545166978697856965869437 0.1069885872534006189619478108268885467716265010785289469123806181 0.007319802817972011180457003412359295164273171669001026507135215989
            0.1088463195364638618180639217366041205353374448892912838831031219 0.3375193681181201098137826708524532348150386963854599347163448223 0.4669907364435750355866130844271863387937863348562859648249644884 0.01404680918146590234456454549621269439791455517166160729848075445
            0.01273747800295209721869752396240416672938440385781290388891356794 0.03380978486796647764867698119867934424119148175431885680079353048 0.5960419032826407267710596171902430292674471456955365027344257184 0.003956345742267454768146912137233712607787193894258006902302402374
            0.01662824118654364188507813965839784960029221365641922662383111648 0.08016530015828685666959959913765667243838236627918708980463675526 0.3086704023087159015949228682391878330773165308065663726856677691 0.006403414034855742442145210243587104607849459950214486825132380876
            0.02809216035859757868973170887337376655844428426856580683683966535 0.3339769318301754180845238824535957947702568605329183835961577898 0.02186218863879017414140899296606039383324663170568873822428057385 0.005009655911036778668030313905729019400019270415452708677214445150
            0.4191308899598238304499841351849141646960010023837547349898281247 0.3267090361518041644395189006623484222327889085844218771019695534 0.2534116251693332944299408656585192145489058674211224339779486286 0.004407205156835656929372759362827308321018123885723768387151857653
            0.004983324642759546390749471394635986686299283771176681569556258568 0.4468429161402591869653746151103997712455179345397546658175179652 0.4547159327902080606025210754010139092700163013168028374062996123 0.003735927350249386809966618042225796280749497674503987017373325155
            0.05959051655210342387181073545858494376557288196392051132651272897 0.001465584001853574704749210466064585610810568112919341423539618026 0.07814297286694042898980450529557106225177606539122143624279387455 0.001275574430446332040909618258420706888194501739783817916634217036
            0.07622901903636151680267474468716165676073290297290004838504905657 0.1079116202865749393684283131140028338605205600234250205739683592 0.5788337891045017751812412895611414986580309732518928891409001957 0.01656303874634306283125430214302281636120745423457820343137811197
            0.5675913512164523109425769021821878364990855445454735139005328237 0.1271247946017582736353844222905152827111129333625154613253737943 0.2580324750619346138162670829984852736222458545856466211962824648 0.01443794907535342851598698463236254200281455066017773792873516154
            0.2052515363754559098069439999594753634139140550556998766012150598 0.0006321387238300995238808879865465813098103818077937853691109512118 0.4445746048500630030033159710878196521937470673272946095355816514 0.003618036750859960197689434657454335227483098465012283733671902720
            0.02334815612919141413079935867870058076931028544984155130749292119 0.02832799944699215902828052240064930124994773833373417900966659014 0.02262044982524357905618081814856048227387019852410048355534048091 0.001445039105866357803087449721994627345550634398056513839098429656
            0.2396706637906562781206571627796597343582640167040429425695715595 0.3164881510787529057344497836663404216809542426384610244666517520 0.09584334801985631412856166349294810472056897225719723861420804273 0.02087663686983139447154827823434066364944284992573371732664122099
            0.8406680663500320225271119417408255599000926768308369790970852761 0.1190408735079882279589660399499179360909177247649376606280312887 0.03716006023849400180125926779298384862137659856448208116238349800 0.001801447140619997360330482108261027263445444412660243454805953653
            0.02622604646661282325076819842448774419107444683208961489457989693 0.8301816228420967335605763675729790208457133711949646815893966130 0.06443304105141772278968853090022012444193118920832654142212020053 0.004354907370554153583134101380568575602681992070336681452956336504
            0.7297120825016440239490742263755677807378639159382751293754017303 0.1746232136659962438422992429479530772950739304544362043854581653 0.01761380210304481353162401786566901959759313477048721545017137274 0.005962593633836066261358894557151119077181582611362076198201882803
            0.4552722515017930146456035917240823772824152368168616332701923950 0.4878561633251922722436370886726938709621089428500855425114229269 0.05646261777012345157188059220832448802889414411331535989015650222 0.002497003461277015557536880869207654005666511927155444793870829332
            0.1228778858886704305129007506607978141792156747084447969620184128 0.8233040849673012888030878042895504798564483066858859551275700644 0.02113427518476824872223442481621138855085788767361588285744816788 0.003769391508996290801861106730301558315523985650373418466282062463
            0.3556760652003148572798747578937617011842486958850250198331045499 0.1101523291408858430856512939870517598198960740094306102749343534 0.4585051516202008473723290574532698274681560098410878479229738126 0.01698253285355117834811607043276127479946753042816283603090555784
            0.1485086395302150772354045147095631056029459532686896395597816049 0.3203604969743520512394872444436692739202757402636628096239054399 0.01515298869521982435358179711913378269827090635806132591886380096 0.007119787411961408306109226086437341691354283125407682846958555280
            0.2551837696452929938397987442305784728684748964146333722371085814 0.5659187181215239208419713183455416349788748791956263428063237322 0.1434162588791990854442590112629358552123987960249842993384636798 0.01148847308093033118173286092001847354141312437986542341081512514
            0.0003017332462818711065708551701911449164513267679451057509027401637 0.6169014026311836779495168015738761118683910082133509365506875789 0.3705859753904056227735212681830276924793368776958031321539725942 0.001138457389257084168393408374158239457865652920497060920084823085
            0.6048119359801534973588105386070083178581566121551066871484425034 0.1181670758440676460832599708967442076934947564844044815984358612 0.1212318485369729980985969586974831290962143896261279280456127579 0.01295733914154513076564249389907515247886741652987955693146551387
            0.1812357648993201687634635437451025137749270623319852471949179337 0.1672880683670824844865432468078992104269755388284504640782244862 0.2075403852637664566152059097244578667303825191418954767501644745 0.01795878964893801960729024308541789775672834876468343924291731795
            0.4556598772841737549146655523705029912207393458583701424080266740 0.2234487776642279830395133017291555094444725923886575025996453991 0.1644001981099326576283368266206757401873244415355076121103663534 0.01774043383792399916497873069568786835761785336441527219763543924
            0.3168082747637329265437961264073022942211469318022010650457239885 0.02039170064851771164124491360948175657648498882135752560594351318 0.6344121132733613197052488182997405625953097102259253892288905316 0.004792532519691075623459024867288620596876207510624311928370102324
            0.5885725878152027872889126129557012093845835789611998559660725876 0.2625195507203611314694793456846002992815324012550680988254041998 0.08421233537415411982883852109055300066671419996781030849140369638 0.009714966767818229485403352481393826932093587212828017234042394863
            0.07975708812736354335219378567475454675785139079271902552132629595 0.08692365415965058426881277966752889000968244976705170154671285424 0.1857122827844711803758959074218098476117870336738258228523697376 0.01286495976793775129385915103396287999629562301919567053675341951
            0.04582169474335612000580612913404066074219262961594040014390032745 0.5087280563308816134806737419520763410044992096326119571767710351 0.3188636370715609570071070731089868647942188274768991221642282354 0.01251939207433797915207512341232698543085914996460051484699418810
            0.09649134534538411328254146562032759589834505876198328228399033368 0.3968927090452546717305416071315444273117265460702698894607580143 0.1774829177149692648261564220076306641223917537998385058920870717 0.02149395820117334878651458991130114636546474388974351970007033931
            0.05181735383466156028400633974847203417535405959811156128528931347 0.04566829912932760506766355955110554513959476075185640493640525354 0.7885720971549357475516138991960207000005462016493797037942499131 0.006110626819677322188709495278229682730861422313238508104843417148
            0.1574832839135006327759993318078009358294663047873626359221193113 0.5479580515355089666600938859711605317926352046802016718979505482 0.1061285796767766132130088227922931497965696023630584436335223126 0.01741083301768565106548382189505580273172969185735505139682558369
            0.5997923494452987938699761731537680275085017157621793672662479598 0.02957590548380784429504861941129374073833493635514355235871561850 0.2451540278487019845108398612871018526352293551755404648355870899 0.01033648581935253742576539972132365966465624072929237269128327993
            0.07588732479651983945656980502163939809205504551942755314330369734 0.03146532756621526113452001424758161237180913761485468535003871379 0.4225063065398066163091134477564080414560855670592882091048777930 0.01070497527244385473680319291520221119491334889209671749432177614
            0.4813940600193163606904057614537727816648076116942950670849623514 0.4330558325519025312444517953414207263054003874540732901479996865 0.01810065166387953462583253878142975178964017680574195211259461164 0.007304423784180342312821136362795951630280753774193367829571508722
            0.0003574237421419194551183708520952335299468038028324525209262499920 0.04326731422362019677180833065541619098020586113810549502048475784 0.1411390293482130051073069006010118421571779466613378988269686687 0.001924893217670319266927039240825134378923405495776648315655462676
            0.005493262085055545740482102100951413864451162892374126327823694696 0.002956537533261605499255453709858085330995054721109098486308576760 0.3996789787135795987745570802137137139908879998686605086167104155 0.0009342024797427712849634743839914485584847560584585063142388440206
            ];
    case 13 % 144 points
        XYZW=[0.4029849822335013645167924504872952783349372782398062876902524177 0.05592133943201526508714071582632703366597713338012429128850311526 0.2712194777935386889735187758252511120262822680951316638550774104 0.01373549356161932410884328019521951446024193663900696020007406078
            0.004853146143288019004547002040098236997569256497576028840227385367 0.3870336508904235719648574061299480663051492495690353500891462635 0.02672836084684379883432258347547241199773762448277240198728379998 0.001822709955127244729723963916388917051770030518527176804223299956
            0.7539576333316478950782114523678843648899375680303011560359416177 0.1220656013592209501700354795910666035351826629937294235993302141 0.005667902372503289788000650621236164400676131897534615366211466977 0.002856639635783518939203514858267133541751829245587028611305768248
            0.3147151874768807930824718179365914909380442393925123818861395212 0.02366395046697712237513560188080317628517809167512505520819516512 0.4457001333729926434595937789811107599028566958762555971037069616 0.009519221172194512471471133461371628953408051521564558032836097103
            0.6541305721853559568529078087547988924095964795998704244358987830 0.03652482482703600680432020912517529534334573999855087610576395915 0.2388946077739451719912134087691716075898224136781213477409391594 0.006748556866137918033517214841681034409411953511141294499641796128
            0.9253107621598265803355924825729258567451694702237628783447076210 0.02440833783241545763237523197981869350873730186434066278405342708 0.02354321455805881236895079920261790997047952461523241326750107138 0.001528447159324985328878295999418796926349859515117997001741872532
            0.003814050373789552953965108939599970648915984982901944559421963190 0.4199924800525905824313820177085145489297840163418851052341070261 0.4911653748588106257466472989772247302607281566414613462266218829 0.002582334754040340675805305446463690065787485872510087495221753298
            0.1275416827815340316453174898429706509544910395594611015472045736 0.8388985057288285304569016089946287390356704562425187540646115597 0.02607303733543889458530867270676410692472912538730893535158999055 0.001487676151255576121038761490211709265726933503764389847026634745
            0.2433657004592898929318834307885557549356693382036127981663070995 0.08331583279663721750647817562508112273468422375083719229646350771 0.6045548024355646815862921024186629530956222776202706169335214145 0.009259068709325052131453097475924534109090670584319348041278620351
            0.2365836119401133174446143244298456776204955072440416499601420143 0.04397170362004206628459900981500265884521759046826490178987648528 0.2358067524723837509961490365682052763719205646203127615209700157 0.01312506215307000436789718179610997211332397941851941231806615078
            0.09908575849984108127525507129859029607214761947149629985944294744 0.08780387857936437221121543945144295969758020256749385412366186219 0.2177883309782537664324390542756195438678429479155147500912987209 0.01042448658734371449403210826953006287374448846620436253519846625
            0.02220917114542529754773686651476878154082886002802139064636611326 0.08351475886608446033958281780682853149789764602643015125963365525 0.07805936045631859714312599140247834703192902454278185879215775416 0.004138336906866739249806910824611533455110757300508367318466174325
            0.02773580251377124194482421356147068290257626514465888223228653676 0.2406237085601421473678778190994143022401862968045259190729957686 0.2602995441718530571343477865500041197574113771040675932481263430 0.01018993222151596774341498822414312826868916325843471863704000892
            0.02606404085845549862492258236678391367032460296853383423853427943 0.6170692484959878006139434123673541427455695430666203056995651086 0.1223129938671531896213687407457077584188971783677776260377016689 0.008106068425702630840654714691913937198072398728303341680885192002
            0.5483776133508337878472372329932329496347225680779743255913480675 0.009975161650579685065780849846353652439608106825158641052371904886 0.2259064723324664214990615390385977731371216237388532469704217971 0.003404118058020086820050041103022911509398171839859022057961930377
            0.2551857129015324929545961189616734475819519939915246302342190566 0.6323279495408989993044709160619189959605881357843644723510113969 0.09449915800490026635209628805831498513540873019356166614616922862 0.005573619957697218588513721657365122284949848078373991779314534462
            0.06077073709875282229111451786034922458717664751465192830669494619 0.1041871860416217385427002765266773197249756808825240731551839848 0.5831903849613917330429249884623155675399492285851912385157925178 0.01110196516871131037296445955711731181847805776204563975257976536
            0.006887091028037954629877469323934472675397340894294870525969989188 0.6494547775419014798370385324421091521598818771053808765134469836 0.02837275336661747456902859385205142562239977566084988012859687394 0.002102625729556566219242525808001600979426177280327280045470751566
            0.09237751535730745423242943212724740863588023475244031587611991627 0.5666882522128536803933864734721586472364448282666849810158048991 0.3284772693557573706003546966610180794507091208390835970227018175 0.003261971658508998056082547307147541813771713858116234155019501121
            0.1038220577689695046629866040092653360624918976624986142410992797 0.7586927299831670731795737512634377213942677661602859718630998179 0.1172679325830135736716413970173632838132899595408199311777447086 0.003860635536487606714288514713946072228456718547329417241178488676
            0.8193818015150089753937091713006884045881312863739089904697634925 0.02229266257945350795560966226770190273533461804366514198373188624 0.02169344390374402390636376008172323530537861772048812590135740637 0.002786089096449534902587645576355042177543043602931045100125685318
            0.5290093055284570292769691731273251351361254001190733926037654498 0.1743583158033034895215972812166253843268779791163519776391512368 0.1996662517287840343766026247718002665304687456381479526319052111 0.01488464149888643466897236488230389049658444029334740765833298892
            0.2934586296939319535645167391629923643316557439966892244915729659 0.3500701161421541063937200043859691011013246740034459327963224890 0.001795274011579795641912552958487923572375463420822407319267346126 0.003451728109540217848498696235970241697751369985944276142078423992
            0.5649882145149829472192100687524269726831946402698777515625711516 0.2685928292132240793582932433418328157148636803986492839615076148 0.1437855827641652471428216342318016528864385840029922316339916742 0.007407036317944308103412068114904984167219348378867372513981414186
            0.2581840093768511051153888510519547002432051354020694282219436394 0.01391481398778134893456162803617015551895690510000403446134176274 0.01601843080313148261636653581485333023568854704162870697240845585 0.001612886202742083657951238755625241376277157799885058168898737749
            0.7333391089762890462321665394697107420717474033982213358997861961 0.1189409601773295003832034489705440852971054065141184424649795503 0.1286910814224007344041220369443485853156301999990043607641701966 0.004788317041353792752404202454843431741788806974932784906036828271
            0.01488154917268644622575893480538391790685999086400785996257500738 0.2080875953294871815643650080782472615664069648105225734266698648 0.5615606288779747407169598344532366188305432980781198192106499225 0.005688046714374978927646114108967932210535878641557363066714317864
            0.6563448365242419548323654736579080468504665762839687012401516992 0.01896297089568418937946296459279872011833834424612733933610323830 0.1090460093830295429899065608927200295480602872459410579223824330 0.005463293623259437280489660735408883462385039573768648982589614085
            0.02037077014070445027192893025650345505733480622228607814173476786 0.8216460831331381272713341379410795727059232177387212608402215038 0.02512304049685118846524272497573022358998267611155408471617112202 0.002863140074840652766071683997436543009354414125675025338785605614
            0.2519265733226840321843715864225186213456080294996531301119432573 0.1178021061756160101676026211319098820503651433141040291912455727 0.09824854115962866410505580140724379406653012247245284709560356728 0.01467020768401226219944650592703144220315855220637694237755230120
            0.1801627755664692097881091421556515779922454006289504759081185078 0.2548986751649479312239996748850020447232549956738324763114037576 0.03432355080694522085222095827771941089667776652054865747646394308 0.008990844587764108982902312472149782004825926642752454936373980897
            0.1178931349555657198535564180125967315491004578240737069747355522 0.005805106374632742532113101479243611582248399908520952790021664549 0.2711596885210727452839876384426876428817834140713436577385592085 0.003845065939684061720560343571696553250992440210972217165689641658
            0.2893417460129522990327504607532731691826056375730742275147657906 0.6733494143173197463326471972484015847557300810443002021585913327 0.01323215698107875022618500752132543017940747073052026675168914354 0.002477622172607569453666507723478983339014310736421960642351527646
            0.09936553397872717002318462836819481302618073628475673939402895418 0.02217655576488680397179584109973120330308495053377727604328202866 0.8683226708871458576316708026401236820544133942615159476410019045 0.001168106803721889742644250966920345196089710417890785128431340686
            0.09333623091861589214561849669142515941118828298910709868098469430 0.7183074630157773905877783409061705494679920530247285319732084763 0.09407803142902548280469388487601305058530985856096817927528098766 0.007870811069218207431894082272538807589779243384891115990087713716
            0.4267581165145132698598426309051897831755539276139426932813672763 0.07408656622542265769617084793750566804240970240014709788145377656 0.02151667250999563945188401531245719665552045961437664164922306504 0.005931367385463203086106055391637817559517955169836981678416545268
            0.3841395987891822796100847658543931652058318789766602700518494103 0.2018768533145531569590036064244355426784436102394179391536229722 0.3753616168386475879256525743641762747554101030287997311854432438 0.01039061675457746969648342844995598825545150292278320618955102429
            0.4765706027666225083994135192191235885511875398093235954150097773 0.01478774674820894271402415524126811784663376547651392110226958857 0.1057874397482365543058867108746498642009841153699069287176617944 0.004780112457402286306446073431436143386505462413999766709767011175
            0.3213650989572799139836670043410495816270896759081210854783661557 0.1763436351087705127968490538892433786237661364493388335815106398 0.3230336022622175369191610053622243781015928688454996842917028179 0.02263013054581210029421753488914813542848652173870270993370792822
            0.1024350621730923631101252446704923930433356933362100673152848461 0.1134345843554455563728714321284286769451433790610244190097089677 0.02012387254923511782131544453179222670580772250210804240336757612 0.004892338621117144120901629749149754280775758769296322860988519927
            0.1350377562424163231404085554993968897923813246999704833856009940 0.02201631625434678763526537355347320158186969280574561277784629198 0.6326367976472450327849889503283490846949733782552073915845469609 0.007754247912945020955623976290825527936295785707440324307575227870
            0.4271559427401216400158555617937761242509754351143929435946087691 0.07557586207007876161153807595989385105667515713334225948450400031 0.4006373352262787954860945653192033108297190540038222288435914269 0.01436605567152501122714825802336815837885518571698060614867247804
            0.007423352058992606395287977357899840222943964022897019061323783359 0.5055800183058936259163788938292459626053628504410369913639967072 0.2814213826273918636828779381264358954876518740715616444136593085 0.004759160510579541471809366940474767523022481845347462266412232620
            0.8415735070096037313865099845738114642259012433960898027225743132 0.02680895576590937200081886933156518836608090431660229327319764789 0.1227724136313379633467424923343961823797627655760729005432764059 0.001608758844511233560187606337845720689890724520352340779835980570
            0.1333141073117334792900671085968959799451672131179237167387588078 0.6184982023463261144584400342402224540426582637767075745501763863 0.2194037991040527718923472849385308185587937512576114750312669543 0.005947564670264931520843820507779806836704405734192839035222378456
            0.6463773759102977965783307947972827398803307158689066786852706585 0.02158828829912845319308414075724083604488034690137568892467274272 0.02183695925904231817863154623564595600980202412278826805905817156 0.003543544797731246082201607917360940336577146507037267500065569003
            0.6682584798179189759938385117190344303176572838967114562020474988 0.01816985860262633410699652505306573754693542095593393628134184178 0.3001418387259552027906342557984778326401029822321829911412160174 0.002142472814463233142257500574536291073767266580368321824441522798
            0.3589348515556770095799259931144541394063181760606854250355206804 0.3287887756031535521603242045256196431027031630271647798515922458 0.2346128267680382556378842388208016058144802350890659331521830358 0.01639230616999878583553914845848009032347209108881686973259872874
            0.02603285207534760074729008738439596972443209511398348786417613470 0.01538537919376644414624028086042120270016008037988938353298709247 0.8344706038911316003864794910206740197367005973057036177570330429 0.002080555027575614713821697520470737845722568336347160729115140117
            0.3922352888408486332217180617649534500695181194563399730061259044 0.1975428273012422100272734768346726618812478235959008934076347560 0.03237216995111302894260504395136366203571208896213012799832767856 0.01160545750040058321120381507918656461662931387929950593253521711
            0.04111026901870401391458528148400574531110698748418798147812501419 0.4635598795730534045811929372291716713431502978748338944649857857 0.07442972956782135527839119501862914184734015666084918482951526430 0.009145485587290452061358378251858059888198163511174469195302196088
            0.6429592341955441402140017769816650346089982074587058701827894789 0.3069963510519800149939539257154945343318636737378750965899690901 0.003031555307071350544420185786152490205826273700452397794322562567 0.002113641782569760640422134083897875163275665990895479375675374735
            0.02371128954039421379382176803040360126561817678609353049099862720 0.01993372390051501748870684250441287651122804320697757516671517505 0.1686919663073349588875962763314370182316005879559032157300535664 0.002675342029048547106753394335334086735728643683052446350444979672
            0.02135303531508865473238528298763330609565946097079761840555405558 0.02407075096426317568590295276839618191360543465196247574073774923 0.3286649267285530802179228653399651471224326799226817378881597600 0.003401097743243374285801320398364953652289642132579629635114282057
            0.02433520352837092902426254592632117699440913736585539778170703250 0.1326057181158871421296722038796195460551194233021444598639258662 0.8217345749968477548075587380540417699550087996765781148464469403 0.002916446523536408886713892716399412344310190315731517294388231108
            0.2500848208908758147851695611212259627143395918065591173151240086 0.2809930408600448797150265765622556979967680615673773276969973527 0.09628105590628796210520026471364838572015147459425799382587080622 0.01688483050819287587286419358142011575300342330039989383374153632
            0.1161336508483467803317176740372620707925086827736916231824570525 0.7906451176427159231454485676826263229394502011111114046345370048 0.01673385534076818741808613030889741638049337195918582122603129183 0.003783586070127428336775575595350895170183129596900726160219189557
            0.2773581941716009392954916798842849724446046810781908260155359739 0.01634087640699857554514028797777138552637341381733339696723446723 0.1056880399225417453754905713437398087117983039178779717246259943 0.006076250369622658277960270689349660069591488706938738618492061246
            0.1447229790543981636769118467123823758408313891428515127179814075 0.1976335251268660429973434410802430759567590574029606200125775668 0.2382338333404146379884929715115282502113293895583548876927226340 0.01967029905878050150391763278067651235642862503124719568869348360
            0.02147963678376824743128926771006879326334126707998015511710325075 0.8658808123179716897034240482551648193160389092200248236915570394 0.09152245318301566049967725869430653420782380412752378861470884521 0.001822356067979386075084235245220295611264083458116795970554539230
            0.04135230779612641688346549424606208328110380774896597611116310386 0.5073127947656974795858277877103598769814619882072792385182007382 0.004708987583563917063523813670192191279461582962276506671380634480 0.001939136966076217362149948875559123051671603698368834580841030684
            0.1251846817122643243590294866469504704612205136604198171058928188 0.1021429768712792998887583558605784354289227593347294597158998067 0.7536582610757931006118405034930534194380349353754686616172375753 0.004277029234246222722175080360280546349496067804340970439611045870
            0.1988872647274596305730474510344784889570395731031414388628084724 0.1095717661207283803382413679621659344673490729790489428257057136 0.5180545120158286449031806711659423563613602158629729111365026676 0.01440977110939026914391163453440998736265732324195756562402136333
            0.08714585193189280858769131170522164233624157123789435734545124503 0.07924891876583893477068696485032736661488067633819126320489413626 0.7355436933501266500584163216957234751939134926940820660853008159 0.007515754617918016358892506325979723043266251233090899116721033271
            0.06767322719005081157673116190168052215926930339639210295366276220 0.4071686220177270066719952402659259927658745229843783396648319383 0.2314662623799704314126320952134889854313153385004775239474024328 0.01765662146432253533559858340369823874360756774747847035938482361
            0.5572406559683179130539804219686519868566054982060692858105425291 0.07931462268300482957407369381668461547193838792343792157679701219 0.1939079638854152615398047673954097145505814462949807971866986818 0.01102407998683887246738703155371160485736155732709177318902166244
            0.6204029048928432089193658692334917444619991766154486302708899886 0.2262059784298142467965192874818188116031151358635599911651533675 0.05855303636372145650935082642403355276025719077215392523208573749 0.01075163840110004510435554464218531134675869327985591771759747328
            0.002398627388752580072179290524222552028566551402375738515584882796 0.1035754803073205584831860973385834699478974130824832484537931628 0.4386454118043428480450555953154195322013242786855191512346210028 0.003170707783992993688828356517363525576079701516399760394668672773
            0.03013397022098228494518269412324591179757748424633278525720048016 0.01599974517434206442900254499832222041349418339738449928309917115 0.6883742379704641629462889393686359420169261587619732244206299290 0.002749464619022443712243296953471975527341017927984434548852466699
            0.1927749023271596915716707542224232919508338432281481294387500158 0.08563086262785727517594723868210634127988348607532479188052457437 0.3911678239781108832820461348233979687503755923045074575611785390 0.01713675860286762730830896301097883822860874721772548499598850470
            0.05407445719993070881610561217707186820125756211755320822255242887 0.1433735010320705258051481434015618310711115566792008608386384698 0.4062652981231874431204795339669106866703135874155575094985164854 0.01031767991946288095048601502188116038769990722114965521884520497
            0.1173581286841458998022057964137425207167002405209496972977241743 0.02716058169895044665632115195821408022712314223164980003088122920 0.1105942460786471693510117718791136967606970034194611410312281823 0.006461896764693838470825456873223755962436366679273653036804486896
            0.3922865164013944127533147177029767573869572204652297668501667013 0.3163455700938890672205471115138026367145907208653177692287906013 0.1213857478349844903914756451596657452991764549325793017846673626 0.01441384614690900225958897047985147940837639987143175037633631718
            0.1285570141986206702984823919757226427741173058504896985910787852 0.01236337471092306987547307253791673781837636656454217054854641763 0.7891861269104634968590147936786728058068520313254330574696465303 0.002949373906783929357166976531632962624047604076219025274276186605
            0.01586929073726076723291613873583153119540892765995310321876666896 0.1816586228403906359380734023541615289017297371275781186195492742 0.01624954489998479211102562305868744822156688309331589875004543322 0.001911073982071574352370436986449099232700056699377534340926488356
            0.01864457855188294122660951562857472146600944330192698872944284557 0.2426167723167551981992698129960194816241036602045833624790808081 0.6581010911006299876476876234713999241637507702596798502386572404 0.004555719340697846767957889824668576273417400831877979671193246840
            0.02000306806911759139290458926855435342017835686428460818161571976 0.02649198278644205394329636786467121943103144124897081803420180741 0.9272118949938697583441246602620336060550075209310298392477229912 0.001352031099753691097512542842377301799135320511722351957163655421
            0.3152211844179504548128012744299219049976975101687289224269832000 0.4164414735819372807692106669015018611524520849012669039390877030 0.04244349793320244366726995565013921079296815288071816091712772546 0.01281190732908922857308597193871912923214019037964324638912322213
            0.01144724532868924817791844003745331738923652085928659775956358962 0.5356138710618880037709954357852492310852409055864795430262342950 0.4383279247641362059397478859242314555792001659439145146703443529 0.001750368919867405036817587458398898442292370115656433589046312800
            0.02162596723093343818160916517617665887914811672211519246031730904 0.04425418344039309879326557284958356026445724543745161672355567594 0.009256170098485279631185817336291645040650318374087257818852039518 0.0008799712357500261314848499804679824379412678257748298427306820948
            0.01767607782446587668692248746044216267715919684850912647390905618 0.01257406262554419716105513092953547376775675631991167161097737992 0.5132429887137185293257404526010412911791572289873773535585150958 0.001797774228137801914166175782217705579619753280209900734889009476
            0.2219495866393964897379040490660360322162072645129256336691930319 0.2360144401871644319999846163924320921374268802691072172880425854 0.4621492857150512043225703912277003766806648253449497473312258490 0.01418086416218590321715368940329924433489969323027148543134926719
            0.02327666683726961548410264785524099011417957271170147031043415787 0.3131957249220535975689495227287899927417592202521437581817146843 0.6504951421463809926748281475149019582303603155139163846774986910 0.002531885029722354188387827316759770254766348907269104332205242253
            0.4192106024863944872289187258027890503322402728850279643053508005 0.4730143699818874876241858091462618524065543316564624683360697384 0.008271076205242583707122376096199969836532030512964936993609251520 0.004076203326322686726142115605814233148569746463007000620995265310
            0.01713465678060134143830613017114264753925375316352268086825318029 0.1150527163382372656877306258797435362794937712016343122127718249 0.2207725751376743851374691627282306052738841026936089147649281950 0.005779995490368557024755724273899904914176999541954673061738172070
            0.2172263610561672039039542796914357267565438430922805159124625172 0.3414086357992757058416686975294300399697100069196316276176897582 0.4316520448014813735227014409699963069686383753008212615641079017 0.005696477769682042115002041546052660435946484047396311473640483713
            0.09326928752302341000179385024257435996633953285159197238160284881 0.3019885316244721588051765686348463119422855828752460294579060775 0.1260258013855673156090274754136431055903403627641705534712731949 0.01389566936544739356494885446725530491677616931603181151768563490
            0.02266845607157678668047812511286603675955353134187495506314249466 0.7345669396913100508718846550105911006732961962659482932020045453 0.2300230193803775930747006294734700603757285459210840435787892518 0.002100306535348137671737940310236551761949356208165222593231404771
            0.4443798014437983798743059082960629368472393679392985945033782675 0.4162472778529355546078757683513187698752951816807782969991113518 0.07053251712674502652303627094917086518989007728733944373312919159 0.01057324081823589190474732863275632887145279104976457724951527250
            0.01316219930673935575536684537254627308267871743356212581058564629 0.09444878598445096217302929595500117191652179957284614517898392600 0.7707888239626122022261843755405194678111870133054320592641794467 0.003403073657215114571485721216662456688579073997648111235428126993
            0.4611214108243424138861755652183842480910535592942249783785207533 0.01542892160544224096499104401346039088487168480670581727435973037 0.4946500422116851288397061808693610448054998462220263375995584599 0.003316626923281069361764862022694125495559349818676685111178064531
            0.4470308206477216625514819225605881222550490203597937409754451624 0.07385988148847178388408287762113435024788798603060827576698024478 0.4678740661231841013228283825381159597053592244588599012984778367 0.003364700168447847999450505152758744502024549401155976190955836337
            0.06971941222801175549701504225439860040483013890433391932335510629 0.2916491938256486166380543331898162760365693488396127402878247488 0.01892395360959876070660014254570485921200934920380847717564544557 0.005526423850961475721653235951235868443745743538730805330786048502
            0.02483806553903419551344439355716323481568000851688609328740824837 0.9308203478615271721421099821030931626187162141110943857269992545 0.01770575664219516158917551084552241055982535809082211526043076736 0.001159515459476759281046801529057506805221241799418587957230078429
            0.5822533058112137681422562173812908886543911266146588283342630139 0.1164687660150405949813116632412097206742372381816586606776109152 0.2819159982692993681761146925112297271670121107673382744345082185 0.006507401550663297685904434024980003988079798847128258646357633491
            0.09643181335804780431605151372111832987691986914799437369349868079 0.5445115549286874377468554816257739114075831024667362033308935324 0.2284281269191424200829001970160655088002676910993264878422173057 0.01428089090655388868598538817192465505585610570612094254238215919
            0.5162290956819030129498785007827654940716135609756284156302130312 0.005268202700851967815906268360577010342298918510792723822931348096 0.3578536322427501718353421978317165911903710508522024815341006406 0.003488988343455649048619737623536956885034754739788202110001026200
            0.07860747443944999466678142850540320699334603236811291644312310891 0.06467323052719519235192299530309093867770822739793861095662846487 0.3545049537527550340886672841275792972572727231002388809095284349 0.006413223093308067978747875787721455408605745571009746856523361865
            0.1694743367735752340953656324970151601493109920080531425586816729 0.3889043114577886395373290240188327346641955865864376833225479398 0.3437675387447756252996016952941715801893224017343687981141591926 0.01613358230446903753060832663346275038274588099222014915466467556
            0.3982449630749531398182347031965340591883803640070876721648932963 0.4475828132982453098539543620746361523853497888153661961628912675 0.1400515508901202287252970472776879728218006076897440778732286395 0.005123871066386284522204371439341592887398646270185070081338225309
            0.1613591426290698644312395819627495626479064212761415280031331570 0.4447468025000249884157523279254293199659056079809714607564005044 0.1114250416933921203327410088142622455517776645668830212513376433 0.01336265688458673701337169105965229450278310897529037509015124427
            0.1215659684236085692116635232351311428702186771250098638615879666 0.2418826354705122473067116593942818944264403043112215399813910307 0.3935270692079265124795460287419336952127089557268727336508854139 0.02131651102451090552604471088044228595926826887504283515675970951
            0.2489202331457709192453315507298007980445039598667178188345980501 0.4723090038059961139513393630257294273927906768028981036325365444 0.2545991037462787895142925966310103178413257034616536689087866733 0.008587422647815154305282565492391988693579919515494958512961354363
            0.1857622213318005077862987208456826588948261362552676362134659385 0.007818863945579304368930326682994241998426806744792108339507520109 0.4287936482781977919305654087737625613073827794876664513808320233 0.004644130839039988363665347963173008800612113091597521098338999225
            0.5900309048580652206572838901509622864435361630203013203733714726 0.1125306122759713891447341835444736263853768866017630603158176286 0.002770978521653996552951752591987739636652814844899740032222576718 0.002475885308757057544341559242543194435082171481325149889027873241
            0.2100994905113329642040206419347268864270030741725005068224123457 0.5911289035552244484785637908555278323405223385518748537031627082 0.01189647877835080472629423864685015887101616191605575836472455928 0.004990023482701343108066316557768950083112712204261552106639945643
            0.5108721338269479181126175866722272516977081138393446206801893511 0.2739475986446046945965028489251684588720756005088448248917814470 0.01778129517769264690082202670089767663610480947090845004246299868 0.008021641770984255490684824713091828124180322492371030239759601225
            0.07711972373841350856105639618214129714273781903181973749381187907 0.4250024379238826171371081100001085467941210132358963128951534751 0.4657535689928028273296414205831966708468141138015427617266780119 0.008082030034637161809698099707571167259598402486471556596976744008
            0.4238091620680213467299199482941858497710890025862370691583613234 0.2823639216825693336854594954085735905570704062210556896314089088 0.2900449514493057074303910425865442639067716549906212639123693868 0.003531008535726620781639203835609603576955180569386736502576193870
            0.07608591410095093618896853185249204926759510386794909638744344627 0.02863217546898741094231788757684185026429098333304451680727720613 0.4902774116131312899588260555185977795678524232864338906197225998 0.006107586168126079500016148001556701458653552522783190875165980740
            0.6716739686851087803385839955286164423578788023976335823832548919 0.2757199754029882544387580635575219484040866689150102450740482229 0.04226232206374509796861855283271413017107622100938154689664627756 0.002686886768605504434242120492841296239612393029309986881039349463
            0.4418630104182825160898641935455016514540291854353148166645477075 0.1909297748079172285338425252073296281695110641848727301941075997 0.1239843468356587925269715574040530634082497251236789894725178515 0.01558671521994095513543854672786905721681874208241277710720971912
            0.2207239209336195623258142682564126674015364107991323956456643153 0.3308182521297616705361222517372691553920861558008314967316349543 0.2262496721182368638557173351073985929759271509731752249764000786 0.02247856818312444456295274992185066373184210979551315815835380437
            0.07161477227838664137356171080675490319296210036600056165233071479 0.6707601238986118854089912894975735734551616574132382862571156673 0.01578689851285779328902279917648618199529558293920188595932907340 0.004327245017464648845673905108874115582370407909708446953620117444
            0.7466185290935817910753484970408712949994976724466812771850573980 0.07992239605195988693046221996883884363103299229854093141761724892 0.08401610467969754123773448050829899486599907490280794529112478653 0.007676675628908108321357879658734737651503682742036089074711147962
            0.2487115477642307659938658755141525846400881424594373688686292949 0.6235301946915855614529371596512685752032887728871805027729795365 0.04066339913180300875807970662683291018977562101442848751886673003 0.006424563042708325748292454215033478850749819435616546600107536520
            0.001471160462491308082917651279431131562239644153039982426422538944 0.4073874156344123733663868375045535026884762502563372552392055297 0.1689786729565603528560778861377011047769998536645334905573757285 0.003601562317459967167262200943797129124405017219430366873638121066
            0.4483099614645295547219049895775908190142439929273583269394125408 0.01124993985975054215487737182304597626059094607157430825508014500 0.02134512090587742544551308412619917888428240510746258511998368694 0.002107720562666873887434025650878907930695498593110155466254358109
            0.09075408475486327855787402994949413882748692148163531217876397277 0.2126865564784830235629106131383386180476950445491386756243493052 0.5875960803550138308565684826245589866670114110996983911194238146 0.01204019166913553568030077157656065059724147094381834448933697777
            0.2498077564825854589587083192699371752455947705973932620947587510 0.1687173926411141038116298660447049105496797727853529197853608281 0.003044334322247282633324739056808872886024273767777896301251707277 0.002928098778407453548512207005925826963694659386106132943863780165
            0.2650890003108020268466037026492757536891437036082190952346596401 0.02446090920024557944559573336837413199371906205561286864384368932 0.6969528239614627981149650937969539949586133362218099241751250848 0.002714314012829964830984131944894059134269346816339649610749815689
            0.8296381055235041555798909443003816116658955214864382068893630139 0.1260307006344858606213597724207981479782479876782791649984973130 0.02320780160932501251156541499315583917150111387194034459377055957 0.002684756137753261082880583066471019354097133836813797568573175400
            0.1169862835549642026728909644761892732947782468067971182906982543 0.2374766179122223908104493659487186656104439081207062258496768282 0.6245381915162592838984140624043018842975556369716803550453294603 0.006363048646196030544273752528787305834871050730574324142225700676
            0.2968471006335101664435893949422406327359400557539746994399196831 0.1530394216279042886998338374583377644122921734838112674306208260 0.2074315845044105665715213552141380934757688126114280633504963474 0.01739408837666783520407831585585487796559224119519083982951186863
            0.003182881770709352617703341053455027892844314353118491746868389311 0.06403389963259751119989908246347872777599742933577436129062712594 0.6408250890751528940698311670701670418638528392048153779646535023 0.001862832057203554496230399663271747176411100148137291504540614679
            0.2804408845407574360493706031571704991024212823768973967225850924 0.1441873868813566417943116991579843169160398110161338967507298767 0.5619411646550633189012903795050204150164194147863663148631269990 0.005334770581409029604220854070980391473053744320844890282825174362
            0.7981793257492078218261920074923218783226981593944355440759693769 0.006781013385787405838879329301407335723202364976444817521588772809 0.1200075477892553544751274519820433857022430231370316795892853216 0.002274732760379829991995299415960807398989678779959054204683668182
            0.01717357205938093414195313962723218303075369467135045341162303569 0.2957059522990602323173014830424524351705414200689197971720538989 0.3918113614251121382128740871711708305583405954294269889520483140 0.007532000482691122916718104158262310700018862358330838540687250126
            0.1071297965938100989271936662719345282955819178601755427198931601 0.1434066920642281426203528406149641477958715817899449451052501556 0.09928164344949991734955060638093234056511887982464541887729297924 0.01036332363272602900286921834483934749023058134131110538776605329
            0.4778657903513425784152871257627226074915507694466446186421960886 0.4839366262237124759700938712014520975131899160489570959418840713 0.02697574580265816183940038499164860659988087647234853478378428858 0.002803830779815179632336553240512901519435764597586782489752336640
            0.01460139598423613992009170585825448815514453035404630628568328748 0.7582780109016852158101938694696948396281503333343739542900220769 0.1287508002461318043771834122687276853803673005300676828439869767 0.003691537027761582566704959126109268969022390298550380439742975536
            0.3602666020503136396404219575784600807764585281674408651540918934 0.003320148291486242379682609245409902364931782364768903035704756149 0.2553679072165592784267897629439317117771613445600763886742213476 0.003837353991048999290504672678144928790554275034023834615227323234
            0.02083927677348641755323960879345527311591921864886392194456604318 0.005364673957560781035372873720020094682911914004266104953361179973 0.05072724763859528558840836313720465640815794694600344451371060684 0.0007076306195457753662216937027177517905345965791941344388889589270
            0.2440310209807262046988355208201154750954708657859051777015522740 0.06752345945836312779371965466751317486310163089778981815129919109 0.02763010816173695058737671352655688517578523994516681174972958120 0.005023476678531097690439261393232350787110725081335292693326969662
            0.2379039858140824473165462719174688290209171676771805550511893096 0.5169491659308930397005866285125604244659609269762544968117303944 0.1334727550729297179694602161134452151310536048395534151089529897 0.01642696904885285242867250626138541699434824960275484257492544585
            0.2869037403279551556511687553515358716111506922447243162219829820 0.01397718572484738763213079089623096282405073627566006281272491115 0.6017948089586866170329601921322943387624067858978813430437120528 0.004715810468336173195384363856489177852077668141185912637487363428
            0.1431067939118138673781374612068868011607481287171008231216248484 0.4462055540788043039431814263491463920642752295292752988900464540 0.02294870132979755485619527060288159868429133368962972810259091907 0.008536707351800089781924037134068754710555529525865881606282935921
            0.6054139224217080820661166249502329741845608562985015490892398279 0.1061964660935339328696146556819613598578864897427016920409353443 0.05690785662707795264118073719687386847834213594225112833967205159 0.01064805877869929619416641855154584962648552398453564515031267385
            0.1090838234150855869613736857974374195576254663878475832792806037 0.01911378077819580008027184566987787643155402088195556294566920304 0.02250036913857926720248965290013803551906627567498348166112061955 0.002072192020254103152644400596868991231155684215780823826562320083
            0.02177834500934184295456466558337532695567890081680554853866138306 0.2383622730401315712027451992482746259734592865580802756255387097 0.1011469516540267973426688918534287077956823161660554730896230043 0.006866650322989464002529186205640433238845774107824155847751157896
            0.4351047098560696466202204897346574526091776632999461940203238830 0.07097689018126598586875613354532323656551695922534098187625372203 0.1135131579774479832831005683273168810419624875936211542872842247 0.01201554417576878858806775209107921132951053554231715119468774326
            0.04924335022179622766077653511033432525580498320521761035134098710 0.3841020928485942645580471223477776967585660580504602068736399984 0.4270594057031102896416272222043626082134702232792367776840319352 0.01216320744670114041910092823351025572149627199098740605307952603
            0.02572118383562114396699644564173520090369592376132106505135607336 0.6166846016063564659320059268943698923948266263807012949568916221 0.2834879933758186300075738109345672807516028953126916215982035489 0.006482951298765949798304278685931518428361775795937650231950038420
            0.1152164498833321987681378380378884443863987758542930641061237124 0.5927527203048754628125580186961525878698483968847663321417781075 0.07362090466445473359040654527066162546009843470352423518038316454 0.009010711207994147810337892757280826585961159607344547692625116767
            ];
    case 14 % 175 points
        XYZW=[0.2314847027026112865391708475310015600098028690022433700039656812 0.6376915332767435108725560014399909053452431038396633273200515769 0.1228446348590763131603645210065262565222695777250501614753783079 0.003209047264666413820722797938690150976347568091778858524347540466
            0.05374740000890886009135366771876870159940165249617521003163737430 0.06528796392981764922578130230561144608074632041259060047020293105 0.4775209726418479486665573310473955197793950354806553213736217242 0.005690517460437244472040143959190227661890076272441273960476903139
            0.3115989758170841774160794325157245017027274877341367169488929920 0.02122667910168606528848489613102038081121291178948119704392336088 0.4201499476244786098863371284547706484791729796075843023809882014 0.008537575823578058923808952739455286191426477372936273420677067719
            0.02060353965947346527094298809461237201246075532246971869707410539 0.5799521781439219139554358606537173196559670693791149431570005696 0.3804659890132598170937676391484831262112856869758136328289696935 0.002933441234411125289164279587092049545602638348979267043929182752
            0.1251577417701054359693483417438351698216623057147794978518348559 0.8466536240681867133596292046276376586021496105415221992707942072 0.01651803668597646519734462730172325806120537266983189307014919385 0.0009789625372255211115446672243450593976571918885713404875915873411
            0.4051376144723313156173685647131445296578221891403890704995115691 0.02178474447088019510935701228270770667260266717464044089710506037 0.04092358747843775965209499018203672144465991525899800827888157451 0.003239300215226231387620287922254178496651293487805749650821625692
            0.1288586023375525130262226651746014042382806971575579189821570435 0.2904995416595343975437260314706194050119530160988145386305693577 0.3723360802526064726294122198415696259756773789356767045612710893 0.01687162629950521210445877834857127023788164443150400252735347258
            0.01017380018186520959937862048741834980817229261028186270665786173 0.3983185570060584789957574851958006183489365103089922073136500269 0.05705791322618209127716132860481130068407154475658369472659635966 0.002970290860719654964659448716621763014715727556701696614128653956
            0.3531857466510768229751930196421622417936732976719803507581979362 0.4088186256818646589951443108951341773398008304427638341617869056 0.008964010430432684158662725768964678323855447307308535644588064799 0.004824006370829444454453163566829312015837839556365583726016767263
            0.07687879661968988096323456587632791922713808121181377836509596475 0.1243032687310529988247248572037513598725608437976567214673499094 0.7392133170763374770437780943905782773521240630655510558472808402 0.006081498867675586276778126091494599740835343469605249328651654618
            0.1504940390832191201185598080342461844779508517079663908812444134 0.003508986593756359480291084474056465739132818638057723997972038265 0.5447310180290130509510945743479744224579002273026763789174110379 0.002397736776948589203037723146276533685473116932119094251508853657
            0.5610432592244580363436516212949126436528236029767056361098960897 0.1246103752155857658120570930170420822909289485710195610414972857 0.01947589532970885999876012215444743250957639553417084184913531185 0.005152923887856844112975969095485038020942842069949325072553730547
            0.07583140248514759132069611257942049026361638735302148555900952954 0.3525491651471062247453936661833351400626135389882302189330966430 0.02164430554794871025679285282933416118647922366417096179485268316 0.005766213503003625293570321832071841165474468524216578860410972051
            0.4129579494992036065284757297791763476265268271465660261496194751 0.2238258302895006879538630141562811439537543687968906386714373634 0.2880142891247817825502252214808979061277193644991661401091854567 0.006244888440029747241426710825063457959558927030796313310283703995
            0.3031876503493612326727528213223164241708576328939310347189085098 0.3540652116195430233549012732781651143257112564622372094514123126 0.06566552840178138070788847744140819311958993245290400058950106229 0.01157565944477449850189019230720925876911344973863113662101168713
            0.5296810728804705578244171000901572098749075747818229620859564529 0.005159256449414001885119462095036550028242830886996228103533299552 0.1116608367604902422855450634124804567481668207789065748119234022 0.003064822979221495294319240283573997801440581733832700753421490994
            0.07342956898875788059430549112965638747549497830319906459670488084 0.5543670382535760793424975949308037149682272571806589925833046092 0.01802831976264215520488518621144538476964282532591542485153684570 0.004618463395097198591917136770405189414881722760278920308142599792
            0.01577807209180517747056696656386017851167949641026197686102799788 0.02192428493679365558147399209074868678097387264812220187733544977 0.5728671765234393719015715567969678448781200253327491735704207784 0.002423571606923225336190105699039105238603044990520554550184325699
            0.2382122812603126609670304510969102953517417286996515406196196252 0.002246995971137464551217836242401913163308834822019225187811087860 0.6197777303998556649577253639570072363839326480306669114839947008 0.002400737084276291171721860276381250601906916041935040720763557608
            0.6446902576542700612442177097247746170135494005087289651410007683 0.1115014438347279396411968682842842715524119779574915872932522349 0.07850236275935559649872758539593736900617773168257702910831330068 0.009516101367396371798426379026067155797740498067822192057847188988
            0.08413984637251437819348466471730407216053774576698883304562967897 0.02427491818836537372008273995916763165616792346936746277131040367 0.8700960271068334505017294960653671814908099335367452404640524208 0.002110802222507716266675321685003555402072545075721559332251777097
            0.2466507485005684046385990564007221550553747148452799540477340590 0.1796293279219591844449740009651561737227416199432152587339909645 0.3904552293324727736875281056278576558056767305025945904601222751 0.01261312331619287782445252563008161641290055328874798355125185364
            0.02610841884466616867410972020685633404521954341664737267837976508 0.2652492895548902855684733682846376208298874867552502992450781845 0.3252997777309938591221275667662866467951141607000041325657947580 0.008887585184519669930499153399103894093307933866950785909982134998
            0.5531219661927219938933453739777369321743705458258548995650226179 0.05386139178094088197242104782155415887054655629943218791575174581 0.3665124924769789387750811183913310909241784966491177744796882165 0.005235904935636209420582448116183209879544388414662612818481198019
            0.7599870156846872335968242904114881314753505633060149794021214806 0.1555388209837724137190501917256185817551257446189525362383647673 0.06853631785869900710119469603859159351470949449495734701383787745 0.003352463358359774243802348709702215523490272869489952057689005862
            0.2519645435605217807011174826841318611180537035411224636028331454 0.5176347330744237305802396913634922352313750188324050399094734832 0.07113538184223079404362215542674467804098644320277201572553502566 0.01010161004207801469518449729322823400222481784570548481601585458
            0.01575400394586895856509577711757269858471390740977207380051610059 0.8817566508392838456088942190276485700634360799572320504087138238 0.07738846271281495043992056326533452546488757295773068936804753426 0.001506026399935976685457085723013656499795951635176954751991963038
            0.06768548505635565748382580548950505163181873493040085651254100577 0.3566638688934275250946439970514857095137724916475514773132120831 0.1104497711601330339664849682726284445226625499338882185967481477 0.009891747378982755114068984726825402566142332719744923870827893908
            0.01010874389072471129179648160710132723310440277075449044404102971 0.1359014760786207656480969358403650238696940989871983839492470164 0.3691718514744732486189924763505383212384748216192195221163674103 0.004376954011674853273135868474525513531061458331497025561591741928
            0.3243476019864201489065812184826653819830788323572145457739151631 0.02203287711722428115603558734064897031641114503204047347861234711 0.1299741638927300699140489792371515702858232452551995630077092399 0.006721396524555166026606775075702565186976060678890405298776827007
            0.06623227804367682053712249328127275221678583676514861480648259773 0.4227112063474164093750575741287562003940865245548539421913516144 0.2261372232864995846542969946389292715689070262161082225676339452 0.01347661865952106936089494436065044829335688424081672235596941739
            0.08743571068487830523804124892924476799466656275565390709141368387 0.1640156529458230605500930345170778383300490328610389851645173620 0.4103544579207127144723517095095500361011019996855836937189956378 0.01221847751762041055774943054721708107190289512758486549968392151
            0.7352761880631612630863031485601543060846303455494063809782736784 0.09499560727466244127265162795511061240265373905920275399602202061 0.008620077036408373181605631663416021206160572041150722195940944452 0.002492111595426449498539380129875651017775459296048923003631922068
            0.04796358179993310666177462949949409660553798272567935633667018324 0.7438684042197701369940380150386310058751557118574694710326903778 0.005985414304433554428376582947900341731412607204023739052224261769 0.001681981087161436338481934152044912746184609840249130354999521357
            0.2085830834781613766127714322655212498881455124813541227230737827 0.06925229525666589052040160801722481450063202297860758793816145737 0.5449968530492212633396278598692169848901133442726813564570352347 0.01211612616965710182795128792521659772489454999127878614883633282
            0.2344779503512194999302825721510372956374755816269304503591659606 0.4872538636109702874960572281954142023745418369158044529949773323 0.2531111171518042045654405471361164412129849055917160936380176292 0.008610774457966613952005074935676354622226429094291991210901276930
            0.06300033076304491158646319106834404980964233665164477607230796369 0.004292946744625914829980564803113359306467412208519498819817471588 0.4184562782709668646941265617057068168444608618904225966389674011 0.002536778746338829473466902237910051778794410589197004936981760809
            0.2934812539555347264596567777738893549207457366927085840665178416 0.01832748486767396842646461561867970053610539338521943821529147436 0.01471759519553825661786542485749295974957857459804853079616468922 0.001916481003239114716177176625996776152389791805627169039730190183
            0.7599329214651482764346027020781782107756391825968350617034697571 0.01984068830546742115547672371078498844463757156396118818554066141 0.2039689841975935145943061390875110016266641358386945039663957551 0.001943461687927923354357689984554812433214502213607912144162689284
            0.4229152513035334882745645039802459981084019340266032003617216668 0.3572968071805251911426139658916185984506182005943786006611351303 0.1534373188450013983278161735117337545202395364838196128467333260 0.008973267620948126324978485917963479571440884187061743778418221099
            0.1952996613722126732559306284078788823388520385298509452805313828 0.3796956166947324995442642194591186265374902693908342592051269409 0.01564023861021158869235136687310901010301101321942184396490567632 0.006413398903513022832940068619732476925435206841610676217072310668
            0.2554023297048646137782895135485193348388548369101687892960534514 0.6997029342998510024736650681014931013968827998284120404121203777 0.02286378414189965502557969445345264314194622122218567598451708232 0.003075516588208086318289585084084669138355983942939090418804174881
            0.5734152355148321887826601668129275639034868040293141537481274607 0.1221715917822544460738962797486736413775527620777173968680800885 0.2381074431796296765762445065974808580346002821682319572915380920 0.01042430478444785829727086292016134440995855514122227329509112487
            0.02064010338693319261637938820836772278254389157898604779399993175 0.2539037772698731893670405765344174273667377919764291105329401253 0.4926804694266224243187823594517726378935052711929237076435535749 0.007465618409086671043408051798257963916489022995716061708059199473
            0.05506085479561900441515980036641325380573536533168879256759594455 0.2760503046384493630228461924053818349036236982010486624245968204 0.5720636575532214626374651686655644363537077299520501521875330902 0.009072472575421035255530048971251888807082939758167594556444687081
            0.2857638474890122615662508537165732153995652106074510097662438366 0.1023609731683444620668646141412667765182942989908437118043295241 0.005651552470431050777020426724638035824411368676811490730351671690 0.002330871288559907245509190180398023913104240317532713246410623204
            0.2468322788679129892345874748199805809345075934290246223970982557 0.2188737401730881139071573325722816720512638696520245015247920193 0.2339990084555604375383361446904418572010115995569769534328293289 0.01669674869504859501941579124779368036174456949849452409779962378
            0.0008852535053425467248850044115963224435013570123560147552120269047 0.02213096610751863048459818877894819902304716184440363694368863436 0.1139054854702460853219657788450647169761358294083514032929617498 0.0007249254070854169798367568986845409060095236592846063035853098633
            0.02169034898005447241193007111414153445856712336005465711655670876 0.7720208858823528189927867831882546322166063516786215632829223126 0.2042914693477058017162701243816152508176856233678729109750948385 0.0009106711592202005077189578316402543760425869049985572076949578405
            0.3589963815557755470085847299817151562760037317912439729740579712 0.3143960859700532499393464994033379783841656707540871001099496680 0.1420169848605688047952914021213351485328247819589328690155475841 0.01200797228973204435730781572758044071883539405812133787961387003
            0.5868902336182354396358984095476487235697898796614696034866012366 0.3423822397563585954968866360464191435924681340092759700431720102 0.05247511413315759382448370357333148867935538411781059417747509926 0.004435179207748586589364004239688928011735931401312893097937725326
            0.5472218032725568900451444679029917696026716786257777790529605276 0.06103533137969867085889534475193106197303330444577847704151675788 0.1890806240955967908952751494787158199816331302218181216220131538 0.01059043447029599032668357771428414948684079782283596836290838632
            0.4565413948935505761280604704194210573119655039661499382566529304 0.2063041863690924586538831516392359694643796277646642438051175610 0.07890477961536762571350604938334735016672661264345098169492697829 0.01171496882034779029418003070183223062619835520275171567331358704
            0.3743237711564725197205922001149554410556276135717614970789484090 0.3079331987427519188994319940539507388732525769789973061199050500 0.2750824732788571704320497969081106896720249147488645157282457104 0.005999929124804787503964462693284794771578881807010588506910791403
            0.01510352136504266742949842711565571234302036201287162301110131289 0.5141323840296039614296675806379530934616192514622083859780222652 0.008799327336764804567639231331735974230516185164554068131900487183 0.001299081889838198467830995157492381047220106339319188429673773874
            0.02014154022774443852870189289036374941596728263099264196458901842 0.01814125293395793353923674503671977380859890380215100054531281581 0.02059054023577482413994817888346616194252408902374511868903004608 0.0007772739079912418527771942145737521984902777747961916635773811379
            0.1626983910802257472304580272923438188109992817729076743923598323 0.07029203271114865537767527780329766245535362311109712165509394902 0.02519989831006128203460896586704982696391760765781233714058964708 0.003993970114980817489240074165726414827597846722127857675726739328
            0.3228171731581155133222215427689562788667613943224027129781628684 0.03885138946160934954458492262551001241461775373234227146402916853 0.5637859720299143871434779654538583589493051248096326519727846531 0.006645269073958711695438876973994014522476778366168409147534658190
            0.01875160970964125790444318407842735580860174357795076019937320338 0.2211914068552564928619108798436824799536011990568359427907444141 0.06295532354805162781091518992520496774433190263731837865752144652 0.004041581908421489303049312748172107394156511028851247457997591494
            0.8440232192908454874520439734958063206177625053437014991439352586 0.004159597347132450619464532294565804976491485758414830017529251255 0.009394838884035023208467171019166071896778555837760476321799085934 0.0005708294490749094199880702940799153211436829731007883444541978987
            0.03127092913102568591452543428465873702154127635890656409280010923 0.01639433148288307316656410346065216329455454104602179004386865743 0.2344913660312879464934007719751417011695710239097542932887028416 0.002947950045411388182066101887981181503270426514521391241830553236
            0.08047410019658517046790459140880930412247419341941752084458996620 0.07868925667574118439025830898344483301672778047490790868077201292 0.1192894301413521315895535349917779606942629032012840111045016436 0.007055081768203036182958693973882373175275934694722258188762204179
            0.2854247024134165439835121147475434492245392935385121692289423977 0.4926735680301441248462941629368760133895538907709909330897089876 0.1396479572112513919132584416395680195412695231838808959223851079 0.009610925858769904963190282508563572439830677274151563529941016167
            0.1246040608452571716193078893023540410329625983002154162316116105 0.4561291972384730569158561351597310559681160653341780948946771527 0.4181733406040197367952751851840534472140748455338481193980785934 0.002484378628306811531888885382029473768352743462069102937144691020
            0.4659538658257636048036344377003080931459384859854108909510171606 0.002842321594193621843652397666063962832829910491262103174252015642 0.4326406094471421526500186736489934301736446416615478528578733997 0.002453077512692304600028825559165720405709658257315357180090754920
            0.5581406473971365248079944971661185835856138285671569685765980586 0.3767195758481862382056615094777671322999172573614330760025299882 0.006878085449593958778799458717253624435255930671081696923327882192 0.002469762611527706404421646877829634561035145377070778570798590791
            0.6789111702013534445504982026239504480407824406852164484107565072 0.1031613732192879965977323481482829264917572481744054280418150100 0.2171630311728097718147066925657456822374683908949376776444809447 0.001979803603927775734682865625275515847338257602372739974704425091
            0.1914051925417945409856789012475828200878185797377793475541671450 0.3244252655056700711942063063999956903949895946722927004330710835 0.4118996074857081016378628613162600043091026443572258764319821497 0.01174163329832997593036231958411340646447785142262849449805031479
            0.3597925630192762744027081693635335016488970531286206417877787236 0.1454806934669920172530177725782334068126467546248880494119688679 0.1431388949480785770355025484386760701016346994095645992178162122 0.01340969904749685456762374475331161050932995255999775137555530656
            0.2190587145091670064945358080866922200648581707272656367137995696 0.2465631457054916804104650601746177026581295664755364934704542015 0.09795829776872093279022256709452698461086090509323787997007568301 0.01591461248228389314027575351703331996698449270071275318989166835
            0.004897292183583327776615324840781002913862099330090881046255143901 0.3867937328807340500758580996933174028773351514839214441487641551 0.5193972942378433879872801865782829562391206329270211438167006709 0.002759748200087773845057395742710088800226668794434635880793993314
            0.2908637587422787014707775927189270130062733480732007280278559860 0.3033737210343642248829240362203437792181328565345229126188343049 0.2756352097639389739453896904469138918302623429722750461201076043 0.01349756123342413844648180633371907169109700835381276136278341528
            0.09547399059950708550396792654950259498334129544229748238241081064 0.8045007568505806052306702907700465876746775808667899549088033562 0.09182231466035314558848238519497808132283826620426415270588653791 0.002027382489537049589752021514971347349623906765885810466956195130
            0.2336668911760918753539800615226963805127470744145378143177385818 0.05432265615734255116950472267311890439316508789079125002726111667 0.7099729169029420014589022225013066857966864030413906218502513105 0.001430171487477176269651418775529923352351007842587279779100685237
            0.4545645719012306869258471046847395306229220187519683986727532595 0.5326031238265766897934000376074155416645347783956768538225129666 0.005804572119455750662159744569926630149455819200635612076029431522 0.0008790486280207989568781976426015157466101241763514044175261648861
            0.06268640081705557945099498199648318093793352692314868092677463097 0.7551016003109543882954505572331903222121657782092234244632066027 0.1143367001062208527659072399358392898873560651610585923013657915 0.004587297128568966167902318809544183802614119056335706136601627366
            0.7470346142203055724510515411242607817702477665265476019449346172 0.07493293081622976304064441316203224879886597861216281419772530291 0.1226964622588806965779695003506257886453834990696941998691410199 0.006500869571373630142749445622170307083946865221894663711262338790
            0.8898592406428813295213519698111034979150257703908909454089387298 0.03578080188839507263182857261213365157075815442057805748902528510 0.06937637580394725727378163504172246849788988971424634965849073402 0.001020201295650193705956255570846906366300280832341508087710260595
            0.1416202530466463322490555774354285645143477214429153537237095103 0.02093690844400411257659704432298143916457365181768149627263543058 0.2174199164968854976158908338169534108939827863073558092736027648 0.006983762548951579631599647400477132077674763230891302414414682435
            0.1375717010361890375145071696215816363283052897279221149259716178 0.4360189277483148024792478636622396172570351681727280117441774878 0.08278405324279252527535745692541531196698247742261216320936682548 0.01113925382685265382445608256554520926062114945225966886449519645
            0.06622064089750860037169762091335598564446349917774108077224096303 0.06060862864200154054242710464886377144701069913525647445040482247 0.01852539931055486578946365283842588160057091930827169898603635890 0.001707937705226536989458802600850068146669776298474994402477501566
            0.05305500312146268675388654503001229202619806020350714946684128931 0.4036072622705165844346003778849397741572929272600036207873281135 0.3752073411997015893492957859271537123277360875814467070973362022 0.009336258089629736430097099949336673834226833366426843950852939895
            0.1110468202954785479187530551430156208252930365276330672137017474 0.2690435536807952409860939976949951710606595878546533115951553779 0.6015552259233110185067597191498421868363768380254965654407216790 0.005816842408930410238941491823504635019488766984570002330459554097
            0.2078231193525116718302438415411230717296480984989060953298142390 0.09131888459571971487907980982810795750063505191023742661930500832 0.09074548267021759241363294178042119102949042046797239942624274499 0.008267796368551581232955752590315885657222314139594391454460695456
            0.07759468530949464795357143995390135158546800847896091349553002291 0.1698109254138963385973592766605168943736171163976738518334081139 0.01682504312212100521632939014969611617421402322125917166620139490 0.003653888242387943421663719629092506457081407966308962579150724008
            0.1819940111376950163964636799902335340673763149518148403742918863 0.5625806801708493599340344470616240395528559147661505094873509668 0.01725975731022311445039852820695146468879446796198251083305491789 0.005758754418573557594200873618123569561466313617126706349003110940
            0.08099934994163246787945686246621380997094461203449236768133693766 0.8354354774582532967353839441275668693548229656127844655439562156 0.02213739786955460860193892545161661363425859385182120502954100691 0.002626348924980967859969111610778968841484716470146844316683126488
            0.7098831305893828612369082838270160440006840774347682296858824911 0.1953464467826404104628967190368101025923188952892322421420630086 0.02230507613685139212454387502754884894878348951807244658627225816 0.004967260591914791326094276389672034450056289812167506766231018024
            0.3944986311068601756462904674362794884166806918090369095541922027 0.09026813026636237826109908564865740003724424869892028433984641967 0.5007746609315956284753403328007639237843919884800882981532279364 0.003821417400776950545895628474657080492880080719959926877603587982
            0.09975536322594360677756458443243298484938919941126258991936685878 0.04092178732935767513448712644846574404435909350145287822583280572 0.5825559234390605996731745104321727845850658409250476902817654664 0.007051230206178890299252173310309942330953513714144910937376019057
            0.01486519419108016051732839764775770848398778809545708973439800768 0.2871515162627684716754527597679555694981569691614303723290061905 0.003465228059210388237840215163355067499991437500061439359846663478 0.0009203024939728352025609850831658521736099267574573715309639016038
            0.3505125568959872458124609271735394588729222840424231297702379109 0.5433752213958881160932213681561826844543463928233273312074074307 0.01728967613329077949175285835530994934394174386122743198924074195 0.005253879413566863809371671729999832273918132040700511185365267752
            0.1972884348296483868057199775789126580839703907115000280714773360 0.3833332352719716012362790314617812131330055035863972407542151933 0.1812317803197832331428055662714558944179367462914819894127752103 0.01558184962013603683902318667629300586107444274661590308297627627
            0.5996050186306239735321351958337657460620812658781124425798066881 0.005683775942445210052090547312842000703178788647283240628263350424 0.3777027932567776124911756531420617179283468917892091245343427262 0.001054285844397431694382416568252171828249094650287696254651083582
            0.4570825410359628835392144135870199066757008355334634183266759431 0.1907007291912775137828015485636803352520475301577423715200670642 0.3396826550894466540789405029540866843445035284997649338409192188 0.005371238477990230285869728235594586690925954635120078624849904521
            0.01863759805270415495524539718583920022851149633784579487241866071 0.5474922147801191442310711022679765846405671712279726277508522642 0.3316946194872884847858427860618240503507468139531585026577473263 0.004702600220365203416342837616432432463697310938357742523516867569
            0.02332127886174483780381119324688897440742716610706988389138542811 0.3873293157888394124639372654374866625617135938222431076758264608 0.5734406116589309346334477179884143429834430392859087162687829556 0.002748189642414156038890523733402320669276314058443341377650994043
            0.01805120197280660116882668513956845888591181899585226156438098677 0.09709221408848583184521160883088032771589389450596914994722948087 0.06057789937936577061841729351273711552035937945601088692660841610 0.002582579780886833126739689398301720565071642700922623487734359699
            0.1951333137655353996713754358730829928466107698574873922210167676 0.01401016362648966194255036984102320496629279680547757132128102721 0.08171990794618500768098953619806890262926479704383701894771643067 0.003494490928600753777896234229655687690021741163938692125385686740
            0.4956908141277186648253253176128974919668736957549526224468972715 0.005599633190162136240837682340546878642554699463928381536431984654 0.2580851699366346529561711978992380906426730170864257839028479082 0.003870086279907699164392379895052526068533351696425488881280604703
            0.7112683211839102321163186094306325799422600322959133450851043160 0.01721868530260947646405888313373909878918123265069086865650780029 0.09934071753567885957912759837239903471693964634733943853180674189 0.004641654196113939580730858236850144715466035662807174507077306249
            0.3458346065560247710356094464142383743011757410791795067178029158 0.1441201027434456164049279419358827543009828709627983466977467651 0.4284500724532478057501596132589898458864720405853603482239892852 0.01268575402497243054956339318814746171387495483762349567421638519
            0.3673120245760062095785575198954015096767229065777944847225715369 0.04627229783561052790877718886709898160993566402824033377502507609 0.2297924996828847917548078118377367584804928059859911237673751207 0.01123947660390109525962734089517108945118318228522673219012098409
            0.01903755588606138310090453997671087692658118123352300635338817333 0.2131430203589363906775890424493393380351754922293361108915016113 0.7448780838668742260916506570689698548481347523861895035616524426 0.002718905668101607896545165174910325931878308068755523827332575329
            0.4735713684036109876781803175518928434890484867802796332409357920 0.07372377117529513367692797890449658440727000358897700043833053897 0.008751987884998887418399237385299718398511104338186946229086084072 0.002731227599726235765553548824515801149647690581064089134237497998
            0.5834693638119762349919387422103842445406527509399630298040754253 0.2375075859771644688436903461562216382269766105367511938660139064 0.1587484992516299126961741554920776803411361225115254255742271747 0.006043859371357759149347945576647442424171849755752882465838811966
            0.0008004278875019239712026642128760737783766778734783026878586840555 0.02804252053143297059162994600209012035776773576820206690450663634 0.3851258757807086613516398094452770290118305833949071433485256666 0.001336686199908583146422939124291992942537475530401934187106339689
            0.09312474570373108230938885337153468194221052693420644229613733411 0.1529358453252343711178527676524076249379863533109954205510475340 0.5602132100631045540315025082969977273901680030958591244539675873 0.01002913387054372208966691673280964987879970529684005684803765309
            0.5397729343816763239556725181928899142056613762231547363687393724 0.008922659105523583784895856943902784788247857819139712992608327269 0.01430094277717293432885057427176660280007305390153035013083852626 0.001159598259487304825707743090881320252406010935680033350702890815
            0.3943801800675378512596971548554576858172801402811189097138096693 0.5107685757873629608458886958475957116193511117828739854010992849 0.07121177301957508330986344330045957088509442498034076732924235144 0.005946005250746306746024207737502046522202375449100049610800975975
            0.08772162054775044121599156933799151566899700665433707215221233295 0.1405974234421546109039079281331834848332943085096055849555722494 0.2356483335099053110569102577751195268755112023570777358509336289 0.01012748832028319666396010327376652167187690873063436482678847920
            0.1858895618725557720677098004314908835794778573764599509747163892 0.07600316226682597568803563066696192462121146495634944420080341348 0.6840670686283561282631264924142994634781191988943757481547831032 0.006464393772438156218121101649791929162080654044480325985799532933
            0.2111615107035253068832730727260781154116245798903731604862053442 0.007315388256264748994855385831383289826226766185026770203652408980 0.7519996039696799024220941722342727604356420824678190099394682701 0.001630620277248115871109326271403610214390983874400758961582003189
            0.01771817511658020376886523149680854418821705175301176757662393129 0.08397767735618085538411925075173107052228976735402656803618098261 0.8931734221214671572397445378905866099987748003086010799371295731 0.0006417974043329464926062215535739526485866775482688605728496613223
            0.8360821766370258003295746846058336867424670978371296704036110764 0.04475731588255608778565626815596809905946345696920614700390855376 0.03238800087675859006686581226128281660985181323193252819334300176 0.003182224612790754914730389271543194246580590369228369493206005986
            0.07889639892651838937953372259898233633184300202171434471951408697 0.5882508720185943806951144198654485818914036384681514644919660289 0.2173857105695993831157163503798798372373872855108171375581717538 0.008403884632152262523354209802290927834801905190718718368015691031
            0.07237754062127716859194959675723011126054428744919356421619505810 0.7026817247528738509888282460767390015401820862562458135179116164 0.04906126305687026018603710848392555866460989348595093189302047563 0.005072072661940632529830443003640342616849272941218242281535088391
            0.2707821307505513475895824619012683213650397386693496096269942677 0.3098946836718330044213385470414163281761575619819415589650460837 0.4050348872368510716722468951589640173739314861376021490487690591 0.005992400507004329109448906522471629717493911369215080384146358074
            0.3420525410028177395685109709675132020188163002802633728938278873 0.1037483935093050310339940466950462735910768651662057630156121003 0.3167826895604702341512243957232278145046147796591994471235636145 0.01192303895197231092809151263706414661571221240850401500471684848
            0.002397173284589438402189406689981610367789696392896505268747412070 0.7021521850192977852492903587018858857275072060032403408797928570 0.02327862030225798459113107105075271227957507604944413995398783512 0.001125650358963078084348301305295109511804145713571100962804283906
            0.9529047174672606923680111853583640865354510793319630570444466658 0.01168840008706958390132209348002252561913921776021836960494613946 0.01181517115627594716534408097120194535171355928366154165717624819 0.0004238734530848500084863078105498810079964425544909438599204057291
            0.05941622765162312671496346025105600334785286391462407363198959963 0.06506846662759918553405511313133370985431577276866131745377491251 0.3268637292613884716831788747913261823235155105378748604252395838 0.007179525130145103298412247236468617181854491185172499335069824493
            0.01151205876693561271864957391694194277419867754005014292522687765 0.008017383319033641877655276446442533275097561230499880509348246604 0.9681102844937589386849032477326345485097772328386682090614945560 0.0002263801126655884524263872935905454336701332752589597190474664374
            0.1875651410979569257746182172203114113835324199041897730361974022 0.1048317104100252195053808730437821740302364035023269425155187181 0.3751512017767904847433596177780817160591180083376826666193971372 0.01311743189849077283677130362352969974666441316382421272113149263
            0.01318766483169567762024860670810334521220887931574528711058315473 0.8818984436492212610985306449569180930057134886062328785724169616 0.01507511329821436793485036575703425936238110414480382375953480095 0.001019303964844575026836172835391045822046219141176579781958008575
            0.007034207706134406585501714091380373442500596542305017899500470754 0.1779931362112384023294324564855162433738337713595912183446081487 0.6962679212195516367824871677596392506960725751022615842973985665 0.002975696248447522592318745176054151054085909319887027652133287775
            0.01671116741330892898320166463487086152651156559725506401222153328 0.06402153550351744506268814058078919032438301652517499704885454144 0.8588413085668976519968041317282833545201603970222963955638492402 0.002246873883676256712118646950768157544809368426484275980610128209
            0.01486152977712650883645849870629995167463088690253842719551189112 0.09907818547701397428412416066922848906784501595874358176872501896 0.1943014318484946378919360303469539452275271622940930023780889446 0.004211370362897247729585119104328158872662036852596049691653141617
            0.1898562596264397833334443576475164922368707866985226509884770991 0.2132864091053829220719860085485253857013139685747261900823809032 0.01946397937078007243045370434056738173070466532873838996327738123 0.006352589300459927726906955647959048759211385119383755462337735482
            0.2466916518816815300115226358170896408492761673190696407344909103 0.1703528045231606720268559145374234860453435807543989457591596426 0.5633671835135334646183571491285848999712519173307505899780998192 0.006222854254200033858324503790094286042174405712628417571661463466
            0.08679328581400223049228719914889733474718444339554128019984297265 0.4452665852152744206270862518939506903783882258444578700323144486 0.4139939813833747809328285946564478123162839907588940897253240480 0.009150770623039259458813579975770600511889028338317655821005287496
            0.5069363956466004560671033185609990936512466600809804885599548786 0.06224133860214626084851614479657136641101450250773100014089016878 0.08725443137175723116830337105221545273351464138010070542532666385 0.009418904950157660552995631250457551639593955172864858082123942535
            0.4623365426901432744152275675621422232033969269395076760838503649 0.3671926809965880257490437974871527073186803301324434630380262030 0.05230373814843905914510200651873933070677207808165997162546828335 0.009238549230940294905838947239241572950902453367447010070647317294
            0.7385444697648062317607143307800627725004898256388811961489765002 0.2532890523964282151828412647877583385051162362614481170206877876 0.004634179715061313761470610467242852697098320513461574684278972975 0.0005757769009885353753975056858964366365283789100267497059344461653
            0.8738764796853344590274115840868970164437217405851504070222159435 0.09199930246067476135535455957707329426302180713735678956152932446 0.01114533175550860339844088364424255557454238228623858401071516066 0.001246535399694325427996874203956082663674187752891081991930098530
            0.1260929136049946543113341724858210260463188128326890253665821946 0.008998584651132683537951901559848713717288951573893969642563425772 0.01300327225752165880283067416426083446659972246548723494214902125 0.0008703415509664026974368770267013124825269077456100931895142401920
            0.4126513207735960593620374395920467743753658804846454354359464025 0.3925404535596258732769332590977876769640384625988314502613305535 0.1903487831191660665179641915518554945583374007902111959223027721 0.003438752780477626014752344915105304428246367485304091565080821485
            0.1679734904063729058128743875135851201306194840958162190006243713 0.1864795361998551996244047430114997471128415427734453549156447539 0.5484948392576974190295667645383647998554481081019295974238858191 0.01049010477638666279961032627174115618265577050598606031185934567
            0.009695819510049381969805899332806354851936726241690047803561049925 0.03603124976693924377930368463559832583536415532916804759304878527 0.7567017873719755997484420993355619780712976394791277745337194816 0.001900136613875483167619296046565119992814271952439182892030068422
            0.01327832189140696525082199755233902607202061443845309367383502655 0.7237101625983119796955331626891757937364357845158556155442661541 0.2048936461930627715297521415661846427947468722893442133963013893 0.002979821632276385651789484836290539948380537768108552705357314845
            0.2812276304402152633690619310283504744559053802066720365179546859 0.003258627391638261690980655061953085832265955483284685197079222801 0.2740563184562950284275885062939169906322650524346066365856221437 0.003008849811209318370988099179017989769722709248298282851347281037
            0.5785665381495505372962581010683439103538291860873882885812663720 0.2223251957765468554946881679652422041710487713330480283653429113 0.1038458975160918784441222892004780191235567393337290530044481757 0.009997290025986607297698547840951493369469121567659611306591958246
            0.4470810358244369698264215062486886602267277679112451588029397765 0.1717587280110997296236184308215062599090780107052538046709297048 0.2088763118828320322801295860570155960144532186301963433575628604 0.01502906673467074739181755307525968070961154563154195550392689088
            0.009343096111758219905012186471296766665713395295086030046997917250 0.4095378104780490866531130629178696509832840107303159323996064395 0.1796558059587851097096093732048306918300035610194557141341942112 0.004292665913174160827376024912854205767157374028817557994087678326
            0.02424645865058567421589505053259545799270964733935718976187674962 0.006104494834106323706986572233651211727207799175965712751997338570 0.8779264347669467780518108785045923203529318272665304923025848855 0.0009217422133469898938174737499576783151620228580357340870066082979
            0.8538367728732515369233158357045118584629437880238416237898140250 0.002927482729120339514811431439186523051281136595310990781547049459 0.09153720228977390442435947938310867550066729081111543978290109390 0.001149395802234229851598112825872883426813952391137885526016545423
            0.01744155781302475464157661352349599862508837711299482441104261311 0.5988323651127695675008404263333837014879782571683448968786716851 0.1804815588056152122838448924253929203715021280130491546997019551 0.005833241615881981240584222294478836019886435251995068500378036088
            0.6779235131450779431802354176882770419065672965994463551158484882 0.02605714313487224687871024219623199687745193705435132196757689720 0.02395203977928549192817562679157469199607742807140010452841872952 0.003562713398096093647702480841404984875340889335878109018530041910
            0.04083475364420002474926392793605621275852756913980937039401015941 0.004055697403597718659329939150734778963220614726724057320088123214 0.7029412784895705467035102051158933703015470740561277809342960652 0.001388178692905946956362210108407227313292664793984141478937790843
            0.02405937647389798060097946601644051218236247948539249863157241456 0.9510190140216096093307494551675987367289809828927843906637117194 0.01343083570620265583689638795521380927733856222111816229840283336 0.0004546072731766212492140170478840921279303712194587809155810070449
            0.06398635292898189681003166436860006110860104485630115698928652563 0.01417431790117682576127770650061786877622518951240387267164158448 0.09212159554668823942085571138983670189443728623535688695351337369 0.002491579627445117364508095637205062706528006194476526645309017216
            0.6513080845324389014086172751303058843732663572276862542673215822 0.01934569657295847472302028196113438148272774251325829840657178755 0.2359808047194266557554432067495156421390395408485434512966101809 0.005325906799754706999839189597727164279860843175791542972665926532
            0.01323051479580047304759897670228932859023963359299530609987762490 0.1157258521405562713033646388270145489541648511820619870310901165 0.5664818699370886681589110299404145947010019656720349417184829946 0.004582119748433643425981554047368699257976203873038516323178279984
            0.5430627677155099452279852473278327044430740460813671262265856633 0.2534804912905675427063877001770242026110978728856125281688693696 0.01476467912911265386921491533008591197045789123118641328754402014 0.005731749558969351806664262532512259146998671898913222872461374035
            0.1725367552869667003772352714588486403411573606187576137177746807 0.03012338657718044785744292708227565696982492852283816993647406834 0.3770142515015193273842054970784055928189449477736920878391277518 0.008701992632426948476684887981408712948185729513027031940256457378
            0.1106445695765415368335905169147326159177437700632586010505023685 0.5766097256951799679753587362370083269461539118144445381017189399 0.1114484132865444247232567640937154753262560841036003367839337242 0.01025963439996476995833233612720343278294553349768618089755491218
            0.1022080770489681250425258797492710445897982647045053890235716606 0.1237065436616793057094096492326125740166606085181185111794166916 0.7709946467455654212765623327912066145472427791347566854096136640 0.001700893224647546719183214621110884625691637862707632337314537967
            0.02642801165669761112324421471667905844648835977532173883771949563 0.5626077365057523412635297690400968121958778469721851673855924020 0.08085519956984634448255072253642572779241764554707494707172799438 0.006414613843541854181456523194633960138008106164765703493276108326
            0.09854031376617983783474175903476531463152196907637287162290377302 0.6377645129364792195298628002432816455571663965312250677166982353 0.2404878198640394719071349800474934334589001221040521551851424893 0.006424323357279731190695697413472729392410533449714418705909455418
            0.003341545005920790337300887915377391155717285269192296006592983855 0.4204189403314316728897173411961571529373517452114337003370048063 0.3327911262775774589478240171329636260081433538558731881152103670 0.003070160227491686114667261952157708213251424034290870178901746946
            0.1721096093997542189073042058603054096732193239527488488800983531 0.6677160501593694863313821819458953925841353349008936309055555414 0.09451247440340891745442585235593435825835608337235273401231426081 0.008281900546311299520871377047508450550523336292007803374817105128
            0.4491638244881406775327560686837650061782664666132232330374875686 0.05384791136846416822342362239446632499935183044874932570989051493 0.3685725096493866358988228179684726269920376498432909202423564351 0.01032497015009840592490255987974380881064594634414725720250395367
            0.02023613677961148932364751991703639466378457034821241856636420386 0.2350815132425931811285793897554108571619834669960799671575963575 0.1803604387580974930410899534429215650658368746881541440332814936 0.006827565694858372887051605560369888569446102035535642391613979906
            0.3647071699267969811879236154249646602133923921821412879005769675 0.2275205924574069639225447283679824656775528173618389846617586535 0.01766362689784773908126284712819042282784911315028279476879577084 0.007925164754969139236804255550039084606638904405720528390640789743
            0.1665402273109914915482614288603517290746974756394351736804111311 0.4703859375131688602580763043619124883524468931982808581082572078 0.2472552829565215275693350423225650101735864895678667084321351914 0.01309999472448022375373195863922065056013824963648794247650230577
            0.1062062171042790159708995959015701137121940849958117645760146742 0.02429250558213802035108057817701446357403134463969317802799950600 0.7487176500483958494523768373029752227116984664494019900817165949 0.005314543709674777986446308575979019980269103944179506301232602996
            0.2111150628770547758663197118370102320211610273817872766463499016 0.09722334934421969000707250537441705506415557270865860308606347605 0.2095489435843953196528520033414575095842154478876220449410199648 0.01454951507664782219468661270660838341869000671841447222736742600
            0.1175572629137547492915983068973341468053083717311974003643663967 0.2529758209488976461325420075863964109392786215829443199031737315 0.2296878335128088362428592816076429269612812596022681693623819974 0.01507785127026341929591837632990213381273429412800991371790202584
            0.3980223742151177638645444168350113993864133835350705595079868996 0.01503022817730537251237549055466211621570499060172280462231072544 0.5704926874223032999780809014243085266259998240198364960980416539 0.001962371338179456878142546942424969151696200463886317072164711212
            0.01387285149144773353251231413174167721659354114465399551761324979 0.7660133000076804109878798639964336728742743726621969756912608777 0.08124349122601198205900426245946804959618872953272511490448619263 0.002960614701580325656043955722844998638684195048721544871992266794
            0.1754777884198104311439917429475074559906187498146926560961283807 0.7005469020980794975660989316847222900445647290850673009566269145 0.01457302840566315358901677897314735030495361068349113332728547509 0.003803527196645015684017113123693428651935334156494620110088097278
            0.3388770319198367534769675989037017728855556050730628860325535262 0.1101529021799191763469956755957881786659888577137313171698691449 0.05650898898101764357558122608112667357293875099992384666433227900 0.009138114729575373859487331065607106928871889199371282650811172374
            0.1035975611172236025573028954006360150776618556762367870997513150 0.1982636933298153572912769044696662072579721574689892981074514755 0.09429049437314195732683489788243533373773071314008085337027000484 0.01017945432280836873285263870494552664275546048504970421470008346
            0.007198974655729548912974042286095842645758619230461919720830477544 0.1171560389619699438483376492329552522045104063949673965727705087 0.001631602560864736844270294296933602023273941126972864038754245161 0.0004146958482361182209562036448088416362171232324308515583945483721
            0.04431560522219406050171325672427810558590236894992758610522486439 0.08967793076933143408115602690588333715120189683644969890004752623 0.6958274616835202710956122008379516945292435524373068858756547356 0.005485971689213694416167938732986393557024313469881130772119975400
            ];
    case 15 % 207 points
        XYZW=[0.1507547919755971307323222521008922006046122206406332528444495322 0.03442732355179202097144037228771538709475709381114145541396434481 0.3603273979439101724249110855554182765500340675751955001037186277 0.006959486833268064266563106746990524130347255195963144346521062045
            0.01881596803556035463162088708540079578809391957724904843138284750 0.5673009248072461636418980317950378062125266763550228339399347855 0.01173852471927616339045125323518624023455020668923778547785265806 0.001653232006398001509262689818094788496922000593722077098721856691
            0.1315462981426609064062652370873094585872092764515907773503400041 0.3507789949506376903001174065578887290403929507935095803340322288 0.1896276868156402127194924640963654060969722073292125480609427432 0.01019267759952557929701461379797706706230388064193357332065544125
            0.4013737643147372729742812356981168790080688032057794143731931547 0.1038334984320445945157364256768962054468602648569866719864062501 0.01033451277787253766073893736103148346871593520651195642308295730 0.003625695784748881093022730915809869271561218517225962744554440953
            0.007309253975721447845991577268822397225385997353165162360712954257 0.5615240391183929744040979786427374244389811792374745179931577315 0.2567153080397775108549280541515651613429634965634220612356869904 0.003017484653076547017621013096696628804148227255584775346441724060
            0.3347024093823241244609369099228269093693248989200486247173326943 0.2041102760068841353761888735696540997462723191457921880281791502 0.3924504627915242045079925580365458812297522995059826541206688461 0.01019352701138787094029839398247740654022303480570948605346327085
            0.1490306277313417818610031718321420995284501416541350253126248517 0.5143385518030857291749408471402767958015600934763244521266928880 0.2694241053915777243105022727596363369457723194259540877451021311 0.007700832793907158789966953582453752285923771805383724520942336691
            0.4805149545889313647035906467121547921452605097499816905664838647 0.1353047347631526716002911892098343721499072202034985486961852471 0.2405485647741998504800630766523767413780495134872646960023147168 0.008279164248124523483249886324271326465844975921375984979292301602
            0.4286096107108586500201233924394721820192853126874408814779281829 0.5463646505338739983169995560610994703941544705692729603927161434 0.003731155449942276110963462763635732520547611498233147300108774643 0.001152975910953993865674357673455012342377790434506989931292369398
            0.7259297304410000650349314580350532203211141932590884734445662477 0.2277614469250161161048322380345275004577318127774527294964195106 0.01370269349031052752736177808004512334046417165876779850431312086 0.001825998177679834274821678302840531213058580215073645352694657818
            0.6887303948099333307654942526279657323118123345206449069199772793 0.06889110999111149890812100094770782012782445102894020156869370205 0.06764029116323667346578532598529062313193069555109278051213826343 0.005836648199004843000485153919734868566425695389386682693402543443
            0.1512475431736191431595520183910523927252671267174843780074134327 0.1927510271920727162301981670565180333508029856800495470080243703 0.3523548900932106840337532910142125071628161157850250550901145533 0.01533873225742341892523989181884730025596661477614503691056648770
            0.003732342124948754734813334603635283190347712281148843202963832292 0.3715937177970797417338582605334940926618298795350895316701338333 0.07307489953308772814814641848840230465037254866092778920443934750 0.002057356475871561866673555757602392096369645701728506989243289036
            0.06159547956574993745772430434098461363850536103924380546743672925 0.04636772283903396745595023244601205653658180383951124171910636411 0.2451019397781406566542480000593695040892996017816452857093985048 0.004710318251195602753692703258260619785802446138711690629227588348
            0.7389490095234590781018513348039369625119842210198338593519592415 0.08136265150122574485922467937046661794738554945890834062631376722 0.009865683214212634691937678349974060282175099132246641782630076059 0.001781962035686336826610072848808395601756680374441556504996086275
            0.9458730128599713495079047623561044440495693892373359751562138119 0.01648072611286040949740668230446116582388546850873391648298375162 0.01813838942008068817393829999969732869585660302994407252584606582 0.0005694019539450177595948494498666442620839626803512265283268420652
            0.1210381671421952152833984830705430850074100875926729462056091503 0.3473112010319771224424114431421928677451883748931373592572416154 0.3490416463811567059304469305094865890623615524904741296905820159 0.01345510486903877100808953241587378481353939366369845933215516949
            0.4329795945909627498935839789866176155758779614377756722545299596 0.01626147125415919231916587656002930033054068397227895561670328932 0.5488277777986533760849681931636006975495968028112764622958340653 0.0007779018995210490902834155512032063074008351574933353724854153808
            0.1817101527362148185207841904059038129581201582281659224540014707 0.04986902322241223252777936962377294865891743880522795276768736441 0.05251200287176029200386945283496901471431051175827305598088300663 0.004065819999190276039376075743102196592269895132539581034840884797
            0.1862102308011067601528326409737766964749590080827535397826199848 0.06842438180268572070172406057487594592029519300907601688443887957 0.7087818447451700470256303294367219089020330303315231664121689966 0.001982152560510525061206829912962479775916355507597312930826022387
            0.2340243238564645702326900773698025200066048946065083789331523612 0.1806587186966420381502246775983985061035905667563604106233006117 0.02307195654197990462286897525584657462562366397712621561338437162 0.006494480562337596163974245414720796035972678113915713504702622934
            0.3019883902959320310332900304981445018441264614227280202630747552 0.01403098768836022372070569907628769434925983263255884324860462274 0.3643679934488349257778313286479301887205393003818337803911742338 0.003638092089883839444313959454713092834386115808121571069573206028
            0.3683452397662419858264468761618595393324151857877733959887887066 0.01140227534114186737892830434397252514368738384998302270422461207 0.4420485550131462186218552567694379815613391169895369120099605261 0.004149753931473900143485286240742283739331717334308617349130851814
            0.01670083087496188478844568001742474884947095822096004601141492470 0.02016747170102688242701799507985901246406952714069282889236638247 0.7380935281581850795751052247277595329512630654336073733870343132 0.002015146169964976222994574436042221680482400005816908505314964583
            0.1853627345110679103156596181283288020067113351800218327381424742 0.4303885376521661143884787525068856036516122494881150788434882070 0.1999442154749932390471624074134397082824010005608617274384112149 0.01285573931598953268071583405176982895839201372023413413324800534
            0.1712945982444072810317973997849592133218018497164458765572621605 0.06893566018463210160854801599406446421033327537124190456346850274 0.1673618347414655118423057258879946879887326308177046350693771396 0.009555893845760890427044523946882727387739766567148023233318018315
            0.3168450320095819373578005499339039486300771634245428385157785658 0.08456705434193444619480186542452387174442634721851222745659395892 0.4803381759629485033981045909851051182582406321118567511477414861 0.01136517396570752629631081707093516804138000264968411468964005335
            0.1040487962422037946581441359843771227775552562003755081673079098 0.1874951749459176058018076974264535177351518110689604887588366829 0.008918653598628844331889215793602259780425275882045742155707515734 0.002551654559989421468371639335543713897142028087379158755234986285
            0.09816597223896487088995656912125672227038607107855547795693712464 0.2224899083369929782467409648467089816692195384108047387215296215 0.6614232419960305804863033658111918965372136259776892110437766171 0.004342924600561707062651698723307634171406221541666308795942711218
            0.02162921317851169624658244113970352212949011271983010339169965929 0.3871644184405773887006361516324143667321720689470230593081846852 0.01156465150288865421698016097526396226477649324925408584804748890 0.001760743377057707576732884801673615763318365036849680727169170559
            0.002529716310619293421498440081443720515477810424886726229240451945 0.7558394112631253491170332640712223428962130363482630078140738897 0.02110755477148691558411254791924460319538045610021693722604667047 0.0008626635205453634893265648816024409493331867504633385224775616975
            0.3024796051967951057228221908670407921252653924665134349854411799 0.3345933071505105526921690229266163179699282703754126791234071372 0.09096733034651396764061619935289287128295691536863885717604112440 0.01296311288326169439992153549107991003175706059811196659289607960
            0.6716803022078243802299406907356011947516980183718961807479106164 0.04642099445779409685139621037141490097100227548654318141427004464 0.2039576863668991707551672728961021869258022069682479554802719660 0.006245575933752215804346762629407028860368468165603974958950488001
            0.2989859830638817302485172277680760239363879758313955722783701015 0.5933404616802458701504954006272449071515829903195455555442319284 0.004954076397114757544338579042693967952586975907810581490913505303 0.002178566573977348979929097353054539137380328180086720666197362813
            0.5566144177763578424355086225204681505592888953801252862419244827 0.01930159615025993849117436773023773869133255470331008913078567200 0.01423676843942446588889465814230764957510067227560840889996787040 0.002089762592294767570878490415204132219885978487095053789684492623
            0.8051863145397753908757381282881524542144254542803295029979290452 0.02953814813638054848326600533621948443190382272219387091117820105 0.08197297012225527343365004716806740824278161525286786984126907877 0.003015808244505484414632485532339066742162075101165192143580065452
            0.2428142654708457861842095898800959540576101773100570135565018133 0.3141717882112441949650175241502998277494866878635675120138759374 0.005405425430551140433886471098003534196410832967125098381091245863 0.002586862899750412815592966180606141704758110302814024434049229614
            0.02978061680885638671094668772337410086499292508030129858662406437 0.6957757395945379131234003937638040452235707997049138796969973313 0.1964192809628879539839300619786562931800259575021159390124020256 0.005060441922324747444160457089569155308838597406741098489406414425
            0.2357988052247937052564507233107801500149518764334994618155976048 0.7440919373937554330631017003787195438353467362496502082733415729 0.007649794086153857124920856496512567345163993553524328957012333775 0.0008711379001063668514876896777698874806264931434428895433802228898
            0.8946270253410304522116585251777265932407660856669437080576209588 0.07054930019849376606092805768507516146312340291620044662101030272 0.02345151562732001856520691793546343081735133573404657103123852054 0.0006453576460863578661679849150413639169117216431397650132591195028
            0.1121792280386707692709948527528686486748629284754174006336660791 0.1043687699324670739625043389979036425747865592942763801923527736 0.3123604751876906073869213667682705784684911371393708537563277624 0.008965800965574891820726421641594919005674703938028001941062114200
            0.05192374003497470559748357995510257035680707242517123662236759636 0.4388703209344928225967837070696464123558814154094784298039218614 0.05909828828625402072846217454032198033502947579467094967187117382 0.005129778925447668800359217859128842106257230831203530527661146816
            0.07502570088072913479638710484069006053082072062737570128518714977 0.1123826095831733947691073167611596284465148092408717685361584288 0.4873661284170799570040542325524352242199355164726111421373956290 0.009032791108007324478192164137966483316950412765200712546668505813
            0.7310623094344363366251423574167128037951032177487383226956992834 0.02695507291950792703961045655071494426950098699115875833307868199 0.2358836750070112494649385460727053123167619746129446822595882165 0.001196196799320896773720667782947763409746615630661951217281149313
            0.01263746404988034107901347212744792193464511665909964226744767297 0.01113957311184543975892302746479860559675996830558292859954326192 0.8810813943593802370438403273988325401378123177523012703290924786 0.0006838006305747889926650955692832877782860531243650597535586937214
            0.5498570192101567596917884967310932388752701982500129881338316809 0.1545734415403295522679688407558522154300471877280399362867648340 0.07045804289735947303048373599635436089584674051615096751030295750 0.008195883062948429102471016914729252015442979085394021936944343291
            0.7007948673386109236711075537087929083453395324690104052304316630 0.2117648658835198067554536274693902394569427273786607554572204181 0.08070555282630445939275845174096932482630140656224755643196935730 0.002220643942565266426877723458767710931429782076632317857876813454
            0.8726891267996625449570656072835645027098369411526860527365211376 0.01093103412948304123765027645631235852749356876697921311848783448 0.09674774701392626071637531499217102546010150339635525277441381624 0.0009697437041884777961496775102402643366182122544057044475101463350
            0.1791221361257211186719724286291912846079033799356606278099698381 0.3050178306500067983056814442020723372127626235873374748754836252 0.07051077334046984912736656413828628916488214499713564169295262949 0.01200729581772904427151907870222914039626666606442530026065698793
            0.006001525602280862480706047812171342265816554691996328183805199721 0.5155227118844687073114827396363937638788046027573215691555938136 0.4686425711584814262261178503642217099113862018298872900602750799 0.0007359342926616280019613641925075171834987359973091706681695567359
            0.4429409397741571156988977408273936028999556726909441314549231557 0.2606460785983505519663560186954254951434662734330112111162228597 0.2289748937800611616751604174968762355438758055586097323040937233 0.009199920254684937463560571546129837969427740279037218421201708976
            0.2021387770985803193303019491362395267567887954699114693670112218 0.06865319224402820155535095589489887281789211478484900384694176754 0.4564605644059131014045864248915196122612994092053199174895887247 0.01198064846807536130351233685392534236434874082326754349382730967
            0.001323710046339756680592668994699521124418404672850370559560082391 0.8493011657409377900735560994497153382369239478267638803567813222 0.09159985226913667158868232109900163684409910626160531760893564284 0.0009109061973714105965879144275864830894680363038865390781543764261
            0.09632662696569336579983239549847958843326820479224097618776948888 0.6823593021354713020962220296677572663240550276021450289255998822 0.2008785859870449194606636904159165572604487938515713680932680870 0.004624639407906879798138583211644145011741728541850486530259939602
            0.5109500637475729782006709525465298490905462099410034331589776153 0.02937763259723758932187783847683869478851392201427592705920103253 0.3769911340852654680992776316508503637518216166902604481046813689 0.005274029458483096667063556058110409503385348387090060513139236549
            0.09299202868359346959085704350002592550514425280336641181001603930 0.06276414496728800427582459857319489561880656416563804376512990146 0.007608602792948092021654968958137097514191939786333554699561450283 0.001100550090547387463652807757287516450564249238171675427205808037
            0.08852505836615761044080868062678074046196837745949767592940130933 0.01443567977715906198792493368888712976029605182865585123576939499 0.6825948910724184352772671492426461241548882194747120291773028057 0.003174175645303811914226507409886919901933823633361642687163047486
            0.4589890395518578831692336894561465678901751372096353818414111367 0.1809534450709162498649237383341220937630895147971404387406807229 0.3516226852953584879117364847906878250239528502606758414837066932 0.003647041029634499416641089340272726925522111511985391007352814775
            0.3041520961334929093387352578684663555440792197207020858109189918 0.02309706781468413996068724860236921691726092582921729122860797943 0.1762486702624577520910906224954710749398702188588269112400389729 0.005853403888949752425402398869377325629285813869233951971425614358
            0.1879412945201879596448731334182591996627192584747743318082560383 0.007331005205403499529781498329272118460972978432828048458370450992 0.7188625909987192365171535855588953319481682154963226251105090821 0.001942913358897424405520161878610551971561892663615998311954848901
            0.1225086401629849995722938019697370807546003138216115978596271620 0.6774976700075785647893408446408760986156011572061461055563031998 0.05434708941667695522889592260493462610050805493908261532543465130 0.005507380680976306577477435362331822726515138788162766604001233460
            0.09192090991402708502426925616508524670516584703778096525434861335 0.8720825611577300181244821957557397326497457245258124657278299481 0.01778947594993722549119299083942072374823370001288495209695987190 0.001346330517157690982540965034052331209531974092264847702899733089
            0.05489517528227709407106905107549697131233639659217669986776158232 0.6969436139376676231688324423058097934149122343674464679017559445 0.01449281799662057474581185158800686473262363426093634876575742373 0.003029919247922013500974232199942162466014136880651669723563172119
            0.01312811426754562143072983158509953801993158846815238522969488819 0.4681403207530772657421684824766786987972428862669181190368590256 0.1709492450634495748239472542442811100069703167492802695667852811 0.004521997974824701569114357415689476480554328100504516194416455366
            0.1599009658043735047144288429977412562459285522520232427995354501 0.4937567085725856969996001336227430169565515951162832966951533913 0.3348584891265796063473991865300422780410730030443584245899355925 0.004043742333603421087065407863267174934487997771094015593333979444
            0.3670622410060861891600160744204708120501742374441389395296203647 0.01403387671761178639037612247968863591756783925861184422779307664 0.5591420955003189285599358296202540832085490518251950289512038359 0.003556164789605407077047820913243347620573221999107536313880037037
            0.6553101134347479187858040857053372821844953607045951016962298503 0.1196095613538235271032795443600943900390753805743676610442602249 0.2038093116111652668297315535657114686508183164359987754764229235 0.004993247339558133509345172152422188708502307245201524900835133560
            0.01846884564851310796979065693810239476736633841133394629329909301 0.2740481592028682712195523294470182982402736931718035212446179761 0.1896951676584623487444557997141158556011930875357802278520614740 0.005830660528553945107377103155891062418992644659501668568497947410
            0.4204626142014355300063286934166341166394447853079159625570720703 0.06106403966697951029465849570483924575952601132202596080416332772 0.1648212632123232793725530670994601958366177075781227438781426860 0.009288963006885027648738447908657408317623692771167280216042047405
            0.4341202277728059358716060309510631628969325820555187296144951652 0.3816869284795853625562145416829761211317103751113867241888795507 0.1164948843090435946870929917380377130852693190940168184024677601 0.006063022833595364790924200145665026933614397605790674420975360181
            0.3388244652290413356298349269239915706435814390756216863217885468 0.5917880052503514665139054194016686339910193122268485047366081588 0.05553044443866386493299345135860270357908460875621657876175211542 0.003325897923605087091161658218007763103344515995038902278770649812
            0.03464184023938156226981897272233508748008833334232870898185264967 0.7894870367243556953204909533379059058041674310727785943583230687 0.05773069434737764626850923736223926115751132235499124761321694070 0.002815616769970525329936127840062222755829239254333499457017741705
            0.8288241729720923968006816095014773534735029754818909656866595054 0.1440834474485152738232102816735387450521260791964908883786531451 0.01634013698408738318778490888971282795242829571867203045651380689 0.0006972429263089693318076204030281543799594047155029792568201856298
            0.5884040536638719381932994885658394252349982537330339280794008927 0.1005581432404579758298536140376606591147233198235172404669207909 0.01110780455303209743796752870610420791519426972090423706260031647 0.003263066941814911578111014351514106208585139923083678951060298176
            0.1250915005787327617598246740761974703581884985931413306054534897 0.2080147453905428146415095536054492417404047212745998222326537454 0.1834327075907607321873177744122870083719532451079625892663328372 0.01333010289043897731548868461498830742305659625813546536352768210
            0.06664866203849174444996013099173428713571879654059187393617798293 0.007434104644336051475312878504281911628232814351812342671509141779 0.3936209682057868558644092118443520604284278178325214590126997167 0.002408386208187212934761776291532039458925614499910381815620859606
            0.3845281756471500474191876975502064924761345872166058063384226938 0.2335558924876750555758006759136060545671669869791227109493095911 0.01635537071496900437341623492831236390201401606388928199105004988 0.004699545732914226658220906742179343722170990644013507275245981236
            0.006352585741721007711364534800837541324891495954529996576029610848 0.7355868472268122389450091620760052068331827119984463753020790155 0.2364613273753508650778754332663309848614100558163611453024412613 0.0009519939982203648216694171030312135212326817952985519851537181530
            0.2488686026308513981857477223242718884556234911771709376509242155 0.5566483277960575179987445142854528638802964519834128951194430389 0.1888853025873145116986828425792705057060411042225817359300741543 0.002803222719809435972415816800464382148505916274480024251604402255
            0.01631238299177849483946904343647361268830055626346557693470849435 0.09900572041563131387504734487965247875462339920944834605141400246 0.01185470410372242880421982451217008390085735792167546701604970257 0.0008761883619429863743224716875292819956201443800997327746853643161
            0.05839226465090079752937934183546746544423946151691089212669818230 0.3019571723483717355044887238023623116941953353427944049930829750 0.3035217089639992780797502787622276699308143393181480359348899796 0.01064254524069106406535018614702993646963140286265566203774849245
            0.03118734066687970039360675127140357964989698583212468330882420395 0.6791691676726023636325142455988472860324690143810702853676575832 0.2847656504954640596912106825992376757492756304510056862258683787 0.0008498385611630651180128248515033993445672406354249278369809239138
            0.2251041172563488495721608916580271875900948902884032506705347232 0.07331555989563757011790796629789282740963584614329867081272079700 0.008802405631309459940839258528730571503756686005986767658218087947 0.002268901035795479957573854206371052878205141091053321426425230332
            0.05418396015426548615914608279408484373130982893409217987519937516 0.2297774295229175491513777299113906876160798166044783076827643167 0.5011596173890558707263083630285683704454112785998960883050841175 0.01004404731144624752849676570995689549655773453043511289508948312
            0.5994649031150477134249450125112212880475848227653131631811884190 0.01721569583684496384310501324779792768073983619145362538428567461 0.08542772448020097674024276544841314633983530181344467927739968204 0.003862812445424578650717910275129529258404527826526090856459399885
            0.008704265258237210087582484093625881832728697118188215492128457463 0.06909052722243415686683692325219626193594578878975249561379454791 0.5536500066870093813515442739224742306108443238097292805233154422 0.002553935928427639771965450101901286004825595063766653769823726699
            0.02892893398251226006039341176123219619456085799404907451001651042 0.06563129069842247392374074861620945084396607855352566085574748551 0.3875428108793278981214494275393089113605912389535226108773346205 0.005089622627509436588185126690369694686028310544752562716404978172
            0.02111003529674607697397481585624786126039867207166593854182183375 0.8751692967111282841850812389990015426229575798859111277366867751 0.01215453089096287589962918649964009887082901575290171794822026079 0.001079651007475946516111516613797450750373394948002968816583845869
            0.08653763369148223183419489564880853510749024262750939338574916311 0.01485971410654425083126666952884583393365352697421517172742402692 0.02414940026958279941273756448840434785268922902262110862826451895 0.001187209937270038438334943256257784442669680409592415024907290343
            0.05199869288159288871037779993450696259090773425501549495234308732 0.5455440803918395907551804681228370083257354891422010133472902687 0.3796873790970170594623582611903115747475282530817409735863487703 0.003570030511045138093450649217269962808026710505171031302462051439
            0.2416660058415539934364775204129798692522114196323334171989246046 0.2541978705902584812538136062922822332677537984556347393162990901 0.2069567277194266811609216128536844072474374693773501294940419065 0.01580198573986719532018693715771946219916769867229320383207187277
            0.02512203311335981202248727998259109566976809998405170536681002706 0.009235690950638253408627746851493869542248703763549461090257932796 0.2298866103320981635180522211676902562024910624900669599080545949 0.001437133769130581380326772645776222436051367652738620674476344254
            0.1705350058692797117258009934740950313407909788994144198340069571 0.003158066997929169251897683584375200057721650963209004903034903396 0.5030083441299574682868070957789418333589872858699292174020640233 0.002434431653155669379823570550872600269586564615915054803221014757
            0.004579875841212080219419848143369865605537407669287495719278367888 0.01712275703486706859933920858386791908557134379811037695863630299 0.3881816778002043876940905603446975787944852514506973769447387423 0.0009632791018058943989238980116403029426076262450569484528121657620
            0.07768037410759336596484275924469169977864732052395049109601204205 0.7931843033165813682616878493530771375637822577264055797480419773 0.08036077519967069026679669664980778868420251673860333640873649784 0.004105295601316544623907036929489056530024098235552704958481211987
            0.5213044519156463023909722044237073050536792615349853866212347259 0.2371888181014152243014999431375934871812906037174165314150556101 0.01082121854349282164517033372354811033630234473881634041347014858 0.003901214118656230451558756760867832405370660126391030519383980732
            0.4889904261281222069298453862665877608904124095968685576812401589 0.1152250819889835732341936990165765362675482206977881301058561188 0.3265723700122942127028913114999239517414199726271787775113010483 0.008387074812620188810673705376779097904009329592608572624332817916
            0.5810583981789681028851661279703534165346773306888842322704336417 0.04719950449839621400508403586422805907021588209813485481578910269 0.1718696174976760967116916273248253443244257680718010432971973189 0.008104803272073645617546428300658499805755739690398706396640728289
            0.4692460088315004244975671220122198670109831372681950632948827054 0.2954489782659366807429247325209100796649592969092359548515498266 0.07270349092457716068864493905568479580458110406376075147788299647 0.01078728920454753041267808072858593032758074083587325129401900807
            0.01439098930146427489256513807845336281060113202180670120139813770 0.6832422484795911353110344468097001916106929913666084189140448748 0.1168662943367826090783680367142188956026031146087412933903969352 0.003539408493577277632585043003279276267446274291411716036846918483
            0.2423376815686060764650431143226927526144395584092131709765452484 0.4168984976791842163836297610931616012319695489680906172818034238 0.02179843997420954535405932153578036196474212661689018544169573791 0.005584130446769978246275125420882869041187954595082859402004025279
            0.5249291274422314257172727687118826517052425514610428919404475425 0.4111132345759422656758071744383716546660323605668736111121941058 0.04631727842617972393893982865736899262649437388243375398040055752 0.002997582504062516548700417651182580781476537769704522198301164204
            0.01860528992588628569862573521386711575296433804389724093655576044 0.008483800270388622233332129211941873154653996152609230907132057107 0.5741442766612084843933876962282065386046274136639352340639542135 0.001351015026860132687835341841941375168125292403712950723728419977
            0.5442114302647621262139561635835161103363142007292043178405524415 0.2425532425030702197314115561127520289062886731942399403578727958 0.1946061114702952662570173082674719118529654927169376948014841851 0.004642573252519429234263672049036317301892238269828350371900599520
            0.1438368087606178288705326685159476893723554301348249723912241560 0.1465449930312459794310576194363139800252336489389206242815820321 0.07954771995996240274441276006627522961755165108855346950594928825 0.007724689465449649833777105740045556172365228373762460389486203336
            0.4188895532639351208193060726309641202720336747341139112583128508 0.1564504005364991485200581268533889310583394377553998842228623452 0.1612319381879840131666264525509955793619869402805162935974717225 0.01343205282096786938850665088705765481475518012414678117948420667
            0.4371657573973153061693824881434078357345351628603706959418273087 0.01043394395830696116646791908463906146489557254888669002872779754 0.08336256715685904008253883194548138521528578797775679398058996352 0.003088573647500701929328734032772466189538433113705559123182809166
            0.3770904327328024758416843725472878952532633204211221466275303579 0.2033223459624090085293843286425253284204779933635403410888741313 0.05969528979388329045227942176057566702426344363628590003577791114 0.008241143299911818524043032416362825709496121294306215117465773816
            0.3775910691695523205100843769984910156228960020744795810884411611 0.4964821253614497469331384350855956059406873660294097041761098204 0.04308183387465497854535036724004865732253434287471959847435051582 0.006236546875747804604339096098667997662948649092092324538693519779
            0.2077042771345455094593161477009574401431035928994945732652742658 0.005839549888236497273396282168702532881717214579471117246314760152 0.01304197736220023364666967816415491186619114099772675844852202226 0.0007568280771387923551414576080368780832992048515528956669405965068
            0.2726149367617908054601071932680413678310820937224336627133118776 0.1640015453863695909212309744374772047292084182202324427063804384 0.1200608089238682206047578851951549465855153412365063529345200604 0.01205801089632172313476032519465873278825047596732599105863138388
            0.06180748703105274892948332118255394802093053892894939981350861279 0.2534708601229811404280713720516827216434351423166510225359926343 0.07385294895808570162950625533901904535931211779195746407318119791 0.007212159881278318322156089631216346758831567402813573237180706194
            0.2005374826345008564177900061413745190766964569606587602703330522 0.3039918590956306782000682879172280156800831469758378565199173702 0.4942397735748566438162599707955628295827542902658620783863332967 0.002062972103329897330358953334617086015866216371981151641589642471
            0.01510548106380341432753842089731854218232742541047087260089375178 0.08514333907329006932312534706086378365911825339405855404100951966 0.8807501752724652846694843629497810316616359791856336317604401334 0.001030897124110867160507873433602732858242493040616409825692859293
            0.01411589304905514172231786265644962408712320711631257056289005883 0.1840211521987845752074395342685475781643768111868881836764757002 0.4388146363239877289873261893824924875086856625817906516440031573 0.004922683058275985085695393274065492875785081547045380375419973259
            0.09760817208029967718996831770991290542473988399889232929255405474 0.3322004791658790408146905913555810517826928157527701716612247512 0.4850680183645971135953701991650293349315202940084439616636760159 0.008425388951623545178876560961582060513986159593336212244268014396
            0.7349352560577873798643945085638205083366722993469308539021552327 0.006942568829339505668536882676503087763604860209493022531790753853 0.2142867438606308977777338321524242434184493105102474488232969507 0.001252948403805796803775011658526102920132340422888851384705117985
            0.02056868178344874455173024188263784866114565883647673247826919614 0.1950049744514656010537162168598412376922742051031224220802119911 0.7687393447103683877926544615022629749267967157937697983624289874 0.001612505227811301355449454568828372329245375965501355003198870405
            0.3203240144399862999210250971142080664104684919186124637983263591 0.07468893606771896511476622054227941432819624455386623441650349380 0.07195443216774246627109956821960457965335504871039505868584245636 0.008158299038197750127839048358116144982509502563735699925498098803
            0.2474230992080855077910843716175280391012595136759043325552605135 0.5217255839013016204628563027206707463788659563697631824570819155 0.07406107531027107782768396402193849790387791691379959218084726112 0.009211996349105280583856145118209746417353000034060226984917720395
            0.07550747606664708629269186537070563638702499928466999307470690092 0.03166904144620076832185105401486670875302443848848536791213207749 0.5400153287459276848364560081519423890522714724730983729856256136 0.005250065932168257976085821777533229850893119751517158356126030934
            0.5810055568239114498835024069594933075422662049370493878948107673 0.006762980936976298302295066666716177479672154063591354698515720589 0.3874589840296432879912179134921562162507247214833507179471158205 0.001342381067128889218062602613789162292050810231532257514034299255
            0.02864722865081841809523794454655454481290477364409402462085306295 0.8531833758867021625405303815683110546054379925155403012719771109 0.1095785456123735975596303618245591852322312719609335412463239815 0.001267039862502019820046907264784293611458277215076041714430128981
            0.01367268957104149336160839424218938159763233873038038201727658268 0.07796580030886056393394302491703398278771600917774155721964004249 0.06857237337433309683774821725330117231154842937114193640568090612 0.001510937541961516436301744984033505462018011058475620407477628367
            0.01895450820472126829762935207871926237982385248066883867665560716 0.2255894683743524218276496361477481862835706242134245321084554916 0.01918325669874282089792692497589139701532497533852144488484987227 0.002091567275015637638518517745641293161647482623290528143807545869
            0.5968891378142529673148358493350931623018918588572743545879243145 0.1564519741627502107632209739266635144414894552352188941125793078 0.1394829813547609627401421486788256725721704446630225948798412795 0.009710714594056770948289065180934308491250213199579482495304330023
            0.01757978737958559085824230055381990084482297865543205276385409194 0.01844643678870181967574216343094368134414442529305342740681066930 0.01815955598172459969489205279024608194304887600700053687418763616 0.0005922297520236371393220721480920380852491475138480782288660950721
            0.4963872111931625258560221519797913820131025242873731704325448417 0.07825380122936020456443741649364016985317230976469069950494426628 0.05949793770226104120255020948346508668613625427002924101465856596 0.006995957583788575851533403103155317179586463590669127039090473217
            0.3722289686073903087116498428205591004785402145000335248063209595 0.4052699196163401726292636568093848482288932411507446339442070314 0.01625588514588891763228696455908746376315199192052242777590228968 0.005942191916197198648210101277484262838657051609289561851099669756
            0.06231774801395839849695052410763964654033689499947350172222851667 0.3707170130051693487177431660890188666887944094747690530980064155 0.1374924910960908040310802936800351958869819639846154279917689273 0.007588546484692812288548073774899503848084686343622319516109993956
            0.5803829662882470519670909801953590067125696873273454520444018039 0.001197709516953249555039670115576744783434909654447013126679916905 0.2687398065461832371177250590106251823505448400726086334096444444 0.002018336111020516006902646833749266615018517860902008424705837139
            0.1928525237596405159207565747584874945749794381788009276829962691 0.3368073173740501251901995523692569057306961781073113468966763011 0.4211134252496633701467214006335792848103121097823621648062136475 0.008785075607693022064995642863716893958664454262416407902982289377
            0.2599559196325909527835223045325208412049074041093769687488864347 0.4175115362091151134228151980631583495909913867804861253686815662 0.2474182613707833237867055302725432289189321084850687696978876151 0.009157563268119343140183771491351705560595998859441673671745061734
            0.009914566790768629023290952878120077802272465984546656675679750639 0.5721633608916377529591394661148163936322052383976097020416289072 0.06655395361501448722629042564194277029878337437332063413823583140 0.002239175739876974436684084770323099362550694699697349549906663995
            0.3684381771449385693542787101419586296858580940043537346040471071 0.02011015560796175277402880543040267678286686135996034632982150424 0.01836364308497698817449031098019788999549014750015974739925788296 0.002491980938774911525700068256662004460793888378368732737161607228
            0.3476327632183010916164783451849858992925923136711344621677086040 0.4684288885712960312196383192471535938662106724645454336187179961 0.1493552065065287175255063773341697603679853910093016900119846374 0.005460143728311229475123510280506858567936799214263362123023378290
            0.6299513900770832745038563218589191836582720886942809854745827460 0.3481941094145954713937073413210268878401650346821165760100669832 0.01213262992129207936793264181735636102726695514129877532215229804 0.0008510019845412764633978935780299671892061006750895676338757924007
            0.1065499285835483031853983425374266812412175655895680960981888502 0.01690323531912160844031827087964698385809773079889450547241520687 0.8583629066155641720074202052436295507954259788107728040120519351 0.001381642204670235440994735873086476736029115631800355237012500817
            0.003559059655362636234080868606238822366742704777701853682349553118 0.1414525224703397365698301537904259542601347820500251763298607763 0.2877561455299664353046443359672641467274943616113992931025068102 0.002204516792188040477599115531619935743078470977337044807849191225
            0.01022133194371154984158118686254287708639734075918394117212319112 0.1811260455028769000860429630351142621578817496293477474068063828 0.09985685305430419099721599396457671429262288632462191338622143385 0.002674964897858329373424265740911033102940149508751507861606024987
            0.2474437010640446714917973629633945925999578255859359105483774296 0.01872342416722929061195748719345197837823576756292368547815423052 0.7192811161179726305563669519413980639958230667454192295298563096 0.001744916063270450363962337278422020954374871352047524752003783157
            0.8668103478901305222890515847235991187498414529268710531690039577 0.01350035545775389031548755894785740048058777773340430355892825869 0.01805162186478097417794540203684522667603649006337904156392527826 0.001106745855227594693948457694933068220548921665380579856203138225
            0.06042164036871721700158909724204492998962868801643504471292987355 0.06944309780553666016774104928322029505283775390926908946101536094 0.1091491971550547798712131224419807596219708001655451200458525249 0.003365976801506138183628662355469981777059349117208509939700067459
            0.04871671673450379949146144148596552642637808926881092233339634743 0.1340832672264463430368583704270531821231319929530304801075622558 0.1799954712863809899338554202873696503393727619774995372302296174 0.006701334517411695072601511325634735818994953911487934138040335515
            0.04654868208162578826234269888578168227579024142010129641788194616 0.1801926704217277006911325023975755305258370596518351877734412743 0.3109152076306296697238123390504408616241392109654968124434759846 0.006587495950883018045237663547486664725493860125490296880092224061
            0.08601395357906062769679249980968054368076925266273895345913341444 0.08912447647650984814742149348825204131652806441084441953495387922 0.8140643888559718014734937228118829931473271242039844472058924503 0.001771425005160162341147294898979385104538298392929209688068808526
            0.02046839639119421518621523870703847895931946807712071427176950075 0.3860786557678877202688447039320331415475122909464352144380652868 0.3948111336496147174975144104925113502995920488868732914229324041 0.006640500095856167326024207057083653161860917074552378693411300256
            0.3343350796967243454288359151078871154780061386873713917919050329 0.3485338078819170460231412177071581871465993941482238793740284615 0.3006512333342201658619159026371024683129018947583441802836246985 0.006699727326119133964749979352420063719777421721861057764251592088
            0.01737651727675884997134381017714410411119243081146758850241495102 0.01465113963193203161118353715831098932843764772003476920628240185 0.09829749440655450294105482925156996279585309121010980688850897468 0.001109148978067661294696926883953627003716559888547534135671962320
            0.3707674421701883203624999811922093008064750005551120218753618376 0.2791111035693254084684531510507577822911238295240492637184156385 0.1914163431994507658660175873645762352498718214059174071328823337 0.01261814400592962393791828038568931740993031595409208323107488213
            0.6695661209893021863467908709266721832511421157337600136896005306 0.1904513232739931401796106005784729464274116906997974868085308659 0.02146763991439929338505414927474115490834326013805862550698712968 0.004877974792835421859946434010393986303875431330973571300831488316
            0.07219855049911622782582551376389340524948964322159032637526496623 0.1093840375282026486392087960583480756316589827211931166626050937 0.04127083240638845041397823535636964183176207561075464460584134425 0.003134907781890950347988531515462301594951300269498929812560174788
            0.01868228675278301831343410231620707907316113243999879692304976609 0.3709366257765843009128100721836565583422827423879939917625239614 0.5249650658952842035905525879044105658379099558933233221583241633 0.004761952014694131974279346184878881247024192548432978131372811423
            0.1031846866069413345509264712061821209795727721158550743543296906 0.6093753916076377173885854628650696545494100274195001521420189388 0.1545275087616811916931175833439505094574067917433691923566326482 0.009412873489607134413696524060561844713551274202082813602541435435
            0.8392477420732161061975695591093560880230804653069391948972402828 0.08116989958962178404738790610364168851046415014459777144918894946 0.01209270385474598762142756044562579226311331240851293235657545986 0.001635654556075609625967791293982805710462020573315307311085933398
            0.2034958255875442141280740469055707852276749244866265281657815552 0.6187136205669969363830108744660519824400542016136206506085403284 0.1239124306751840844100549317138362800046493035859843544425813081 0.007851158619357932118344571573650292451679886454513737418072200054
            0.02330965261047973849896764788872513941967451397900947691344114746 0.1028882422757021156051042049964275734892082437038944394059223396 0.6640838461205430673695711573531694446404872492099427138539435625 0.005175158179429429823787181948877451382922615277604919779701461240
            0.3368208436557667990109155630123493606322240286576986534248278739 0.04490543205909051432093212857384959903860206012264056118310413592 0.3113438347181388163807141064995828359133175802979342502957934957 0.005688642812156178096274413368385220839986837372769420726092526853
            0.7282068821124298870149928845110050560191961290781768076301341536 0.01698173514301396009552228338774039427919220083643022754946984570 0.01924931519172721727953743458783243070280254929709780770844537883 0.001968045897115115005387801803309530035760450503192408242140360814
            0.3170003442313028901780777715302389101445246989753205403780917178 0.1412925823925570353000003804111278549254907872327011811813672117 0.3275303024676495632857832665792079448507388316567881328597325273 0.01475076819813514036233311312998264286633589648651931324679527815
            0.2123844962260473262983939445019383957269898465563815095730431521 0.1096879266291392855186755573766080523982974947187453668219283012 0.6751942473274485874761893047033156082085080671769370132076905785 0.001758023024351833987826506922736454960466777733058257147434196251
            0.05634697358925610443452047051864876628628965756438453134961995969 0.4907799028276332169485393147270028540007541237138521694211922862 0.2182123808584973640775825486140638838901034714278321408349463750 0.009874428366382120923828490658043266024519494015178344243148080866
            0.1361049554814906635766250798945740981103412880817576526680730009 0.009243534373402538072769792141426691726918564875158877056406375638 0.2179353479870543564304077518461909170302373472275973638136293077 0.002946828856968204709928253945725878241290897026230798596988941822
            0.07758787848438789733876153085365558633119394488460586524980479618 0.09739843368305145196684646400615492981381328327072866000978289767 0.7601226152742639560774147593222682147191844265107464564691935783 0.004346626874626566028566186659427282860317615981139535844766188735
            0.2079767681742354687383706291247857381281552247872454460466341050 0.7020886340920630736970254562895332390490914617328103084325963476 0.03328655229161106802996308407776201209228871985949836050518809529 0.004234555400613262313210931122645463667323756685865155343543734994
            0.5939570802259585341822327821883987038731320002019629429573284054 0.2703256039756099502770743428580833251712483916742289574368599910 0.08086139883776580037648110100195324184623526609263543898764664947 0.006516655096301793611261947572102564106711802444949596263940321728
            0.4714991395072986646496062777850277252576705464793875164197481614 0.3844669666766358633267900966578099490263605449676200776320524139 0.1402395974825106512996543819645746319432569005650357407525183621 0.002294686329357263826599232519875269308284984494579305563066052286
            0.004601338722531792157786985212290246075996963782450607645634347522 0.06234625666401818252090857203301664356831921992157256378751600048 0.2000158728638744487055508485655142043134830319962176256332640537 0.001586242490535612027600282324286025303097720320132548902256537190
            0.4625289360638351015984529032182798002812172711960976982284254423 0.004574714879934040107552691715956141160097140926079686387386959337 0.2164099641530858227891007948179260712526512803371682754194323557 0.003172020727126507138491257988689334708321952103069435423371436445
            0.01707733270740993047229585096634900779516665395992316408979346742 0.9439685635830315806341430263721608334892364567503517921706940961 0.02133318122709948124146293921813941730468425040936119212082256374 0.0006465467005981786316479660966668705126150815355204394071709597443
            0.02134979644513946025128092897414742815324033615565794429951109602 0.01752742177506245759453155107371676663197923857708866393134854777 0.9426112327288288581620341598425551214113230763035427953917116335 0.0006733458423383813479299565671828572250584783413232393903527877479
            0.01743901299750168511126942876431024665642881487012648388121447623 0.3396835311000472409505023088677903915056595321659527820139424872 0.6249231618958005030629199567445072880699733687644178045980189502 0.001889742082263662023112072535039328875873865590022773913368286938
            0.1060117474991762537389797662631232968132834019322979022970854479 0.3346199463292362009131823056842658902495472744297167883352475054 0.01527550712088721652618108887287770056200069206680747478650854780 0.004190769422185754942448874519821480811655689839265065424078914084
            0.5348605283146589533763666771219580436128643641537508672290117496 0.3648585952420912340814027363698416652752969466929645469683490416 0.01237447669790847986263274010603836998363304365621281358671567623 0.003883717266467965447180779869001016597245543341436237657219644716
            0.3003444010033492990870416099293699250470681050726983833903401645 0.4320725913631632718810618493342416160887143256466308591188680388 0.1361477535839049814567766980240497396641248817193883456953278052 0.006797643084003488630765917148458461368549253985021135277025632016
            0.01469360805318148214265535116488921296062711818184970010680309059 0.07578041520150174641072699574908326572742131463874939740796835664 0.8130279671505999009797721147813858473903786833906293947458069005 0.002266769797793616354877690211399620309010946014288921044299738023
            0.007964059531271890224309627024461892581150572478113100326122138683 0.5552266727354732383784261802476905155760160714328582712349603479 0.3692085538636438618045233832917020703027998967497458948184219257 0.002255195757435371847705665837556889210482162322314526942594576602
            0.1441171047353248648038473788256042180149434174884383752361442789 0.4771103178995261322405837728883522605342741242347892804632213816 0.08896774167577381570456858819763515419819086710759806391609668029 0.01079776168820218523965156361442216809860295809615803505360486229
            0.1688282816203245328301247904183999520745443724559645184735842628 0.1747929157957496285972234809284538792477289240596689560401072870 0.5012006851987394300262509296152837183344776555118470405705607218 0.01349499143635797403026805838968663043856672718920894712704938071
            0.1606314967563064892531671220957700832378521586150128853896334979 0.7559175387252305893648431905380845861157745174718098624119682680 0.08019571405965699265210705393310589287009364608281730552907068894 0.001658952814057419443869521156554273099849223984668564154616428709
            0.2513197055268167822162985704504016066859775029550868369980250890 0.1032690797294146711365865772462412172849518467572004173080513003 0.2498656215815960772456021821779581720328328062705457481883939039 0.01362180164548869970093029548383765699608265350675750887923258664
            0.1900881526900539593674679478138958788308153132828047208985236651 0.1781553480393000681025705080128516654103530240818580485479739661 0.5790209474792402378423181967597066009940121070140001926944607052 0.007986803570994515052121713313008746182081979032131500522713887642
            0.004038616537634293054968657906698807168432486968103735960015948410 0.3355107077241521947835309938783754408883205648768278082431233925 0.3105722176466401322104547084884497899029599784900387627016464570 0.002788838303840560546963531550566316987245404109654156730737903773
            0.5554599033042431103909711857581309128276626202379979132793179620 0.06302218092058025664871083377086745464414272756956645227647693559 0.3673223808847935515486602371427662044135694394764831147796226335 0.003170251026336861000242136794134467041598037733116463778400708179
            0.008595572803145900771059003432073742723630563615006502317840888203 0.2011349184908165530362238940624675185111821802157824544626365961 0.7087768407760259208575003296772833082869953767876594792382197546 0.002088884346098397001193955811342690485206636186977705653924741788
            0.8150218502694453784134097390516783385455705332353841877141410796 0.06949410757044580153641727655279827838938503705092994244879052605 0.1039192562778495575266555060033321553917980213448878774366551601 0.001799301346231207606739249159535607654757565715803767360652761243
            0.06385888041855653924760600845445267640739815365025710683541545807 0.5047615455526867295155060724937355966746483106497316548510252097 0.3303100960290545959756112408491948045648121183409886631455077408 0.008272109571646415510218873996829917079377135775853078174073787681
            0.2507630912383941868727064054906470633897674858578160250331158537 0.2729258691729783574191022270677563008304446386423118200086720033 0.3323277817796908276943393411903747428536894813157362906409946793 0.01349333600820038322869897862600905618471451194933941031516169276
            0.2007511057900461727839176549313206262655638953205842040572263395 0.06609325673356700568209803149560348303456450299735907230953538091 0.6590854663954879179839078194134259489657775417643402366399250438 0.005232594663621721093321880861941947035655755090750110705561955579
            0.07331714994657656656129595513968298164860838162023159305583157871 0.02042524826417094020561857943014885224172983652358850155439510827 0.8126580397615308170095633519177460333621638273661595535229135925 0.002748796656217003055114160961746482816820070469913491667617020736
            0.0009626844792215000800168526295231546079480889907473732700826189134 0.2311365207415770197781318804204938046394636374178031710559999353 0.5693507233350823012646011097039915460736634840954852152373206019 0.002038278879576294218028552931778789702031959021258368197897641014
            0.1094511511929398720299703201795927601455999965261098378946069823 0.7868397227728236942339964555592849543326586026516477440633346283 0.006424530788768692498511462506061959580337653994776488856528000401 0.001756384390467258195177440104631469386589098755784146337601717099
            0.07668795852972254795094248620215236769730259363361476829154529761 0.3923722327547464840044973362782457482378902973745580431249517705 0.5143632066521120226991346141110280431159845321059520528495834673 0.003736829558465618119978838735989845750745276548082777765387512341
            0.1074479992257070197961927109870455381056374808942745441029058236 0.5060477260884437462823317617905313914637682492083433161510221343 0.01533130888920043816024087753568219343676690168424409260161940422 0.004538440910238359989258207723548026831090738108395948822284566754
            0.4482726203821599851253059079468805951880810932113360945949439788 0.04942573243578767888188169609604354358800946451683062176398448609 0.3084600718838055737366124658265054735707613365280737884513951758 0.008036781056505146307471210749255962088257290028209659563707448671
            0.09541388635893701914968032104344103875954912915498761995847315450 0.01600066219787816443573154693391317427526276586635229471967060545 0.1040386266009801031356315703714155503272355310571371027851747180 0.002688701262344975191603449187224698092095074278200612769510403133
            0.1072877952913166410384517432913590922299788800061952973577222648 0.08131181445705297187810785049369899043255611188787470134790207232 0.6397492816679954538186028762091466158289548377787971002855786660 0.008353076838492788851058714509397626520872964737025838770602983445
            0.1855859375501377070002561832116980211440093821108643321125653496 0.5908628883501179764581239658751471645061530609986704042260612450 0.01433091282168571305995902884974514200547416752230491912468415146 0.004453800707077988558693023257538942114373175367817915312741835064
            0.2218582405832149023719700794794996121624355213386590519450066604 0.02204212263715950113481978854545585398611898769501434265644015213 0.5834232656930696979775824138013689729466259147325657634368826936 0.005436312736023179774751461989571919699918629935634253652612914719
            0.7302384575738921177507541766028986930209309439965009803248038145 0.0009378524123601055788909748563675521027505320039130927454631852239 0.1125996396589630007687543380783479563312583148193574969539706778 0.001472025475959041247087842974391452594556093862521596401561451548
            0.05805863445850515399516443451380115961200114156747627456696373424 0.5956519501082720895439786062205574705080741294788315461801725254 0.08108756617418486220738140066237370630539378063751658817185925407 0.006130601056359508254990928571807777190898825526292535085428861821
            0.2389171278053806616211642262619754866628859518418607649863632392 0.009891313694447862975915063378027035170572216842746659083717187128 0.2845231219454098449490750839629723563990519680894587361617571222 0.002845296020940792797278347528288238802775973680647892669906563266
            0.7532838197915281532518292349830279555140914880842442126217722026 0.1168000003080175839903833601685490385570904048807156771260140646 0.07368614499399254201225919701940530040014415824221933281954513351 0.004658618004202648095798186454077009103627989540812627271683810657
            0.2987340413848529311037535196768544429055480580917234696771657570 0.1943515362814522667404883558211514050094765566703899496503062985 0.4915782218298422743756947569869623695285919874728036458252929165 0.003843350644637568568237599142106578243313115388287595028773209984
            0.05873959172055632770366674946181407879421186509459922385924171382 0.2042665015523637143312673980163569810718716006072160895138750664 0.6401885484190853310023636917123422283352291330288733412544820296 0.006940995047471769129889523254595097280400281937325916834972475283
            0.2424802498704876430635565418511761212376899416810786267301321810 0.009163824916307248017328036954446130362720847604646351895687208206 0.08915945352307181825061097811563000694415735445925766711280474333 0.002679049792630195996848831476337079408086304080099806268776215012
            0.3675217889189664829873742335531708499020253218459684333378634386 0.08023288279198552354309049390183796909806805057650631622157677343 0.5284798167399619108876250194968634317568965139343790108348001662 0.005506989544300104818555894731600419863384260764754922420357582906
            ];
    case 16 % 247 points
        XYZW=[0.1977670527070666027370078522298019632639055722684009783039443592 0.6857065726893940056866784790541476647994536020412542954240358710 0.05180058598761956133720745052241791341346898658418559569906963279 0.004918225539483336411639289718365051385437581539809790036774370400
            0.1301622401702374243076602647669885863465783009454716007098052144 0.4373461912169003634626800319735475207630900857142831381204419067 0.4259657503936408621307433504362128098796442471186915033479448674 0.002771393844216010659942838128563985471165586176578556187797549163
            0.2481515000727497644008310498532738679501750687630720937068985459 0.02722765536268149246159521563947289833752743617396400547167519927 0.2375688805071617434301069647150216974602627181910629429005025794 0.003484868181822526211755474423694856798981577551435088851867737068
            0.1553321839627618145789549316461619094668179913580670051764420765 0.5697714841910085537364241350382176646247742620073462402378441332 0.01550115860858741605213609135582008927203624286697248341516809954 0.003781687727414067084836115742569991860081223578740736759349901868
            0.3673181736938783491614266425344717461008883938787699967967868009 0.5258087495334005860987968836923720363469046509872129075206952839 0.04595893804223797515669492803795551084365622186989158536718419453 0.005252824189035830042673005498969383486059075601785976132444396014
            0.2339135657524820216463321846000900321616482153046447954736540242 0.2568468443452783985841832940730393131299248801091373819184640380 0.4167355542763272747272521150760121626479710302747415912176501500 0.01075098176687715508106227736061185931719186711866597123266336681
            0.1009528370180086368488111733282712674220636570432568679644371718 0.7850666490617264424251248571844168121562323048864315429672112742 0.1000339229301159731686676285449334244512844187684144501750259104 0.002328423552232365464000662122370607805179210753307188832659576064
            0.2484871047085516879948188126879537138072563302160175452650309695 0.1580067517728241790784375970360481207609810456348803638903563051 0.08236825269435949231852525646054565362520990847351624751210772752 0.008438672947362619262117225072754594539987480020821516412545399822
            0.2944668894911701667601658642352665121153975883326444466437942030 0.1536514620131981849128285182283825788550915402895937110491634412 0.5007074217614422486532757648452962098665233080321975131847263961 0.007746139606565013975838190865577616959009386511834793906552587769
            0.3326040902219016791544299778076792637449629748296265688819263578 0.007888519715475832521509606136836556013517623419305981695516478897 0.2589710251091693930271893291878278542011231616052187807971092770 0.002710539395133481525511563942620559526141215682941647911349101064
            0.1386791040919563284203766467947378206618942467343835610013934045 0.2571563309665705673311854102711460879999996269818225649720215234 0.5866055042336919014678200420297496389191789108894210982735086908 0.003919836138284556073082058582275239663341476311857635309191781441
            0.01874616149989119863139944185213658043459154038757069182550329773 0.4695069520363310022579887103808757184782010123162329552369733188 0.01300019791805651807670183835806048147919243898354603709214737968 0.001612273937220395725973590855322646690020630496376992813465685596
            0.5734140747539672858969675632132535517308388764604010461804552152 0.07847972623133376156331654620037389423057388386537596632163940459 0.3333002675700194833929961663196040018174175083428347910015226238 0.003307016636981856857962061541081990304375218232742571351233460518
            0.07106292657340873625524069742250002055713467862429769771067855670 0.1608825018336317804093409910062480720375394233109717808061638091 0.1919580728171448777844678974930719115607602302764651766562174366 0.007239015485380997518151369461925875989098783099674461761006660453
            0.1760484321844539596499743260558994341771630278599502120153222980 0.1250739805619904312531370383613865045972710063787075224623988223 0.4024465967033989223614231173585791057517482862928965301481342138 0.01019914172623220186996227186692160886845018112367208806826340853
            0.4254637623071266229758234226569945163132564098445500681103879145 0.1884226545692610977533206463661717489274589168024160259563388085 0.3705884896337405876137050201960519393788568850623083487815140824 0.005020922355399944581128234392296623849919210018668109371637161861
            0.06716240679114472291101744453365050796724911622250157433451331532 0.3107405029695728282753922509489651972545286569798449311033611741 0.009230275305652723583908291724084144718653000619648951788413097300 0.002107813109722771131009401663119331403906412036025727142341455419
            0.08014588285566466301184443811111686004803202276997684774899076251 0.004290626001659389970727690710790034325379321980739465711272537499 0.8645785295360362322863609693307813964249729728626731447558987306 0.0007327443275821558799559841858854005066731467284238850372357275028
            0.4686198385188656311813097538670505051106260998395546688495033159 0.4408056030063336846000057057437471925870891082389113394896524208 0.005836903863252539275397324641699658303749305740701131491537591763 0.001809092925221237949468645112415793679286989274907456825451985991
            0.03449330199246952816620177935868938936456402993418998631707230861 0.6092722567882911799095549413280931498818809939768723697509582761 0.2840462394356311071234621619519231917873658316912969490125999810 0.004884664806670971678243609068593883414310990393421988708340027144
            0.2548606364578350023306098613524819432221074441884312858379584593 0.4814414816994239371735903816427511108825546125650528709223739370 0.2561010144261701173623807779651131871393408134374223691673699827 0.003160428316612219253299270318871181413188993049810007385566941326
            0.1752462630811294290548271361040135620042729871492217700290729036 0.1306736719513650628072126185002922875167108990577751672441866554 0.1718271526977108244924045371881813038794605490899932197410286324 0.007577324533436814627083424927108455974656668482860828202021538452
            0.05947358804016285350916372171638167660180091441076644532291148743 0.008775632332373652946909377989739261869005289004827639327070994952 0.01030327642693472601605430202162222087536185778069383445445185986 0.0004075697802755621925774312852567215786158133512661960686827498308
            0.5885589579829193562401130923995410287371895358882434413426485410 0.1789080564459688880274824689629326807711552677953566430160050889 0.2149238866949616170868910520284615287700342019202516841333081015 0.004963800573612631776494299772672288086423451846846338200956542045
            0.2045446086937125183748414795895352570623773280448212310703280426 0.04553012526637222326953214118857497655049461669547697558228855833 0.4854178870802851280574979093771944136969892141191031438313094833 0.007535761089783956819509674185942788846076917873496870512733790820
            0.009756776966587813860291422013198838274352448593258588563190640821 0.3522927466638845638939825492816057930233479380865563228418214410 0.09454400947042667684540686273949199303244404297900413828500549856 0.002576320802720123584734836533226242468168090834171013266541990183
            0.06508931397492767129669512717255920596944437533603033850946614788 0.3281850274335972791900738246695409321692111113122688945538981988 0.06309517349766129751125562419283987300206854334827965670396638669 0.006057152431259083892695880206769705071011340336914039821619039683
            0.00007340111639549574384686833150656378488768692292057171009194418282 0.6618982916458646827668032112247909534334855448621775096113906682 0.08915592382140257082700314982478181067984646781832814061538616371 0.001236918888909614017202224745272978110183951406139495827653054230
            0.04014669555255995281929036833902504812256044253449165014786210437 0.4222865596897427778903085807201825113912094258598927027667167551 0.4741586045014906178953795076175798931742794418444360520346652061 0.005113699568555957238225547550366226088490476773951634479222284542
            0.0004861059171993463423723729717538878962043946196649323860810056196 0.1644480439122869100429890402071108143280307875857315195800270482 0.008391423999090982923178590313799898649939905689844172583227719695 0.0002759364403453064636981760104212644847125922539938893912307271271
            0.3965796651818665741135116820880096111261645500174942938269430275 0.08296638632305122230655813759347324294438707234222899425481655678 0.5124850021344343926040033530466346230794983080389893437280439050 0.002342951886658690414053772372854616923986100423423548411731618290
            0.05679682918887471680148721814050068508620513021716823754878332367 0.6293027545796048561832541815830553249202260642959031595567334974 0.0002668921711024965824894982859336952342837037071400283354510128016 0.0008813038595198250109884101663613196713360459152029790441902785146
            0.009800593135275570720483697458745794879818058648621500593265228610 0.001631133033407851081393348917119998110671202970424272734935076370 0.7916782079393774474465806646444788822584931542673849963844934444 0.0004101866704228950676294393582265132028628969415638652103098346544
            0.1202174011426764792554053647456528351433306761360673269652900199 0.6156733456260290973924959356153667501913612134886492414344862561 0.2478616746999620649983338176977363856606491973916259141542536440 0.004083982289857721696408290858904171112214182418865648759401780250
            0.9470979444867986740298854417503538610213909859671746103065580597 0.04494586866144220143820995867517523341169756198120693990613294879 0.0004554006963633079303849164447667615405752137546112076058045827543 0.0001689037032990991173185997690520547600899902809592766105536463654
            0.8151643055578577050673569194108198304948823064045890492826267456 0.006433094819694760283259636574266927130807396656126442589670842981 0.1783167855841214156824118525448385166061738834829555836382094288 0.0002485415689150780960661718661945383384373419950992496552388451900
            0.04040181538273675382133642160617333548113187816867700434300267754 0.3266624727182218858415482156247360664822115620115564671522731643 0.3108026503000175523407897000731807643695306604662465768278569247 0.008234485398396694544815004347868801624660410413578541959364867265
            0.2710382216535689500170142477412366604043434641532138925643434933 0.7266040914894669382759937608022022317833641693787198890861515297 0.001942883653135390114941496003485242358536260419374562029330544518 0.0001652518520862581706449658382843097201624929074807534788012961552
            0.3654268136669295525118609896579765270000345042242633863964125705 0.3074989787743537282907574864963006870675656329468766776082089715 0.2663478578991585115716477588286561227653310302959390405498846787 0.008737471376904256935777735332439612826667490624833347298601595420
            0.01800318598010567324269309595629378385100472316095596640576629982 0.01662459798884125680988850797149626518469750651403959650936596001 0.2732199179096597160266552794649474643629106861012367511058986953 0.001596994260707610735821916964906384331135821921467236498531703569
            0.2962975599560470460489142300229717770966706098607969034652316767 0.6500251578644639245691342065010089887290270398626539575673346982 0.03974169527415501764071836840494055356547814687696710879659778890 0.002146592755348467840201005567242130686118574709254169438963583173
            0.09165070996491004723475425153142504032218022393177288115957525258 0.4158417049121764348880290234699362483376389670789155475621689917 0.2217088657743508012400352731494953145266627593756289633397622369 0.009616633513435058176545197243503751953947444467467527927143024181
            0.4429433239706817581231737422668222632756205564210068299485490539 0.1578610362207127515377243710454102636518005018193025027912586153 0.1979849806926998498019920279909917596584955952058084726828945325 0.01186571881442469738686160719066461287364874379346974195092388231
            0.002537413073156058465235974039413787771077421827370215654505583459 0.3423081932957895271765990163557692545760119614453301125154958697 0.2244659307268145024549618589389622735221340790871185122270359699 0.001988779679627508346238303281126354724907559709278335423670392370
            0.3991218481465134698477285926405786774517338575319164110610522143 0.4927436281050363166044438075284716010678407836801585584536173829 0.1055666674190332406018208818399950506122594414646630386799080542 0.001655911487092193384342376155890319196570090359678850828592024680
            0.1390132664743282848140006147043260134258470696110792315837076075 0.03969346474529112686660656940463288249717543173036284721477374730 0.7504323738330591428640759006945689947500335609244172927278334714 0.003707217019999215395382071675344082377439630986670099169364805352
            0.2319703564060756075297483836517741662284518206561635855673539178 0.07328622894178449810826416234143138571440285981441228858105136161 0.6760671140941607433755386180247397465598648625068370687368686944 0.003110448275500639261147979293977398699437944375307229182561900818
            0.06170964991700262662423165437340764606029682872679560186135492373 0.07516525524158817391071882689756145044292173836323490864864139444 0.3209138963979860450536077630894170902431597792636674741974283876 0.006355485611841911879106122231487779987674283029125851659043821075
            0.01593522683706787115075808570959405373876699411580080056850561469 0.07979645763097079037253361747451748606716434347090954160417736732 0.6190735995807794157034358491047560403528716283032913752465177882 0.002873929931164730806168079392634690481212207381178925512797454942
            0.02088766772553662276171055272016325704108039153342663899215519471 0.001149277830744254678260443194821415718587940448190551621945096015 0.04859157349665273400923458631154089178031555453628425078359189557 0.0001747400771843018968391907518630686198403722729617871857975739870
            0.01437359594617999740721322268813767617584083173326392545810249024 0.008024864895907197961122688039288371213036013976565571760716474863 0.9086902793517626333210062991807189111205176436584343144258083330 0.0004270592565282645284813825614051306554142246653934357503131767262
            0.2755604119201217030405508983013068368007269559459570083251705283 0.4898255301021443751034044856928183928742793038749694155221738975 0.09670883137566300287163649317588697850164414626127933761001372073 0.009494809920466669785481694706347595699024184313529867573044282700
            0.5071237093094055476829633847023172605736692632607041781238917052 0.3142631701785262643239165738331861078427276736999972925406245391 0.1390970212209420145990099141599343741155530244451953706777286396 0.005443973847400670860016918609025665013335058752199712715138478017
            0.03029246964053137981581560408874788995020944796687344332533555658 0.06856345481996262373112084517027558796792337035505817046752722662 0.01284585612045402346083306622447862501130115319975017539040436194 0.001059295648724477997333765099601562782710237236822723657479460344
            0.004589957011234967027280635236587756843429262449216112462733902100 0.2525814920753785259455608761802367317979772726755070594026407445 0.7377869892637272467954165617714525434059518377340695315905041789 0.0004140715885223980042225647564775298913085980436310265241881690315
            0.1546131639572702719360128924514186705289594626309581917755802555 0.1786483395495095080574625699014087952177684392479654363566032590 0.4901809337504258391771571314611618224801423669592067077379735728 0.01131013861781615616406980912941644547820461560032434876387322104
            0.08649776318548103851462357516638430849407887246179914971737269068 0.7650478401915205976624846749779781389181394805243536137122903159 0.05947384901020027501157623878658006927976707904208894156898780794 0.003590687391079640597191128600762986512120571147424986507409015005
            0.1526080964530896813027877785529535252677652972313413339879941660 0.1549038015096663465566356875791879473179657360034268428137592130 0.6266305112559531336196761185109995206535351632361174513667317118 0.006993506059418939900303778236396535518917844954310906825331589819
            0.1503903740589106969628893925459024662246154197683544834916556974 0.8115910395957914654369223212791951234676199922342946492948978503 0.02401968818815947227068027038765161748501896959920145486998814247 0.001446606242309616117531167594037209453229520945876369668956120708
            0.4566268858748746743167955823492444163812894494745192326409477178 0.06166297435901600190535399644966551720046310501916276956241215897 0.0009588982797355105379975819295056109217426214486457377318691696145 0.001222099837884573747109294281820525900618515087069071573248356182
            0.01976819292622626261123879618824738023067281503078060445647625186 0.1150213880769216438307748944585071512701782303899526196585614655 0.8530512100338738645712216566237223176826919738490813336734633663 0.0009719595444729360854722615708925863154752680738510504432117136644
            0.07570683770158320822024288690214175104682121896182264377809598702 0.01505606661695833628666866687707882930777191914425710446482658074 0.7569637282784853854433827057763284686405757476965201370929322628 0.002634655570593446542581787946940861650893194261276208285613291245
            0.4855662431492100132599387405576056495345584904639085748834978564 0.02174775125539682503718978456092973518201233855777277827662330565 0.1748273154642978790550106654001995336735758600224665416580597057 0.005632828002982902176724863943618445822615179854875224929665395969
            0.1787134552020605357956956228709056725082260928626884056095358991 0.3381227081467364735111816378169685197358338608066721281901825795 0.01280554600995334154570842404170423654197040553894882532542596022 0.003742607958799031877345092481310678363752678332017901531536304752
            0.3124582544704902590754871306058666050927213035202983633174809896 0.1133438945316999417438781128846946688876751946993321277081086648 0.4143631823015439846904059782460030910777575564099645039360008833 0.01177257344012116076087667696787199708328809821229020121447939576
            0.4723028333204654586294581063638573157167032828801638093625807471 0.5027618801749442647952256715347398243158861918322332459460710450 0.01557288611027003311695854669756263665718703121690135518079269560 0.001212055768951526001095850903345226033557389961078815328117236170
            0.6878272667874377477759300871590406425176783835112116030274015448 0.2888505182235947900564775882318639201667848656306530524470718226 0.01733345106900074294662086664609858578138598210298227229552809462 0.0007749247351327881313262710475299521063640752308609785275073706883
            0.1368166061760075170333362196339201878746329612039081217056895805 0.3378570700121321773302643897630081586411646462915019681430678422 0.4737005782748008833533195439585771166010855982693030390148258385 0.005828051203695102845452232839458406982996189238802675157548097759
            0.05101440958591232009274477119616798402362508021399817688371312392 0.07033164810546628335930302557751380004093486557573267043066759323 0.1798943847980593315640606940418399010206726500867221156079249944 0.003347475129975728072869917743234030279332485844336611149437516797
            0.5286002796400066782284748220286269175967457459075776419355453353 0.09211921176817323383757519946999455894783330140630875250647958491 0.07496183669232178448476268577817810713986088297238156266506605642 0.006802425642343731587164997550467771828391663105171050158795382742
            0.4041887590994516133914397518322203186877527726361429474296285925 0.08777697406349257129336391912822027481119558848256328038693969279 0.1710159167049238897427948291566378743122608165314445043004155622 0.008652159290848692773024957455570401134482236935032592081807264672
            0.4986016578594174630630926242655391457003183627526297524999809261 0.02720401477489495745221640856956229734031694361421618880627780977 0.06626439982863433304391449098754946723144127751537867872244743244 0.003968932914211923368672807561777756672966231712005013132504417238
            0.8794401689674576973120569319248235562970848399510395082077204536 0.05433820724261860368191311077499714723988672569367171490845596695 0.05554361881809035287804058342218370363436075161737356240786173957 0.001159337873190477894768727463158150531646375031270549781275328680
            0.2084507724401167511976177024128790609298892955121051135911082512 0.4412443205840170011900839999453009139166921514149768965678863575 0.04858256259116615859159935780345126723188035462085333920132828129 0.007657352100128567344070778709864163298934872175193609853649793621
            0.07466954598990767517815360551534751517528850726949425770021063753 0.7044630583216496525997365319359277398047716742543814658072666565 0.1515480441914155752597203817344505824706121627091342137112209792 0.004883154005710238674466276018801687658914037370234045484308009026
            0.5221614297168649643610395544648755983738039083947456850954462811 0.009111560549080602917029781467557102660868804657423464498141904787 0.01812425547596696736169620788442168566710818478405339535995092432 0.001134274768113322068298749571432021940681459328433194335316178337
            0.8149029504539449042341663144658279928632260256818308529911958849 0.1523834357130093090037971020661081545160355998630952128352509648 0.01931291809688984688061668618600889014803230219150649931720808119 0.001200522516192466571878835654258736667275477101977677002403331283
            0.009123085900007287654406594570531742578835512379104481406833451564 0.3105865863296788572610212953087143980474560846673017433355824436 0.02006725185042722583526206305033146701117301708893095524397645285 0.001255819450684123428351069349370336880392502134834649385358482946
            0.2029817061920517651367817748992197171388480690840455305871335087 0.01034438944645686384851452886480994079394400037135174663707731393 0.6308471316047665259808648586332446512697207942532340685873284330 0.003130508248062054649135327722680018208648867584711907480313788140
            0.1500963282091070098914472209160533852418251439524345938027727312 0.01870676217346545727296518004581663761567765106180920745978609223 0.8232580558889429237754189163996112390471921969749325966383603320 0.0008881534816303357937225208237419980172399292453025959095057005792
            0.6432569924667936921118518525832680984097696067540252873911608005 0.01441019425301822282530560680870617444662125123166262894964749858 0.3279111121683509418240503195727238616689790583509022478591244428 0.001317305612884365777660535853008065496796145024440310746052060204
            0.3516036301033608408834159889976482436316579992285348808334909891 0.02344119344530040035140506502847705006199748087018788949023455719 0.4610494469322063148311289693020347761884666615501216274808839968 0.005888946134294498835084555016823914767498872722380048253034095257
            0.06291317829691707738046757052574566320271403448512997229124253309 0.09794103738128989953384039629692858874353496243373912030682202403 0.7584105933494280818473654794924206317603068170181290702847003576 0.004199085929681469375507156860952602811318232547377602070802516726
            0.7186495987571602712185011295691250517403074375391202998104950248 0.1327684213606619977470881177687057037540215831808466932576288398 0.02516443503447245408721636434334469837230291560657387056490444127 0.002584078595309079900904843040748570362893559321292204560134100161
            0.04408449201193112591318833544588831506441097507636082862704777642 0.1794912004179711827268330697850637327371300885149859707395589107 0.01780784875007052817044838470318778645760010846039702236644555190 0.002329595378652023371520051553804011151263040061047982419293274216
            0.03244822163822764148829434555636939646951247279301209319903257247 0.0004566977175258663165009874912615711826186160755198440681880988500 0.9638256571020678686965245376414234869689979386297353641354020513 0.0001073108568416040787284144676763258549329269876349097616196172686
            0.3521975029901035564405493831275284411226877027196456041010724524 0.4581326584669652527127967861155467835442116322026389765191302278 0.01726204628506148823130025849248928958640212434615568583582105654 0.004471965079296137688341248721238042889265135741058399334127396071
            0.01525532331654366382207896713590510226961660224785891205523838228 0.3241440404039025075047614317595141468846498207821611934703863528 0.5014469296291045857043999333354949113009960052968202822212954280 0.004207096247034206522027431947897292330437856091951729052979308567
            0.01722828219525053212980226103497093188614050430303869383346811977 0.7859906255830858928771242430406387485259055525708962293967670887 0.01639706281127570203259976392881396840147102897913939301777686411 0.001410253019881351834650766437554949690272697070354908848011430139
            0.5763670798847841803377631480804068171743983500295889549404180188 0.2757236257062549089278669684973608160004629185409236071800509800 0.01436078086934803972591148273394105223923017842423400070447161846 0.003329735450923311716629317259313986400826587673091638930043751363
            0.01079140798194281631136319857114190279225910403618108910536038159 0.8521158326811483686882985850928665718921312578938263020490616179 0.05204849640105619351626763579296067798685302206068499532196835661 0.001189858759722240988634680114496554658189513124694383406144210373
            0.01529489892125755062365031784200970093879790764615792832527682768 0.07852506531793469730210985558336777976801437262041824219502939340 0.4622056494842566489204386794110838834495018397079265592861674723 0.003432227657454161631895939491519722411070574457783340273715934581
            0.3395335172642993794875883017959808070061758087797557771180205217 0.2238545284568717996539886118166395548598858679718278460237705383 0.0006158772772660293567530730203380281954595313259567429155402340375 0.001049652745319599814697702675138931973209555409474197075835177325
            0.008560537663256436795708157507048965362553638286310503182177265421 0.6441005776636571167873044902857364888108802954088255487553104877 0.01553671872889671360235141892830949194703039674478613944122166923 0.0009500046220203521129261683499853549717092070435412038858924312966
            0.7392096844770409142423593221825596178716753392874290641451466006 0.1472208338682479350178645552611576366251846782166277885551669691 0.09658728162507202178011309099380688935471523098988311852952306857 0.002884249890250260215276548387557489461461220511001392560748895669
            0.03632648615953443396186525128707974614015485547695610077266031742 0.5601926156679603233383092999393317150979268555121187056327717608 0.3906072890854663875639116085310627356013457948571216922509499483 0.002457104829321265383567820350275379612248184466770581863743727109
            0.07373608275066785502781510485748014611997524951784154152244882477 0.01182558177910503639548058044763973159886355575217792402286653968 0.1551074013394425168911224557032249547691234847129741544939225923 0.002066529514785999598952221440929424811728857000578406041911746762
            0.08700140731659712906447511693880679444551377439670078171522795083 0.05912563421862015435166405016122387503180339283755140096053101526 0.4774704942707446573487575321478790689204125421577881914024054956 0.006240122713500488685927653458434792134200040953596746921339248059
            0.5639003301656209398324341747003369880744535016262274142829352467 0.1946021464060351554688099089463065551807110777381909334869542870 0.1528788861536454423586479878008791109125361646480901954738144360 0.008808695150335298658837685272463850168841306172968491647611711547
            0.03877787308369615927769517705188919300189561019090202175664705288 0.7387948329582983890183817448950444521667231848318207880284453078 0.2116218693218547562489338436271042560849200957336616064479970623 0.001716138569101147699435119532619969565941145492045611961978071726
            0.02814616401470630143735722981507236489764388782817330857138637802 0.7351661448710731451382682311543054089857159704277322389969883086 0.08858559326738742907320473850234300138259092109114014919257320440 0.002971268031660654977939369162046700075486544262418669845180606839
            0.01781443851914660964301032625366238929652264588703921363756646449 0.5021779302721064806086362215177225434129058449949679876674958637 0.06533663111467802973663863506326072246771509372861117673223794895 0.003127313251880523401173810703954918751062634258538764109396996900
            0.8187605126808359100807353163925254271508523721073410840340432578 0.01155147330314626240392879894577646043425280231412381913444825043 0.01905418311845789781083704785793327362405415553278894679219296145 0.0009389772158320834719369603475642903790772089854495363406788427579
            0.03941154123625683351572440648029958404646422193421817787132590942 0.4455877149187203162604329381226794482735456310687119626722506752 0.3511965971366223189065812598347509279524924442568426489121900484 0.006507011448603330898983872248075557810073319324942363335786973868
            0.3296013925908725167637571307730818058038655417132902853342802568 0.4819544303485450949954284783666456922003737493676313940938246855 0.1471419081205022247467829060973479976344502225785294109119773691 0.007185515063054365345139409128639553190969772581482843094865728169
            0.002590167395602102433477465404819360961845071529096882222156597575 0.03265271628041966525682382802300849527136520852674035950792384580 0.9470951953699347114437464357496169575889923469382361835564233511 0.0002671501304265267721688770791518887630618051691087324779968649924
            0.3566164651697407766670023357531625425121292335265093491688770797 0.01306433488609433571451083842075183484685534168870473382146894427 0.01512936292407755307917850977148036630094645463623214781563156238 0.001231876503127519955494709390803176361364619401829777185295539325
            0.2238620057753748306441203663439326251832004505607109103035805285 0.4005691289286033017585958872033455000338470661069863128222276694 0.2308486601509953028263444903853945739438140061508046451072955342 0.01209227740504161008637078915224952509933363873742580423505610828
            0.006818343131565581394816224978239360909952397954537228475672396195 0.2972926619112877388083491459334445859109048648563121573297646502 0.6330525181728100890539137182694348013407510346168230156762640792 0.001805438710209732119066390167600969236847042294779082729085732305
            0.1975013231240658743120834944664855322837916076312877495594576210 0.01128801773625574918398499902307590878312997888725036829756305057 0.01566215523495959669306171408344665261909602864305837849885767028 0.001065916166492796196064589690553152312177808631158384900776197085
            0.1830394164334555987521126330008016633697075381212324820694392606 0.4168163263084095212168465321048273711302805381890797245529490883 0.1492002819758898047956214261817630163240225602248494498416182605 0.009967073759750882867092561306758373754645745970039194451144633670
            0.01154838842793436852837717873603608096149486941154230561305574124 0.5045029807520288836942128955302419466985904112008817565842130277 0.2682411979684957160957239551956910398021578289373804227423881621 0.003298689654068937568293018723934263579503005199383776290987269282
            0.8052732340215182793325967476663645652112353168933664226907199177 0.07586097478919062704617135354003669929721926774491582741515698887 0.04250761319305863316977046250660236543735627189524617866565529040 0.002243460452556013461772725845828072724886945051758668092064113497
            0.1668575092086173355876969841395121461159258476641033377519295069 0.01414564726758250266246775527318919843538573500867121951700595736 0.3879902774799147197344305129874724198813580340774238935306403638 0.004331873315243768567988047077954155539585519324737652901318211645
            0.2710312180143658385619887465233766130679482204353632348138372972 0.3080207668519443997066511721412194635114308865837111840178986563 0.4054398142104562130167928364109427421118993776207777172750565890 0.005468906439891876262989935303412938685983003843485436533644483884
            0.4479234507307959612412812853664464358985651088126280930670352242 0.2929686820094117290606203618842704726977594687613135585218608393 0.01184275236515181694927410417205413011348056577986598600512838349 0.003940575208701151878591188564822538118315930430065897144111928549
            0.2144559291284534709342121569001766889981848423289910286719880358 0.6490149078022880267748016930274865451068214884570709207969255195 0.1240410412004488029462821097834235655163302388557543887217732142 0.003117174139248275779317877574723292049583646167149554451451270918
            0.7418398055642094001050016231929121045839530522003503491322700436 0.06425548939638640550009991464624926845394011861554942627640258608 0.1818751923702033975788416863550615308376887681787524364541480638 0.002251267305126320558427638201241843593519132004401156800644041554
            0.4335958946567556691395643384433439356308061712835037681429446147 0.1757597796057621052400878891103146246871348873962775370833704266 0.3035992909393988092670633924654836793790274894952171822107319510 0.01057482096831172706665999380591981924054588271478748810113859573
            0.6407543468203217981463877356812609048352352621575529212166678384 0.008670624682112583028271384267056996012632962148182039699316027171 0.08673915290364244714529405317141341239104199520765181614178697330 0.002404771907953059452817429796962949934390390263694735608004369817
            0.01934209304334884401768161402350483026184863067154856328684951153 0.4893641409024914297254066782923083262452607536889253533102966106 0.1619845186780136007460567290807589166161018868493297928011068927 0.005034341763726736112174655924503446517582751924622869022786012302
            0.6230708708846380232170925697393860297135466572872783023488270321 0.05110114076352208261733250376316173950988354941481949157274801111 0.01431575459119076016828683521614010623085661327122925714852367026 0.002571739365774268294537518948538031774677085522171532785575767895
            0.05032126673103731669596538075690434625942006395353103952158091792 0.9404480030584127658145232355111109096070207220149183338276635962 0.008677273561534037262476041925267637312490185525325079437677663869 0.0001968782242138384382447624468516334666527770603586358078893019204
            0.3845671228573105393737739021723100972654056506864187420817212062 0.08412404193717962485580271083251560445965734603496608938802119655 0.04365323085077761885755502559232273156421084208504470654035169432 0.005801625789439186848855092189355684518167554731126604698148094370
            0.03790827219361694183334088871807650672740003913540941742680822708 0.2050981330399192531129234347569081160810112759395159877960567031 0.7144721779069616520470235952637161982269956927434864892630683573 0.003154531407476796207010632568572751925837408382952185814628984716
            0.06747783217650599946820665911674302143487652228167618735147632354 0.08686150842307448182072362442092387355652135993757468482876607751 0.07402273341505763547224702042570252738855741126344674621075726649 0.004265183593043566428444316395843381393041782680928801280450186021
            0.07293532871326000091813119238550057094661952420920199970180531504 0.2555733876591006579800426147659738575508017235088679019124175594 0.5708803239766715453305190185239891420314639372698379478127847836 0.007483638274675253943329581907865057285930022061468918629973388770
            0.008139557178508007697940355161833074486455989379335929336030643007 0.6654324595585285755392213993277752146117766016302452098331450749 0.1974801965974459851285938446295590031980787258825535999528202402 0.002177240236307017885473141179392109981717863807977743552731991258
            0.3885507028368997089517593605272247274353636026500344156977572947 0.0006997549287968056574719344471853133899382883887120792882203188777 0.1190530389822036359825053324551976896046333602445887944975657739 0.001592523032556855932321495926669420378383285099837375804319180129
            0.6195676888345970453493464218780476322999268717865554465581360891 0.1473952905272757012208735659917667651255884508050699259819436070 0.01270094618413276595586931790672310873063161630058213410676400947 0.003094249789440537790644678253763043203046805212201187203814931054
            0.2864761197064974657314816049598448505508284959329725976965332714 0.1603488017024008578260531834697559347220346825897317898674741840 0.01467383634790590631265356192973476700686100842729203742585879181 0.003366070402381878680617631400162296868969356278803410935602480912
            0.7256427549702340353825751921426259646523773433278264291097071494 0.08170439505001495041370087466605563102943175743050699586121462083 0.1191991910371068691614001938995440835745688335174732476607517406 0.005064442670090322859379630630614478392492380456347431969044476603
            0.3239434707461210423070116073341753262191361034759619382659668871 0.02355505432260260291925095752325938278303081771866610402432775430 0.05820340311732460437392790424436930615567480086966331818151323132 0.002522980168211154613528781072883253544109627271016428248366821328
            0.2194341677339438198277869897344974605978300907329597457829743977 0.006036621349577086884936036974716883697562073843577465808554678981 0.7239939993980278338848001198500859445883887449596300042630097064 0.001224599671948742101571388633139902953079053917778096180946295344
            0.05275633882399241919013660093011826260659603682560312549015291148 0.6256453738531274617542373621890752380230310391410591257234678322 0.04926263722371804741402617707796628148135431525443097227991196820 0.005055999655406184279723532718876635781304401190129423741919646903
            0.6003555469844485849691972374807955334832486593847770502652293594 0.3508748193221153184879747665754553327230102253442402889293739111 0.01401892707623869164096362841519180583471846202251660165007084146 0.001752146768145377843144482723025661624688445140543039410727713312
            0.4832448967068454207674225605303276978093072529043634384543664536 0.3457484395692094004124815368748918572515371948203310408201096762 0.06112006885153006544606060353447144484267034233304490323032075766 0.007544493022655318031694964402466213875522150088862897188597281589
            0.5972539152277265540567054519560283407048413033312688323970313396 0.07203880580291961285406825925999571111702727356896863033547218377 0.1433570176883251902375779125792543715768889347871521617877417607 0.007304475283189379227529991082004649605665792012714565572373901912
            0.3861091521162407932244495813829210482260630442870606190755477506 0.2860567330107484612490199978082658443130686389069102650619679202 0.1598686315377380854086299047280344260277952261170807783411023436 0.008371491082109867596148084270043964629181456585327225921150876690
            0.008244117110100647941762426459268926090123275297112955881922876413 0.08674085758745969556632191485755126518720130126966778825304369781 0.2941722260238386395194374614490149597680381230717821413233188895 0.002166357609903295577951517492724400467956131882511499081408924844
            0.4273620625723682845295768508418808779700506772576359522902934467 0.3324203601018216319334868173858906202871918017183015004745368851 0.2316935957241476698486682655867276281691415030650803793149862542 0.003564596300566498709538951129556499006382567964779213212990359796
            0.3149502891413282058915574428253114536605368378175096958442358277 0.01501664045159101043111745701669047406203139052338994076794353697 0.6607930254423315263175845843061954277412341644103421208535905090 0.001021252364150246975987688721536207300857116923069466997468043133
            0.1686235881586060558298338026922153676326292777399504311110152419 0.7683491046807376469050600358912446680206196520372522938337333490 0.0001281625295447169561238219707776649433594540336692763261876427961 0.0008230047970759861050397161183976926382171224861830455563498467466
            0.1107051670133388942070647004261000674471097346544541682765588854 0.4911203677534937261425255798993722195690013450641028231369497664 0.3408606987606584301503448189917430923871379952223642568439943114 0.006167886580323209177863656628497668240102319719545687442641082836
            0.7869930454542242783263889859962437090286489948757627312923270430 0.01450876348757584799387778876681709365098093752113130521857927563 0.08794264193112243593735534560930214240741786866946772587468244286 0.002263808034144244789192348895575226791573615807807545095029327394
            0.003749937920247321391777279210258131309319783151612294745054171734 0.1233127859809953905751262478639038577192785211245477235767287939 0.8055318525222386121967868783731814297653177905304222309198124955 0.001069525595964044718911034315173732036894797711990147046144773163
            0.1031984357830612105591425242711851332417926831141610448175082405 0.1883641415680803676393885413457751955841856398887105713070268366 0.08451283902390550794501719883653634798193332351035994719769869432 0.007257945343527347476720794905123284574200798577091137263876332728
            0.2120617701009230115236867531814243691110023275055498173341244418 0.01246539493779747415623422876494388905379173841582217295742207474 0.09269316919916544486696975429824306674037726265755511223538896424 0.002533703189673676917066391886409059985648751379073426098434412100
            0.1884593099415951664710224090905564462529722591242982031655371003 0.07733819238855424882166108865899187457045103090643695474625325075 0.06559635408776675718636891522119746317514295816430910232944935365 0.005912791997154187594062151516219679448818111752228800633719461078
            0.1024147913616582437576461081592824682439322625158963192595629670 0.01680751344586621099202630696610804649776557575396460137997675949 0.06308427819340380405266587076737496342865180042950670006237877846 0.001901251063690673435722431845389286722764665786169389308269782668
            0.1039487992559377073669803596672939218604632511710420138504243722 0.5303367878686970090382262233696572443061438307339348299547516088 0.2278048915191014969032665375897793633605595725462609391025406838 0.008502107748820873686663377180641802350198903421941125473315289392
            0.3246950379537776088424286293185692141755863202639561134682045075 0.6289004364750518305914136574352691285104036551397078669868764599 0.0001581669498442743746984183903785648807397907424534166528894836259 0.0009597298832538465439524227117392768155490992470932650374809311379
            0.4740553187635024932806193160410562633732345378977907463523991249 0.1526276841777173769264913456053923561689948112420313358264253934 0.01828145802550024761671669340019098373245699913391572722349283247 0.004713737731515570216351554131604061694579939668731191222402809945
            0.05414666327224048383299773094807393253742876223198386473157164466 0.04309351116231158234216746501000929257317054961412070288757869095 0.8840336130148235710746512890671765055584888184413240979354836221 0.001415769280102371571887248651089080388272048083547267156120944533
            0.08694757898112567194757459908539267150185410841009033851812139041 0.01415657806206638588085965425901661824105967446224133380181883524 0.3001250554216778756098812073471468975244690044718410278588939344 0.003172937650276073785924214913506082380522734317446387957252119306
            0.1422938100747858640360118540902763248967649072660627609946054482 0.2622451566361569917434421740053122571556708926642176414016132547 0.1822275691456902803627698150046892734989458456193582469615828980 0.01105846388137659890888382276993110407223525991699140849991960124
            0.09286597395858354643265745074234265932734311387020962405788439765 0.7381026052244259320177142707031322073416394565106096801147502812 0.01447695600164113330442723010577727894647901001193355051649944423 0.002816685254144883381582331780339720729578440661774766554135595878
            0.2580444976043375104856165312830911354073004396272270083680570261 0.06913476243098621600448037526488613588282353167290583233574770719 0.01102377266481772982381018653239030903171538406932227602551429064 0.002310090189744299229489425027758346973710235438747516853404994312
            0.01939122052506002995771619474075976880079027880588980130081875309 0.9006506990266530637463145600274704845832369125799445288143542148 0.002501318530885622841787370345643727044626708213469566968024740277 0.0003484944671082924113660910752249498177392854927636164700698621655
            0.01222689868783950525773718831399689031069850844029050269600251460 0.08456422891609014046296715284106768556563851868055279931690489709 0.1541689671190371763698836657946843584021992902555493587035786336 0.001937192121953008633424680423240182875635264447494636051842201602
            0.4674293480564264336477969524506353612991204867657481330708944172 0.007260243140568505556251670526569550449001007258812091897863764490 0.3026495805119507701802041795279475408602461101721982779679571969 0.002927865852684498403603058280052242582500249297585090408752056233
            0.2744587766504187471838844617891337793024298975685264247779903273 0.2780623726406590499329597637070648347186486654359714616067520802 0.1540389913122847351117660121398713235984439753319980228537545384 0.01329332074401798237284052773385698862742685895733368102951272136
            0.5946581145641845114496003015353471623672734606247535868926280223 0.08182981400739705800191854994842067165852881467749900982724824969 0.2456494256699516937342168775141645732844206392135704703765844111 0.007052848069612741741008018516387203355383592295811157372696384917
            0.01939951910625216241145266210457473372027166832342547957519362720 0.2002064244062748641990952340733977113714845821797552025331517638 0.08311559873054691577325994106212067610082494382567191124253241359 0.003530183419467075991653043459968377496437079973161302342434959294
            0.02240072948098742276578552338276274868074561616353961186293499773 0.04194495092679412121848939708212019442706595381487534354863446140 0.8480937363472108469290837058084460485800486862580838531831353760 0.001436200139285426246112627861266375570024961177060458254575451454
            0.00008354780690668454115638122601489321200731259331107232770631799571 0.4610844414098620012202561506596275693280717156086362212320025319 0.5234970199791879380038246634218446832332619248662091186196235620 0.0005876467339485000530710583252464780619376357268846783480816844895
            0.02174413386891738406101630395271640481052722781280855355063395831 0.02720268436117133633355774288888106167322577884273624796789819081 0.06496645032583040209356539192801822134060058653394865070859188260 0.001092265739897471018157418093432006791461312355468729747611073159
            0.0004509623091618993071694319906590151648105453704926111612956720686 0.01943507229426982789099811675511512737904825507918369703981890613 0.01031869744687278423923679776084357816618012945847430657219566260 0.0001231488762924479115545465754271967967025544562523275018054585104
            0.1788104956411348559307610425650930287256386298262678142612888921 0.2979708009915899584044494783835308733776933570316410264130938216 0.07896222326150818846734516760564534055992503290542785447942049262 0.009630979435272378012097231514734116404424708101271613249649040497
            0.2691840945881946070026982925830687556402741816892610825472037031 0.4166430242923207540809863644853327706012368737956790758428770207 0.0004594881723079403739136191017916342095879301007240110320400294064 0.001874335950350894639434954603205350430124010250451739009749387354
            0.04951524268863122489500482211399137894116089566200920871483101247 0.5961445887001805637821551181370765313623526309478509390333212063 0.1539138168566125230731614635854957510540645130442728407895673843 0.006031254940027357269463844530564403067149018546972850631560090233
            0.1177329594073638636290691968700790006566113159367548205278509056 0.1069815935589219099836120516403674907850105917949277974221464961 0.7612162473415465315481799148421090287923940214546354220086285236 0.002351654883169884545547378567810878134475810724811870815187327445
            0.7888806408595927863704718591425892325504449877449576876492843071 0.05690461077899402056663947747600822005944178006199156386384253702 0.0001584240141445709190223148597328517487386372463543701301708744500 0.0007586710789888989223242405571417922695305597771034690543047940965
            0.4780328445649114531104234730158304303004933391672285136462847258 0.01641196797891865506056142473107783985093415560118609013990874228 0.4876076096034971961343824046041128499870369796970560322687692244 0.001868114801981064649485159797745018350756352925110127214756673081
            0.01188079563527177350166614263653342825889638748680078985228004906 0.01630871909460088760199414024689027020785073471029341664421651028 0.4203379975713523049116130301600176101949310292970048853333132651 0.001202743799795733557163657135902599062808368804789115776151859209
            0.8561703812826332169030000935022087566187948069411863341836751733 0.01498072516385381498850597933179362498020057451846351797237122309 0.09873083163700270949586959715386334607470543625714463069713670174 0.001272321522475327337751794873389751054829193119825187279511252987
            0.01554522203823204410662733981198591487882208373626747715843285930 0.1714426308907444517683832861814803344026283658774284184250809340 0.6552790055436346283444975903190151758924244373873853855034514051 0.003869854944440657518229320317490122431540548092543126873391050688
            0.4639171497844720876388362905556824362081771833609574019704890001 0.06360182358119985435068107320526645327450183182775293612018916231 0.2906637668782012252316043079601868732450859896630644896431521126 0.009094870652782770678440078712586549166745843265172415556646067148
            0.1379784984860559583965087582701022110024595577665155551881190231 0.5939905122040434819246930536844422610564220782892000736991944471 0.08664751108325745854877213634613948149300012086329344280125890084 0.008915147848863428730871553269343453502809400172974148059549822696
            0.06325739127407507039376827048106911808823952815298626713453580868 0.8729942333248957990430096783378124005736441757002761329017736888 0.01660487881812894978706644897041497152798836534671618959767946974 0.001307090307434140703048278768200272048049579464881587625659993820
            0.1927134424597249302461124592145245342266849403024025894357769141 0.006694213840666102342455420821765435924130370465624256905651075334 0.2056686197355079780452761263982183240989787839640132293968858399 0.002165278796015686734695267253187134222125246952569067793494364235
            0.1803499185508771568279828860319943831390071903991160013312969298 0.5886239491622857570006122251644344239052598567179350385973020731 0.1588245119821879583606852313890566661782409752452320435385532515 0.007608161267461628749975352906530858897735230112935444768285034528
            0.1699681026588144030012932191384889069526337434157411624645907366 0.07242366871980576759471203344194597861567060119169770367314969158 0.3033740833422667080446281302495800001228733086359775725733400777 0.008733260156989133639147388843579262355275381001684092575878026450
            0.1239545299400667648437717481603504304528414784307688836703527509 0.008051520606762474350341345707269612647212400954329512059769491139 0.5750233419703545714252321054128595354947077571572002904568544804 0.002436360432446618864052211485076801983355236383584950118620429316
            0.7173503751666073709234078502085691973868949004992467602873492173 0.01578553519528123317283429992514330334393739621917917211456533103 0.2136384801842992799347460707913757352513293902850679018900781295 0.002493207925712130285430239954648286870054527612977508448857159130
            0.1570656150482394974962692779287897564229386120502242097961990055 0.2669580224714653050532427108989201468092937629765085544794691712 0.3050011867527907141260628414921626259781319285176247939230262027 0.01360652451058218854410080151765468806757961338273957987424475751
            0.0001401628477277679038613346238478977071849841443007854954337678589 0.3421577512329646772345659555447958314130900563507855685086252317 0.3766703721352599453227204004751418308417065817039716759166930675 0.001621005232391958038338307297313774215199339487976427732771663804
            0.6180228622089090900340319696680426957301568629406765382600192740 0.2830377842887094179760835253005303396099765047855129370634398153 0.09506990433658796167276108166479566573240976811833270109044957477 0.001632257232197700778108561318036572489708968083664194879605744647
            0.0002733737687392729158456523003904511977341470570265296291143853137 0.5085811750385754199689889587462236408109427292642447325376425806 0.4028312486506857794379957061015089430656104620743566782712935313 0.001377663038771117404824695230495246211650565063597782779672889561
            0.9131118783471189485869485812461657407371972228917487541661356201 0.01578907432356950665231074940884918390700159940803592183152618554 0.01768732946830918523424476339591810318608563252689646040611164199 0.0008349356228682203386593739557901102691327199322096469976209426413
            0.2079037015729775584943793037524526718455491408941321502682152817 0.07565279663300800609181805263003378051076432524569376424972788963 0.5893979608803539421711027452527194728116621854523836570282758668 0.007313628359556857807095052817424163597827075234312810770302849326
            0.001963816437671349477960507076641093012525560593778821360588217998 0.9601237371891934373924620846046157550721955072445018447599136923 0.01846309670659466119045337369679723464493021646609620217914431026 0.0002034851120794924128039571421685889480535346208728519158798476002
            0.5110184891156848070005947674604122968877433750857398401429652745 0.3996638929268733744574452518602952392363955065065359436154423046 0.06393357066050390912248314086368089682611054307946075630165780699 0.003113727266521766967327785484379501611178326256973517254219709097
            0.3780701629766941723021100281436372865976935483086695054347522649 0.001873888467703799025811695374060896284856820208090866669101210276 0.5436968748704054742238217499750964497166948667979974381073693387 0.001218126556837761794690670864428127814254644012583766414296849788
            0.06868131379818039756965559961231650091508380956691492777468528679 0.1271309328841465006140758302622033789646663910667578327780759142 0.6338796177287577008910140508501938772240260964156983620248883207 0.004775609676902077515257176385161473495487989470102476094084789865
            0.06136550432887361796095632212252027333608286180330854651541694520 0.1365396198581448684995856003754785331598158285318463254903976587 0.4921676824612992018411388326747266492953393888594966513710305132 0.006446991348798300920812248935484231357598797090938402077013308617
            0.4402956885534222234778424047627237300400256410729422560072155698 0.07254891510567147066704943439406291580037450586855929300322565398 0.4216559353880972930465864337817118556713872556779707324552566371 0.006521819680930091791777206408959013046408061003296966272385981431
            0.6967271066587500479130105642337401908715532097879451609966212399 0.005045878286231283706849119669406945739242525402843939598641847423 0.01390540099150051092977921898975294844074869738656574320016944731 0.0006456111406031578864929374528022718791864846491303151087039404320
            0.7326757176049370796412841026254435989148815986132012247790391719 0.2017360550606723643779720318281847864281205254114858346449381000 0.0001389996304499340619250541016141815465042188435634329142083702924 0.0008752431405157758875114145563716472596286257178448895881772191188
            0.6689529673198509419465804497517874901585698110464939240515840301 0.2222958548081801813855317306653717543419512990047919111951950649 0.05249328443890520665096172353412806991893711301286712957382429025 0.004970815607410753791369590788031030963840553458131919638807273822
            0.6375404797079031109378600374708435596299748306305235969085408806 0.01504514542193731390236797743228860772359285439843420713265930169 0.1920635079861887960669091947869520493172203826298618806411606300 0.003758834020771922151887737525007418272519529169537745586587792652
            0.004883430088078787642069060456287290617491693735646567870080271887 0.8733299176928930746191382676325665982574224682871154832408209308 0.1215470742659140959814382329693335146787230014136593655801972439 0.0001800617501246872348938512841732636718590663430747064272655853423
            0.04375537658039110492377930315627109438294483507712600279901917807 0.3578979543807628326008567852913208173737994368168052033302589245 0.5853207451902326141232492938092306655143434632387769382028689315 0.002553570324320331851705867387057114842202865467824070788394080053
            0.09249381117266769695280822136905052304357009237615969142429684854 0.4680526366145342728313019242745258812025218100495567832319363639 0.01886237315919255978665407673747426138738815164354441476384389564 0.004249629009346016365443063221228169982387680476204797170506777055
            0.09939528076427760726798986376423062972302260002538464769078724387 0.06336546645215317645324218443181268689598524387792951069451799751 0.6281478066606357424669074373333095067368052520405134867879876717 0.005890151428647484646801421910209276068475135305679908364066067629
            0.8525651122407801311358586894331469781644474285277684448592936273 0.08428666216030737726048053395652496243043034808285213478275297888 0.01220947531098503049386111705485187918348854445057863473734110362 0.0009040028348617673640740601875343351414920226291170525367447327536
            0.01103434366013720442257792236507572899741646980857216851817306194 0.01532914789658429012646615709848340343685556902536339544080565324 0.1483884804968489919855545075026233646632935505799521751446978918 0.0008010223481927178134954687699586516315412893899035440699308098372
            0.7049300886351169033517789045459102292976037924777126269682902460 0.05083940479736236328838448742853856916022134772869900183637788506 0.05030173003411077695037962197513582089386856457498381018631156769 0.003855390429188505717406926360720842239335947468197096362033286434
            0.3653169395857554836000959864264567681968079106438754916643111156 0.1761297017406718193792981507918742979501069598430370276593937987 0.09934272904792356098660478087113880236936408562913271124012645302 0.009693257876320675312098666298128707011723396364814491159847365370
            0.03178904837993392182306909136615996746107758916416748366422490983 0.8862070264244655909135676347546490800294245144788541792164136992 0.06369566218917034461062614849261044347404218473022763576185262548 0.001207991436765972579605805147869980055115685701625362570008279216
            0.2240652862280321491613691665429699399189512757767842918662476910 0.4304036027387158915010158049249738772990858085927890542901767276 0.2913095458386299305023185681540754484574867817873537514945787070 0.007694332031613710495842140306471903515356138717768056830511642554
            0.3159742097653183671515240584522088474331633976200446955213024006 0.2716783152532482701308269952447918934830561994758005032618525177 0.03374272112465000855189461584323712186113881520989039456368174059 0.007155835600586895646091859703099303856007276133669259287698251875
            0.0001295530871528726159808330102470849846233381666968455396461808849 0.6827884006741758969266918447644031680255763602326797304371245659 0.2964020088850770049672311016298954049325696305874930922073377807 0.0006412599737568192797966625868006704429175134053705029251418059179
            0.05799197211304155951635392141659724868910766984369440094981943040 0.2479899967194976601695624877671464111567253384502699244557205746 0.4554066081402105087119785222640899784206733648273902295615901960 0.008343174038472928250456265980938084182371580413036973005569352561
            0.4777760028165082502638074356953397267186038737292547062746229006 0.2237951880476382813484626925695657892186976138263280918848644605 0.07333233161163351427734016654107284402767453040602024877530045290 0.008703278051787421299056577726154880237069992801000075702618470499
            0.05021910192540609830590325744463658957211411749727409047396286408 0.3127511165501896734158950989145607879243340271197332258652401318 0.1756540296078177772096757579141777832688700118752576376771090727 0.008263332398646098462099701234390283441903254847212386783685143470
            0.3180878559333984631766299767104917773852250601835585472274967001 0.05063545313361337801803014646819156935372346301047124910999168749 0.3175015656805641883223237117242094331176665795879469591208120022 0.009538795266770978023856112838179063945100206629724774229213417617
            0.04057615039526054146069542160130242533956764379226029383939949947 0.01807361948827851810411266650688515117944339070150354516569944613 0.6436755817885805827335751260213700209647056274786638750721137657 0.002381885411675126515333739091372603362685612470633361494260232604
            0.1601874732125395092590987078500546309184795699782883966613013765 0.2088038790750002908803683810712243378659894170070489502171698546 0.01939982552834983425020098337829083821823381223327601658433454502 0.004146312888213457313415458882632953939598077818983934480828766310
            0.3143614287609899944090142061674017386774769986246722574995057167 0.04175264341212441360025099682412636649980996617332435386855127878 0.5810828434842966279946342249237901466959931150909475035987572607 0.004514906462416352220534917855860060836213436019325824273588820228
            0.01334671015145449063380431427669664370585395559538069576718613090 0.1986838455844919402625393453148523679030220552215386081741002268 0.1940867382613415461058502004808276245253020369905087821018392352 0.003554799345283807343239108864811814455259375663891129482810306347
            0.09206713149762184259544842117872666403357442858087137615195786433 0.4523840291346422393961438200013182903278188650234190789790222304 0.09763007476080953871334630206696305086287090866949517760743679908 0.008861597789870892421029877510276231943960413340014454981354716873
            0.1232081549727831747020050970774325006632375807077956993920534395 0.1206610715585237288912697109543722180626465919232439634932444386 0.01227712139703208497907643996253289058280990994946156754738943247 0.001993543192797624142959477108109110595319556270349789365082933340
            0.2339426586555506451006745072477291033891054836943389173487763464 0.6077604651080530753369618487233067169987622833444350562315400462 0.01784987849403398124857242468631945014918823284859751355748444467 0.004145356952875416274266974559010080832563646381081083303724594723
            0.09337141068477132032905242802519369090598121095194513853996039448 0.1769948177431515458862249457916147168917619790676324343451027573 0.3197241411591448708931032973600630574054744922009346636686556057 0.01061150958820860204171787548761180649588469908822017965143148729
            0.1192271943782571160628151354990601021262529873989195725654212694 0.3529245448999198536851801721877114583188192569876386858662439954 0.3774023214724739205525423645752866276768914459576207021444900823 0.01124568974486103418162064461145460851295957874616643118279826256
            0.2840327009287874846752416012859638703292882522412198189940285808 0.2442403331703994522143314618693797805740681358410655348465568033 0.2960464999701048589039583449637952580938753515227739319424051959 0.01190301532578814624762189380085168162873283029440918768772942935
            0.01367187211234910305973401552830299228586972569658303404478237228 0.05206320623592204461624121186567630344893704154342636204039993522 0.7559691691085890902758826713043540601971781395517825428667720826 0.001978652677174702302419010855218607344530919151980977947993467880
            0.3832195594719441280392948114998983145635533014852270333393066788 0.3543927000968683625175127081980795383728399412007324707496056481 0.1529973018859859849908051500019669356054622362290414454765964280 0.006582738123706442596103514830702592320085621100769401083659527858
            0.01859198698898202944045329178346623348408807985289152317865662233 0.1916073455261429448990963979098342973732854488599268995189014224 0.3417632931865948876899487128334589288037814689956330625823922812 0.005655144156006876486203822356135224803415498636020984247393207808
            0.5381289530545871997968899478763788565659617858251690159223957118 0.01521022795718391476991421747776058708391212526915098060946340302 0.3581042362550454773813352959847444350311760491715737744196557946 0.003884703104585935585443009344878545762262429305813648264327132551
            0.3115941082879443074938040715415925071734219746891563467211459614 0.05288260643681920900334104875507187491252486546695693261414310584 0.1408531011316559374666775201990019027388990872653879389429312306 0.008303800842611250995540217507201854933221152665292950552724592879
            0.6095891066136227861648910698161230621048577376549727959848188873 0.1583316458750808843752234679713156531470796891072007466822109977 0.07625342650914023865429899223172027791677310719211652221068218745 0.005836996049295009672326828030038554700715014581454174899834202565
            0.004872865915402102302566962651847463613893717738135081923168157063 0.01546170138178116944670282450968199073444572548244309589469493553 0.6007706359326084733978185320871964455253940700975192601435323265 0.0007583237515549898153093555878148117705983557605921497465977258555
            0.1158252309805302806567329157874235416083265822023987272942494597 0.04722409103141988370812554768976897987345296148451713232172278931 0.01355685414379169984981254945870392280902437216877355044142111800 0.001286492521609441855860217478690036989024965376638834077427139758
            0.2935428754370042592908317437868608126597813058553034416005075827 0.1516410054470694494273839613786126821924466240566906827702892822 0.2856422729953109012586729734033910757470277968207516023066638594 0.008899107005314475716936320356397865439419729167425914731781250304
            0.07049784559075366195830410075246361646307406962945766216998793990 0.1982848498423746415165085375306367704240793857534323196287215111 0.7239603861247411356210369342259971473769775974267093145869014060 0.001189236480668926105610205837633563690886482835274741279832497282
            0.007784178791856039625520638657243979959222138987146483694106233538 0.1895629060075985601607737722842246543841659425076224491605758976 0.5020400009768182991693231252247886685657233349385126713187180061 0.002889839089754711096024998644905502585759748634455700493998778302
            0.01278380910840668920727423756504657354220055697524172925183997844 0.7950737537378282727437544826104483815947899364784692424582065579 0.1440795863006852792338673024317056735926878942085716542782242852 0.001699214152159150511599767468517933300657039136885380430313310392
            0.005068453357026726150483043944203844132601889220395520056048257186 0.09838475455486387326091037109722533978142692285012864377357436462 0.05778842760423515599276285378541354608147364238664737351211509656 0.0008757433108103077539955828876549504999249682481078320287434869424
            0.1394840062580083050096352982896841888005713678614552137748404232 0.05411405905298610480059905585916707649196823436632177048971078855 0.1713647521301595650581705644454562777289706275678131957188019386 0.006116902847439700514776643587898960884456297487402897399976830311
            0.2815580130777105423278413489443711396880872434901944145970079396 0.003178432030266401642545710631715115907055703703731701829788236158 0.4271247320489238645727503410521961747018219174030349582715034904 0.002104688839985908011811249406728593292952863620707006783592095175
            0.05424603939641863706839159282288146902791773875261869063675312193 0.01204644988067063407810166694945940605386882175762347886156739537 0.4734828329997077020684388841403877844225712893796878626705872397 0.002322739483044276940699419566869934538585747231407095335854907968
            0.9661486639565208939119801388513602550195938890839988136649650226 0.0001313671677027616845254117534819356636213675324033548349974424494 0.02916143387097869616517312770854315617206467042293039397327303519 0.0001089514639657660116786573215901556175073644603738288735754974275
            0.3344428585721257397400951515223128450886595758433749119586218856 0.3659911672857662999683972526437721508049532288645078623763215275 0.06429312176496736133595119274118022194707998699143832043479770058 0.008024454157344759040581451494208611970224829752016635493539177690
            0.2483704483770801642176753775280006420382109160922183133527291516 0.1474111372159380175348198568646915777533728366870175831301414730 0.2184372494588203062879145506970102034950296330539560750002664527 0.01042764264084496803362349562915983214028305436414657961770266040
            0.2458695583054052290882754946856299422282950704418763047721684540 0.1776604268780574158269359285691919604439649283828997444703717989 0.5711146838825317099181859381541411330455914738609620642239259570 0.002181826101794792400092393476393151623301346642786816982899130987
            ];
    case 17 % 288 points
        XYZW=[0.1330260982330632895105318222210457382314194949085768241471797956 0.4602551960004610435309259050149488081159891000411513719764748557 0.3961269295220495436814924162314780946926596908896376170117854798 0.002872059451611216936923524123485829706954653600614902628186153280
            0.2531361774984347563062697813473761230154374833153062461473510675 0.04164191312744571524649732255747050448846989476353392634433626187 0.1481747112792672760711120899682923903735551419734830259268646988 0.004358723992927642359438589560335173887023375345024442289198248212
            0.06898365856408913943630228793468171157895315685655127512518872446 0.6502128850269820694772418941736358815470068985728311192289518368 0.006792271395674992636535801185917795653244452815281364029330339888 0.001594144432970875233604343172108350203445987321306106241159757749
            0.2940118785738613230973642162830130992269146896671883006042816721 0.5480132903428303493560202277548532109715683671938694019300624685 0.09659001573838236958253932341023474302093108346942918041255804388 0.005679080177824058190778569584890183749382774820424756106397098950
            0.1153024358653840057336669138252022254084793103241200875999020208 0.2824071511788448340521226133037860009723715060288834863711448589 0.5286556318300595108649753216414884831548859624762867002890392258 0.006410460511354727953668734459697992048481417381104184367023842145
            0.1761675109361403984185364756363529339912376242459393205138131457 0.7666300463482981210315182980062606888584549788003320187412888359 0.04279050823792010569903929659903670046447371679433284520837614481 0.001626694982120542789069474470699407969139695030205389446386589025
            0.1386953383305265245846581407410399648932900945877036473257222548 0.3252286923335633904843963714924942285063752017740738195555003600 0.04931140523103574508787715560240509286133710292122136749718744182 0.005100481261331615133381782709488000217095213470243554713914722344
            0.2653353315625729913638934384144761267158744620136735868648669520 0.07480830140460867731290179340676106595038694001142728872418104830 0.5969381491403378857907183731796544738247269340861251884542624898 0.004809818595746665857534685665411100193291389855787287911600396838
            0.3256806199143691252641273879595629029866717576883564444011824075 0.007874771181126466339643222818674473034195523267951724372405036300 0.1933293006143390970711089909537358128056980569272095242745041342 0.002473396744591584689651936860483816089765361186558504087219902137
            0.1641334438057187698233106689476747844814785463629941371926409353 0.3017745908760343765223570560589302373902464676060640211235235818 0.5209343326608669608672215574779839402893066272587739164035878670 0.003288337911549313481848854206610308024936639579658841729212573659
            0.01442020656902570339486920509861339887075829376075547870440099662 0.4097406491737708208253168896365215750006758865854115944711901985 0.01251101809757896468516856482309224437151769546912154151451279696 0.001171592162151048068104433840262090495225897336281065428811322475
            0.5065381291665307440521478147691091436613692631868999693887374775 0.07314979909155542680967193798339838146993498393381910309282038967 0.4058356367778804357070092044470843273110233838779183402229911066 0.002451315055249140777046270366637112684642961626306837848232961127
            0.01571501974140698161975823128342414600841125784758623247645958588 0.1508977301232610286765157987204443736215342141352750701543495438 0.2916390517417532951034476826696837342222528635174472363813542209 0.003914868116017337700556105089191079968906369606808040987934954154
            0.1665192011354268681743500153036818553089953580855704566020326981 0.04434427839791802386018568516943641605404773638599521113155606046 0.5280810345509243494087764734086393396273206464938570952757403320 0.005208978965417211803686455300131368294714845685776203659650998558
            0.3722418619811923926384753658346848824999988963767529494688858914 0.2462923968117393013684920374667601472136685337753204006106623837 0.3319505980269829902640709296800724212901253681306011948979694108 0.006538523509222127540606968252556724602982421393428281947702908108
            0.06359610396921980019033500589594203555000442952151047804961869929 0.3089300192382075118002228864989486910819603415915747797928219579 0.01527202140249548773955685622747201441537031114709638650342192989 0.002642806042972791500147348361270029928761037806446952614963356792
            0.07137549731887018558443750078257813195342031154115527913179989700 0.004816333630342450049027836960325540940670067918952262321995753175 0.8472520235057565925615130830891663457972165526192274897031616170 0.0007139292648896286555361233965670459070017582691113115860407260489
            0.4368555029869961947502830310783375749648518334295313467244408442 0.4744899899768428174560061299396254131041043500348366415183175259 0.009760707739329590854396291100155605413489928955278322135695097804 0.002191463494579676949242203308997691519565047828974300728692042013
            0.07379353185559161319581689099621863334338618544552132140275553820 0.5902695315530109319231097269967617382904366069294879149001699930 0.2734027344143507685712507625105343126824230347360758485860345174 0.005001733398529394195190355185885947600657067570252164500147742225
            0.2580516832211307822323485901144848874421648518182734486303346819 0.4729139897357830651271419142392997092997338031634631239463103977 0.2629499319632637282919363778507371482040109747695514000134602937 0.002476004107292321516597326812918106832001865919924710080783442065
            0.04512284050055036773812490785383614442943638949660621286218577579 0.2667592758640409575797187312941880767384360668213850974864787760 0.1615793080356838881357937870604612722034678710009707233836691689 0.003179169015729208178507571669711580427409237119722840341164318661
            0.05463567843325760887748921208222195241195117853724238514625001923 0.005513008229521898014320687483073201199945654961193746360602457857 0.01338858885143079681655431241105552125663431416104136260940874454 0.0002912283584942433266546152668437637251804341857565560612982838462
            0.5548474436195232376954049324477410814357137887348615447429584505 0.1919669784300894307524845479478811449425144882822115839976243955 0.2468181126417058980134449137660007045572912593116849919822837453 0.001888246698949667963919469822232053156063501861394648002075369476
            0.2540553672412708085854948630737042072772525436712976822500752235 0.07534930375930766919566043486022988066382012069850830836412850896 0.3995043328500744683216278177489806435878348224559080132737110806 0.007552186231209929394723469886469834005170502820109190764662546646
            0.008732659293293893588862404329504415150477631244458936733039965953 0.2476579492352664436735694978952683255164981314769355085473473069 0.1661801680576712705950825022894590279559033825360899307771866925 0.002141090473604738195037588462779205827770808061610505069850334797
            0.01262514392332734240569693853989750372022977634002476285507629686 0.3123416140582085287235632071936562713013077415296451098635463387 0.06692450110141618147395669462877924197661152852811021609959265329 0.002401799128311987485830274060291492075889766666170080332831584957
            0.007660325332249557290819848019002390481929636529739747502318817880 0.4950916652338424681722058944406828952582495302823402685044942449 0.05409277438404434533950742160210819694219887131047360238077476466 0.001529221351314696115129133011532816911370014758246423871272389383
            0.01244735128040835878325572501229052687937997124096243829361513808 0.4530114613188880906940988489949697949509217430357988740479003954 0.4822768859713687841181791939986474312744470974544157715546647931 0.002079563932370655229031171345014079759314951263251941790161542148
            0.01137027450928421481104425325272604573124458219497851797297547624 0.1037667627424630807919899368651999708348280541526810431819878181 0.01465018131410715907836925138380541534883722021032566114825158245 0.0007227649779490461089811719270067283054725277312146209207121832594
            0.4067692042452855626132478599075094165073795965806362378386276778 0.1812613365746250030752105489817783976662688055209245869293653481 0.4052736813843362593776321081230903646449352236569745527752267536 0.002516455338249036688028448142087079688438299745276412387340129798
            0.07648929890798682527278261194857281634611003569996618315335913808 0.5984796835537891105575799395680245783880784799141683959653427941 0.05867549843567726003401657795756202055841221479760543616559988851 0.005344320153362385779040411122619715642897364453666662255273022664
            0.01112433445955809893960060760313726988280888667085351209624678189 0.01098134880364954336688404596634980127292350197470276259304155779 0.9056540775378761335249728619230556919958575460597307615574146394 0.0004251809428741779335242113468734782948325720109156807915069740103
            0.04336872563577048084505848910050118268899430932045445406493925850 0.5883046043736223946680595616422806943499294925649330737185895762 0.3556730385567371536303782611735037155272840406417618282880073378 0.002010298433674769875327743439800379482064036517111461974201749027
            0.7698753506752975025144361393972999548862545992901979672609218047 0.2093950273501985396878300367724105587136352142491140781745589515 0.001570960039093291475281626473951560259546395948328860372427837344 0.0005415527617765123063559327482765353821477521004684804789831932319
            0.8982220758614531739140263757009209355736040230420002101058297415 0.008619187774470037674999117138521281413967892370876062937968401411 0.08076596113290332738470165099270178141568465130972654486094615412 0.0004139150430582410984670132324383609430164803420114494489277857951
            0.01259608744840295859721837485061710909874306413274424230079227614 0.3140845899430901679358437643078822134155450496108878460398904248 0.4140091276987788580231900683807845558915130681166767212383172376 0.003812014481958078531500080499882916298050578994714317836962612866
            0.2339131002634372441265717693507374142821291646842966909815940675 0.7522072056485084703772301126746321544229387507576003839806669438 0.003117426772700751992606033214575579463710002143724597297029574696 0.0004313020122432883124803566748055032030848648034616852307927026561
            0.2967178127759774201755716138642575749199548308770587280217822972 0.4187969241150985019829108910207259850637320937158859441218390215 0.2288149976852295852568626506683148691230692085764406641935186119 0.008104378744314446064676156557385707315868112155243550486852237092
            0.01600467312788303563204183539364933134409558940846554401202492526 0.01143457451693363275481251999592950418816289652084004878824780915 0.3114525009596651435152629787682088419632576295136374415968192552 0.001164316418334189108153317748975709055426163494779562298113841571
            0.1720331498121377343557906547760369983796929323901028308263710713 0.6885409489511205537788982402785889091296041208414765845375382057 0.06547856947017847599095957785394846738316727277739603534067043491 0.004944498594461365886225641315992906523280095618970203116453748495
            0.05522337843079985197839133940332180015896779267803879609920979822 0.5010346725940898116147860027237239506015772756981820353139669424 0.2900429551027771515147541616896805779316635507949563415680206520 0.005616823608714730121171691761194447163592082432090468650370863347
            0.4789544284300078718153306540498506707872321861766208687073726160 0.1542377395025469852896910171477125925769736976406761318134326994 0.1776157824555033803104862937314395092121695860286510585875320192 0.008427234234465046280847708230702285715226985907504227250292699066
            0.008874100797351389618987487719360103891653565393508582074986844833 0.2803808475372111056269758565537003089271642677470160492411846451 0.2934409001346021414963932514802819722222916134089062966029443154 0.002945682150659510616282915558506914090278672566376799061084830575
            0.4809568873407216427146026985623695398322986259880207861131751039 0.4697593820880605638600365793063837052259155632915403807814404641 0.04685585696079529366104669682364017521590475101593716658554011474 0.0009071967426770912008288465656431928887772365628576228548447804629
            0.1366753800894811105824873483384207295769505277129825362892440677 0.06057892836880502960994305811505197405751696044125603060252771219 0.7409740202864489599873248347459842567584287306187949412556774710 0.003303824272293298888161041416026769845689134721682741836589198584
            0.2544361488724906268684898917793945942320102717677347908170092086 0.1757961847726547074888129792268972404360928432857307580422699854 0.5542511750234477826179760553951553774325493215368090020467356550 0.004060246140298171456558436998410587220175342094692925620892208717
            0.08738344140898258174598099623768200367305972853317230315797535942 0.1614337240454174410911532572297375397679249317831732847989886982 0.2748328581314383123122811618969829566227623624179679853718217369 0.008361975043479351411889259508146224163469847131660376809793980135
            0.01319816307590478409151635515757752647300852469274894439226316477 0.05578637873252201319392540089082083994276333668757191652210215678 0.7441161596858732647483706500120586381576949921518759117356612147 0.001562845135043346029874299595959562577264065105090030311599181182
            0.01744264571961498171848541684967378585451244658926223826151034969 0.008921674571814648053558735738502382927427655909672263072712672293 0.07322566001183758075470267954687475313183728227892999855149017472 0.0005243175432724898305792189660987681990104109753036639900209869770
            0.01589833107643885699892925808918349998591830734745153780951651339 0.01671145735777742012095629271734802284795303701703479077922991199 0.9531116509462822149733476955453273208660734376039551062334239948 0.0003698152932255108029851763466849585623247632277448769988847284249
            0.2825065613537211134279450394041200600820754442447429812260099287 0.4210152651420652204172646463469779441924739296229548218744600438 0.1534016594687249992881294065338318347359830662798009099535145282 0.01035286916058179586010291501793231088212800157648873629795067331
            0.5605119616624778121843202733552746209626048258983642125488765376 0.3198345688175852051566047030652917062856917912898724724789338257 0.1076487224152758401937964106991620239078671451216539413936310293 0.002888337523921141078902169528015692355614988614454054894652222969
            0.05600783718262605245229649647761918708244791405575719435055406453 0.05372354433042926099542959068061910688746131127889421513439878740 0.009235848652934166543865723558028969865213277081511843586683175153 0.0007476643371215642522695901790577909405221497012681847093200004660
            0.02958023221548225236383100138049191024750746357088192594144130203 0.1984735215016000207392114649104454599026170047019704382990351455 0.7700797281851625973177640748279635578768053481756148651775839552 0.0007041526088762627148651127981234192116983752715191994918263440533
            0.1510844744910682418796822926704313150164864636280727832647689661 0.1320298799379829801015996301248829668748841987488772347545799716 0.5394679384031418279628259125980642481137351940899427415562675221 0.008244221575506866687659456674465374154798724058837827352604676978
            0.01374629908063479718285410747633602777995495793441649835428850159 0.7399902413993615181663828115185665635223163673841628217783476913 0.08221505777590382461731572711356035114516560183571123646024966847 0.002429670731275056113978272651164206498192448084969808564747088622
            0.2510295744357366737769381074015193632912208139646159120244294445 0.2024094425093208023186291996074851501840058021130045640419117849 0.4709905527203770572935625782476597219086557574916489524510311710 0.007887063065511253231868027655724173712863563919615447080110734326
            0.3203813403745661270639442730984769807896018987985080707739561659 0.6227979023503119300014054973868539329968831577509663218561184761 0.03941047816289740059572418012341585043751043985614327377343866952 0.002328954860720971665351314683050935856002994819826062944401664129
            0.4297271702985190107145813821293867311059243752840339521909316423 0.07582105994544456094919806717605807896008680224726953327266551840 0.008521854292099569028092569453214655897138588007696799111900548109 0.001965667423197245694908572545185143234366572716799239304309507862
            0.1475093519338830896810638951145499802528119109103815751620613748 0.1534409879843826979406292329273832503419298420833228024355256226 0.6291585565528485804150151743151760150431155936694265473028508453 0.006284744790577945620901093205097638180877053528186927123813473000
            0.07587275362645525097411262242737139927753546434121446306925948070 0.01910394063568735628727587390548232976911945372181330728378092213 0.7312529396628587114054599195223551310190453541005168227944032765 0.002589405532846723692827894721572863865640832432688739341078497005
            0.5194269241321140640777498704405799103050669813109444737899037329 0.1154013487262275897501839049973949016901183656740929641481715467 0.07231402465247698859849012713074625747077620743099102602645792367 0.005848827450654102026122064450038121184463231181237247187249657968
            0.2662646965065005936460140728967781699692160747123098645457899223 0.3720694418092919820032593862294460914741801632484920829455420492 0.01146763799275156459052779180991145678745994805687630199269393148 0.003157951770162865785532211242819766850250919771639643701471619308
            0.1754717057299223348474632333079663396371482639319142460771599003 0.04507304707394356388145273814119934265581246183940293551363525580 0.6324265241583830299690545723012451144542336004560642033900903909 0.004592290205297181429484773870526179814696808332330053492556166027
            0.4132446475763806511901149736624087361192552031716045187010410074 0.5678388336789401556962156709142577538040306436859111036894585039 0.004273614837742556985951333306073237763303563059814046136998858356 0.0006455967653433213254672966793619587652286726534813153938556097727
            0.5854232844958041063353642863087533480402788741510512263754205069 0.3866640197550218908601521515029525769384600088188908030042473499 0.01045944753506821382454504691492722981872751817552030248529069783 0.001162550055779088689126525107835366421867211172848960951302402567
            0.2041882921095283148261817589830207503352984751006998433627506725 0.3464304997473041072533819483761330482606312115208742418908600724 0.3886664696568924610765624669187144479910273559354597982356324414 0.007209526355993858877508873798502136979165476467961909517114515289
            0.05038161568045665503659491916384360743486273645191660344280573985 0.07123354413472109956858610653778196333438723019117401966109273277 0.1380763482332838982729496766371327492550964793538032204940675732 0.003324670479193772023076501489397176248231014871072894876500921899
            0.3691310880443485225732765643893939683933415599978760525400167566 0.1282193894182515655599020744489506375673877615826488204156903782 0.05966228542920320134357071254324050132131278445334051594637163050 0.005771241289568123953776315957125684143210747155718666265387346891
            0.2867176476000828450944079454536085444375120822070309033434122426 0.1732432017422747518564052855281333509269230595094629750201334838 0.2723908200634573473995324757139186368666382642219926837800182763 0.01109466120710480539920400286823322177105696246098755610532803013
            0.4993845667497946776593998852697297863691658346809384671509612720 0.04349411958623259502943751837547705572836596373123848504662782711 0.05010054522706404298142594348694556949238351641064720908716452588 0.004036517310835869116744163106827357971216687623251023186899520913
            0.8614907023096111903960070815317127495036433864627724218122243367 0.07495938037169533278030303075810640823128087843057109001760761767 0.01135727378115391344978884185934430858405171399625100988676757136 0.001005445491591222861329793020613947099507081783087395500883989931
            0.1816641879003950868660684267406458098949638195461843597860225063 0.5568810163907723673247433956846249660351149600955439741012005094 0.07758971149112335726763281289590504834060165148233458300623722147 0.007971592298865192626253024451130780040728629336237904400248906337
            0.01467294742978819460282345954832588420404371506588589665224638492 0.7194592994613709770935532596692544759456815682114437388982096563 0.1852123914471918674769652774072125051210625575698245243504855131 0.002411744987472076416376175306827729046654904449444506595986038144
            0.6113232170477968031073632556805169960152075995425809577297218130 0.005714732752008685284739539753864295646471913737551211058974762799 0.02050786756885550897450666487833950312483496034299807932469154622 0.0008061905120713723621956943454330814274829335405512454280668155774
            0.7715699841931909666041309287330641018821466752044415766521246737 0.1596853266554822333956923164909650825548333462750077565666015069 0.04984644832632517624776668880677691277796841107068323437776438626 0.001885852348392203090480042408185687581918902320613293579788121218
            0.01010514954733680595261804308424360262912947123879395555412952668 0.2439726083105565108327655239322458782203877467385402513012132668 0.01234963920851936034600700226403970708756641571661935380864438171 0.0008349635208065051361437610819222828863988696260565344734239949284
            0.1476973790109980116274500059165173470433572279583642684427510879 0.002578922209919277411837010466717142851389527696384101067210971240 0.5987132544720075405501352829489438335560821895843398160983254564 0.001465820026620068900731761934115836884763332985904205711146744531
            0.1810559666448960658527937936479375097857357828991480251064719451 0.01094718496097329057765202162749411509326885852863593747109123751 0.7435320236404720105204867239209728272173366215479015383269958688 0.001378822501171739997457813445814773462592289960551081000251110872
            0.6435860034947708904623298458788990023611965109780904533256304158 0.01346719831778374591095160180537594911626772029538088492951931543 0.3275205632518979618428443756588667869098740072191410786662163203 0.001257578373766960913561772020287045455005111346542452143781391342
            0.2907021674286945474703265752668861870680383995953047139408960475 0.01305729782442698282442243284610220218496394901830479751651813466 0.5443817666849233231914530500946174847482523780642658706954251450 0.003499523156682044987095047019131491117333155206699209185427491414
            0.05957227585210779712801955496544756622747630577971647591871045816 0.04189915084958408049434019810190821763581884383032061493618159674 0.8311971246425040077691705262687713410489128628527134797492246787 0.001875520632862347031240325140155521574185807007897531987081487655
            0.7351658108486308154731451469054138434743621643204510527550617050 0.1615378425398469643014974707210007126738304579834840467032752200 0.01875961775751781720045140856446462700901886068559696233852408657 0.002839981466623601654705625663066123866309688622823608375860086578
            0.05876691419031626312033625880547871579488580687052386291337778857 0.1664718235967265396803053536148874546527192321968777863185824012 0.01140640828403992443245228528799606952623597620939015124143657801 0.001676176729092151258039242180960775958518650380427876045070404306
            0.07862563721086740256401880569365570259365006252199040789055233644 0.01305173495107617727381551382610188009312516212681700574187072724 0.8939272918626353601415640547331101386869929301689257150112858835 0.0006260197826906827958017156959536783843390412476828057931567377148
            0.2053425885783679738497785841123750187545742951978721668633613635 0.6039555236239804388186404266205492094242273291896811296346732039 0.01462215266410657085752286132957170133505989300163852870323799818 0.002903724805933885122927752787791979311910182120632848173953512976
            0.001639977537915284681684243687761603897559334327774527402912739135 0.3043266924786774830572007239336055634891458249112996502136006914 0.5779480083049168795912673307427499862861421662236450138732322371 0.001203060731035911269099751155264847825348850490815371756530385721
            0.03903566801908693931495672642251574210428394869330828966048795154 0.7865352885017725585863756528084667909930618338673149870645093908 0.01940799918314325951293630916318293115638147313408250123851951730 0.001587216172892478420513264556974985316492524500334356933573670210
            0.4671368528378846974691504840063124316945516393989082800448438210 0.3298288211323092789127197969150254237962133185588576297817227764 0.01296234223653209549256732832181003303065179371612262166765468093 0.002725724196189008751682307696570307337315173327835567321174801049
            0.01573839168654750331962860133180758548565138111476942174003166124 0.8362176023595805984620372968099687461039197959044807847577105607 0.08547574062407714932373751718493117228334623536257989775151228418 0.001370160564393640297825658630009344864348992540307274714500732999
            0.07350816452646714451323084902506033862205185696782245848799801504 0.05868141836161026438458768234684567115385332410640232515871074624 0.5816839292148302172129819600658391062616089954025589607662257441 0.004989299225838532904986470586268933419625347331024835957398282364
            0.2996261187752004127630940515043200828715715355004864934453722050 0.2034758709664014324654336199329106195873685169344299253661623155 0.009484856519511231240764807065415492545726533437912501292747279937 0.002957392231993726789536436298281585593277059139891779621427952609
            0.01489141043174219959510445321031332096689788626228918845155903651 0.5841350621471224233785589035484318488904939176284704495381032646 0.003477593862488966943299285430772653790023775126224932516985374943 0.0006169120837079447128680051588111607416990778461323452357024130635
            0.8516251779979607867815366445850277434013831606885267486891970419 0.06017615984270859180819621365263087990333355190532778986849010192 0.08381664133911846282589952496340751963558855237758094194270528664 0.0007180881096267455603705704496097952705674027872867710085305671676
            0.006780202814486034225638667039798023020780881900178353780250877219 0.5229167338499293893400096492543163705165729534577355976389479734 0.4621619108856923852353141026524929263012510158825548519407137908 0.0004944559334522292919635022389595940487817807060326129446998917636
            0.07262168039229125145793664090722497464329894747601393303364903543 0.01291061351574474495578498023388534282051246082112168952601675961 0.1850597451606676710714298095772392885902318863078958169521755677 0.002075839035741360294223087767034978518421759321134965085068254546
            0.1262244044595567493084014052366600653299007037573488012257231772 0.07236760265575258670836094251828363724800637526158193823514175417 0.3960654383226956978694882443779458012751506937524893115291585666 0.006952497543545402506566282506314697347140871198909994475109015784
            0.5342756305467679708034549735235601721667284176122109797832445867 0.1461650382050491378611815632193621763494311932848852320588657430 0.2783329910167753869464052914464966091477171204522643117149468956 0.005405067845229880022022806238061325916934423194822499334000751520
            0.03767834392920617667213095242320016557914935719891260647859096526 0.7477977619219280539094724455259981023547718394072541664624049670 0.2007049531053209209110809310713288486142539324387231464002514956 0.001645757589109701945091686543495562143577508834823530907506825570
            0.07871381848489629654398468271194889118837233384643981156247036393 0.7020298080372936050436269432871274300496178742300403965418304597 0.07699083331636926923039033881481749481193606842426485049180455677 0.004130276915822178838415137611500205605366550731116423115992111088
            0.009321181973730906556209221057710866006222472211858504840050808650 0.4072191491975451420374050740978267987109976866902741844586746997 0.1505043143131336508445288721182732587226281510824947655806683482 0.002778417573868022554802500162811606162763717724769329387074796070
            0.9024327047582083464083357971784215039219040645335289087792721400 0.007210987705900086515102066335429683388437191615659004851269207697 0.009873149852428524014062380822687853348706053814641583877636209947 0.0003296575581648018423637722845946762108024395100895331724015885081
            0.009747842930567717903836617844997625672620128739138648254819014576 0.4660027174800121563239701870107841490147699304816826070494472853 0.3676487267017438115425233778814214092755561916458596941115798234 0.002662132127189970907243502090226627116329893970043911495692790391
            0.2079994653106960267502820412658222239637992854819640767624547945 0.6025014562704130710232753930382658637650316235195603000502106536 0.1635592268795771283807780944865891308520339086364339727130402653 0.003777965370941721277506274164005707231369698941380414199029387049
            0.08008806967207610344142999922901022015629371001044733684765569320 0.07299716154135285008762140426004047918285614241169656047259264093 0.8360604474049584728416314630838295412270143716813375181904810849 0.001369452123302554574352247648348996673021084366379302827625994888
            0.4400357066664315760152647912771583678231026704587597387346121449 0.01224165279239111054644795653521989641171122423305012345446490928 0.01528351755292808714253976358264319165298202544151802565006994980 0.001274696804303298317498935764851824576693313456490608373559923474
            0.1627110283082953118868084448884678186688644015050280404131748480 0.4122845526890843348887609754421341486014504752735599275884150014 0.2834015471932764791600344647092575435517544764938375527240821164 0.009545955519674171510033907340919908654400193438153047090226064043
            0.01171173960375751896207293149609064408847777259250799905010151506 0.1539754845280258060704036317487157096862733590708486704971389200 0.7148165890188617712486877705773653248400510544231003675999460411 0.002263472317943210773001235945243043263749552872522622698004151044
            0.3371780504375378137933819724504347402275734993696937846612419419 0.01019298440077490435999511943361978516598772950213636989312277384 0.08098475661312639023267756512571426852112670078337930353043016967 0.001926059432669593215917599704797984635650568821024180783028646188
            0.1360634532827981103827012355398341937859725255340814112961579826 0.4686539334293068126156102951913092873418775810735279891155263183 0.1705578011514215365807111495456931819211199746848700473985413866 0.008724093164095422423359825811722243868727649853555739070218096206
            0.01441238182866636106226780813953039186219905185512067207534096304 0.5780077346948183842058728945555842646909671786131166910116700505 0.1177795042144615929206774042849318527922999274577929599391418350 0.003167305850970473113123112103819832063397045598016419261604956528
            0.7191725302349458866652182096392361987588466525331788053425833623 0.1294975433036046608340967776998095867228143914577425080872959357 0.08935220309549381462977660118085314154153938438266356218401251891 0.003488151799504968420074424350660357383710079463017644802920498843
            0.08915946497790551184700935861715637173622374951117105840764925068 0.009379559668201236433517703413795945723620177869406284554898632613 0.3237280518383410420556427806209045856696274556700366885920735790 0.002201086678625037453512853023987220474371861078069374309467086771
            0.2858243186641413057384774641077945065081054582167129679061385627 0.3186568717003583672211300753710290102778941240537820819467442385 0.3833792998543487286885357591044112655357619743739312775437090434 0.003798207430579716612116206942412490238164047956398622306271072857
            0.3293164391171081313327726780179260572973148629404000570624379645 0.4711752364604548587965160157249648789556048041236215816516755624 0.005044105780342005496936338733657788256470669436574474312139693932 0.002119146506334762667858413316394194225916420881997860103916361856
            0.2496151427807797368500851689733606050523931275829104472507268606 0.6407891095599759510308277713446666888153947432612389831793380933 0.1057689813787515230346659425396103826285530446196157185252184071 0.001074094166726095143785214893220574263028635699860493485323821641
            0.8825721264626552829849942612319737458653137579594218372374898124 0.09327208470620770103794480179381392161346483634604712066933273169 0.01597608059628909899728066253671058666363863523587036550630886411 0.0005392508623820391874701217745279933693847716520819639570181559333
            0.2762199941352209751782518317469465428105031577977790217734716621 0.09202664390151026005821686994894240886413050286720643915732147510 0.4965332031369959435898819381781928622868605543480205430347341748 0.006167125823185703527700211474977230156850225549052439306626584219
            0.5361648656618452717492113241890301818666213984506213163869889392 0.03369175569451729123917639829280375041934207510158641095679519075 0.2649029126103457000067898492843218108295842369031371356052452171 0.005068658980240279812904380123315548804615528532098784301086027739
            0.007952113157501166515871803951698319251527468277119176402856794260 0.7728290517630889025559223272057582084742807585888632290703934119 0.009249479368847982969931970857341552213235940687088529569234175557 0.0004755653439907559968880554952958607868658206253298143452595326638
            0.6001225528500763734343528221403652630638502939111771129152369011 0.03526308161594350786662231523485287965360118900582472518590283110 0.001563872699172903046807561796019237981392919860823602492845091656 0.0006818289836931988690423783781124331901391417592161612829182872224
            0.08741397050564231552216618614000282967814980527330119014465888456 0.8871704806677671483622065625748662219856635001001376364942457910 0.01467430150282330686921848119348846873535829404170819895856400459 0.0006506929475592716854504331020045563295684743849641095441579088190
            0.3291619329555125497909847008483168054437225634177303358915734998 0.04985041925048468152344089901710471440935440477651310943240597923 0.05663618978216819254214675837661262158096622314062593576702584088 0.003792227492854478228872460232097552156060767168364789172300702218
            0.002156883753372778039598327116362625262096388004011800964172112312 0.1877816284745916358918148239131318567547106400638031178760604318 0.7769333634552825838268500133126463261013255768778654002520015602 0.0006871994125363188447800669669730479218325554894402166447584219226
            0.08950678651430454904645429678960809502792117816157503841436754449 0.1222477829332258511982202457698451208530378772913699130634224593 0.05801615608457619010347873290972996753497093145583664994675559716 0.003801007555727999458148483461060474603933854644910144513788856211
            0.07671718905243203524540321649836331731051209099800343764913719601 0.2762956063859946381165801101273661983348934015014403815359554244 0.6291554180625582908112517874215684571627068195409646004672377841 0.002451091752397885447131189197425112489648439805153095008482635667
            0.008896274367531152073643650932003190721244435424632881253317252011 0.6027009484415642703205542289422311081414064335679218798234805860 0.2095621953508518769082694364504460617557165380523792577930895604 0.002363133404651103865418725158297447340813472201085797017429825682
            0.5115425057892981744741408871549058052845850865259665253764512735 0.01105440382037711852353415419588815466171857884913440961188856537 0.1885267387363210350551392431230257149879709939767215895763986139 0.003087015466943540839885794325407875653234810764212655226434669994
            0.7105344391369110186050317093173245637852292501490753742247777569 0.08159048143669555922182506743334933666019751391844492556283712664 0.05835869104166012387678054783966927468153684075460445547771798181 0.004741582371309264652271567669980802465710578625759755937462017054
            0.2698594522424118819869931710813944701887610038139455952501984055 0.1047360427536820138253245529090561756031683672903804703963008487 0.01647440228104578163072027882742139070512613639932322474397772204 0.002875168379413933405341421201947014488431275079168226480533943161
            0.7193103735316320096230036244133980177303010064868601083107377829 0.07778172605414907038876474313124838910013192434429835609110451023 0.1785814593611891780686463853442808709329998137160871710478007277 0.003228765255922797723681605393879598657620155532711555647248759655
            0.4848392483149078106384130695949483340592839683816417350905210132 0.002267004853158171343498955721873875848123710860941735953744832875 0.09049886196676068365441436337188631223558904596060236422267647261 0.001269231754543740124945865230218801558368449763947995681119246991
            0.1653686955409382920271820366411783846225559387459200069726540161 0.006843967007020116552977184416160928149855652626375693583419435470 0.7002238190287926876116276392080421462035930523872490547422181249 0.0009552949760276802202668293920922824963188541421701368605883572290
            0.04985145222588263358341928412141224236929966972316233457533030249 0.4990765450450780789673864038346020440652026055267909553190079226 0.03916629193841472113705612395765157187146137401843185275112814627 0.002773451885166569316675831199266114058774055802930603154738603258
            0.6115274053878625234498256856339866210258085368151793463818330149 0.2992009149611054431943421430196556888535689779279261583555571562 0.004594802519005919153973685715609328262008758623593127391354913483 0.001446893478197051558417139591241559255368399567826364383427033168
            0.4277713601849925551551189276175848118892407638797664215583526953 0.3328294418041088960466231266919881963853315542997901613132503391 0.08692889195515351534447621044751698530691538997797320610669555052 0.006905913788976312551713377438522935309850882029622619474446605249
            0.4873592171355714748769063850425097561803454033588839286771384889 0.09395684030224348007944847116963507898283337994284996846221603180 0.2990056236866255809051500661274935493843338477040755333927297717 0.004622922464818269611634359501782013664671494159318362570060514192
            0.4619680844326766422149082931060012324388360559550974087110863855 0.2475035222408303653602217384598461653240542924524034484963041812 0.2015537882604892130533055735857521319623284559044641098989778801 0.008535590979053779003636513804075355771527050596522548245334906508
            0.001962120862808291237717077437961374185788163403115034740434275848 0.05777511483616252228010272303391877407971765002761557863717495658 0.3859585235863219008159891705002515061949110476180390392936833614 0.001101315616431748800920990885945071295187898390603230244701566405
            0.3759926853442798184909828547856492149057704300474436544973640889 0.4772679701189830650305949429224764776832507504606551224114077440 0.1344978987672648154582509089653643209715845920788764054510130016 0.003238862497994167382690326552591513059776334796525898620485663313
            0.3202186502136937298050633454847330002909466645072750482434370961 0.01278131018459104274894934823499447455011196523133152783340331099 0.6587891890533250960548258577854459358122375350761828761945387284 0.0007297866407685377630998172267891153803954561250079126407858989673
            0.1436242060553368107622881364266040167312571558281989730559760912 0.7876158974971915127242729895378327356393016421786077687934482232 0.009284231413564186151778939516859014578807407173057210537273641448 0.001318640796270380185078941119288581514362092734134932768476799603
            0.09115050311371069564713836284108505569209120214042021105664773952 0.4148963919643738536482144451205758803048533089025666302978712915 0.3877010851003097531942760427738993560849015289467553504592874919 0.005290639029559638192280905101316530368439095122235007751473857295
            0.8302711663821783789373898079755419439113781859051120506389785693 0.05128668377984266589517173037286873657692235546231746296391437900 0.07139393213081955497103783234786330743710859084374260164323170977 0.002294225070183168216394233351884052086935341132412201183110847795
            0.01244092277191212238602259166919053747049385787839301946637361021 0.06547192980037536346744858040847466600749284020766788515533563067 0.8430614187978810125982827796822415729468344736564725057830968989 0.001301363750210865664742476770631228408191786923518502237505148220
            0.05876349193214818465593128339734400862818358826273049250426107989 0.1609934648460256476409638196490868416366058195005764603309415778 0.1628569537956465214112692294299069617063210966639574709876734487 0.005129641660478698225613133618941377849394564355585411532551162588
            0.1961751359402574352550480944995761783089723023964612121186776622 0.006337925963617491666901186043363927222117525143601416813175044166 0.08392750458615257979853582706828093810127256061850289401080554073 0.001427130592602782163853725287519361431968492608255051270845234460
            0.1847281031905274173496302252430582756618374598769692551188489801 0.05562296680499988779048646935427526733409240915030290630037211520 0.05692513578591693753693875398539504038334093831065760112490814592 0.003942214099160150395660435293803011758973001338960454252582871267
            0.08501650686051917265454393599117446478020989754032925483798383783 0.01339167964506696083631210882774378896991613374745602106295457187 0.07555069249254678205869965945739006219400657112347059575058026342 0.001561882806556214149686160067378008187131517849293618714890929120
            0.07541574613170015573242489131312665818527395636530101802718186928 0.7179623019889959501536862297323861979197782818436985663899877466 0.1441679091893600995335831679631900228966032894359592859251822754 0.004036156541747763502743428769316306673979838095811010679532457174
            0.2754149657986187819177495805826553845787010866314767736483022556 0.6365847880793885858854615429070647596453860665164102245057183508 0.01370872814603831345282725537510032406942034710755267249487859231 0.002712432571779287948767692922413282517940648966120755876110020951
            0.5322611774815273566997154952546613288487437320110362445895916060 0.1356498359548061219214078348303959657103676070534513982347054702 0.009954635815040293937663728532158563631937347125506514487358388606 0.002298610805461449063150697529161127278744014182847442893671048514
            0.1798225251877369693618465112604053039867848167310769957486542030 0.01521726336847024239557626040482711969487351726834066065977546600 0.7913760896333797070647628482136469909724750370079704300476392354 0.0009402044953813406898285240728545191819999823885633380844327185129
            0.04514812023359301249526837630061240390079323780596584854027820669 0.03115221559717967610893895102639723905854280003231764662699970115 0.4151005160830156402260030709138999103095632100290288780703924648 0.002350518031032876745962024891279466666497614340443150237512804855
            0.2459404748081613544700736186686517606352890584110908606199407433 0.3557894167484739661434909753589860860857955745200412236652609982 0.1250698655318216493512016064722471238793731030831364645738788863 0.008675125716204987145978796988925031516116155591344969487404878108
            0.04395969737905017071510616222765724577000173688237794006572873685 0.8817548186023698473224484031284031805460385589113869677497213141 0.002287633944252364652949978653950122477008963673168528969580602402 0.0004631182074298617245611250868211879114594417007623921154365061296
            0.2757297017656464087285465717036291113866946665726546708106142725 0.01043586798621964609440373040772970371494195760772487079649633594 0.01853787542347792334267038129464941887896326573783744767604744479 0.0009683815531229357637812383766094051216183514586689085536370603037
            0.004622555388209751807214469701016909703774454818331931215750129441 0.8887734572196877248813703150465336050849733522362611135272352637 0.02139311758582015864258285337064174880212418814588493005516859379 0.0005048288524241754422818218073984426914795571753054502309300802397
            0.004803089824142454063957907483587194624807696654604718029593577890 0.06673954753023248207228914087409661950961241463810692713890214441 0.2094551463602713622631887714444728467271682360017777658121825963 0.001061275086373763300215195836724279011263972629931756441906354783
            0.4573744811235079864541662815104748985935918469686312598858750488 0.03730673889094454383718625983193131783741134585254782974171297808 0.4250636153268251769797998956645363602954685761151096690215700197 0.002817844306290422879670156885623989425988724680082910044499242485
            0.2014237069996699066708154486177352825878320770693912841896344516 0.3049024847086658908540427446266254893115219365067188706875523165 0.2475618641577314196987900820789358723672468464934852058338070543 0.01031450765160973579772745292433862157854215247781883795040275161
            0.6322225577055995539936974435600029064877888897053423618201305624 0.07722396260624430433626532932495233328275331641670986825592451187 0.2869841854178058021569200727512102393380534200019391281765061516 0.001039642153842381589631395504910517686217119935569354314030081526
            0.01687844292379849471370046569873307635460578003087285943310065020 0.1673297098664068451740730454123613954183434680260792529562140481 0.06574742308446966817954539759082330675756926538356560188079089919 0.002344654490518320351725921771340347340970822805974970415663665595
            0.01435831172687220721754958755312118907308356308917287958295862170 0.08360627549183930482832125439628140091888750892983606733416927834 0.8872253179367634964920198940688676782311692287990664301454006594 0.0007674230245128146264281963500103038008121738195572402483413142702
            0.009150281621700694139699999130135475533891299617478174334005303601 0.3372343919331929809472799859437596737263385097817349218660929797 0.6389349881262826615818477721707921664032468374129723286137578528 0.0008983813495105961750918127131035351962338930359137781757916322094
            0.06202189587602581387737287526626376686590241142303933836454391030 0.05525388575806533509359522920494079850323200508099548698882458076 0.04602738488990848247648939265187957903839142989888230228763034792 0.001570688387531082799836545363036936786967480374969028301406482467
            0.009018782020859701553691266608372998206561236358653289253581792657 0.02091532860671662207007489336693724854803870371596752816701552189 0.01530003780991745020966179597877179314821815861725101036112710820 0.0002989353500802511996431259632945881787829695260095005754920179584
            0.2637087583633548649193230809754025519567370751865953523043056899 0.2613407828449512620341530502991229043364851213837967976569423461 0.05529338293696685672621252897888603217283984729758805328001384974 0.007822692173668848935298008541942252021651250283767110717842423042
            0.1641757733334821174660950597480064777517512784312431123943816152 0.5153459580055597796641291724053054564410910015093061707772446849 0.01543325887193496580116845533581741832555381537118691117309911616 0.003891091824487213968311299336924387266184064618845327508776527315
            0.06038286452584390373135628287674936834757310523895101521432818221 0.6026745945466393272126469330392276679840419077125550126744663278 0.1666210306425383542816000836000660829163032752719177490607813585 0.006153046066843930387912379379535816657154512257341364183262955205
            0.2050399080440975322130879053166850864142416701789754123974833427 0.07295242106981744137593439016912640495767454870430812234971408090 0.7095729989657400492153736291235836873016837490464424060670721755 0.002094811305802378533721003673156682472407694068169537262293778926
            0.7363485787761154392685430431135514269159190284989696485835285384 0.08815948277681083470104520459690175794390053045444024560080741684 0.002576738135949552468446376498464496042313153095293539739969697593 0.001042157027689545689537959667133518210650040845612912242228449679
            0.4820601769681943326190894376788292206026355730692628606658925874 0.01436938350484343345038319534543165127540679587389038880542208645 0.4883687799666249167615845799610493438258274650772189126037425307 0.001416955507756163121296936031835503659938356047355270525775677444
            0.01392028126925977603037553760972595037060990056799085729939465846 0.005918985239768515508739930028845026018924610858763677010299756173 0.4926300744376323370455585447874744161720521286980674703586151198 0.0007653588119422877204367332958934571517863828853484414174851644320
            0.7668197010445239358230524479807154526719403260803628345082611782 0.01244534519091673246234151382003078125075310445878912022819819843 0.1741255831061748823692352096402578347238797403507730666924455862 0.001697319630304319660256633587986693182060275989557040177937382562
            0.02921797612851480638752847947820196923264857664706614714393343814 0.2788888736888091587286287764058770754385972740981968433387248897 0.6329799682314416603563962178569988039750138373837657502203793646 0.003229071386514664818338544525433292532874933116140363895376431061
            0.4014664650709986812064079157825328008880633190695061958938716784 0.07602203998186815705656773070130226115743358748967635787992891108 0.2588043747915636603089704225837113953342002168042676728621127354 0.008971874946501769645834479379212209204411196558998279571479054161
            0.01430018629898489160225027514664207582673903351638720765988112833 0.6547184652673813002063171500047047231293363353270779120429932336 0.03519371387707744123742373199124637925306777239933249203729118716 0.001712557617264919258748319435366811902575323507497027661017802785
            0.1222870227042022933661191642397154236331693554258652396328911873 0.7458442005893575254348481363951386187136178787438564975903962285 0.1158409898404799649352186034706888289544034128385342535171643441 0.002011973871730121864275926869037676836283048715811094580047676883
            0.1789965181489697015460905637950168188251202140181338886920757111 0.01094132735074663152854451344392231151459274940902036113351337714 0.1928131568999359015118998144824518940555766204375186212889302985 0.002470443482158834757097178322077526942612335816859862177443194804
            0.1182737817994424782472523393654438528454031762575983767786023113 0.6341777489905091807568058598750114780553430104594285019865775704 0.2383994136078240245981698015189403113783578897511400344834420774 0.002222348829640999898692442529622851028435482881389995083629329588
            0.2501746382373079347453499597660686967554229285779063145215477719 0.06687762745398370984106915396967981800844454435046312168311395815 0.2836340958464020959416875216574350586334142922864903334887485865 0.007111707685318050373663030015788122175153088923844903690077672159
            0.1509277309659514609593485537247625704752485072913380317335089884 0.01470935707029101032545461215959407644222434111557919808626917851 0.4415040284369679236857463950289649945317476480174431035084457615 0.002336371107769955223865636074586780860137614377179289117394612118
            0.7867807272347784621511482473552283950403489806204117594299748845 0.01746222828905527524014949136000169965074267751918413803537661327 0.1915562248300107256753685030701245882533471650833034601513296143 0.0005429295842682039074623280223238586565971694458936080285537282194
            0.1309869744203093684299285551820942251561495049829234818175854120 0.2461085286888168349666932986264503150912232226774540085848759108 0.3015225366245411994428087504844607216181155517362764370914054490 0.007972743858233752758477111890118333671630486098747477808752766734
            0.007083020138221714551649714286774832971913373548357814365967518152 0.1565061716241770838726331566287542582471294083871621299412491506 0.4576939883219989772293685650752434937260981132544197453198641293 0.002507228287410719169966454171776083464514359219609429173638039047
            0.6855508795232490988818594322927898854692581804874461645733293734 0.2711071083405543660381465034942707042440345526400271090927543311 0.03490920774360313957255456941739960103412754970084511916515599414 0.001128294484437958126995288625528960013084578356457211765537019847
            0.01507282248128277012131353026239648520308543899492844888806582127 0.5877253188808434041572478322000246649125023075275112928021833741 0.3258588474071586767585336768778606785975794919459042510636540490 0.002574309857214145152035279506294831423359700220874731655539363858
            0.8635223660290971363537264746535727528939235756291606466105600125 0.006658044436029420470794791322011991016385170830287976032748611438 0.06499619747773962234222939147662930619516391764703853180895146735 0.0006408705525378849570620241205742251138176133077315265387781266342
            0.3148367364821974085794744384609473076120982390646471799835263915 0.01578787405038777925635151624748456126018631044291882291353631272 0.6156167514494411697962576296870600192292563927586010371181118837 0.002406277109423532905330202330759458133243786378542335273948615010
            0.06332952347699625688146164386214175892520500039885419276939098162 0.8343309262728189993119967162029317362239809145999589046515750357 0.04500257176845129330673102588086395357453745969325099647314547944 0.002302363592551740973517229189706823316547480949522816024183800435
            0.4879299534774844924975623431772086829715502801949152904209630896 0.3475651672983891962819188527862210250080361889148575519727875892 0.03306182658107846207161548556088727428368226228914193560908438505 0.003162746070025897723085941721340169689662560465576091053233209017
            0.4589954744339691510460277534645098304049527763739102112618817807 0.003087120738666054607539294907347306274681395022492405373377710235 0.4568156846915753360642557738195817299422813511157428311996190419 0.001162924889592519608971647475875342528980273185298564710168096188
            0.06792455250774202636335990907764293076578584451535067524181994228 0.1524162333599477242532247838546265266681145873717454341063938131 0.5044613752867710886685872524042196161410147759858582007687821916 0.006915205797360864825939403767306383266919283146303904925681702615
            0.05282464673643002349573859922305183512041403999970209904687881861 0.3058552102216762929156013868938251088281968127175387251374204793 0.2217098774828075139949354608384986542630217999787875505765543747 0.006133342234506380217305326042599036827794825783673509702021751412
            0.4175144507099608927690372392813920021774369221594135514525808233 0.04925468944527230146662563910766408248547815649259926978753272240 0.4630121602306485602964020268288783042068025190195747508119338114 0.002338880806943595132770730559519281676391494365567679394315023057
            0.7628536049283793188007470897791401423312973843508361763025652456 0.01341037289885730038354545178160759059979246367061064127409053009 0.01459080526717367218342752853451339570780849912424912717181123087 0.001101148492445343475862226272727504354691644903532530085975375572
            0.5977917272563969216188973455083052252890448435530491735803558561 0.1946786844043173095773966671866761880256483848432377241936538209 0.01677990058445948324779483263174827041375434483763926958635443195 0.003973609229789495226028077281520677704672455844673405701651165134
            0.6111398518958130561628463845706928716859785080268548640748215084 0.2837686663603776539583407988547385015137420146872067439511554238 0.04737952717153630320214654075564697960082736024329825104541924853 0.004127364040716271320501510055709030592923514334464781579773960296
            0.6977635471845069689929503912888181264641145441690591676543418006 0.03994276887455083982386868360624813547049669864612896922385790743 0.1515750388568888184973530607005636520479180932439325080717417494 0.003851505921501638560325345598037439290052041921521873555861941559
            0.01673124979920229752162849595419116389832438083914680866564580692 0.9541438374143897174919059559769230520266995530688194387004366833 0.01247500846816063564736118975656463325489645613709498326373933022 0.0003301867118158493683381668681020912277519499720625110421751129377
            0.06726004715540564491380740890427881753418595406845906811705383056 0.4322953801161753081658234170035098653597223866009620782445858115 0.4544799818144664434348933188750232578091558050159119758026616353 0.004198913326870556615046820592905377018897996544948188505865086091
            0.08338032399811140933633755726091244954692390554023872391883190768 0.4610494533218152868013824653162003505485401550458577841193472446 0.01037521412796355898053867829466275560576407917233197459247550427 0.002250754128249657787295734227087510645840581368351650525400750365
            0.06834563243300690596662878797546765781851925179992227885362242524 0.09160387333572998847650309285855283838034511426204005952518056507 0.6944641345310379023143582403922884679208351256363098383876219691 0.005180297410927923454136937392491874264030740299936706665307291121
            0.8333205367538054756198893349880020366964786264115286441052604036 0.03788509112716545511021544522337761259936337203791991859311154698 0.01916800353511108923799484309789506821476656478707335664173867149 0.001146198111687503832057486444447773870270401870399947695185899177
            0.01419679951535520215858250155466177655689769342557463290277988932 0.01489763325522053222459182220593279262993871968585484095978408645 0.1704387056820209441545274187701627594417406246087445475028699536 0.0009564914912127872537910796299531774418793113495834761454524144633
            0.6397833127668778499569688342087236934012976999016659963562327142 0.06547624032134967657787775383347355188872658885399384141675479291 0.02569474608060627140187541737566011292801422300335141390653114075 0.002987676310574477605178351830707988129523079718774856481062196208
            0.1831385590189008846238722212024646440844924104054282690628543506 0.1703881928194572116945725918303734999889717805112044311544528038 0.05415341943044676145585324079337352698886392541452017386420456380 0.005441465610912922786322301196042596470166030834085073805740324861
            0.05776450949751522698179080762792103673281420544569207238881110967 0.8504612603318364297059382276913642166192957831839314166125153788 0.08351062382035915050378709634407629194620242228979882764705514932 0.0009085640433452795843277089564516221610404284373025141886350569784
            0.1638047831326753861312791208039326530221575511095357218402583020 0.4953982819089869372446260756759101186165950340839981775960094756 0.2960293056806931497140792744940537297630208363929413583773440757 0.004972728056697298610383093345092946983624959786537356249401847192
            0.4082549458789515782993042554203981472618244933260401932850873163 0.2689569054234117457631798671572175603379704793854288318428404930 0.01091444055354453788721078728934343460651454809955906550230873017 0.003164117317936241906675384296022511969667364751366623748393940411
            0.003592680258378689878178047583055066154767315762020287763348356609 0.6916978974650121143538917785831841404473031599397936872466700597 0.2881808985641247574990572501859988159115323301321844075685548533 0.0005926880656274868854649787018876025693772186667770621675169067450
            0.01510643841434435308248140480257446059805632673760376876061004990 0.4405822909090163686831869148182378847037244421806000399279721286 0.2545299037565788710508428231026012350384024664879330889536800249 0.004348135064668344896694195973357144245181718791300212280464886675
            0.5726575698272523855436408882057237080599934661824729752156543756 0.1685189017052629286658963163277990094302036655520359084491755401 0.07551804266381479348782683570391024212182001607885305791908184043 0.004388510661394446680806544612039805742141779591486167423012647775
            0.05522007687199834175032481721813066268898602543396502880429025985 0.3829644638598899593230553540704722415269075290516595101412680616 0.08650342622836223127189391017517962750501432015604638205639744849 0.005231944387393155242797033418709515892744656128780570421425869329
            0.2670661652898634019874767557683466422525677296700092586005138329 0.2276526635401719940348738186214985070037538758865828515316777209 0.1407560045901440373081811272229424552407674652305007404300556157 0.009056287357069843206964386502104653302700699761471341594246956315
            0.04450192015057431481544214520817334052879125776840098090751292154 0.01127906878899064542035293423176618392670095211953517617859245468 0.6387028085335669740826052288738631471318069706277808157480952117 0.001596883277403917031243541014169845933317843363580160414244936207
            0.1730572260803694973579627209714146356681417805220120977142109086 0.3359870557154901200270522051170213766865233099828994414029411134 0.008717372424814726575324811581848383462372575011760338082547324151 0.002333017860394086294209554744602954045482486461585545222108522302
            0.3549365334884683718190414159199548172872066117060682768187469010 0.07220077906759270067796150985226286455168687000250467964674109179 0.5614831447836078378671666117868784391972501182020906447520509601 0.002344810608009623368077642369183174735459634671523744314895191868
            0.009784582749263887996720691826601335198779396397787855068033690872 0.1405551878876217682406650313543802486302754241452144659012573292 0.1530603352384587242900216636812478211788125945010130171255841309 0.001561590109746253261414596243827260319973216172654316356428038490
            0.1524725920545808865073538782327552825401299839284696180390551200 0.4288342380043439817387391027737004748956139416429876041998353825 0.07252963193041211991943870171364521090176064235947289861596728830 0.007810190359330929991767074441555268655033055255189228550385117000
            0.1470749667009834862199964561266930137056470392927129286246841263 0.08613275108865183703865150853570611555738522707132119109209898490 0.008704392624971726391104141248483872574414794518524165658132258643 0.001617556367591868923392876984861666961071902052991925522732549501
            0.1168088992405381747460180920953407408998832976643122153632225104 0.7192093160545461605527268286411716122827972988996137323916653989 0.01520749457202130207805087315470913353433566141655975365846532922 0.002328590018870545471794201373398615776452425778148314831932791984
            0.04708188621462194105478826103580620873297203380899922224496165177 0.2243184775638901798959065177637229862140512129067694768223829737 0.3723848902190363849282828003047987033211402790073324996264460948 0.006323821251390441682308344689100160076332403677899395409913473032
            0.07226303816147677644053278923356846301869148937221726782379961980 0.3817916901205109163551005523123687682807924769811363930356667305 0.2854477501648887363852626907327529842471415013990211408107000645 0.008155776858139940797091708748275659017828415849036279004516804945
            0.07612399333156936001766948561406331888787655446765678825226548886 0.2923696220281781250081121242852303866636463802984630124736605451 0.4228422441435631662792834038711097373406770735521155501378507133 0.007106947914972356870518484025944760383890498084713057389299726058
            0.01621671704488663181255797267983969609538389722647269033929073248 0.01105931588659199701410587143792341782579555177396829003807201278 0.8024156668539981879673038477530905155428066136137627419451253857 0.0008092810517762090916533905480540905789058376958650324928593756034
            0.4402509112809005910798909068335365736226009096325147155913260446 0.3666156080724581552159841878608060130610621928716134931168117936 0.1278711378241887088870592466855806154510833950249813173390192185 0.006523171491456557018673880560916673221910073383036886962798930496
            0.07128626612038114466905224902989946039634029515783884023338820834 0.2356946145614150324870585998821823945433781511863943376538926720 0.07029389687097793292195138579132541419194954306122214475813734445 0.004647795741681687830355875490936494527581904845734397544480225802
            0.6084505318379648293083623154615978498975802960567846765378802335 0.007803451688882433097859750254163672142295388019627846759619259613 0.3030222609612095296072432625235548782422719064217435940221344871 0.001650567490343984755450186551321327678439925670793378196650701101
            0.2570902273122936489623054972621777896165341565240798511917717948 0.1230180379244389789339964046616628123090371207111231687620980913 0.1037697883987309213227602164742748592546451350022462818465429602 0.006269158247542694861674618693205635889951448447201098199495839828
            0.5689632399376703726569935009781579231287032571323628400222989866 0.06255777930765594482212522586177467983077945519690873242306259956 0.1405172533847782555190508004239130827285175695751127753362236437 0.006526818222160471869842828596797051936646537127500328427388666704
            0.01352667265598112944966798811462363612905559452947229898984888804 0.05056997879903069002933767535959380701266174005044918276537929810 0.5472008871023655742886758389950845494095244706538507294596855260 0.002048338526845650311568614539756568605565194216572471974964206579
            0.1475642583172108779937823165160896714265730095252435959425411894 0.01601016086656806901574304913118650881543647266870012028252155526 0.01490727174385623129520745340490484051033790483028127245920197694 0.001066389772436326725078960267193709338064454341978035585106863503
            0.4164030467867452073107082762404802414041562586203000157229943020 0.2394038927660782819578566455324813483328697579078299487019009962 0.06737660451870082773116969709482226881009504848370642081938691341 0.008805629911819510303796874154044877823446777439886843771046446682
            0.1274417667839028829144881899724045046603220186374605352109737667 0.1643607398995022723048285688560296736340703364182947368360543812 0.6968821150203752804424433612371014939439844336618994056018657300 0.002247512259254915047432801703946387761948967900067750201498039606
            0.01429917707947883479118180879028364455156258858640058367528605969 0.2041189641313306173145424320115420735086675999322666987593720699 0.5619891331549570487278905531441914926295839315404729778379060173 0.003251435801264693195373404316651592664369917100826995211660666897
            0.006679331971214992867865190872908389593523779663928811803607017380 0.8394527274149806348546175395467240522077477948658638689403951907 0.1388567292824782125449376104662513699970164858115062826406444313 0.0004768928950346184658066456954342663249513056461429038413356061652
            0.01185367392146374281430078244762746487536790533430282675147181746 0.06094361737252485597236452343215778052420699636867680693827318981 0.07768824745418043495679215172810927653614970333949551569092580747 0.001274073146459314393251941130181487887290467593032984367450970482
            0.1313818756661717828065686727205482151821757130261654992166852960 0.04794236625348342367185430098368851142249814994206056259148752266 0.1328505505354489870731419764451039608268542028265009799317739045 0.003012142307869945768536990749835846039925575415877500448008051959
            0.2073189362575542103949196094607509365410433329957338317340223344 0.01365756788083997590796577980743330854299619132836885412848288549 0.3173939676536301409113585489829598409269092906340656080638712130 0.003576672638316933326078069400402004511378952828549552267660036918
            0.04469297405713084056411230940884766932586556813690349515739861451 0.09632370777395019365987397299738572315240960313304061527159359358 0.4219713356970626613478591256824885133579230268786593698243846667 0.005011658629272546228653279097810580489913010487231539222471919911
            0.9503940454732770984017391617171099431551755160283321863490318797 0.01828786703935130695648034542327910079071450037579727871602447977 0.01576856059267601030298033478718149256329459510520389412359835546 0.0004426916154762254359308465335016354140827241635050862047228363194
            0.2986040463382630255079451934387620562692596427237934255131359990 0.4006926839029969621518159163182789567170827310747850787003363834 0.04861336540704691310986949393521310560164850671264541292790678777 0.007225022186462908469355190565069901366311420894599205346871458092
            0.1961847053565622453444517493059564501156228446300812225449343225 0.1560660737934920128997366784703223158511969276861432188130528718 0.2266492472392956100029416841910367505829970518510460259948360464 0.009658021301488447210941646536411260357351170532257886825605823105
            0.3878657454092717210958267139336572208289068764333398215721648698 0.1261981636820251612045199648953860941183103573791191928644605881 0.4323620789006778020236198186711931931441291873426163976966303824 0.005696614031948223317079459656171083706687617832130422372084911738
            0.3677766697238509807520400762535158688326573442507099417789549382 0.04738763274309287149957417746028059654943520118859833177107986970 0.4083987134722951544489460743202242339052037298205415306328413912 0.005843293691034590047634491852464907948131890877330315189673044437
            0.6002744769519031210483145139843031076623727201176521052200987517 0.05127133702697078138611159181937802177907340376570873015614429408 0.2910739448108588055853089680041526559366060597023394585121984155 0.003885324034722815434167706340361670925410978377207220863738904769
            0.3249648782566412319772515637666632622999057339166845457100355671 0.4960624732983755416002943023769245966385045809256092439186602598 0.04947483780277595892149169978867075472030677431759878506605862837 0.005711400143948423072035923999598814317088400431277451169954292556
            0.3975298100291997239805725124979584344618744082253861931064569321 0.1468788190962069435978033465011497614131278429625421695119971316 0.3196394334436629650466450459378100767295695923403415474809814820 0.008174059489384490303825993168869342875076580519535465955626173300
            0.3791283908122563712641002395666847246227464463190624458618143304 0.1336833387142208821122479743025619266762585164456164986809564667 0.1491193167945977805263249233848890992572772030607039129660787257 0.008874012018956441609627975933151778880804457000060126616193913074
            0.3632040260406164933514031198511728735465362376589272036290373575 0.01702164190963188815998992617251509029389148826931999113560073798 0.2860938857207356349137725951730749998308105101885432899878108523 0.004315508256098161554990743779973487586275880957722673481713131356
            0.1604453900224612772295167673939445033932634874018374593193420442 0.5672093847770975053552684396129972600853956032775627247494260425 0.1742966183649890649935650571713364956905980551706533652356223115 0.007951342078076757171974124290643177412599442241590188747903480805
            0.6078489561052803830142250549589207231552156646665346840357139825 0.2089573729907412628269313727671139437162036790653153596346593087 0.1455367193880491877866342095842512622735422733478677877359923760 0.004560799182739955997211803706358770069627388032126237667846162919
            0.06156379365337188761870431914286734938672403209849100958380193562 0.2099728317727427355718241255748394253339050454300574026481111860 0.5895800386243434701203417475017334048044573194242411477602101957 0.005993033060196541020822097123932124525173307415391489764664289596
            0.6539246183124882857462309896921989289836458508109331352222034104 0.003151296200617055096026034101468899749633324971632128658699030420 0.1825151376570535183648171987358522900479888653040163641397819036 0.001376655636243031108201236929247818287770845770623669620187568069
            0.3668339160564134024418406392296779100328825980850913385121354846 0.2566315899131430284977459382724188080223751905962327455452540895 0.1702737720995846005933246489739571275638919181574331727451801902 0.01010299034856923699612277871243281974809730508069780068716159711
            0.4248268663935461312432765235048109363759836763744744993467867208 0.1599609621897638935491549284736691447790259590018751450995070464 0.02136610121869543819719804283744819605062224551323449377404398200 0.002334476046520829340519773368979503896976560222362600565718595744
            0.4323381641681334891427314797430529035562031662955638061519342888 0.3211990643522900466757875450906561959108366185381449181461356707 0.2323970651260352021402911727495293832875135549407532386691848395 0.004342544268609077350207765421661499044867045056248901565982590617
            0.08060909862960744954895955562202023663958840915455525926251442969 0.01280450911511163049423169923411615491836313994203306458921532265 0.4960299069924930215465837131502710619290680141135153475018994888 0.002075275258461139626383596452431900886740834982834032243640289069
            0.01532162541275142687672205823260777995738594065057307902533460251 0.09793117437130607768872836166400569884715006639393240833031256971 0.6264993700884918788797630543418244112418195235631097591221777823 0.002559527931392636149027364450133445641135971375997423929986247475
            0.4414243908268058047033285074796653957236128151304965080083162591 0.006081176083933556757960333917827761938560640698406816292758240788 0.3582590899072870991921717392626444074320958776057599855470370617 0.002316611935234274749224123827172064875865349986513397600284800128
            0.7686320561524009095382420260191637758240075582896511824894889286 0.01261906331569899213543198325063831143154324840844199133744682909 0.07525112458494244522242937467818308627327816476104737461261839041 0.001688427205264581329993150986757428438945463139466200017317154176
            0.2900059646272469393164517155191191619566640565284835761236879462 0.03964343653006997544072149656894266560447236561743657669164076746 0.004913353935034458042047405862644098016189039081822899495602142389 0.0007545863785517249563963012445601383655122509235863013143347970437
            0.1303530533454182268218554960717764253578043804682774965690262466 0.05996685218353370915171939965848688881179877807166051507087033386 0.2600824467190271360303978197237415632027694763124107359894382986 0.006317466131307462572058486441891382155589023401131227886622065602
            0.6084041360039834273283010958096626183023791604101126975587375945 0.1132627303160991588009794581338960951438181724329663503984229387 0.1751609369549600107992321398314896532707561444307551297157581043 0.006161450321526425097611808139552429139761027510839072744566280211
            0.2699294603740972331845516848066462354797382707077754310036876006 0.1657154367204928420315005244390964239414899115245263643131693511 0.3886419645947194496346803972625618339786057256055349450135507466 0.007239670647919535353472951789434122748393938529830444768442156437
            0.04382600614486732612952211262490011129442681228635532970725591059 0.06433314589488021841805983394040543574549091101353823270394330251 0.2705358751774563927405439070212883431386438147807825883554542308 0.004230381079332578719640772953315805662989705274157620380615982123
            0.2856416057686023393094863342369760957996264176439185478756475393 0.07590027381480896093019288684526981390916972443135657234822384068 0.1880741984106961366338919477532044493994299027186497451171572501 0.004419635018646849202102878925833748503658283472574325001700978627
            0.04947036404047568843235773517176635124292310424619869068054031503 0.4083246572164080961225151655749344137067519423218746605788098327 0.5362607134923438382917274908945957440599749437639460676079042472 0.001247489386771595396643197744215312107167141608259611601076155698
            0.6388699534584460979603354388255353368870979622945634332055849020 0.01702753050207337852160362036929812769969576652413283051450750556 0.07951478282247317178357597083193242584105072094896270817795267244 0.003052035378626341925575621117606543806675460350284626387096765756
            0.1533961442198183437817237484463643405086086507603144192376109104 0.2055921428837944135961847867456327853020416454535021405236422956 0.01064093725166802849347681729167847752193298238459224720448194316 0.002511373826151209297750778101657684383771066449594710658764490714
            0.2970498370539195537985925133084020305557520723869446873499914035 0.2874781194505749691627896088626859138117124659561642799953698270 0.2856825535345462139771196514678265629935455419256001679419059865 0.01076473748228728380774985532632020263881641761777719960400087732
            0.5769119497990762016820707390076086436030710150666758445383648196 0.2172549037748002744160603983955749135240370557283185629438830504 0.08787611978699649628313752792550732989578447497316839620164373485 0.005476544253146303772555025569222338594769610751774517209804373925
            0.1641248518702123824155524976136817152122340140064344989889211737 0.2550037652465302707366505367407146713463752009560778299094182069 0.4140412729888447399102089712172726561583975449934894267213451228 0.01016850970115560536895771815500002229013774154849641459993455299
            0.1322458678216596861270236631794436484379339369407001111886210782 0.3220853653010875115550342302712114908585699928744040546509576561 0.1704344544997420317198528640100411736737072698222547792995095975 0.008311919268431013907124857860529506899879037529359274220606231796
            0.4230783846148094691726441464904249187853243424652388426973532347 0.04422126006623382596582104451475028742801791709502547869219379136 0.1378400430633978051333230554666570118493107276138112983049719641 0.006305047376796156416577598691189126687571142943741583463136837823
            0.05125115193767894040519680305893796296535015560267576282735445075 0.1474201257768999902689894364179321761262210233531342954350114569 0.7538205592800438976628043846789821715403659332843066328300082538 0.003528670597824708883176112743688330255043262570798016252414996546
            0.03210561145704624964474667949327378731808824174781614492794921175 0.3577067416986585373084661919096186690147064552897603027519330380 0.4773700328223217763966965576879899395467589918460321272954615563 0.004355322649519814518946115607430444973061587880998651918338266457
            0.005458684509067707979691991111013912416306880147072615632111399231 0.01388015079087898904868195678220188570450479304560259997074555321 0.6722954044071273233295337774022312455007434697499790875870609664 0.0005709627074365219428890173637125996126101970425153244285417258014
            0.4587263615875107831275860741861831093811115933685834173902626590 0.4476495678674431227637804561024971477821543988955398150316063078 0.05002707974245227496071108294766877263562497303248891965855300615 0.003840677573964840819398524767059791008199098643977850372716117153
            0.1439972031384626651029848378624983384336990146120810928794772399 0.1140884215347720544703945519266213242918758940650956511916084932 0.1474443151278338547739120941228518594179356741824804055347100211 0.006089541890980723081921667768422432802992030043909007776949849076
            0.1563057357699276745971736681308377868818759795787966268217451582 0.1498100214475116569260206806327551762511420747162889055213074802 0.3907685567055423996439577931954390554816180837969286000179627327 0.009018122860162392266555855286706207553949975095471482881906490975
            0.1408237728218376139251616831000727250881842551287462315375176603 0.2365393433763527259137003880008444272043832073212361955825031067 0.1298351960144175055913813951073192872192055758081046680645015157 0.007776058996742692067502774265176467427214637342213393341580241657
            0.2674000909847241571094223164173128774941491751148360664783177774 0.01209773082597899706575882942676223503069013982372852200642248716 0.4303543396043457111832676154800096920368064720770788644479843941 0.003527377908890686969286604357884344352826091588791690492867937343
            0.01268386527432667114646477697353408947730793056985358996546980113 0.9188185217395794880737918064183003169587678128181980488978776630 0.05614617218342261790558140470252618294837253378947219771520906945 0.0003636782201202358191926562144092061611280904315130158822277016597
            0.06206088804759205259607257253141055695546890521151465327709230121 0.4709976980351967428132252341403024754240913421596277058724596763 0.1434762936643139839681138613204234161099585281669300975452631008 0.006665444447481129810444075311572136590372566699500356996798248855
            0.7062476043955224727422729828093657847112466374706777087831692053 0.1614452889942290017330418478935724858168215479793819678488286434 0.1290935055945801368546620535871293049711673172885896929416972936 0.001159355607708698052500468383827933184492481222906833617685074683
            ];
    case 18 % 338 points
        XYZW=[0.1683274265762306420355126045451713147953983858879155762902786588 0.7171467188816288958705880541784698178165419739862565994530092341 0.1024225290800436125864402800414706673382271212048145221526921061 0.002243538639278397876721850776445444320917583325668974232643046039
            0.05266038509331212645661479847035036380208334999152526585014452828 0.05428763316419117151683958645372235026865855710923318502541249857 0.5288789328731775218327252547953697799672481091587198972667118427 0.003152762520087737125852201279519018297943488118750170854184106703
            0.2680179886119384594852820595191881769203177068824947124645151806 0.03315362849402530892768019844555446794633222560573181806364736508 0.4271718543995294841476031049588094335564687798573333346937679919 0.005494201543012681799573003048336193705184435237851384747613722905
            0.01165202882194916982019457716901483654581558533115358038589028817 0.5778505741222745538934849073555612876239001579982552259391448143 0.3976929014766024834314743157270651303043651579972139911046024087 0.0009031935473721440165436983371279775383142962059474090154356298970
            0.1426724237368054684467210369666233973773258217799769821751816642 0.8364718061168705016857845309443186061513750671315854241336731556 0.01943720803609089192546569661515975443496230277075040843101403712 0.0003235038618898432252152027021643260574906570055036149892786727549
            0.3747474510117391662246344231747941844476302495521964393015161055 0.04506996395394777022975348710302292910821963391639834954575239190 0.04565181423884463894729209243736746130218321950173868589008409009 0.003227267555001175498201185286229615611800981085543604804999838289
            0.1259415895966729927188232340628277716679407140580205298250767280 0.2364983483601896777327875632814415937422308776179088030256180445 0.4476963805383269856506460606594921859830281535861456010212766382 0.007248310727428282966539370728728344395201835894622340353784683447
            0.04391468784706396217086960975693195393149891031272696884150697284 0.4684734011767374748577548416351052690243599337011291399347077923 0.03382073302117223652523182801555526738013952157152067644652584356 0.002327352145975614761048023426066392210362475692196239181166722298
            0.2759149948385416000315920227673576409893942392734380785398757082 0.4414605477923806776355725950356935804444557983282165847343024626 0.004959984622689418363689855119921882960819822061100465727874634924 0.001809899276496269906149489752194440044152227496390731801892855470
            0.04448794497331236321575079676016446023159122246667167702958761595 0.1843759704238228382444498644135568798767643741489746612382480942 0.7123587737247805762217785720658511686094214974265912536004336171 0.002718781333234786804168397545439538786305931649671502969099701455
            0.1521074553933510546104675418325191836481353547379601194051002471 0.00003276660990660724077700029068548025969295877900316222194105197970 0.6984023270439966492674745840828988988331802671406128446026749713 0.0008693315373855214426088968576929681154831399927912162044165447919
            0.5283761411200380512916654921984640197389409908749753683569054826 0.1796655092065580367906159115653967475433275410681862970107803828 0.01555652934607857660859631905252063772060683093651360239093189622 0.003319458696985569207567523969867115172495319299922894287617570024
            0.1558914925324167368616721834990531541419272254983356943478840015 0.2833730099073712460896624191063118316207632223832119835172405297 0.004771231257693399446016621342652095186567033416078663591661015872 0.001548010654622153791683073159423256504559361501395977165146768802
            0.5063810674038869822707742800510838482042949339244250184046218451 0.1825534557375921824216102170224779357167905927626721539440957841 0.2334471938288975592668417425846415938357335809419166091997427464 0.006611243267164813983073922333528882557244510568404601694066099780
            0.3576905860457069439376773601136786741831822243961648478891420719 0.3238656117974799232047784683529029104672932830846238799241421838 0.02723849630498044841977181844497415760148808885680695075510939027 0.002860931023631523118842718940583597335198203281456178637059596837
            0.5841420914856665244209329132326576597193333286583986537679120176 0.002540704978946981105665973937844211711846295320105597509511972391 0.1398992617632788986909033411175203742696150489468263978823446370 0.001210871179108164037116427942825044173394727765679389216831705746
            0.04651593971018430711908292596877100804203921752076837859523945196 0.5822274997846256811058833428695714753157704402408487795576042762 0.003884539661620314005273584669415761784627336084096570980394792396 0.0008476949358401605316769958068622703308363498629690752158450747811
            0.01293390119103585710413331682557735136787203359804015571462361598 0.01616029484666579051559983208128314627661047814324368891634663240 0.6456882301397576573320382764808791954085188771772900400178708022 0.001090430007558471084951165980661209437346725581203345113181454322
            0.2598187787262118534360634142167464842228520958218396871073348669 0.01983569782351146555955010468054790596595070284668219399354573041 0.5649530302924151925618765773351832038951564095588958788577951899 0.004102999599319657451988354944095353860734898478735761194803345970
            0.6733625891557631628391364671659723197171736735680122902508155253 0.1420115855452878943932089768725705635100963829485584880789865630 0.06714301236270326690975595591888301775990299820468891620557889858 0.004668086805414122656917770519744446965829743340797720053165569035
            0.03341382571277493573853455331039370214853501515921994327328010996 0.01953408909878516301306542717048343162549353275010341192725724100 0.9175207095520148879326977896373273479396755392952905795311360445 0.0006666601287181236105416455328502239840349184064666745479455534295
            0.1412553348791873088789157048725083071329739064730270464794526145 0.1211271948677318387966893894811158165943826429058011646479650080 0.4920020058898867581871236117945316394342973371837767033307224788 0.007252227897936232682042163755079222304220649928295040507751020434
            0.01031938932563166926889663140042736717394612060991829233727471837 0.2673945135019837388507408569799060930046186851803633423806567543 0.2902351430792413180393448324886168467448763311065727862285916862 0.003153511717273864002496170023114847813267824483153591412848400693
            0.4427212354285602576575960035690412439785944557967216799970207699 0.09464234866912460918671189975536445261605941202116360034528257569 0.4223834834827999413073140880947826253874060129681701175817221426 0.004433522492403445296394793035839066592017833281939948784447395178
            0.7666514376071695405708899364354827058847007602122820236284131597 0.1484623066446219228946175289731169669163156808632620350762875982 0.07455996577890362949540031155908187939162734666193027067817253444 0.001431571695572909961465111541083458208693589816185499506996834533
            0.2439886978894507016857556825743174069218724103515600863056024173 0.4648217120292746606662613803050161137989039340305440027311245912 0.04413225846600323241170936343079100312371332081729803534850192544 0.005504468617623612943072464109237938365916055590712853187549530581
            0.001003637367565592143524286066656015068237164798063245053163446833 0.9545471198034619013934124918010182708568166673578491058245041171 0.03867152232045123764106955879700346951322509988305194436254167609 0.00009012230550226578233849772690183739533420364576949834920746453874
            0.1488505051857411208985531593497891043729217999862890923642945911 0.3002105044097018736845876867398616336522949661233604630760752389 0.1149168656682911291555658799480261522574842112091526864258349704 0.007800379208284770438325052154020028206818135340021104080313732094
            0.01093614426297327396231730315770394018659378214449360435979748679 0.1449699861376029489821653272579493496535669565775219602854638966 0.3998321116765977926151456086506183253654738670595993295335364870 0.002164739139918845361273811385772568004097491716122433388581154284
            0.3378881314175249884867169423339557048810549911109145767482972781 0.006860519623986630246193341606750605535125559581925600092102720480 0.08667118313011621480520118195305198987623560004637229206136833769 0.001583422577516778413123348560420935257913187732232706105073395415
            0.01201076738213312712004065881667090609646567471696298022679654846 0.4107402605953261119104851862574154722298296546280097741024861246 0.2904426972606392642520641112695150026570148131040971366768310242 0.003263676192269655793994511336042316784374616470702791132219074965
            0.06035880315685298798925060394830654225939153270312702633335798668 0.1331597376335279496877609611968445772070254293262869894508049588 0.4284958711920983562593560337214284230391086375152751866726182185 0.004788788796012572572779741730135924316462313752373118971893022300
            0.6788045079919969973510255829274789315452395401765619615362186019 0.09483813233351527975103184254105581249010624185435362744266502288 0.02696003856520882185677478146432447085094195898460253764394950071 0.002463872257029119556729155102034335541463653384720818282156373256
            0.1158021262997357490885432755221931980247292229770766226162185984 0.7572221015333239765124768066432012730289606629306259894125790264 0.007560215954183777299525918099569373231606198638200553696156042042 0.001354421943687365456211267113288037048324578241020262418738249614
            0.1947716335687268933084597331964835732565688880958569634711356752 0.08089551462428246111458576877807746209789459256619444209503695487 0.5759709062517591210991288172300464686217419566104910516035269328 0.005242398757022854569458452445552732083979875593239185572120326451
            0.1733322951928096017037179936023700647121656645568355783535425239 0.5923757438538803601821940665463760891613439667016235005484476027 0.2170287220022910886364193600697148244763742258910877624398949651 0.003220970916967568120515106838349734091595531346520621288274580420
            0.07620309691093605142224981869846113574160678994661167120393477841 0.01202700311291623289133454152548489266594399403879646401565643919 0.5068122176047081500296883728530647233916261437713916387406559487 0.002211833833250518823704456943150645102059005088288577928555083636
            0.1938613165237963162665156428454331765792537813385893194457067350 0.00003308920686064219452589015838682007796796071408694021202934435979 0.007143991256849199289300943335686675425863963826144380686815678330 0.0001818431323647096506414074140624873262760534554161605376582041558
            0.7882103145198279882778062425057819744129658215974797421607693678 0.05822648792101907541242090883264749478301328761824740832814055056 0.1373637971930919204502701006073469295122771043395536612228549935 0.001881689746127302597079128169763954392785908876473896662479662353
            0.3697686795345650791761195276820663051934438843391010662107975033 0.4354360079537772551876985793782199830080228050495500390742525858 0.1504109053409699782931991222340645120406920198159669896003052158 0.004080466236363526062458618916330096935110222985315264451720882906
            0.2014813062277828721408612950466924807557872763863839952414607188 0.3848084196390787049211294816466532539016853561666578501227719467 0.01072996519325078488711670103043246799517936059232878379054736935 0.002318954149133790432060763449453438185748387393824850501120359822
            0.2306829954912067929806741403273359293008577877605802884155096326 0.7241873356075289287569286427895763222852948075281843472194383777 0.03108725097953924936028793958312036220019346600678943321832875591 0.001115601411207886438865802744434801130972702077767772122339102257
            0.4093026925867883280414740597627065966064342304192617370323960038 0.1416505932296143738432160027424088477530683595185939763816034311 0.3477336001874800922386325750200538790517720545659186742088051095 0.005646860027277702564091207499654779394359782810134502548719688892
            0.01028375333586873834208625179106278764657530362514596930143289966 0.2629315451199659716380449361815929010087192521882102665590111310 0.5673854581028186851370960086654267830333358117305592898743336540 0.002369176153938054544279767336643976894682540068888276004452630378
            0.06762637155006784859476471304192931215550824296765133532579536816 0.2376519648905046252710079515205193490559841259342462432085452948 0.5859929838429854000181568280991098167814171398457430279943683318 0.004947204387439134775116536562445173030758571547415420016924932195
            0.3106365066096704652157796940942904198128530436688637850830246417 0.09903725092095204783636775271773909982489520135316258719225483348 0.01313054652798026680026120070606125355685221620440668951920318599 0.002129977922881121823620258291441415291255108192699584880406014840
            0.2748804223365795832454357434710795418032684604353399338707992316 0.2082527281489210589348580415297679357587956777781920533499499716 0.2128132025602344004896455994099723024167970542616348129474583461 0.009687790827188033592019591799930243402297224743823445544822997244
            0.02976550230377661902321779771373407185846155761035135051491585724 0.04527850794061827004534462806802482608317151016172883011457238632 0.09661501007009479333730069155988890725706696354122910871233536422 0.001718055691805735036380745652740834125145959285068381902478219126
            0.01599813960903710268664202475575657460691990328111466067603244185 0.7200378230458122892980001435831982079920896154717797068730666800 0.2529319064660712736372311168298773571798391321805088924923149518 0.0009433097783557659586415075417246645621681241188252434863694773159
            0.3653344688878719670461738725951835766999843383837076515738073859 0.2216965297712433182929920657590230850366676890223413771783674236 0.2423407099002775108954732645854064949809908931134518276592404200 0.009923030310910219182411097266540195599190087198550673676662476706
            0.5835502179731631735177830487514799463354281112673589549244086475 0.3417323551670283258161730634797060132259644301062429001626771687 0.04193849995583609262669007956511747410911860319970034245250189611 0.002269230709397179942753351249491453856693660561007622994507377677
            0.4752762438622759436823215577845624240243467036865122575461810009 0.1095782904776887264252027515001816165028911497133918344690540438 0.1772161970087090272229152206243471786010616249138265226647584080 0.005947675698822548338487427500141685179145295787430223742718123351
            0.3971393019521512253975059379965943794482315740535740211700639262 0.2216604515774755474211427210937739858086809910428318691965115157 0.1260501870169569260808458852010226904744311802312414295177710009 0.008628429978683923338749645804958086927067216201284768160760641199
            0.2896667120636869359668632209920976262984068929708612831355919478 0.3250334729342835776823724729122991050484138815063204266941089304 0.2969215024689163421862686439043183011068140916768480534042636791 0.006029075203231221548603342476359715252491988471169220507698125611
            0.01027311928513874604036425707071115772807009445908744325836281296 0.4600521706290525108494738869855009092912686935923906530819196572 0.009452644947057997253480975819405564231215614909195084107158161036 0.0005722598142421129423179831822701329472349807517363683977848002314
            0.002581783143252839801126203488812810631505940656146001692519510443 0.01577751581877196053621535143039875960062441184697566306273790762 0.04199394161663546827460277039180312255352938143265855275950291747 0.0001960102558335677162647083416245535025907795765417016287799772713
            0.2557752912498582089215160411472790536602313737910824704145562541 0.1845511639549907464470804982782463437117085978911926915874020711 0.02855711984147401125310117124031434608364454043028888962520686574 0.003530800521318158448812370174424595493491461641451000207715072924
            0.2394509295700134234219222141301135295114316359249174917527549722 0.06598899407816204739961260536114460059084497236487537333353599263 0.6333401464158564476106510242489877523365525480133974078737472161 0.003817227453466264229660887706212119493544312283535550366407160218
            0.008949497346622354134102747234526429599385803976148884375651255476 0.2158071614860515707543400823172899797366558902494672683273836002 0.07353927276508041151709067666005811223185895394047278126777815134 0.001738309776553043579786203695686457558474401895534205209647806068
            0.9027432253215797101528466628240147537995208170224029729286597472 0.01030478734091108176703762568946142802399865533531102982161507613 0.008436171906512358213963113573752503040243389864570097619659207471 0.0003422159252852178361255788487692575277860017998911987866497005662
            0.007317316423184885777418021749973007152438160212922112112060602583 0.01651340045699001106881533441336937464072879335444879667626860887 0.3522982721297537669723273184339238995389164482190736188151503702 0.0007757588947080969782014381637917884347967915178387209080174674362
            0.2194937685233065331329478553239981707540849858454046031842316069 0.07149978512728677097796438629175169670812903933889062844336847325 0.04949805469575916740415134846883149349562021164117865491100330283 0.003685741197213769921683599835236261712925023300798092029980803188
            0.3589715109267678945242641285099506928105241047965538598081200848 0.3695011376542612935509835220032046280562935237734046813653679251 0.1359000334094554289922494895042256752190912583535756704830672361 0.008835283459660659261676249970734094618361883426339858770166069955
            0.05003765921256834635955429098207692155642501205954735006723394099 0.4338040145041179708499292196923998062122716960043367581536434149 0.5090915966320477348301073129592316848945247872642259619248192076 0.001312545431114469431427991509178127058125948315619445318648843210
            0.4971792598683715409059883307494903475754826486043861767223467070 0.001037017675873364333439893807826220242858207905972111634626915656 0.4221725323053695701763401612774122974251462672551639330644635157 0.0008594406625405831673788366172659836211489470887804685692063201670
            0.5036241210800540362744396989385857554312463890647341767328783088 0.4210215756798554884113776847180636042623842457391561677729370399 0.009379465154791228294725164441849239284545788629766821607163132951 0.001821658417384747329339353378406427623950673727651364771756888085
            0.5182524090167449526399649255820564080435478944421100990155154785 0.1572828804948833969885646095393553197838507132146782684920996022 0.3026112603024530465078439222339212000346651396400432974326123880 0.002959346505401071675119978087066139936050293286052277951529017627
            0.2167633184424010389741068639979808044856068719313473798872523822 0.2848761367817193753601609879218262109168043786575621589318606387 0.4436093535536825490837232481977306271465186504689215537301601413 0.005699547841989481349405609470414084801947870398582775442854821858
            0.3739891540028977688944160066201005499943855228964687364009005877 0.1087527603242101323220226878955482778322075443788108274614057251 0.1606892867598232931929252936632820355004640601316418569571508172 0.007694798338796779820763665120580137745730424373584099019199117913
            0.1579129341193875349302453198644773241867451074541806678127089721 0.3102382627323137732066055359260672948643730693382554097345564260 0.2030409719455489204361191650127239138862866123045771127548681081 0.006156930184921675517726777035906336221340851867595777715176592864
            0.01440757932619585339022562823990222193165359730965720992113471570 0.3207824402434313930420238962208164617751604967331237660123679899 0.5922522538288934814543269287299658366369782809823819274296842054 0.002178671156009655819448921918040489671292115665556129453727554419
            0.1757061175704914044059703288036317946139072877623059431376886923 0.3390843453175567011005283312509229645264528325212631758524741254 0.3434154309939218601425843686730303688297873232354319576862606504 0.006979598847506803084139730003398518087376438539882584773999966960
            0.07041812668575839313630159605443448529817770534180568235586288116 0.8631644615478987944079248459481896974194901693354885124412137495 0.06126077285216001531918515358823242662595455892656639798888669581 0.0005885878522325545781981789740745336560431489223881820113629690539
            0.1989609468706957297340754272687046507093281202652414887752571646 0.07068218663778228992258450345399983509750914193030508134312190645 0.7187871265529461950244244164703303354819428692655837843340898356 0.001892214308327549513739042562086121289327092371669136242908122401
            0.4103374609450581424558903932686648208106360649509291522406956648 0.5607695702911564307149136232599252468555386427795167978165090749 0.01484192203561482254841811552887620723523728169816538731714525336 0.001091156072629953863174545087371074772807222621203872474737868739
            0.05103047231540017854788919604311936877461436160194871105189761995 0.7786260834078246493469812555398390121422457337040308943465873841 0.09377910195644077162791290257924193057174095941591833992858874368 0.002377527137392173627786509020659569883996848605972686751667230295
            0.6622875743133952880963340432693129444219702194629849770663513265 0.1531085857192350814922519254640925377557230698388842669553501544 0.1411624335778921813795596273349183644633796361013337239058235490 0.004683351102299813557209970559729817324055515679510587247808984615
            0.8830469638029366378019886276074800904321237355373787174874160867 0.06305154813726116497493804501060444903768773306226331774866962467 0.04464892280640127853759603264890566088120683157944481948530046685 0.0007032734517216652544737956496221090195959164191179234295036351610
            0.1500479370037570174536361349109860211437714950294737636622698734 0.04212193944984947791504070482688281202202134635196409834049440416 0.2344328170206288300321663256425028243304234234378796801647971128 0.003623102829744814395361279921884800969827960952694194641567011648
            0.1429481482290415908710017255702724837474340696092858694989234693 0.4528607406310413192312639565574647513421616544176677221428888871 0.06707878576264771119538749407168971496192531021943398801086933162 0.006293275048348728880899285540086828804026506918203308669705207973
            0.02612842438936695591007764575408103207489358859914835511746385847 0.03439423632156479538020046176114795164315889062580526704594028278 0.01855844212803777350980511889833820869682677485243229055152252064 0.0006136318279291410214435017561125435392423931506801260989432232662
            0.08739836318247893298949043115501212996141921298621355099644089130 0.4507568481756116874024867712353934748235399237926012425566625833 0.3652010324823770427533808344982247670662202582372981650117885902 0.004941777055336622926108919900036019491618816641054917570394379373
            0.1305352118419406845783900001362227870925897944505484855651848398 0.2461918070517771943667659688741020580902136357906404616829935655 0.5857409758936516610463007707698533305321850326783503169386461206 0.004204593091815081347611486532719022751651482747008747351671371047
            0.2554941954861784678668799892576857467222277859506623842633987980 0.05555225520412402035355917499587085774105665915132233076653047861 0.1289951877313160984786631738979647086034357662758407651078946680 0.005130583206081992332648247291898341816977422984612122702742963996
            0.06938399285669665517769655852086733802332275336245434113715319994 0.2176648902213957598630493827244478273399944492482822109317466271 0.01230994324437459559566419106787628076763843724742678335891427840 0.002023009862274312753016593622343591398261408373031034907560835551
            0.1500640532916830977243388833684182101576916383006507729776024660 0.5872287752474394892999968509617262624903788693759686460588621331 0.01348599431720181223942657183131653108380020521358908299568866113 0.002907455202339925700430560932774539109477831590261022693930785546
            0.1133256963912607294433946603719868570106018120962258110891046159 0.8609008347244262030582874707961639421850111223403019118429253320 0.00009469920757928326683137171954536771903215975824102498015686454670 0.0003249841288964359339823341886324242219578276878110099041105169384
            0.7065157700007623684470422253147101943069271681989783451825039818 0.2073609880882175045999650798330029521679582894404851611157399616 0.04354749357285785589718852066185575921758804292704087257225479204 0.002380688141813300205159533013412166498275769173363905258331149188
            0.3462052387455926249934248305779661051515688391854880931574383008 0.07343070548396231683697326352366592892022871676585932016290964750 0.5693059409162642537269014408796797853253642025295044395997847222 0.002213663315970313208094407296778499099546419426217900207357715910
            0.04634484652786028214148727368199197693840587927808074364809491738 0.05667891276456391410111860970608599023629449191742488857771807919 0.6724731936109160599671923790512609618797493238702507742786594903 0.002958586887645701856118831755320462635372413286100212640214094136
            0.01265974507647292981328853236975456684232623137184511085184023915 0.3092550375752282463028496730658415643650618942177760925305758012 0.01455976510401633782567561611302965549750849863136971611488652316 0.001037626892269469183117573197574259399415534587421387245413207515
            0.3787512551846207850720334355429979137441598963385131038626444739 0.4504795623160287697313995305859885960504288423651142223546013574 0.01268366089512246068993185801487537244198235226192563650041846615 0.003075662326847410572333606237264950125530272473302796667860833903
            0.2453201654935757978504188598291801443345278513915184850307766986 0.3453874644671402387354190312708708096431267924631079309107941875 0.2260183536561162998558831544718706341058799858422289582693543266 0.009642724919672397231875780616753338376267362991961876382108157743
            0.6780406524704728787072119510594689362105523054611654953819653023 0.04036339333587550814309997147033470912390397976758777050590479059 0.2733991716047112485276942294074682617059843232501604600729414742 0.001054660009164030456795609320406782955357695825317757155464945581
            0.3649163155406229409012424687773770607421476359421409290807851413 0.1727311076339111485458288692772307725458729813990154880862635807 0.4542658771005915367871270095066581207424787514255188694115308253 0.002293555144973095908668217792274933979041142408557873360762044959
            0.009661195000577170802869955536109024600169981585586588712810729370 0.4628962842298255871721429567348474599156674414135506077840973255 0.4496580942329842196715450117670962675193355172872433217580812173 0.001807719753033854534904875065165663847575003507181372711665299829
            0.01117764625580903937644893026034686193492385357455321480471704594 0.3049579237575728194671148955958314348693869426149167259175861731 0.6727787082926584229928652880398063559007789746654139040027727004 0.0006442624428206756720040995368940215988877529283355584811606882067
            0.02587951084428033498843629290827373762319214937000580597700809371 0.1084214927645943893550557543863098279559628755596893206106066620 0.03705076361435993070386841169755255550236326872155393051744829580 0.001281673898159116681434945671128073007281673438132554998849780187
            0.1857896458335199082511447788360672596284213180857545901113337169 0.01223370834204710169651045595392403431293960014051363559811955543 0.1283910817427421974399324404899474956422693669837490792270034525 0.002141539176727191043164008441991210939030803109634262912411210766
            0.4791250546141465268456829418887185134319454454429085448261021964 0.00002762557992462347825372891334665684557279403928939969257670801075 0.2778333058118851096549011761539126046047326207279467818102145643 0.001048026088858913605357624218090002925225648389376149041498061416
            0.7114923159874453600843384203647886448928980908848782492972249451 0.008778040346671093068274422968527980777691695838471037672113021884 0.1390668560207197551498638322378460291538622781306318436455252761 0.001809065624099949878499179097571453992143714876367151412444690188
            0.2764361100325647623101268343763511711710718600481107261575320126 0.1468662802013844045844331312410268664591088294611358993325167335 0.5227964642653351595694407616386534386820080381798783503725159888 0.005597711947236180804245688268504281482761768491709957952907590624
            0.4117683321847329062971682261660458117120324522718516820281705171 0.03562219342945575277294038401303800275409070565262443411755402368 0.3340554619577884081438825458599353979590015683074855542050929066 0.004491124605173814052861164765955133071084504748732712482574339264
            0.004962739317542861504975457715110327828909806689647276828177614333 0.2138760125012745209817941870624293765402750320257543371966198084 0.7315243971305241653340660660931858367442102256007746641389047475 0.0008124539299351452575163495496223622152823268270053092622307010210
            0.4778257003720322403122779173739279329833448992991553211686781694 0.04409174254205472177909316707571547343236748716684491670395575948 0.008025391835259794043552004060086937676007804434652949530356419005 0.001303362197836365274693717301482432488693982648309667940012696226
            0.5459538308258388475188616422995958047137252746450394356570980372 0.2431525765642138926197651986525819867970709902268903423166360332 0.1955325428666893905456173619250315440063983717167356335972252396 0.002980040257581438016528308272950104781410339835956649987230909313
            0.01299679339383472176873638602537188035229425524608797040581532836 0.05593472818460970898587630836093475833639366739011403187853996688 0.4597710030592316545164546143828533515076790667361549865628145713 0.002089048274641060942172817060440593568977122791659903968342226463
            0.08088381364344342802230456975821932813334531334549498775649736544 0.1132857890535819261628508186383094976976240944721028317885407904 0.6598275939499328448042034563193474271274483301215218745341347026 0.005345874563150710105286892569317882071295582161425202847061234180
            0.4345220982509993784516385036648430979968468726700022046962932083 0.005660459712299900664041380942990381759640243773280096442695717542 0.01437958837923919808263371456421369183063067251637931905373031969 0.0006338598162318641593646142671417957413023230683997789510462680485
            0.4003590965763676378256646602201599341566756567047651070541222872 0.4818716575498396447179474925002518042492920047858054598885416972 0.06546076445124478133906873409039576903388595213331226502814035840 0.003916908834487924532677707496620731620096826762979270292805762502
            0.1491871401465661200955299605627526609001554609648208093880526739 0.1534055929131697939589068676754571158132814424183553686154383110 0.1139907252279752209879919273965702138297828433102033322483511558 0.006763554271977335101043335090393879210009787416166016638010490902
            0.1269562476853716702801950840858455803685089676808174409445159762 0.1241419997551239892717527805403376532759938925318430092782062953 0.6964989600347219185282644809202902571746345666602408339500008798 0.004332635878644466449833806087863768948269723854887739254672065094
            0.1584312082025432937558177665071528688822485161080937170796634316 0.01244900519031287328480836310509996966001727686356036297427765603 0.8177448797019383975177209537993880095400384867689217805313485436 0.0006761346658209420411325862125378997410072797549123456581202579087
            0.01225144649859458909445502650242820976771520193593098620541092225 0.05659905266646685387081818749312332783675360832957862996675972105 0.9206688566503744068758120481134660186912510876918471639639500148 0.0003902370881060002382608445447223741621259652670875799773511823195
            0.8921362713564417789586906577337628109753371667365269228575498666 0.01455415599392187910851964409096161720031007416867203292770583270 0.05036298614822620490207731446971523604341629487531358816816591828 0.0007369446452572605879440487383555623805159696621858325459187399891
            0.03431191820491977137277112955119828701703676008779347932178739709 0.5362014789731096017735414697149407593756009070514440832376007923 0.2844023669572800495614799339919002058826442019789606552087235080 0.004696294744752758889139832563821432850135789517511600471566421287
            0.1315829137517876504540813761183598927343278957401583409133512342 0.5946559793532211096428694026294696975750079444347316852522311944 0.07252156912342061869303440571526397925249156542153838930531779570 0.004825865975760623323160446397122081656900654668005056991416747169
            0.2547544906324717550495333350128190195987124292266881142250925800 0.3042070146619329085216769038851390542726651782991291263255400204 0.4305728220446992791351267949698693400678082874367434807879762194 0.003067318042066500616345832567066915246890678065541746603118969694
            0.2869110817871233017009878995079198291375126404059160056324760394 0.1015748409528116071595174548932696760976932475546960467245557619 0.2946170523389787729896865666271467278626500281064216185134554094 0.007502069479782500703097070144534533006941905479382680357016535361
            0.01048870102988506005848844659562576687971327454240857392943021943 0.6808869860400346364236048094262874459868593434583965986065574562 0.01548365491025583982483776899471050144868895781062976120637392799 0.0008621413878015358385447550431751032592517593463632384891039461743
            0.9623581896083776618716947364519872056555595703689485791168903650 0.01028262774263544786652383162442157556694298677165857923505281894 0.01412961496471768627364641464586838692414849816492071202882483705 0.0002018103332966904844442786030173515912751614410115131225256913752
            0.07697010520784840456687707391987493968862705001196010455237162816 0.06532087871596312327436647836140699167527136760420007183527823798 0.3604808093748673853646795979159804751429956251181938044818883819 0.005450628416282603182598332642482000176475413137101471847208051931
            0.002179806124291513195531936727748418022181033461490463236212868038 0.006005112381287467961684837506225445446939961155763765255463619882 0.9775228109755917534950950254382333581672855690899777139983464072 0.00006570886469202407526774264153700501796774934792146146654389820716
            0.1571796904934289234902128253826579619667104571721963098737311611 0.1165546410505212143486972498129180364090191394813722316386651613 0.3455414864694976811144241109969453343676187006877704622382844709 0.007766034253768797763348784087940047373630266668182899779293797569
            0.003035872333356313326697856938008636057431600378662508309685631394 0.9371860966690398927929255055002488464776830298950661817480158994 0.007850222803032614615028376779705138477070113560546614370191521973 0.0001706914390444034408217924667231070190942143599788713343642543440
            0.01427430041659973326210070499364928792494590960234438528186252712 0.1466251244224936968928507388006753369687996495797842043401296853 0.7097611378155729633156121791620428197947464499424484983007074261 0.002318746493438608498513100519089649704000826951183720519215742530
            0.01591134675308319663608321645891194104730819067542453068843382137 0.006178510998908397243971200738539090079877736555034447657189445606 0.8914922270326858415812342648843691980146441807578434782755765497 0.0003149807824171385748702075776183499886509429332588505415468695643
            0.004886249957931560293262600963586008332497029765835959382493518269 0.04021678597277656654612349024709758157395295310579614769092563531 0.2069433441415461483857232161099583966023165605885176823031457085 0.0008115971954610662117623108372572297559592620915641556966839530137
            0.1852192886744719278682360501936977312986761430797407212827944676 0.1465849076633293749175176890566996344239285421553127564682961452 0.004035315490436406871544851330870279482536373144300889653645897049 0.001300736662526197803023349583817997897765452335601420179416974426
            0.2178660836324134121978308566837217384467451210203596991846103020 0.1739361399687737608060880012258274630549764090684112640409296188 0.5967875282224048154922721582018295838752973347627556907007064630 0.002786054031804286027211644982613176567802550771315170767371498665
            0.1328284705932165561136117054397804197829210990129625529541728614 0.4275029193411817715788531228683412744740108761797715007464261508 0.4160084114867152000176365773491879895649188435915468134982433986 0.004070719464925106100387679029961165581901553053302739135326751800
            0.5232418327530162308491304522754555684055482801414698814022391978 0.04008153002248534897848847743595335965744010231835446077288508511 0.1174205101686309166567162697131302812414478964586503375842405719 0.004513363175714121109018716597422456154915155722091547187241544562
            0.4682552534035649762087474928970326775948940779452981405653384731 0.3720296175436648006793310524978846165521237119860044387334062768 0.05364048753954104599571010371852708723657729901973880579551569120 0.004646319017514519175982388052673266803546739737179733618267355573
            0.7016045242001031063857432302104987625838448184068709603821340201 0.2734727443924528419977256412778625187772765216188900131739564913 0.02010868578470003374291254232899214009762908081559211061844468840 0.0006335643133151814250411284560429051707302461633002577743211565362
            0.8002956646448936427459296894888601176335291969402726618641511899 0.08119332517974551140545175678520368708372453382230621890332241281 0.06213692766424307476669865166890612441767393037031793078709050081 0.002369964733688230893862317979098938139533045611717442669471773413
            0.1377218981829194416377565224109330835000765995222942942169633898 0.01211814371876444702965240142129845909648687323751296950281023063 0.04860434211092543534163410474950873948758981895877304048141099879 0.001222213976625129012686051875303087248931144255693407335327781124
            0.4326268181756263306502742641953483333681429869004087979775607144 0.3923922755927199613574328192942506470176778142624947541358263663 0.1681628768799361917823851302986429927638942263935289358244058808 0.001956719920225694876959733703639643063914608444954529790891811885
            0.1713157659503795680643757259287339353720806304648495186556666161 0.1742089040085498965608949539722354392708008333426470931131929286 0.5430815998388990667156630497103793537539411464640634270509382696 0.006904579932844458704753661761755164373938196076068125939070684263
            0.008882322218387445074228807439007085292859751288089894560643981535 0.05699005050118285283550543428913799341717020158569108878190556175 0.7571686354032568895035489239535352621244757824365873969354342692 0.001252524950080328908286599345952369071346927044773419982440668808
            0.008108230698535449912780977864767128331481135713330220384527828123 0.8062327571506726725430661178683062147981351703228033582142737308 0.1491666017813098027549213624045010672407817924637518611318161645 0.0009506099737481177395917989158155654190729406453385725635888375434
            0.2530170321533840182744454931709947508165265303525175540600770334 0.009204467371103880855420719094132071200758802917170805028032935823 0.2133319657270540777479133695892297361857492245755362332448634339 0.002583140461795589586748098564730007396959603041128640137139075777
            0.5736447083733106723087769963151305386687265396474477437610391881 0.2555190160905199862334132334299515689094046095742462911263888998 0.09221851630144358297715184378443222323772157490528552200210066010 0.004039710607038482944861012078021620139812583263851137436714009730
            0.3991924841198162128595841561053700210261938636679488795258068640 0.1101457671044623309271076692169967705086052211575237536680408069 0.2854417353131195184997022652423286739130235716894096100780498240 0.007955847132884729989174130389118127846410827172020286037451072974
            0.01116886671346468915523506120450943646588862045860745294242089535 0.4180276721986238981689336403904627824877065100670954495188188957 0.1729701315874739893204709376297196258965795939138172402129932918 0.002846433211269694910663402905605852455772974731826442498063477307
            0.08996837858455338542724118662311674139539611826232543311453677501 0.005873568057112973416806589019022996896738256239849149690294808350 0.8510017151700371716312768858736786942558925039897636703380658445 0.0007311598305418091253429150772920602004837723570104703411549831181
            0.8172118107363919967121028849685094259677805403970106443475703257 0.01151485658204512249911455844836455782150560164591350148700233378 0.1201638091216644342386225325760813350626141085533021255419775004 0.001134870523720443972238638005635951382494132487082647644605271903
            0.01142554577253106674436291806686072883970096906604416816506934955 0.5739414657563047105744675591672989963451690866144164047789487475 0.1705954798204837503005743535445500673349712954808615818006953211 0.002726362721876734991468842759181019017294113820435590303475952763
            0.7004990155493500044083531266059654195688885189036020445422924015 0.03945845715566998318293556966543915449808690911655361639602171784 0.01185236628020709223872643461390535528235279872502742433060782920 0.001046697086487727273002790772350928536519421217894756245313649190
            0.06491908076385639236314536897849323971190660800476265060472701160 0.007774683033064761820598558084230373452354119205355565147846086290 0.6594369076191255289092698646258019460909464953652497259214033742 0.001408170420964925091924168422125281390334805497031376735548783911
            0.01282303219773133373375954592127372694188424869317055897512637673 0.9834785402057119841467863570416679997286424553244639690399581760 0.0001053279674068596195906843973812541942551339742314955816218260540 0.00002614544584193723153350920034683827692711586108622748462703367609
            0.07836634537404384130868856239937366505407959162835571632756164800 0.01104294734741077993360720495007727926930794600415636555980377348 0.1311070652075122662124695458543903492234742589296672399348165195 0.001539275664905196060255074890956532764877401843844507514264439837
            0.5337974145683983950591687296486504426014441179975430786197248964 0.07167910463831709868611368028280487436709428057275817115300983510 0.2853512756308535882355344081407352938293154231048382734980600703 0.006081067733401102548513134607469719747636839479986860710216486133
            0.01367109199423635340638762295613115290180022217752523559332735242 0.1649840716251160049896803004980445710023787599462413153845997008 0.4813325294949778525416884922899691041567563922860844674439669558 0.002016695022599950807171693122132374939223800747277525956603618258
            0.6190240554715241664196683691125064435759311419970647029751990497 0.2644613512212419865213729422148939362314664772576728111169415626 0.01821305790604046243913224374187043859074365160844757912679818834 0.002622204447480185768192576887273550370270212483428866058678964235
            0.2744893985339129761357044341863299996332297752843642389136260490 0.04506205710413906379968448932575067843190502794647648703000037319 0.2616044817369180783700628853349584397414761531764646891723015111 0.005038085326383496871458695962957286588886901713084936627747133842
            0.07055162427446294071583784080121569800814277833204393247543820428 0.6167470030136966975539307758809446834878730549361732210697934824 0.1679657739334441497845574874793390399667496883545772542217872670 0.004596430093734454785249018159948893395402087715366267965070763739
            0.03957871499059648693651744939920241959428178590253433433632200619 0.1133853539254026971105640947219834050129181069643617715071752747 0.8230756726332422782478002275924649923418473777971373396616697535 0.001372737513980610097279595431441523091983900686102829462599949502
            0.05984215279921688668459631063441038678583506372479025900174950904 0.5824456218112976249356364328117662229946937323245236804596430827 0.05135020813856370601964510296668370516580476620730939249528496889 0.004046733831390046073326895869295463086986227239524091420285474815
            0.08319465217620490359081847068709323548920937966547247932471790576 0.7009972960049565221563939078130658346941549112732617371374069165 0.2087831946878639989287948422615540905165363930846765618691397305 0.001225877143282515351160482059860081595577444500808698651294447251
            0.007654234306419429606078519798148765060084178699661194062664598638 0.4136973292653841167317724928703635062990144200519661599838436901 0.4061986142336464919817421929395663763144618228651344426947657719 0.001937722827662346124337886825519129172807207276435866176046099412
            0.1129810256183473626769613759907184027895183459763174753822269828 0.8026556329144044244234517751766355166292314025662014521266724107 0.04102805741099924180092794207906812244489460839351006850779579005 0.002467456888073125245915201671095228216695666868182378988919412348
            0.4225946383943964475710171735292034420551458121619951969652292626 0.02908550676852247322512734560924981935846709091614635269372994419 0.4167952176521580919217775453264747901674722494987630949352510975 0.003939342579571544188882266955882175432525177798505592663594811495
            0.01654881741691923819876281031938806950359844728963238005290385066 0.1928663625395530170525730478957093162476283163668949390566681525 0.1844217605197851136040532286794398151552010052199162008306749118 0.002366330548639692169327957532487284820501086321430977744766796826
            0.3859756111496367717240847474924613223253883284916226243406965868 0.2022685764187495421395978342069922958252506265376826957992998840 0.05539941761622875141289853535237884104212996089164399236118535173 0.006556146376638874203230945970358126679731712156075064071344013817
            0.1294499263959364552582367052863887386354530903703137643539846173 0.4834491672407117206669973290432012730101563833898093593604881969 0.2373722715372632797121598839323110768185686082267676461480376193 0.007659801553597347522327439730610817673306390502781716324107849134
            0.1272770485646732743218209306259124980902529732745117912179017054 0.04096980179460025966023295374644570231723190045875753242947395164 0.7287785464701574916122965127176408937406999525419665634004882294 0.003279500221604578159349181205919878565519992408543652994899745559
            0.2390527236756704805962886878849523477696949915719585359427782059 0.1152045852219428706700721921980119378730393767561432197699345633 0.1871607659506821558267333039683276299281985203829636914341109094 0.006715293201084725017894701292893249107093760603738261283622432160
            0.06254164379999928992493566872953296775570162478366017606542074751 0.3147658962388438991555617283549252946997847873015436908043904251 0.2261030057480523312089007526535667265257122974057537851774418733 0.007894180622828432527650332616392656397213464369813315595965496607
            0.4509205143955304787279282613747683500946400366370443719540637086 0.01268306491009438342359041988963299270526891035878147707726279733 0.5223839107612048208329089485267293701077990245101346471463663169 0.001115178573898693727540439827381758268538816407192257779586972479
            0.01342688286703970690166652125658581380784296881583506375217973462 0.6973776582466187180272224542270225519719846737586619377624542313 0.07082383571663130486628660959144002195665322661516990202159672432 0.002153574088746200726518693573793664153434222761693664381940168449
            0.2122120652207624137631040668504543556019492100918453122637336825 0.7250519843640960112253912871302032405050650483105163292037297897 0.009011720412176058635392155757175959048795450540788997419093337676 0.001228829501272185321613310168099740372351617192106777239996466816
            0.2713754664005237311903643705695954221217917014158468513685410998 0.1360287938406649601194374825746873664885544582382814516324081443 0.07757388786007158586617724899392016084933992786989830168647418621 0.005036832935145405538375960695084979571696553280470617332233827568
            0.07041804525111942905643778179855076934744257234256238141497933620 0.2353757909980922701457606468906616656592549899926586341774052591 0.1416424153559836135423675749491611614007849452375895562068504005 0.006032639389594622054859974043089584999139935681161183655689887133
            0.006380350291803261042237572383881261669413217152111750375713860045 0.07344839882126205731160534515910127349392782763762195032433438090 0.003401198036759959219029503977673908573962926151479104966900090395 0.0001610316629152658714615008415549539233990488006357992254600866908
            0.03864553167589004901272076441084884900653074070252104340590163451 0.06863532923036611594006487237234806611728256822287934381880504054 0.8092600737946832144884675419784294941754916502444145640747214462 0.002400518667976442904382315465889253247356935696985565441338873065
            0.1853266322178726490928094613361746143044134925515320520436066093 0.04856290813123438743152261932505753271054690572632335899727856657 0.3465199948301286373808559049105731174633709845251754981886260486 0.003145568377257819041133387891286761203865490806394614652636034382
            0.007700222007518068161361858981400396126875376689346832783831268549 0.5588264724702957856538547557267492318531524325266617284148854289 0.01844177240178539527707021291801081250869786970657674752374266666 0.0005390644757856885108611077232086266377132455927998890286697378186
            0.05489917295687167608765519388432233690244511796732376043201829503 0.4960259342460459383815554754600158396043852095345049150693200089 0.1485454277790165006340051468123175633085188102010696204998320161 0.004585036703692315478492616213475093671975165561513391890571660333
            0.3141299300432728526523967763473404956274348326175582569079334525 0.04495442805734323707100430035114953561898268019342615356420335997 0.001298806960194584242368969703380693499361901977293472907805518249 0.0004717034637084523948934677801137530041490947134209899407191416606
            0.002940576572629289251172038042082462018937988109053711033262900214 0.5611089497913973896120994030254758808024547451399906630800384448 0.2747699082081906904574579215881430715198320380113417298092567713 0.0009529353740055312656940310041182565266335401114191823566374693044
            0.2661630190370710831582087685658584273118489072263785269638386587 0.2041999052180869540581538545134564531683685613474883846923032073 0.3898651930416155538509127415473799918538664700572074636413974804 0.009199889868519090457331312988345205142345337215017892215598108890
            0.1165241410026222518126810005250899928186915614976596074315769198 0.5660749515341731331633936251944930875925612861498780772819256763 0.2596950770266511809121421323157483960261599092061766744596604533 0.004536494451644440582970681165530324671801115305625572364520010597
            0.2609168107572921228149294382578586892065746599783585404255922521 0.09788855053430124148471420754992867678819192486483877747151971617 0.4238216487594789416066654593509438682369030350731947008715240253 0.006611092199269854716114405017185575399054919462691916115215684599
            0.3458741211777235271345294872940968960101078752137904168202230280 0.5747056472915952172955439025400123767895293016521197041160461441 0.01357070171658237037057670378023906649080366538571018007919649460 0.002199647167377883117061683793146136338718550921034386102628126415
            0.6902677516791427563655670563962438694861766523231699460362642999 0.2722472626341713362813346248969936558292418279623055301816757251 0.0002425484360250912435849343613292074618839657617630917069351352018 0.0005594384214035185949925378253392520132394406666910533536063433073
            0.6690890427255639326418935962490224855426075110958834949607730316 0.05399048004913001270897981621827618862394560623664268515135168389 0.08692937446906422606049169143865338931403229015099460089625937659 0.003998605202321794819805548560137089840357584924896784779133262547
            0.09951766756229925086325396712596015471701792986887464277171350766 0.2190067508441801201453483237764131721650107870183966974063747610 0.3558844085098739177823143448123049461197523438565750520115269351 0.007206917883523231481547064260413092968072833986647941521488737701
            0.009909907353461673478886543504826340322275703821302393317718206210 0.3852547708276898697669334119498881599122589356331836008432702803 0.06538462190231484667665774362928344069200864548659284017899121787 0.001899577616877680447124164756214689769830490247192130853941074557
            0.05164625518401525863455114292552800536228204886110515833004514071 0.03862310371142001341552909448455904646020747488721196729747366780 0.2220487200883573455796922651349191277285821529048016902823502747 0.003016109236344023420583851736278906108065064549676555351093648387
            0.8933263411890104926118869021258033365517628678059717812819058463 0.05719018259101757286460825880411706923847279381962003134699805502 0.01023530997959563835469888239494416819997944825651874334118218806 0.0006795898743855720177326375723111073082762586190618546267188701102
            0.9016037306888656911511193983212931024470525114685633482435072462 0.01228742383960518661143416553657575518236687245023912177273735693 0.08172469307381075601944084692569154521289892970285650762516231920 0.0002858975873281269975231454554742330889205335543087424091771588274
            0.09646169292311048556995242969367448512083177928372044829054442534 0.3537975318928540227044656026628206157190100517841799199679874048 0.3251517356954677292212726129470298608588525311477990086946635626 0.006067704900740562315804967300301496511429922599238721824166897402
            0.5237105523483669324660894324224604429455558901317467294024211753 0.05505157744500640545849043847979824834548430904366273012534818893 0.4171489476658070006772133249046366578076193146392789823642828559 0.001040569440062169713413711969723688611718237102968601519023493819
            0.2396333792041221216327144514508291813447635655304677243115545223 0.01572395419845766263958287965760687807329765137923368273463655272 0.05487788692992225761607085632462270708466499558209302299619637866 0.001321770513131459319094432083301765669992343303360344574875378140
            0.09860626296365636150965225854741144800048139669652237605939712693 0.04975634092482324789102283818571953297120959673267987862761251281 0.8210008015445660064809346161834849077711296902150988920685283415 0.001743298493453371120845034608934813981090343861179298445536233226
            0.3027778179756795734550098889221566560416354811076714011259136626 0.2407279278404810879413432940450194255044924854508398263374187624 0.006357753495312102688273617509553996264408018239578550966085596910 0.001999136113034981025991569224980703811023657806027822814099639051
            0.2997205154819913706779554897723641722002363551596825034056787948 0.006830285346986225596733893302229817454475523193188700071273532064 0.3334598540401591379563075042082843928812296732978404828600226226 0.002385985590832379652006953257581028567062603817170592826410523303
            0.3611583687209222218392145310487070235805649302343513623953620243 0.0007281767921551526077279989951474976043925054851090230292314785397 0.4385964046338668804566489778965100933755157762595239973076206788 0.001193558001280536373969150470210790409817872796240689355772966741
            0.01099654989818601506335378535717008621120812652970889301693762725 0.008538759997829111682025878391036562845863222663003036246166372316 0.7832396249339352234938937244704257570024977849080305771261131605 0.0005187105211318847504338327731268962007453136013378710306784208059
            0.1324480979656567320092818296099911898449114005054035247096290624 0.4348493766984547838709211806770635382464308375201360395986914502 0.1587494393419596644236364183419584881261580646546503875970482829 0.007556921713559140286752784604629046492531953067260631410540048877
            0.1242145092036586897468001202472853952849588485246222276063472993 0.06303039966741256969127719333931789241301272857134498014187310838 0.1179263635175650497187516898579547793776836012079026680931116895 0.004416282659037114334751585829208018197044642926434778461133109834
            0.3656486169546797771475136593383658640170816285116153905339740156 0.03489648667781381793745526339224253508800852196877932551966546438 0.5388630950105140002252277325079278978789657250815936681566554616 0.003144753140924226196671415039500352849548287757847504763444754157
            0.1284711556058542413519184623518941414920833312388892964836235856 0.1291900279296747123778898516674178209538723962679610795762784535 0.03484084325885907616685826537158545340012070520619783660234761939 0.003123578645482498483631184378441898889694770718133315256405514060
            0.05298264114978927168598283133416936223593256950146268319438675157 0.2641984083830640767930717533795453388417167865177840469852207763 0.6683995106257625802130950722303251964793769232977867551651958301 0.001838962700556622458221448748922315575580719220668038385216498789
            0.06123520670733428462858492977847516356511131194001282306351897616 0.3762039416106041942683645742679277437609606564304712530650109659 0.006843749904378936221863534452522603471067336967526330562842308975 0.001371741011512351886016866202316330324568530756669297261189148268
            0.01365955631524758957217872017345965474701562206680501885226399146 0.8114674710816154637828381469250797180561488019350060387157949352 0.01092251250696433532084834057757886265398812970442801961640418562 0.0007250035956729619045516875214269743575655195188398548623610202213
            0.2642407588082833805681561928446796495907774882006700606207262785 0.3501184920184749819309293058833974122043714096428285558710445957 0.1167277491148449370672737954818790365847067303768935581753310994 0.009979280143512792172727487198225783245765488680516494448994945072
            0.6970023712740518807126631362495084616247338395960547827398482981 0.05786189744971472066141815695511155095316851706694759086124202125 0.1595807631708029073084484062171133780044078760254280442882489787 0.004151309193801297920157413740678409942274944287306067138359211253
            0.2379954897830224346404078292587630680710359194052584562024380492 0.6039729427345990486113201328404763537467932736709936521036717664 0.01100471788272129207865824516235104560163335214267780344086791406 0.002446298857661357593767942368043681192645062672204685749794066208
            0.6108209166857349190423213183611547037694740158877275898430442914 0.01074332605245851623036889039154703357554117273907067407223426079 0.01140019325605984273674030248249455405486905807093497954889789852 0.0008239187203228653666098903032802294291794155635213355859662349608
            0.7865837390224444840079657113350484115814788041378665713373724425 0.04365604779053957594505472401756124414098061215519727819430036870 0.06431536991359185153774519058158891611745526667471679341756972988 0.001494897399751927337354445767860190349195419921202256733407324550
            0.4023976233869577504435591550802625442327568275400739238983628370 0.2833983532798849465504946893224749538310765722489494967319087632 0.006314026462342483372665223000538531107709659328062559253856380845 0.001735909786054742895816512330652963026536376654586336157941963858
            0.04611450825076572588491054489748304452601897954226482796089576969 0.6704544648285207459210830399183090032559275679464214237421968059 0.2256925115823936980036998090583746542441972257922836653855764600 0.003381254752343965488178886577315650632687528793222296599212873305
            0.2874958576029597160766206913654746112933694275073621518643634303 0.7001653092635984053738974128433867813884362981954698059638893113 0.003562659837436353584555460822116852786308437660690690910715740002 0.0002895497806949332825743964395688754490913092458718429143133042971
            0.9416474213359066363714529307518551152604901904674372618496737815 0.05612040951192621338368683423027867227676435624936727023717893191 0.002138097995175756564044753449576662674786858945745835529546567771 0.00006067796242894806643856559447893144259716038375175045514212109497
            0.1444131636518444991102520808836974743703051351447235779596203711 0.08599916448628220637161615534715528459734881927576979724560780311 0.2282375724901587095348211260794250535378846616535842945186947594 0.003953512458260677887575209457949554886462315340592959178326482320
            0.1124879934978856987654842977924546002561281763693190413153476864 0.3632871221787742144726457095323973859068127482331752611833685778 0.03774235826772141863858217472790469584925344494690430358311596951 0.003451106013029757356519026405755267378734956263771967849386256194
            0.06554943425495139206154551944603976434070309783816300238131806604 0.1064468381034046808277501540455659696254210417663734179861348810 0.5493726078134037304939992825177113361650216715634718330918733195 0.003102229788636366746519776576943776962977606831896628799614325862
            0.6748279843220652580254325057789595269436412572659982709567161096 0.1359089898997818731782469301378717169694086460244227268478775573 0.1892306516958439989528273231830383389837712380641763057118835348 0.0009178934466182610872788581370546118109210651120948815857279138401
            0.005740626648285173085442348963786534571241444206011929505430149222 0.03091912797410039757366122382232841851277796160886754544648788538 0.8816723829182566329618070588510720011715952856599453031047816734 0.0004009304965349830784486264485056572345877844796313455426974776969
            0.5383898636920974250463253120248135038143160280768143106644083152 0.2258305940436333831708248575319572023435692733761621680577727948 0.05332438468558471264815411615211898123917574333717561826573447622 0.005281484787446606729503257577709982788750774174300619055952303783
            0.6238558584543988556728820912443171980006340621877248275170808334 0.2725824710713934960512500471687755021683534430144168377392979931 0.09518933704819147235030774521471294952135226763233188982316304657 0.001766208869011119770431975931631997446724352139858399985642592343
            0.7860990754862090918782879693858586268167245950210752939675533329 0.009009356158990222396787175689714520941268105049195897960207330406 0.1942368145213331376018219523736540187332208031191878843189485496 0.0005297190960718472393523365709289920888796148523671012556480430681
            0.2463527076395980421428423014223298898003607035280058854335850798 0.3116230398662880358357172002286805934270167226371886739480130318 0.04574106355644321604791875724760152889455952465438756640718948275 0.006007183203868938124754685071511104789652636788090100458930305942
            0.007981170597137183792227400260056644973949329477622980887276882585 0.4317442945926710551535042747472178514708099585380464946012353530 0.5430537450449907475044023837283960482066263888426598476497704301 0.0007621155035206956412732673645354191703697594407168419418120731007
            0.3963517056969446826576336076416521895072403756838391451770006894 0.3081858149172699126765662384753165725167437052551643477510103570 0.2302255695393589328499667356902979149923882341412472418311513544 0.006218161812402578346862058173847155472460029342295442973446331181
            0.1460643068540250703397824036586682805927385039033069826861063691 0.04637033832144088810385224795420545627078675962631993718688841366 0.4743259231344321000563169086477047362645487834393600374161045071 0.004938111882028750180506254399076473441222531610121622679293780673
            0.006454163571783102744504476031574409911817037190072007805517319091 0.8334609419015833464399443762998146520336028696304566332596302391 0.05611666760135581295590371602818750112541589239230307279952661722 0.0008645092624870250254250017489404560272437717159412660117246519116
            0.06697805240807824335179834989899748351350024298366840868360977171 0.7777064651361687448587963589350326765105543072957152636534569733 0.1321867037570240892789307427457249238987181464250797449131208432 0.001885257273372779018084381152519491860617596960545681693922639533
            0.5395881052500308558175026486910003320217471540675931489004690378 0.03136471999896161273843148283757225766592214950316447050999265025 0.3812551488650968053302695029563338834699365350723942662402787765 0.002840224617278061793810291299454624051275711098306469454149941408
            0.08468864289207591287091143905066059307530704573699404030298945530 0.05329239482701243951731690366856451970005197432004826561241905252 0.04794689656281525857406088975962794122038444582730143911830054282 0.002066014916895572956307014017176809588721854269335468571245743360
            0.05739268752921877938775754930833291972315877058634865707573257523 0.01592695931926255362176053422350036847570931139019045041558733272 0.7875217211556792223202933946552478628255037995198609125227301957 0.001699097037723494877785420581922645195999970685982498777818008656
            0.5229366078584478461491353034835233226889733330423635657685628269 0.1316357972728570289725102396499481388875262522494956838083251508 0.3437423400385008127929946170651625503286948670338459524025046068 0.0007122597350873770932189689755209401610308128378163360688045761327
            0.4198751322578820832808543505168950718709577983507315517859349758 0.0001973448480502702553299482383353610856137592719729970614112763493 0.1780907783175740183078214903666900200324403478581388037977599901 0.001084380703050693243960075723557846251242033118635326370507130302
            0.2095722296952700812682147187412384974735711702195659595984159529 0.01378690996587224998680342320873541089842364622003570181810285612 0.7159888809919654647287239821538526710220493523823287556244762586 0.001837917296156066725107901141553071027629003942560108117738146753
            0.2441836480170445864705413239565354684140639256973083618655677394 0.6374172883622983400794445720364845401822986889067454885484061351 0.06016393483422860632469405794247243768957284949531583203717802113 0.003353735952748793655933936711950458081656504242037857719762046107
            0.04082163076881841411953068160408700306364771196303851399579692835 0.9287686455085670923192439770614963588286510464981933588655613182 0.01541617281416981579001189718219622220972267109211876998819949692 0.0004729375635617101911453628298392771741481588987426306535835321165
            0.06370902142816631079834328729854509605296394701821514415010688834 0.7043829715957584596921586287455018154867605301112564754244340614 0.01265774276359397045632422525480283948554919448799687737757276640 0.001797814412235931596708531916163636582069651221475684232656301126
            0.06181587146347279637740131504355787509093035866800688272803040195 0.6081140923537803505795218656725121631025165416198597002154363178 0.1208015626137734176820621658156135228389465088920896149451932716 0.003030370039396793925627819913524850217644446425227821381184792791
            0.1803836668631959127005701442338273070440557333572063674282069173 0.4879761999993840924394201333832503572036326394179585566887798232 0.3308061917745222181898971886562760658580002310661762217523378011 0.001073981636371553174815986867018792692038983609148709344474843659
            0.3412629890317380796379065751229432622044792031855769301914011197 0.002418100210343175937382354797034760062398869170071782077711791135 0.5838313022202037349134123053553195332133133968912578538520142755 0.0008889768372057340829551939735616597959032201190406000629523387370
            0.6269100034634457615118096622169475321263422234626752963207666733 0.08438193985895922447519960334193062996299476793634997295082974050 0.2553285911768324431966583634339820697384855358280323325812682169 0.003471672922349037452807346953698413088056433969375186362233365344
            0.01335947698052965213613148483902759485995674662216459557920101190 0.2966546013373161417222757044402528035766608377426383122196683855 0.1507134305128142172141629617835150014976675536559526807874089331 0.002643741521489810791497808699020161781275101553593800596462038234
            0.4150883332462787559946679746616315557520531145669860219776542275 0.03191570908040631159123742917059019982909692108106998597129441553 0.2370973825908837532950835727081690487513671673657537545694754490 0.005052959878721789834918840722119909868859949771613172586030350780
            0.5020156724787777451057848836810790050669813935629942852764996540 0.3195542786857742249186084954032062277069198072124831185398834340 0.1311750664149032732694259955990105045353220958533494553062340481 0.003983912534347848144780230126771827002406588972783540297433888222
            0.3271232389035919261348186584755334205703565230746514875431723956 0.5974101530182538509068516113160217702327671725644111646085977163 0.06571737218330898153053157634026934608075676499378585382694058148 0.001669834876476379740474672478856978856833051800339661243346858520
            0.05982137767355686874385547821277175814155325601728075983094000099 0.7401696232923349401940098359960529264617866046375976917490913823 0.05542504826791235700557916055495416363435587034445764824149207304 0.002948504806262537286127150292611057849165000891773778647243091324
            0.8267245623719647295843334672283377707156603320236364073688801857 0.1478396495978842722601618569993054652423410519818004192786335847 0.01237249715504064408450768304706597869666859852963858793781723363 0.0007213776081853596775582649751156763644596844964368878848868956093
            0.6034797096538824558898857546229816692455283892344804605707065003 0.08842068785397129370623928909975568073057505297755864977167708595 0.0001360535248951856112559242287105461674889995244347756952163481331 0.0006931018538719157214336541222721868522695554361373850849745448682
            0.1472607724393446734766257455972468624320537868205744029719093801 0.1908409112302961209965332661592079166023727791691176811812985726 0.2259629905492380873917754982715237179742187846020104190332936391 0.008878167007817710135892606849754198885357509734400227226865690140
            0.1576410437124083357693432739114385872905856999132684398909891045 0.01119078175735080534958402046789320052973669575864344502728296315 0.3666347226779965587021307852352884347476098052366166473473162503 0.002933098260392564605814179876415053195700522535727104792174139712
            0.4255020035890300220225873644363162176756776174740158258307886382 0.1347112636738802306449234659319121107718561325602208956330591383 0.01272439970394625396091519512447929823441005057272552876581651529 0.002937307582347090997262431318803024132673070672807020627730266652
            0.005348802204701800338410437153482469769079830860313779581019660650 0.6366970241869219886814801262604241831807682045794875200799602625 0.2927760844357443461890150143988325681500892290514530828790979619 0.001231576015833403394031804960014747677881367753464434038956465496
            0.2898226352371191351905058888000809881351541372651432193302109799 0.5449472790350773212263135076739703584442963190243419594395700216 0.1564221797450669523368981694273589723520697472897887278236229908 0.002101971357182153924992231366653297853419438893112152365945945186
            0.07070646720825514667187468936728070455104098313403703957132402080 0.09573216990857833881310758726327002258467083442840653612165268087 0.005746440096747628585204273567851468363788315600995261698781542797 0.0009258882184097140254301655215929493313695038921945513228227175328
            0.03562486110908057162023331109731856734668096553808332563550287705 0.2779833256329803653227221168195394958089671798394446836623697970 0.3890038202262759842911824434577655977167216199260486737867170279 0.005384348811132494628127473703689275313737838874904188412715788672
            0.07031897095089737558162509140810219895803086682797391192634429714 0.5757267527573483649662192493167495094618134005151221531910668983 0.3417392325307424834641576621324939536987675480560345070686437502 0.002152148701742819972560031934115392357026404146327656481853662722
            0.1730143898806602496051308932618334111797734370460075136748262417 0.04711657848405264600258739117384227447751945441132320715665058836 0.009881902703518698093679295524168746878906695234695836475238409700 0.001320078843193075004201135895720721632697936315918471920664883931
            0.05262564147408953150005085207606259110055994817283546270889287798 0.2001379106894433324425008008124464078878036339542400152781784945 0.5286301635202305111310673266741898948471042433279009570342481400 0.005106216702827935796026610046189673344999672824859905247419536250
            0.6758456304202840166602010776925733436683980873813954006086896693 0.01287249410865206805118700709251437525844748605358290766680811030 0.05370649953542008464434437532998087396748746146826066338706722950 0.001843082990798216588465666729865147360961992730182223928880076714
            0.007274475551749325964219777529851298116235417455756811453160622121 0.07364044525685234084802583567964025032321998494971193702866332168 0.5981239953572215187585027472887616262112878434121163871109733200 0.001354865343639099124504112556364293786262358167468289721279558443
            0.01798777243576984836586652452814545198540250331986224989966930716 0.07798484728982714735342787543676950118887581955830476085877170858 0.3051288322432889650916857668664037054984389514222647056386992279 0.002618472939159648075021605117585833874085713685598881967921541405
            0.04340083774891521584416482739529704014585848101433226410625985262 0.8635145953359120546346745607518303542469801767972051710551420974 0.01412676965841963671319625998484852492121840906270560834642300052 0.001139862389414863756856512331782578951783052815597453674543666182
            0.04460822224359094661446115515841377869666785685460033340811502177 0.003498603625876361140645873598596308120126078622974714110266781716 0.04986625756203882962996012576048435454596338076919575428161146514 0.0004301377792166319460944625199577395008781187605982117154717498714
            0.03933958928068964017172117460826054699429928459051528037600578157 0.5253463236109069738438804603777606388322064849183090419444243040 0.3800316327558551943236750065417934439488356953723398446481483231 0.002857302459658518009396729579279789843160835354421260802519531489
            0.1624197211138341695195809393508820638778590028371430808537563892 0.2962088735945388818104098465077600080288312477655908599722518255 0.2738162703894372716481940608788317571650166435983409278723808973 0.005115502832706720571213281852328844846288151090696747727367156855
            0.02314870933030146057732422704139000760640442735968169358914439482 0.002221887413861601803148814253056436576796244776772420732942259264 0.2351956697855873947899049115190167594417551844675620477566318581 0.0005343831795023204744780507242833266144108438512121050363357255632
            0.1835607446352020880780433219381676576194017157153345872790850736 0.002301712673102496050984621965020621496321135703424122837831672697 0.5177208197315380185765230544157242329899020511555317385336814061 0.001566934821184985033797802742323618931678861895856923326914446749
            0.05423363943674417702359278884412140391581337520324493843263632384 0.01319331653810468904063769625284895995012199680198359889849089478 0.3661772829441070443036593969816922726526233467834945897486519372 0.002071776838502478505701651488358379571484057851596393890901671871
            0.01889523182858401126566601726997951019911244527370216055430584850 0.8883334567672470590619405350879205170685020470213347732035614356 0.06030910586435826252343003654218873960281642210551338792367595511 0.0009280900943766046699468628769923610981647035898347867873057456983
            0.6706943150242932336019829037641454954218653932122373633748681907 0.1617443733165663358943365602495019528924780861450075997862419122 0.007510289777648905107725055140549315935950068500901816490725111585 0.001563585857466250684902658988915079731935216176958761811456761860
            0.3577139502188097820522751735594446207504833063009721580215790523 0.2224856771796807106275123758868070203858511056701435799931361692 0.3711704678917599514330029707099341529413634464706141288885168918 0.005605017412785016020310608186911900251836170160782708553596677731
            0.5685934064499803233318732707896064868380494626650907640731826641 0.03594625897654123078044967623232604794605175955263744864637482430 0.1945484066823190093254007837465149322932973420316111110909237940 0.004672415404600880897702140342737450269505622844575858635675409654
            0.3968372587702373414453862495056242587955705162205614492801734385 0.3344034066909401409736494645225513513248202935177968317814187961 0.05594925640240089363578115155358735593279054775401265258469976934 0.005737130442421236386691544931466555833839738009675125107239383379
            0.01312731476863628609096918941031390479308106146494298278946969166 0.7118193917815844279947049928259469409065831582377826238123963145 0.1545636632631585393021086423239441590452569760293335187293052149 0.002376544368337984407844390123148087943141516029174328087986014149
            0.1152635245580993111514502353682309683849359352307125119932668021 0.4873909027215010793272378018003850060435883656883830106995280161 0.01118597063232400364477544928467728884526477475863374259701644557 0.002126436300008278810976662760055251011566359151624567142262305616
            0.4832903542577469371406783108258245315440577415484597541663173876 0.4366478405426259590490564839258849007553403854261311355296408348 0.07068264595558336440276117524003746550609321423101703219182194487 0.001785697769181053497104979988883888458973431236576417696636319774
            0.01411287128719688385274743197831753610305537619134316540351538640 0.007892641896378739128389046827247216953931175232761756825588289202 0.5085315898950174746303750071499689820060044500975187270133417053 0.0007601968726567811684990857589713947034117502139403991363134269431
            0.4110640273617693334972696320512559441425335489639348715846646478 0.2793900744598538230562596741073156470473998960198552003831309512 0.2991702348301764447695291711196449229199963293692621764333741219 0.002849031933722959829312750229669202501520596420340643552243986681
            0.1488390541641035672845412451870419031885847132354623579812923496 0.2365891512457089291343242120718488942334591817132756331132581073 0.04618544928481982037743043092625942732867402379383219373115473349 0.004741037664367374188200408560232223306292033915387578596938011009
            0.4040707318901932490034571149207356079463167234599231246324076218 0.09609764140159854894850328075042190642165989128753037161987055786 0.07410819661390301445778267960790912448802114567721291098709910388 0.005187729950781390259417714102924340323175133898371971349426513190
            0.5051712522147195649053035010404751656781105972657307388254918443 0.01206088402581996649686174771903750243229579557381198275933332104 0.06285286990018394993431048887882870923900850700065345785482056735 0.002182947639289968863135966562598797060689480807526751663208886048
            0.5354024534925373621841790165260625915518804491504832365343923368 0.1258089196835940170057908191938084238532164177926097940186837465 0.07841639773181062241664050579371807612651949720656784741142176145 0.006074878966506628470174033384122755840542298229688777272066937464
            0.2926273773933920917189849000337721183655406340249381083592975221 0.5084408770560687036056307796048215711867556211056887009244263200 0.06560817443541946830349883986627165185515714808159497175294613117 0.006111982690817997276306537237514879233743053706546445921680977702
            0.08420700371502688353622159303744920614891863210908359193993595009 0.01561412434085215343345465964189037064571349845784080785393231339 0.007807710392639943749422109283906709803659050249990501826632676244 0.0004564491180366047708047021423178236817620070572817242680195899048
            0.2577178059328004543546810414172150830412922085978481265500071145 0.2171775609253054818362384872946969984804000349641158625757029792 0.1163673677888993617372700160990598493784117247987836463170738532 0.007689386314478297003754801530093533530188674050719321056717964315
            0.05227524223775071252212155300920186730475667893555437029709444081 0.2873022520809976411825730426795051582144374411820377658219742732 0.05676310565676014055774386239529810348558188187782713853332585187 0.003692456574436430912027693509569102139115244055895778846324708663
            0.1318180652451385391035819719679771646260705973356013547723457992 0.3153078668601356031042948793284325828949898971519665126738859596 0.5490464706445593148978420985571000143873459351694291868813200953 0.001392654475945979008535712117914461754420119057481651125808719244
            0.0005840432153909993257072278551635429515666576290474252169425467087 0.09841637359042399677280638745025910491339558501787455088167087335 0.8480512903173947535188130490003505354318072239746052613578457433 0.0004491319216868606731780456531954505336244478327327213367276335934
            0.003001952483197130994935354847488046837568134535289365679280769555 0.2722600813733426571685548925360799947799213489279312401412773566 0.4478627482930688575012561439530661316523537053791277402408313291 0.001385301687857629314635383179516990386587370493770999285947533095
            0.1242160132668586890548652222594368690701341258742778560232183070 0.3246428482146313700401737594511611116478278416122632948944722510 0.4571388007725605250088244788931811521010821908504371022286777639 0.004901375579567376903658016318675486585272333861729883494988305971
            0.6840563294536577128943969337909472124371510850069081519062505354 0.01376345042083642557036639540682168590382636370491498808685462800 0.2442811111074197711283635341387858891189914917000851098062023317 0.001972139379250137826532128615564380163738238356290098410212100242
            0.01289109503404370620552246359421428883795776158199673614625859824 0.1680234115596743098993166916541332418723081376210236444536377383 0.8125854538378442300738585279501954069615072117577952756357475370 0.0004871482877502437135088030014983663225638855410787696868211809062
            0.3768950375145050739576058553502989175798904646982980450593759638 0.03165769371536806059516934748426888327975652022970440385109670239 0.1428089830331633555609985957541368661671235381599922883555404126 0.004138081872556880661235472275583000012830502513624127988192905534
            0.1590557687099552263288442808498094804106448303174684735742263496 0.6751927323617514106750238134118384272704898573578117920601587953 0.04687997309684717326365037518222814464909342903528464824968203310 0.003563065591428539466092298656712834086963304823807842072058623685
            0.1321178366990884756277507501571934541803565380758989433609845732 0.03268338546257972931508319938712096903308379685790553507397018686 0.6144284775380544967708240487342113764378365392233177399830467620 0.004182080530668493357573096087380543925106951578766265318513158195
            0.6218815431664380949515388660186835304033305258769786847135204130 0.005237971396018672306004269202870089967117107708175428891981777488 0.3597345872486769741342506352182555342952094871244107538635247428 0.0005832208987795256386967971610569157694903596212338770253906377046
            0.01742240919479620116464308560756811404948942468250330746619179174 0.8584411215631877023052212747685296375144352173888980735421930842 0.1240124602418006606332062261049186797517913802994408773279716716 0.0002792695543876048738544393648532313048007987454446701855773065787
            0.0002037297554622384883602457708505621245597423435911529091360363311 0.08552145888456492249045594287848515534517870779402907967107825109 0.06472404580869564749331816016482149826204771639899710787130867729 0.0004237322784237322343920160435047398695851740644610604556123858422
            0.01318866266716929478368763530284037785995750803011140235319409763 0.1776836117524605057117001535931704762494048038702002272798654038 0.01229786656299167716071636171949928960872465169983062479892630318 0.0007563453801254342560313271251582938171491438312207982424687956393
            0.5761552723865355762499064265297914954456694363453118324569392777 0.1228669812717952103603042910143507602067336249037209161550950674 0.1579820244375103469723892372285021770023578226932195792636791032 0.006317007457328734781136615772842766937913518900514735300776030146
            0.01330011953053266795756746529409777645262244459741774155485184647 0.001593389354212091290520920122262307195144252151466194580110180288 0.00007876723209230800100146845607205436169554010867939061813351940467 0.00003840434639046549908052779599954966467643080624346027933086815763
            0.5537013025460237823525840308325473002303252740868072402461553443 0.06618089448962841955436050093575000597979133416717029615591724706 0.03538345627368971082839967111975073214941276070874246430790083362 0.003585071998299355765190188357882405882858793592266775346094463995
            0.5094220497774336486451252506391436667459692021405972718769048309 0.3069224227010623557554772543334668567504073382073921795654205273 0.007213070308095837611946656375691573201255319187399612436099554368 0.002016597265465739315080908314378788954788529603706941999764521375
            0.05928487602400647144964553290628783245113248194601536258941397092 0.4001699758982642713008259599398643656194682918148296194481131804 0.1052689350575260292623827866157663904709054501416474514579181111 0.005559052922826551106633660986865149222996433561825250443587418211
            0.5749433865614502985294681739818390772272627769969372498265146245 0.01167336044082906252182609330650915950033716123819035471651870822 0.2792776758283916884606351658904256107309060193393695950278162245 0.002356745205532086881044261557863132991187871908573356473439526064
            0.1990511346856533087070200114164181741760533633344722634409629583 0.4346118736374875361717932500977127102286568513046160447404027990 0.3094726681515035814144353729659289794210420211209094067304849050 0.005344660010479615662031173931317549268823943543162538136146851136
            0.2436614929147198632207884580942250775116908941471912197813922884 0.4805176433382087885205716787263746328762028299857552616796129847 0.1915319428037299426906529696191858485913108427972805097151923857 0.006475628003766006106094523390166119498053032297935169791083166035
            0.01043459556552599474309807921261546040271234550281729375520830991 0.5415510881877555285001041766321275498139885038916959067608689887 0.07945643060700702207038856490883058758513586855376970820745779609 0.002146897557009688235049780297048760787751331482631613960434320959
            0.2970131799661837312742218844933009835851925438484036369017646679 0.01426238728033888302283339673948023409662821064810108300493707431 0.01624806013188638398215283470716073799370375626028759802591055236 0.0009102734037769918503399059741943278236740554750223649770101116891
            0.2552982024123407027544703764730816484136146723759293884236455708 0.5655583017478620271465982524331897182249365654659685845081075322 0.1282136634809519003635200859090871154675507291248420550516967166 0.003229648829181954394118787931251898495422314856469400744286610669
            0.5563891196787218152886056976647583485981354905682118025336682053 0.4219825170096248336498146414325720938320399884780246250645354499 0.01129840545776684875749141133177514541946164429668879361672313174 0.0007266096763198319689314741761394040818214584814406727460587112085
            0.05733028510656885514302289725307130901341518480911882832195521670 0.006196141746368588991433459829058632521945828005391393129711338433 0.9361805600406368291985798400659897743393099369851926265050329387 0.0001158939610687374708711854167467454702829261004902352346919544499
            0.005474058163658866538844167407443228419444213447347619366341431634 0.1470107438988169802661630128322798825380875159365097069383848533 0.2689059998543344321350691340483762861509994887534316082620931697 0.001126105186566237255517223111755704550063696157075853162529777449
            0.01090488408590303366979569440028872851425224562575370829727171266 0.1119548859021604417387847873527628084792037630973986093736224442 0.1478124455076277398558423583382298211653755823085166822713871856 0.001605679670923283444078307007419635490964229281802997659413088943
            0.2952929908588748566568846548880907365377910685507454863625864122 0.01414617610864711728524494473424185401592540831712614578160806500 0.6782904595886750401861128890137336777222328717900626039358564656 0.001025273078460148281367850438751022079413849789679301535410436043
            0.8101418131076997636501985696599147103927571162737941937313179372 0.009672915521003086107918822065606883392956242150183985769713404903 0.04678296879648255007440047755588147511963723511102385979359537340 0.001021174814127189979049037635835632561295738890936224040683268134
            0.05091199715494668815344767585787984483893169401350585511060130587 0.1017702728700386468715663191484832622529084980265864165546528274 0.1495326670817075370778222320801083111068238959119210333008541702 0.002549723549546520711639461681246179519014479775477567085885427958
            0.07308520360662558389765319672161997973095453739688908839495471079 0.1243152952271574147683135098303908628685436378392949941166967561 0.2210965743830085900374527463171002321951129505074344701515923445 0.004537691624526472552641210024780914969216336284561591547462683304
            0.05106956563797829212111753842685167109106221531721050004592726014 0.1783059443803421811297282592055283164749422119581403481470724959 0.2924795837648175250913018576104257878033370146868037863466192846 0.005512750057328084288471830556888359979959857607355445685652810917
            0.07688057777084003475558663298599751283763446557726774798679030157 0.05939299406042703646372971988243640894270675620741962782673369981 0.8635710446093310565960149647040934262848026671533138445371521664 0.0003553383765863533505609385704749901608500222071425753111240224562
            0.04494350935392333161563176414081064182910948010805717730114630720 0.3577773350743115380710560894871210313206575497681350319420060018 0.4380103665558921629281647576474107321492318609732257345782129485 0.005738477054476798373617919316617272092424426848690281055264366464
            0.2935269590415080520328303270428154136498007712478341628600265376 0.4127727986506392206849669290968409424055925710106102472758916890 0.2763802816494873984624344537018117589806865140770999559407780712 0.004252901281373294762249650842667243176282928001403414590765765740
            0.01103612719900978176071657941419504145124210356540191392827564306 0.01089512613838024041347934985972315962190953332626768203632642126 0.1248025873927093763731064337611696059715881672408903670266034980 0.0004854158084704463306872247034314678492250317522038701646364740896
            0.4849948738153608117791924501321872637520796366056608535692323516 0.2350546750637005367083063744178088026715957654138294322893199205 0.1398502223680118447652069303950708406799380949235927496308160745 0.006491165610712514607708054835975391620874162266929517581216107390
            0.7787774971481311699278866770447127826715902031489190100563985417 0.1363483739155516972818959290216276089357182763361889964663106630 0.01208125917843540608964716677729480616649001618402007816930220718 0.001490562474842187334032474653299689180428209396486926092018123171
            0.05641315499858742246424072767634702835707528875774775420665222509 0.1539062906305744220129775363921590899782780441457143887063090696 0.07550658131443616456907306342672657382180259734605296583575308971 0.003532154682168133411057339495408635613842043848622418077557578989
            0.05905159012517649986986683186876269321454598241325731727019619370 0.3690090428428888716035099380926284863713462320424916051514977779 0.5230963428551018705396446768626422839337586812838772351023319558 0.003704375681144793626925774056329256312971357932198090047826442091
            0.1351391645097208853630647352430175397996520898941211496828909718 0.6583534958508798616375921849248033323717748289150661885356055305 0.1310804855628162119891728780488306689435293543201853513891459117 0.005641274908756847701445094943124843575251063489664144458895386477
            0.8051919280077130393250085394056712679816800876407210534119535762 0.05394751507274243479739401407213032705474363330930717222921532692 0.01233892811437701231441267099014786743494762015063244855723939977 0.001346147058995723735374181037232892946971075716512909147418962660
            0.1950800264713013505773537206929965272469893646466276273688695050 0.5086494542314291941565938570273767590242879006744029656002913115 0.1303952042869710275850393642429225258449388968704711887228161896 0.007198123515896918409238429788663033327807132042799364466376666699
            0.01311935607495536471137827486411331834561223613646946241180908067 0.1432588490460768914704653937283964869932381702247373433754194349 0.6055597248027858527982888584583764382855313416627943437346857387 0.002151389529383200168909860977306948524587658163236135065559186178
            0.3322886425949623195590769843177778603662460257418981380874995668 0.08655624343532401514773294588779425581663154017455813632388901408 0.4579465499092617072006709103784824845625852385225285930549433060 0.006103285791486630920378963959573108833570980141088334337586701250
            0.7842012348223137562182558700754022037364328633095848089709972983 0.007421212231249031747416317837133908608517584958376878789045640484 0.006982577788332991942250047139947147665888793147180925292303965314 0.0003652029042575332560671591030611022172363570202939802806937228777
            0.2151019094043805436735433767549355120930786700972144732455109651 0.1943658719096279626628464477434482933130913270229236957576931132 0.3391414592235348694924467165169692087742647530025930865398437059 0.008556852441551942339458570283929227807099510275837429076761322463
            0.1007063747791455101701524809230101979693872929831898834640495716 0.1559909940890595729856945361874248381511313585527324344951687159 0.7373504127277439071526661561991082163369736734728495073722732295 0.001317084602786487756370180202206416005385773762697970803615880560
            0.05736482090101584454698612458875459645440606513132026861596135305 0.4355104025422004717018648421297176803226002395379280532021330546 0.2585388161942409265440716929671341913552993746244876943585329889 0.005299394686730561289057854916381971792975949015960431601591156684
            0.1145856431959906204510402909047380543062380392335317879952159983 0.008017505085353451831053399910641042630571291202842504293818192831 0.2450820039276393322157176031577319067020597492508002900463837270 0.001712274322172314918861926218841661639868542624900821257551095617
            ];
    case 19
        rule_type=1;
        switch rule_type
            case 1 % 390 points
                XYZW=[0.0908144555371566642253575166974738037584904448774740309219222234 0.5831575778485622641100412320882175962075989582581939841462170286 0.07389120504967601636949225810049061995397376846927424978698865658 0.00421031111150651188059819646845183784708836310766057356957336768
                    0.1313935896470820360018871751960735479099918193323723813899010485 0.3831749702390847481395046954101259931756476157773457884917985654 0.4732716629890030057161618431195267004378018870464479924885734114 0.002454188569402210883277362829610147470231183451515188501334230873
                    0.2260050066507794748853885120323645454982250134872686092777489804 0.04861707851372964484711664468062853635808860749335846323313152719 0.1655805756312699773093283695618626842512753508198747094769564633 0.003384370020173254390097560882712849708911683368661424701887557606
                    0.117564666502962802169161737025270156031472941652222070782505326 0.5334576262390838781512003011649890156762565465670567044294220573 0.008682523473735463182521625861556998224949430010302332137196080476 0.001625193740469099463386552408081055273998427930372193159080703922
                    0.3537361954243038705411299326332066937556793697230646043498649997 0.5032810368632936072583679583178930585901656146276191733921118092 0.1042780137628434543547049560635568476913766267692510265452081124 0.002777705628225237734025293994793214095450848305787370682469320259
                    0.1386853150187486514690980327017997833739001841543542412715389666 0.165312562055792606348806553863635626782603163686691738602186438 0.6465920519326435567996281568271150843093423096210446369996896009 0.003324609016234577286324924540081060229972504934162594485473151285
                    0.1401222551797005540414570141320990216153278571927041911053512733 0.7466868164402783914371280777198889928095875790685033378372957793 0.1044621680662114322962987059127380002839636772850915619627340774 0.001306823149120261182269420714893291726858319273725341700579684249
                    0.1072744744406235331362634648535230295379293651514286522278686419 0.1043168550460706638187642592281801115698976621927517994586125795 0.0537134418277085918954233300483717988023043558677211772535669308 0.003069925805527099594534465139601069793886447384280575834430668743
                    0.2308690587492580737714945660723176129101272490438744350090016735 0.2337930294622723819962126322517198353561238459459201094058131374 0.4956158113671677244117174088877558032295198985594993377187828076 0.003434370536185111755454628270446416900340926730929068921075281609
                    0.2521385351057191857925377255948905829569117408790491041661861128 0.006570060565159558469413956143095176649164516248931918439059592933 0.2116473649331681364296600282179288938130519116650248531650785923 0.001859943588675067893092590146433052294691684783111758541693593265
                    0.1209606115373286290141187363122547255762473635747144219113090245 0.2572416568070785256679505992609361269203393881683976322876338088 0.6108299860641801678182779204998248467655483608784923815130106819 0.001793427733076927843192286620544316595066145398890337631109994748
                    0.003804185231877843713922754816136287409872030151109163396826139318 0.4281688164850964438463778407746734226468247955381246453806773151 0.05256729933005487775782689855512415903878256792981805604511913272 0.0007241087593562885784074438016186039007034978006265549885249594137
                    0.5430976135782680673389338221037639383516763642066872794498654797 0.04721601693191351193340330645407490496666646226887150659526013059 0.3991914190543422279001050654579342400711188273556019599326257947 0.001322146066429581904135230735026888560186234416372150670819919881
                    0.01028778629378632360779860771684190392246300259880054428782189946 0.1397082383978054357280605866561331319239046436236421497633890783 0.2790621905504609887693337291558525756455975428766979960504386217 0.002023041240581856399483586786279886376963597656043164363006070599
                    0.2219395313402137142666985025979926803895059282993426633952583175 0.05988551638980271118918410821333706650088751913027311693863756864 0.5117432022688642986744784771200052186887069566096276664977635328 0.004630753953024772330059876825382160410107116234661235617876448088
                    0.3716492118680023211992679497709370599840943146966012213007978896 0.22373305153053724289896683515968821566519005183967550385350172 0.394090343483268979983799237984543558062714234398928863416776516 0.002624259683734234750625737143304036563057346150548285587850438874
                    0.1424123455624368131682892394576007386984487800996668437722089759 0.2680760575962285535456805481753418292240403285759782201077995664 0.002334483162463267670450812602502514078790088285127972355569346742 0.0007600098607422214064376633867428403539263357970394819448906638287
                    0.08160107827177880861126541343618318383266405383137078317596499479 0.01186224554129044742516883027911474559122144988141096406796752097 0.848444512367194274434290580201110293290957274226817291773447793 0.001027896074184272190046800832762571871230687964198838274464474117
                    0.4086341367623811831008481263296084890534767674110210955708220257 0.5321706813963030458253901056844666160486954848159439434714996627 0.007054031521576706321378827685037169939262696109648816839900395169 0.0007222277397761738080746567133415173750582136481947024832013749708
                    0.03688563351920537444120326937111537005665931331792599051106204086 0.6581508451034476521825200754547941706705733907293383031309251312 0.2556384467853013474884956902776075391846651588780701208650226511 0.0024876157531381172538646815047661184288034440357981626374145148
                    0.232944906842315564977498950131888627892083525708164767945660928 0.5058713830870367002641375331048906799630745143101061922435279032 0.2582387239395660527045112915529069215369228963674382065771547573 0.001127886864372668637841835490481137553980823052609913122476248849
                    0.1211123182613516507130562270223664415546798834396263307313812034 0.1881012339663666116994798069818874594873615698435228549104251069 0.1345604783997737482407119948811016124868629458338321382230898906 0.006054388554132758619245400911895997942856539554909284396764682904
                    0.06625321577775622470237592035238763436720463997455188165317576757 0.01075450415989921305577223243242797080663448530566870819467115908 0.0134550784911721067148023387063137534013437508671347452443719303 0.0004308458257940726307222000229364284868230575892534711305221420657
                    0.556081566762973362780445595477080385425834978543113638786953857 0.127433503799463639653180881661971806894826636231813317523858328 0.3056848193865337752655016789928910769448896870355623901919947626 0.00202767856229941395182104440247430954637115986952594421403681976
                    0.2326829475334662853564638281308421967187957157213412355091295883 0.04815168600269433817476224549654068035416222415801821110789731879 0.3967347667443137516275840065926605136276216018093053686741542316 0.00459418837993702367265984146973619252409067397331953250270452859
                    0.01365096515435366469264924585339863491298475915595125733031455663 0.3864748212870980920366820264077654917134882874287210411234583324 0.1201389704333990655552111608862881373648339460725436386943432947 0.002211404356392398286823001521568409308326480104786374936281419071
                    0.01163899648188401821554571461595473892270866240296609345774216433 0.279148749239774276896714557212213396210324689460990744890786016 0.06057939774830962847997487738991176796811322822052016708474140148 0.001737231648596367566101141058649545784888935797123320786199734457
                    0.01073619434358564035992290355595208970034334444085866448998726565 0.5505788761897649930742480153430525080423861196416425919842912271 0.07061142464984533713652810707599255385665605379772914206491249523 0.001728072966451716918702883260118099933799349940619025165664591868
                    0.01097646369818422312711046540832092668807537722392762610913195791 0.3817640697542036610548308288312669138982849404478599905322734162 0.5449762390861880573189729743285877371898521190124506069564338444 0.0015620052743934820308411151859795418700575822948374757144894389
                    0.009286172349604963134947770690846512305609344578791412764380870355 0.1202627522655843592677883563533998062844193429484027135873691535 0.01086422258239225962589700139483232344849695660809176319222153065 0.0003860315998947477760493667109069141751821329933916409940010059805
                    0.429247470712772439372664387978489329512117592184815663193458975 0.1185218882776840430699259970275267201420569541013245297279627215 0.4406104712534799904558952368419985694990870285279703241337388254 0.002052414441201119840485390625160704064828392535146782031189143891
                    0.04090867859482736132780395445671923929533783037150255556213426449 0.6667474411503127244737971128497711742021361687724790641146428513 0.00234348805437690529200442870024424206695168417320937836891541303 0.0006657836160313547614941331105365904634183528528968005401456110314
                    0.01605833123690937628510348676394319129058970164741518322808000553 0.007865520096778605035859068153145650734716348837023149873764143613 0.9119263605759495743897327960571238971273338854760080969055061083 0.000372367331629047412175265507069738099056446679832342080789532046
                    0.06636042518603538242706319934385082538099867814831465661911495558 0.5699045992406908828556608794001684862559895324883601360347210262 0.357226644686744211687154134222450307497117004328467978414526637 0.001026478580242318930910116381877087167699490543190621730332222588
                    0.8360412404030199893087565804900427996623248326906069373738765486 0.1380885042980192674689351094052604785819426635164824649239123106 0.01273872585902312548725680816911671955364233854021102936660478882 0.0006372375637563300954918702028608444599872775929063770494034370722
                    0.9627177775795950579722206739414061880075117978132153075483147942 0.01054665849543661249894389034115653922289824987152813212697818905 0.01304636650200600994760809104297626922087737855332956672662875864 0.0001847887151790688786205531898358126169331099467287989459100239474
                    0.05361685685004705333437825243520904811254262272922458687475160265 0.3603370535569422839968001910211605654392376217344881852389771299 0.3648709513321332223947061949083807678154670463871980839651847121 0.005564685133937438262718788214137507986977607715663250666604427942
                    0.1756723639129912490942501907104421540369606143116124524232133783 0.802043871429816498567183351220631799462941230004689071504083896 0.004007463865110240092782445334881933200722803107486278631141692684 0.0003646468645253081819532053399037598968861816239830247965110755533
                    0.2701069809524832921070529552091691208516178826297147551346673182 0.4685066572531891514146334878880652979691887493337669111893069284 0.2238077820844905029919907309925279733478957977356203833517113566 0.004793013475371931066010622555099726824978823450370100138333095112
                    0.01066647701112615230827600956475532582035631924777309332997482007 0.01032445629273091388527603276160034889073385211833617717908418738 0.3085192151680197784446663196003734107605681424779681866633879518 0.0006189600147212258369987872678739627333461641916022883936467659875
                    0.2393066303366749416928462448395137095583230529443618893333072059 0.6160573588076363117633984827031600099476089446403066869167909905 0.06301988010002282437209251169639509326410139843613200534228864355 0.003912039066554202556958798477291656604521795746092464984955668465
                    0.054523089639882332064082936755667293191258034933423569726348811 0.5670853919729937890284323100519503440113993526516629525375479715 0.2404704093543406525284999896638727277773226926688697120381819741 0.003905807184189168633440429161338482604115905402110068408833071722
                    0.5536262487238270043285929173780835812893583457192461533050180338 0.181163097187378043348056325552481705445991405925601738184115243 0.1100675478766255195648980400505302317910063619750321129001108447 0.004417490252501384218348363738875095979663424706541272216409964024
                    0.004127141515635644744833219882773055981505380551398095241043227925 0.2802501184396295417800720652617838976705855180469651726309563328 0.1662921164713025756003196036447011241906139814438750355179746755 0.001056542553143356025764325777743469353050166699067354103574665701
                    0.5022415989706138254359100060319967879831064119687011378845069017 0.4221681597220564585125219268984819780530425511491033515021588047 0.07304237484345497978910581257497022273131698377319571257152286792 0.0006417096263222754506439920474492011115539914772527806349813032824
                    0.1367236073773629621115398857979554956707935540857577307406593457 0.03220652149458000898260762599988537329944524985426637752640620106 0.7303022869757954759411282047433774750552399376242722116453168946 0.002145028649362862354731215269876978729077632140680569395316413705
                    0.2618146176288455325269192836545489610173808995300207742096654717 0.06172138245502488271072927000901551504655685649778769662096920481 0.6618356128875957679077178860674123998246524891185548000041934462 0.00172991250352769637687005310617266706056103825182848679298986686
                    0.05038289196017814532597108631237056446521162713910809163202542886 0.1230992872317999531813283954700594566132582357502397515845842192 0.3359107222438124534334274916853858512812562988875324185222383916 0.003738086760515378354048417492320093726650713878817526073955251966
                    0.02765331396909763916826324842289854054031841485655786227267499247 0.1274003073382064321195436105237124545910167201354970224866793969 0.5424148118692901439608771050120028222660234732380794622457207146 0.003295284608812557005838888569630500092696467657254137464100375631
                    0.01470638026621560337306364338031504779927406795445420912781227699 0.008271835931797513379017000283168547125951235408728588505236219926 0.07499882700020075503425323665819869002053291917166918003649594647 0.000400607032817672747759653195373977699961771020846009915071256795
                    0.01075789050297866191187270799585065041204610834600375888327707476 0.01472319055216183166486988924531514019787995335753838934854238312 0.9618658438718151192984961417301774734328806087672801236559237974 0.0001994474999630256662930671272047074907923525504298548244391835406
                    0.3303939566817408720860693896009185106553864257033670062283856227 0.4429407623971507150932493384251831450595902001769497649683772682 0.1258307454201013374476721816183831673616146577652307249080161389 0.006564324484034750636481345417631976964081793249131384564244482439
                    0.6424473493718026702637416114330065634356381939088077497046553771 0.2892916097715229381407655169758978942372394724611410646167016729 0.06518183032482186630466448246085251198043271546600951969916188592 0.000739366990852722850744184618371317496445548012377227636175587402
                    0.01917383356484140583485067326166227271911854665823030058336928832 0.04913596292162260707898526567179172292488680021116118695073440719 0.005269502497268644165069675600413606403503077161419406679559987096 0.000255129632502713043998872271713960289432124031164024738710999759
                    0.01063205224643291562855991133136201300401374190530815884608945298 0.2839306942449028086255450537523220887324939367134286419314738026 0.6910149869578767705799492285250269315011968067648106185793097669 0.000655991727356525020197831079005705657807645903891683069347563063
                    0.09996306818075289596557918107826203294010067142447976550478583616 0.2005538217726183488136251925854017039369118737403107406539313602 0.5709298157946934440498049227641068174188323663146521444873860229 0.005355736742651426346556067394102675883370838319749647161420964005
                    0.04515686913411789720220454451616997874450312880845144668897130208 0.7905981193869606120624587471694459959392995492639988745164421245 0.05140338264452380813841151058178409320180775823258048291677262056 0.001886765168568716517413813169750344882737765267907839908078957289
                    0.2149622061066821674644790207266740236765136043782916474984007372 0.1187911897664468199698134069235006122527105329684896207742109247 0.6174507452357103915007835409656320583807872524690418169494238171 0.002847016774235930171263114237252329722292574150694570200276843977
                    0.1727525156409756713575419946897081640247981704140694747086691853 0.7421677217317140136667994867001465113097469435850971277122939801 0.04846954632614708038168052415244170991111011171093959668812616201 0.001892556804368579005090758572956968418245058899191509160736103882
                    0.5266909876643921739631786195270819597848410026867617060827373598 0.04783757543485129301273563720699351744770354040211574912302261563 0.006260328899369644791613970228401118059132369296160967094149161368 0.0009402020776977847830356634181625713624465171249513621792239260473
                    0.06134343116815007550038585575553124124047897191723140951374278602 0.1503757613096279211083877909952635742835231947134661225464998891 0.7748595302547035901783848278141651261380912961346461970702958464 0.001632569567143983490311340515857777731164915235885142100911107982
                    0.03696289589685608355237111677898676160796346557316766271155206576 0.01119753172038038257131988567967234511268328952283269872902926645 0.8070228395977467266991783675479401460796193734729428222375998148 0.0009581556001655249042073375287377205610609956075902952329855352584
                    0.5924280862619358454130769648712464344979130971796938806923056252 0.0681922948652099706691259103674892214722211024554158584195175252 0.1187225618681488580691811779388790593937039761241109944169548817 0.00440216139281407133903003351893905426204259526251700360966253333
                    0.3231299413759054533279351704307566268123231236536218216789076976 0.2715316687867105592086028855210486258119413475872605452640606084 0.009676420933825212040584347283137574546464717502878094343009426717 0.001873787613438779038114137787270036068123795450100712270651350287
                    0.2041276180494299705280110458560082868615001250509259487890700908 0.09037473144785797204741097657005705154641921553611242013599877626 0.5794683978069701963708006893196195318715642287476783027792870882 0.003867000154306669420470961276728673702325464188482701602477734934
                    0.4305618064470565929551465637569498268912600333376713152799303233 0.5479949140612848738906382706918116678842716899349174841519198357 0.009039834571246306834596391374981255083933559933371737388933030132 0.000625988746306532293851226105829551051421216008359944491759330694
                    0.5755528977272409249095734068515970522608894974080928850384509255 0.398440635940284007722116645970832949322286373895785800739174053 0.01343034561650755432118330613518439145792186889338616341102133808 0.0009912079257418126584959352453600211155147502172419450866048375734
                    0.1445169279611384371255957980483127545934773126066089746188068047 0.5021759565954679417449431989071461025532871826897218615373120647 0.3402634994303053145628863987293672941855745832345880966059096434 0.002254010021000655160878614336340723245700676093400774347314393691
                    0.003907695743084102349124874954503496939617365010148881748223415742 0.1264863194361007434581381586263530274165846406562609778096772576 0.166095485104048870575502168398120256998420232569792937146955678 0.0008494139859657969069990796820165109299939357581037134176288684666
                    0.4101618168834533171004748750478232436795161119907915126836138242 0.06481929584283567636829849255506477902639639040270725561208462848 0.0397691096175723420227283166198735680596060767790236454520495537 0.002900611716359907175459304988027194780084541664331811980766560414
                    0.3689160156163590789899793387688692092490668799574245333436645858 0.2170784657147050880541004892372030443204139858109137786918740526 0.2305914226033864781800353006113056561862027108257524654046644337 0.007021859148263614035584884856757913291989204427298979274985817262
                    0.5613526240759389226734389297707067280095005561168949874695962206 0.0395507231321306079513300572458060820677686837726609778613760438 0.04218638834505347298038505692917477884871422858372803610363638444 0.002362840910400865406171930731320575321884116986929427334982192555
                    0.8775119662983445479369986896771792428655191730486767826398775351 0.06488827622449777728520408801775528687500719651044352163135095332 0.05189164346965153437405030300389936344882926243566113154801396851 0.0004431282236500917775321986572843061614819984160956060707905302593
                    0.2544264566669925876766311307236025030425365957849990900242628802 0.4270373676673947388792060913688667615735420913610537899581910511 0.1168066993537840405707739515577762541473628595253343132773211689 0.006791449607731978648249349526049641860333372366763682871911641421
                    0.004202295270963138059921548545855704015495612967280170905710974258 0.6783025781933561982622478827036116776651688662980634159631136995 0.2771276617541717417888808007405021191846373115444253676187708967 0.0006439882637198862737337084256197664369820739136010902710111862301
                    0.5777641598219506994720955599103654827251262912522268148042138285 0.006949268055204985474856796229813095203362626491963700054757836281 0.01380025958695740385016484574671878594672789019967358943874852369 0.0005494177740732449832642390747093344596431606092925854044829295605
                    0.7695989405721116561540015712375656167067820631917952768773716298 0.1547684992838462714516462249604161787586394486488554432034468535 0.06475967923885185205747263981239765282653984444574468535009888318 0.001284128524435536868945606507017469235676096047132662429458518877
                    0.01020455166781949902178018304169300820426569155153954874912333233 0.2346807833532764838199772561352802895851905104108080506647278595 0.01244855305463897739998362212267996590886621546430468407925148756 0.0006587826406382351652213792095518351315591623404397417430368945008
                    0.1430904220891698949359683540495678163902485915218339035697104902 0.03217646286092577047788763128545157697705102262338860869752302638 0.6317594201854962363744642404482414183824577046717298961068921233 0.002833340087023183048410912380480661269909315489763198090021161107
                    0.1455746255707541182495639723565302861398671923771049786967281499 0.01247367062468125352183610106171241269842221625249067914409060555 0.8305551059085127503838817624195315995471814744354737067558859952 0.0005908263537564663220505074085201234209316400898905783115264643339
                    0.5881035534522043646677176520850561354059801086685026334587659537 0.00906757479529889930172193374002942283866806128061981465167461181 0.391218352108773403804951284667317725851723564236946669725015543 0.0005806400984887383322732773350537132168113014663764354243681283723
                    0.2356015620949915013980699049948616703540019853900538343463621536 0.0102971212982209122849336133588966322676702849854084858862491207 0.6160261950296403809490923842838064436670048203019835766614407413 0.002041509719739372926035933710748637700719350786016976874543052325
                    0.05340822130120471109793238420693174887735939182233577939806650132 0.1285944156957477935832648760681592265136607527939454477749762446 0.740241386643427977735807638399222947749256751734926152555791849 0.002830399500439217866515655676960132075569762168510542044354461997
                    0.7519483783022459814317562143792600757551519692528387794256715964 0.1355419818297502305462057177675274955475242851165823070797161383 0.04328257196644489779088061557950375461132884329828390859768058067 0.002251366673528062372286140684727853951745715210244701111090758445
                    0.05067812303874899557917765042915349213980413864579300555087103998 0.1521854733174534488033413359101284018762762382998956584188960919 0.01028460141757334971434896176192415610153775577061686022078860837 0.001038967523546607134348907645854102141834325670640999910871247042
                    0.05721988211749053143560511328861455545143734016056924944835166502 0.01054580406306674590137157716354888967473499698917304233989917821 0.9209087823562998247653859216013913246487395303531406977606797685 0.0003220136581762890096206470371302019191865507400703103356509803087
                    0.3439121191896588669574934583748464823614017206421535365472339316 0.5255410862979661276211025059203166818126860999326271176787128051 0.007413333084493217286408236848249677621415681842555298504506798815 0.001596075654056352486658535181769590986551748944970714472933657763
                    0.01526127541845253137478972769554763617990790956059622473543453783 0.2793356631607934137129978947167448772103023973037311246124064032 0.4788731467307957696909770376523121417609933071756512896451063526 0.002742597985219827893638408678866371558679984967308822618029861643
                    0.003354113857577333685938682082492600619316305452652173539570826204 0.6877453199136533624863335999590773582435181367483816504882518328 0.01393895629104991208844700269166912549726067509844876907511165829 0.0004020931599613593930741458809142310097483082060326069512320283797
                    0.5810617009617965650003489967761972790431001737928578618397338968 0.2648027665341731415526046945201158872275750391280980884917232419 0.01093419824027396035503010791349589537562386295801229312633123829 0.002117118825991986883222214372331192405845104426095904142676186466
                    0.01526012585250785259918049602691702058173139651723297964850793364 0.8764569640215679978461952043378158853096808062712130817453480374 0.06330937972507674586459353874747087043233924750647483747125719309 0.0009356650995841976884724885265973879535321116807356141734946346255
                    0.05253453441842915190994363230951690071130543202241762193412606852 0.04246503075013440206881638643553923875115808439198510765673570092 0.6399596901550890136840027858458133648410181865758934616834032536 0.002217582412454782160563306461836430884691266519497480454021608653
                    0.3847332958024613720104907385994580100564323858063364362947488391 0.1608316485822216710490829714094063135662187001602021152902738655 0.006008174971004378264343396509983621094929329027422652694705190099 0.001505525252514429051600338152050914682460152429087870513084995007
                    0.01127176361983810654331183834542985504855824577005800904250997878 0.537608818342354194399347152640754028101820743546788571354621446 0.01159939695970161749837186890158067779977953596398558970376496096 0.0007888762767397335564835400967758963327961055907880209318306537932
                    0.7925636996956552974073967421127096499374065370014065227693975804 0.06379342979495043057412849183260180661357749497059655444408874919 0.133071324575927490530957069516222666348335886786513853481926271 0.001165365671129393327787787349898951779911580309183405586536016039
                    0.01160045081765102855251723850570681467242699685113870878795950607 0.5576273128463574638168877931680174603107925593341380304699146891 0.4166338270702516742119870129626323721094881947939396105036007042 0.0009068070084241923689099998743562452796396840383966508017252808868
                    0.04968962512649946121850405916543901039897502840943249087001919605 0.01348296234525366408973275320962452413843399501821972321618222862 0.1593555033862671964234780957689374963401535984336084238581297014 0.00133020287917563697149673569679201375303223107190440138223173971
                    0.08697860806964587524526179667876774880486410519879815144023507477 0.09884285742731086861430697206996481924236581095613234135879405543 0.5901071525533767520467002894397540599902413274384707077763269959 0.004242774761742867564796531766927332068975924798713322615768089973
                    0.3162953042545895979455935915070569331401818526720409096761704469 0.2467845989222561916676475365352949173143700528777869674943982983 0.3827748162620466140422654364713165940020233647097123099624207883 0.004258489416879791113494553204632050403626911332079385597076804169
                    0.05631741770709184633291457573463266111785308398579489808831727373 0.7979500621842867069632729530529328848803237621678932863588169489 0.1339826872340595750852918522730978308530371796583176721442339859 0.001052310461494388175674212994617996786170416195713830994790805392
                    0.004909668869413152220834064125839712963359834040806504726662199525 0.8045814717621933643665479685423222091854090959244540257578723318 0.06300025309385864984242633876380499938026642564008935307707136624 0.0007255368479325727172907047684655138738618786371144206957588445295
                    0.016663420068483608690171928053784914169384609944790743670139143 0.3897656014019497532123129436806493345572245460052658431779985462 0.2148223437382892442500725146954827477526369654333196857627488143 0.003213124310705528141786541788908172198098953399677986554510209437
                    0.8156498120103193291111185689522468738406036999443587091246474623 0.01277729102053344043696090738316268372634215898302043712336996852 0.01144030140451273803220044269030041038873226756077791244494339065 0.0006192301434797405948085264072939911778456046090791853458023420602
                    0.008175319253387582470033862632634898776293872205315497357260538121 0.5802076425326772068884488637338029665261422078658583082486832639 0.2995101784964188891688180530013935617329345901025409818574680972 0.001439709369907754105388577219678873917594974668832776804391347435
                    0.1877244772828194569714115831576061496369067198106253638952344634 0.5466210362050983979803073024300329304616040632411703994125664132 0.1443738642197543480440786871473580830449175273382059771113654139 0.005744235858407219066428612574576729216649874526099920569469443356
                    0.01306764827832688066154958214139190150413608964250363699279802548 0.0741520874151002036883557985296256629099059029556743137143062365 0.9024277850946485962783830865591808278077453747132374382367160146 0.0004301320678647936142062375705518332862407988681447989018767475714
                    0.4344228029914099979380610248365708470983770732938639448969192324 0.0127817183436076370409773874314674251983583759121402121642579693 0.01102621401093054780632536267648442867772457436551158977905337445 0.0007853089125961691604548650312576105807578680648875791850383479359
                    0.146012790597379938056159388564672304774501745705734691900386013 0.5234854686925638801801801844794237251428769556706932609564081724 0.2747615870597359463606563460896006313694413640442542875767537672 0.003764527898113499395725690877903865443539995056202278311993936398
                    0.01210880110110308263891254074258938059896514040570778854685053097 0.2428028983790198643083180617988559856512083390452934282426525905 0.6838649165305065593946194869833202430277566607801824950668204607 0.001428674071262272155732509169899645676504679471112709693167319009
                    0.3077403902959244245424859722882662383623395839610881563083450244 0.005896426170727903475859621450078137691389919806589902915559793347 0.01957032478444183534882017782616477195838293246935107438002902611 0.00053111716838712983851206607274850333143639048513700297556390304
                    0.03497767752125661663711506813205133919966776109299158168259232359 0.5708688923117628281563208981972769066894590531246326947064340671 0.1319111606114237469485078936947933559776054649804867020382576007 0.003504143991089031204694408989969203794631109914420141299489632043
                    0.00185360455797087879451714542834791987432464602779999301614796077 0.6682366568276580668527505877178580449642670452476758795042682361 0.1215797119886440817442794476386628844568356271646925679916128438 0.0006975873475795729697671239114625148784700410625920288455869825641
                    0.7648759735767660221347446998156123353336117036463668627500438198 0.0388281497153317861153991112892467950645302512160138551077071033 0.1349871282299730656639190092072285534986816816254031135802387032 0.002258912342208729716633345717116350808269196316966428628958085235
                    0.1427830648294019442272292555339369461402574618225716390559007121 0.004192312901429364785943113447257004131894724735347035622089741119 0.3253233929897421593513347140007245720016116374460261200641861972 0.00121357697256488651187600652280871421161000370651403892735670334
                    0.2227877564075424428830904096582132820156621710505306248075036876 0.2467902597313501457972073311366838207462015025728177489244378035 0.521636684380051912620128081687878342248071456017787709115863314 0.001682837957444392182024875379994452957512130777556143025458607697
                    0.435372197791800927088855272994725743232112010947050036739089947 0.3781117082840676600858408966501409494389271689137668523978177108 0.01019901731950109381709707978190038947691676004502120483379309152 0.002412232914639553486856677217103422574134852100471281888548123577
                    0.2443730286241659494270529597607154951756961596632924541215510665 0.6109579124282627697187242394266771366059190136754241988907224571 0.1388369481171687757242272542404525643303771900256877716243982012 0.001093124126306832566923732722209543338247186115510538921403742657
                    0.9143778238066723853966530578256119091710811716987521214594423519 0.01269665079993904067554069720103508558092314198744660063097960087 0.06410844240003542923982648713537805288251277356867722041868210258 0.0003149912060202061086413792684806808914092964519700387362820301886
                    0.3338285892393951981032553928442424087875211294574026511010031133 0.2071995681241361094736887591765445838784689189273508203427703935 0.3373256286695794385204126247074767882679208870411218275181355061 0.00670351386049862035891810614937149188389377465039830422896771555
                    0.4827728463359550813383875505792205047673109175176738580880047417 0.01311175253233433217115493238929056059308151459187608376619620942 0.2306972136418186826861405414797346050005669678485336991220153412 0.002879050271825334857991026195321407642317169524193159704066235332
                    0.01553225418944945815252799007644163932190111547820233469250562578 0.685513555255988773016128436419022451022275949330786654514803679 0.05378094387935549124541922983284212591155096894988578579745695401 0.001539035055724715052278644622606303097883718505375614879170276839
                    0.6513433381026430298231933445315055664926843142239239052590616432 0.07136115986020584439285395568288475978063789874515370127712325476 0.008986696548730878433288955191582207115084631433546336641449039283 0.001305319840219148840156463010465668343234191833409553240097032392
                    0.07356710760299001413997857062298866806930178291850830241766756159 0.9034157679182761936536291991993789245695964321870479475905027327 0.009484005732726046776658474850334432185588047756056816653737220129 0.0004135647978285619408053637401941713906779165898551434074696094045
                    0.2693999397165189955520351899810093761850607262639020532819105121 0.04157845476495594612907804363930247613369291257393042143845823998 0.04144201998155271268765406527359927350407332313573402180660804971 0.002738816707117116055778695289016352514152990649882694075007313813
                    0.0120932548553511390919497670074848914762376675963814711339423169 0.1333491266382964694710353627796889899808436668071569379577429915 0.8053077391448439638249394106762210616615170452795392200784680241 0.00104686494693773161779484964187783618009747298365585209647233703
                    0.138082570903541596030113687170923694300422633110341680028846636 0.03445314139560157876391696385283602243138783772572233286550789779 0.04496283202594149203888567352818267515544654762231988448993688567 0.001736472639700881307028930313821056073230307019882370405115730024
                    0.05968263439399616853246784848024250793943346571806064666370662061 0.2423571853750921071482005292967325878209171788561742213191213107 0.6471933555098779545549242749330679437205464690489565344459404614 0.003171939399754208616746149386246324032436180977642377096386917545
                    0.01494626261283958856303982819081857504544045703360037630594376576 0.6732437920389461108431682280161221444288357128054848311275818475 0.1860659206737642038480457837147832246995074556872278012581527881 0.001714394819757318505984768397328581836589062850206069185808811762
                    0.4210135695787407001071812805998889606685559940548353501004692983 0.006451645712925178704420122614488771560149723635587954724970990942 0.1576054282796144973460005683467730531858491192825992752830296733 0.00176750598976213903866323581380036233989925429775400662285443154
                    0.5493886017149505530124581734266672903394458289728634689801891971 0.1217490679982928806900778207454377615551358568002532759636453729 0.0577335904250166118897671028703864890464873265257589544337394561 0.003889550153253573854730338734991178165916223902823643585103021231
                    0.2525505128989815362118214941987406771100301014339869817072534583 0.1340887699549145913950339910370189859773319407329853400243021274 0.01390092885551073313005005574916384060451296984400967171511611741 0.002338735596323714706749535915896937039207986190686943658594070227
                    0.6720067536676764030929273268213775916273071724983610847693694885 0.03775308244683484860640482344537866014298427403087256917796807562 0.2432553619924354636771509132141132231490568937315567247088385536 0.001954100057829109680522409740975670931432439206882590311263090333
                    0.5341006994922555369579973318421061830874920240946991785831866994 0.005826101255960438218782635794737143764884525008441132134446562134 0.06673505960116641609645777083987573611523957704973112344511393837 0.0009761253541640386106803840420221337074440097732001088501601800109
                    0.1988967447171603738687965136558209785656951177743047145808208603 0.006937683300858515851668545872240583362723012653386715278428328012 0.7401674219478047246690919508309273162355718506976321884218307188 0.0009517058139265367878137571535606629346661311719885755652469959803
                    0.04583327836150241424372387323614862613538852444538152783835015215 0.5598758560099194915179440454040826656339249665328725105500570028 0.02856073439030903608356973186091466960811098743771391683745425636 0.002177173276003606373615195583545547668458754393346455014877711046
                    0.6633731368122135036527746925492683601255699128068107473132957434 0.2767388654322291026509396135663882982173350510807333013649833806 0.007682445053064459687893657808890196122408066693705612780788369928 0.001050636807024803938434976323194809148495479386833460328377406759
                    0.5228227663428694665888053843138012582213642741043878508857076159 0.3777052024448570785488950005169170133114701965710133694747308056 0.0631827984894750174539584381125379456296611302259435576181987333 0.003368763266920544104131555314361075328983090851653354531007906889
                    0.5195002719073971689458898133199964713255753065705204328174394161 0.0503753957172111525627981457188879169216147402065485126935641055 0.2359016985695429953978694648426002003514184382016119041370323525 0.003514416400091137197101718561716048593348199145953883561852035301
                    0.3376824755160522883744878458608669026923915624832256997151768136 0.3383545226397203852670102013736359823062122700815958386860083938 0.2305350243050937286651777343942406500579774890843794412127795596 0.007182152892997749954522767137215884338462739422467851708239507335
                    0.004464656789454562643488865275408315912700087644711598715836017015 0.0597290886614659856629945496661937090160929009677042561138799595 0.3713232745092351734451513874326355223669457716831537654902151999 0.0008555148948673493029886722271845406793149038628113771306177472473
                    0.365784841470591068223791023077267473675540802461746543886598969 0.4827355818898511375773539835950131775539398724010465889726653086 0.1436961685528771190446719252470235417007545046576290890288753278 0.001702786667713597103087525509769102386298318368145134805118561394
                    0.2746352690296337978881445916083696097046437067621805552752818674 0.0118843225844164358375236124961468975858433207765604333585690954 0.7056892382191575779131348394046468867471548702608965010569500299 0.0005732916639437206245956070842015115674615580494446443717592328261
                    0.1295880358831451015559705509173651200628434274351832838109407278 0.7919352019140043828921228800807688041994100432184564483265127288 0.01163354684653859015817562252905269894382822390696582086956299135 0.001257094595829612389802473868770029469098489944900929923719208343
                    0.0583543808186262105013462067213532508886929562681010482451079462 0.5242791751335485581115141130533519541662865534756404138644040991 0.3767071258156596369169598113382125129559833949033439405130616846 0.002932343088950718185441838224397533276968278898101389774703366334
                    0.6579327964208752691463952478359463277224921128456281899016359006 0.1367875319176175819444332806801495715943253632142138465489373928 0.04625809613676362378353384164657805343973063678205669256426538765 0.003655281646965675163505784602672790899772221811875497669421820226
                    0.01362070701929058217411561500423566608913339479063274691623061471 0.06392363672614566430818064364233315852856873169956475598297198427 0.78341742049199066078356310387265473854995257154165671517993182 0.001286620970614536072381055193459841259226711945840982428251959286
                    0.06424363983117369225559818543653694868845276721051853537833722064 0.1462109591212825250798798078262697572384421417906082497636046151 0.1147049141781003713811908871559265999725275654363855123916447278 0.002891807131485478549328822593279742306775404247809354624622684484
                    0.2006769240324652222093455356098853679416062664281265709917128369 0.003069637134962655090277824976754212110307504602446023953371863917 0.04497592985364970937869482826885732223996442888482306916415420865 0.0005154169144330322568218044483913296349626329896065299424485738835
                    0.186675896923528118847842159445420589964729753465690326432484293 0.07372734006564273219175610974806004928447139144247521354556792441 0.01105177209012419003362431690281721648527008629259639870954329482 0.001451312676372114073100817283466206451691090587050249441619485489
                    0.05087331235614990546666839610432394271967178587752158297841460821 0.02408595610052985843030915844511249131087987607964499894813861967 0.05795334147215779144822113154314594724990832390786403587856517566 0.0008644100446028230419993376832040770298617798871762092198116471451
                    0.005035256317191821052826273731795814919648121601440609187654368647 0.778987687328750821216557049995065891582293340418709144513211637 0.1477422126043424355547394982702431633984106774901832858381772987 0.0007481282059433074054741897370791259008150153607760619248747251161
                    0.2745175222789010322993236989785936763612316720614466745253880243 0.666021917730218935479780527305682703420717990238886354044102063 0.01215885352590737035893490163561325706905675149395760982650412329 0.001522513466679915010362157873185199300140892113375464351380886581
                    0.4933313096422240923166475212209007988435277317630817059473929906 0.1172651323102230898832678196389710827480459055000110004801671348 0.01414986333559518751158676233124436594009029868071972068011488954 0.002151213366008324232142037495427282570300053413728612443582185268
                    0.06481372434771869068743822903562390850971716792154603286644504697 0.05794618943647527772061014443930931661344793341666800092528978828 0.8665005993607454171099167143407661797103443038870792351778114468 0.0007914980808098582868418145878818443996598486217291643151948603861
                    0.05616438158315089152183233946737215495475089257677653372105070853 0.008806993773022533063300249656598691233612681818911596003806940627 0.2750165663935692833330545738995641162211047021679841704010966764 0.001181445771935395346971691149065296073315623154192867077715069062
                    0.2184889750756446572747589723257024180672200149661825435888871807 0.3285977418326434894643158422588440790849727987967776969664686537 0.1387282702899552866393077787588168815604009855023894960752331474 0.007086565048049684746787305998262234681339442898580065132722393583
                    0.1387882929797029688119009279132251992697508112915642718471857775 0.621761475808116567455561853362950177911768819079566954490323228 0.0134433302038397083160850662323934217832729730014607226616650327 0.002211651279504756565417487715757725415299179874209425937695545231
                    0.2698618054727722977448713513361634278668074639480018552577744268 0.02360468314519292993239466731882716441906921358166008750875107533 0.001813729896507027967372761028300101513002088459997708955949216639 0.0003757429771959854058495863955020821377490110180333814639334860203
                    0.00462502150456576180530202692147438300951884323302788317785718548 0.9126905614056253863908671643703067290733025338547998967570675452 0.01265554394438087247779204110767005284722336907315726013915539112 0.0002417302116011035708228236043301918544918448111902122401937248404
                    0.01331588988847041909911243810759052957546370691681306801039097187 0.0544394340037422547441702666166616439005178487857121861144300327 0.2364963727114361290975982284679505292735713116744478382098016807 0.001593867289425084450068381378411169855518489197384065475223906221
                    0.37589606218874838216871563112027213835995360440818320717107053 0.004796777868398031848427813948525408368608213154745680993901794772 0.5044203087479198301039311455269591748165226964269140598347754499 0.001145527351261638223013112483853192438405456554106332983971805203
                    0.1330524415033841105463132881608759595768911685348399168961263638 0.4413353810539593182686876834258601103577937791339892544175281519 0.1237021340397288003501821810152301364849070411061807082485482606 0.005575679734432781882668101616164694712371703006385919320499059138
                    0.6834946897060393682394815580766255354778853868524103820107081262 0.05587894699848354967759242333432823513668833676886627968566399108 0.2537877249133676104532790406489788535199414462250659620707009774 0.001016858808919886161924561965400384373584321583939107421097481514
                    0.007735058275486813087263567552496993439519330845010137300241054459 0.1428257585122364090867840527048260559165824693060445168477976871 0.06339378119168890409475388967445386479008537439601359622311414234 0.001034809694265337586391927384699612661433320036898860550371396555
                    0.003961030574951322804854529725312440316633799218288318876148227952 0.1430648030260829661903668090887225271082470611321995711166124909 0.728984829335566649886017110467375576582551706751237638207978764 0.0008130984755495598982921330006535557095073591674620152356376119471
                    0.005245366589380017493557979456499074855948113734877094574192790771 0.2830799072975236398204378544542704560952985299412999658369836391 0.5795773187108055343203024673695938298347896269092334628985543684 0.001063020752151778494781077468281705468633232249187210007605710635
                    0.03286727682583144391167532285111861022306040427572943407508230744 0.08174150095315630914840742483074817764388654109879342163800762113 0.1230087299515923138230600473885202084063053143614713067547100195 0.001910562225276708663676496123267764411356963863655976036234548502
                    0.01197416783692382810981211494091144064026878092255699467103384734 0.009786796572550695626191099943397399863121527409094322354660854135 0.01443981276382287320944662422285659382657135866694119215400225471 0.0001710807320741127178749278235762198482263647575855342082370166341
                    0.2352994494290767338180588976302905608274152914654985646187469626 0.2626771417765120559429707917654266059714452762633191913614560678 0.03520416575357772515362704659886576839182981806870508804469437743 0.003235912162915476278234159215497958476598777620355960331209412778
                    0.3396186358381542674120347593989562410064482954416378568211512457 0.3588743423894672882255397110932010338080921890128633647508351791 0.01197372701862810806450585442996100880752108827191974323002887504 0.002431076643314102972572662070400917109678239499511049891323340481
                    0.03573897044234978574138150408112644338250520988295968817436249707 0.7011290403590731463355020194270780563939222488225427049054400016 0.1083556928707559173795609133104177706928366402884115274751107694 0.002337551829305719376785923008694712859468940731997806173547568771
                    0.1507717844791458115884383166822906636218449204607404227854904853 0.06627617226276735401518494841929841637571740822871345250479979962 0.7803404374004651233247337973000445044562916049462205027582961296 0.0005347777666714168362046344838264128552064864503645466750794752455
                    0.7780963295306838461844375580735694475253812284129991204701552135 0.05175158982186017261054619866765135677012813884802974680806666913 0.0527586318883552806218608136548572421773545692176956130615092625 0.002097256323874253707370971516223919821381283043460176352842740669
                    0.4345366920658008072941945812973268825018491813416039627809338534 0.009276416141911464817618519212252242141715710076425136824969886229 0.5461722836756865299223580580013428590542677333938663278537289031 0.0006012419941064273948366430537781607525108727722862043654219189747
                    0.01231654643472919977851575837270120179730879831703069092687986599 0.01238044294582324153451792096371779468078570240933323636910680622 0.4511473673622270440084008562049441773053851640027595734962347635 0.0008573811029150458155850757460201035452935701234614836697996420118
                    0.8703776668835427144654247887034848195913050285681613420054158914 0.01002480308570168331328993184797761184615606616340210547565148088 0.06863483523706760984665684299575786314074762692524106157957356492 0.0007232801059780583395930944352973135143504279499112569911761017777
                    0.02993946111143802942794820416911522692898955339464120496595284532 0.2104683479718685768861362475093725327634316200534745380006738911 0.6158521098800938397057563880869015283760396272271320925919906968 0.002868674863591587451031195376247877775644893202525636341488321653
                    0.464917195211445592825974824016190433682469317734283636422085383 0.0704929785389419583187100061985604410219601209680658448464814166 0.1840489787336550031582908490023136478280707093665314511734360433 0.004305032020404793166807139722161849660797698184340897753586052746
                    0.05863756058041736583878659688853998215759479210461589474061908278 0.683056849852100898550552034378239556552394715299756828236944871 0.03361922479414371517996237002788663397422891993148889213085895946 0.002060354464337196351963720225720605983118606332726186813075428299
                    0.1203380223488199141329601526787290797584353453999267818780099268 0.704085310560317106457676773378043679353949816600293619926586473 0.05690419176812336496936229399849846764495728121140314769478386608 0.003680809004833598134085632853116754298632923696498316738664290614
                    0.1405211608711282281818884892499581185222616646299972707425882226 0.002209969247889380546091401166146147921726424861237684710698706474 0.1595662365848338206253805509182749636842670947491200167482125077 0.0008216656799225037740778361496483079360183016151618575879148558967
                    0.04514253577916598071207864910393299017794625123757226232115959067 0.755822809522532594458704207751589345037184171009822151457742095 0.1360526209419930094349803146050573189929768619919939024293179545 0.002372009648928070547128923212688603728409672205717476464144983388
                    0.2109695590435435803891613242326419706086563631612076783086689334 0.03970997805326741202747571199393510705297587179641983882608941608 0.2777385574884518306907050196437586445555238233835508844389340221 0.00420677638261086994089678886572470329813368696958502511997140544
                    0.1154006208329456379958174061102366204434556710634203080896369499 0.01312710994713368630374856737935338934151723719558019841575590451 0.4946902419444486426657088613716212912715552188750342478789563571 0.002041642396648166150154240779132750880779649717714319633774267915
                    0.7150147963863575004811807005427627859363957513164066438043593407 0.009849618747045996557555081987428707130326150561717150108660156465 0.2623116208507916419344795982537957940289934966084105254092991287 0.0006022163011652692016871468761306290115801741102012468035697405565
                    0.2061729214995968898121242048172906461772557116171930400688503879 0.1712263368070904264852366804839651476707196184322424907789913181 0.267587434310546139909788746239012150542303520971529423804807876 0.006056495214882797468765579550231433170434561064830847655896554313
                    0.009731040506822651723493536085609775538077841693858551167404797811 0.249562442438404433679039211583309955903685055448631664929716739 0.2754042974582738315803278547549966620456859765351858157003998679 0.002259046502031267994255027829596360759983369969310212413377895358
                    0.7261971294195701463997105294753274919072368215936191950984019664 0.2546047126383032862624374047377058017746084746792034003368328429 0.01104251579614349039130809904177613988599842107152371578160770151 0.0005143088568928592037067361651610582131117136812072619038365019085
                    0.01179041790428235365279746674331655061098048895592349200834249976 0.5089342766327588506976256811316957373866297664040639486589924742 0.4134041910526009130102294905442814809878125785922244425531271081 0.001567818514524605106642231213998905106408443201343456165896377252
                    0.9061138964811945585544342415695090466962747489110266877200205043 0.01279275784724954420939192673430199573481225778703612529074795432 0.01290607235745937069390698655795953271064545731892976265708376804 0.0004793234971008252555243831997479263025970459718236110000260550053
                    0.3181029440183398194047890906754164069912578881481159421241847135 0.132610875122909713211205279346808086156532468954244724208251128 0.4980974241991218502726355592402153077745058671378751611269687394 0.004178895109875177850094651544253194207780015630272718140401929242
                    0.08061419418332768930888162262350715628421156259350887641647865986 0.7598048406064236491261544704738248778708615955351789155725209637 0.00816641680950379110684870203760606103629082580198505057500663648 0.001173569720149894833797935144363295624884231698156599449214792367
                    0.410530902630348553176683889056503865393696943820804513376061013 0.5305135094123800385612052660366265595740239876287627714094303773 0.04899879535581859024231984694464201440692168830662575795348447967 0.001159332451880553826932040691526919284678536656793185626160546822
                    0.3310783093301691332889920619801921054015377937654201181437746929 0.01146058254849067877554998313195029843769796949455106372177469441 0.6106992069599274859660299964508440661476638244024327726003963361 0.001437501147403104132969084086621801306601253976673573271160143704
                    0.009871166514739684261703069388003833527776720705947672780225948164 0.2231661739469247845300483028947714297331175699845159986450336954 0.43652487268531993596429099291958954542795857422702320672209616 0.00155056755456964317884660888694366550962389938828353902175466364
                    0.0459895427071478200475280043323889082866800993386639656232922378 0.2228209259047590988083919671619247325293923915818384439857320389 0.3535835589074982540259480105057621314442345289589267437907833277 0.004543242715637085421326298027136237314988242125679938894989889249
                    0.3912730666951927966882498677347452273936798273671159586403122165 0.05643859104998958335779096821000469436798875019817061465747123868 0.4959597389901428411744447016948238609052884963968089372779031189 0.003562778718724868867259239313501887424621963977266342824553127963
                    0.6983227985440156352143130739994047753111316798061132767043158504 0.01544815232971902009951155789774997608107822150050801669464348614 0.00661504002870126027035944199648000336590614722782316300228722488 0.0005481712448369296753141687727345890133081514922990828774826345413
                    0.6970516343617074136672244182479992077987651933895309438318765044 0.157240411604780682954503505374721192087081513389633514643645223 0.00683572384410165948750709207131764763980338156343915580578678017 0.001116193341217669595144656346102856909976321185271663107072183724
                    0.6738759266569840973524435906570239528675824539706947357141914453 0.2479361214584165579203608362586834685630090586264243341927411709 0.04714107061857974266829150144489134481041480152671630358732323281 0.001819081357295570534538592877999331022220807840023610260022319183
                    0.7568348261513824772333724283939260490029409287460864069810019601 0.002925903027835212619137492311726515910141690289114430727702458311 0.169853261612610072748060719469472884876303578828312052047313742 0.0007421136973509085997663416071621097541685664839242569646735897038
                    0.01404730189675040116624281150994177422283941941612780195056914254 0.9595536421680580050339385763027214261631818039572569082056659686 0.013194845477222340900767244605559784933280907787543342892019573 0.0002467969598666513236420063262623152675953667978193732516300538733
                    0.04988033973917880111709944625802054754145728381590188473636528243 0.2835492905906742849784914307377595068998495200508828518506857619 0.6602550259185770504709690875947583360845274785119377971858278524 0.0009011847744483346961138760545312890249866591779945288774701369201
                    0.1234564560193107756887274173514548446656691716480554033537185578 0.3513954306587124701433483005550271821201344824488322095894820802 0.02114391688442669448378879758097175392629064613640610790747570048 0.003067341660290766845075247079012838337916549109481665305934511137
                    0.06442685576535785539415592886582590682334934230677880780073974342 0.04853522775473208805442018912840275682610614738271642955668558122 0.7457474994991251027366416027358078059456098244342105276450161788 0.002331382023707350995020126736143175621716841319046130333196705261
                    0.8609786495412737683946541276531766262074275698562134361990077693 0.06663422336296528320458844754582583213641626281197837957930309622 0.008324872456045920629052416827634527092406790757663689281562384885 0.000719660234403458315941066611430817176847240720887037909823823726
                    0.008255141787170683433433227996516380584397463022786654165599837508 0.01071624686412952433822988494285691278376420554642801884129272094 0.1777412175541119081893866973760917351652862555815574901073684606 0.0004336741442713923587328341438434880658472451641744791027741494836
                    0.4556604653525540947331565290674937439131691245327541419642685452 0.1138238881487496019438916023565712897451631041793404232520055902 0.06985009349005809096721627188929951140935588356382031764727774496 0.003498171746335134465767933404926860818551142367816434169207523121
                    0.2052997818460030370424066030359468541011803812577912201122783143 0.1058953913448904898341013174076288414282948961049897003254544165 0.07323798505403793609917271322877472091426203026985250048186664273 0.004847892389538725248711006231771441174163040885987152245675608241
                    0.05466179150906917643283178435446098330196866024457114999324509898 0.8849339395402401521930891809440219665588482559864181363190128737 0.05530915679013781371513962643703904591704063400552554681503686419 0.0004250583837706133131195743898044327929501079181999110031236685935
                    0.1479196568831410355655229438558587516685824161691050739471165426 0.6296091491818501297565877862347975773111087276239783485487949018 0.2078217503528368655875356835158957067343358553513086549519094146 0.002098112082901056681087047234509390478375674822950612248292042086
                    0.4751433042787569367776735152928550363575842410318678917678614296 0.2388951384142381002037650825049056289989498025505303954791253385 0.008403700949916537351402653068463175712803336982962481253799221403 0.002098291801492667360628519322765793057831526060376703835856270817
                    0.01592312665137679788458593283983212693410744822106919309506853846 0.6966586776185144517946719959985767856036978631430974485569691173 0.2821667387297293625808008574684991304102768979380597492958618683 0.0005107358735237145242662528625927918315250641604098038227358967148
                    0.05600463584848424716926141353299169559470084541308779095471349723 0.4678103929212246006184059183450426626341797580903711316450244418 0.2215507243400468407247015263228280604394489979136591461288642321 0.005361115555180240531336303898353581243941896652547043644834036788
                    0.4929761520633642810030767561616781706037132165832051448681546116 0.2478153285165286695132722989241578397773138031138433748786161068 0.05246084504957401791360643696476036964176688688987076018189222234 0.005679767526496487643519113797864273581317227509891575455594654591
                    0.0552460201181396294882121492726531448911218579797294832498243025 0.2956866837249317678080689790379661662698360382901424969648321278 0.1121539618146599034671059982628041710976979440839425828985313541 0.004237289662282433635117210531331361003995312879756128508708852125
                    0.3638064792022035551783978077959066619765955010430562592378136016 0.1222725858735558234793134946251024856550881863593627082489196096 0.2468928885895250964893800522768323541355787727437752774393648475 0.006577935996344337149131160016064726774629714378111770921773983105
                    0.07438610880509219902420125132377750346241715184875716613690245559 0.01011074128340355605092648884080033715167420697464243424952843937 0.67046914381751732272205383916727938531984702011276308574612042 0.001301158305458160587659416579989956573643110114234430553031882667
                    0.2343005324386921844935676965328529577468969741345115713839551702 0.2336877779359023958074729523113007386645498494852960170030477853 0.006664836795154413030183056144989710439586540028634947798620862051 0.001296064211807716456399240266058106954071337076948303017931986202
                    0.3915194580357885606298899375111041376883875305048201753227522218 0.05209767007716452338918847418223465181056868843856731686116225444 0.5464850585928415124021168315916675547395102930700962189238327403 0.001305768842402451246468198823667693789044864397542572474783825461
                    0.01703742869858228696682686558060717339614075149610823285566077365 0.2031356142658009010588379752662827274897010585354766186486575987 0.1361927589563112759128023006100245732133341904406274599669777406 0.002136415997633129203358031586462529516740035638143434958904614839
                    0.1321160390790840349895706711060868904725065689636292470779740726 0.3297363263795638968749129365310604147080386633777889791819718547 0.08705413186282111163237804225172043627208516588491524288921856316 0.006464772371314079761579330539453174992816465507083941942288777723
                    0.08502836140924545511327615204746754565947199032729823333091029235 0.06360314455168548475763233254838551556990171056203520724326341753 0.008127580287717549838873783108384053685816610354309643583723249613 0.0008579185140951100663140617591971368922330262726981449564535536136
                    0.2528136646874855236043593362808164109517874854401186363579847464 0.5001937218814212666659935940131996505188198326551602500893778748 0.009680144630366350289978374202323069427696103727066855618472375659 0.002475739731509300898634598273070161997622918970692321999395872688
                    0.1549363167609048929835504607212600854608714587499961654799071925 0.1454525240580333109684184118227436207379956218218445193630502695 0.4745826608977640049407427301851834247970589648765394979591386422 0.00729991848348029049454081971301684922656648113387733083976846904
                    0.1209006639209179253193569131161207154411378537454629522088411546 0.4351490318782843236689318332546517266855062141603416111301001685 0.2871393362306746779958825841581512791300160427547761813030996825 0.006812722592098799625730090412712277819436043512991645937300818597
                    0.1419645872556482628253419606054781494976866519237200378891430463 0.3263948360438551964089065556253457142691284909396348427769109818 0.2619111559521262240645485247417925821178968255851107452719152354 0.008471797055685149805303482863778782903366904026727201982428018174
                    0.001482304934469625998091001062093556835443494224092767585774792412 0.0149220000342926946655917309117128469167321051722119792869241755 0.8363902770626195213105647833591909803572509749525257096274909162 0.0002414790285588126672828452969539357520731545159812597059628867762
                    0.4449564087744166871404971856564192002591545610942234573558820815 0.362892567065745735134762914733922987483330685369382520394952236 0.1548552120475672552233380789170121638405140802321785598670289493 0.004293369248848352956733696323698085356656454437560931467529936323
                    0.07180354637930536217210846240050633533369491428003655532028939756 0.2468254342125726615860449090121158045839427649131295737898236532 0.05219783342056960327416520861610134100979997827480840879756897247 0.00281298516889015452924959395222710489885389686603670927819482364
                    0.6206825753839935661996912022872731987206641511126823807943883933 0.008562070181738049901715708083111509451921016010650036067714215934 0.3040475750769399454483451254913710664381562013881304794612228801 0.001318512606522279147090835817406821180407028534912774900146202413
                    0.318556427770040941579416779905849627747616106067058338446012132 0.07006847416831254744256890647290888915645459090892325336664898569 0.101503926701695589634029225719671961016711607157029860336659637 0.004191722876956784889171701203043453147476972139043235557750441159
                    0.736094299924273478573895993440062950824098677931825837962948535 0.1166036121580819078248624862366775829110699198306238478181031136 0.10590241475321527682841796047272650072729583417986481534736535 0.002062110879225057524192453457336621292640113150208872141478842762
                    0.01024219342996255571881807842423258188967114912600606732206394697 0.06056573690121588092740944004215912817216054140494188391644411498 0.6757312018626292473631277510110743936325720405915634963479099848 0.001446194461721202738521554504545103578019721447507473203731179022
                    0.1586259765432042678803654485543933035625233605752147015263123539 0.01253814938003311726496070870170186978078610996489954410996409653 0.009096530154604034684433727288828204427498282627163058797230344204 0.0005037164397044637815126825766328073147594902047255315122334337911
                    0.2774604233196819662800370131725494691025200086674937766226868296 0.2144998643284363702205616888213345765148093623402228263171772521 0.2062955563214999801869377989163084884854243186453651622711228167 0.007791252117014832640252067721404737785127129647088851273687774416
                    0.1567752856969475442994041017432703503219172482770571445889123679 0.1447481732105247680694588459634474429291593534385399586996434562 0.6900032160611878573048450550601641126386648300345854004636820635 0.001435943541056236069857139941416848918999343696402286141134132117
                    0.009768551044938895431639666812149190499165651486310996188122252407 0.1526669471850728406967133876119916452050454525809382670855983929 0.4145038148839437196325140777903152642605939593398598620775201619 0.001978858015729742436146644861015568470779680264000794002391672782
                    0.01169420591126550911455696288174878255343030537683556340900564868 0.8111708772354951065461707986416801091177573363509045510766177574 0.1634251218763663820483546963498145984168568993188812005405453654 0.0006610706896942865683113954006028674861965423158339202744380067517
                    0.008572596247039247551948237035521207689001647866066245560970492578 0.04983530774597488183709615953747280989683137815989850492516707141 0.1153200937463942483036698124758760589595402315153529688651454781 0.0007382625204080292894228041389310503857747201111811910951125701233
                    0.08954995492236381200644772042248676553865073579705982351790187788 0.04417004599618862535129085842178489578716979437948815958454003024 0.1167076301033223583931440711395984568544138734264120271788294305 0.002344970692664289450706608993837076765969778504783349712968066045
                    0.2063936367738689711284757870078187507961340657841412698753072874 0.009981265821033662810572649882090213463355859492899807028565880658 0.3923266504741728303372118207812307773846803536409933994195184973 0.00216229204525800732263086018566681847358774707599248008486206309
                    0.04658075625177691234995972624774115602577441943383987675083819749 0.04497364263671941642948782203609737278734356545451511760898321622 0.3414242204988839909049521192081555809454775026512278106440899669 0.002264736131743328839346511142201601366040068358964921875051236262
                    0.9243684772838771633486963768178170788957573272722418757325540024 0.05511768057244049001108139887263970218456158929744961762997257932 0.00877602423233256780672463026661424542256578827819288355102961626 0.0002739346141023648303477631537253698040314726085408162636117071887
                    0.3347021013096671829764500161993898734419921680871496434742262603 0.2437125176058114572250423372048171304991970067244262648699067344 0.1074836904593371569413442493943967821431437535223605145613628819 0.006403366680860336628968870672786766213664708161975035376653227908
                    0.2752958593214506954403433073018107543474092454774994716646644139 0.08482533995823612684285208329881803343342192586135631745031340321 0.2681756332498870456069374714961477965244186871734289437342888322 0.0049745838347742526841162873495034492141519405246803590662365683
                    0.2848663800874231709043636894434935328232233618486666830631135714 0.14167819532058704318484023846508563797653757485772713134060504 0.5655188948769848823558908664761720917826049116755451720889515889 0.001711412936860707909098602444565742571605567049165381743973533257
                    0.4676552522851675252876605224848828757549299412166610336488919591 0.05989803750128275941062498877803039090525250858420698747273750083 0.3565094161028304436666550612383744608798554687221023377228197202 0.003573260948215774681650103651835462543616686851390101791086906963
                    0.5404628530676135175971208921486936231816090103676578398189975846 0.06200936021663795633000671516906877759885161452666609564164696372 0.346675738480892450834149287729147229237051043809880771483768189 0.003162365026150740100971364874230838612411969860876445457773293619
                    0.3026456435756295826935575618070470303556454421760674767842524305 0.5022832375272073652797061878677457419799640550458255032080161396 0.04401276524204270270126389931593029484015540956401412082515785434 0.004238139931296378677979322053061451398680145230447255252294682327
                    0.4278731708424607281812712030491540449007828987592883742044809718 0.1122120214849268376178686880668617224321720842043311087037241075 0.3027942699120114774396799134798496427840702142146085905196966034 0.005604202600992477345661878700582906204113447306876332818854141383
                    0.3541384371877748020637999759553699422720078533043060113556333976 0.1262653504078961100147794038184115216859298281505617249577874048 0.1382151265690389422735135096530358458554580159737105586404303233 0.00671464406340096059595939595902132089463649546777345773474611227
                    0.3724967648560959379498114892933240486387997969722322733541024664 0.04262456788104087393381166912655932103967478840375254751853532272 0.3407980426581827377092721834486271575081025249829913690768013758 0.005850999605668070028191122910956861856984333707438323804158108758
                    0.09059551791213114857994482430959351401991193445125807351642295956 0.6172360754302888245165714916932508586558630650892263826362202927 0.1401550792853221711761143471146410546229799534239735914724844152 0.004544564508797197411763909401733240215694796701725628966446718285
                    0.5776627922922818969221159397664786120946307980004557112597514564 0.1089579912645519848948460794384316349509557632288709916565312406 0.2269482137772628409880351226004918913492068309733669861227475937 0.004439721007103793097073198139286459427117027179595132394320770433
                    0.06220636304456167434536922270901424179087346057953182538027442649 0.2130438598114454536706441746109308083417720798859031661120447492 0.4839029449288692038460430610411241473404206669668517885772455373 0.00582007222632533271200649102266980205050768809399566645922804096
                    0.6997772516642231567381351714106906138149882285935906562717314247 0.01590175178083243853591582613674623632426255636851368521673151523 0.1252245163548670780256065806832325786443680704005058414411586012 0.002101049524524465868530553829308025006463200082171293306696819905
                    0.209001988992203983369287600563751339341197478850486073507268538 0.4472039823721379101509523582904499307109036149511041762203157674 0.2360242659670522025123068279285899702366028062868414237636620419 0.005611354695774008832935591110169249971436812643509990464278353787
                    0.3208755548509699363415082645220063007511253911806771821307408111 0.1472003894884585977203772855515063936461084482349241184606145659 0.04801703113923302549560718871146670685175143266566342011891551929 0.004596878156958330846185895927223492801125854793429728899174449051
                    0.3600155128336365456278402601465268160688577545627442158738250103 0.371884083147130466866620937279988117849282160852633967737022513 0.2589616408190745794529480090168539009794075004998302608285720716 0.002187362066883068830822585185461296319907851206136412330765053086
                    0.06217798691208318741036295203971123903629820844768118035104060301 0.008786351961210628172486618922101668036819893233896115239395168533 0.4156438789272452956348261552604001245688104190948210012119265207 0.001319634050082358387193675229128139426750337600707188187432277775
                    0.02493778034671210092588260704828224366761724452667513337212922189 0.0716327899868764295383248861929459317981738860385244394985513787 0.4186499761940203667690799603553969249171849783720303177841773387 0.001529285091393099833541261170346968684604449459414185346183590615
                    0.4024664015261936043050850464246514006049718072639285899271121936 0.001324610238955055213098266878822478622718429467920746242373894968 0.3703921943545860096443592398236559808204931178432193270040474846 0.001034295294289106870956443048568848309723475376937124739847668605
                    0.8014950023695399446486871290139871541994579177888560352937889454 0.008761713405488191768324012069096432892304495836867928483872821852 0.06127478196909193051159719233789558519460160724128209683290509913 0.0008727153585725095914075179644304736787929267088732617639207033985
                    0.3577512819498731560715044844968379915157963850082948624807068032 0.06735254021005649010240820273938072350476258884528986702532818511 0.007562353460525031790733320132957462398619153543983480924184414087 0.001339824454852404630732672111910600572963222918051344047444055379
                    0.1024424792193860671582210725967165532637216313317336761664066946 0.1877553560906202232868582049738215973742409671413330378962705234 0.2588449494611513625042457339078096545569610826331505749182761466 0.005570590933779075673904375385416103079221321485685542952601199201
                    0.5144529668766752915606317271752300941617055171763025736163072402 0.1312305012711026371641817134330669836316489384043024529287370011 0.1905997130598377780926572310114314534304559194126542412138987628 0.004575804511562855146539416300056346649585972958375357485376905341
                    0.2732941762225052141097694269422011373272683314476352419795565671 0.1183692644778683617527694040643795264743271207614331986432980661 0.3731543675220706180195461861135425763893026768043855843156138512 0.00728510233618563141226416923151838061607297482535640604615729504
                    0.06231328494370206028589005794850761230722087759260056511114511535 0.06423245835172656470932133084828941947712192483132707746885023129 0.22356157732228624238129991939253707740585109600445191017298539 0.002911935272507394303865156596422054557845647333309456673501807498
                    0.3546188806684391304636811789669121315666750538227965274178028634 0.0459188977116012160210266606863563711484385900894700311253481078 0.2135931355908018197520135099810583538731390914577069087968276168 0.004561764791020089523975779546416531845997930453190360048783053502
                    0.008503679294258241735974962997939980555050676566728263171763460989 0.4139130299606199591291410726831265332839163213757832163661740889 0.5660764291369487516368446839997009339078679541849685543317660815 0.0005785259828475123079231591913608063264339484704246763024127819359
                    0.5887004037993800070682242364552890666207305625022796432650591447 0.01255673897852349666357423137841749230279854914075551708312301524 0.1241215642900050110673175790848352036083912518395212546016046702 0.002273866040526136986256861552453155267931774390374703563781394296
                    0.1276131661651861408321544340285311418646473849070096874931017087 0.1622128747428282637070634503865892235286546391476285228968809902 0.01092702049576092239154564510042694726826356380545823766134997953 0.001709640710083266428349626046500459051098258575471967467347563517
                    0.1375264717771083612441818890456315653402038063009629939681617033 0.289331898244371024736311580143629251323282658957477334886762012 0.5110832669330296663213894815966589873628259858852126380315283035 0.005018183550445371814415744532660583121527032005650807047750789686
                    0.6072783529814260864256571753869954080594732734302276724524264045 0.2522112787360198409403969503794437985041878871292227568884968546 0.04895693695330034938008485255491768357177156675726228157073151194 0.003652202187639595783654423196255280242060767785133035613493570987
                    0.1160607820318660209324656029637022390777203285102756569853513651 0.4133132872921654045733371863257000289986364398996185180738098699 0.3967874146231588055085711262630335843328943795439393446863683535 0.00512311470022469266519664983458131703596817123097594872165925737
                    0.1544853831062414388593120215911962051358096286705585510343254488 0.2544325108653101654658312719035347568896389216251249506860037187 0.1866875511403251639202390687309246337810544986845550444895347433 0.00686497427847005731791506207084636891951826927393666246743250128
                    0.3355783311806375006749971563213937563358546186470562735509125023 0.02685150908472936589129900737973542692797821496257861193488198591 0.1392913341297285954930892867996343542961762968900173155175302133 0.002135341328604499548432184020668126198185949523202398111271307726
                    0.03540462060017077514770829780788591032052247442924942040138363934 0.1235662947097553779158764782148808184114262205544567899217936598 0.6652804975371982349552193940417612394168981567876725959572115705 0.002595540242879194984544108777549218471988545684306864965970545568
                    0.04743208407930154030814378151892719177149690272449181460307034925 0.4730238878178189284627527702156355567820304015594375005617497608 0.360313248638770260077042534087253031004337140941865825408920493 0.003997074819893023271893570746733843598400766326206776039659139161
                    0.008284014886269645732363805972132027935269444403794398560924858396 0.01451691799129920269726432315862523279129787566387420531730845884 0.5936986008267442259061762202070248718630012603636635469808843337 0.0006629766732837238775663029569342179461389399016470724318147678177
                    0.2743723962752817266817028196973663819038659860562161019163891264 0.6511800333245233006288986093683567024240628807801040254788585487 0.06127327597891175353545835737835458682335859713529851672451706412 0.001728623035192519853427956760130703110320528460804150908139779014
                    0.1263496011933354728481386290819228731559735619026697483742351769 0.09988991396284406742948526027683424258401777992044397064867563855 0.258517142010036111690884408672602334331751433319554892788276428 0.005393934677616673387568281528211973090558922612924694638551253027
                    0.228318250280194094767856409716833729980547235522108331395833583 0.2235996954288536308549413623919731593594266736125026638655573968 0.3355669870736485158411811912637975828860898966365432647230920001 0.008463006113400005961341138951824677928195195409056575820520461178
                    0.2346101395551196704625829251853602257516568485046537285482204928 0.2009634442906702244977275565091149911973423004727364535569947173 0.1051045910273925849208422243544041051357810810210325796519633315 0.006932978720394400641243739006620611655260681445445012689073260219
                    0.2738734217408086854453543357295025503069865662991083810960461083 0.01103280626176115203864690940028322146216455129672619220187240658 0.4681073434828054367312464473109253840693936517610786001532466654 0.002527198322112064348669563040076490496104103960445121296615083351
                    0.008626552676740369784103904522189469651455336672124742019480697529 0.9126027907203063421407635534595868562476568918178197776694684194 0.07134346308119340202289510223805591429273078949294684573199805991 0.0002269606818024703719217369166289069535946015298660271834111290693
                    0.05316080190784124812299984392807193724760914942557081984593356096 0.4569426942107864160893992949737117353661328350633073406393791596 0.1057199924468696621589923006096398296775019331288706208686439854 0.004393710144839507203666406997750734520732830587277961316616699977
                    0.6750686228695721195409703936578624752461780093086433820025162331 0.162018831873208550063047641471100485442195117335156916047819769 0.1612989245746404763197095098553675896235555740925350035699367206 0.0009347524276665438645210465488906018651779832989917162984077426129
                    0.1068902968464183873155994609262577849170852742284108994748654849 0.4685260008491214329285745601082735005852517290141466252013961778 0.04420138791766394690278162834694684312061226922911132569842186828 0.003304270905842476982670104745423643092047111645095445216428945107
                    0.1540946003014707522779478599316686619523594689474225919402426964 0.2095162618563882224123465901922481919063453894456952117424648342 0.04692192057652927476686789038981909520420063770129502721379176745 0.004285250448850762038787689151585152124895499561715864768316273984
                    0.5889953461075929448597693237187264660631679006756745258429274019 0.1533732117632819509904859357346988223440692928103048871680108477 0.01162445994925964376732215756440834895989696509158563064499533319 0.001824661349411439902983623782083981390614710728793052836122840467
                    0.007507375197407561359917818114326308089368349229273102399437212431 0.3378535776004882981265596839592686465980054096216790056240539884 0.3390635142568524880943989836022452650502013568091166295009169395 0.002011792077038490886742985729294636818707404992255315638000481941
                    0.2017424079830154894164304323855954721722126814693889471454294051 0.3971232936972806153222999310878281790043494632812749238966909685 0.007925966190713067940656884981977354822370443344833705369054847164 0.002190979060560776710712184172218890280444592326946690931693871081
                    0.04406533633428076400540008012025239986823332356484591080849970282 0.3998774096275662244335952094043297892931308242412327009146442993 0.04099784685074027179962167972403892757146647520099364515225739414 0.003035114559564630813820502225482708572284605160521421980179316604
                    0.04529241373687262861984776746495997625064484817803565279285619943 0.04312562678342001677477812618195503225283312890863444668571498498 0.5200368574721614785012965708786944430460095263257884104375828593 0.002859069706361223304616766724387162725709440419720063343210330469
                    0.01194048723842388385260720953906324717627435193418471085190208044 0.4653553367734580217383659251934916689319908224601107509573414098 0.2986112380612180194348488652308624406077730158776443001054113816 0.002374006846680903586191580470094245423893910933325405167876807737
                    0.05682120646576190402075400852241238467328408985030879328872180858 0.2824412816279976948521795355996057507667593795282207141281352562 0.01050583618219431015297465405275947498384775922006470326037884864 0.001494255414451594276228965071493353473099269040127999327414152407
                    0.2175428662010581708183817157608251974570618864876304670603427558 0.3194721485273899693771341861750460936782434079873759955599212172 0.3526288925478678351018782977955111468460715867791598643413755905 0.007175099400018186629254241287301962026723599266845155532263681832
                    0.0538084623990555283824424797565009635885098057168329125834235891 0.3865824459758739526151125778922787828495360053935049645601940755 0.5262743214765079705763235759181531289030043820011290433295944619 0.002804355807242947825040435188101785393930482048312353998560057206
                    0.003883368586444793949583156216872137742556189980864163844966413414 0.506587310931755843349600270991662386098425430102134677451547894 0.1650085090756567494313389195045687419719352409664715204459410746 0.001158955709952175268221070329402104057196016035280769178500324703
                    0.3834045235204784369447298767886606433013509195434223141979084789 0.2969067969897089300759513068721483216578631628621333752584493309 0.1348232791626318988709887918481271609690400279565535348021286175 0.007820761675869634789961830121651886415073927744627422108664090573
                    0.1223598591298975847836778720938205177921767656125672461311601385 0.2151069630624702180081774192200207846488860642024075144547794062 0.3548610299396042802141938309236328024329439346676339868330632174 0.006977917649969085735330086574405014438217028211089042604580357105
                    0.2122817612277926013553344393392807969941732914756120436159347123 0.1882150766127174743740878549305499861509959729203433888903128497 0.4831914267028182530185858141423025078683532900610779650863506603 0.007202343391293760232990994772973349860922047799267240543166115392
                    0.2873702236772317632823198545358963858222193194904373468151838576 0.006666469612616804782093878619897179118999842580706710641007287137 0.09669444869214548452105595274984519693236175970856828179785578268 0.001265427880438073167543057982931304521382165918447617776825656273
                    0.6207198087683914907063007865926442049060632568026000219215544575 0.03880091194466966940801942819325449903609812443395456569902782863 0.2167794267815190656823765827088080052946213432899546142789174475 0.003192059685775071411612449206129784319708879654741267866322460962
                    0.227458264231495361940050160258307301744262927518919477870642982 0.3753126766296137255290582464458629597318142513542496931322850615 0.3589668222598838784840730503023876552780370646461236512330089267 0.004846421705317647172461975815461685169652235555385403614288466013
                    0.07905552294907260511621686050867392596427963312076957318452803492 0.351399011679445230154965508113694165892467037632407256054967443 0.1885963828744460547256969811976059720066536066067677140451873069 0.005854735902054122384766992835413239596748361613837520923345790285
                    0.008387551703099463444137821787658175508851389733139193163769674923 0.4078472401975844298894862804094270310963547931493215690591396741 0.4280077229690142997501480452425933867299748439128446169043986652 0.001953692452078114219286107719826240981982082533963646122779958302
                    0.5136855003862043866233986586245330408686808506605212688303239293 0.3257782725561116017614292556117870995173151656412491187623818617 0.1564764042547180500165780443989279781272070291990385617277459375 0.001185912602493444637431940223196125176103413754975911260612793595
                    0.1650429571170550343111792493483261139716952642621527833090581894 0.1082495584051440486833968388548231087155697423333785000344863066 0.3624855376846750192154756311690102774429302848858055399949836923 0.006067028317288034899269826170071639566119764874390137110299374788
                    0.04946384057437318681111566512745574472141411558031448542849053288 0.4436713088830810835612698537071408279782347413043282408271054665 0.5049400076512319055841066114482621899227982384137125531513957855 0.0006099843476272448004269696515309764670756208632285923486857461053
                    0.2925220967766971983860007942226721254787803940167351894338837454 0.6895402587406871727732776536248404600356960769708583001718651831 0.01206988527930006172470736381458736239168383459464620940814123354 0.0004475206533702892447790209372471535993666716788752508169572241341
                    0.2564223251504046175449512962031009057238796128946657884518266949 0.3487847318892432870590440706088280945313406716827969861463517973 0.3898185888749790320121998535119776460046751630197408665116984468 0.001539217121737253863217461062751984043555409951627842823215287205
                    0.01376435725035402383456632963784142383883334567894332591476519486 0.01028190985048501447786238118335696510157349691960597650675599856 0.7223933125476066838048068829847765333331413772112192868336023905 0.0006842078417167913329153857979784141953329796891244650565061517398
                    0.002334464587153450698928309838664967628819410373482752290261306429 0.1634300072149349443542626988618548236936361344632600658642024683 0.5961388121008691450842076232658103959174064015551075543506085462 0.001052592012203087678720552877519657635615348441993201963843204519
                    0.3430396837492214225403383227669424607462398553235306139092719692 0.3631300895763641450957270708165788148569637733704303736950204116 0.05536331475024535677509663632745415989769663796558672663148913968 0.005653384680230799882969383398356872739950437528808589452694720159
                    0.48815847720231873151841156188576603812176080556951799261676932 0.01226370786390501519073777976880736811653714147963094592394210761 0.4450478677654089871149882897202130012086406630474559539217021899 0.001688598171779806015641238965458460539970962465787322851235000523
                    0.4942664207992518250303469591974361606821659596395377355137795039 0.01509722407564137863007441150145944704340044864846763062113021789 0.350786228428630010597913081824622001542862115550875017924879443 0.002623365279315735494174578354068257637978410362815731520383872529
                    0.7691698318849245950647077180284975890885294418968038973414257879 0.1616817827786955635909332372802531720315541440599314271018188297 0.008991514865125144444366911836719845745746353590877635970996956972 0.000908334278624354161007052548798484714903383831495802494644199098
                    0.04763554257638223989044613761924644988757808182705835662096848343 0.00306143296620837446885159202951350238186488490047809358595804574 0.576424598760048302964580034016285400137905760878584385971309725 0.0006631264807691319190619426240363384390286996790735183454064193004
                    0.1002200501116529064975266070281006978155583175475353262920971698 0.599883314488504433159609038073525206709393488035939916848254059 0.2259985629458699564198605783672774244521207050659231734608771331 0.003580777888496280128884932176419484911205440636145104096413885104
                    0.1645446957305356276158484359792525958104198768735635080675433782 0.006805165740062002713843764872492144781702816034995195659448031464 0.565220691334959155753988010602977847797819250264563443441819393 0.001416723025049204801584145229118195302405088705152372646857198479
                    0.04672540181598504153483004664712264595267310622637490089999844678 0.1416139023402403525764670979654428114187644837382333803805494779 0.2092901143084900481171796529400895154305782169284176272225225396 0.003337652161281981710617239361738844012065217080386665164657902482
                    0.07334484788411772436978407183689140371047253549237903817968380519 0.6854552009908715496197230503505400766796456762589747852284737797 0.2296022936078501186812607792532578894738848244581219875250566866 0.001448245909033106851803747067611999588358754360238243732121130782
                    0.627156958994821042235079797372185943800680539550681338064660009 0.1768393551494386573501041738165566632521739066381755383442270206 0.1190298917085523288096904775339511147652343250561139198428909023 0.003487882058077925557506159230165734410168555055943084014008267918
                    0.519813169177305208308349757211238910819153687653203663322864032 0.2107810112653915781865449257797529167945937587466931864988620674 0.2366578291372221752970682039146840207928320040332358106748286401 0.00339433646629191044904340226784532704731076988766942375254017368
                    0.0554931177398553352695495240744187280163891370626088357378597873 0.3171949329842841040368243579554368042593039107165238726861729964 0.2993360459562239815616230604299761838084239286328147597670746398 0.003969321316922113535611785838214630962895234413806029082987639998
                    0.06330321378108539449060313537288104918590527046760134387205533354 0.4515048122373594403667229196739162572098468603962869802225280912 0.001272481416318620257283322540374449755550218833348835072668462451 0.0008206017720542251684505563802073976017623114707719394131732654183
                    0.03729243767808300318696033160360758121959134863988949055157907273 0.07107188091481412501993700018676590085299095026599597497580606612 0.04197553391507518956030662110315334120023885952571726376979427177 0.00134318704402333027867648233970611763680784912878123137085469851
                    0.6051072614034177063313512069931579558531172833265621343406134678 0.2375481048949368096019553486023211896917603859609439869028138394 0.1306395215576270064584729619050949231383605733159570395975062106 0.003253360360064695202041711401987957165694068783851338548664188213
                    0.6546799935767900622104902112736926090583982070813477967480763688 0.1130533889201088557549907654638401417113495766920013828828029 0.2003712834713837640750274910874283684932527511386943749285239238 0.002771983261404232626390440798611346031006461133771497707628562901
                    0.6757847138954073876549668434381553363358596055130084152923342605 0.04723054328664542151655373625934732242564645179942322249034975627 0.048748525236562168648853490452545799892748728361798953799172094 0.002516800125686125416387498892377525949562090382837949845855262342
                    0.4734855994831346669157168901842240623376212790144333363931246072 0.2157398073547988307690111256229459366054284967144399837782220381 0.2171104397810974897917383866228580317554254828550204274432249432 0.005898430888001486814724466183466794749105597100702114464505015975
                    0.1205251764985327334011730306919172690791150219058573946135206386 0.1030241175861522788430318507533639688698130837281729341787590138 0.654822883412035793382713459063011626808402373939063246866205714 0.003660270422602134359700398975627238015402166869544271717721064812
                    0.4440020945732524564749967968824477120823163557052936544310925145 0.3812288193249869199064413200625841276918285383839379794327811383 0.05575866456683203580058494734985994199910377749674072147954619612 0.004968753774606105346589318554832912236287270990463444286525342847
                    0.1558988016689168121990246808175144205859441240117717753842743678 0.8099654688956848889463196572839114480447817760217207885021623066 0.03024303664961776386209435197271933234198681873771666409698475507 0.0004865213906075066358562558257908407617697616229739813883993717055
                    0.0766900217378983446508966975245204534505694812088574827943595396 0.116005573606704452047709481063346481252549242194254677323566295 0.4401815492081524104903004905542553452901867561880550032418028653 0.005288221121181251578846524740064997774831749457042404379786470484
                    0.8243849501456240086296529118930114645929368832854965584593598783 0.01215909552539360467840804812628077231220854530706060124340428519 0.1507383079546644175053742238493156148900240056443239527773219396 0.0006140964709442473028169552135762200643513909792868544993262902056
                    0.5167627366716639784999883659697064716554481994770444103193010341 0.39966754989872600038053071986900583543629859275887368723540927 0.01181385926935391125117035632226271793132451803730227581131748236 0.00194996734691494059319156808662794595603918029920428844331251252
                    0.6049836777992787047095400078912150795132426182770671162543732302 0.00405098946653184293841568281861635349626836394012533658713175918 0.2273851097572020704252605440303323190160458613067235087112814193 0.001124760230688222684854876049704667187542322708868356574934769625
                    0.01495680534646867509070768460374789947172519356217193651563597197 0.8076209873014465676585529059936943552670151806112360581834822942 0.01153111511005522333731095300176829742785562765156502076168483274 0.0007702769680281050643283519568401426465706563186877407111700040824
                    0.130450575594780745452336083602925497969388798321807123384092523 0.05468226065051648607053222640937642400723828213520212530354178504 0.5063568111162120434091467672045219622636138440547122219962025497 0.004255137408285855413334543364681395719310511305245014624929630024
                    0.01090678797775449341343045104658988147538573627776092076289251018 0.1735537437654549053669266764333957222217410418624133827601753247 0.8064279411463709769739836973997088414361679276803385416889561775 0.0004503987707538812425474794060539353235403578254940257789019807944
                    0.1267606562542787417022299924001868522510336878421075395593821721 0.02485075538804705040836069840241928158730094582905675004919337625 0.2304685708693577836554206030914541058984916902942256905698823163 0.002956926622276354565163737836974622604654638087988355883717664082
                    0.2455492538917794001179807422150725035774647887578018261688617245 0.3388000457246144205483358782742466319100389971155727661977974383 0.2224036144541713256997448579649902265519249691033765505953953168 0.00862278129052219842763892991746922077228644241338264479992311038
                    0.3416761496917343728809994939167207671386495703185789389780734778 0.0353693445528059154120502618116778114863253309678601552809859607 0.4777549770562387167616836399294015760210138408937712996229166703 0.00391813202567523202794353881662413203128487012546476557694946365
                    0.0107756958813178316827551014680870327358412228520569008272367624 0.567154081214424137981910172237795213008873271663350822858873166 0.2159745800105105587281612642388711718577839049788108066393057219 0.001367079690681100475112373620162000104424632589572703024192894442
                    0.1187060636899572904497648319075999741145062538070910181820580692 0.6981829279490476246620608148876689249222609626133249935720702093 0.1252658003125617409999977986280849141542062146208400963920093798 0.003397627939959397341346051995407824461384224926965578381564618243
                    0.3259395059310473378634724568479103105886310171716347455101970666 0.01003270970106677432873042972646825827644211715485774897447039185 0.2957110794208318986229835098646651298724750039685159831876430302 0.002675471711816033853766935697970944730723835967982976061271506649
                    0.223969820078658326219871572433962166426070748153413821604771211 0.1216856850072188757542560195182444033956752988303206307255601733 0.1819800529270013795349824498846790750412923193755494979097178966 0.006615218903650196362260140280211763790780044211391234829612825934
                    0.489680753690951659059272196858530029947166106014085468192342854 0.2919138737233359836837258530273274376146070279479862948379951652 0.1226143544115844343867291635631073443440997737795042712074097232 0.005322981686054371277189268552874238586161638002744801912402302072
                    0.1255028469112535562387232834615022351227724755628201711295900683 0.2922528726476664896107360901615586269335966721751801480747449578 0.4126839474864585920817547816478792145479444588151454686362396626 0.007694938801642834999559114925192428917040622998763193376069497978
                    0.185362495918718521086813541657086956868404098769247901751428026 0.02534716840468262849253305513315636682378351645871944693540613989 0.1079879321360525333103835444720494940927452557942059442065139612 0.002478123313669804545310355963292920676553199091800170456285186651
                    0.002638036556522482198893753231333510699173686848462849721859303182 0.05020598623159046543926739320946613408978890934109342865423197636 0.8880860035988175070054785149215266847032065128273782216171770609 0.0003211425914577179047964482581945236255222463767159038717073685378
                    0.4936432039929027435700308278909511470758793715361855516915492414 0.2386255007426014683925832655686447610160123197590163301496719096 0.2634599651859282422717403633762804609862465634561332762306385779 0.001212966171645572398265864581918123941676244865632822373475704143
                    0.134258182101725046030843181922386545687610237714580477991725488 0.4767059278047778285497826486244746544062279963616005754633964104 0.1741301214254318448521115715346788244544495283559010611609681867 0.005584082970723647287965841098262756705646521212887643586029285334
                    0.1974084380663115339027521863944390185328385360019288148519496463 0.4544482207679524122057851687970637471614518748588512203696778541 0.04819434713936380723558855145242405001958223619999052900184284011 0.004798248536235073670628469599287523566480844017842622175199904665
                    0.1352327840037519796829718071319513355294303167107951706989854105 0.08809498011538329078585851163015188796897397032759509767790407687 0.149496184470844226538994208265233690478967113690164699682285929 0.004044716697696613635201450736330745248710888041275705269632730626
                    0.4538575246770524998442314579658708581402506090902686753665725404 0.1648163791582875378714187527214582930231785283977209572438125462 0.1322311346771809546293853255689498972585996075687158294729597034 0.006337390984377343719772027657618035403471417743909451313680439384
                    0.4729866579953057088188385106412353774443514127055806591905735015 0.04609989863675759918207497176948109211493698796488279039637713431 0.110010144193541215676117932183393394594280178260845962920735846 0.004032863455868742156207428970804252207103403029413771473392765823
                    0.2535073677146603053311436997447165993787943304543945689007845481 0.04789110726403224186992236192783189452170565797231925438023136529 0.6226262396502039179857302337651676155673191065390024672250244484 0.003192048377622075926439713405020815788740163279618977497020914369
                    0.1128784254128246741322137698123904915647738571006957743043164407 0.04575481446677442323894035736870139531582259550396160789611554053 0.3713221828622418031553305376129048581225483130609495028861213536 0.004252393060271645771402039558960359871900291222943320709063578243
                    0.05103759371918092230163519890437223326450747493753353540919537012 0.3333248495864574521081457485686633321098886477360225875889795589 0.5011355499345420723142722621044766018823205475797335740703596082 0.004954042479185511038293885407036157282913270858130538274867238915
                    0.1739112546641712668426331084953743994184818164059887921157938218 0.04435783180130465039508580202788694674007948011232377879922181881 0.7481418490795873660603458270642753232711929818835850592786530752 0.001191550473408334062511455437463793773700819045545894484985856277
                    0.04667302801124516521535747094296675914869638465163392349725548639 0.2484812942324444828398654182339748242114806936418232804098408792 0.2094753965937451436132884881493508613871760045071423473846232889 0.004349170442213787851219228156432519932075536874727339502619736534
                    0.07663729529409250788194021548210086519516463877417644213717667755 0.8324734053540721784332208760930288034476444997136570723536176602 0.05290518361367363875603236757558189245036132621609129734672787253 0.001592267037606894232646397739725356253832413048074619881398962654
                    0.4437285576453631124964210857188292069139713080619980846247432841 0.1467885175765642977910297139987299346105534489344992596081246309 0.3507748036638861332769759328498126555785248671078984131282777969 0.005484613681957023142701244614920430161175044228505122023050259738
                    0.3220987541780097430527053145624965169486666498922859558357133883 0.1075130330432638236131348338361340118708968257328325898422246851 0.4400083510129007156716812761030114954830417624712856208980230054 0.00585041612268404221943201457091501570426339819473833604611956654
                    0.1158301202881540362267663277289049190574755016797392327883715921 0.001057544627740350125768410377708434110315084849147775341016585778 0.7427876438982526457531733483135079371799179203716625337837685714 0.00062953509154178220486691255689437084990914894393207003131703993
                    0.04321756997123792621938256573199737143312507063996325910536565802 0.170109460637454241392455595080169721471784179962562689187943801 0.05286615398266317526665223859630829536794874386725994013614491513 0.001729223485215668515130804646186114711749254544833443492515188035
                    0.1816865847722731716449730690868529967761442816259653928676878804 0.5658992784654516483990509335333750414773220895740925397841944293 0.06328479308152410783503269535009091053629182127921215280242172907 0.004653634523128499803153222005820660859553079555304689594985673154
                    0.1125156082560990943315257333873896264863273106936231860212624507 0.07495735073958995032962803916315173073879775048821762378405603091 0.7684970446187415975984578268857899424510744237408972565395913481 0.002210631824167346693887776199953522473383189912011895807639839984
                    0.378402388858829124409986244284819219187892762018813715517548737 0.5214394197853024168560253100934348408651305664529466982790199581 0.04233510625824574136557820036737852861987523712628253122652937141 0.002856402092431170991959816920071023627460789344165269664884445498
                    0.3875211022362550035620438517379168597777250240500035994426408204 0.2963917734804445720503299761083348136583552842394452890492620074 0.279012499173038641028256830681766120876963494781558718482839311 0.003673257431440969506083264683719677587186484979652360213911339348
                    0.01177773369622730468039559792300212093357795024201161875642013646 0.3793551797632863994297218309598754161112186628869792486148052014 0.00994874365088474621960023387366552197342031665651490814784009806 0.0007191268213088583275002264278194871175185795426019190223351524823
                    0.2212379138821593257268189025279319174320035797035494282682918817 0.5952210542672161142963067506572988440183459648346038708130352787 0.1391889814757077950290269012960989517710996826777985550179852954 0.003619831598988452027158614925885048171476069407953124574804664748
                    0.03717160206584737448262770732450125740122301893298264916898180154 0.05150248894046302781917973092026044835696121319004550028881230153 0.8538361152666260020329192177119526392081410002454711970889434128 0.001371674283610740707616780274927213648278133918861993685444138307
                    0.3945794618413093058859521113408134084218739740487365593215541363 0.215348234324363332747490694590253435835548203502627052947889411 0.04461345869940123253369480452701980456279720187779379153146226257 0.004986665866263824789503395643792943509302193373634888918885031115
                    0.6768547588515363863639759488254419049419950060877286783477278916 0.08545647893824179861390612741710854946584783391011953122530559406 0.1187361606772159134770068001468420418018898340630214330647781172 0.00378613104041565538708597378894834574222899811970418094436475866
                    0.005454952128593440651298624261402913632173037711464361432363143105 0.07104331906929497741653860202778700848744667164468402237185299252 0.5299052335094277565701578243257515661278733776800829371611996907 0.001088459776834685362528929354256051569710066405820362519731436398
                    0.0428465299334151327758174773331084778427753095491003991150023835 0.8789900107843750807214881716225416973936164703236611065864648671 0.01047706673367209651189617069898061453803550460441984914879466562 0.0007497259742255372184654769853892018663987031177046251430567370943
                    0.6964945330743351355350083831665133929593634000112847320178676929 0.006439098659426897479825343130603541659510479026016487145202090863 0.04656409911343805960908694873899933086351590033642323944457441118 0.0009242359071577948802795208598266897065514968466860373673819904605
                    0.7707958813504378732171123462051439674616901182916840768548193224 0.06739268619543024538729783609999211290100302432786004619591827017 0.01028567232162775891682589499184827958206620011223642438671330176 0.001195113573161335035410615582729566765281004650192334953437634184
                    0.001980435661538682230183547468635837987777163910123224759951263134 0.04892999086999416421370029074395758135340384831276159554902328138 0.04082397785692029566037316713023137414338949523008912867423026536 0.0002661581243859050998851111727035526447711933430934508610430123709
                    0.4073458580372372122105798183326018454872808085551983021743928258 0.01400350221159151325978639123215780032889556954752922091893233647 0.06008122837386523030958103344754998548960194729612287881994893563 0.001846893770884993066153715134189131741712329346273269369406295973
                    0.2467440737286551612290016653342494756635204484998186185214372895 0.3286389610983160489742190901105383754001531390812013442835436293 0.05655124074663258323303683460817784070602519234792144968151925871 0.00466607724052726267621060027261324704608563114147350473771664245
                    0.8523742548878930708046449556059871375775652131928842802540420606 0.05733608877474997998392447761108864015529046206732683726478013871 0.04964554120505112692673378153619527127149525459590475253276251971 0.001336250959912812720720178071340172666498574151927225572470425451
                    0.09486916578078333747979436964335152582435558673694719643203877548 0.002558620507061097911709021449397487973070463201682405837452427486 0.07364287990554006343412886740640241249003496131380459297028675439 0.0005235480578796971035202818671673910283928167458158022909261648105
                    0.2048277365559580885652913341785160389327732847644419381920833929 0.6585441666897797164134045515821981645960703003535958384869190956 0.01135651127398575557417360407697522912138980965756969301265547048 0.002103497553945909619715986693789596783728120119204618797086801548
                    ];
            otherwise % 392 points
                XYZW=[0.08595277150281374198413937064295672239580504047525492126691918055 0.6087880055810979614882960488623699825732356658421775135675113896 0.06756005400803631841534168242089266879814893921860967719950179678 0.004154919804528042931062720483968200487969840137880278365128416307
                    0.1090015001138055731775549751919690182012199550520705527683400500 0.3875410751196880027090543855349678444665384136853483336093921862 0.4908707496497791651855606604556387545605264855626070262610070048 0.002392173195320248550055019410958092904443734624432977966569535546
                    0.2499824904478421781223656312969571523838494622623132641650716154 0.02811658745199531268976821666465933013919781665107009056716212492 0.1884641197882299684805190758352910758662437616264769519985668564 0.002218225839844525991205631724983106020216528477164279839765056809
                    0.1186764856841286904223195587736783277849870275569069541194894581 0.5469215735004998131435457227714340909109898368657723296544026019 0.01039457339221452664844140891033922001732749568149062705499998065 0.001889790943901067053841205340918873458318830131739228486067742906
                    0.3667754791512117087666922891767216706842289381995247597465823954 0.5280703583177398535467261745450403292174608360738834531606066643 0.05986156919148681556822045245638146086101206732490090844502211520 0.003109857218034563446522994436689312355281136031089823149184664260
                    0.1095746670281619424777792731520505992162107712804030677777027888 0.2078905242262133913418370513987522362765040528939462975706413331 0.6349009373874566668128092351994116637359136568897003005398239551 0.002976189043353084890013142177039426459202923898700012602901427385
                    0.1079279265194175193840689590833967265111702547579215918520268926 0.7636890005383312109761010412810797877285657581325744084176061931 0.1193727936581795014858994378448355769601395402879575098455598932 0.001158144908122907170993050794234284699820999511612403525650689511
                    0.09804068316413551291633985934133117102365951600611390393867316349 0.1320374103743844021416296794847506982510702738438021854600436709 0.04462834413595848953030763182898315086962951866230881794522606340 0.002673049871770348052427560182673289180761210906940895556856793548
                    0.2493626215901382949242026681348016399750930570299254373330826390 0.2049278718629509325482505333845617937851738089063065261180389178 0.5035724960597308278234823460409686869766203169878530095105465871 0.004183139279353761729032939881472574708286675610536414963162265416
                    0.2535577313390338766159833055464137031628386162264448425541869419 0.005996306289523334032734151083972535564929217382395459093730623201 0.2223795664602224930180175887595545859925723286709610961754425113 0.001511118995326505602221124310086290532421058692324786388132487129
                    0.1145887972243773234150550013671497116428288093283935300056127936 0.2512900519570757870038308482503349076961502347497765943495632941 0.6236657937033636170156464260446391744834061549442595765367292877 0.001589112021812844670451047854671822300965475601954277150528973406
                    0.005303248558078621110437030271031274278291857791531642155981196985 0.4252667071553375528431433374853274195936991299769881793791267147 0.01739559318525725072409679095867592367989357910759506808952976166 0.0005625398817293103709910776374529075599854743542832063789804288118
                    0.5530942054062970242691962539119447047675140128188819206416780440 0.04420537333658915413903289381095752726050233286370180725405597174 0.3889127679386170324497219146150404625040042909510363183568322192 0.001617477556246484808191879925569368696773509070708061190666139016
                    0.008858640528210462228049709631790424710992689249661777856338272966 0.1346483773461656518413786453939936473739654694283642448047391995 0.2614189239297324674417121739341841797217298243946994279291640323 0.001806545940363436734179641456993997403162015633430961884395465398
                    0.2041612071655005173278757079888261235932745140865098850663955870 0.06840010044249792410198755452536638340741632811436478265811328723 0.5414392656774946155417477118711911372490098220758024292902627500 0.005151422236520998901701963245465942232651832161700710954362719304
                    0.3531909214766853171780345294073403291632825106963490641858842926 0.2426836976010444781570587585747461230369388635063293112298185364 0.3941485375732723607027741698162231720402027288840823075501950213 0.002681626793238655468158222346544460727884045784696928327296586222
                    0.1306925295715786077945806275477507054233101909008954675274866015 0.2960275948996222745598774499967564620756191733721553384480185185 0.006891575419428691055333154678449531187555176335971512934569924570 0.001311645587712777096722295194130662442846332470461256971040989772
                    0.08037397319388048212952347258548379024380449072794304488402743644 0.01148155183936245803069880577795160861360450008591854225805829422 0.8576738136071622691508142503256512960987522901905901877102947605 0.0008496036520287252912697804608419417712985685397283241284871506465
                    0.3926761635246666396652016281688648047126619312077130160726435338 0.5248874375333257922174866975545980205339875208447734009713343333 0.01434545307927027787275580260671781291190018345944743763886506458 0.001530242588223529344565546529233129098231004480234461323358364501
                    0.04630881989107995122702230357241054101840411391149228266780016603 0.6576245602866498668492589949958419382370715651912279944720836109 0.2478236331247827400634199144397172940870139431908086654625378781 0.002379388509508912616987108682497320679194211338548672276037396232
                    0.2509342787076134969526239894849756606487779049360070071943496549 0.4870494535500439729433803135247213489769007332560849250923255903 0.2551435167245432521764585694589732844378902405087669055957949389 0.001677687731264120201697326896258646241447305152459392819065478643
                    0.1110615470548873763469971536371167123981646518870375229169993350 0.1854049593961360988537012080896812876511461697076364912002664788 0.1451753199934011836323691685569502516977777968496538039064735082 0.005982399198531061624344739153656593373888655481902465835208906035
                    0.06669778465591210133189518918712942938502547995651176783332423955 0.01138800805439519319486801774013157150204469621082250948665462667 0.01291544420324176540548077753735142111443652839150887805495667630 0.0004323750475775456633640382455885141659316966971118717313368977401
                    0.5801996681351095255885406557715246578831592668522661202496805745 0.1063783618294865551612993712945253859096512969469572931341483455 0.3041452709463955362208704451486365560854214250595518683185435886 0.001343574699229585536769726563824022098011863879870537065567192579
                    0.2456798487355095971629791784636861119641455072636076241892932846 0.04803247796216963742434624129188908006044623180036741368445112356 0.4114110085473692648931772668155167937714141297282824373602308652 0.004691725415859371970940643394465754650327222233588008769645265529
                    0.008599876339902938562160901978603516813263322740714496848318805616 0.3968824972962569352714652675511690240882610172771537240560825856 0.1039392064824559924553010344766157979309452207932935217917880476 0.001831937887689009903930132568418152571535961085201712100110351463
                    0.009820060043474005139288229564995662489063257149685264894355571766 0.3029227834759397333082798685888611011296962975734223360108262799 0.05414032376369956386321409236263716008165825326116787659708636558 0.001239366000527990550950679379948337434722169886475971687215969960
                    0.007301789777139665329675961430426517959809300024397743680441102531 0.5323789599726438147777726085601240872102928038456551946811228270 0.05162550570283685872146875333141756035189657390429368490741761863 0.001121605430270670479584683645220584668892178510287945510997680237
                    0.02089577756829043208678327724494464464599339465151148927356916845 0.4339432784726157958388429971756559190213837989377535835108166169 0.5102442780721617093790986104752405482527822818356877588199879073 0.001247657593532815240006803671916351662694355036799020182163833244
                    0.008638532936710954063925599673817759747276476018717834625089328205 0.1350178628921903862537421644488760231019504513285637855292106313 0.007359526036820204367440794529165002764390885645278361859323529617 0.0003063272220541228645058945672831110060889653083890067151924359655
                    0.4391415004444721258199179223469027775758134798122567313131332183 0.1233026153907862091483167267510022357118905875815250600622647595 0.4254898378110078966244337358750028579992520569290439720471467618 0.002387805966079371057772942264730359364818214189828554970636844193
                    0.04820934013846051504944403558992290354132272477151790820235997139 0.6706311858360371940884207032146217334711690932136009315228127564 0.007354824872404593704861816045102317884490313170483501916405623624 0.001057065653405506439352028283527071590837720495334846091152394364
                    0.01569903574073822845429883306382699448487280500932160795346374957 0.005267546540388222168384695180745919885058677219499547978604155748 0.9253467674397920721804161197846868237676608364792791520090923238 0.0002518387932000269295208409709013959647802081939275581482160263747
                    0.04153410195195091610226581671748148917536690251525568266493105966 0.5813863028508816237647724006783501826516735062806406935521752618 0.3658808978633617252115431475406176606975265456335574600159863129 0.001293133948575002105370088493223910040016131135572076046088654783
                    0.8428202507788309505341577279911794850938899452128703933015017282 0.1332502311861600315909706296768441544283633481518420787096726098 0.01173663074562809108039197446372819400085938948866267842977587048 0.0005408176005024798819380401961531208915784978932175433723998411100
                    0.9632590443581167087781739182526421293391822354207883616028065961 0.01004189887907351352288809654708716216883658195386289545546845372 0.01363345124626824696710525205727470258133049578584049145602079007 0.0001773025528700091089855113433226028877070406655336310161375179891
                    0.05223997836785756363865747789649878698649401993669822973747281760 0.3415413868098625002303985513653770500095361393288050928400745187 0.3713868630077356144369855217317711615857846770220865908490746066 0.005077944252431479426408482304696204611962630906637247504741990207
                    0.1865012228363573960528687710077722102779366782782650932371135373 0.7938161523236986719359220752471831681471671911955014552289762635 0.007663816491854051962629553781794311490631692990417958606711881216 0.0004513612774712552521556195092440891602604084140059995616570488001
                    0.2810777519687947593742253247920166748380625072594224090094390290 0.4407764737197624101561439164169198313177673172683890935288156626 0.2311857683150206093838030292959064247667037245152819628177995609 0.004788077836025001569824401895759753211688942685337309328635362921
                    0.01475939271070616016394266420459208456983952502300181940696259179 0.007410473401502261894632951366902565803852105055654580017672517794 0.2952864824637688052210885315615595976830035894581223464495822943 0.0006444580077492345925751970243103942116252295453415954623867304375
                    0.2445332440268856145681833974984972542309785666232162192681590371 0.6093601709763757599930450084729183315335889313118849320046941436 0.05794933940771363598160649096260016115575095445789103833767481111 0.003723036085186382586601724002435886019886731000580419907240927532
                    0.05480177763581717911324104634129162907023718612372207928077073189 0.5807354878251340384246580043302267793358711019830688886647796747 0.2364740929675533274514352282678157987785329486955574229910236452 0.003872998584127431155766357642862103985793937887236039244861986354
                    0.5961735848364711836673018083318245710827638909478122646325877978 0.1579228467757785042950887385368901361585507638226827528533061272 0.1134413599281562062220451078610669364078965821800856898933378774 0.004644186587725121610681209242035476653916350564691286361324087081
                    0.007469520318630954269337440867128446240239253297764852907304079494 0.2725796532873784418834680500768957739999932751406523066690738076 0.2048241605199215653707987594403693574399781311702265667090814074 0.001739080879222253665468639191385872908710345663837157456441866862
                    0.5421830073368653222271060825800790080643901561600676177513915938 0.3733185320385800252444148170695691983391630349264198183050841489 0.07610452981482698230853339995167319281175085952895364469110229973 0.001340642778975186174032604076112854209495384656278894383078215981
                    0.1224626763454930656258020222197444559990951715942433183770187994 0.03931535914126610400459370136302305484326152534122922382527956878 0.7481825694063941283140410591959098117733288430925701037630984997 0.002344213367199290446939248884454177173355251608268057578210739520
                    0.2629499827226380312713834834515523274478921146139071886725447455 0.06072474263454779813724104241946443843635892122295783448790676410 0.6623690178291948588964567971227110113495507222794835405999278265 0.001824828601234347254068649832358927163596144614183786879694458100
                    0.05027839033989678055518649439970814893381349368458985472888480017 0.1342121381308954061287618328112340129077501944466083257933981125 0.3198378002488488716691597555615735106689068679407717219110850957 0.004507878785961934788107709133924423156083208820869674471853794918
                    0.01544075975902627783489279316297571859399712331870145417204815360 0.1147629966750915921753711524624672404584210567104417882439924139 0.5775138005577409334205884003430012365933659808822883485656216259 0.002730566358951606100185575399135252499199084169345037802967154043
                    0.01522965276972571457913930111612566927559289119412660392714060325 0.01044196910641267301699590929227126223768086067982594829553687515 0.06866907111382132493650560457646608629240942225626934438861195784 0.0004777121283297146391273704277407973393908850422205446350696721468
                    0.01174222368539725138730996383577118123590227651826791318435328570 0.01452570228056949918728174326876551148406722627107485096720477052 0.9641583619145040334978107197585467526634965182714853461515452024 0.0001627249397208698775809703054010244741029119901666262737573016663
                    0.3499450749880391808925111297611631028656208747701664465874670674 0.4336739025993947220709799812896587718371343967413664552124765670 0.1191097529724983877336355888884302187634223552538350262292616141 0.006028491176376566490033021244455867322721726633931170050328000862
                    0.6671424378783264753156171795916103138045383831061228274658611414 0.2556156228578073711730518942385273409753241336496898406014935732 0.06715137796155471109908397912284507382787182854405990495302424020 0.001320345707164726367531181441633084462341610276752602675920095826
                    0.02274867806218675992582576413173282555388391034181669278714856000 0.05423641183817120065049331021556398403901173477712631423500400005 0.004544371879441360584233762216894108877014311213245481131513658448 0.0002705353208381917607068074228118935872223733877402684408657921566
                    0.008471306102124263794688029097166283624715376515904820497834567934 0.2836260822168098796848341631754206126676530674304798538754172184 0.6706880041862203627557525070876007379793710931277939725237440594 0.0009315670277609773137791512353843073507112757283974981416973642058
                    0.1200460685829433647246576910898983616682798072439475829279301313 0.2113495978490938830056320254839540502292059016092586503790098310 0.5519482162972382131181615310294001471587937819363658493363761448 0.004401460998710311039923867840894473841020605277617453682172695582
                    0.04559242417494775824502963017269650056013888823238571918332773478 0.7829871296943571145415280446250889040573119237582325382266857616 0.04889424372075302481029294937208211690424987095820275754595667436 0.002054249593322150141873188110706117345442720571688721219191734076
                    0.1802020760228830145667315526692723825433750792973823343937695917 0.1343418330410120035200537749190246267773172675067320787059443953 0.6467356096430874239258942715610104578477767679178256591410896108 0.003296803791643181618112682277843613872868306396679752677832268260
                    0.1947248590240765676003820609930218577079677341324160609995243168 0.7169640578877227981511396023997682303410156326027935663296706229 0.05371257687004464043476896516947764531656257496372818412485987915 0.001953780204933396636472723644285531011222091868555463725068000619
                    0.5344861388927187636452742120845224000290831083353471843232956378 0.04729063464592977062849052945967040286336689665855438874427095397 0.006655520609117573512616376686712760849343508980484157560136582197 0.0009588825003287664413801156894305616691436396419073408657656697990
                    0.06302106813410877503743894323924236463546602420327604195871647771 0.1548392583946775116765412697912639092216452424892575900713106288 0.7713477631162040071178980977187150181236044553776432330478780711 0.001400888804584954861228502042704370290021169634680744959699427252
                    0.03529092798423355172357988557807930536015495082788278859741209523 0.01205784831098427698238130012915433171945602422526448292005820087 0.8293742401132130816518220849196850741456776939591426854746461436 0.0008276650521429633723374944968172878847075662054910184130825900893
                    0.6051643799622817773751554202677376954491041463220758722647596288 0.07003929725454948326958547088078778478599323910380087776604960199 0.1196411143897451059339667606892745250188533118000534922407031769 0.004644297676778874140900081736055014191873347415055407324257178260
                    0.3185816136179423000133155383138232200120335667710908256348617174 0.3007265720208951997251065209441409281654340804373595291442711553 0.01182449863731334043971534393018023280186649323344373268939354489 0.002613022236483667155089286470879482226144048050528615164460422636
                    0.2044526407049172195817361954820791442816860675218523723707485338 0.1208884149227918943102266577661949446590978085616401800110003845 0.5710695009747759367762497581210664761038841031944955787203464390 0.003728640984232288037513131931571130819913931889251197747044894481
                    0.4401629017499941018894853443770490445342021732987552367181961119 0.5365629678085231278060329235235172613831667160523983825185025498 0.007783755496796969803889451341402892739364376918025348929557972982 0.0007212847787317560640646027222979203736111591630764635549415144109
                    0.5831288659049985816925580759048410301148277765299846943165092464 0.3910694573354751008313498766211826065398877032579252988994589347 0.01433394821191756426512283046055731066091145423824948638647100033 0.0009446424834421033563088280426958377298913334048437017803331461635
                    0.1778544772590157550412025002876691046436946387485577636065906261 0.4291895957414070962551512329924557040402983960390476936946884804 0.3489151414894534823558637498596254077752465734082346420217930573 0.004378573573093388331562066754163256082914255926603067823203154269
                    0.007040707855058582036282180974488521598519999266112648818497427142 0.1246404933309793874790039763129331697876153479754655291845528936 0.1439990977412196574031332510757390606098316666933658953084954874 0.001154731741186770888547651169780348429594498058809517165357658889
                    0.4052760795912554657448173377031536299686721955680841308678193557 0.07555114129007707874772391731195122280892071700564208075608468548 0.05347337847161375668190466032362944138084463429531927453507029061 0.003917090588040402702197960897708479901146462353791656269612939317
                    0.3285561997611669527550571417373459020341806533168790869146013826 0.2384425613469881199980737993282032246951794779453846571884568423 0.2259285168664441191659925858574623983710119477566350697144204699 0.007546521474122915472248060817372045349633892855552444354736759653
                    0.5468320819659782332390975917587378219276629675896238746366011920 0.03451386265000845421834771628590188667000794180616467937960123919 0.04041190860249548476628765270688552644310651829974143289374302058 0.002212838895341499494444170754368598728632660538030292046267877063
                    0.8813517574690288774134159221437272106630650307193074919157127202 0.06235984467772732208436029959370510095902583260476877308531615005 0.05296881642621043448860419755586136569163313340417449897215972882 0.0003443622370410026760702620761379865678546943662845249692559282175
                    0.2638028978421740738546419585629496694867664073778963364104849300 0.4277583771357004468195280803656499662564117683221572953259913769 0.1099955887700437226574771369305381638880872398443369476288344392 0.007126260158402705120989357194517282721093655431798420471683941596
                    0.008159601801430384997311564578430926888175604989042493897982097877 0.6925967248463685578598796248861701144350559512063058487356959761 0.2599325058810087819084367100072315985393902414886199626452054378 0.0009179568401603836072482767519498456700065649650437819965829802235
                    0.5814435342797790706323862109597388726664655468424531161993274572 0.004995108034057408129826579150852619396151038673688346707208645581 0.01229215000497998719106502244849970187226647882757968738896298175 0.0004420604554268965965300932577475767748986695184534815975954051452
                    0.7801818149481343092729922696606415315461022376571954924779772763 0.1435474826864018901805261734064658370886269148216082241275600882 0.06191976985353933780574921447932055981109154678936218928567665984 0.001500731324415507921017341244244005978272025408575638607979222617
                    0.01045797613382906196566845738587138494228625685458879119546468255 0.2570353115110574762060991078930144163326546701070418619608795896 0.01175829634280920810536952393503694756786163223560337095052318630 0.0006579965759751037209546676595874492435332759452026562137136456604
                    0.1316622137763387504780998785338762648561072360751766742250123661 0.03460765267952799873280990077672727662588244341644035726244429998 0.6495446778272601888052041038904636305574650292129342529782445934 0.002936607268996233974670194430542182058072775480907383991154741038
                    0.1465420223391598678809057361902358769133870850363813952657325871 0.01109581163982173430737924762513082622639743811442286801058410198 0.8332137883418926198760249707642461498643483134675463706927082325 0.0004321181597474436535951322476711014280726736069739742223573973885
                    0.5637837545997321494920725013389016249976631848640051185856857563 0.007931639852386505027225860446264147944184185736356165380707941918 0.4176661181264085839711630223739380716438002840148306971987585714 0.0004897259038480388235410402901757361799543373873457488017499665631
                    0.2175978554027095234429452236622693051144123199945763508543215812 0.01073789379295408818735219292321641051475937991284156526198049275 0.6352700689883631276475429799725016128426564384404519235653770192 0.002026686741632794519883981355272599639524610738013061622534329541
                    0.05140095362756144696413183571792088771361670358150255750363899271 0.1261758320514339772403246254316070945064876553551381864178223982 0.7511362354515488576657338675127283853448746723990405188716531346 0.002535866322395084023677012500401475822228169381624615415618031673
                    0.7579659858092529390472892880529666380332013989768483401653214160 0.1243568190290063017427456837105631630069288698769651169667846661 0.04098282528926012614692115791649547061388775732581685114054377532 0.002071352833699424774503413768189564890256570307469701839426358813
                    0.05115750658020129539279430310425588473572423815630490964225327667 0.1581845078956141280932293215309653630972513600650492081077425875 0.01069964403526443800203676881733902049243624232264901010115972777 0.001143552252046331459266249477892679584986453208578003370857877270
                    0.05950243403143403251586568963078758862108603179007897732198619658 0.01114018207068864918913999140979043905948059263238107393183469297 0.9192253647730895309990490392401036371569144978846340541278617113 0.0002955791779726857465342348395345548228337937954724664210768645566
                    0.3352998257375770724159746643863058178930374143066361759534443272 0.5296584477784021209420409692092968748872588286946521944496584430 0.003985269746681355279446710402249293887415452173773148505778482120 0.001064303309785997953803757040752320815795472679458286710910033335
                    0.01278302517997949553331606050707731050329925634468353546763284853 0.2664236562452565058681821676020973796357665746219656835715152399 0.5009362849459034559689308216105569930221583230994025889490065539 0.002459393225084751576819101050420897681639275496920466794323827110
                    0.004642693512687687122722941120307811528425163715338150985077460165 0.6982364402587273396637404150350945201239518015427179386288728201 0.01363264693386934909100232741551441424956633410708351240749515544 0.0004591831081145950566850443386468876054442606179825088674176462996
                    0.5878786516537350263500279183279840151001241846362932103849526769 0.2685113853079579027818324793341452230683336933261375089084493020 0.01200255444086702623412169039701514884785713720952089104817308312 0.002265912312958077568841560409641190995727929528153541039193966757
                    0.01628179437735309333193895884703357727111132252242981318729560869 0.8690663749873522304189710860019775933812526252027906930710364870 0.06167104836854103336389550476386713886750028047661448534631482897 0.001078426109190280045838546178688579417546084592245686083453760675
                    0.05280400989327445322394415604849975825678534751714483522962732409 0.04339139561387539192518426991127882083261395911452719483420925218 0.6149043027612636039185570958269385723352980202522397068445740925 0.002211061146358437222575245362458251495194279571733110770445377618
                    0.3798176827008322407185815102891604730380219334902567851446814982 0.1793288346556778995304343890688295246799379091514137043906266255 0.003708744557399153872473419446613033538271964247351481603522090539 0.001187204878820968752759295164349055037423086974553034915505000704
                    0.01441473942122564080945023389013585146215868713588987132044585340 0.5602578442476459104626861192914140162946364648682575596777308808 0.006933673207687182331041679661827250853930915781345921338156070260 0.0006089351711723383782251081623250478871657492346636112214866289449
                    0.7995300338248803170400603329103622705835974943059461461459930498 0.06075944280360903542870154921951773217857991850316121198351668171 0.1300543075931590935765888629109541419809349085053300424726965839 0.0009085123534772494667113854978673003283493934778584059796202684212
                    0.002210561072545111195032127293407722811036313438395520308949040388 0.5629611280058701258249645729069848751557264579803819631645015303 0.4232009026748650281856431464987122269827237025438443263924037107 0.0003606911422997412440095838996313362243294590254897086855659721314
                    0.03951596086500149701110629474121713264400891079132541669200887423 0.01014458747460872649567785243524750751873470145068325585632993599 0.1573677007230223367823385573972244945869674843741831865077408050 0.001025012885099574564803770372143717277349029026771593746256236201
                    0.07381987783966418537531930831017418538160952719358132347584387984 0.09697452545103059835982640345853816853093859244526440641125733705 0.6028503671687967370999350812858637877114060493625115303098224623 0.004139572157325285630862139540849604201971294587432369900852027726
                    0.3850881000679209115400523408590221860240768927758243488689130198 0.1748657443381481199737831835672424284500298162260296466710393355 0.3855356794009747984628494049394328778235357888947886911309918078 0.004774903022923129758505661402485335278814666803260474902375569218
                    0.03462664344639385856217686394254372647389865599707403053146870517 0.8039756262995586876024712840612233490569452542615642604629486695 0.1491705721861676950478830730587322168106032435878751652674214122 0.0008995742994173481881445968188121735138867876746682986006630237706
                    0.002191050090599287050525502796366499860476608770519844700530026619 0.7928633463325459663794586166109788019884488871242719180782050700 0.06030932532024692779741189624319932575561147276655685003093208252 0.0006394018623193146851677450869283715243006134131963829399222763528
                    0.01477805160945973580027708705943819401653784666200679203203901959 0.3877665020878549108549715783169137532612341692235067332970583872 0.2123102118836795641250524781533909374343103762902927837447263737 0.003162097660899969769207800542560346102155489323075248932238611375
                    0.8182674839945287336644143034701816031686745510285383772580924155 0.008588775014994295443755894863699646728428154614435223039785615595 0.01150868652902271746106908752342153487259387767207513148504600915 0.0004565958900106349216008414496780191233177663988898585441374415682
                    0.002506311197411498907687647518828542697042384697750177633842899942 0.5470030868258113577179275986770744283600892573852213775426600219 0.3308168443170016485202915920223973269754937469136991688600065454 0.0007719023958373823260345071170085984762798607475095130940448234650
                    0.2110561170781058002325933971132986642534879089200832121472563301 0.5454506960615396951509003392808912174443089656047753917974320981 0.1368944153874918805622972631204460963326770444599978645056909007 0.005552109074592875296594822764870079011561271825501482438276571072
                    0.01225124317827410495406092673313973203289807458800925862631751872 0.07900377330015722197094372556009764683490044555793613550301692924 0.9022036017187264642678883608319861673639517728417459851505462154 0.0002874122932779832543692169617848252167495540217155628693889041805
                    0.4314532022011592167359435555766615920484953611456969090299776757 0.01327362887228967496195419100854229324098241033991810254562969524 0.01069874101877005234437452768841722330464944526828347609138701910 0.0008187214549321791844792125471414830305780759063321225370679860703
                    0.1864784448587538716353834583609059027478107257500910334887848676 0.4752588179887475442955626098401267442375480293745656712553870820 0.2447353634764088566402539749155595466500846648555057923656035969 0.004795659494745895146577977355665305056165634166279527676295792131
                    0.007268353580998054450671398853360263624386454834863308588676130224 0.1750706814716547716463808804148680317020977645377216909997887842 0.7355302372802278854467374289018362083727141865658374637931778814 0.001023442479277455546315825589931851338714944806323295085268004941
                    0.2928386779566035233137068333098086658526132386527926148480849559 0.0009770706508639582592971219244564967097631208261674565285068369271 0.01966239051365186305705268246907198642609167163956230752666843269 0.0003819794130155829306082721915222294875810249639983497269108069199
                    0.05157862786612776146723875598511640170084594891548495004742714561 0.5538818633882494634348136238945959735034810929713531457587877269 0.1557959792928289107427439240564226625775238592097117626307291201 0.004820993144050171514733495680060210782041015819820832359231642856
                    0.006688590528920086293333589370479903315884503735434073797590370967 0.6484525187057251146707112219125416791275187123722805867850345257 0.1487025511341665054051197343148980910475208817588514405726681453 0.001289968583599425091484914786158610915026500735296448461743382993
                    0.7904498261393225059094410798065412992237521196614659129117548721 0.03935639750795455543060023641015818248107508581878888277421155557 0.1149518514432877059933041912090272890637411004268620706508916077 0.001459576049106601170334860840848939017364231813494562657527613449
                    0.1449043265160080100135638999576189176521218650885620636576071298 0.008459034955408793518379473790974846242892247474578293917151112430 0.3070577237326287267308987667977949201218870699376470228282127641 0.001884406768758861775570072747217772402627878831539910984900274772
                    0.2047709163361093018292554687369973419004474860197658177924490755 0.2607927579207870873946872070320795991225045533237214831837055823 0.5242679463885313181324475398857928847823084655560446649638123108 0.002024741618965744219506336111201327406166827257396630751925572340
                    0.4405798942865470963520037170634833392312771830957766755456190949 0.3769355985690362637636739710449714675904808673810851168186614321 0.009314760342423574035321462701085274538521882995825877799309737572 0.002160646032006107704417686937802998185377955031899474230342586067
                    0.2067339448327233245983097655662793807777695311080311253167298444 0.6438092227779950217078129214009657475581812667937071921033428198 0.1466717085096166058961494353362821101217349968047068872706697900 0.0009912561963656035891021056862522184893744184276964639651430152242
                    0.9089789965352639262866501660642815159938632371383255736693386711 0.01237934934495957169147437675424535364103629753931951088117690066 0.06770719361313174777872805988383810647910751568788935888980793281 0.0003876575612968030193332038895282889946025247019681236746316272954
                    0.3392015006861278913037423795111113275433429926110687223615517093 0.2119040475775734223359417094148657363907179433140131254784197359 0.3218237825857956133211051774049301211917083832279709055389176545 0.007358501177196831902812377989831649206030407725008527740794940797
                    0.4922523234564637034533664153536275627576305149061936354632595778 0.01242901628451939900588793328277065457728947821268531735021604423 0.2326302796620177448513228652048176703547052399773914953718849185 0.003076555820535671054402922509455169306645876787307260306433671889
                    0.01282357714184942412398965775104724770302730769907070274301390554 0.6520458920992602650320143929292447102147869855407338999028905267 0.06675405155039178026747095162161586018670444865015444388224415848 0.001617075947923160203125583253033722462000281913186521180011258619
                    0.6351689596106041634544866661166598497479153855580533786027366124 0.08681540162853746771648486526224104159423381344427952263126242626 0.01069993412255101318349603290907538754094732016881990074775683692 0.001806338452597640146379424904750447405553873672989982253646011447
                    0.07831683347882310894306057110545032488603458150993801461528880790 0.8985857500010212248737057463982902618212922084963797491479190297 0.01121592381919861615352478736042059723382328738617435291286855617 0.0004561404191999593801304178911612242497319783431255041757625908490
                    0.2841940405825054787457357910871794194435667487570436329263229083 0.03494080129258559406587487789400083776375387879706057714418055232 0.04227480016560775929118258577883388664687366112133456238429405076 0.002593422984810426561906492789621666404181508002956601291989824014
                    0.01303936175883619567778644819722882447376232873253413927621712117 0.1421966924418519346700677709664078789496858358108138883250923667 0.8201576435993507725742960591689078803682789061502897376179257554 0.0008134013797783510247317113491493744928780672435256350689993475470
                    0.1285541919864463060258493546733424365815689023298016973003427741 0.05372637494080321662607772426215219367819657308174074948973013998 0.04779545552382299545015364830326151191182716092091337027540568034 0.002140565717259238298656741939168780759572799955549926311810879768
                    0.04283035398079642369657437728572989975796756502165555744474479864 0.2357659191213900489619725273546415450986122961188487475778384190 0.6719704061637602439726567955388234056001580943348706042428688249 0.002159033692514621545745795497654634596279193495843130516301168482
                    0.01340544403964138463260027479545906264060733956830473058695669740 0.6380337394391977799732471408345714725100689746039662336306551969 0.2286356576820788148584840906177049570533673307158839773274424878 0.001825086565061901305648904308801026866125150805958921799611118700
                    0.4202750236176670069732257339921362617320446110489949158520583541 0.004855529159701364993227112725301487611118067117885170110848510213 0.1622187540343085742164430982667817077620665206090135723623533599 0.001607578505936310086762394911443488629963711898753643216280085773
                    0.6461010093527092302772586624358241138469566711486040732318337745 0.1306521091680621825889765550860848793369388998573324292815344514 0.04502017775157783465406364378540579833449270220085195114212350015 0.003726274394880948299918585245122050753393925508825828918580086200
                    0.2719735685638001918751602706951300279841360509018638033706593379 0.1313128139505995197664510833743263439975198892947816609006630708 0.01059454921959053403500286376820504427781564409122404677910331423 0.001877347668801728396713651381696795094085045546577988147946434418
                    0.6750056711852731679840190422639109888716023925777401142409959982 0.04352568807934990272574997841613811685991129854881283125943411201 0.2400922048057897345346862128238877499130376953460750386434457043 0.002377354227471039906426418745909003433303912024629289299748069844
                    0.5396089666373174259995254970146275518461372671245450796626008456 0.003136458633600789666329107925135688556139935183522586045571057876 0.07037937257169374961661259817610449525267543398623382381184734018 0.0007800748143043201213763864603054015902848934511966352543831440506
                    0.1916607189055072538071152935871553351733387421889916631043005418 0.006174679543437606713780484538248069379835847416597374576962904015 0.7496377679888276417760729017633831663988483203891123685367005078 0.0008625159302928536084426941444516464605926577330193278703844923740
                    0.04620216248642328980769698171261334764291543411482247980522425973 0.5495392447998961959802862852100053156341964659123681943604206913 0.02789537988910653319918631828071466964030995497573384717645677650 0.001951259683067656192691478023954535018601457421343217841380821962
                    0.6588357718436020904788594446575101598523246215880953361636947955 0.2878217951180066452507539671023337321209814009779984769857629840 0.005637598910803735523571567516023597919005922481171279695457811050 0.0007951884447743055813251416717581239111699645705442024177179523163
                    0.5160858623648312528438522311233588080229236634649883118107382585 0.3798231705548206118565792130263594516112365448296360090424237443 0.05473564321294446454225738282264437541310290394305913958115377334 0.003137972126605337619645528835427852685980402960193148414842953397
                    0.5286059251768572011028566195493325547828738391647811869596302485 0.05642632784072538286937410396359267255178253201409895997080012988 0.2394610020962577229787830300781975146723489520617855260880827333 0.004241078010944259740450550621738285304451540803848866839774913728
                    0.3492524579581972465694909168934724422579089336754288422910980706 0.3325421178681014044256468967484440797467833861126932805747506130 0.2191115858721736124092177804432716545706821461562272418636408097 0.006257851396176473537324321657199011197210640355885157394553001918
                    0.002901946251734448546240182893182414519116369121577331799019422929 0.02979506133952605884902649027700200785364890688409375829544775363 0.3667751192760232956154013901900954151876652281230502503832248634 0.0004079094863016746394531088700923854010320246816087832392146581090
                    0.3691049227623072001588299167111549415570634856755895254826071511 0.4823296350544467413205160451317593199814169685174899189091848432 0.1413946937025519986974214218270703311229236827785833110806588210 0.001874407641081192440018763721256122564898999712655456284793881838
                    0.2723925235064168084489256487616415315101922844537662287762836663 0.01145361873201993506291978922055571212439910162842576852047556397 0.7079460781363003795760827363581433949137382691738179345388502014 0.0005533644193207775378007990149027021596864890501772428632733495131
                    0.1390634951650220282670945229215726795505619912580215090944884564 0.7913627858715787911746118691931471480509459651904270811132195167 0.01123341100732451638228342486872289669886475943909645409337785967 0.001232081788414762182895518251346826075958983182913872294784471286
                    0.07705444585215993186437317791896296627049251937716757884215917542 0.5006102692901794340306494248301632738091005844836144558725059098 0.3742247150357053110673216515229708764311006412162913528644077010 0.003889646799908942386270240093885060757189738500136706685487289213
                    0.7541315958098894553882623022128586160141556944366197707434674079 0.06073391759343366854779433718546463929274922649851664061206385917 0.05963521854223580915039767718370334695568438112473547461057805003 0.002227848469588251461071900637329085822239379479669915164222395193
                    0.01030151844817355650550339396389448813545438641120128286129735348 0.07594269358278959991847349058700458080255368781981898236337288946 0.7519657937342242153685337128429018282877071964705508263833018543 0.001179781364189440255369570783493385183642934729877448734832397909
                    0.05313078548227712311783791419792505967368817991431310304496484889 0.1249936045483163687628341929802036523606641548332131211452209463 0.1025925025893277419881780633574765280196342475996326398007065051 0.003004953970787520872107859379827715046487074469932706340673809021
                    0.1810976535079918311482402682821758836906674201036663652572364393 0.01213902885475821623496774857232141916019504053689034521959383662 0.05946434546560352385907124439461418663730736339794061155951872963 0.001241115810115305453477171500321469222540167956739988501664302195
                    0.1963428903850944100632460486793820611703061045816868891784603889 0.07123531158242592568956300562231877430714911489138299527902475547 0.01214345736152116655135866091883948879384034100288056599159845966 0.001462219653619987830270855283651488176543479096403837374368906858
                    0.05772510078346185735864436268392513942817341228607596463804882602 0.02745497796541263483438715419229483400593227384160364454497670516 0.05159601744844887471902986272346450820906127086688586093743680143 0.0009272319924537519430881659997589846793666067484398554276841720522
                    0.009270363866212998347051706732658192418822430126624005169416197057 0.7719108999512577893000949766851435880876110995404109842290690352 0.1439495603767781647179077643438663725700939960957000020070656935 0.001195997343092115502371737039521253581157360114843402954845256551
                    0.2831397691859082883557338222674391000360615384212195343772642559 0.6617769771529046860485381935874689972004899492377436211191263433 0.01184435701790652228358137324099867696917154439863803501186607887 0.001455181289096082818109739659412583881881382942228754890179545057
                    0.4817430322470478397535542737333187376385697389696729449226374709 0.1176989494134170438908958137614601788912720454507570952441337789 0.01303662305631866334785956109192758131158928556170399240568311548 0.002017345750584754214224311538395949874969862528581741645576292111
                    0.06024903424534146826535270461899877880134560621656284743924261301 0.05923752693540257183364708708360722606299274634288036276210108290 0.8688018790428956552500010025864779638893650343029803371072125351 0.0008228000166966678775535713289736681335996462623884995014056610964
                    0.06845508623569607714582606285840445885436441770950416101574048498 0.01519397093856834590363974873443739533171466238851192709543389258 0.2533286927454695435919621939707510950593509430482416560692008314 0.001887135885845097594398506442210151977519091230017817999507611658
                    0.2171398253282302088544285351693939075188267155833047752604946239 0.3298024031370629989410870625848543802295505907912199992731475667 0.1243698391725922041809507790860385484447406855988442426934350853 0.005976512936093831402550864150874568392798159787418301801876818582
                    0.1079323789782269961046465521612390203144653418762390068954920326 0.6799754300852289959969230271776386213065089355285469333386983129 0.01833210218249356042338088705335921910920187919056599090282732642 0.001618420431245793055267951254207990129330069607791337000389292045
                    0.2801256734891065983686784495904683620309348556292698588943451204 0.02722161394932391655388432249350034878888932143464263445950500059 0.002312869183433803492975958498446016673529867419147186663514843118 0.0004728734480327994042669864347125679013637181771111811337780926883
                    0.004822024286532813072249607506494969188157170574123856789380212079 0.9118570107000847918236104806762294404451154076308859228146169616 0.01236264475395773147346760920314770648673625546526352769556070738 0.0002499249268523169637361878108624248036939042928142291756701283293
                    0.01422958683489175295602159171330588695517967961335211512597110546 0.05278387503603278933902329118312667084485722608962715585378231446 0.2272262201691643900546822782431169111719566365723937532639553945 0.001726622610181463037782504793232860331408938948485035893384462365
                    0.3582953735150841094816429434801085757167969047830675429335244538 0.006010894163038834780475351129123965930621435455521164639883956603 0.5187157291675721020979920482016187923613110687117103749501163879 0.001320784168672946408156527663680112997690106091768345191495963773
                    0.1334984809856116522720799198012271521751057193831415408889654988 0.4384796294350961679327398016368850704239518187217531859734724257 0.1225339163268590297619076800518474702397959124521940978655309609 0.006288984828167509483080445093576914687329364681215290746404969494
                    0.6989769802538035231446321856725370354144753367193549657713327078 0.05757265537381703206991053067624381816625971056914447913873023647 0.2396156664324301868070394082959611057721844832680544239805139056 0.0006722827918496559291789307802966046037847586506568653933073703274
                    0.006299534650874357785828757790543705262716904732693642518265104108 0.1416031691511888023175347850740840525643790948825434460571390406 0.05172141098466921495077003552001059502433829533810145316340715763 0.0006643195899892695533485636623542669921122324896184609743665250997
                    0.008722971038270437668311409380816503067341334863191079466379792779 0.08482484148453630173452835593088333629740036123076226118960339898 0.8276621843008150217369734565832565645853444108395213261315376896 0.0006828729129371707787671843147475412209643688287582839656886509878
                    0.01294178128038901227426392269427696008255697736598344608548506227 0.2798401611236698007017856757054937020655651476082348626234565572 0.5916452451798397834699898504624801382389952046510515806841112832 0.002108163495207963298889774310474124106847311012296135799650898000
                    0.02917428304966072965852375782009905548994344709116351280153078592 0.05835361609809111898975818789286218816466916648653475987374710181 0.1218399977636024164683516196931379437485318397780263055685169110 0.001342756245962379428104405259259441791051445815797293098651614393
                    0.01237650753191330767342101905577780595099814218821525166093827833 0.01013924843380992385571152810006136217204033763769059141313143706 0.01292401369441550284656081680725191470697657340568187097386382208 0.0001663187041064373045867841828276563685556530165862188180667708780
                    0.3164792113621352896719265938935711591533451896959723158140967179 0.1646072665217280510817218281914022862707910737407438031381614795 0.04700545606494943456303384028031159132735717931950261771567456464 0.004857455516947402081189117789862149882293632480593367954326205049
                    0.2847323823313864363844251958214840593425537226818365657831720091 0.4486024270426212251449385680754812011087486780324343832238960503 0.007696373253160313950176240387329255938594895506109230609950213207 0.002134155020458910678244376560822552255560630673235399110862817005
                    0.03379500016489753753608076530826115986237208349752662314933725531 0.7027052682986371210375830405848770013484290374670640418310714116 0.1148677384183933168293432985530541768200865179917392790119112577 0.002543591692498350583695801175431837196272224399908346987704162128
                    0.1433120245166042849149223644224714914034447176688399093270852499 0.06223380111632963014033852820193413496532694578530163983948961363 0.7899478628456170030303265653895578302258727810893208547201423407 0.0005747349422849558877312071985320901621126074777063627505398527939
                    0.8311408956657346107422420987148385992485305380831416613333098522 0.02530784120097069346857462242684469436533877862937777794056026225 0.03141957716603862781637836831911739061729923772294544438719546300 0.0004901855717909979088880349474282230153333432339363448956800646922
                    0.4180349838064569635701869843348415912070276602436372602047464910 0.009349018585645165218664444643701629094075808010717435681802168722 0.5599493220438896902387644991161629389776928726686878648746904394 0.0006709975800671327687089536382497538292205827740874585694722207683
                    0.01334553691042869854188657195277531849370716488435517803593204235 0.01011330233616838622722888591405869016006028856461087265983868457 0.4653433571660054370692681199916047023263983319355551962525316040 0.0007798603908971874655489547437698093889288969415547588905688481106
                    0.8692893323203175734231796039330641007594311656507405165773487225 0.007542602748144407628841011877253958098812342542620897041421633411 0.06449327355699129244848511697362727504047608485672838636994189666 0.0006317949829771508973716307756262681784275374922419771713700979731
                    0.05936275999448619437543888379741232257406518411141805652806549771 0.2226655093020769296181570687687394260728168580349362729526098378 0.5934251911617299065208036221283597691053059303210892525278142571 0.003023461759386744945001493014394684227853596006729004468623848891
                    0.4757756637174051589907395466417297019800060583991664370506407189 0.07205243289840793235462110080872700610553438665891803356607166591 0.1866572168708286581878352433698352261175677561577080266376929461 0.004601182030960639708175561605744564854428870910399635835837836051
                    0.03745944254660399367256025900770189331619213789375617559818889450 0.6939027902150486829686581390567222922837753856820287990294414331 0.04206772249625474167127542082144979883088164511769701754210642619 0.001408388405759180050632579534211315170896162039369101549224090632
                    0.1271938371797284136444512503338670428311955517821303630650780304 0.7089993012480119979962535341380463487268253530194426649885135389 0.05511440216191420321340112978638809956249141619678958193731843635 0.003396799095923000507029281783549408723452097662714242012211800728
                    0.1407817012735339815789779358656282534795961083548453155771161843 0.006750521460812205742561665427705933007425153179982475387253896863 0.1649825976197442034758818699601325389588042760981038024430305111 0.001454536019993435657925087710600148354754533740103250866494817610
                    0.05798152918130894244453673015688333117168178933762761987153950479 0.7485098787375016924117010375308379878719236226978721179890195714 0.1349679407812342461615005600544680808719624996361829460808918269 0.002696863355897097671358411687715403357697873701829135918598131732
                    0.2226394144965386236586639637593088903275029194977614495088284146 0.04221480637672158867442084362548387694456627079427000376849838276 0.3026591413445078859803178241852821452774860734114936671018739250 0.004345709060592211791098842718418595333649076671883242831220406360
                    0.1044318353047717620654719483140579272389782667972273402750356262 0.01412001427670538299020120264468393371704979104109957961121757295 0.4639749269641197713090922255744800877517734886013269304106176797 0.002016937103056351234177094453369384796801795846797996856327771246
                    0.6995265949332508762253900287543326771744288489272800507571837385 0.01099095040220838984864820332270938381026633827330928753646301118 0.2788751965722586283632327418147980714115866814270917700591954751 0.0005937702788718256932071108395817914099948438330257332670649884367
                    0.2153749590715633423371283155005196490393650232962779368208344411 0.1904013951736569386235902852640489959648959990906379245957408698 0.2338249873127818701904261239534721012759386648775576727270014418 0.007848525491922308854649703591372902434164288813956187659916827110
                    0.01046624682290930664040225868341775070872737094269409134947199591 0.2187454958807096914614834216086633282944231091309664332468724314 0.3187602469787128089119334919825318296051958542895793758263427368 0.002255831522313072279365493538326949634556288168272010457560792218
                    0.7306191934621068472258576055687224659348142662334788725007496836 0.2487280227932742642146567074417979130651886493408757760872374699 0.01313783436914064742581227519314313646347839753948904054872912338 0.0005556575558200083883977960061070337721553148458797844237671475428
                    0.01448530502081496838065144486536077725295979231180471576608711714 0.5535407243498060593049070477296253005737094532944036550063513228 0.3773608974031066721139805814550701654691844614649844958163190406 0.001570338724360238642054879209192154214133978872102638557117975039
                    0.9094001299702989773795124031813517817967164040565863586699152112 0.01318748545578135343377554782533376902278446602216253156061275585 0.01124832529796638496846646528584309045215439436869509331754686834 0.0004264421590609781002156346200882636608603451542426990167631644153
                    0.3180611870417530567757269683876656206334945625786876269406662511 0.1086952000898211731632055007031239530899648735738573929628880538 0.5379382885334287281108769359033015986812252630579947537650702227 0.002879818033248844189029868520715699691666722455765997250152867269
                    0.08414245880429874994783316299189263489710125673704869403417907531 0.7697968544287001344285413394310501249126859290849209556559924843 0.006868737655609776409004672100371014859488809034573447293407105556 0.0009348958792478718311707717462809152910549801382802863368301564012
                    0.4284380789830273543096320614213155097565474463873409849458955211 0.5126634064863334866484421758303753130554179853660786407099275104 0.05027302492675735183964080658035913210634011313783585568678807942 0.001220891151896520818674518115860035079729324404464154033896536710
                    0.3157576761880558941531367887967105751037428090588547529492003170 0.01326833541431898024481623361525152531629166066703188988992043206 0.6205459731170819553831534897389541807922398922767563344796514651 0.001572532164942313436376684742848100660560195016665166530836161821
                    0.007755897737708482700933412899625318854681372596396949746367745981 0.2071509477367018849783965844074986101874538765707255821452112688 0.4537631462410242747783954832886264303015417637122887384301696684 0.001681050676022432643796606040840380163875071382278571019218429654
                    0.04375247031092719495518290064395994635471591668885465754154953577 0.1827244985346609844648061559645336662642602569005243843835610379 0.4129514728651410637432568460746427966092105277971436401248599456 0.002803598145758590480781874806368879541625306979083938954095443062
                    0.4052874662101079349458564557794452903586769247485930919259155344 0.04771534821006267242531026944503437116577604025254351912276956332 0.4953874446968034771613013841347639872908567493523925509205318867 0.003345807128808899514102026582424273235183598165123716324459505772
                    0.6973701104781675574085089802515269242744473486321788291366024737 0.01897075972729294891866635996292401813029731661209628149847989648 0.007114359357501120173114621441273911499740029713685898057156020221 0.0006814953379087848165957693052172072223182329614280702976865577319
                    0.7031818151005948097114603377384447839618819552730007880795617831 0.1509297215549329747149731073161119237188693657456623057285587851 0.006971058507883068586874554156302616965560646970238809735061444399 0.001315269920696471833771540424868372872661837102547737604696727430
                    0.6533997971348083558993293606068891539792568673324051778154919273 0.2554381886999160292681066067747164133201676470199932458609052419 0.04420509049196388047490937180222219595144668173172235064249196101 0.002197242455977371643182854278532961108156289905604552756903864821
                    0.7579549762757517779256473666653547935562749795046569069392956658 0.008632128793909775051940266861835967930686216934720019227366838015 0.1637450627172126987129128320784516483590011662986280490578478558 0.001147923614308583781786446140505367985556725815222727100558986528
                    0.01488327893790584050547900718018498428546022223441589588752511027 0.9594295038002265348612526302216873759054712122681090181064069693 0.01226358122989886887466228827470242157975304493460193032900497504 0.0002460977228361580325994954788709652128470662511633030481836712308
                    0.04331752291293232132248000908402963551170387218166911922511233104 0.3013502465557148548617902773140183113615582980764284813336925975 0.6471922616243308170278035629040700341843386782400159461488323271 0.001088690277034476525010446849062934067258792028951422935568562915
                    0.1194506297657664609100389889499522446154350697227846697540790165 0.3648491034897142259535647580815618955907180413095887849899487307 0.03381039184669038395663781756373401420479972579281173542625529765 0.003536439637531099938414805315625895057965881198467838496559142265
                    0.05059624637304204665714046817602905088293798203946695178441401507 0.05214426906282641850123923723646528980194994617451715882796875212 0.7491461725611474512450526682858953501828209316920668312338366130 0.002358412875571700091771922427077961475913312229859711423522704242
                    0.8560285004301059515294881837380794229785553549509901612771013062 0.07156293267663742547598430349748727782038803952141282446085666931 0.006712947340178530775178766876146466292254864202123270863252952494 0.0006765233895963418178751506758793945502674134662829346797143176353
                    0.002233092050994338059122539702673154165040988465825967589159467388 0.01251355543716936552250455535854339992086057690869213073357907391 0.1681476883050160591772090860420146068173825845708546000847832954 0.0002818860395815868003204021202128066736352457455138069139808159077
                    0.5231813472600317009737513397176710646800459254896510027621866050 0.1120791326662495718690147126888131939297869793048461936545421034 0.06156754045128973099807841125148946965907306144657376991937296170 0.005336893554875468046497833592147050627228477789231648037673305191
                    0.2186090205367808039975100874642775974384611736515120762519431556 0.1071771206559154012119351999500810045111625736636449444215143448 0.05373549816824526986820899513125666084505338112923207380978729981 0.003630585000053163268096600294445438514933370353737219467535867164
                    0.04497086190256097040701190510171553800645862991095257696806804443 0.8857973271901770699333245719465090667841643951530573586507050470 0.06083353026134824090325273806007327071946599524071607814205657439 0.0006113210446342206294093390523119602905911358888527735272475589232
                    0.1636913079731239504157537849452373383676362142664357677346049727 0.5815690423079391353602689936127840936634144627261432579985065968 0.2344782954778802671048364348987520450256332591990728439355000060 0.002595361610670090113475387498872683561469361057984070006184281671
                    0.4298862236280218103389253418357077479128204572247635360425040130 0.2880484671497849765600802612836882591289805839592985325258773131 0.01011555221823971152360194519555811849445744799001642815610252275 0.001940123163527487215192631793586303891738083106307505188851457127
                    0.01427697690438825983342246742977995757539519717743621822765281362 0.7105648301901424026921452030479434275895499165001557964398865300 0.2706659567689295573307458206995978233685528606739882672167549973 0.0004392323228579264216152457472005343233977472742582169065121683417
                    0.04323583917635186422483771807880116088244807058222244926391556492 0.4572801797691881258809014587227262472470972250953781845989265170 0.2734208676947149232618841458872262482724116888316229892121544881 0.003446074264737631221543158312340364513031832003363242967943512178
                    0.5008659019211249703937382259724327033298693741724320753234310460 0.2351489285458362361844858173242965006560632733613873153271319632 0.05527812104591686178991509632146020261004454706010034579061227253 0.005533709883369567797204785215624204944074297583334384116734531089
                    0.04499343446866857621580010570984967819563298270640882330490805290 0.3021967879675745880423495228171891565145769246002471990267532562 0.1332525177022275956052501969235561247442536831927644457288957516 0.003757862045441609208857433988320691146264314885341505064523547204
                    0.3459077559427021886441130878149464336807189141029622737977789899 0.1233554931296509848238863975525572521852111581526700739243877730 0.2535360236622349910420717317206490245465211654525698246220412598 0.007847174003746434344061291411508386343269788448344633111107930221
                    0.06316973520185842815413918856039140568463308550018475041926952887 0.01100387523362812939917137398994780057465323654450793260843101659 0.6934622242603130589562573678819573574521290103220977806263427502 0.001344424125801848645639461526911871314242199359300279808807388008
                    0.2323656802015384539416454803040980758441548701946660410192158467 0.2416280335128118515118560805200833363572869871031375793523444794 0.01150967802254737408769190631228481496026570611521675641210208534 0.002418533316479529161922489385583549635638709563488027239508446060
                    0.4048922652087130245051425380321521435053893925361200593632086228 0.05145649447165171628347940306976939926789105287590426178016262683 0.5376800514312109862509844558709785321030416244292929093792541683 0.001015763323214613832667341703640421435698750068231108285043326530
                    0.01024124958861419872459047515143587155935051204048977121819238430 0.2304216621440370620670463342617566788416192234981249571529069257 0.1224771618067651724875321846121686855866193457413213844790099934 0.001642394889591817643278485952131886765563811126238007369819132368
                    0.1268643206399209992647228444897238339339214701796964972032927574 0.3167273088167407891660610039351893578420926774342889953486462777 0.09648579553341316818742764479092567168393655831495575910914901214 0.005620068374296718564654066216514097366508600154703495686776834154
                    0.09203169684973945948819187000280072607836709331239998505406317493 0.06654197888499777605338740987188170274508316136285225420725930151 0.006516882744971700907643447517013088411516920720990029912153759437 0.0008099158148062208984516891134771326649024742401510525262975025596
                    0.1834557885406043228922507675680698133714892985421652436466617616 0.5826504701776864145107270607361925920773709076906997979996485579 0.01121481622639225930875301620098418137189244670756694954046356292 0.001965788068806115048276108358535332028359447680662695674354222054
                    0.1420434587642287217210880945131644998547593287135776207072138708 0.1466054251994289856187890547230819496711175954291571364739939087 0.4877368521577567976207515655078081923337742930134200889701574843 0.007220662075842793948825679861957926435720923998059157840999038651
                    0.1068539708852586532551182854726119107749956718628206250464145345 0.4472289826001640153363800881434741754190112471204855074704142178 0.2844155014977493911249740528200566966721758162602030238697229386 0.005978600588543209766021177751618440084463592689915956983815299292
                    0.1344692030881566145963471191597315064280551953538852552941666358 0.3368083827134554511677740248019418644580888777311814984296336784 0.2717454816545285876241098568616238634131790182062879241353906371 0.007517946288947784121808214151834113623462451261901894557332869115
                    0.002554908318008213742238487580089496178161934246556977403522994018 0.02044795637070357215699077421399782884773549962484408881157135854 0.8565402648060665512238876998973257064639128415492828143358749251 0.0003126633045366422125092928855364285268660134350945688284375767199
                    0.4408966176273476034645550643242952029336823208985084731385622354 0.3829185162262692026680600764024666107805086389113332990559796717 0.1391261396796738191160723615442485727725134827136868846699861502 0.004266189258187493989839233751009242333112442944378752612933076687
                    0.06485753028751197925933505626157826200348211670270926598759378333 0.2454047815056738795281384331675032598773111243706143751788284379 0.06931344089640226079424155140292805321732742714962471933074607262 0.003773162066442044266684971728472689011755725955880236883832618601
                    0.6517307139710275412623442557432279373144568589004358884577321923 0.003571071153912234245147034599054071916051395324260298064247431955 0.2892566634880247956648499236757341716502901641015593799603024883 0.0006832803209215718148476087290214778727830566477896129880031465254
                    0.2824035877709408887680124058622338446401969186431446134986976310 0.08406800484467673321034216237000640066448285112468700572288747495 0.1020559577981133177760227338108749796304642920360564860262697701 0.003913556313567504576745242313827931578923357008530964877702466805
                    0.7376951224613909244894039088207562288189351013959831003496752209 0.09419072934364632471901555314450067888231937058813800848184989787 0.1377618231757258850833232110023269786952467532755578402412754498 0.001462654573658041271126779779682935469456813599709380707076654758
                    0.01090581439123931902771761610922767188454739994591788542923465084 0.04416999690357117005484994899563193490754729577231941565582222893 0.6844921242680185608647317923467385236052897094384134359329362560 0.001165907841845349447508844844186895604663351522742660742830718663
                    0.1600329962622842704051568082968815759184733743869739322158296522 0.01302536918788998800495579989087749223379898492901875516947388965 0.01182810194814166723586763167884053528067807238211611445415747159 0.0006806150319894086704566955080197771507358483187202854377301125696
                    0.3149252692483614156528398963487885660853029817471175145063938900 0.2082521776687901168955388759820902593280590618061132357844015098 0.1554449839142258737419923934604760697916126240359265279112532168 0.006507612293330705981583226162733394974887804527400089864207196612
                    0.1637217305253837536830431388796350897050756819425533384751638290 0.1410742652900240925847001182044417260444298017302258351115573801 0.6911442976094195073628951910615463880176027086646565231687382611 0.0009701729630164918199656692902330352711534529481483571458637719944
                    0.01068942325276542294059823050618949569953722242999003382521441533 0.1174314153796329769632234056776129524086177330335812040826792653 0.4333620230815625007757393865281179060425254880489561750862688644 0.001899410270135892584314289059825454957355646098953015505353327725
                    0.006027160376848061464398386091038659624128941849582982095932426459 0.8282488616372025809328357664702950855836027617669143076449660392 0.1505849438641980473454349783546461075295371196056399158078584275 0.0003625332980975108466931737421772840434937292035637095991342383801
                    0.006226874413468867444977064114137401169997633254256119185279675401 0.05607855985767846402177107980929860583274472231938074199780193051 0.09374188667080962810288670980117791968526248621465650617042766402 0.0005306487266039331273200693896573116928142846840909972409209983841
                    0.08997464438675424451005304560612464567894899502912283706421480683 0.03997103920226802797898890550653704174789282871326772718983868652 0.1225392265508571786604875113872969086031275051099764216853911662 0.002557991520506325380792525846373929738331610774690521667275367328
                    0.2036352272312049231410488244267832435571542414063238126232588065 0.009037004496300932613532632965905319767113286663500820229922148620 0.4049290992939442218381574744072149202655461839920236483352566937 0.002297813230111580230353359228035163468694138527175503684643380783
                    0.04337603375221136320697975794734087804196054408368728224187002363 0.04932041323513612397829805908206245115376010780859130218077095158 0.3476839610710102597587515609624006886273084184657832705581920870 0.002689468424631714244734224693426416041365690503571913402515155177
                    0.9252796246429687193404150915487806173362732707037780727200787199 0.05302828211170333861154787170146690877897780030917093451768650552 0.009402767770162281212084614065524199862816878618672954919122129677 0.0002872212484618766983043866509241807068638362291943539966170110803
                    0.3089094965018963061981039342975793800637114644841841026320516099 0.2888994105385460780576056671433754664471627538441897104182322455 0.1042373498341085294783850432485045320218442079491224325252264443 0.003257966661194633108971636590195646582690392871839546312144094856
                    0.2486559432097139663922381667226621814150590775759751204934465349 0.09709871612971095595029854243684916831405510413377904229957624278 0.2934672378519280066822886857201832678520597042256873050413177175 0.004471894714311672094415273310011381152076579980167959453145297107
                    0.2890578427140760335454986586238457087323480876540906692349950330 0.1508615140841800643550959151449180729678236723444953817813956709 0.5557113550285408116328462557087968400574018584707673170986209401 0.001295592281173132010747951695120241082272711507410667316429907208
                    0.4591270378764738237763307464644484604015415156758979029344106228 0.03952748870840429117164528909838780872213435868236796027552048771 0.3713337300138085971528214611628194895863381266388210402143440364 0.003088740922171743940818970530960060302051658562721381842305088816
                    0.5539398800945140400885630259852156945948214258909068439377607933 0.02339979042698435674298557677535710416086381688974517332177356623 0.3569189332756404796223457277907095693575109197036111953380701411 0.001817730148218938787964357528484187086340730954502665863941419791
                    0.2872191767935580000074362667687857496813994091546184979726680714 0.5127918871534896037851605546786291413692596397179587124763356702 0.03399967001027381238672102722699242124801443713581645138863066773 0.003173314799187793190216109612103699962762348642424326669051425750
                    0.4138121966047382077875800750568779846893590132158301678828856415 0.1040228124704250249187029074530190940840209575159765369978506827 0.3408830898192074312420100546328112996855285392699226093377114361 0.005170300975198320595642085407616677352902929018036112916965754427
                    0.3819468455946183880443119317834573853548176863013478807468394928 0.1163792737922747840279328040406238033562607832846371904497336818 0.1271646315845564720621109431394621543379488292297097459551073781 0.006851437190191055942178201314036194037810426711605858902465385617
                    0.3750863808776789365028241653496529911097226641373218138302308933 0.04543722754152005374334846517880157844409352061830192832550356465 0.3343301609622336345488361747000748787109267518840310628685346815 0.005549973208535719600473683715141727873673141429607207335814623467
                    0.1016910315982372671688566628669556203310793442214620426683988241 0.6368273194822887985866133251988069765622286716226157161346611743 0.1334261669051737465539106935766715986646461609463197092070098797 0.004616727940469358393488769585261895609279052839257040638233746856
                    0.5703115650309976201680412066474791436781033815481094201535685007 0.1243379524605142599506882643780879611794481824405790852717843788 0.2073338203620642159258777525462331820984826199569194997655752993 0.004865029783561290352315154796369006716019921177973503536548862066
                    0.05461822265463124741063043380496977961036814124665634158751264269 0.2030810741420467202048534374671623416357878698050120575098637621 0.5042548871574086892343235679717432359838040204313801295104446286 0.005175405034237403332477117971862861366776772061792829023470594969
                    0.6736596305732192697666181711551935667576629663561273487208295440 0.01455649411157261509716519111609766665562039989123165979848499792 0.1513207891612286560300890460969870528888591168199431294194761093 0.002031259404068344041147317941403435864545443680947535978066851563
                    0.2442781869074342661371961467695188557478656955214993686132831380 0.3849304006982516244052877561996382361990446338675805594143510029 0.2175603967406178856760465021244615800621544645084696659066072385 0.006388759336174279349941700816399826492817577398844681354197779720
                    0.3822600533535676722427506460002869630897473310980461930262655798 0.2176161172526403156330528969859236565922024322315692488682470277 0.06922157011009716733498497268851219977594845898773126431572750902 0.005138742446776225401564001060430324835040166694758690698640959681
                    0.3647211003164685434924803753390206335356240167116089581721195653 0.3587379389603842752930137095663756080358614702921077957652805511 0.2663359973568604025723154608334023930940387858896716922015277510 0.002465197387176348623504849073767453216093969299239927672080804989
                    0.05887859384667968716737924099125845604035224579553234289203220303 0.008072992711970363752329969191257053837659001828529608886144853083 0.3923883933600481130098876247274929983753807878228216058663763628 0.001186299709682681934606838146373923570631289982092669528444767551
                    0.009434075532047231520804374023980866012038842564168996822545562452 0.07284975998502776175727986512810202584441434933268091992067169261 0.3670474118293337009003678148019682698041095613108950949588149697 0.001041421057915595233853113990012587908690206157066139877606183583
                    0.3666912553934710950299646390100680325223552605164118865555133496 0.005104760971127389842838494977831422587082791839270582304718050874 0.3840670119339249971993099696480071900088607246423356858627181727 0.001384776922256841196001393410538099579804945372805724557638025241
                    0.7730453640243989236848404678530886931599641244572157128675576272 0.01088571121225806543085619483219985505491687624153335536123870190 0.07214142486679068880266215973389103588634758397201093599634451529 0.001238554788286861392492132201022999883308551115942467543913432518
                    0.3743276639452759798586282739265638757149969986306016527993254318 0.06911426983177986775961311685332705344697430390843969859832064631 0.01130640408335228151134645552908893438918218930953386469708560997 0.001757333611304789318663699781464343921197058235216908166981206011
                    0.1242372500355299611940516059672974915968519893356589614353303133 0.1395971448924533410216808223325979666706709237091420457735761764 0.2629738810719239607252074670798562286270000828268144022824546739 0.006724046385719511343523126647495632075541843560999449456038814724
                    0.4474960639685789621045580367793933865174703394512187903370254226 0.1431429401260028051888106542040971794090773874740360222507585806 0.2414817759493239937928437591098174290929889788560700417530158492 0.005316194450859139482030989774216892055726220629754706655878691393
                    0.2778828720860007798154440533819251149928988692419062813950073209 0.1170489945437161914670101743882361178812906504532096839306554217 0.3975950085479843526680294530486227893047743675953416231038194771 0.007519491633601485765068169705375508727704023148502535355278449461
                    0.06732050522187275080317611329397550277719274626326613924387218590 0.08045338637047610273749285942152266881785243580381029042986739797 0.2142373855718151540024609125984137919837403300833279609266309222 0.004151836107075217600699170728485954093457433930660707763494255822
                    0.3612642176578033820581433389000925149434388365217197195306676481 0.04504107214852448892790030068947553090594934681121941648531499541 0.2182269217766062249550682601030678724896173200207148160040106374 0.005039041793726196772373843022826730299086007567893159738798223781
                    0.004892389501700218104425551851592197781350380577168635423832134015 0.3890383170083868372055710772159889187854729269242493123400807757 0.5964083450904423123049735738222478789558676181246003149740524264 0.0003707251630658503089027110792133659816077692036817160801288977666
                    0.5990162072243106868627384592369704915495957344278319335687443660 0.01160073873419978963379598730492891178830207814642936869182301822 0.1231715379035730615851985791854666539069860092689214993778129467 0.001991481506689804377381280050174606084690099050604004192641587501
                    0.1429467257461913879402572934867737228849056721080472902989321361 0.1629707588490883202172319919606658099493675584795708378641698428 0.007906563260407270288750696685974624552190067432723861804210477969 0.001562366223394621355499942093586946966506780328116334644126590293
                    0.1526824663698871300328424092493095846039972390012768694972465314 0.3025156519349789805416904034869054789485562024568036002545131631 0.4838168038682168610328090567708147779775443869128706623867044552 0.005473417306940747909440648785091833647835602537521109828995807319
                    0.6170695055583639707708556878328888446699755643843692068797231433 0.2285979263351907743101382127237334528483973728090671258929472911 0.05091002481617129342992024179426758721818198462811102108175490551 0.002805629091923018197896122742306651689976516611895728888013627075
                    0.1157449164393855290073803635712304629119285134691565007610225288 0.3940017539702470104595314446617639334154011549696860112726326404 0.3854677865819185951764011075335024014019882432425194057905609758 0.004588144576142982498997561768201004499935773454855850491115244896
                    0.1367254240223778830875285985445128178385716845983781670302503629 0.2719800738322982362140441233196842267531755258250056684229811123 0.1924300175801547244564241781200923803635319577529234493513372711 0.007371328145274920892086303302651263238992240266525383413876572686
                    0.3469739325581284055814389685032702292578620079946739762052038606 0.03055516759693608780273066646529139902612284609736329678028062839 0.1234971976160299090812257867790909446076477712950797908782607176 0.003195362027550446300241111948875084799504273607848890448923236216
                    0.03111633392055489768912510183344854469069137274694494915813518343 0.1491021737723402298243222000182701964183835401494050757308756234 0.6610734242894145151793354838583884278878845904050838597864726908 0.002928734639394942004656936878298755319927347091060553869271268548
                    0.03729811265626783914369143224983396491908158388900633824934640837 0.4786595870791660086042921718195969290174010756374894237868157484 0.3680214681432501243002936359199540214320191215740250601679493661 0.003620774941895954182171962355691593518525967997931297240477984504
                    0.01160275549835021123924495310187675791448450250293796628874150975 0.008561173025177784513550881091341023871243039338200481258687709141 0.6209765799338847082914031659955610246259627390549364311629777563 0.0005865316759656672123142030555629682122486538258022539475607025872
                    0.2770263433531341940323741844663124149775432997596910510492155221 0.6447959444788540999029519802453772914569346377634943385044255141 0.06754201826283452659469673279549710194726733892906882822723512427 0.001494949971323195115736243643849996739535931621319531567391410814
                    0.1837868277798009563291591823795700297117993758475087669016994385 0.09872443514667751985194932798506424673979911213385192188172568000 0.1943404713317379869693321816505047279272372063287808020898470781 0.003682317917172221978558010659326179490471177492861934803162448980
                    0.2202970578613748420797868009630064178305649987712233690678143686 0.2125499661778209825875192314251207740058311243117827385382845494 0.3312019568148393766638912501876153766855789753287582282470627221 0.008795160080397565659129354674178246251557689740650309574175483427
                    0.2248606548150579910505814769887331754807423488635109372909728093 0.1893922778525718088322221008942633785825605670458287373430282595 0.1169350095862368326616204216755069910730440292197517150088631990 0.007692554336696465594081407556804019737035819308997848908544866437
                    0.2596704607167642721940579213495485595883078189090615986602343854 0.01231943361387935280105320270496979765234042247904430296346809395 0.4943330603323840914226520462280409070390557156261718051880360903 0.002554808148677844653089883474784109494399218036104838694664280391
                    0.004160654104664752403768383022426189331075541772466949450807732916 0.9215322702145175050055009078440153391675949443482530369319850057 0.06453857218950149302232233270166063039117406490591443485251272089 0.0001734209863891891796652927968690610756358295031318508581011409565
                    0.06012084215526280060451620820107459500607792081746389438085119033 0.4029199182157937607893335320617493403391811146884849951564525472 0.1141603266018256707210732738581970671676124050145307859094795741 0.003398088770514190803969738652317208592031706811770266326067218571
                    0.6818843348975056599303279583896318337567678062592917098825057504 0.1634903848905491295114727271077405414207071642529500390464345572 0.1537628492993982332890595491373084927331284094862189159005159422 0.0008939744053831259313360402984308421771339819841510326257518340338
                    0.1056103192557230138584287754813588317628542484189368967259734763 0.4829272887521248563950907004597351322915323991147992347371369231 0.05157058925094880295933148118505134074204121469525482003490950251 0.003418965188453945645722102229716581092617162011065266595566583324
                    0.1576872469770223540351583722473407526569190624819888970547327357 0.2107732304361199226118842550150916801302143232760396768752221777 0.04985587615650729859455709781840461594342438639000837717371980378 0.004786416154859716750534077383253813721752542443531202104817933053
                    0.5482209655107836063277823439475540669017446781127998438407030439 0.1911779599681637930011160304867227239041801900822964024233206078 0.009558425396448384666862506615209672880856699413324554214720382958 0.002197076400147387626505795930316286634743704086302204773445553186
                    0.006051707529802369525949858806931944439031717115309076901149021876 0.3456488384052450317171337695062763780844116635832825573651926129 0.3376748415048600739949637199848944426451793169483710191907911912 0.001863566720075331839732404829122774813836430489449881549276291405
                    0.1908392469729430403385426836929818800040879794769142815846558728 0.4002619959602171822486851792576597236686288315372551203908085773 0.007790948430590397220995756653960821342041743202182248170832876653 0.002093343058280821568215186220251767014148565146114541295489569480
                    0.04075260663729035068116486177703573928220557675345857667794350767 0.3964701677681301697561732641553855761549161048125037435396601713 0.04027293361293945036733323174703146437015314301690907836165475832 0.002893537058028225388466121913403578512928621599782452808970028410
                    0.03657370671953930042077220144268136466534395296117839248076802845 0.04803077879294122750356386431801133205532634019184718137461436092 0.4970871763237189535035016530306603409357392875320094484954414699 0.002771034966737731092238227536814918070610823896977455733199921270
                    0.009528294683739213306030789761188716795687477379525101565136822762 0.3985677467704367737721876253795610721801110468070044161882819526 0.4157547475379142961443768617040597357104451535766256251331724572 0.002160617004387111595001444740443640319280672047488904387343608922
                    0.06347583304678873896991579997450005528060903223423717491773321392 0.2706766372988705210318708567573401665210478541065746004298122779 0.01482456804204179308067552422584249160127167474004986132015690911 0.001838661268470062637398585489090716712608552186872627534461396964
                    0.2097445807970317585883345471888303852519399361376997165803554609 0.3193655958301392889113663132365116529450131533077558594454962101 0.3384395061678875865401596986367011897509118576957047612464425055 0.007577366552657964874996965984488319539348188954357835413574834249
                    0.05984804137591042314975469101227297010035445676865532314513317155 0.3551848793481822292315701604771507692385015298705311111718718896 0.5323615180999668183140332513905930249952313499719494634206697298 0.003368515381087520716614675117708141123461807273731054675548616642
                    0.008497736458430719785303809834166992436207704249441226057247127882 0.5221429054311363312442163462021602599355537010983432322116751583 0.1464306046026039640024462668604133638007185109585558472135315700 0.001803589234069591401382191945712530891894900013343095181769431387
                    0.3890381012313203735056995136560582302978399315149846242426056500 0.2982387697165016860680064054472194504162195831325243437688489458 0.1275074594853796707673623677704971055497860812093614568319441179 0.007524085190752683446993410050393526141135444733274758786252782090
                    0.1083451155970357917151379900122733480597079980293271711187294977 0.2174733606252993467408879406165302793762831373754514113332728421 0.3436932145087708338745408092722840942557614942246844634489227306 0.007806043076985767735700011797930401016030202569830336319523377907
                    0.2247795259876792422968190445498664070347898739807944262644162101 0.1974672641139710784481465006864244491956953808630336087400930227 0.4565869196058689658085042452866481157420359535337540501015081213 0.006856116097029500456475663370745281416741782818061550316788127016
                    0.2854391047841085567147525583873335516739589809991336813078071111 0.0008434144131366115531096549919832585346984392768370680817446666011 0.1012402954506772259991197693149683597297358185253641537827599498 0.0008459643184168090947776021450555250575506699580910738473734964235
                    0.6164863166312418682065669036504740834680922266478636466244025934 0.04581491282907579263212847437785542729371637191749467871258505598 0.2348004929828133583141795964814160533504184392321732030397853755 0.002516733942589748541978659827138957184448747961121990299514571303
                    0.2861261841986337493082591639934366233471670291639803375649540772 0.3027807935820711120222014715640506418394800975943408591079676902 0.3558476318633236592253431742850628206189307701451783660193620486 0.006177488419064699018583250378278976660871753127025090764753412491
                    0.06916534603284266407131136407218209235268830353123837045212735576 0.3906257499302909244930828271244495504995021171824936804902803409 0.2101236821405616315664856831506614371888681103923812146439333298 0.006269703249286099295912429480734327423895189813932370200209741232
                    0.007322647665876281600604280923891723801187755341468364204467168365 0.4056544734465316437003360317164286920525274472480327897443396669 0.5079560828718447975659416809075949410642862786693173872782584721 0.001295210897693892928782898926377334375680263669683647802027186833
                    0.5149482404379074204621495493195979555147393617730378571423316000 0.3098052794021530221430286802195379292250142597413058455318785564 0.1718126814694822213579578388739403423481649918469985922267855471 0.001262757781505264159593607636531333320626173060520654276852411983
                    0.1667638387615530426166455531894094463471088990780798744970731933 0.1151071190537566603837506266407401756459504700195363498889037955 0.3783357788437605744276190769736828094770996200877617103201724622 0.006552003131171302248240172371929504981323873299019559012644808300
                    0.03781956717091467899992670684523143766551322596460491975475690116 0.4564127103943539216576485631302056853951661830881847684257852385 0.4997547824479509810781045853537988368017399873587599689515478108 0.0008096951338452611847542715949165690599143222932691057839834090017
                    0.3092370946604168081670941475230935261952780313575010692769214178 0.6712325166561010143897660685766718272741523977566527960703138615 0.01482324810199125273399454061272075077693171374096917938151849918 0.0004618621488298762242155290838827801848575189292574685039418746442
                    0.2274267025448595851327220295824844173685804991303501368775772186 0.3698296586628089380413858432105418547412724447687252739021107826 0.3930220152089306655037819262071081639828939583912176322857062195 0.002436709312907541688433752394419081278735684943086237575538558382
                    0.01233746884248698524262016906323128844548434670864342003572926478 0.007815220450551604979470440661482046913430484777848133426919612261 0.7607693676106550534832047145918073449654539801141441469376954988 0.0004879204203801834185070596745401306260534666947149032634218291198
                    0.0008887038653497185098367691405479834768156505589198350371368987923 0.1709587043523863162838603121491819374159939311695641493965676681 0.6365580652799449415284507771202171828408266756279988256682419392 0.0007467818230260785357467574272917099220940056802457844657363252266
                    0.3439784829909684578985583289121345098141078586004626609722558328 0.3577907191352539745239920080642681073203641958652460380674338856 0.04761915278296432642827746578682770752394218756784555614953887035 0.005514817479740678433332592737255268777885387228490960158291114804
                    0.4809839259360869658719535826569018458063923949776541508497102845 0.006745001351801976023974100276906715116599834018537600979908895721 0.4534754016632329894112321530223479907003480830393303896413666115 0.001054502551467144375498406537110883043102913417663699147164712071
                    0.4675547114714950739339274207548917432247181857003586042351590275 0.008210815103144332113439146501221170069032906546358486290321915342 0.3673913886377485529432776330221021733377168270578200147546901559 0.001560992327873134548834332683980893774410704316961939388043811597
                    0.7627867014244264449326855877951285382143648837357235045572043741 0.1725373652941705288381014781032423113127061258496428571957177330 0.01076490900684821425654692434393118307964579871502689925023841446 0.001055517032983477748403946281636323203632028768487060991306723813
                    0.05648731551674311293521875889599950949610937743884939972202574804 0.007308908362817571706802766491739596145848266654628569634995985443 0.5674506786142166636303860057800970307625339164022903967716373743 0.001009003186273791561253674934095083317811153519697303937048976027
                    0.1159122470315714936281820848967452174499780214911463650625175007 0.5778538593912987384961688060392277817107847413983175344112281525 0.2373340510660815205754757253789684758173931505956482405589176748 0.004171388444256949080365770281966022813809765170887457813772570343
                    0.1464175980944511477403557115080818659864902507361655488598068611 0.006406607788896062494069789127485024261272509093598027469107353960 0.5781407150858946403256126760458545000173059514731849740778477028 0.001582696034431353872235621814836417119846855090879333071101449465
                    0.03972009692009831567348112170418740839168940977511575193483742143 0.1707545494254464264375593496444076396963917825316313650132857521 0.1854729089531094709561820254626249725655386187239333141606099449 0.003519222455679963674082750680619464808959874196980679848148878653
                    0.08076027896580794580700663421156195772096792433126271760753396555 0.6781163031864639935481196406253197191771843111986522207603876961 0.2309457622920834913520210702709953793603872894301968170509902571 0.001576001897958086019992300653699902130609097525847594116759273709
                    0.6921739849019276308654218757656030082261118726715102203690716006 0.1471260763384878105982010748034797227129080156385394406095638038 0.1021857230826886879468029342685723947451334905547419477477163048 0.002956587789709623926943919504141521407660622626780751457697172783
                    0.4796302148308850789183913273804375857597465261412428980516038427 0.1971439481341879663212543272405427617479737282910811690444606130 0.2664619914088485050078524697196993997426687935351045472251126478 0.004129296387173560424333963818390602038817805509837431818057992335
                    0.03805093175634197207684428554243083259033530502882982485863214093 0.2858552580841950847745773242929516782643208116001208487547630630 0.3340981474659707831038392213658442266824436558990285695606694820 0.003651049698792557489264610457287299806956853251932895176193274739
                    0.07229036378082125415545158462878219273184826580589440102339563518 0.4523762660209876715931186742347679384968642661607137448082647768 0.003758598473927021644623196745282777518414319475792304209029869043 0.001037502426164697090183771528782442994933870049271377024731668629
                    0.03611994401528278168008310169810940474487736530591138674576523006 0.07723871200848542279549347335628510680454787091724404078507460832 0.03864531852916379518384759200878921143730081977987018788844925183 0.001353738113465316978559513973022000739909962278858121227608248312
                    0.5889747438161144209057813502856904974710184712662405759366590628 0.2351055489290995303006495093875084762759339795607078286075178049 0.1435495916810519400995729243726039592318870882405255349207635127 0.004094427948051961360054673537086789085585711947778607060699257471
                    0.6275091561838315822249592845292468846037565360750875591999525851 0.1270541304624834789628236029867942806900985638253618393029089108 0.2143735820518296369238849447392102704194741779255663659863784847 0.002894233361013237423668591324538888226736971647228745415328383916
                    0.6674636161281221395799062936845955090830952133767703461778562280 0.03837903007606730986835016690427985388349320274799194942065278947 0.05015970540792925659576356976077532014835804231427489171583128513 0.002666454766999008196507121258605365979041162532363994998670039910
                    0.4669151255195071622645887330243035846791652349595900689458541096 0.2311252322741572784191481320343977700462242703074762331697737306 0.1933074982705743736756018258044489472765624137875247742220705372 0.004966046216530961859513781358073297599118140327148657333982115806
                    0.1136342767657953409983323428825858783108193225306414154494301872 0.1099583127406723187034510907782686728684637767639789724189568594 0.6547676968021143682353853745706780411627904504530510656984452956 0.004376898107464522943207164075812836814314286290198934174054392417
                    0.4568457360454622496830618036134058442017129139388412771939870385 0.3592551606452130617793781613456200882785546903596851540631987861 0.05237115270627880495696268697064979396066184147143560887027188432 0.004203328325643839012986736382876619431485755996160610107810323315
                    0.1490577634555968150269461117296651656026806223637571096644558973 0.7987978211839123356621429827560909055410112849084300647986659486 0.04652949273364265497692546777299052969078658350783465483637506558 0.0006822600732125751849451755438727944898229669943695069900033080566
                    0.07084398343991702992902899706611508626063069793656366168133374136 0.1097889320203848878117295565911361537712018396176789323814811802 0.4562715490190579161110363839109799827532672069141815388531107524 0.004962256781586205237820293225982142851607955962618842610179291182
                    0.8170412844109655470852069263520976214592718854498844987532668758 0.01134489814881364540013122475673655612790489938298092807057906913 0.1583644376029456025269140631551700827440921054904900638446995378 0.0006339545752075363721528142838372513868821779153604062507206462346
                    0.5142553342053729580285715958044330392709709233314545064789433601 0.4055773847497423476791948604464265940697999013762906184060577153 0.01051603265518346673070865638673984040063668186078063620645217112 0.001648593369750222987686791201000892035393615714553633296926540348
                    0.5941055137947341661904314896981670400378595862306263707431269425 0.008159103179123706175036982565739966488246072055413768024001462868 0.2576006367373720440725471235523834989050654356799410666992601687 0.001527724445316405395789487877030548701367686379299399052002621411
                    0.01651195337758764774371647634172252181407282139045055082194225168 0.8108291666464255573422682273760458197990601415586440892006795629 0.01053181350376029597526502271397165792400614769592337888428074211 0.0007390689179369703854690010410507407849744663143825281031780303486
                    0.1334155114834045160064530020017172899610021390356646764194027997 0.04877429493700671888460646261767202396559386804441982662138902201 0.5069494873079652976294713625270966211693117530445085320177121887 0.004736246837559430693648877065183383645317745672387572989508634523
                    0.01006153008257757225417031613827683828349977891727458682285390676 0.2214855975176216004758885022029271051747812755860742600778254266 0.7635447829266803874800776731440187081239968740965211072245860094 0.0003299177331003762474980855748569032936388192663595501366241551367
                    0.1430630403543728743629319599059224219672314911101121761662108258 0.04622636812143390823498243639101797246889463405128568762265273090 0.2261157481207355262241397983871248815714035888385050000613880987 0.003725288040665356038418443132086294274343225668410682636821093823
                    0.2208338384631852827577259013301913354578360881963852164395996320 0.3242736950476317147327562247294852983682767791227224871609325402 0.2052481609286995193664638652534123013120892799700052096406883387 0.006572576094667514718068372361604736416403453969064776776111967555
                    0.3277371163225625446950549126733472467269505774977971344463448649 0.04083290412329316243469019776500581317888283421638783122582485732 0.4878416497637716563630315537068604044877650280199866276744174892 0.004231814811956698029423407576498073120611336427300988782609383824
                    0.01031370837036166708278305062886535928384380746988801229370316689 0.5006162000270808412305514227218021364096498905741070333572431284 0.2601393077737637439609445208059498995818105828921877338180309732 0.002345231681482286228591323778619116568207987059721394298600109087
                    0.1507969957406616882227727797546213341964209317949387697469286093 0.6743937246091745078811472426447163374921796196431013582558484220 0.1304642017317093014240089521020967694243907757378890880658790376 0.003502539973402937429727871424427165356381880250849919878864859021
                    0.3272882545686135438910690043671725190911664767621072096073911247 0.009496655249970299650891564651984238122008039751484009635668160651 0.2965888133519640798241243254327362052627813883957792334700694466 0.002410950071183383937549810246574309365064766778804180101962777774
                    0.2632513134234530875822977957228213021362025040017195234462956648 0.09739606866027408141373554757873084372144325689005470534174439799 0.1886161090573033662571809095953652089159522158188179346639928864 0.004919652717619275361902273957166566673068741800032296998272128310
                    0.5137788464193817967236418254628794899255118259940269664539848818 0.2788279089166380691841363231827606047781943721534914943182046630 0.1130838225318632645061788371132056888090127441110757481731756316 0.005025890133876744408245228760981922814648136716065793760952853083
                    0.1214723822991945224387222379251225844365540127706140754747854597 0.2730992336393622578533126308911366505021934794323280748883210580 0.4185889524227676923589431269446382719444200514868648109293204455 0.006942192944710034699499979967081016362777700214897034052381903261
                    0.2052672373061645016605775878055419898095475061744446308413475649 0.03363114982673797802990543175574019922798938595111901441447794493 0.1166467690530874624984543355962791837810468445776659391649689962 0.002930847054288017271051108604233788054324317826364213825490140877
                    0.007544276445789790151978909970196334327936059570052805192946508871 0.04352410639209408264947859882295330388102483461257812507365619601 0.9102071244102442306542022500214134466666651294595324320687867176 0.0003385432023782755623672704279419181814738874158642103986659344769
                    0.5073684938836504856260818317163579635122874935048786324300374896 0.2055874095741024363284068043061193846318963263074548741643099047 0.2772639392358733843023104078105458481582489450913593634892214750 0.002302391288415197233973105790566047455501078249062783422557126065
                    0.1386766865309443218262224804038224556270531942302618055681010903 0.4915423548976412193055015547998241785184190040694658933865043234 0.1713516035584197873371087420320458836070717911962868290091352592 0.006606434135051010999541757853370319229889329072595440310271128131
                    0.2046056261918312616502149752563834428389562743143395043563288854 0.4425008806165124224845417539972132618589138399846248055369354542 0.04605324979559444985395708363721339928867756505789995203001440279 0.005058069236893616032556739802948700870378587647776307295117926099
                    0.1439773243312637229104977114412960700986610812471863222349393840 0.09988974066412993366369615062670560348693334742926866216418902744 0.1155651138948580955167980275732001308895117485620466748706644771 0.004402581473882988665626117614280370281485590406633102403379474957
                    0.4726467262261385077311170226867327211108531067303765619065942690 0.1647876496109767128873091359013778605564459949300077082976881649 0.1377891468355157066547583453490427008330641737670868569011546548 0.007377584971272031080350120909300473688703071184154132572527819499
                    0.4893349306128652395385465457283877456064091388305757201044358404 0.04011784448504295895650868049413775702895593778572344913416636429 0.1150384384045977247295091236274246967771490444066260434454967162 0.004014144882088527538132067298216306868007741888314702569067855500
                    0.1239356880633019241318199720046390019050135922221426813348651340 0.5195738380885110333949830364561796207324914903346381416922791618 0.3495089053595251203198969292242868414568176211429897405570055280 0.001593989106041045947164972699645870415641124317305184213927070482
                    0.2355567829855806937016977597067061645068030700810620950324302321 0.05305390199422886617430855241636063823572693718907269259261505288 0.6343922684654064871473978500513946456543618303961551081864355784 0.003703846649117610277330088609450710019338893241733630347351146181
                    0.1143441172456508650876804094792885797072248700449858679721713359 0.05678690216092676987222847590895371001505216938916223719778068764 0.3605929038932779672019220479307747571259146233996065596571186301 0.005006782277867077854699673126783849749274307159194179979091282583
                    0.05475271747429427774922475502434508942614417356531296936465316701 0.3387302116200136361389032924098153238043352285570450357716590188 0.4767983776312641043813602388564900854414981153527484018133878525 0.004248524202266975579671363999066881748128223335016440050481274291
                    0.1631696828250109462700013818729020101506589466007816231904721807 0.03982990703615583965881582895911580488426955772120260691488475757 0.7663246234959557414163833338944488011466505439154049072832277818 0.001179357424844290811134291077127354296310838516720977268398694294
                    0.05771554449877874986603697817054954889717154939860799383239100084 0.2512096550143683093196579604295732915521567036525970326233224698 0.2387613854393196140119598653719799155490425851647596776009480318 0.005514048362431835660044981720390071265664607483461899446582825346
                    0.08417146355924686881103104929730580423604250126547547842669679180 0.8182476101555327324826557976095332519484351902155074101469895169 0.05523459273903422575878904872526620769109428174052338390092140188 0.001794391976413496602137758991253566269992571899576925891819700362
                    0.5076718937835470258723830600579951717363649834284602431303053258 0.09011925015797213303145447951822942761861668375434369080154739747 0.3419512636889052794772572235466153593040303090208708973559848307 0.004668843649175664509823534775756796191513312986477149423801717666
                    0.3264460638947224052171795895027887101409855158683271469847279720 0.1084566324697749810813992253801227737484687084442926038403727332 0.4666322968555324080207725217934507053883109193438794845729670988 0.004816803641500511571771032448676641256195505522664074922789549257
                    0.1073777100033773039865216956070818387789292457306870870693686054 0.003473652130789445962964630674092348033848135205860052061092223650 0.7579078272848553246235300824829030168070253344194263885484730695 0.0007738839412500866704953468582262434484460577098976037120538695449
                    0.02299984262281564559738081036387761143215363902168280670574181608 0.1881679329336693070976286007190250225380775756781909006043307291 0.05447656181832029809333381902190627949858658082564904403182063629 0.001312635941887275740749954417124384124933358502309935910735529727
                    0.1772369956717510481701484547144759482883919868775165770160108201 0.5619442967972065751719611055530196620143218824346932013716469498 0.06696919215307308236109501526117276062815636806098583836456554175 0.004936646362265110938319067963834445972047132737458449263102842927
                    0.1085728099430210083933660422053559931604782912121610932074296132 0.08946972352632177333762892524022345696367942166565732939249103913 0.7623717880729585611792846424017645674454607867277663172923200938 0.001928944765926705514749832309751560531877347149699226096966138339
                    0.3615037289787641034560582010172952714464564666025783476076173141 0.4660061709376554020467306297274037288225274034912109359822009644 0.04729039238113300001937686584771410194089843586263848483253855649 0.002552119315478653195479095491146940322429164895586923747555618223
                    0.4223835418693319465194808969367678657442150693612787926658025159 0.2967267764228133643541854503558991038728877260008393342269401840 0.2423290849389467825117574369015654759689642138568558985253773921 0.003514853845365496647110185062131749587968488553478561950743801960
                    0.02582524641801002795769058159801772244752149374767655444258196108 0.3741297942623531184822979895609752891265878750441106741981178141 0.001721601852366721645150451870676617901072861486359629195491585705 0.0003997685043904873489524244640163081352108781221452937983839184682
                    0.2712389433439547110712660250163595087032066326683677603186370242 0.5535237991995571254608492072132470126454241417879370369791901084 0.1395124277548756137733909153570525664429550170674829059059546790 0.003724122331919436686632775636162481178839687395588888883509251380
                    0.04075315251297447037892204064357209685417169115763903014975437453 0.5083066212515908755882044436985343688349013925283049388248141159 0.09425694059739584789218960743101819136178141366125445652806206238 0.002926635851246409006274707094920944251223201766133775488528722733
                    0.04032155301515703192871602716941320801042813885718300520164516648 0.04965712986378594324290021329803855682695189860443245385268261283 0.8518054826722888401433522105530926278209507227409550715047247634 0.001248862948510936341235594944428305953993580469991203318426147527
                    0.4228847257063539948046194359106349331971700880895942613949727555 0.1936583675016742699033108687095192107107597971621589269529399698 0.02885910016703120216023470205868277202606475424574637402950021888 0.002297236209399746777480985842341273232625465653920365025730954537
                    0.6944311938354575477041162535462688431634064816445960980261537430 0.06653671351242129232109588162088814733647506806019320274853197633 0.1348110124639189772154596231375235955360514516571683658796809803 0.002990460868258238320172997006893538568395681767974246742430735960
                    0.001999569923274427397496899832802351360621419419566202400591999415 0.04851037163912422564758249560977817900137655048205291642179272517 0.5446912706177734743259126024957720837967892006544299142349439603 0.0006303376879051898961023157330650522718574211999489106656974885095
                    0.04627936902644481498018607232119313675442003946724819711314896273 0.8791318419467080660979615218709626572157109838668508619167982318 0.01069983073271708470110674769408800719082855955143200436954442301 0.0007800179260901237889924184766383610203611919369515327261724535910
                    0.6968532354635156088595408746302156327140534236916302113173709800 0.002037034038988559396107503748834273099632370638416725174816687750 0.04347885072887794741957444663239906438833080514462612144143929128 0.0005402377065332065365633819052735481101089353702962135794349480823
                    0.7742896510638277764369894700402108410346876651491423189942459316 0.05814627582484519183226863266301223226247558030902169372558746711 0.01075197853092715459000942464422234596392035269989159017331896682 0.001204073261430135683705102124920562052693329291929731575066261163
                    0.002202463330715364304980260941507865934151876317973498180937821453 0.05213910020563746170329292421329024634169382295429280161408840369 0.02832554040215829789979067202035230531333743315021113561129827488 0.0002196310583697609774425149858862872042034258910104356300438658507
                    0.4125834777024173963005922400443012379407150836794194702813933295 0.01326914119113054833311999257185661722342567130994171874182862681 0.05731228323026868343192162412391123556366226151790727883918588979 0.001769444817878116947122402268437843207784063348793658066867933113
                    0.2376809845348342609230060477361749432284090819138680180682015882 0.2975978030103459691107186519626108373820288886544292962130875396 0.05302949886174449145535120312947350963365429293822602644981505729 0.005674723307449480251919809901621645422735931154525208754760646941
                    0.8560405320015600125099656639068848108974631546376642350649818492 0.05596245357065579311141160147206059970616615371303624472967126037 0.04697805056465681240159830735851988719537184098924417361928105188 0.001388757557557436376582038558664374624244251219954619816804264619
                    0.09093719696461821163985217906348718044451709124374668440350962227 0.002569587079316577054291728495815002221912039922693370274182212655 0.07424685683590799804880544629466532325125624620588790724496266860 0.0005396017700250865273418104563819530593509608429705940086427754081
                    0.2084600791804546118442184027224572114772719358663926043151456388 0.6634742782532601086475788198048403125538482758034725493796171028 0.01046987823717813311728443752271321106789071099820749395132005594 0.001814781869268588277767445671680840682289447427501144503656207468
                    ];
        end
    case 20 % 448 points
        XYZW=[0.05176189099545021902135879427347211609385699309964415637925436364 0.6686336731711792527684089935657200206567996712402301939425826972 0.1084717597130366898407523825757671286335027040741849126837828957 0.003107085544969521048852193806528106447055998045487406053769472636
            0.1445207728649141920988974390761767456232657919090914082570022880 0.3112907119261582577001617261393184073322915085487855497134064602 0.5377637861812189307777551973913786042019695665458166791313471118 0.001457359171577578235122973186924307924986158680740061590043775897
            0.2964408046307035564280828755472128286034266201095315722582412003 0.04673921784084519812768860900477599479968731133210032813878191925 0.1883518725331032574001969085259240120085908426985844169053192134 0.004102998287258934908447491838709936589057304095602074329822677436
            0.1491917004775791954044735806162564098916263041800069090862368518 0.4974334623576486473295379959722321377215542651987612870533973803 0.01456993760690673442063939760187947160775849341882566463473459873 0.002314025759577693386804589856771994420605591549022210409944518482
            0.2731420335589885231240607433608782466631228446673832036733880313 0.5983515808166141874988053321453796865456753286746208335040405066 0.09512437283765904344096868088073949243122145123167814110084807590 0.002387636875935567648114230441532491644578709190101688354961442546
            0.09912346934424947346495062024277539450214384151838513108660742236 0.1305323288740949593668219369823821686753225596700104883917578125 0.7356464504143902553528465779289155206837194821414630198943406350 0.002082584674997161175949604424790767864502196264428872818950228979
            0.09844354027183770708775753473203827456417899672717211154990154506 0.7695012479283296403257371463306262107150146980472971997598967444 0.1251607838194325923358394843413348273010186123045408529668019707 0.0008074410869178663861244789814955833017277236894555386886421207461
            0.09534269257726937867068057601891870344467334581898402274933300019 0.09931718088085321298758864905463608693857912972074481533964521063 0.04477772214571750134810450999555370526299241491138952519723996284 0.001525255430642390708435072830833312741249896673163300805917386277
            0.2254435654542074651662138193358378701537007645889156518484780510 0.2691838068911688808225216737371634335139033384314583819182114317 0.4535562914842734590228616533990727380872516577138555988835827430 0.004433041034531512515469285638687447344709916081633643931845786491
            0.2985539133507323810687474109745767272284540200111134554814077899 0.005795689292760731805680565359046024353523383591044176252294183309 0.1711683753511131096727972983457618689670578560069043375431917966 0.001280798861537931197877761310009550259349876495245749765140965772
            0.06398041847390389770652793086464924497988963949812881901371783313 0.2538386204798254370071990972258075884729388620508073592383913849 0.6722580571068343174671515443140012107979901501184785658722815747 0.001373047498209577496384159807931348349421858864350908907430890931
            0.01128460748628307005273092160777621501576518415219555853935016217 0.4539036850068060943828257894686033517186446401641997602900602417 0.03705239957106509281623401281765414581171173994096717508665357441 0.001077358466893312230312126479169748075682690872946280346364964755
            0.4773202785359470644181954930772126225234735055188738151588941654 0.01128472166412187770282077572453829561588386721739558849192432359 0.5057243227014259125306888805950565720624126215103071623118556354 0.0004346304149454308390092045493405064906617646939336096075872223892
            0.006158672958109030278120415656661709371186003831305645626449770362 0.1867697484413706705398521815991929117424444838276923781570933962 0.2393224185228515929035855821094678148411898777420739083482438928 0.001495681407548904792487125068282289972278206961208761596251914598
            0.2377672604694621488055065272769968629470394100165233267641113121 0.09537421495653203442385636854166698551720226169731684896497182453 0.4342899086096565334108760553444172121078493584198537065806767120 0.004899023309460350149339587426658834163998468920994521277635735823
            0.3116753266383840567296958991744991951564355859029213045549675412 0.2215153191148421905346763208800568080414546967770234542168170431 0.4600711622973457968350754058439174697600685820448198853187274379 0.001319215615288731074413490784912733409363787903260109742845827193
            0.1246916775528968938071153812176926416165408362649968128675540362 0.3446229031476763682102732245318364052767069881908315653312282931 0.006847492601894077076528561628584876506270120904831090653175832059 0.001301910837618921149014668594436783900976071842524150199731454908
            0.04104972108518968680811113388778155202622568731667482009524877640 0.01006312064617896698845879742513769310639950946170380435226616671 0.9172225323060420630286177618311712693615656567533441895805121818 0.0004540315649027884347353821283461777879105443488187327806016164526
            0.4732507308685079920552989545434559269340479417851859682304745740 0.4581512871510728660949115837097366151932036673290166047412852547 0.004914288892300145703750988182334708352879353341313863945303063355 0.0009945388740717456513217092128738730623102426721189000994163200169
            0.04575710049430664489380008934902454455488168821618665765015418884 0.7076245519546613327827098029860219101212481457037216932698634584 0.2382364835469147824455875797857100201608970609307264712018932573 0.0009284116344949077666765057200009928597033790603089398614768670052
            0.2115034712846594189975697423092917629646511537802049232298375276 0.4929048453453171754839643113978216002570586857749695071390375131 0.2851397881398363407840093768693685587765851771076893915086368038 0.002298624236284665021001217903397710456412486194522239580145747134
            0.1077563053686785261884141825753624626971116589800866650985520528 0.1694442104469602078899275994725158411958848845867494305586353389 0.1373364430252521680836036593322385577540301392329087254688860061 0.003872853883294352636720609968885972615367576664274857599400212759
            0.04357758269384519777908492648906892648766600149885623953636203324 0.01203247177307311200057266979958795359624517559352301646794358457 0.01011703544655921697481025963990508150856427994973332497782057953 0.0002792929263145275676114550469897286565663920186424574348520173324
            0.5976106352969847944793829378435400359527841749835378522862009568 0.08454755090556031753497487427999843598254002642907995779889183792 0.3105284683423150184558508381878680086456268931759471316754246558 0.001166534226610107152269029283714492547547584563686465210718991108
            0.3136344633387390616529231351154027080233673662162493987428902514 0.05283670158787985031612835813083819012446249815859272970253535337 0.3052483630949164146903806762065754448518292837473138656657817920 0.005394824697152904037293925260714963032658130610234029928930256260
            0.002765126393014829556638670952551343557413521568632733629100462982 0.4216015950642834246256410894785038005268426634474011194302592437 0.1063588986759462446012661192567855471429761577692585866132400081 0.0008509031749583619740113673037705869675909317353972355860141196227
            0.008757229789263022302417202887438158191788532011573695850641823231 0.3140941804809294111554854659932288918896002888883700204726619169 0.05589625283611541770716736938966150233154731250063879149873019425 0.001132496252760835656411431597923771561604866712846988234200814106
            0.006835691214710020822142941391360353210468247387630342179244201393 0.5881819201450984069564032016126083806466550168935208993859611611 0.04498077961856312730643176759833353062332790727692819944278987432 0.0009500080463538882685544097282242431761294753944705276178647762407
            0.01126992278697366633366425864545882414187047810013598951151737807 0.4626316340057838006253653619592209904910119728477304926625234386 0.4578790895669504726140616278100177168095476583645824101818785316 0.001462792058534693634779481112340977490483834777087868493787960423
            0.005197719831154271859287078838323997902281430767748372663714694480 0.1451242951158649682378504589806872315048318351579535574265671689 0.01094538160391803921878429859308846017087402115737417236208363422 0.0002829847578586198464745497766714093857885844131028429424580603794
            0.4540448959009589347206904453436889547025645257929872069005064428 0.06513007964232350764092272382431629912273264999479013769549548306 0.4714395297492401298204295078054808191893239215020615992350366146 0.001290903999494937663725038754948390695469299415488268693332989824
            0.09995149034926777657195072884044083340456007960502731567549886468 0.6501281726357598161704987644951991200424441836712913103498680563 0.002908756243317549319412169764359922245172366751418074743145194234 0.0007798359068595852595219640024721367438519559202170645085516936704
            0.001821967329325632201147442055099750938048237376718928535195648541 0.005971139180519490527764644466954715599203250414334058868679330085 0.9614495886817080831685070719366036655957414204117553811048663968 0.00006890837031390207998281058336387307253515754933222734532177596049
            0.04733515840221362086284968947594634797265010642185834878929127577 0.5532567399291971507212377068182775038691098821250828529826159315 0.3924335149053877873801169676293111601891106532777939194895189067 0.0009294695312315423878725990577550796033969059510275538572703504318
            0.8524663495592798722244146632504709048785252048227207558852047707 0.1338198568450839122148052167661877803465654835108411190951619338 0.006401812143118802329605702682136405020353120573130750985191171050 0.0002215285421961561656310311675342474825918769706762280490034787245
            0.9717209257605102806083585148239690839820486091340008143075139162 0.01133580480787668627080715278693534082916014169960812037009062384 0.007508331761305823504728728963546873037324259957166698573544298982 0.00008871229095724844654973476100110585406612418072230931187188775682
            0.05308607883486364068280185759772845103531785484656082368908864947 0.3097165928390035864416076024045047512611196591807131690488389748 0.4442947794507916765523069614806189915145676692653328745959014150 0.004303918629837161018144382598980792073433298965139882967619900362
            0.1492896170322573921868150032476555269698826725329336067021055482 0.8321351685069382388497408647816306711777960439375086327567645949 0.01301389805100033796452357392320730125815689769880684996150496710 0.0003619269782063443744208662993640144213688348067300317989146193893
            0.3424043403539250780891560296356852611620318191779023183399601766 0.4198413288071035944089603061209260648735231510770774476345175157 0.2287006664388674493065473625171652084299354350102227075385353360 0.002141429799569417447498061778698354994930230612268903773002094036
            0.2254942094799715307414709119756682923596079683226692313559213580 0.5765450510200452403474330960859386962767765795667913811135334224 0.1010188815784179614791409721158033169897076293959246309218249870 0.003706038177953958098983900455237499825198748318393026404824557758
            0.02943291941206383040123725223012587837127181424410413256128063787 0.5689972869725863503089938002438454741085978275576233028880208916 0.2021484220079198862741209251359139609529455554707256537077356096 0.002891317887187616500514022439075689816259535028162710016657012168
            0.5882518344837742344967967667664997298943515040327806852491271347 0.1608981033336066163353737678699108213734441544724157399158958954 0.09437148720481567008719186675878931114147870221359194864690735362 0.004817536745952302004628411462832041854063111569336357762464028181
            0.01268084872242645585597691213541066752059456723734478538150594328 0.2804748687305570749946463892475676832275838877819056591839866636 0.1382722172240307758506918712741621365770805559714760356814779743 0.002126959799624318294866311718700150373576687132503854789747850551
            0.5268335938350829158999184073432951294973597460931923504864527552 0.3509367776338865802710333755323288347781408213203466142775435315 0.09274963859395745105422989621125610703111502379155373919243981523 0.002958713205789752841073225805137281724766804600130246902438771686
            0.1035347256030172457728033165589805553902396665484971913322169274 0.04212746582951786818243053470140907441852908946985827434199832020 0.6925910151028342162962920726750305263908633001575740688333975549 0.002801442336951572526940666598424453946589030345770902261083544781
            0.1206557117283765479614240456611573141791792268479314485413064473 0.07735664930279657851484177784862803781350444516464941884229890509 0.7994608571052476610183461684782253008332597712372223314660035575 0.0005013878147064998762611940193493770055685734343403323997699069062
            0.04880608649295586999364619202609034746868096485477994409990615087 0.2109600000949483709173699404571949014196809727492247868165080037 0.2225149895710807491910361732826793123794787484635514584499474987 0.004563629683466318612625471908481812112859139855832292314117053343
            0.03606329794280483509817435299309448256245461169732692501195013115 0.1278223721709937904138483225295462393602204942805541208242119949 0.4995348506484732656472198206107749223987348278443919232042754479 0.003601270457797034466192556518071893870503563715622675109470740461
            0.01525614613357650080982608112453719528112829518510847789503150549 0.006003620177660945468113812423019106531383968115932500515535196419 0.05034211379204804220793825758706452602365762178542610706635064535 0.0002326314187314994037140065334424480414005108677324155249514523162
            0.003455700953823187312837772469361021681774348420243877457208952858 0.04514170400002618592732359985819174739127506495391743701061426814 0.9395125370971743636964321263224021232574526632729894380164633381 0.0001354583959642321194353268315182678329477256613178786028286682253
            0.2946534476356800350728357056342692514433569362420450140322428094 0.4763634653889578455722316674569853331559921671495674438495452065 0.1814467004687850385716127616328698808492181935271924358624031179 0.003904004305001412065124605362680156346670140809013831210354428140
            0.6304189997892001959751252158281111501305650765522743453370627381 0.3086836314855882933893825271599684552006345774999209063280541185 0.05231185802880303813706336179776514915351689502421199264404941587 0.0009257508661821250756584244741267929061378275301684076371376726293
            0.01460473188441083883052733324752068350482873934973812029051423237 0.06219592778925679356833948873421760298987338959542869414296096878 0.006993261494587333164853749896348660057590421030351316711590224529 0.0002872776248935797074370923534809806992077862890186181931585081521
            0.01144159375579308935357175288756702143882247688569656181233245832 0.2315570597794161847583101462699932240750282288188426617189295730 0.7468929990447108261982967319744267292807344048524653945684382238 0.0005481822835240644996589600522825016376248812002466045014964401720
            0.09510922632349887219114654469467485500014379150377682677398317493 0.2295152263496843778967036685233895825521469014581550031233802576 0.5622895187206716671733786411242617584820436792833639885582839282 0.004133000989552432497160200274779364505102569282216468540152351600
            0.07495567738522908330237566368263354360384884258515531973295965357 0.7860477811937148391093596795688094848748915133255878376102333689 0.08103088738930594957498864918855171605856363884154610668791654994 0.001724902483070527279571836740055317799572771659753700279344415843
            0.2387337576286972478053340819984152655180110450135106184150524526 0.04557590494083153172204093132448957564176283972407585336918569057 0.6636555269091916524649437164676113826765481414351601757595665760 0.001728497178567639362159431799562160142721223236720595647710861104
            0.2257416719890206043258606385177863328106713665749015959350684172 0.7201223877784439685865162197360987432882438772588900788930674688 0.02884233746020548566355186489012436145547772760743172525400721071 0.001093285562399677237959888470816373316524434525963532908226827523
            0.4852264463868503079415797380160755339628527115349435681093727086 0.05260003866616920501773840867924820979965226433670024432280936135 0.003748345476851786901594217285811259808233335713311772421610955403 0.0008559062672681094728531883572905658629830965652941198967271614546
            0.05336218498558855603208393687559294672389394465455430336784189389 0.1410304325991326522819769496343284160149339735660423791183408315 0.7962841499262752054083663698331448653011322693697943794403242955 0.0008617424236163074336124583213038412412449632949820848687672848352
            0.005316580874929026711006189064457465028893480853276105147261004543 0.01089642581566372476847997125740664830690621842756155582627622125 0.7886643328514608059996357791025645651721997808894494295622028436 0.0002979838464005924099379496849740509600830066429624120171742217735
            0.5206658827996005089315431787928375928312849143577906459861697228 0.1117833518884275609929282192425377538613557925912240780271889039 0.1912251468413161012139059975220624618797907291351492266013434123 0.005335424242303601976042871152031783520952263287940827569912096159
            0.3233603520217880885140907975852512254782554971848552118572362480 0.2149892183859858647937965929234328745538247103957543805161573213 0.03828281382471039793273551046430812441778192183419318615512560302 0.003937681394715385520017083842041433417193642839206577248816965762
            0.2261328653693771238506007167382973628119313062076944749371859924 0.03163774151595174407389989049532281001015227496594019499793357136 0.6001398297241915516729313991242527170046299255440383429334690927 0.002685956292327512673482126919511026106206012791477742578851911292
            0.4320583606753729311143049712366264488048452268786960344620186119 0.5480033837891526533668575012797748570286730406107263762885294997 0.009282029967344873223434076307563615598431086586544021937658705114 0.0006021875186302359050315711089619704241798823815040950258433481007
            0.5827684555839118193358992088347145953525474559092826615984224748 0.3952857115095319289330323791007473999548294810240505525867966001 0.009053274750857095119563243050363759144594228954694928196730759337 0.0006733421361947643082335707253651553900797182529870196825665999202
            0.1228001619232911978685436759939124891320045211231632733619283948 0.4623868078097165304587550730534859878643058916248338222582615710 0.4078461610112604512850340315516171736621726338910985308118801665 0.001387908499150692204139085589917113140293598606501149947494770962
            0.001279438767825376767303224923859009199640241034225531148680087780 0.1818666890200446849869875900407402587349978513210190496934264520 0.1197452791708551679587798496130773523692317988578720939890683590 0.0005231279269557368692248863810902134373497901576257754305179112025
            0.3853217002625868296212163523351639516687131764421260808460258568 0.05896150711553577951409637256705409223457433854585271139768942666 0.03382039126904651556131145221898836030938645299244654675254203452 0.002643246068787598398704068794843260639694038823214519167096494805
            0.3382603614832800654507817760910056955393009680406991935782983566 0.2195846594237845362191523166580195907630296216434615130923424946 0.2809820055149861927861550412446823811123358122634969983308352821 0.006406698323570838671855959622656338038287018108165787569323786732
            0.5423695959569337939059961495323072204535564597057225772149274359 0.05541674177001225124488310883005747691893213259237855463566532635 0.03480947122040383993114039458885280319181698516558575589524497499 0.002453223286973615014389972932135324713887893089008179780501026941
            0.8301677226620462367140898227262859566873353322430622501747469587 0.1064164453311618608091468880005511995410237012776281093139960816 0.03312194880742013469681912310749672617891496659566039214226587789 0.0009683296970747728161629983911142604442907337844580189594224051976
            0.2930082761604346681563101243892013100444482812555971744437399671 0.4550554264683603250504057291430582706202177913976059986348945975 0.09651559792667012436906783459559539227025982811119278420815236044 0.005372764191351208772977322534135376341900955006994058038057769143
            0.007112595066475316213845789406590555530767446723989566725078848405 0.6019313559107443901748290625335496793206259355246681010985684343 0.3514366731007939652191919549690738299212141915284944298042755937 0.0007580357509198351548275227061451701790452257945489699772756337194
            0.4974130982606827342827060089566264267230008495652699384323377568 0.007425507765526787587824805265439837343186236963588532556908214145 0.003470938986406548210384297859743304485860184216030476730204703191 0.0002202934178773995817464598518427860897502939274156457849187761023
            0.7946767280564517288286645050419508861852080782472164529655860724 0.1522844919788130524986135872917670648370266502725628727386770132 0.04968156857741971660559834063657971290247875480035368216231664901 0.0005400981124716870454829560188792325126938612900761109531847059792
            0.01071460060750092569633587509449062205251674469862524208817145366 0.2544201783946531102054212982385855988923942586007762095266208877 0.01177059076393613537570903116035733468445356353459813374214263639 0.0005693134841683602377455824064913848556473774011892071478373959256
            0.1809343988567309385518683401027219525217413619784820471866230488 0.03782733158264522727094281908954831901678847644506790100319382395 0.5500790677701005881757264150474561832078501397869027318128251604 0.003359430123574140224852898961394103947541604511932643710316156741
            0.1988322907184049815294841522625121497627529912540275527377550072 0.008462114953086817117249786456478857609418553797771562399573996531 0.7816726480096077548727857753383950231689316566015331515760899577 0.0004170580338430783898688157182246980887005841829580472921548199302
            0.6016903749760105075844121281906525581676906761931291462121472165 0.006122659699897681845611578775041322991232837433743088174044444907 0.3735421041125367989703324916297203671039310006648940545569282287 0.0005868856477917812185660240534423530945427453561051504149900407021
            0.1726239908157005128645975058218774014299548484285365367497298701 0.008391951686465747008085802585028681805496634852735848639066261280 0.5220423997388107572211933361257964810938972952944491685469740813 0.001566674705471749232511906427942982981487835998871817617044197594
            0.05142180758471749309339324616504185936850312547267055333295683650 0.07994824017875561967774282516698216314190498244846608499957475794 0.8066607522504215500103098652555015130098109164568103931438316041 0.001725199538202636432929876456017556458226816434398825999260505907
            0.8272104413112004701588118866259170107252078580734995316866170948 0.05856936190362168362261365485152204870746497420158900245703107908 0.05518233546319158569600439679638738273355457499281917775093979858 0.001304555592208003072436235634591696408343345573101935826548604500
            0.04173389772105281446168311359828814241381758676008852939941278186 0.1209687135941080347784599063273072731025766925831168947275502217 0.01793553438027054656095519191385143602042635550736910040899443007 0.0009886139633061802413801342128040497565815594407980585397608729933
            0.02014284419495195473858956407407518510465745557971033053651406656 0.01034715943700542658805079783172096050371650193688843903570937672 0.9678415931465242863881855084094670660173670397045251053656839037 0.00007814554030935479813193638601523591910817252536935347672625639812
            0.3377847851772909260255778093825567021874254406697752152492276405 0.5233133916160273453447785854011807597139053653163983926533589623 0.003717363200149410792819049413476602254729887942707823911919677337 0.001017270612395557569505230346652785461334701968311710043094385904
            0.009084114942455479546582965656302395089122426400440460254845016718 0.2669346887611835768721982710313356834463884470351175970544620249 0.4961850979916116037139123688556937198809569284468706825574710169 0.001931715456331735496075752476033742237576886762039155307800864919
            0.006495578924823791939700288884210165059125664235405050052097717603 0.7087190306790355317757245315369598372515663455787609674388416639 0.008244439842430231559107225646581633264441699222964122851986629561 0.0003469862992628505889551350972760080344187444834320308308782374866
            0.5991570943645797482105991891456298504226588266732340795541077787 0.2071845056902976469859483545627590284373567430276368436635930306 0.0006765157101435335181411014518495785031846774154368360633514107265 0.0007211025214609651974319776220126081872729370332888737926278569004
            0.03031193014347916323675804479182601399374617444357709605001723107 0.9008788791745106721921711305156183359334927675628148189280622486 0.03536509131904274228832428094538911832212096905363587300059780736 0.0006654886521182536320362162788924763838811667625271570038873192616
            0.05160329249060285339820647326986936869597931351283189697533235380 0.08940635905022541326699590312518660297550674045921885761899957380 0.6293597645873683671786784142915303198097385598875186394292936039 0.003179983565131586402443773236843900411820046846455940675511704710
            0.3684167751897960330455682942139599239531071987489093038540470978 0.2468967806918972215177035752730095726520122702019668426003941413 0.005105075886735242413878246235257702103759603458701061470991876314 0.001423736935372220389863879926573851836503025148571001956265746628
            0.01131167314653621436029636978101343743299796051643251817414090881 0.5508132092345208840627739712075731793847985110305113464644459130 0.005753951427008825287045262740255680798134393962297179159079924976 0.0004616774959288515105503546757890029264516968017081564339593512136
            0.8597584766831858157174398584479131465456364160336433485559690399 0.05560702389242012736370731093428812378106810180405149654345781984 0.07564548922068136372983283491070960230500672398408328615497086014 0.0007213006254535522402240433030669067714754425269022773940292157380
            0.009508442662837031248809356757471414649315712964529926862668670917 0.6616071996435595215601541832162890292958883724695687916692318358 0.3227539174200895351813486809154065027312801717038935288226328215 0.0003754381517241716992217457072144466082680523799459615175580428231
            0.03333702157704420302315649886559595597398976321986027176947446490 0.004554412769692337556117543384786620054608159378325088620881646043 0.1466369373988023826759465524832836650593151034159351997091072458 0.0005118724378285561955346904350624917197389048585387843743146645196
            0.1000589337459292979089728845627812898578386298574224165092888025 0.07321433338180728857296505194762345320149358533469906412415616461 0.5198563891221231442016652080374710615899592281374136315533517086 0.003964866116688008333718957099238326424201379030777737420701607269
            0.3131649270413209279015608234829907889400969888944210035370023433 0.3398285891856594401652428110922977034253086650548306078750432443 0.3068290066108602896895835111953814577167301351082979345261870367 0.003870711051509834637954021948855658045601238157154311806855778085
            0.06800585490035671427892138319679297420283054086187243366049249998 0.7144658194751460421035497977355107723556100275519983528076010293 0.1769452845053561774166540886982654021938271292949968891772381472 0.002351340915737442642442093319136788396994012186872807609008873628
            0.02855617687440378411300666820793448613154315136027610491436277898 0.8171949978191198192723435477414871292342735598060271460309712973 0.04800415187444813336907336921497832280771703217801764604425740315 0.001407190660854458258848855738795028052506607982940790694483279004
            0.01168225528257991324493770801707086048651638434808899101741123364 0.4457794213958078747976396530067548026658675826177174932538144211 0.2104903331227677748294818170189406045554987748160020532577984898 0.002203088364673357906089195831473386228212052492024499124747995962
            0.8495255169208516094208575440979730291877133865426538420950650554 0.009318727090585730187221065795155584605102207591007618016158789702 0.006401433978781608895195882127489624053745206431318263844198627576 0.0002425461407347386005248310532170090577423698285642116334561107491
            0.01163346291511932810261417430393382971128488318359812973291550828 0.5522783577227994762953006072276310522206831652809860031536414799 0.3269311240228479892319488710177359688874810360282249371851831589 0.001846786201713081097717837681973468350415861754494403800770992370
            0.1014517387074586365994283727236003245993288455353887013164769574 0.5350053049351604182348504978650472307870206079654631868211384325 0.1960318868245018094666715638791477957286689948816166270858293667 0.004651979808189261736287235271548789223410654563241464765312849688
            0.008741768120621972620723632110621236695461318983263099093398436203 0.1219888285961629131342996520435441189388555261711791498595516255 0.8613371656600620685462352839752054692614484086293971669367259614 0.0002836709016661483373233821406800795076985463406628250779238107635
            0.3616246355159527808004915246060256536858145831651720319533908891 0.01267650156015758135448336353039364115368773672101130859492266517 0.01142227726097214753538760785775966782350429524388026663899439702 0.0007278594392964987251467120897369266457537512668386766157027438556
            0.2043058899087793551198862635103259423729742155190579821677687945 0.4997305804711788935277020642077785347893620442212933895929874375 0.1839786723263262032659526781852871125846033312705413869273290529 0.005583165743900571968939810092652902030298752935477419542258697535
            0.007665248554209206319318488526478920952970874529858736811297832460 0.3604189301960600777072058766561009929294385680850297887979879537 0.5917478630502644779976184568182370732447767585203948637406939074 0.0007782908454910449347176533921934149830198413373564833096788662966
            0.2357490856086489432507807318883571337399765085843788547943192133 0.009194848490708377148545719376895755136818745322883951317454356331 0.01246435329455741042408889662462765739220655639678206368977420943 0.0005614416186076627747175511070796363201261325724308892828283322952
            0.03738144283773482841688537318899551773587030852132193761285149868 0.5838348530834684836077614894817839909532112009138623520894255211 0.08207994342619603057894998944721373593534093160423542346935643655 0.002752456699584889019841836908737237129561904912709161220916358829
            0.005334961623043074096636999829900413087270432426692491521080595685 0.6728003407175328951083845152309876366845217079539214533121148789 0.2069601833083924646606521577710350941533373022137172203748205973 0.001033437518781633341104148462531404684172731494189390413446911065
            0.6955907690035447299490837977652654526863326469137550367901289055 0.08475887521478513788222342899970627027971450279789155242526118368 0.1384347782350878581073584344413280630432762984900563318670591989 0.002853437406158075822768400350354252891058748209361640580258274597
            0.1630478997815297360410654070481137842252139659468999428602707409 0.005774909699740583039440482181435622657369686865809783540887839803 0.4036838408051474976697852559908143480323168784498213236283943996 0.001371187995821543992462502417001473427761166391616536140174609872
            0.2322434571655255695716132265895382507971249361088144486361047982 0.1422428705607415612471777962279530392202911358936997118707917331 0.5725915591485994453904719403351419984008710435955192948179149451 0.002899848760738934693163045833438839811051018656801698588947298039
            0.5437380770098057858492709709314072391564261848111773530718505074 0.2399181731264013044412302912576678401309069006794975939291764981 0.03941288050817119250505923106996472477856329293263682274108033462 0.003593729995297211221586026200506684277417074298868284049479318393
            0.2597714462570104273272449074786016493733676741652045490010028879 0.5831902206272757006011257921278104298947263719235747416612539996 0.1518957817497859374412277464715752592776659509630490117755857559 0.001270432092788583586835602280661041645130225374590694334298364926
            0.9338694804567333647380094321364141863861113980228220375507982664 0.008823288829974026978355962155503454689206623378698278583213980871 0.05013165002365454150038719616788751649960433899677938675786903512 0.0001803533600011887431963880186552065130819459261855552994600258435
            0.2595232326850147517894656457072275374607448491944578714952106218 0.2660681505027375727053940781513077220247722695203235714598386037 0.3615408124823333329245250773987519400638017314744887215353816952 0.005905031044167648460663868920039308405332535249516718141718217627
            0.4219899796605084038516591806214776791126272208379594709857785658 0.01196428529707447333313845534950259765261608437934008598020598760 0.3081178142744179197958400308925695372572871745026889479051817866 0.002690990763869373562968912869663900567579097330121853584090744397
            0.01190537715631277607131086295797363051494773252785653854131136435 0.7173585098810008806981003084116775682710599191437421463818115656 0.05117502223716721146431852089624906459257709976563229158902901047 0.001332039707760451326739752808813834416647580472581715264836430401
            0.6448518456293845611552035946340783515073090176535575922346118938 0.05862414142478212982180239248278081118066162963708465173088619122 0.006405423840994858494837009090230771177429526785495472283640856380 0.001003289429579823475135440561937797308209938306547491619699102226
            0.09544537324923807468491379753840506491075913746601310837478168449 0.8423039777061626980275647405082460348443420121132192078733633969 0.01567103824833175134915942820336418703026806346333893901256791824 0.0009448620003418094451181213983915378656715796815022479397921635255
            0.2553096962036045094207197399408094101479212842606978093334862821 0.03767139537884794890141702378237125357821593381286258881846174748 0.04648609669731106430788675585230631251979148416268179966323242871 0.002201763851910720284474554537068757743808008144902385048545012036
            0.009368716943999021331185826356243072036128882863987872977914064879 0.1105939588579164196838526098042005120808897546821138012177219188 0.8302459539314704112922141092096308870683454719250844048030875333 0.0007435516522374451321904432380971709568518496952325980302812581257
            0.1310021825388717139199840801094901752397992952850719082997889292 0.04180814739419271106980651512835100084591908722635830353261042775 0.03504218252223830196844625381555924393657622016422899912500019921 0.001490286741876701917013630428183670771673020093549634109765263300
            0.04638486553629214215912165091221373925347422877954254460365026897 0.1949146549358302114709320997258026041050189572626733925222782436 0.6865696456067888473295449812232167398498925364162134615463247290 0.002440859933656692264703767608012977782019383675080221794943528746
            0.05034371836381399857108565475625629773976797657412265670896151686 0.6030546910488420573391209876022548849864433769000492804899628058 0.2420966990017419960752290265808276101301661715265485747502112350 0.003525454574435088026249898827397940477028828739440566291244257481
            0.4363399951235478569844983643046437095268679626570591889988721170 0.006073811564740234830486411506678484703243147111393775583105085465 0.1162305376666131418740914476791327609611457976016057789312107879 0.001321321141622693996301284834015540166892117575153283266044481196
            0.5401384921987155534116781342874066868707839001689845155975377107 0.09619133745750177552768507623474372777489432906555848007779898895 0.09459362468672832496682874771909775709467812328443834792561964224 0.004083235732951548295602698180697979048251460877942385939192262167
            0.2349800616482051787064699745704928308937930844572244669659935858 0.1536270952556976682423843079787732446869684404396228996962569364 0.005789761886620951686775567009252470408242548909480926342056039875 0.001236835963606517526314552574904525330253644351500317818038411671
            0.7529084791201436858799981642786431814027308443532838803624467265 0.004364112028496217252605973552656233847550792412745474533894893596 0.2161552435596374013052732948459817677773388011654748434306342962 0.0004407339007727628197790550382735701784748233268505802651585198123
            0.5861099323713078990202303394325314905853157108528426702822065293 0.008846921233996608125424512516070076562812644477524465437784591794 0.06936040166616533450517811720246288534557884250580415477134072299 0.001356455496377926624131604514586770033745767921916679062359032837
            0.1583288022173349619901353202195471400533094543753226461334414662 0.009643262202396210766573899338198818724988705864100919860521803354 0.7119816799574316958942481472186391196021458010549941998144515752 0.001281939525928266972782290626595937810836912216438900817230858911
            0.04011212602146540406036775794602984675497014794710981044315827647 0.6305893731421092659115416773269415291315781623918845248533617426 0.01445625810158590666281321774914529020085716630049316664085994766 0.001366764295806519303075790892689204035551544116171142000729453907
            0.6654066869609813781711942231118865712286473289437356248749712515 0.2782298231314716204734915646931317898676943088055009886670435785 0.0007833055567634623238115465183710620315567108073698821988146718167 0.0004981970537043813648545896912035638298021890086162895442528118790
            0.3764326290539069616959562392020186966697807141392941750677728459 0.4455966453731874687665153368281428924660850371313145035576599818 0.1051260422142914870225679918330662758555521876716103161846175387 0.004690225012083130228063872513666651276066617255172780975846965975
            0.4184655612238410241223647680115701047561878130797017197302300690 0.06517176428454941871509564314168888887754604300212484859693536166 0.2940642313749996836788281279549927406194045110705971124802622881 0.004336056589390713938220086791397587392624334332076653506984759706
            0.3900524293890743029381888370947880104716107181069255551223800305 0.2549631918577740170188342274934095010645321158021947204605912074 0.2783673118254097783351985783638126383979884853051236771502836095 0.004422137145033111862632843669085266677277974367598369422075159303
            0.001232075172825196999773965505204704478951488147935182117329163976 0.04942797515860340748996562941062087680874761688235342787199339092 0.4305278688997172177931864664738889695018488750286053606647763139 0.0005456291637637337217798113196248673778659325206673062546934475598
            0.3972421605721792082177936093967368270873452494756922636351744542 0.4746977280685457212022833898454122529371425682860637562985404737 0.1158179327767269825744846996213796137191657019312934549457819433 0.002197967621693058106465359219009015215322982188382138086984410248
            0.2139401329571666397836296540953191207433411771878251198371490156 0.04241858351596687987551238791746825170244825449636604512634224388 0.7341454240339477065691283620913492697380545903493501832865152869 0.0008440278801933969876385194470169010427010220495677710416643596503
            0.1821084040539138135311221686346435051975273647700496859108742047 0.7733883122677199101093764095284829398868873854995997295068114693 0.002077984681257305680750716285372144960663727841655740884026539734 0.0004491227276721888957390726576035560018950661263070330919449436426
            0.1180750950795488360753006406164539496095439418073511152872622707 0.4714980901730676113861948739380430827376871915041377719202163624 0.3674114255013514400469252186885505587607786076783677205921890191 0.003295574170438482502822331305840194246358333138837405975075079486
            0.7181462583273070297627935702917687575652208096272757947319737521 0.1040138177712048026532041309321933465369797146469131494594570292 0.04731369319978569025299426361267886109083122315409724718314911198 0.002851288946792235463126054571574482643592120799182587916165262010
            0.002808613061338581568740586123949384487294773059186426453627960114 0.05079748259498374012290593432814727468796882506607090675292882136 0.8272862814625609880359000208744669774200872571448562315198515872 0.0003863442904816514730308077991482783582739853721140625246605717695
            0.06621387023236316207522334210643054747420146376772017697358819899 0.2004468627085348253223832054738964393923927317358422940879884527 0.04366826677221895291556447536297330918615123726273330778496035261 0.002560412843156728951699739460584406969917353290434680636769199692
            0.1756271727348402940867512263263632653535306865918125339341994372 0.007525031127982804474255101794694688207618873771420264373762261251 0.06352564316205085194019053410092699628105586967494794648069328982 0.001004991303756126403011021697056413723409367097126982907611893541
            0.1278270114859277171272388651165870946909810279103767174956446650 0.1282507801687583017965158179073304443808162846486240710451629602 0.01110582215502189598654615506034268376404118829417710579909931234 0.001369003623713428483644643523821644392737678333741483135413404962
            0.05168293384726117976508602690774397201227371250066245713081804789 0.04641922224627415494228191079268971759285799903655611993458227620 0.03275274415101358021240809669044443536331193348771003808621219082 0.0009702577449172968333967699224120019141980861222331905710023604745
            0.005693626042203222448423334543101983867012641254887306446497589560 0.7778597291369677381250794727475947631134405232558046443187351792 0.09114061690049925396600286141328851180329560163313715748199832477 0.0007135948965897211451508609670233300052817509338669133109625723331
            0.3256418729746251352869026973544412352989863885062395478059054594 0.6160991617259141065471691243893505662266229116350425491786030657 0.009668387803010784161928451676137252148975237406492938907710472223 0.001266407225749180136113124407707227178928666291172733369717937461
            0.5211687354548281674272160667354349793468383457879220062969515686 0.1385778667271367093625307392360423361177812591185497554283793038 0.003721478721467202354501977822114748752117461540464548806919617894 0.001126332222543569515581970814087786533193053203158293568794787399
            0.03990555943329014073994549677763520940816865732448210592539142841 0.05764771090048597521131046672634562550323092690534313071861479780 0.8909428274406821454712179375888830417530018424190142883379814305 0.0006577427423781322168808074121412246534622315455905851046254404964
            0.04580494075561376201460219198647673033292879850638659601702675474 0.004723685874891335599623430252382536122349170568221840747569404252 0.2712637609068596727352450977029523029663175326751106612140007985 0.0007391587702430544673039715583694622189261470165970884954305642417
            0.1931026252722276500672327654098245005029904581434489464298664346 0.3106384570945808429863379402264171412936719464308499352022003997 0.1320585182928657709330019655323334275693606931840833309230549781 0.005886211358898821617519716850144306358182944162395017622662170478
            0.1626353549970753960616415760982994955267924641817142405835407769 0.6260349658488038078573016948734146541550528390701285062608059788 0.01798938947427612272673730764176268171523591890138175244837616593 0.002111953429997831960254459835839905454599936064471735355686648847
            0.1918627433642978620298168043698695087228790235874754931148177394 0.04858656263788022054759811058506284416775595869873977671258859479 0.004301467292344719557633813301966363364521601843077131388232145628 0.0005778643522664406547410591171349297746669962096172605584949250141
            0.01188232325089154700964244237874997450031746812616589544656204356 0.9312626778533035160553051947093273468667105794153375223145907248 0.001427347115464506080041135589397957207148964672448530202784984403 0.0001241154453749459712647994407539939273229491311152770151126891893
            0.03567557910587030712852028341786657313752675633186612461112065410 0.04138894066127405364685561734718253428364266266101876972529260786 0.1924562787617776472741069856323139695334320479743707383608346916 0.001959202101336870729206950600231648777386077450590138910873509234
            0.2980657865654474057865593911699869385892410996947527260035713552 0.01055547159278136230439132545466348404815569771794840489611535388 0.4501929292080627149658957308193656637372163465379525518892933825 0.002021177281338247088660574372966519190906873803030290365546057405
            0.09949669572686212319700468430410191790209818181864023637220005288 0.5464416185883128832867424800450174168805389674539176938313843915 0.1143008357877377330830312825972038688051754529622196366115614133 0.003918439653540407157769694866691273432049702396686132597487436613
            0.7483911349186238406767962291294641822673785364831154748819169886 0.06025101259733271395819758910600062406740388247978750456301481903 0.1865735833947932116799575994422373631890868423673264022603263561 0.0007310999529451517138978545731795290857148696046260042255279635818
            0.01031081473728030932062824076307127792528292961103735744603683486 0.1025760104582760561640413206693388743246887561154032336239005156 0.05831432797541216522964467362364947377618339094463024039946909720 0.0008466530177030312138243149369807574284629733166752773903012293476
            0.01147872535257870571698269093493755520158845618958512214738774505 0.1298564376050223081966776019488011398159381839222132521730895783 0.7311070907592991275536090386377860028445807649326518076983999691 0.001594870420217780485746104546192161606417611518481420255999811030
            0.01333102010251893956587668990699635957771972758922946792932541880 0.2357966747547336174733094452463688472448915579887014403837670637 0.6192941688671005654499621646810965237243371448551756377345845314 0.001934674683397932241938941585388474471343070855115481995180272444
            0.01944990009248423105570354491109331936539241907942566388602658161 0.1334925254817289384072574728819981533718844423865327557401425865 0.1539086185355281586603551703847362855593677075183079564122220175 0.001888061257507424486716728624759278478507449461668590388906670959
            0.005498557769617641890524358265199704067756551910280453655905141902 0.01126928400888254759216220947514083219337618232985742317297692970 0.008685204477653815327847586930202194282896080914102006384344208733 0.00006922708788562311478790700004980067189261619023356253134379155014
            0.2055804683570703742864275803766242950166220093953104558574193364 0.3345629175707401159901819788307130889473969298298448825436922259 0.04869203430037039801384553449914161265556662893504945680587369065 0.004942888866362724438192191292681008134402765920674450347699429659
            0.2974808895443804329188963049185930692882185995070985590584636125 0.3717674079962557968687578529748699622728132232825776653459364534 0.01266135572278842727003273489908529646122411984746248601836912443 0.002611507757077515412449897060860415771681938551882767481313727519
            0.008660024343032957255623281072588659929690756966591997707470488273 0.6481989880092403190306358893063581073557318538532314328896388172 0.1370644849908401483264046801660117629218820084649953033397189171 0.001284116359707223354543499845627373941753176528412193437874963488
            0.09576031461248948022470098580473629059969448250264879675010493603 0.01306332471587937903831399479327654623177269416109801595909893983 0.8858517777694007605146994099338534945819223727522273581033508239 0.0002950447240331354628571647770947872393175539480231123931357373402
            0.7739001165590207003654117691686441249441197863086219596813300570 0.03113144938247869972233668170446860841651838941225943357049288843 0.07310737271143438464214536440275516370965471574101214269488464560 0.001859160419150342762324573665448117836301010783608300787644900294
            0.3254969083722496060872223556700193587379036356001170893010282699 0.006946285246366315585446816248897698675449987386786284685828479419 0.6558444335691706413925930460695806650905739142098732246811811357 0.0004520569260417250610176100261852264925278503773567748646550973961
            0.007069939046528428651323480506271522040844242351178213422629189972 0.004258810278145131747501410325381892275457050990456025625133380316 0.3819230947289452421022462470601348124740262150888832709919858691 0.0002669209025135174114016425361813400698204102464019058967829257671
            0.8631729081310122612378276956058879394715286873925543217786407372 0.006221674679947479558769743035233035970135090117290581674329542341 0.08346458248177066408643851239534810615526559228880784032487538754 0.0004608186265968194430994305787955641849147649742135033113404736627
            0.05341451155250988934596960226670298060652930137261755051592891574 0.1882762441705843995404420290373607841726512851818744474486749811 0.5712141028777057545766820654813672525514984713850701094695716978 0.004141577987159988648175728132292587843937407662279967573556502667
            0.4750030967950392015392958738212815060945135881997994855499340981 0.1508973216312037874082474254992766119629594584912227590852021192 0.08982349130805258140206200620047403356654208896532968377867609763 0.003494579148040028091000726946760725402840051476387219695707189805
            0.09678497431782780710925081122139382789257290369419056056189677954 0.7374187688275300305573570890205705520599719662492207099458150244 0.05338801936298250807477591656163566794042717438753493858719443842 0.002096606063967921967217995766351113496396311829188884089373551200
            0.1764968228565630660200396507859759215138816073629441981685053943 0.7117408118227280108894542339834157126301962042673050105643847832 0.04217314485892836403332688014284816030041015854583354792640557182 0.002508973417564114243547127197677124888066458354019020779236758528
            0.1097023054312313412834974585399888616628541178384510495691256967 0.005550132916696420086312688301619589975982614708616781891184154065 0.1575396070927157354496193122964481723362746751745557351321544045 0.0009147382007260088055420507636615885326852928116294670329378385782
            0.01310954337699735613205757518172712600198178331369520994792872898 0.7154502017521988476770971761778822057040645884342530288207241767 0.2281347778159294985461328109945948497897925942682078777549113655 0.001222478584076633772942211448370175758880048423876001233114352823
            0.2252413410326398438331855374851202436737715142645407407852793468 0.02157494738063382716018571548997946429703004638174839954895125616 0.2803823109537106270389200611758780538217166021775000163941091548 0.003044995885334689064361489827444273085160594681876250762236148733
            0.07050642987134652624210456984911095049933018898159943726492288070 0.005262881274045812869799372477315866590499052748318267881561919590 0.5496560225791136210732365416388686244664540445203243946935366429 0.0008769003845836994092005506046498572686811171482286212948731636832
            0.7149221885377979613552877159161431478780925720310151827116624422 0.01416785063086838351925122095611188714380026820452148243057927707 0.2686888889077759772480149291942123596300333992582552886630613686 0.0002861088397291472785990144509394645338983522888640452308761295510
            0.2119369958793148570725521220076177666661115654819173027495483519 0.2179948400062595484675456660032407318211393827628445548893258247 0.3398612778758097205402243168828887879154196660902358675451611930 0.007312407163240606978934127727048389277504088073671193763455355269
            0.009785019198175308752432241386935819507455620649420361182546294980 0.2493657516253271447489438984658415746394159293382826405188712370 0.3137826558464174755619654025614590780605800749818108580311111673 0.001798516378985652324228895781977270794194098828296720251446093402
            0.7239345840654063529340410931595816594460700338958310971594384813 0.2565065405173495955536064876039124019551611862861087118768407299 0.01153307062940681569392351718608097949530881821793442933591229406 0.0004979016269957098921661568115436597553607519940417272127680390702
            0.04820754075823236321181327114367590739963950015460095824576660944 0.5855404915925623464075714908092576945036680626348826107315122816 0.3242540594394636140725354036362179998691354659348390017573138711 0.002450224935920060669669955357862857620740341986924527625970343478
            0.9305060890854501062961176835222302617641609149380996107648990278 0.006699342215760626743684637191562454581817313345319913838474230970 0.01076871833167019714886939484397537690946775236193375868400557183 0.0001896429694234815816711666925619249893054683318835198905611236967
            0.2829944678367481484490514390044818375866105772774753089761128000 0.1070941830538099859842016775616585023842284194341833430363000769 0.6070915258909137128059649244108137910978978986177694456979232774 0.0008775250887651393521390784003203030139629788262600270418920270678
            0.03562596508906133264218239672385507704789528577840445065322490648 0.7732796577158539778906390841647696310190322168206411295844593305 0.007370987970125216885363326074010752384977347481070039911881878660 0.0007044722925256920703476435854826940486835909324864585888712965805
            0.5092385981435523526842857629022261208901027744831702216603967987 0.4400004694804796933854857510875555174142269588372379195239261288 0.04944921762002954968390855610881129131722526555257288356473786643 0.0005531734433362856986045876155902010082378111010260181189965173472
            0.2705353588074682082981646726579252813192896415450182058044085101 0.007269114360840202164253176303291883489528804335527279869048950748 0.6614124348250562252516554330224964056001163623102343084840430979 0.001039686622756246700200710312231636332926272957763111563140289610
            0.004155081551997229901380105405257491402088632835914006558284859715 0.1825946320440962524924410607684608382723219708028620722995818867 0.4460154825827147255628091555170417500489487803736065239679200126 0.001270105115227560194339881643617626687479462749435332373090910362
            0.03828426760998771443678219063097605324374497908729188880728918941 0.2347113636461147815243476999166647857202299242824957015939966490 0.3278379434275231017006830352715902071257569784365797849783023954 0.002569867061606507230112101205808948302738689710785360826765614273
            0.4246601797105580146529557346217728129036288302388388214306589896 0.007643062824792753165817207357203777008133655234149410660657100332 0.5253208805973940626817659474085150856825880508134028204576743386 0.0009393273732350615742526702309849067205520835510135870835858862009
            0.7416819195703650373799058845244686393891527677264383464969455890 0.01095577564420836079996472642887008998250934920670100193188087896 0.007651589911872025169417880787377368505538693913104228822254839952 0.0003961350567837847499370580295021247830840202467691379993457666513
            0.7997322599093052640056356956898635348246661583640447111061272465 0.1087640270872334240898573110565069117691183869353014150826686964 0.01046977866451555328416011406819374682408608064480846472832258076 0.0009158949809637588110794701777894871542981821034073851950356841684
            0.7115480889655864289855612987982153464616052378785335620876574266 0.2144875275789839198609880260777500265295216837408274878656477652 0.03983306949603426432861132160064044069680817403940247905362774019 0.001696634278597054163969364199059898281906144597126776811936614073
            0.7581212114916923082081537028303323666511998873298105848113350082 0.007336696805318373108333412648250476681859121367213250887863629058 0.1473918682713598264105070706273771556306630346938179388428685068 0.0009555991314651279155697715275408161484774146609919091190209859977
            0.01063870535184245660744234388777489078463902109103237813769150052 0.9667327823952234907134120623249577014298912715479724602552288985 0.01274813998778700192521855351403665192687493935573715154824645369 0.0001398215996160018903171470432288572865298513393841914457617931392
            0.03415179205998724866160600994127509722051020374128569221817312372 0.2965599498470175865638399683981291101674699014177821694928199698 0.6237912179704831947985364178021944380549399172197527508900486159 0.001812251452464922128799323103311774411559480590058254562923046229
            0.1086549219910283222932524362840326768015507234203827251695954663 0.3935016173656225380877309445132335786637302495001440649123680047 0.03367510313602265550219881151828101744344140079378399584185646018 0.002699177971518235167563678645242166922413115024262645969829949796
            0.03225489046549259531692731921821715884041324324978856912927329144 0.05096175291742458614296191818909543713877905725164698030880959402 0.7764308463512639135948785149648909320042218505939343208636989969 0.001500706112702225356131962226534742104242707446229752138842569702
            0.8596756133328084957927973775449851258202133211591211410029886924 0.03948409028271516955245082870437540242211897194492920691530439525 0.01306279671246168611705945171800141141330597405956013198133510917 0.0006384494459364576179249277710530705786701680331007968091104250732
            0.002421348152249474770004280266183779036704058031256127499764392194 0.01339485067103282068817502156033655334143617931965588452421104892 0.1196831364835487142546616891935522492354271152845212400466557364 0.0002067869358128779281788908063672205071872305856664796613969191800
            0.4656971730698691440712262967255245321545617263762786956544562488 0.1237439020043878026633995196824891532810243908038960312033624287 0.03218567058228769735190698930258359308310208855940272300900909450 0.002895765391012746821276364503150917788746982060902972250447179697
            0.1597348713447819082313984494601283669372005758770745280727060409 0.06196210501773992182391879978233666555750348372862742341449633865 0.09600008617826542544591835104675529764705815567962614766117607153 0.002950548558603064243428157820328172669571111426774872662381850573
            0.05482676144791754310563594124454301360110311554408439199600221710 0.8998024241512658686474232195712992490515402473075244469806532498 0.04381346463594868010216190153302464728329459374773416223233261253 0.0002609358124895811254441746756516257822274419021070250388548359230
            0.2035539815708976772130688812155357411433796442277786949128846081 0.5599803209539647332099200018220004867864836930856349866705830495 0.1997210169995029483188216600293589895293068400678053295397502573 0.002465883944676778397067176503545173136536445757759540272805976854
            0.4677461301900598503936498585399204155887308389254071784396531657 0.2562751307712825905417500187151702256017612985412492126076128068 0.01180965318781632643065546874696557938576815309933385532113452158 0.002121838489026089082872357538850478611105505474679344513518065572
            0.006171629001141315432108729783809783382240206933334682542994755122 0.8005149322612030387001087179607566785876100356667500686612031359 0.1848170095514990532221309149980178859842052612344064912353969758 0.0002909893980319717217496230740095557905065206546399914710827970028
            0.05360961851347231577180270785634362811729600477414317444616841890 0.4765635606013642012044318480228777495627592708705277344508759443 0.1711262948297935434245002215875739960990432952750595187284846132 0.004410216448069014387540380339525185191116280479147114324804905872
            0.3568318024328195240280341037921858682797767936441640043686137823 0.3427264807959187756015546630464566529323118061011248574229226766 0.1353464875630168051693836847817918535759103542419650448397111295 0.004677326144547813243037860298966913292448072525683178230690176460
            0.04614107692853971868865035729983608906575644995224308352638305845 0.3382304649205438435376193041656923797747198913619169194154964068 0.06047905214217174128422734886806176112600892485977483322376140937 0.002867234822482908677991142307564918123179966198348419591284088915
            0.4025638744542339036587557934983890829590764403699316893759521136 0.1598475280027804295224307009629330331393538414713880980174385564 0.1888601630563910397039781510687761256109624925119172020847367217 0.006278559063146218767362660018863099486597022979299505320598265233
            0.03698458271096192696668240195716184552284806663088398395447833232 0.01511944381385080906427013174855215857058703798929871333847318179 0.7394591033064870655805652895791479777931008034626671064920575591 0.001119400809646823073562029214201028462932180964225483098149447819
            0.2381541110814721660501952475815250562303344765299987694875253410 0.2596758895798028335848788604905678807211741441546902851448828864 0.009890109213538074892874947694233557354384443018929659392200299754 0.002079056051400792278648000163468192048999790487646160508999932884
            0.3414537857081686024931749825895313396294132312058578235126692222 0.03930395399675286389498744409275297689702157862883440690925384984 0.6094089400997047918222646465758541529011479350581026719507659074 0.0009405693465555374148246869944163545780322893417915293497042585655
            0.005273792882369253048749929032947399579583765890924963365197496652 0.07715283658562105984107547969217721482818831017045878675648774911 0.2604351867243432396979049513425812804682720588992289664863868048 0.001018615758177027982237704532896193219492940757919895405399871998
            0.1232099192182548918507064655345970598213307605271662360294362491 0.2725790326112460362193212545775972095710855688176376029211975005 0.1113498814038405537046561942292165126112719110456460260177503030 0.004678507345877932542489057308919701311467314392083652795318627353
            0.08399865774790325805620352252124002052099821834548605479310386199 0.06149320426419448675591057701796282509269927103567758433174921499 0.001618164325751759424605450230414543216643051709631088507135702263 0.0003942757566066272635116715096119028952342808458667483332258656217
            0.3989882923698022528775846554747328026292746595397579839206480871 0.4582541479569104113436312673454881563319665403048745136010984580 0.03033015688299108028374989360970443944625458715283908376706638103 0.002953505293648205971016432811234823690438754972423437019969089736
            0.1702411830376893553235895854125494394228551945214404667711521710 0.1249394483772396561579066751790249262465294440988628392004643680 0.3967640872614694757893515275677514173039914264204370825162058454 0.005079701000791905280311887092941997804724733924005481592300379589
            0.1692512765733540121331157212610088924574875755736490168409143445 0.3633797971707035636616670206948836746917578750979332467813610588 0.4376090309534123280578118542509443327263706423696185861777629531 0.002151137313142206300921036563808127664922732045262925775530675920
            0.1263910353564580352186620535493705444222088728747780586209370849 0.3754255913971685332969964645790456322806859382786711938663970986 0.1830299059508622612237733647488348445614253915649988881329016903 0.005452635682969574264716590170110838773611683959530093810309595589
            0.01482375524242676804065885673292550217036429381746448155603702598 0.008081532499432996858319609746659738999486476887512484354917297627 0.8767491495040140630749081452870954696772232097968815039379941996 0.0004058003596977502916029272683332372667788157753325047627908074061
            0.4338939310319813927705221913719619301793221452099055829942510214 0.3397956332483063208982033384893162448464295851150331738088336902 0.1882284730630920956284308371166090512469357848089527215482804014 0.004132601337580603375982203083279646132372512855708752469372039026
            0.1123246535567806863558292476606152330202235077619987401836584036 0.2792776978594956932945178510447562358611184126011994907416076457 0.05367976931918064609545810451071252262450991142842821417994900809 0.002697850205846381858292287933052951791007561168465996798829917129
            0.6454176194022803338281164084193123262668766731221546872066894726 0.008721537373540050206139786375318778714411557746868757668731827982 0.2739448409210263109295118942891700330069805461153063847641413618 0.001093188656118695091498650346786515289958386086990089178454247016
            0.2502623813429631204200660932498937381978051535285614770326506088 0.08122795571646076648879937064416389214039890169980027371311304212 0.1070693294846072199729713472284316209748902249885806401602049423 0.003871547331155341289918690109990830892851414781437419677473794633
            0.7458191546434136544115915264314848072723918426447274235820833625 0.1171879322329870836687167158935381168582898648259265157678896135 0.1204922608030182812541480423281495963643496584567861707598454769 0.001729166965518824527398147459235546174390931360264149903631693596
            0.009251983094851737337694226152967670645745552766020958579565228299 0.06423485609564247213008350378122512248115815480188906799953037976 0.6961460656122464567442881532644243584680512136187605963470703044 0.001190506807682558643368571727433788789665516131805237362390459659
            0.1218779612041469050484290630402662276542484834404932819133333699 0.009726981020251406649767420230055043370344002800314935056083883709 0.01096929408069078020672317919108720734727152156263527848424853836 0.0004188527790401606339112649738591319948087163342654338838351322106
            0.2400832850604309035819222723779330586981252692332853469228260437 0.2015478374171375662599619509319763780007658348302726482143114062 0.2037187162103391320762368883407311698093038709454625566773972482 0.007355970536254658225728675151474897606688388109084802926387662014
            0.1450224744943145466946936935356270856654252934307117345776402295 0.1727108657965138711169268002518996653290484032675508709351190500 0.6784363339825433180616164791709148924730627101882235140720416186 0.0009128829825648463502820803105364102353175847639017932091446321074
            0.06569791204754417332060415362175192196283696085970470048742205104 0.1527462335841583041611862972516181585868173235406585841953884039 0.3509968215600091271858000821764768444328344037512026697874307496 0.004511001012047151406593867939174115921688324502252170964654945370
            0.008347982853994663026207639637236431868633440021633230743461291540 0.8258251894136949978952427214861161817273478829832184732203207102 0.1177424112273780896445623270341349491601431424960482738758546986 0.0007203999894692848157576026457625440832845424891713556489196896474
            0.006313447493827306430261081481909100956754485960698532974754942342 0.06633038692614009218630771471954398231104434295668329705386798943 0.1386689415258628537669264527962979042471249110339225654234187281 0.0007578274046741944000983760940712275140720265255748549802485482245
            0.09128736212205472906894582696040412584143992371650760751447582197 0.03119319052884467497778472528850694456054682843207929397560360596 0.1096741824077420228178381649103995736901029592697875470745205161 0.001872960216815683375073588757449114664905114993516591262352210846
            0.1993503179250778952519549265429498680290840286522142536604559933 0.04705844370570278476841033513405981253502508129494064029800203153 0.4138778358565984335405288433849935542246940866331314876911518154 0.004449425691183075917409774638859064691816581637809302046671026679
            0.02006047536186076446455252508286613686074265290221973482815767559 0.03190204782498205386429379738931983095060475001697667403853381405 0.3379443304447235684005133895240040463380545948256509970520098044 0.001433319762969361803308373542070337183202393704021140064528933061
            0.9053600898294930141478992343764622234809714693124702291639819083 0.05570543020491999094830631194106631102453622886087607089343384615 0.002278772835569286846187071603735027287786354578182294578968604084 0.0002181459661535774225360743787142529605689469362601475223704050044
            0.2716087227065310188578605760780111789741686530297869296382336115 0.3428078161825790540542784550209525903328384048909174497944888465 0.1135042983522657586096973675903956813580725375273248600404847785 0.005852648396882095181540654171788116459763905892981584836472710786
            0.2981995421455806196943524409635597140529432491356874266246613886 0.1363683279252672006695291233099812841966223076171969709242083066 0.2895670077580569750108101214758670655002702778502294289438718634 0.007362230489239754916872863168867762240649185835721425921287930442
            0.1936308358562809840317972981294135807606772774303107472910626917 0.1020337230666698506502717136981783324531804506729798034275095555 0.6801677601081738003046762317584152719652601526108349968370108002 0.002120009767612292328269375001662706667201281049887060257048419682
            0.4848212529737510468583839782531574374740190020067799460054242125 0.03595128161526409628401490732229912282972389253083424605716530205 0.3429679144296913912827505697910299251960047633737946657902532822 0.003496628144000879967766220373113327535855828032187029431749162194
            0.4590378408520170973623526683813862924506656274214925765911097612 0.03720094107721546893572348296818804545913140253470934667775471623 0.4602888485942644273746311437449742242789790676368288479789149183 0.001648009787982850722593824978225911419915509800809008693553446651
            0.2656104689522507693357805476515515298355968060335137613392161400 0.5742906151256090473689194783697954832005773476871044595066024601 0.03575832569873157595281249298709378018123553043373423208745238006 0.002442301156486366750759370100594864080200655570508126571857479554
            0.4591003092892829154068927743841458095896794312419721369079389989 0.1286087714079592850730262890012795870398198444736081618775922539 0.3015957784992044773820461205743987263535925606102320020447959049 0.004084365521153371338028031725800763684487361956627580283473825024
            0.3167465917318511201425457287300912437298521857808136430841073304 0.1115473099586585589637697923715720913168794815101758094166069144 0.1748989949386558546663162191721828320810040316275583032690722226 0.005439307675636189966158333888968485249800996937149727534241727348
            0.3215501911601760848979887463607393502200607462813443930312507721 0.04693209020670127365445532962069180877441084498103247989325275655 0.4338151127004153882657722248722452681094099965761679244559534302 0.004625712673718168925548903358600632271283410322112338258029378108
            0.03313862236211531467041265839344840264654798570694382600915925475 0.7307909732650489355807055003361834274629180295533576944397031903 0.1410210989580072579198914061858692292712901680709029098141275447 0.002001889420144780779417699434517861030692265305320351660718834114
            0.5718900185462562019344665075045246792234949089248597232231713925 0.07289326746475271816404525118216521215440280623995013050537272864 0.2552275638405113222608503052839069211077003840897907991164895201 0.003760460930969792161423594108302474257886565809467083330581138250
            0.09770188651407068966699586329491074439499100933494129477311993434 0.1770142548016339886579936260834977447464319359484439876583909717 0.4606826787345927275788669740365949703391093390636246019119093989 0.005746840620022217291973326152981881186645497551180732418460224221
            0.6802842614535705494279172409776971347171759753248486853523952222 0.008643370700996695045108300490023417675838053196703632994294342700 0.1242344027184501987764986092321931232134026540963257223473085169 0.001517457454555533449201407804391184383746540877562414294225013955
            0.2153595743472590594593300890608045370985154399098556498501330263 0.3819394237604471435290045608916172345154528140581736074437806804 0.2142435860822902244677460586139500245172444743908918671875254793 0.005371745563370709513041405979146600551267191112001791655314370005
            0.2726213741941845236295446934381415255117788966910622919153133161 0.1279712807270187821955394764946995841676904987543612635073078161 0.04580100204577557176793021983055094835091204568052420823860083461 0.003587220836191970468612284499072817723823513714157548040763937715
            0.3984451853729349399214372389797377857301358109021520118628415743 0.2854157004822748938838365464282982833933087455367647005825466120 0.3091983426702745437094234165177243925399062900277348246800428012 0.001751994492321467556232397784585155943543051306707679474261363796
            0.04323598422595850792589340437769073140549071740040309346441676947 0.008889303600267966188379800019032818992669889116885462058260676359 0.4372360648354571981248501206967613834856777308216301624296585399 0.0009055044030176570358258165707363648194093541503692697262078055612
            0.03900172368806890544590749000009556352044473472953484069625350897 0.05476349798324179148983544074493262268724793324328813964512016805 0.4550419468526132970618451552508136317464483046737741304729170078 0.002849568836924699709714838164329418310310655034526599979359094951
            0.4275780460998280478585421015372701422951385702351858929823935777 0.005809531812211751430199422658327025134901581389142105168191293157 0.4083577423686247442993276824962480187211509803560434929357310295 0.001282023431245292254151193380722825027583666509218082085545811503
            0.8338281530353678955887777681816955399776422485004199737599168419 0.004889123725581052196276652572815221864713842945806068168155868102 0.04409390225232897508338146909057089820685942497031532300183024320 0.0004965018703848840181878008632581264728499139445130473197666762211
            0.3142529459181859802672342437786703533982860330867095108878135551 0.06142837736376139515423329234354819934200877363170872845141464774 0.005645878842630619562259652357898676609347637402119839630595488229 0.0009252237691650951235022893383821688544292957860143766230040693575
            0.03924791261317247060551725646648397188982337401619106139052779087 0.1140138234129837368149973818767081868954214575998071807326748551 0.2395092746883467100436020140565146452737168388328490516306626741 0.002694003885120917537429946552138262429237640687969616807488297831
            0.4539541679152803812649405527003206577380642076557658582031305557 0.2104952778546835558943998045799220019308598570248665687525376283 0.1953533736725629891871416137276851841892801732533323970089273676 0.004945171878536036089199716225444702633940339850072658866771490436
            0.3812992908125129102305544957177949621428063631631385517857966105 0.1186730313863498365249889133480411634836084801318691638769359175 0.3303447247285235372180954232213635571288561833604554049430324785 0.004836813790922910757382583766938959554648995070231216207579550750
            0.09634781463342883351190790654389184282103577964306091296719153251 0.02712134112937140031262732334357993902405963855043175042016511386 0.2329508090903605958246211713324359030768696389351355186757083654 0.002313598382019313765939090058707847565778795865470845293404362618
            0.4093841427681363556190256078473563866736296733333026131812056271 0.02279364172315966581146578522457806872152312849985535145086317034 0.2016207652679590756411633627266065846976678497123213790401628177 0.003246700904022639799872252994336562955549842152000517592411265343
            0.01218601666391607062260945835256938942425214869509004776440064561 0.3586513925109627132829861529575491531171977642886143201918134844 0.6224762236793449474225401384491198187271981640963953341516213865 0.0004542959465978239302631868843830088555178133441450148158208875242
            0.5677930768819939191992456554868856218548828074308318206696401299 0.02798010380355909671248182781014529978770139015657844571569051054 0.1260485984319074406112601322258395295342338088667408174126815567 0.002406468042227640858049449357305733190618096350268522038988263895
            0.1218171595568586696421416439662850289405309080567937297842602061 0.2280888665052674725480877502758746917115274920217722108248990186 0.008862222387529789467093322824908411317773097024807690863591318354 0.001368907255599423292536868320317589047502618756136882951673314700
            0.1247980328776612053589188873276802589796782663494462467913317955 0.2226702979027820659422742800185204255364484400347230080416954035 0.6122708007192082559051746019625727169819846682131117506592600534 0.003281748981420506248364069843525834832287820720657718902413267532
            0.6042085812697324021618923164836815797426018664623488301196124807 0.2391625111643122188442360835735968577806398008953362220277681515 0.07243671589999448643198318633785925901669373825865786118611219673 0.003420279960737267302926303878925055935484808728831680270080375120
            0.09569448959359237950749936159441128738311225203024499128022033513 0.3350098492020641177074109836798716060577651471654846889805365264 0.5244587945112209413865625949615473877165321556077553352165636110 0.002797105972416067697181527019926479986193694098046545171149240601
            0.1267487143705054473353234181989858598630913357695647943210990759 0.2325142606306795657458361176284909550866809953237382430965024798 0.2102146257636650617579820129967744120910762129122324673877384166 0.006392556463850251191822433789841383966995572895724960452897779493
            0.3424444735195554240979726085665747949999446167629656820163422240 0.03383912803333577338098208870539262530482122620187056652604221311 0.09799656473653479991246496746870889672828488314863446678159271993 0.002851182390541708301016861822110484279238549381047733995024087694
            0.06414346632589353156506260847643956514305313911684435376322477688 0.1150706711075193337000704793324738157734790507689559340435862683 0.6938491183614875463053205572111133748351664894752298784646609253 0.002841416167776898037587521798696408293251208525663057228021505854
            0.06143584703809244135627048766807638513766033819142737517871704280 0.4517540432764376672716272460654081715638797905840890502300152510 0.3770147526003654831719007742218726774808874831895973834974178905 0.004450013051192151471037251712049469146980159132739229444964185355
            0.01121170716683483286847251599364622622113896509877904365570737379 0.01223002862354909949608942167318831428200837770558366074589417585 0.5284797265787000465856135926900679224975439958946289086508375123 0.0007344273912936044824224435229606863016915416924943707550884089555
            0.2020516780565925342880109949596983563913589244874329524719009919 0.7204162230916858455789949345363539504868158692286366507651662878 0.07134442615272177584057472036605403121795059301487138569058212877 0.0009386717527008197970645745914937135439131510525391624105141225788
            0.1139414460736873085795910549281661552559239980789304223610123793 0.1237370590909660967714256192118673290173555170626457709463061637 0.2484170616762457677680127171965313260722155095514200965998422062 0.004995291121766719539524495307441063766935572940161104613407719452
            0.1577134277233417966482615843596069767293614377540020418088357057 0.2108506068360035141324994350352796037786717385027279190110548414 0.4618966307850266817911335206006728517382464482056512557155509158 0.005723091170155212642818589576682718213112380855577273500161420409
            0.2331716811697654388767951024220469025629291090792202714657066714 0.2089557275559732634221564519055385369341525275664793736995661643 0.09374785496221769498081834858604445268396858632974336486736447586 0.005808311044230109176397893459625641174092540921767422418935765942
            0.2728052102971328232912119676052681862526493229507129303035749540 0.01040532069820405814670775107877041124145177598638800133187350846 0.3698992129765975071371070602904897759113417877503052269364079601 0.001837639254849340204308541932294123231073336480000056221397152722
            0.006617366186959541302303726783227258178281997111818449296054100411 0.9138493363341031174965615999429564527995081273159905428354102927 0.07157035819017593342132683241738611168500524683587471430359861461 0.0002064205466615646119267674332116070087320014317999891406646263655
            0.02979013824047166121603283123213852816725898101076320505340414595 0.4447114957983630863629671916385955456862961031514151790861527432 0.1047829231275807630455454709655855316624177660936606965526148220 0.002843084238134369323942518769872180317100735958350109954334361233
            0.6297833814817403360453167721354411988496537554671089156381978385 0.1649653692519393705424518756100257906605755684686033429182230637 0.2016584109883031733145601866888614636050540893425717154573539516 0.0009753766770590806615675439018560950161181102089940940625059740061
            0.1999314275790575048485658759045827330784034078639366439389354446 0.4575916380467192968047786600073908659057269373524879182476601689 0.05794434131837713041671263793032182557049673209168350922556522967 0.004910298383883447733187148171404644822441047947216148275173645611
            0.1789126364370497230114436006910325622532891312264375637655695416 0.2176964371459946329776669667488090412672553962397788831417866860 0.03680542315488973069640497010716836984351991828342590288286972815 0.002918453096107860286974288108025657107794527870747125199174068520
            0.7226749283162569613410822179115723248726803732326173596005431696 0.1074891289982141782361703258181655589600160051056966774640504151 0.001623926912232709334493888332838048672961584860396741818546815791 0.0005836474571542556931857915806089310949208177957922112358470934173
            0.004417837093680856180975044296499017369316445336491162278345067564 0.3449729271306788036049546073819403374327792955366348378554208726 0.3500785284866001518248897019418912094554107351641995040680976535 0.001418391533920987725958943528524521469105586370730663649118917739
            0.2013393819456967313675368519185731790078909570327387210640517353 0.3889485645986112711208343494823064996827435019835343505836844521 0.006615246922174220500372974549562949921765368546130444424485009858 0.001249521945450198540598917041128982083893129913021777683589281464
            0.05246489959344481297741545131741101867694878492639622722761300983 0.4948759256868595956278015304002095914943294013894865029371551357 0.03864686630194216772692412383038547938209145602823417920739976112 0.002367672014251483862292963476761519062258765184387738992853411542
            0.08410178318342261208959897342224175704520645547509905454308732803 0.02597632021280211825613773949498728688832003844146774718248773390 0.6239111990328436333714464907542444960519899861307742127670065229 0.001712400393535437590168262402885591648844487293473394357454671070
            0.01116973014792802887725982089341876684631997519101344053012414210 0.4066030983916656535586398028575748590601710383536816714544754741 0.4141149144760336093341105928609466493439781059409234235175657728 0.002262967500499489163508134813571481562924007406018936144882220035
            0.04962589946543302389907075147832325538342589093837685682192521358 0.2999181946054127673796719810154969763309028880277776453928033304 0.01285311770939948027846236613727920526104463760756605929927287348 0.001301534672885108524011797785540853197710082489548956273867849671
            0.2012185597596565809481734746443280153091655512181299811113598749 0.3454874319080430466445344733307898956224665043138798895299617794 0.3077486353784415141511113090340554902411744940627716076453262170 0.005338872926814101389821436222244556375939401266332175327899980815
            0.04833618434059810375093795328011560975867023974548276496791978922 0.4463696663551180310747955900930773918197199327837891635913993881 0.4630417171508816142168117255558117916077921134019965991917463398 0.002196971742616977245269957680210091723084334080332545691919333671
            0.005744041814107994813480270428663841401108866827501529277467846409 0.5545812788217059954462751468512348221816175118269150692588755927 0.1251386056909018492312108820329359673597100262697880061063252387 0.001124985181844444801628888119273846949640588900981813052961861255
            0.3184239430626558618688512451034893179785234284742413528131996961 0.3620115771674341439378274303727683204120059644727004776014457675 0.2118224699654042446711362382494084590213962635649524982995783265 0.006700604753353523436162075624012055547286613521178448234443174503
            0.09826933438154410102451141156253944573702263469294074378854615495 0.2582900448159864491401279393023476414495965961452120342224967670 0.3282675258913374988033151421943044805052101139543402442506270310 0.006087417480404480667084706073000762706960724304310451144271275521
            0.1981860028067536295143841760635089773142096912849724012307253575 0.1912682335719708831812604886235026552703253604747442971549060231 0.5174421530683052169614090177266811298327631118386392955550668279 0.003817092387699928683595291397092367974041031087721034073855744313
            0.3191224053454660792279003471076802888960192012195395241441516684 0.004894329279020903775897885151896392565792595813303833112140672205 0.06377392314630283141468656660191322561105796756969287869244860782 0.0009646994857794641755373396515843137065437889904867067830854834509
            0.6596440233607632158756496627986385931731130202172866558271569864 0.03173793600401234646927558258133201135583440560023442002336222147 0.1948421976898453245570506985154101137211858650074555186869137298 0.002311593044717439243627817710282244581787968534363838562568959937
            0.2153066992569017353594095042180999537742209253316167947816172268 0.4164301028061427882750098932343764128510992259937326089992351523 0.3073892178830423268761746085349657107245734447184508340868346120 0.005106647074472658963073376048493443930800239710699335373558903578
            0.1088081968082913578609934467668523419464173024544235958814158817 0.4265990373617757700181787876360204558638493126178727555846929128 0.09162980805577960720659130910313413976133542564189010061363607511 0.005526149518886664276357519700370677877701334856790104406181523908
            0.04166941410207935460638474161858156579128904276183163253624767234 0.4592219140814187862688159658313348790024290237418143648273764337 0.3040399826520747383830738307913213328362967764425885363689722732 0.003392365521179919283255360212861384710101411139449968921502854451
            0.5136552121180788791575598829699296635709020940434299111584621759 0.2400468611147611990117534175807384099437132752571863035502161759 0.2292242578227637287302346807203298527228381635922520143529527932 0.002249580592214828703397705621151942063736388986856151600271740371
            0.2126178150789534168837220117034575123025531882368215462789614815 0.09598766859124911455417295275107246818406358799585523846779216363 0.2740660246650779693808874195540153590385187089443321770689645462 0.006090909248038761914315372209073119295829760764566367900230283707
            0.05725313282469317233099119926369854104727455221135644242028580527 0.3958830581030188921787723846651975817574162456957962794164699852 0.5379385095068713576048269965772392817529344638929176099957859622 0.001255833263438862824865132519473036721983010405163811075518500827
            0.2829705562803279075109298923323390343383466394683878112749872346 0.7017138589590552638580493655585772983099707715057399758675771398 0.008723928203785356470378248258742423487089167736347777383487609355 0.0003665019646548781069302873382027695017693132570470644446510958008
            0.2498186838962400167184441406494796861879400214752420250765126865 0.3501360664494111907840842398811007460768625761697039988963797427 0.3918003944197590202705847003117783335034000937242247246952363513 0.002084441066379973362763869598360261057132983570948787661717885647
            0.001376350816542040761905393745942045650430736582253759757725278842 0.01693371269665175034703800202304849402903371518827022160888058866 0.6642944163342224752668437505823796130650979453686821986096431982 0.0002906579867450674296995004850495400027187062732448872855657188558
            0.009313214299848354523544982288186167629167277745367577882725147588 0.1508874164314872785882230130465850751496011442242585830538689268 0.6030039501588478940105451449960964031552097705644778200939940359 0.001806659102175077246911254912482362179064623027548013667306068374
            0.3957364033740548676590382606329393032577327645227865825155953622 0.3502365939950622937961137918877517190065691163907199275993636290 0.05253608188833040713719079391286686795999302570913430034880952597 0.004378675357276637618142343535033685780386587849123562557836017651
            0.3469636558826824005990273787078298119510116515276193601024414512 0.01449785414322121151949232152323690425624972367425939469063609991 0.5273964518291913598216036398894754369895690126840258894452799713 0.002248674981636339715071394988906947557852631458517882112009996080
            0.5207983240340919101187653550462939835747633406848655770764223382 0.008369723707139067709222705158037483829967422979566183862882849603 0.3925727829593155267652104428727539082902894567741386726068818360 0.001206565235477291994529207304402210036075285649918128683287079529
            0.6769373704982428924127180279787719832243667661914035227606133808 0.1978736160431062819111965422124570540910828790047066989565054035 0.01790019050943462813215549278458445906745109720991590690586680656 0.002271407070436375404264941389784089294084576172552645572716641355
            0.02583952066459174768566268717393672815284412242341649937172505176 0.00004561629825734220692488325422974115568048964448429114826745518346 0.6616695792446511382585990902743283894505134382915317371328163366 0.0003029294866670228343621104525929559771644963473849130459627556615
            0.1232177180111713962972946304371134685942252173307759392363246326 0.5788617905907081270718591564919993830036588049893342108915980898 0.2396481736585022766595518334633865895959407699412745118540309895 0.003761972816525272266217337717142332176862824754051020710149625356
            0.1124873524192964268736826143041331006355873886339078781579026155 0.004429371964716670764951956712534632172502661237721628239984307798 0.6594707526371292176546915275718389805190272423859818235311787463 0.0008698882742380015251901844584524498991782678913781120241666194822
            0.06031656879466539395662881629618038742573323770205954276588553543 0.06409688887829174674982324530641478288365859533065542134079151181 0.3130012405867649333655575202675727397525971296520406430279123472 0.003001976854467972496253109077469068647257984558457121068289690280
            0.1484873471768446540556005457094361791326737294262375492433917792 0.6612747595401172381654096417894831840563506976119387074145404113 0.1817084550763693105800347946761510852684679358848524410440033916 0.001175404607684483867199887852387852943822775365402269362655439711
            0.5857179070882263937290090675063510663014642475032651141316370083 0.1593024314067427710690669630508568115485642435156524938487900828 0.1661915137832260419561668769597886572706826579526603290712313889 0.003893701382682149315899101224778412629738732835393397779806272208
            0.6298557176528284280475158366031889375674778452316862910239117763 0.1125218187719472167378825705547821099605703073368559156398615252 0.2224756181003075918115547923534889758047627313371794192534850105 0.003082607618786085946783983878304197242041601238256988684615021511
            0.03498066642336913528439808982213030635661493917985059949174331137 0.3644593548369223972453876017848002984409358690425694117597366665 0.3168509975813390574574034060162572832657531761478222912023153388 0.003806737375520551218941323903557266091792725456178462831450752968
            0.07776091924497038708893168075438547755753274950338037069965489625 0.5121805077580355579209878067894402690143461139660537357442592854 0.003313898391026028675958351484236871272614235372837950220045680766 0.0007695831123860393595687641906875073035826454750854724618062365622
            0.05153780056478526245534570733941324821757074764727376774939482403 0.1007454432644270394422328913568273043470722471869934367096305363 0.09010140309266689946225159964686084043735383372770978385426610685 0.002389048666278280456305634205472060671248632880069327652302546268
            0.6655841540549818229366783749124046796576532161700475092153431602 0.2240083619965189675365439323069927790536085452540956890383575966 0.1015009731311900164277251673708940180797523967782214917214635004 0.001197325092050622477138158295640087287907741884136901219982735330
            0.6871747611980704034991901586751297976304235713481688032652427413 0.03822828217175332066956208919913881221959105520582227286261928795 0.2417445450100977332808739937969808974178482755816202675777716317 0.001707016619740958552646095768083620883597396264266466397143903954
            0.6595784268000368868426737282159081473873195954420027620970389793 0.04802607973448832764307584731515250931053892116283215353289198548 0.05175361208754705398521723728370166422904677905347738513383913745 0.002485579315227737272106372891263440603100537111690886562822715714
            0.4922342588556252540938600994497457726956869941663256765578514203 0.1881705104933070406629374330997881585477733374533133986392018156 0.2617409950377536058642529135180228335794623514446257213979446917 0.003795623598621170760207485022569415185234645370460826119822978566
            0.1444181831042212774832313398646456760406020219289845658850640819 0.06372894803089824894636496941131494154958519778601513202596174479 0.7094936506725721102853777444232865539983032757216260178835572686 0.002736623521361724406696781480953299333299427546583061498796712185
            0.4805761385575785777675897383975551948749662741386815144985118511 0.3475180058811052170229826459660628311013290559441106898233820349 0.06536366953850585773636673897335830272674981603759700977464223040 0.003867902026882856292789921776091029628652155713661132557206800900
            0.1107499200960137534438213753846022054194168794135623883980871345 0.8162946197454982585722745805565129084352378774379000414707649296 0.05484849244949586518389652933013579802077537691483128642679391135 0.001125157427674053212650375739701901090725906020905684748524936908
            0.09731667640897163167252665715495043182599753784720164701487900257 0.09550833484722897516687688647813755203464899763075572519553245971 0.4050175849474518732014880294153158844660039284884466119962329327 0.003309006175775688650575537817496997724139449034253321950992547392
            0.8438620925513104578726879081544201575227354924653363012425110420 0.01233109653305761362717108502664740824115295788703631413962727725 0.1342697840838138218849906079609847948709986806127938349999017110 0.0004546496270371121698898320498385746675797058278077303706629748161
            0.5309291938609716723326633968894705299756311179489493909226813277 0.3305348289906579538395446488771026569316849605590309597569212590 0.01056942993684865892284203154186739170729898343587067782154533813 0.001995251147621976354068094880050605309382547288505374502032508158
            0.5768865293374503486511452927620677647775494294582440706009973706 0.006865062213149566281149978649670362284514750514086840367469567199 0.2530147738753639151233547366115033071861528974449042073358770120 0.001571189888203280240378348154541386403347855295138170034116774573
            0.004721582541848811377033854921640881481598529782193974433165332406 0.8390286438050167627619494184571492513133811914455208283166670802 0.01360274891266285819796440999597169370301600822340096710535859036 0.0003299349850057559169432094450535652822144285647846929728886712200
            0.09990975823182479359955166874739993413938881130101039967885522001 0.02614599666902756979058118825211149868661343549349480216975147294 0.4802541842011604945319809280258819190608181003298005102088835394 0.002268872641920383904849110591119469612420016566581177997670932613
            0.02611705010450717711573929501399396429270225767517406204812329683 0.1717046875632755855648416323302830297683402157379273302898077536 0.7636856572883670546534382926431700265210790161558055040508252722 0.0008127782476143453340544435331160844975617242711212536524961184585
            0.1652400920030116058087445932953230117616077643995106624987856943 0.004538811247374012325427299828126686651261245415681567404525524139 0.2537354983594600295266595472136831155527048438023156785396919561 0.001169263579507466400593889605995229219892564237580135016005077987
            0.1817897658393963825626907054725803851496792553629406697956183202 0.2950038291345949166036482253376562546911274275980653502218550531 0.2497060013797523382113292070944852999847032931884623207687088853 0.005995937512478950479460254811712405305547957550455540331429203833
            0.3863743889798157786270287438283207020127992653004846986756078421 0.06712996148981621063760205342446616390121281248471314325765171944 0.4475547014316801161619536705963456175611913802702919941989201388 0.004418128057450593906626830058163288850692883495563822181822270496
            0.003832581510357447889904504351868570935404838669904050597105811527 0.5095335243170066534347477761055421199701953494024215428752082391 0.2709628058789787413378391522294991253030380617467930887966974756 0.001217428683590010197015531059790459873014928438687292551805479213
            0.1150753461722560669263770090430252246625598613146267441523167313 0.6438737494333785905056154101136915640549361037793003723614965192 0.1352625389220638124157803127013854128211760383259488985840302136 0.004013188230286464043322340388708263766408300996826300038985661163
            0.4283463119616319459685350758954037782691789169083601357527652698 0.07917341697175936590325475591233884657894094607854565186866381819 0.1806080922896931369347784231394705753272705250159523497661783709 0.005075338000346275028997801990507523961972334713828498645968124924
            0.1985983219450729484055178546317639144063154558195880421923621726 0.1391270453167234487915173222384649192168853284737958229621084241 0.1566615245521317519087384911086612665567929327354552118179989358 0.005655884190984700156212906764226695604903854009172292575714632791
            0.6002470457564128961063185434514485588941535973999237902806056828 0.2220387861216089309632419835985426176459203519751084668930460410 0.1403905589381416485682093026572055563932406937560831857085105131 0.003111167569598760042001870536492030513261638436784848723907647591
            0.1128565942060627752918360633753632700048769114264965629758009383 0.3167194981413038479721280468398183732591715053822334642407814067 0.3739722795724686371658360855227527441208094910194273731407838567 0.005604374506747772260273762921568245316822177520335154627968208809
            0.2103192316621360845170467958643173924393376354058377572874266696 0.01552474837489454042355043358928244587173378947339643977442586788 0.1363026540456084594952646509932359247113362171712892114174496393 0.002319030414207347186056855724718444560946162030706358770693486502
            0.01371753317807480546732352180438391137127573568477588558895585624 0.04009413129230368637367569971405986251345382822940898303441045591 0.8921364817535163970212525406189972888849153962745684790758883670 0.0005655176369721244110306429933751348223425751504164914660229765847
            0.4095532546763224239495240194570505357396460010778188775124587718 0.1401088071084835452033666233466107948134844264155285279766625226 0.4445997721175674844373412125862348064370169910637528516349773774 0.001007045366706616977518376430340595430879719146252146840279617055
            0.1020425878943817645604468292729111590585403924312619460724470950 0.4101534726150710927845183801193487656376903702158261023052603782 0.2625385335134063951394966063212593365458305361416962685247812080 0.005419321003960794510414072658117079189266386480806738977921267305
            0.2794226588242804439447118270741752717743587154002599363702860403 0.4762606657643668855144897555801162590410534377610761121195073135 0.03512685053912461828536262388893564820930920435176896285007455927 0.003401887474505065483252140048616493055397536662322033954163343203
            0.08808581324791314356649980218446604988519538662218821679406023845 0.08945054601050688459342205276970454468071753403102905780623970358 0.1621167300032201032139452498157702036916718247690112225500855899 0.003440645109055953721311243922299920667121532599817050414440888997
            0.4456414188876795250801376344927956585355827615983304878378583521 0.2428390176402161441777463042127705694649163808993407295716940134 0.1071325434349901192770293845632063227200747660836127088536741069 0.005929134910155106883385398170978532739644813695338423385080934160
            0.4625661934754945979196439069571588653521940177855755845516439049 0.04723006149328280698527151509566817665230236807991221640781150485 0.09822417640691973017148517636205385105571050973151713629427807085 0.003564198468487837183081357691980835136365605513687172433570843469
            0.3288332893573012807408911236704022800595660269259314826800078887 0.04834966725920090554589057476939613594505680075225309923893372128 0.5756176864309201345252491467508541315997187194570211420542195153 0.001957456173386182614096915945496603269433308762385101900869113135
            0.1335656999722368949997606389442782750261362367331116385361616675 0.05028808698754465294483168010517607924580256405521870312253535398 0.3410738458008965951691709164650037583911471382788385720713656964 0.003904078315128854536843738456450741012313694723385515449441239137
            0.04412106496473397481612342225856587229441129381781272428179074495 0.3310759658944696669742013301336422281915450859924245373634974614 0.5194320689258736055924608029386631846350128168934581143481520811 0.003271426029383576733719715348799782269239556916505354100408686959
            0.1129377243857096367361936269635225189576824670508393101252300262 0.002181789853243472466923800676350337273475738147240016056277446151 0.8416370257309303515013808560492183603985524550329665460417054296 0.0003082195033519373671755975794950180826758455341010867055795277789
            0.05835313179746080162351001949167980129612317130069761401483253036 0.3332132566264622282982459682559316387375190680149199895380465131 0.1407533810092449801734417456112975766934118900912974485802538518 0.004044052462395890579309916306890307789390910235778624713535678275
            0.03554637686097484987777021538185911041586500309490964343435322783 0.8447821263226408020110342619660420743587398777921223735210933995 0.08759980257233939393670197845282407746690332037509979137956841765 0.0006163219864831145008241596626424990273727258359881042809885687092
            0.3585921279137420430144583264639057113152482468619127386759665881 0.2178988767195132823837610146502196213972075154802751103220532458 0.3883764952684987173878476325456829107826109504968839420412111164 0.003619883357177967219139316748311847401045476086372090941884384754
            0.2476519042036054512786597966627791629486571056984045770728949366 0.08394304139306759234551603227355578511808864777916059844876167007 0.5558926384336318238142409602361112202484280204855714588588878138 0.004500833120534223199455517321045578075988811716806078525522721196
            0.06742381343637162105749565150724216746577166301915487315875453492 0.001483019495330369395383713326128399575173924021155902237084549208 0.7973865185812035080259736232201447725597173440596921905613571519 0.0003875282576439998988898575339019484271864142038665651411321183891
            0.01471335451915186704108575201918292939893655832975098010804526881 0.1962819790320516348033173968943211841480625700617911009289455080 0.05508756760719335952328908980604626727083975683424896613001070895 0.001332291820895265995327850111701980443038961003702691309401924398
            0.1596673906124820793369735633681310151358908015864947399883100860 0.6015916380389369210523580958526413763414005120413065574219109419 0.06914755309035657110153815623010559945608263693582133820443030133 0.004335393669013530687582439532135430332744679389732141913529348108
            0.1083296738801621876563584778978173152111096730075467248715887673 0.04465899922717988313183515478411485486718553065454002000478150331 0.8182490408798226804599155125843685755588090819866363748662361123 0.001311875730076925760905974940743853684148614164982014265143981167
            0.4509699208749987626821100490900041291498667391697953302766623888 0.4723855635314332621283060552880477795259763407958969208889544715 0.04134653097742818478669069250395382361555941097092794127893828654 0.002299959410320018169891416538449104696563297754059940867003705587
            0.4928679346852856287282398923349726990771863971488025131202770346 0.1653876581891653314053860708792592562607712513224750298040765720 0.3310062108874100231018274337452965419511224494761105928633027186 0.001792230066583980749368365473329792984129270850019171844855539049
            0.04430252521708734880409392979608439392306308123628416805793671230 0.4191431842672385186852963782473666059107809771607535620941043043 0.008771364872225592302274744494442475859876276334052588664684298037 0.0009364863246828208915719003295470047975194387589683125436311417329
            0.1596456622928095915025466358387361820627742176610388222542604140 0.6828327333048914727231929962960900437591283404924794979060444827 0.1167828427126090521230451744326640149343260153639669904001071652 0.002808309027881169810577336651194888489331491534490328925434486756
            0.07034656217914566708105621593361420546032865497040401692105489257 0.02234086274380269497477608374557519965895462742920267671235383086 0.8214758994091701549456450468350794554980788895589819164530761528 0.001297056223680144317923598681601490397252231360177802091997585549
            0.4482911469457969586173822403601920442662669833842563667566736993 0.2082562294800926185915976667902478996628268129853062077086415683 0.04228523758589527598850035804047381140066716357273007863435983303 0.003362199073226946104443586623713681979126742865799036802397800826
            0.6477445382348014488013290611645823019887224109374238787643763560 0.06520052933674646277142655197675543227526420918270286142175014489 0.1184629753402387463191341366380297265942723186478598693313705960 0.003468742682912562357062870182581114827258859382928425807614790420
            0.006193343297247673558942941454124214352020037054607223805156385524 0.07952303129728287724783460374089422409294573756366760709563904483 0.5443430766872069233073134133809725441464039179097846701570897431 0.001186669683131997933746243615889357761280772507038573867981525526
            0.04116342243464060620175570281293346807792831756267466864751831341 0.8639429869296815638326332004235426734272982575600916935595420608 0.009629994789811983061194301042948443020680126733295958440029024655 0.0005547344244077158237167708156454602412164043231402951776180263108
            0.7231307762020021371536261881784767175625543985900995360037030312 0.008385493595601985684136214897187211209758727280571070517193507993 0.04520754618650904311614368157023502544671074875625478265306963615 0.0008598781985995450176744384834480366718351739620921405584272890629
            0.7672134870959147917660108039488621384741296014141481242476856855 0.04724821849161677560790522053227048625748345171775141417818062848 0.01584090433282462690324532432150867056780979068908728599917724379 0.001186373364452434444821793394062697907373633370674353422099272587
            0.005478809002120475461333427655533747197267159767899222785062492986 0.04170557338780183834140290940454751598081639494543582405030897763 0.04040908977091766908356949659574742455179059067441384251801602862 0.0002884525188223179310772766947886837872953726795984879419833063963
            0.4671125323714270701897442465404830843326265705562298600520029400 0.01215634379021324804353324295347960887440148133129769139487218789 0.03889689938331832119721402357900096809298397442878733809302397456 0.001407542492198540435513700608604519765981485697442510241997120636
            0.3286544117724445757468954076904131826059112399940748067267980432 0.3090941608959385163892711797259195421685243026423109244775878908 0.05360948729038173316982244099605530644823851757335789319761076238 0.004323818825121486481452699745404438127557889241737522769339783573
            0.9026501223677552580613078956354896540165084880412880793846574736 0.02071177749821840600553946774147477994236760268783885543589787496 0.03851846890002942801287930100968095550635543357630354801330588166 0.0004817434932166719723525496953541981789479251852474911155998349112
            0.07491620994253573204496362842948584190530762923205482136079260707 0.006280830980805791104136832627911403001997519662939670015755771888 0.05874585756430478260425234382131758654938075453693972087578717575 0.0005914837940940468450304837972440778928022190352142332593709612513
            0.2167942969509398065439351147483700029379793070272237878874469864 0.6649923453791831353637368829337238154095046904565775674171285182 0.007878962600534706292485580320865143325530710318204643470981790407 0.001249799506378087097339604180355943715231639756269967046260575442
            0.7221260525215253709886386933987585174653921210728116075749966701 0.1399512041762062294549070256876040749884752621484945911389288406 0.07822679885088993269576232420005742603801658236471055242356959104 0.002469774028289573708259206795018689867194902319639358204991624494
            0.5351797240596208596957011732316597336967208704687771553988928985 0.03925751145747827956796129455863954620931882626000158835562030333 0.2083417871562153426872583387552325722341576494285257106033976748 0.003621271784508280919862406964444457003639612742754969849715221331
            0.1080970198552070924428562252211870752921825572396797825539525003 0.6029324878956351623560294516734434500010156042653263554786049201 0.2787013512780132534339322619231104304331549683537776232491997354 0.001527723796927379761989084990546784527299665264551463508058361244
            0.001726411452423023051001564002570027968039467615199673046958927889 0.2270226439595841691772743248222216613159421048089727685576681676 0.7138307000557900740900560281160392088083739504158451330848782305 0.0005419571704031694262121901923252187430132849160967939442134008151
            0.006991218397476372885187400966894454586488624373670253426814416578 0.3867111171936998528351798288969310640394532523755631488146666910 0.008390873447007354705654455284669830768023946183155464759766604707 0.0003788634531039245159858914317878191714129460366750656857655222179
            0.009756481519933474266655355744518832920764417543289590164536937751 0.5004111199008980006216635630013795388411249542560834648742407613 0.4778268595424093448534357576773612870110621280645174733069994291 0.0006827562856105738681083324737813686854352690082474213903666658151
            0.1689219676251743826252631311462837230862849525939547471174142609 0.05543546767880016591915560239585063495288335102468346297260545286 0.2003835170879287631996332782922596802367513937843747312812119213 0.004212483768830363750226628963756149978285826225548276522703721615
            0.1466305636949959828921879561742444680872742225531632818099665700 0.1430490359448050087641823685288249463274372955856113290326866633 0.07067986418772989112876948279206124639248733307633974993756372136 0.003822622606705256732424279100509054751269204943809313146828516491
            0.4007613726773584881817762587474682485020364652220849805862455179 0.3833776010008132948047966662395179046795916619165705856734520587 0.008092022281750748290980444193184084424664193181746398051470591047 0.001869610312204847156157638256316603084351155607621452836916934966
            0.3405350667537469895476340918100516446053729174925847924136214015 0.1595068831498934447459767594726862982414021455532952349847506965 0.4173139057429387835662123128912501953242643757176364227573917373 0.004851463560630363664699584162003951898150335122334897463860061256
            0.3028266823488787900214215290235114370549882732284202329690911054 0.2698207272312682064406601115302343074715015370379570519836084925 0.1990692003329262408163006726281440359906169013800073709944415357 0.007069824656334226799923871331793183009916331869524205458035748864
            0.005580809785040648826278672325698572497292728950712184388646449348 0.3416063199193121593184426359660228712970628366692218930912847014 0.5412971191435469492620338836438595184512237416461807066936091685 0.0008923243823646767659546831930279749237278965831493032593767585650
            0.3378519912202578217684861069998883805208167655286077392877528084 0.1155153365139387577913021541898504239815085286091197924514176225 0.5146001184860771084759372256719778330300987834292555854210085600 0.003074083767704294382318532479296462694331849999672849326474966797
            0.03563980957967465370834650676544160686704254682178811899730694267 0.8278953416214580286386585391355621112611787117832296560251928068 0.1274737565916173473900679489181361793822767358461993781549518534 0.0005906049738868782783261725843824456790317857762023020296204792863
            0.1431874363578190445592503690622645821714891062923381312191515306 0.1072990034912498762748957508603945473890527457435790463832865869 0.5625771028587881624198189976380568079897265315748483513126439170 0.005663127792200320908639239804293067173194386453292505219931594088
            0.1341726160802248201675816478298446787616102910601149016546324363 0.1330041277671017781970458924371594687136425932872195650545309126 0.6379684942252781505855133883094064275209626016034743054125408479 0.003224472424827798128525334408492421947214916177867393261897667604
            0.3701562845503331300060179453722001110554654230071704029969906291 0.1145346163073186889610338647315506042360567927323066608919901073 0.08637072880332265306670823751916055732499229537144815777146212580 0.005249165970709499232944672297572553228241743880490615906567455918
            0.6151268767765500024949406104539586807255777800146202275182091569 0.01144727474871260894112282630149658714425988471827237713184682357 0.01371212719256321401323059961537412132813480232747310166758605557 0.0007762577919916859607142707392020933663539823301333695620676536997
            0.06085268640352271711554427160718013039195858540200378727771596685 0.9213489960868316687759922817910335447380833933014900424998041182 0.004911339580585479705205704279412076509788415545244428198792135681 0.0002196904462845814907510748647516847547760291147206735463010752991
            0.1691842433613635570734685871243438176015259024871857451523599296 0.01587629480704478499865514660775980682291026924360415430215984335 0.7624637241121806533383461668098046496230400633894334456744947217 0.0009012068382840767847503716453290231096403929602270705911835356607
            0.1284452996446493738225002252910731090655094408318164759631804399 0.4603013954095078057724701834679015604975994711531157614589128905 0.2926840058264932305142015743994113977012827858097964140661328386 0.004794411321492987262370368695396684111544613411215967690304574341
            0.6110060427336701882457997621885650561211694491765189783913524325 0.1308334295306646394728039595062468370234789810188907087672090072 0.02699203842054827708873945005125572708835694046786230810335456372 0.003369975597521328772291724157299420209597813729810273885972730285
            0.2364201944276332380867634299059848761943900176819627855069145107 0.5205786008561412766455242100896416735879652571445742875097239855 0.004920592396507445134765779390309428200167906326608174780184155036 0.001324351171401327217105470345179167264505839759931506926674296646
            0.5857951505050797680976461123071861181825403596381833518615973616 0.3343383074611678889835358799736840051457130263009321603268161291 0.02735976646444833762177967268868947925743368438181029713733471446 0.002155690260602618580280203144751468548934457695756839473681907793
            0.1065782228461135796238059798816725658186748174913742391323875698 0.7622753437710628817000795035564208817691244146054731933226912355 0.009156528701771215943669612464717832310972841069151734328304802746 0.001171568956296674107967375987926271075430597213018104598967239763
            0.7845982724834032011636055353806119212187915452228157578335870738 0.1704778877827232555435794632702767524719359333931342639723007750 0.006733042543853002115503757211615520123503706299179441438437258111 0.0005062254840452189797361831818672250148978524375156612647462085317
            0.3689208305315530745158574062690649557734522313286331345526107712 0.1364024507893260323833438533322850097123632835472193045345010344 0.009632929586525418380481233388129120585705629542397922066667471679 0.001943493388212479754437098151895121935167107576001829599361844854
            0.1538276210710102601925726967059681145517985725667045899984708831 0.1722509705751445545859226214522057993093760674956865333351924843 0.3066364333635019118897476259479995682966582637423596166982391914 0.004996983047959258430424523658337198335881250450268573131666248775
            0.3418181111102015047378723995029327716467169472299026611491858413 0.5996034536010032139286451301129670564018019403918049948682560403 0.05018685496826429496071153402682038867520158660983791300446330089 0.001150075893131915931918522092008920336436161207790677421302617831
            0.4744940276877384431147235665140632859816861257666576545917134517 0.2864297168261222537575702958327547854167277565305196986182462984 0.1434441808909390986952928075389373868673871407793548250500011006 0.004969400680796730655983146827415170982458280086072864610186964884
            0.02929504524133799858734250455413713741119329473824045374834820339 0.03421761720885159492944858923991909474798576347628224159925675984 0.08949244469255716893954313004670169691581727211399461093673330735 0.0009765069642999556721230053366498192249161880158327575678695509013
            0.3298804158708545668948255462813218376883241419273660898134561415 0.003231704083125481641406750946527398254682720284579001394365946032 0.2643728058803550623430735743628724935618042837192065168760066432 0.001033932388978825919747671138807970941737373141157413509394132783
            0.3166479783653188956090562327916080557963013125662275734964857195 0.5743121869922133964789216987174121565460093052357185510874884214 0.04827573252845457805730923693821442437095787110740870614953194762 0.002160761952979495323751163812721238368780155487032300205673091866
            0.7830451092502252663232119886502875843424438922908273424179022882 0.04018487767975109594523786606989070572590431035632512503911794986 0.1333305580560966397928688622010532726753489424558668103472731903 0.001666253341087117014745101378941838675977024146738067618854154369
            0.1824812930356860846953586008157036425273565761622660192717913145 0.4610183776872661879851281974029016473780070046870303165549243945 0.1354698818823978951534116383026110687828610675387474865713652602 0.005972898002766029949081509304734507986340571066769357991019831027
            0.2396882062414563181645168427968868051603825130561139081106945831 0.001712958789244692717508908965188127514926503471016099883009771085 0.5837849523921838268316896293530292149481907192527132537819735134 0.0007787999692140297548564125124253192609034873634683980721925100402
            0.2671521899547668651663325346479517586478525127317362223241069874 0.1456521061864610452378103563846285533950942642560205738410323941 0.4318662979534095874538334182650800412660120937249073204896069744 0.005382644833895082196902175265197505592191501128837842217063877007
            0.04862940916155615841090252468471071166727127074064447353866475780 0.2095816406588164154047193524945795157320674068033546659498675277 0.1142654947977194920272691743546020665372576087677185113468662754 0.003295764865120781858124658840135352021847782712141642876877451679
            0.03391958453476714779911842615066556455143207454485462211882412834 0.2348792391163563307946218771333726047294187237639180470898021090 0.4306441950650781130492400869212450029575940560920079909984094769 0.003392915168558253659713410428159739934830674151238709940747687479
            0.005948943940372541756705573676354933236310826042951514023955529955 0.01366191411968686317672220678443606386085813975138060173226897699 0.2353540196847983388567766089204054023406212165691838917729800206 0.0004646974534612041244083440179189673449977389083757396001571613610
            0.2083453533600838269001717935351109625796536107487700430058501881 0.09015763740948055391140055796181413990347235161159707787854332310 0.02487316155374961759228743173807938355173847120947322446428224757 0.001714767805411967837802816490714173051538385447859109553732593938
            0.1338453714970628301164186480348818104303580256776985436997262481 0.3330775798857274111661176001906054160758926548545073147944077316 0.4341817187625914408270808197640784700045971513337910287189302985 0.005486982900161855251723400113194489253495078949859148322625314320
            0.5836583705661269311207863662391455844799270565180899432239124489 0.03519327054001234020477850436618757894954560707872439833682496482 0.3713866896223358782063795717966800630660896555881182033766218040 0.0007454578414908177921636368548980515696635881445300078929048431838
            0.0009089669537077930468881996688270615523439797972470449903614113458 0.9043997863133920735015145389031619162971025994120494156435056383 0.03657000213610424944449863805522525708657033133459050200879628579 0.0002330625235781478204122035014975241623618406677334615343388099990
            0.06514184191724962231411029874841132996504372176760923572922690005 0.7003393806704819150066794548378048094487838642575896665032017156 0.03756488014863942084314594130211867254923224802172193697724367110 0.002150853522720168693518139990963954034834429801133758143547889000
            0.4823251851268609702219022405431653732857407742187907511097240908 0.1063872202647660606082422166913570244847142217562377571376832406 0.3669367361101126543504042398292692109372138052934314536149672936 0.003595631154518662712348907472409247703989117616154270296974560443
            0.5131571800123465421649906816856000094237129118782943086992157029 0.3293713890289411699634379741941037189480080590437641848609124511 0.1564163597786528064990463862000579431310987954511819015120565620 0.0009589326000236011475902900734274863963360747497434216576008885289
            0.1006710625212378537682310717423472792843607924614060058028965833 0.5695723695494701704725061595391402106660654344823501071737837095 0.04356300297791438848281326023200056117658637343415662188432765040 0.003036561559826645559306134840772877674291593514142234163221106294
            0.5237396975770622302247198604757922783503967174791253505917021307 0.002650160838497455364942512463167979314073963919428457434616697630 0.1810666909012912891511345621660026290369901529866009983273867056 0.001025959726776522214835185744708640989078816811414928386295020686
            0.3406780012520992671650763654221422138395524987994251229113167142 0.2101909689592662430912241416749728153990065435123754657373624389 0.1150731340669544196310636680677498905372468273886299167232001620 0.006524771824097624209674343239077570070388512530283406408040147778
            0.08981796915031379608711076316179923970513942277840553592997092781 0.01182450804749051663356218634294462582681034393370758835896912271 0.3563895891683424740075902871883515108414233257763715216467377422 0.001581330313774575160265138340214486510520637528810729571294651356
            0.04838562814958744700003008581463802708074979299056625487179772236 0.1857653865739793259378473419434714475501786384048225311230279938 0.003216849558420410040043485814618680152282387183315338242804472398 0.0004719788094707109067676104703650886054904987714501864932679901901
            0.06095178993749332723695255349269346012735479185247606272021277805 0.3313733778222626754931214081864906636507900013247100762961059612 0.2312891820473547033347477522696303710062844800122857046499159414 0.004644861444207621250778811999806167031753850914391671765809791660
            0.01133241515186766050877116215742636606811948609971369257811841671 0.3350547591818930435322370811431504263717002328471016182944011327 0.2118453225543168169066896269216571459849654634671254706341266431 0.002195869619124833850836549718693330197356401677808273926227174348
            0.5627674555604480709696796756462099581251004008083897121384820576 0.04321245408638639610069306014867106908525462448584476560653643547 0.3419137667269786680944390159413832710102020302612774966183254636 0.001943246365838183703583382478415741761117160863087292585988099906
            0.9191384219123620339672310164554526964916678539333049722196320665 0.05546817040486831630512984218832308753054586434925433317381760863 0.01695384208979076332698421984416434132748895712969652500133306525 0.0003116858149771484943992682393443874631478226723176972239376432418
            0.01474501083029658882511821093276604535242541873143587675945675161 0.1217905347829033262886869329624110503280305808324841763093609036 0.3659668756667056951330068112709436195838761711347440666664666853 0.002713100110464541214185567669660966272569265182737822255271226258
            0.2253167036077325680377296068721662531758713325108242802456569195 0.2091254173067923534967684187698861383361158285777551160013319901 0.5511816295020478900428758575303235021735633762974819827133506929 0.002191515869678084559955178675178203437865231754085531823828381902
            0.03053908637261667142914207120808861335534288541537825295867953208 0.03551004025636827479740055758118687629950717993911499992924618861 0.6060278663330303399276373352213515180188665892563311312780937136 0.001876388751228706516592547325560969538792355931009536737051203457
            ];
    otherwise
        error('Degree not implemented in Jaskowiec-Sukumar set')
end

X=XYZW(:,1); Y=XYZW(:,2); Z=XYZW(:,3); W=XYZW(:,4); C=1-X-Y-Z;
XYZW_bar=[X Y Z C W];








function [X,W]=simplexquad(varargin)

% simplexquad.m - Gaussian Quadrature for an n-dimensional simplex.
%
% Construct Gauss points and weights for a n-dimensional simplex
% domain with vertices specified by the n*(n-1) matrix vert. Where each
% row contains the coordinates (x1,...,xn) for a vertex. The order of
% the quadrature scheme in each dimension must be the same in this
% implementation.
%
% Sample Usage:
%
% [X,W]=simplexquad(n,vert); % Specify the vertices
% [X,W]=simplexquad(n,dim);  % Specify the dimension and use unit simplex
%
% X will be a matrix for which the jth column are the grid points in each
% coordinate xj.
%
% Note: The standard n-dimensional simplex has vertices specified
%       vert=eye(n+1,n).
%
% The first four simplexes are
%
% n | Domain
% --|------------
% 1 | Interval
% 2 | Triangle
% 3 | Tetrahedron
% 4 | Pentatope
%
% Written by: Greg von Winckel
% Contact: gregvw(at)math(dot)unm(dot)edu
% http://math.unm.edu/~gregvw

nargin=length(varargin);

if nargin~=2
    error('simplexquad takes two input arguments');
end

N=varargin{1};

if length(N)~=1
    error('First argument must be a scalar');
end

if N~=abs(round(N-1))+1
    error('Number of Gauss points must be a natural number');
end

if isscalar(varargin{2}) % Dimension specified
    n=varargin{2};
    
    if n~=abs(round(n-1))+1
        error('Dimension must be a natural number');
    end
    
    m=n+1; vert=eye(m,n);
else                      % Vertices specified
    vert=varargin{2};
    [m,n]=size(vert);
    
    if m~=n+1
        error('The matrix of vertices must have n+1 rows and n columns');
    end
end

Nn=N^n;

if n==1 % The 1-D simplex is only an interval
    [q,w]=rquad(N,0); len=diff(vert);
    X=vert(1)+len*q;  W=abs(len)*w;
    
else % Find quadrature rules for higher dimensional domains
    
    for k=1:n
        [q{k},w{k}]=rquad(N,n-k);
    end
    
    [Q{1:n}]=ndgrid(q{:}); q=reshape(cat(n,Q{:}),Nn,n);
    [W{1:n}]=ndgrid(w{:}); w=reshape(cat(n,W{:}),Nn,n);
    
    map=eye(m); map(2:m,1)=-1; c=map*vert;
    W=abs(det(c(2:m,:)))*prod(w,2);
    
    qp=cumprod(q,2); e=ones(Nn,1);
    X=[e [(1-q(:,1:n-1)),e].*[e,qp(:,1:n-2),qp(:,n)]]*c;
end


function [x,w]=rquad(N,k)

k1=k+1; k2=k+2; n=1:N;  nnk=2*n+k;
A=[k/k2 repmat(k^2,1,N)./(nnk.*(nnk+2))];
n=2:N; nnk=nnk(n);
B1=4*k1/(k2*k2*(k+3)); nk=n+k; nnk2=nnk.*nnk;
B=4*(n.*nk).^2./(nnk2.*nnk2-nnk2);
ab=[A' [(2^k1)/k1; B1; B']]; s=sqrt(ab(2:N,2));
[V,X]=eig(diag(ab(1:N,1),0)+diag(s,-1)+diag(s,1));
[X,I]=sort(diag(X));
x=(X+1)/2; w=(1/2)^(k1)*ab(1,2)*V(1,I)'.^2;









function [XYZW_C,moments,moments_C,err]=rule_compression(XYZW,ade,bbox,...
    compr_type)

%--------------------------------------------------------------------------
% Object:
%
% Compression of trivariate cubature formulas by Tchakaloff points.
% It is useful in node reduction of algebraic cubature formulas.
% For further informations, see the web page: 
%         http://www.math.unipd.it/~marcov/Tchakaloff.html
%
%--------------------------------------------------------------------------
% Input:
%
% XYZW: rule to compress of total degree "ade". It is a matrix of dimension 
%  m x 4. The first 3 columns are the coordinates of the nodes,
%  while the 4th column represent the corresponding weights.
%
% ade: polynomial (total) degree of exactness.
%
% bbox: to determine the required compression, we make use of shifted 
%  tensorial Chebyshev basis is used, where "bbox" determines the shift. 
%  In general "bbox" contains the cubature domain, and it
%  represents the smaller hyperrectangle with sides parallel to the axes
%  containing the domain. 
%  If no information is given, "bbox" represents the smaller hyperrectangle
%  with sides parallel to the axes containing the cubature nodes.
%
% compr_type: determines the algorithm to use for compression
%  1: Matlab built-in lsqnonneg, 
%  2: QR extraction (possible negative weights).
%
%--------------------------------------------------------------------------
% Output:
%
% XYZW_C: compressed of total degree "ade". It is a matrix of dimension 
%  m x 4. The first 3 columns are the coordinates of the nodes,
%  while the 4th column represent the corresponding weights. The number
%  of nodes is at most (ade+1)*(ade+2)*(ade+3)/6. Furtermore, weights are
%  positive if Lawson-Hanson algorithm is applied (i.e. "compr_type" is set
%  equal to "0", "1" or "2".
%
% moments: shifted tensorial Chebyshev (relatively to "box") computed by
%  the starting rule (defined by XYZW).
%
% moments_C: shifted tensorial Chebyshev (relatively to "box") computed by
%  the compressed rule (defined by XYZW_C).
%
% err: compression error in norm 2, i.e. "norm(moments-moments_C,2)".
%
%--------------------------------------------------------------------------
% Subroutines called by "rule_compression":
%
% 1. chebvand (and its subroutines).
% 2. lsqnonneg
%
%--------------------------------------------------------------------------
% Authors:
% Alvise Sommariva and Marco Vianello,
% University of Padova, September 7, 2020.
%
% Modified: January 25, 2026.
%--------------------------------------------------------------------------


% .......................... troubleshooting ..............................

if nargin < 3
    X=XYZW(:,1); Y=XYZW(:,2); Z=XYZW(:,3);
    bbox=[min(X) max(X) min(Y) max(Y) min(Z) max(Z)];
end

if nargin < 4
    compr_type=0;
end

% ....................... compression startup .............................

% Computing moments of shifted tensorial Chebyshev basis (w.r.t. "bbox").
V=chebvand(XYZW(:,1:3),ade,bbox); 
moments=(XYZW(:,4))'*V; moments=moments';

[Q,R]=qr(V,0); % change of basis for stability ("V" may be ill-condit.).
moments1=R'\moments;

% ............................. compression ...............................


switch compr_type
    case 1
        % Matlab built-in "lsqnonneg"
        [w,~,~,~,~]=lsqnonneg(Q',moments1);
        ind1=find(w>0);
    otherwise
        % Extraction by QR method (possible negative weights!)
        w=Q'\moments1;
        ind1=find(w ~= 0);
end

XYZW_C=[XYZW(ind1,1:3) w(ind1)];

% .............................. statistics ...............................

if nargout >= 3
    % Vandermonde matrix restricted to extracted nodes.
    VC=V(ind1,:); 
    % Computing moments of compressed rule
    moments_C=(XYZW_C(:,4))'*VC; moments_C=moments_C';
    % Making comparison with exact (non compressed) moments.
    err=norm(moments-moments_C,2);
end









%--------------------------------------------------------------------------
% chebvand
%--------------------------------------------------------------------------

function V = chebvand(points,ade,bbox)

%--------------------------------------------------------------------------
% Object:
% Vandermonde matrix at "points" points, relative to Chebyshev basis of 
% degree "ade" on rectangle "bbox". The basis is of tensorial type, 
% of total degree.
%--------------------------------------------------------------------------
% Input:
% ade: polynomial degree.
% points: 3-column array of point coordinates.
% bbox: 6-component vector such that the cube:
% [bbox(1),bbox(2)] x [bbox(3),bbox(4)] x [bbox(5),bbox(6)] contains X.
%--------------------------------------------------------------------------
% Output:
% V: Chebyshev-Vandermonde matrix at "points", graded lexic. order.
%    size is M x N, where M is the number of nodes and N is
%    (ade+1)*(ade+2)*(ade+3)/6.
%--------------------------------------------------------------------------
% Authors:
% Alvise Sommariva and Marco Vianello
% University of Padova, September 1, 2020
%--------------------------------------------------------------------------

if nargin < 3 
    bbox=[];
end

if isempty(bbox)
    bbox=[min(points(:,1)) max(points(:,1)) ...
        min(points(:,2)) max(points(:,2)) ...
        min(points(:,3)) max(points(:,3))];
end

degrees=[];
for ii=0:ade
    for jj=0:(ade-ii)
        for kk=0: ade-jj-ii % computing contribution
            degrees=[degrees; ii jj kk];
        end
    end
end

% mapping the mesh in the cube [-1,1]^3
map=[(2*points(:,1)-bbox(2)-bbox(1))/(bbox(2)-bbox(1)) ...
    (2*points(:,2)-bbox(4)-bbox(3))/(bbox(4)-bbox(3)) ...
    (2*points(:,3)-bbox(6)-bbox(5))/(bbox(6)-bbox(5))];

% Chebyshev-Vandermonde matrix on the mesh
T1=chebpolys(ade,map(:,1));
T2=chebpolys(ade,map(:,2));
T3=chebpolys(ade,map(:,3));
V=T1(:,degrees(:,1)+1).*T2(:,degrees(:,2)+1).*T3(:,degrees(:,3)+1);








%--------------------------------------------------------------------------
% chebpolys
%--------------------------------------------------------------------------

function T=chebpolys(deg,x)

%--------------------------------------------------------------------------
% Object:
% This routine computes the Chebyshev-Vandermonde matrix on the real line
% by recurrence.
%--------------------------------------------------------------------------
% Input:
% deg: maximum polynomial degree
% x: 1-column array of abscissas
%--------------------------------------------------------------------------
% Output:
% T: Chebyshev-Vandermonde matrix at x, T(i,j+1)=T_j(x_i), j=0,...,deg.
%    Its size is M x N where M is the number of nodes and N the dimension 
%    of the polynomial space (i.e. "deg+1").
%--------------------------------------------------------------------------
% Authors:
% Alvise Sommariva and Marco Vianello
% University of Padova, September 1, 2020.
%--------------------------------------------------------------------------

T=zeros(length(x),deg+1);
t0=ones(length(x),1);
T(:,1)=t0;
t1=x;
T(:,2)=t1;

for j=2:deg
    t2=2*x.*t1-t0;
    T(:,j+1)=t2;
    t0=t1;
    t1=t2;
end








