
function [T,w,res,mom,LHDM_stats]=cqmc_v2(deg,X,tol,comp_type,wX,dom)

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% The routine computes a Compressed Quasi-Monte Carlo formula from a large
% low-discrepancy sequence on a low-dimensional compact set preserving the
% QMC polynomial moments up to a given degree, via Tchakaloff sets and NNLS
% moment-matching.
%--------------------------------------------------------------------------
% INPUT:
%--------------------------------------------------------------------------
% deg: polynomial degree
% X: d-column array of a low-discrepancy sequence
% tol: moment-matching relative residual tolerance
% comp_type: compression algorithm. 1: lsqnonneg (default) 2: LHDM.
% wX: weights corresponding to X
% dom: 1: full Chebyshev basis,
%      2: subset of the Chebyshev basis depending on the numerical
%         dimension of the polynomial space (default).
%--------------------------------------------------------------------------
% OUTPUT:
%--------------------------------------------------------------------------
% T: compressed points (subset of X).
% w: positive weights (corresponding to T).
% res: moment-matching relative residual
% mom: moments of a certain shifted tensorial Chebyshev basis (total
%   degree equal to deg).
% LHDM_stats: vector of cardinality and iterations along the compression
%     stages.
%--------------------------------------------------------------------------
% Dates
%--------------------------------------------------------------------------
% First version: May 28, 2022
% Last update:: April 15, 2023
%--------------------------------------------------------------------------
% Authors
%--------------------------------------------------------------------------
% G. Elefante, A. Sommariva, M. Vianello
%--------------------------------------------------------------------------
% Paper
%--------------------------------------------------------------------------
% 1. "CQMC: an improved code for low-dimensional Compressed Quasi-MonteCarlo
% cubature"
% 2. "Compressed QMC volume and surface integration on union of balls"
% G. Elefante, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
% LICENSE:
%--------------------------------------------------------------------------
% Copyright (C) 2023- 
% Giacomo Elefante, Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Giacomo Elefante <giacomo.elefante@usi.ch>
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello   <marcov@math.unipd.it>
%
% Last update: January 24, 2026
%--------------------------------------------------------------------------



% ............................ Troubleshooting  ...........................

if nargin < 3, tol=10^(-10); end
if isempty(tol), tol=10^(-10); end

if nargin < 4, comp_type=1; end
if isempty(comp_type), comp_type=1; end

if nargin < 5, wX=ones(size(X,1),1); end
if isempty(wX), wX=ones(size(X,1),1); end

if nargin < 6, dom=2; end
if isempty(dom), dom=2; end


% ............................ Function body  .............................

M=length(X(:,1));

if M <= 0
    fprintf(2,'\n \t No point in domain ');
    T=[]; w=[]; res=[]; LHDM_stats=[]; return;
end

% 1.2: Chebyshev-Vandermonde matrix of degree deg at X
[V,~,N] = dCHEBVAND_v2(deg,X,[],dom);

% 1.3: QMC moments
mom=V'*wX;

% 2: computing the compressed QMC formula
% 2.1: initializing the candidate Tchakaloff set cardinality
m=2*N;
% 2.2: initializing the residual
res=2*tol;
% 2.3: increase percentage of a candidate Tchakaloff set
theta=2;
% iterative search of a Tchakaloff set by NNLS moment-matching
s=1;

LHDM_stats=[];
momtype = 0;

while res(end)>tol && m<=M
    % 2.4: reduced Chebyshev-Vandermonde matrix;
    Vm=V(1:m,:);

    % 2.5: qr-decomposition
    [Am,Rm]=qr(Vm,0);

    % 2.6: modified qmc moments
    switch momtype
        case 0
            qm=(mom'/Rm)';
        case 1
            AM = V/Rm;
            qm=AM'*wX;
            Am = Vm/Rm;
    end

    % 2.7: nonnegative weights by NNLS
    switch comp_type
        case 1
            % fprintf('\n \t -> lsqnonneg');
            [u,~] = lsqnonneg(Am',qm);
            iter=NaN;
        case 2
            % fprintf('\n \t -> LHDM');
            [u,~,~,iter] = LHDM(Am',qm);
    end

    % 2.8: relative residual
    res(s)=norm(Vm'*u-mom)/norm(mom);

    LHDM_stats=[LHDM_stats; m iter];

    % 2.9: updating the candidate Tchakaloff set cardinality
    if s>1
        if (res(s-1)/res(s)<=10) && (momtype==0)
            momtype = 1;
            % 2.10 Recomputing the the weights with the different modified
            % moments without updates
            AM = V/Rm;
            qm=AM'*wX;
            Am = Vm/Rm;
            switch comp_type
                case 1
                    [u,~] = lsqnonneg(Am',qm);
                    iter=NaN;
                case 2
                    [u,~,~,iter] = LHDM(Am',qm);
            end
            s = s +1;
            res(s)=norm(Vm'*u-mom)/norm(mom);
        elseif (res(s-1)/res(s)<=10) && (momtype==1)
            m = M;
            s = s+1;
            continue;
        end
    end
    m=floor(theta*m);
    s=s+1;
end % while

% compressed formula
% 2.11: indexes of the positive weights
ind=find(u>0);

% 2.12: compressed support points selection
T=X(ind,:);

% 2.13: corresponding positive weights
w=u(ind);















function [V,dbox,N] = dCHEBVAND_v2(deg,X,dbox,dom)

%--------------------------------------------------------------------------
% Object
%--------------------------------------------------------------------------
% This routine computes the Chebyshev-Vandermonde matrix for degree "deg"
% on a d-dimensional point cloud "X".
%
% The Chebyshev basis is the tensorial Chebyshev basis of total degree
% "deg", shifted on the hyperrectangle defined by "dbox".
%
% If "dbox" is not provided, the routine sets that variable to define the
% smaller "hyperrectangle" (box) with sides parallel to the cartesian
% axes and containing the pointset "X".
%--------------------------------------------------------------------------
% Input
%--------------------------------------------------------------------------
% deg: polynomial degree;
% X: d-column array of "m" points cloud (matrix "m x d");
% * dbox: variable that defines the smallest hyperectangle with sides
%     parallel to the axis, containing the domain.
%     If "dbox" is not provided, it defines the smaller "hyperrectangle",
%     with sides parallel to the cartesian axes, containing the pointset
%     "X".
%     It is a matrix with dimension "2 x d", where "d" is the dimension
%     of the space in which it is embedded the domain.
%     For instance, for a 2-sphere, it is "d=3", while for a 2 dimensional
%     polygon it is "d=2".
%     As example, the set "[-1,1] x [0,1]" is described as "[-1 0; 1 1]".
% * dom: 1: Chebyshev-Vandermonde matrix (default),
%        2: Chebyshev-Vandermonde matrix with the right numerical rank
%         (useful for algebraic surfaces)
%
% Note: the variables with an asterisk "*" are not mandatory and can be
% also set as empty matrix.
%--------------------------------------------------------------------------
% Output
%--------------------------------------------------------------------------
% V: shifted Chebyshev-Vandermonde matrix for degree "deg" on the pointset
%    "X", relatively to "dbox".
% dbox: variable that defines the hyperrectangle with sides parallel to the
%     axis, containing the domain.
% N: dimension of the polynomial space.
%--------------------------------------------------------------------------
% Previous versions.
%--------------------------------------------------------------------------
% This routine is an extension of dCHEBVAND, taking into account that for
% domains "Omega" embedded in R^d the dimension of the polynomial space of
% total degree "deg" may be inferior of that in the whole "R^d".
%--------------------------------------------------------------------------
% External routines
%--------------------------------------------------------------------------
% This routine requires:
% 1. mono_next_grlex (author: John Burkardt).
%--------------------------------------------------------------------------
% Dates
%--------------------------------------------------------------------------
% Written by M. Dessole, F. Marcuzzi, M. Vianello, on July 2020;
% Modified by G. Elefante, A. Sommariva, M. Vianello, on April 16, 2023.
%--------------------------------------------------------------------------
% COPYRIGHT
%--------------------------------------------------------------------------
% Copyright (C) 2020-
%
% Authors:
% Monica Dessole, Giacomo Elefante, Fabio Marcuzzi, Alvise Sommariva,
% Marco Vianello
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%--------------------------------------------------------------------------


% ........................... Function body ...............................

% ...... troubleshooting ......

if nargin < 3, dbox=[]; end
if isempty(dbox)
    a=min(X); b=max(X); dbox=[a; b];
else
    a=dbox(1,:); b=dbox(2,:);
end

if nargin < 4, dom=1; end
if isempty(dom), dom=1; end



% ..... main code below .....

% d-uples of indices with sum less or equal to "deg" graded lexicographical
% order
d=size(X,2); N = nchoosek(deg+d,d); duples = zeros(N,d);
for i=2:N
    duples(i,:) = mono_next_grlex(d,duples(i-1,:));
end

% mapping the mesh in the hypercube "[-1,1]^d"
map = zeros(size(X));
for i=1:d
    map(:,i)=(2*X(:,i)-b(i)-a(i))/(b(i)-a(i));
end

% Chebyshev-Vandermonde matrix on the mesh
T=chebpolys(deg,map(:,1));
V0=T(:,duples(:,1)+1);
for i=2:d
    T=chebpolys(deg,map(:,i));
    V0=V0.*T(:,duples(:,i)+1);
end

% Dimension of the polynomial space on X and Vandermonde matrix of the
% right (numerical) dimension (w.r.t. the choosen polynomial space).
switch dom
    case 1
        N=length(V0(1,:)); V=V0;
    case 2
        nu=length(V0(1,:));
        N=rank(V0(1:nu,1:nu));
        [~,~,p] = qr(V0(1:nu,1:nu),0);
        Cp = V0(:,p);
        V = Cp(:,1:N);
end










function x = mono_next_grlex ( m, x )

%**************************************************************************
%
%% MONO_NEXT_GRLEX: grlex next monomial.
%
%  Discussion:
%
%    Example:
%
%    M = 3
%
%    #  X(1)  X(2)  X(3)  Degree
%      +------------------------
%    1 |  0     0     0        0
%      |
%    2 |  0     0     1        1
%    3 |  0     1     0        1
%    4 |  1     0     0        1
%      |
%    5 |  0     0     2        2
%    6 |  0     1     1        2
%    7 |  0     2     0        2
%    8 |  1     0     1        2
%    9 |  1     1     0        2
%   10 |  2     0     0        2
%      |
%   11 |  0     0     3        3
%   12 |  0     1     2        3
%   13 |  0     2     1        3
%   14 |  0     3     0        3
%   15 |  1     0     2        3
%   16 |  1     1     1        3
%   17 |  1     2     0        3
%   18 |  2     0     1        3
%   19 |  2     1     0        3
%   20 |  3     0     0        3
%
%    Thanks to Stefan Klus for pointing out a discrepancy in a previous
%    version of this code, 05 February 2015.
%
%  Licensing:
%
%    This code is distributed under the GNU LGPL license.
%
%  Modified:
%
%    05 February 2015
%
%  Author:
%
%    John Burkardt
%
%  Parameters:
%
%    Input, integer M, the spatial dimension.
%
%    Input, integer X(M), the current monomial.
%    The first item is X = [ 0, 0, ..., 0, 0 ].
%
%    Output, integer X(M), the next monomial.
%

%
%  Ensure that 1 <= M.
%
if ( m < 1 )
    fprintf ( 1, '\n' );
    fprintf ( 1, 'MONO_NEXT_GRLEX - Fatal error!' );
    fprintf ( 1, '  M < 1\n' );
    error ( 'MONO_NEXT_GRLEX - Fatal error!' );
end
%
%  Ensure that 0 <= XC(I).
%
for i = 1 : m
    if ( x(i) < 0 )
        fprintf ( 1, '\n' );
        fprintf ( 1, 'MONO_NEXT_GRLEX - Fatal error!' );
        fprintf ( 1, '  X(I) < 0\n' );
        error ( 'MONO_NEXT_GRLEX - Fatal error!' );
    end
end
%
%  Find I, the index of the rightmost nonzero entry of X.
%
i = 0;
for j = m : -1 : 1
    if ( 0 < x(j) )
        i = j;
        break
    end
end
%
%  set T = X(I)
%  set X(I) to zero,
%  increase X(I-1) by 1,
%  increment X(M) by T-1.
%
if ( i == 0 )
    x(m) = 1;
    return
elseif ( i == 1 )
    t = x(1) + 1;
    im1 = m;
elseif ( 1 < i )
    t = x(i);
    im1 = i - 1;
end

x(i) = 0;
x(im1) = x(im1) + 1;
x(m) = x(m) + t - 1;

return










function T=chebpolys(deg,x)

%--------------------------------------------------------------------------
% Object:
% This routine computes the Chebyshev-Vandermonde matrix on the real line
% by recurrence.
%--------------------------------------------------------------------------
% Input:
% deg: maximum polynomial degree
% x: 1-column array of abscissas
%--------------------------------------------------------------------------
% Output:
% T: Chebyshev-Vandermonde matrix at x, T(i,j+1)=T_j(x_i), j=0,...,deg.
%--------------------------------------------------------------------------
% Authors:
% Alvise Sommariva and Marco Vianello
% University of Padova, December 15, 2017
%--------------------------------------------------------------------------

T=zeros(length(x),deg+1);
t0=ones(length(x),1); T(:,1)=t0;
t1=x; T(:,2)=t1;

for j=2:deg
    t2=2*x.*t1-t0;
    T(:,j+1)=t2;
    t0=t1;
    t1=t2;
end








function [x,resnorm,exitflag, outeriter] = LHDM(C,d,options,verbose)

%--------------------------------------------------------------------------
% Object:
% This routine solves (possibly underdetermined) linear least squares problem
% min || C*x - d || with nonnegativity constraints x >= 0 using the standard
% Lawson-Hanson algorithm accelerated by Deviation Maximization technique.
%--------------------------------------------------------------------------
% Input:
% C: least squares matrix;
% d: least squares right hand side;
% * options: structure containing the values of optimization parameters,
%            possible field are:
%    init:   true if ULS initialization of passive set is desired, false
%            otherwise;
%    tol:    tolerance on the projected residual, stop criterion
%    k:      maximum number of indices to be added to the Passive set at each
%            iteration:
%    thres:  threshold on the cosine of the angle between each pair of
%            columns cuncurrently added to the Passive set, value between
%            0 and 1;
%
% Note: the variables with an asterisk "*" are not mandatory and can be
% also set as empty matrix.
%--------------------------------------------------------------------------
% Output
% x: nonnegative vector that minimizes || C*x - d ||;
% resnorm: squared 2-norm of the residual: || C*x - d ||^2;
% exitflag: exit condition, possible value are:
%    1  LHDM converged to a solution x;
%    0  Maximum iteration count was exceeded, increasing the tolerance
%       (options.tol) may lead to a solution;
% outeriter: number of outer iterations performed by LHDM algorithm.
%--------------------------------------------------------------------------
% Dates:
% Written by M. Dessole, F. Marcuzzi, M. Vianello - July 2020
% Modified by M. Dessole, A. Sommariva, M. Vianello - November 2020
%--------------------------------------------------------------------------
% COPYRIGHT
%--------------------------------------------------------------------------
% Copyright (C) 2020-
%
% Authors:
% Monica Dessole, Fabio Marcuzzi, Alvise Sommariva, Marco Vianello
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <https://www.gnu.org/licenses/>.
%--------------------------------------------------------------------------

% .........................  Function Body ................................

% ..... troubleshooting .....

if nargin < 2
    error('LHDM:NotEnoughInputs',...
        getString(message('LHDM:NotEnoughInputs')));
end
if nargin < 3
    options=[];
end
if nargin < 4
    verbose=0;
end

[m,n] = size(C);

% Initialize vector of n zeros and Infs (to be used later)
nZeros = zeros(n,1);
wz = nZeros;

itmax = m*2;

% Initialize set of non-active columns to null
P = false(n,1);
cardP = 0;
% Initialize set of active columns to all and the initial point to zeros
Z = true(n,1);
x = nZeros;

% Check if options was created with optimoptions
if ~isempty(options)
    if ~isa(options,'struct')
        error('MATLAB:LDHM:ArgNotStruct',...
            getString(message('MATLAB:LHDM:commonMessages:ArgNotStruct')));
    end
    if isfield(options,'thres')
        thres = options.thres;
        if (thres <= eps)
            LHDMflag = 0;
        end
    else
        thres = 0.2222;
    end
    if isfield(options,'thres_w')
        thres_w = options.thres_w;
    else
        thres_w = 0.8;
    end
    if isfield(options,'k')
        k = options.k;
        if (k == 1)
            LHDMflag = 0;
        else
            LHDMflag = 1;
        end
    else
        k = ceil(m/20);
    end
    if isfield(options,'tol')
        tol = options.tol;
    else
        tol = 10*eps*norm(C,1)*length(C);
    end
    if isfield(options,'init')
        if options.init
            xtmp = C\d;
            Idx = find(xtmp>0);
            % Initialize set of non-active columns
            P(Idx) = true;
            % Initialize set of active columns
            Z(Idx) = false;
            % Initialize starting point
            x(P) = C(:,P)\d;
            tmp = find(x<0);
            if(size(tmp,1)>0)
                x(tmp) = 0;
                P(tmp) = false;
                Z(tmp) = true;
            end
            if verbose
                fprintf('LHDM(%d) with ULS inizialization\n', k);
            end
            cardP = sum(P);
        else
            if verbose
                fprintf('LHDM(%d) \n', k);
            end
        end
    else
        if verbose
            fprintf('LHDM(%d) \n', k);
        end
    end
else
    thres   = 0.2222;
    thres_w = 0.8;
    k = ceil(m/20);
    tol = 10*eps*norm(C,1)*length(C);
    LHDMflag = 1;
    if verbose
        fprintf('LHDM(%d) \n', k);
    end
end

% ..... main code below .....

if LHDMflag
    Cnorm = zeros(size(C));
    for j=1:n
        Cnorm(:,j) = C(:,j)/norm(C(:,j));
    end
end
% Initialize the residual and dual variables
resid = d - C*x;
w = C'*resid;
% Set up iteration auxiliary variables
outeriter = 0;
totiter = 0;
% Outer loop to put variables into set to hold positive coefficients
while (any(Z) && (any(w(Z) > tol) || any(x(P) < 0) ) && (totiter < itmax) )
    outeriter = outeriter + 1;
    totiter = totiter+1;
    % Create wz, a Lagrange multiplier vector of variables in the zero set.
    % wz must have the same size as w to preserve the correct indices, so
    % set multipliers to -Inf for variables outside of the zero set.
    wz(P) = -Inf;
    wz(Z) = w(Z);

    if ((outeriter <= 1) || (~LHDMflag))
        % in first iteration we drop DM to ensure w_T > 0
        [~,t] = max(wz);
    elseif LHDMflag && (removedP ~= addedP)
        t = DM(Cnorm, wz, k, thres, thres_w);
        if (length(t) + cardP > m)
            t = t(1:m-cardP);
        end

    else
        [~,t] = max(wz);
    end
    % number of indices added to P
    addedP = length(t);

    % Reset intermediate solution z
    z = zeros(size(x));
    % Move variable t from zero set to positive set
    P(t) = true;
    Z(t) = false;
    % Compute intermediate solution using only variables in positive set
    z(P) = C(:,P)\d;

    % inner loop to remove elements from the positive set which no longer belong
    iter = 0;
    removedP = 0;
    while (any(z(P) <= 0) && (totiter < itmax))
        totiter = totiter +1;
        iter = iter+1;
        % Find indices where intermediate solution z is approximately negative
        Q = (z <= 0) & P;
        % Choose new x subject to keeping new x nonnegative
        b = x(Q)./(x(Q) - z(Q));
        alpha = min(b);
        x = x + alpha*(z - x);
        % number of indices removed from P
        t = find(((abs(x) < tol) & P))';
        removedP = removedP + length(t);
        % Reset Z and P given intermediate values of x
        Z = ((abs(x) < tol) & P) | Z;
        P = ~Z;
        % Reset z
        z = nZeros;
        % Compute intermediate solution using only variables in positive set
        z(P) = C(:,P)\d;
    end
    % update cardinality of P
    cardP = cardP + addedP-removedP;
    x=z;

    resid = d - C*x;
    w = C'*resid;

end

if (outeriter < itmax)
    exitflag = 1;
else
    exitflag = 0;
end

resnorm = resid'*resid;



function [p] = DM(Cnorm, wz, k, thres,thres_w)
% Deviation Maximization

[wzI, I] = sort(wz, 'descend');
t = I(1);
p = t;
thres_wloc = thres_w*wzI(1);
C = find(wzI>thres_wloc);
n = size(C,1);
add = 1;
for i = 2:n
    c = C(i);
    if (max(abs(Cnorm(:,I(c))'*Cnorm(:,p))) < thres)
        p = [I(c) p];
        add = add+1;
    end
    if (add >= k)
        break;
    end
end
