
function demo_qmc_union_spheres_CQMC_26

%--------------------------------------------------------------------------
% OBJECT
%--------------------------------------------------------------------------
% This routine determines a QMC cubature rule on the (outer) surface of the
% union of spheres   defined by there "centers" and "radii".
% 
% In particular it provides a QMC cubature rule, stored in "xyzw", where
% 1. the nodes correspond to xyzw(:,1:3)
% 2. the weights correspond to xyzw(:,1:4)
%
% Furthermore, the value "A(k)" is the area of the subregion of the k-th 
% sphere, that belongs to such  (outer) surface.
% 
% Next we apply cubature compression, so to match the QMC moments up to a
% fixed "ade" (algebraic degree of exactness).
%--------------------------------------------------------------------------
% EXAMPLE
%--------------------------------------------------------------------------
% >> demo_qmc_union_spheres_CQMC_26
% 
%  	 Number of spheres          : 5        
% 
%  	 Cardinality on each sphere : 10000    
%  	 Cardinality QMC set        : 13136    
%  	 Cardinality CQMC rule      : 84       
% 
%  	 Compression ade            : 6        
%  	 Compression residual       : 5.161e-15
% 
%  	 Cputime QMC                : 9.904e-03
%  	 Cputime CQMC (only)        : 1.427e-01
% 
% >>
%--------------------------------------------------------------------------
% AUTHORS
%--------------------------------------------------------------------------
% G. Elefante, A. Sommariva, M. Vianello
%--------------------------------------------------------------------------
% PAPER
%--------------------------------------------------------------------------
% "Qbubble: a numerical code for compressed QMC volume and surface 
% integration on union of spheres"
% G. Elefante, A. Sommariva and M. Vianello
%--------------------------------------------------------------------------
%  RELEASE DATE
%--------------------------------------------------------------------------
% First version: January 24, 2026
% Last update  : January 24, 2026
%--------------------------------------------------------------------------
% LICENSE:
%--------------------------------------------------------------------------
% Copyright (C) 2026- 
%
% Giacomo Elefante, Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Authors:
%
% Giacomo Elefante <giacomo.elefante@usi.ch>
% Alvise Sommariva <alvise@math.unipd.it>
% Marco Vianello   <marcov@math.unipd.it>
%
% Date: January 24, 2026
%--------------------------------------------------------------------------



% Cardinality points on each sphere
N=10000;

% Number of spheres.
Nsph=5;

% Number of tests
Ntests=10;

% Degree of numerical compression
ade=6;

% ........................ Define spheres .................................

centers=rand(Nsph,3);
radii=rand(Nsph,1);


% ........................ QMC pointset ...................................

% Define pointset type

pointset_type='H';

cpus=zeros(Ntests,1);
cpus_CQMC=zeros(Ntests,1);
% Determine QMC pointset on the outer surface of union of spheres.
for k=1:Ntests

    tic
    [xyzw,~,~,~] = qmc_union_spheres_26(centers,radii,N,pointset_type);
    cpus(k)=toc;

    tic;
    X=xyzw(:,1:3); wX=xyzw(:,4); tol=10^(-12); comp_type=2; dom=1;
    [T,w,res,~,~]=cqmc_v2(ade,X,tol,comp_type,wX,dom);
    cpus_CQMC(k)=toc;

end



% ........................ Plot pointset ..................................

xyz=xyzw(:,1:3);
plot_QMC_union_spheres_26(centers,radii,xyz);

% .......................... Statistics  ..................................

fprintf('\n \t Number of spheres          : %-9.0f',Nsph);
fprintf('\n')
fprintf('\n \t Cardinality on each sphere : %-9.0f',N);
fprintf('\n \t Cardinality QMC set        : %-9.0f',size(xyz,1));
fprintf('\n \t Cardinality CQMC rule      : %-9.0f',length(w));
fprintf('\n')
fprintf('\n \t Compression ade            : %-9.0f',ade);
fprintf('\n \t Compression residual       : %1.3e',res(end));
fprintf('\n')
fprintf('\n \t Cputime QMC                : %1.3e',mean(cpus));
fprintf('\n \t Cputime CQMC (only)        : %1.3e',mean(cpus_CQMC));
fprintf('\n \n');