
function xyzw = cub_sphtri(n,P1,P2,P3)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Cubature rule over spherical triangle contained in a cap whose
% polar angle is strictly inferior than π.
%
% The spherical triangle is on the sphere centered in the origin, whose 
% radius is defined by the "vertices" of the polygon.
%
% The routine computes a cubature formula "xyzw" with (numerical) algebraic
% degree of precision "n" on the spherical triangle with vertices "P1",
% "P2", "P3", by a 2D formula of degree of precision "m+n" on the xy
% projection of the spherical triangle rotated to put the barycenter at the
% north pole.
%
% The parameter "m" depends on the size of the circumradius and it is "2"
% for "small" "r", becoming bigger as "r" approaches "1".
%--------------------------------------------------------------------------
% Chebfun requirement:
%--------------------------------------------------------------------------
% This routine requires the installation of Chebfun.
% See "https://www.chebfun.org/download/" for details.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% n: algebraic degree of precision of the rule;
% P1,P2,P3: column arrays of the spherical triangle vertices coords
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% xywz : 4-column array of nodes cartesian coords and weights
%--------------------------------------------------------------------------
% Note:
%--------------------------------------------------------------------------
% The code has a warning at line 126, difficult to skip due to a matrix
% whose size cannot easily be forecasted before the process.
%--------------------------------------------------------------------------
% Routines called:
%--------------------------------------------------------------------------
% 1. compute_m (attached)
% 2. cub_circsect (attached)
%
% The routine "cub_circsect" requires "quad_trig", "r_jacobi" and "gauss"
% that are attached below as well as the "quad_trig" subroutines.
%--------------------------------------------------------------------------
%% Copyright (C) 2021-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date: January 01, 2021
%% Update: December 29, 2025
%--------------------------------------------------------------------------


% ............................ troubleshooting ............................

% working with vertices as column vectors
if size(P1,1) == 1, P1=P1'; end
if size(P2,1) == 1, P2=P2'; end
if size(P3,1) == 1, P3=P3'; end

% ------------------------------- main code -------------------------------
% We scale the problem in the unit-sphere with center [0 0 0].
% In other words, we make up a cubature rule on the unit sphere (by
% contraction w.r.t. the unit sphere and map back the so obtained rule to
% the starting sphere with radius R).

% ... barycenter ...
RR=norm(P1);
vert0=[P1 P2 P3]/RR;
CC=1/3*(P1+P2+P3); CC=CC/norm(CC);

% .................... rotation matrix to the north pole ..................

[az,el] = cart2sph(CC(1),CC(2),CC(3));
phi=az; theta=pi/2-el;
cp=cos(phi); sp=sin(phi); ct=cos(theta); st=sin(theta);
R1=[ct 0 -st; 0 1 0; st 0 ct]; R2=[cp sp 0; -sp cp 0; 0 0 1];
R=R1*R2; invR=R';

% ............... vertices of the triangle at the North Pole ..............

vert1=R*vert0;

% ...... computing "m" needed in determining the degree of precision ......
m=compute_m(vert1');

% .................... determining quadrature rule ........................

xyzw=[]; % nodes on the sphere
vert1=[vert1 vert1(:,1)];

for i=1:3

    % affine transformation matrix
    P=vert1(:,i); Q=vert1(:,i+1);
    om=acos(P'*Q);
    xi=cos(om/2); eta=sin(om/2);
    M=[xi eta 0 0; 0 0 xi eta; xi -eta 0 0; 0 0 xi -eta];
    h=[P(1); P(2); Q(1); Q(2)];
    u=M\h;
    T=[u(1) u(2); u(3) u(4)];

    % nodes and weights for the xy-projection of the rotated spher.triangle
    nw=cub_circsect(m+n,om/2,0,1); nod2=T*nw(:,1:2)';

    % spherical triangle nodes and weights on North Pole spher.triangle
    x=nod2(1,:); y=nod2(2,:); z=sqrt(1-x.^2-y.^2);

    weights=abs(det(T))*nw(:,3)./z';

    % inverse rotation to original spherical triangle
    nodes=invR*[x; y; z];

    % cubature rule update
    xyzw=[xyzw;[nodes' weights]];

end

% Exporting results to the sphere with radius "R".
X=RR*xyzw(:,1); Y=RR*xyzw(:,2); Z=RR*xyzw(:,3); W=RR*xyzw(:,4);
xyzw=[X Y Z W];






function m=compute_m(vert)

%--------------------------------------------------------------------------
% Object:
% It computes the "m" positive integer value depending from the sph. tri.
% with vertices "vert", so that a WAM over the sph. triangle with degree
% equal to "n" can be obtained via WAM of degree "m+n" on its projection on
% the xy-plane.
%--------------------------------------------------------------------------
% Input:
% vert: points defining the sph. triangle (the k-th point is described
%    by the k-th row.
%--------------------------------------------------------------------------
% Output:
% m: positive integer value depending from the sph.triangle with vertices
% "vert", so that a WAM over the sph. triangle with degree equal to "n" can
% be obtained via cubature of degree "m+n" on its projection on the
% xy-plane.
%--------------------------------------------------------------------------
% Data:
% First version: 13/01/2021 by A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------

r=sqrt( (vert(:,1)).^2 + (vert(:,2)).^2 );
r0=max(r);

intvf=[0,r0]; 
F=@(r) sqrt(1-r); 
f=chebfun(F,intvf);
m=max(1,(length(f)-1));










function xyw = cub_circsect(n,omega,r1,r2)

%--------------------------------------------------------------------------
% Object:
% The routine computes the nodes and weights of a product gaussian
% formula on a circular annular sector centered at the origin
% with angles in [-omega,omega]
%--------------------------------------------------------------------------
% Input:
% n: algebraic degree of exactness
% omega: half-length of the angular interval, 0<omega<=pi
% r1,r2: internal and external radius, 0<=r1<r2
%--------------------------------------------------------------------------
% Output:
% xyw: (ceil((n+2)/2) x (n+1)) x 3 array of (xnodes,ynodes,weights)
%--------------------------------------------------------------------------
% Required routines:
% 1. r_jacobi.m (www.cs.purdue.edu/archives/2002/wxg/codes/OPQ.html)
% 2. gauss.m (www.cs.purdue.edu/archives/2002/wxg/codes/OPQ.html)
% 3. quad_trig.m
%--------------------------------------------------------------------------
% Written by Gaspare Da Fies and Marco Vianello, University of Padova
% Date: November 8, 2011.
% Last update: January 4. 2020.
%--------------------------------------------------------------------------

% trigonometric gaussian formula on the arc
tw=quad_trig(n,-omega,omega);

% algebraic gaussian formula on the radial segments
ab=r_jacobi(ceil((n+2)/2),0,0);
xw=gauss(ceil((n+2)/2),ab);
xw(:,1)=xw(:,1)*(r2-r1)/2+(r2+r1)/2;
xw(:,2)=xw(:,2)*(r2-r1)/2;

% creating the polar grid
[r,theta]=meshgrid(xw(:,1),tw(:,1));
[w1,w2]=meshgrid(xw(:,2),tw(:,2));

% nodal cartesian coordinates and weights
xyw(:,1)=r(:).*cos(theta(:));
xyw(:,2)=r(:).*sin(theta(:));
xyw(:,3)=r(:).*w1(:).*w2(:);










function tw=quad_trig(n,alpha,beta)

%--------------------------------------------------------------------------
% Object:
% Computation of trigonometric gaussian rules on the unit arc from "alpha"
% to "beta".
%--------------------------------------------------------------------------
% Inputs:
% n    : the rule computes computes the n+1 angles and weights of a
%        trigonometric gaussian quadrature formula on [alpha,beta], with
%                          0 < beta-alpha <= 2*pi;
% alpha, beta: arc angles for trigonometric gaussian rules on the unit arc
%       from "alpha" to "beta".
%--------------------------------------------------------------------------
% Outputs
% tw   : (n+1) x 2 matrix, where the first column contains the nodes, while
%        the second one contains the weights.
%--------------------------------------------------------------------------
% First version: May 18, 2013 by G. Da Fies, A. Sommariva, M. Vianello.
% Successive versions have been made by G. Meurant, A. Sommariva and
% M. Vianello.
% Last release: January 04, 2020.
%--------------------------------------------------------------------------

% .......................... troubleshooting  .............................

if nargin < 1, n=10; end
if nargin < 2, beta=pi; alpha=-beta; end
if nargin < 3, beta=alpha; alpha=-beta; end

n=n+1;
omega=(beta-alpha)/2;
ab = r_subchebyshev(n,omega);
xw_symm_eigw = SymmMw(n,ab);
tw=quad_trig_conversion(xw_symm_eigw,omega);
tw(:,1)=tw(:,1)+(beta+alpha)/2;









function ab=r_subchebyshev(n,omega)

%--------------------------------------------------------------------------
% Object:
% Recurrence coeffs for the monic OPS w.r.t. the weight function
%         w(x)=2*sin(omega/2)/sqrt(1-sin^2(omega/2)*x^2)
% by the modified Chebyshev algorithm.
% The reference angle of the rule is [-omega,omega].
%--------------------------------------------------------------------------
% Inputs:
% n     : number of points.
% omega : arc angle.
%--------------------------------------------------------------------------
% Output:
% ab   : three terms recursion
%--------------------------------------------------------------------------
% First version: May 18, 2013 by G. Da Fies, A. Sommariva, M. Vianello.
% Successive versions have been made by G. Meurant, A. Sommariva and
% M. Vianello.
% Last release: January 04, 2020.
%--------------------------------------------------------------------------

N=n; n=n-1;

% modified Chebyshev moments by recurrence
if rem(N,2) == 1
    NN=N+1; nn=n+1;
else
    NN=N; nn=n;
end
mom=fast_moments_computation(omega,2*nn+1);

% recurrence coeffs of the monic Chebyshev polynomials
abm(:,1)=zeros(2*nn+1,1);
abm(:,2)=0.25*ones(2*nn+1,1); abm(1,2)=pi; abm(2,2)=0.5;

% recurrence coeffs for the monic OPS w.r.t. the weight function
ab = fast_chebyshev(NN,mom,abm);









function x = tridisolve(a,b,c,d)

%--------------------------------------------------------------------------
% Object:
% Solution of tridiagonal system of equations.
%--------------------------------------------------------------------------
% From Cleve Moler's Matlab suite
% http://www.mathworks.it/moler/ncmfilelist.html
%--------------------------------------------------------------------------
% x = TRIDISOLVE(a,b,c,d) solves the system of linear equations
%
%     b(1)*x(1) + c(1)*x(2) = d(1),
%     a(j-1)*x(j-1) + b(j)*x(j) + c(j)*x(j+1) = d(j), j = 2:n-1,
%     a(n-1)*x(n-1) + b(n)*x(n) = d(n).
%
% The algorithm does not use pivoting, so the results might be inaccurate
% if abs(b) is much smaller than abs(a)+abs(c).
%
% More robust, but slower, alternatives with pivoting are:
%     x = T\d where T = diag(a,-1) + diag(b,0) + diag(c,1)
%     x = S\d where S = spdiags([[a; 0] b [0; c]],[-1 0 1],n,n)
%--------------------------------------------------------------------------
% Optimized version by G. Meurant.
%--------------------------------------------------------------------------

x = d;
n = length(x);
bi = zeros(n,1);

for j = 1:n-1
    bi(j) = 1 / b(j);
    mu = a(j) * bi(j);
    b(j+1) = b(j+1) - mu * c(j);
    x(j+1) = x(j+1) - mu * x(j);
end

x(n) = x(n) / b(n);
for j = n-1:-1:1
    x(j) = (x(j) - c(j) * x(j+1)) * bi(j);
end






function ab=fast_chebyshev(N,mom,abm)

%--------------------------------------------------------------------------
% Object:
% Modified Chebyshev algorithm, that works only for the subperiodic weight
% function.
%--------------------------------------------------------------------------
% From Gautschi's code (simplified)
% Mar 2012
%--------------------------------------------------------------------------
% Optimized version by G. Meurant.
%--------------------------------------------------------------------------

ab = zeros(N,2);
sig = zeros(N+1,2*N);

ab(1,2) = mom(1);

sig(1,1:2*N) = 0;
sig(2,:) = mom(1:2*N);

for n = 3:N+1
    for m = n-1:2*N-n+2
        sig(n,m) = sig(n-1,m+1) + abm(m,2) * sig(n-1,m-1) - ...
            ab(n-2,2) * sig(n-2,m);
    end

    ab(n-1,2) = sig(n,n-1) / sig(n-1,n-2);
end









function mom=fast_moments_computation(omega,n)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Inputs:
%--------------------------------------------------------------------------
% Outputs:
%--------------------------------------------------------------------------
% Authors G. Meurant and A. Sommariva
% June 2012
%--------------------------------------------------------------------------

mom=zeros(1,n+1);
mom(1)=2*omega; % FIRST MOMENT.

if(n>=2)

    if(omega<=1/4*pi)
        l=10;
    elseif(omega<=1/2*pi)
        l=20;
    elseif(omega<=3/4*pi)
        l=40;
    else
        if omega == pi
            l=2*ceil(10*pi);
        else
            l=2*ceil(10*pi/(pi-omega));
        end
    end


    temp=(2:2:n+2*l-2); % AUXILIAR VECTORS.
    temp2=temp.^2-1;

    dl=1/4 -1./(4*(temp-1)); % DIAGONALS.
    dc=1/2 -1/sin(omega/2)^2 -1./(2*temp2);
    du=1/4 +1./(4*(temp+1));

    d=4*cos(omega/2)/sin(omega/2)./temp2'; % COMPUTING KNOWN TERM.
    d(end)=d(end);                         % PUT LAST MOMENT NULL

    z=tridisolve(dl(2:end),dc,du(1:end-1),d); % SOLVE SYSTEM.
    mom(3:2:n+1)=z(1:floor(n/2)); % SET ODD MOMENTS.

end

mom=mom';

normalized = 0;

if normalized == 0
    M=length(mom);
    kk=2.^(-((1:2:M)-2))'; kk(1)=1;
    v=ones(M,1);
    v(1:2:M)=kk;
    mom=v.*mom;
end









function xw=SymmMw(N,ab)

%--------------------------------------------------------------------------
% Object:
% Computation of the nodes and weights for a symmetric weight function
% this version uses the reduced matrix and eig and computation of weights
% with the 3-term recurrence.
%--------------------------------------------------------------------------
% Input:
% N : cardinality of the rule
% ab: 3-term recurrence for the orthogonal polynomials same as in OPQ
%     ab(1,2) is the 0th moment.
%--------------------------------------------------------------------------
% Output:
% xw : xw(:,1) nodes, xw(:,2) weights of the quadrature rule
%--------------------------------------------------------------------------
% Reference paper:
% Fast variants of the Golub and Welsch algorithm for symmetric
% weight functions by G. Meurant and A. Sommariva (2012)
%--------------------------------------------------------------------------
% Data:
% Written by G. Meurant and A. Sommariva on June 2012
%--------------------------------------------------------------------------

N0 = size(ab,1);
if N0 < N
    error('SymmMw: input array ab is too short')
end

na = norm(ab(:,1));
if na > 0
    error('SymmMw: the weight function must be symmetric')
end

% computation of the reduced matrix in vectors (a,b)

if mod(N,2) == 0
    even = 1;
    Nc = N / 2;
else
    even = 0;
    Nc = fix(N / 2) +1;
end


absd = ab(:,2);
absq = sqrt(absd);

a = zeros(1,Nc);
b = a;

switch even
    case 1
        % N even
        a(1) = absd(2);
        b(1) = absq(2) * absq(3);

        k = 2:Nc-1;
        a(k) = absd(2*k-1) + absd(2*k);
        b(k) = absq(2*k) .* absq(2*k+1);
        a(Nc) = absd(N) + absd(N-1);
        start = 1;

        J = diag(a) + diag(b(1:Nc-1),1) + diag(b(1:Nc-1),-1);
        t = sort(eig(J));
        w = weights_3t(t',a,b);
        % w are the squares of the first components
        w = w' / 2;
    case 0
        % N odd
        a(1) = absd(2);
        b(1) = absq(2) * absq(3);

        k = 2:Nc-1;
        a(k) = absd(2*k-1) + absd(2*k);
        b(k) = absq(2*k) .* absq(2*k+1);
        a(Nc) = absd(N);
        start = 2;

        % the first node must be zero
        J = diag(a) + diag(b(1:Nc-1),1) + diag(b(1:Nc-1),-1);
        t = sort(eig(J));
        t(1) = 0;
        w = weights_3t(t',a,b);
        w = [w(1); w(2:end)' / 2];
    otherwise
        error('this is not possible')
end

xwp = sqrt(t);

xw(:,1) = [-xwp(end:-1:start,1); xwp];
xw(:,2) = ab(1,2) * ([w(end:-1:start); w]);









function tw=quad_trig_conversion(xw,omega)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Inputs:
%--------------------------------------------------------------------------
% Outputs:
%--------------------------------------------------------------------------
% Authors G. Meurant and A. Sommariva
% June 2012
%--------------------------------------------------------------------------

tw(:,1)=2*asin(sin(omega/2)*xw(:,1));
tw(:,2)=xw(:,2);









function w=weights_3t(t,a,b)

%--------------------------------------------------------------------------
% Object:
% Squares of the 1st components of eigenvectors from the 3-term
% recurrence relation of the orthogonal polynomials
%--------------------------------------------------------------------------
% Inputs:
% t: nodes
% a,b: coefficients of the 3-term recurrence
%--------------------------------------------------------------------------
% Outputs
% w: squares of the first components of the eigenvectors
%--------------------------------------------------------------------------
% Authors G. Meurant and A. Sommariva
% June 2012
%--------------------------------------------------------------------------

N = length(t);

P = zeros(N,N);
P(1,:) = ones(1,N);
P(2,:) = (t - a(1)) / b(1);

for k = 3:N
    k1 = k - 1;
    k2 = k - 2;
    P(k,:) = ((t - a(k1)) .* P(k1,:) - b(k2) * P(k2,:)) / b(k1);
end

P2 = P .* P;
w = 1 ./ sum(P2);










% GAUSS Gauss quadrature rule.
%
%    Given a weight function w encoded by the nx2 array ab of the
%    first n recurrence coefficients for the associated orthogonal
%    polynomials, the first column of ab containing the n alpha-
%    coefficients and the second column the n beta-coefficients,
%    the call xw=GAUSS(n,ab) generates the nodes and weights xw of
%    the n-point Gauss quadrature rule for the weight function w.
%    The nodes, in increasing order, are stored in the first
%    column, the n corresponding weights in the second column, of
%    the nx2 array xw.
%
function xw=gauss(N,ab)
N0=size(ab,1); if N0<N, error('input array ab too short'), end
J=zeros(N);
for n=1:N, J(n,n)=ab(n,1); end
for n=2:N
    J(n,n-1)=sqrt(ab(n,2));
    J(n-1,n)=J(n,n-1);
end
[V,D]=eig(J);
[D,I]=sort(diag(D));
V=V(:,I);
xw=[D ab(1,2)*V(1,:)'.^2];









function ab=r_jacobi(N,a,b)

nu=(b-a)/(a+b+2);
mu=2^(a+b+1)*gamma(a+1)*gamma(b+1)/gamma(a+b+2);
if N==1
    ab=[nu mu]; return
end

N=N-1;
n=1:N;
nab=2*n+a+b;
nuadd=(b^2-a^2)*ones(1,N)./(nab.*(nab+2));
A=[nu nuadd];
n=2:N;
nab=nab(n);
B1=4*(a+1)*(b+1)/((a+b+2)^2*(a+b+3));
B=4*(n+a).*(n+b).*n.*(n+a+b)./((nab.^2).*(nab+1).*(nab-1));
abadd=[mu; B1; B'];
ab=[A' abadd];




















