
function demo_cub_sphpgon

%--------------------------------------------------------------------------
% Demo of cubature over spherical polygons, in which the domains are
% coarse approximation of continents.
%--------------------------------------------------------------------------
%% Copyright (C) 2021-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date: JULY 13, 2021
%% Date: DECEMBER 29, 2025
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Define spherical polygon. Input values in "define_domain":
%                 0: Polygonal cap at North-Pole.
%                 1: Cardioid at North-Pole.
%                 2. Coarse Australia (no Tasmania, via data).
%--------------------------------------------------------------------------
domain_type=2;

%--------------------------------------------------------------------------
% Define algebraic degree of precision.
%--------------------------------------------------------------------------
n=10;

%--------------------------------------------------------------------------
% Define function to test.
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%--------------------------------------------------------------------------
function_type=2;



% .................. Numerical code starts here ........................... 

% 1. Determine rule over sph. polygon with algebraic degree of precision.

vertices=define_domain(domain_type);

fprintf('\n \t * Determine full rule: ');
tic; XW=cub_sphpgon(n,vertices); t(1)=toc; 
xF=XW(:,1); yF=XW(:,2); zF=XW(:,3); wF=XW(:,end);
fprintf(' cputime %1.1e',t(1));


% 2. Caratheodory-Tchakaloff compression
fprintf('\n \t * Determine compressed rule: ');
tic; 
[nodesC,wC,momerr]=compress_cub_sphpoly(n,[xF yF zF],wF); 
t(2)=toc;
XWC=[nodesC wC];
xC=XWC(:,1); yC=XWC(:,2); zC=XWC(:,3); wC=XWC(:,end);
fprintf(' cputime %1.1e',t(2));

% 3. Test integral quality on a function vs adaptive code.
[f,fs]=define_function(function_type,vertices);

% 3a. Integration by "full" rule.
fnodesF=feval(f,xF,yF,zF);
IF=wF'*fnodesF;

% 3b. Integration by "compressed" rule
fnodesC=feval(f,xC,yC,zC);
IC=wC'*fnodesC;


% .............................. Plots ....................................

% Plot domain and nodes.


% Figure 1. 3D plot of cubature nodes (full and compressed)
figure(1)
plot_s2('sphere',[],XW(:,1:3),XWC(:,1:3));





% .............................. Stats ....................................
fprintf('\n \n \t ADE: %6.0f',n);
fprintf('\n \t Full rule cardinality: %6.0f',size(XW,1));
fprintf('\n \t Comp rule cardinality: %6.0f',size(XWC,1));
fprintf('\n \t Moment matching      : %1.1e',momerr);
fprintf('\n \n \t Integrand: '); disp(fs);
fprintf('\n \n \t I full: %1.15e',IF);
fprintf('\n \t I cmpr: %1.15e',IC);
fprintf('\n \n');




















function [f,fs]=define_function(function_type,parms)

%--------------------------------------------------------------------------
% Object:
% This routine, defines a function "f" to approximate and a string "fs" for
% possible messages to the user.
%--------------------------------------------------------------------------
% Input:
% function_type: determines the function to study.
% The first five functions has been used in the paper mentioned below.
%
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%--------------------------------------------------------------------------
% Output:
% f: defines a function "f" to approximate;
% fs: string with the content of the function "f".
%--------------------------------------------------------------------------
% Reference paper:
% A. Sommariva and M. Vianello
% Numerical hyperinterpolation overspherical triangles
%--------------------------------------------------------------------------

switch function_type
    case 1 % Fornberg
        f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
        fs='1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2';
        
    case 2
        f=@(x,y,z) cos(10*(x+y+z));
        fs='cos(10*(x+y+z))';
        
    case 3 % exp and - square north pole distance
        x0=0; y0=0; z0=1;
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2, x0=0;y0=0;z0=1;';
        
    case 4 % exp and - square centroid distance
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        ok_vertices=not(isnan(vertices(:,1))) & not(isnan(vertices(:,2)));
        vertices=vertices(ok_vertices,:);
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
    case 5 % north pole distance like
        x0=0; y0=0; z0=1;
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), x0=0; y0=0; z0=1;';
        
    case 6 % centroid distance like
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        ok_vertices=not(isnan(vertices(:,1))) & not(isnan(vertices(:,2)));
        vertices=vertices(ok_vertices,:);
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), ';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
end  % end: "define_function"









function [vertices,domain_str]=define_domain(example)

%--------------------------------------------------------------------------
% Object:
% This routine, defines the domain to analyse.
%--------------------------------------------------------------------------
% Input:
% example: determines the spherical triangle domain to be analysed. From 0
%   to 7, going from large domains to tiny ones (the higher "example", the
%   smaller the domain.
%--------------------------------------------------------------------------
% Output:
% vertices: it is a matrix whose rows are the vertices of the spherical
%           triangles.
% domain_str: string that stores the geometry of the region.
%--------------------------------------------------------------------------


switch example
    
    case 0
        % Define spherical polygon.
        factor=0.5;
        t=linspace(0,2*pi,10); t=(t(1:end-1))';
        XV=factor*cos(t); YV=factor*sin(t); ZV=sqrt(1-XV.^2-YV.^2);
        % Note: spherical polygon (first vertices not equal to last).
        vertices=[XV YV ZV];
        
        domain_str='Polygonal cap at North-Pole';
        
    case 1
        
        spec_settings=32;
        th=linspace(0,2*pi,spec_settings+1);
        %th=(th(1:end-1))';
        polygon_sides=[cos(th').*(1-cos(th')) sin(th').*(1-cos(th'))];
        polygon_sides=polygon_sides(1:end-1,:);
        XV=polygon_sides(:,1); XV=XV/2.1;
        YV=polygon_sides(:,2); YV=YV/2.1;
        ZV=sqrt(1-XV.^2-YV.^2);
        vertices=[XV YV ZV];
        
        domain_str='Polygonal cardioid at North-Pole';
        
    case 2
        
        % Australia island vertices in degrees.
        % Column 1: Longitude. Column 2: Latitude.
        vertices_degs=100*[1.462300280000000  -0.386970830000000
            1.451162780000000  -0.381483330000000
            1.396121110000000  -0.361567220000000
            1.385678610000000  -0.348266670000000
            1.374529440000000  -0.349083330000000
            1.359337220000000  -0.345341940000000
            1.340596110000000  -0.329141390000000
            1.235266670000000  -0.339379720000000
            1.150004170000000  -0.335283610000000
            1.131528610000000  -0.261491940000000
            1.142312220000000  -0.263158890000000
            1.168041670000000  -0.205237780000000
            1.221729170000000  -0.172633330000000
            1.239712780000000  -0.168258330000000
            1.243913060000000  -0.163392220000000
            1.252495830000000  -0.155816670000000
            1.259183610000000  -0.146770830000000
            1.268749720000000  -0.137471390000000
            1.283721390000000  -0.154966940000000
            1.292900000000000  -0.148620830000000
            1.303179720000000  -0.133733330000000
            1.308121390000000  -0.124083330000000
            1.317678890000000  -0.113258330000000
            1.347483890000000  -0.119512500000000
            1.360166940000000  -0.124987780000000
            1.359095560000000  -0.132850000000000
            1.395983610000000  -0.175387780000000
            1.415862220000000  -0.125583610000000
            1.437729170000000  -0.143991940000000
            1.462604440000000  -0.188633890000000
            1.500416670000000  -0.221254170000000
            1.531929440000000  -0.259316940000000
            1.525429440000000  -0.324442220000000
            1.499796110000000  -0.375050560000000
            1.462300280000000  -0.386970830000000];
        
        % Australia island vertices in radians.
        longitudes=deg2rad(vertices_degs(:,1));
        latitudes=deg2rad(vertices_degs(:,2));
        
        % Australia island vertices in cartesian coordinates.
        [XX,YY,ZZ] = sph2cart(longitudes,latitudes,1);
        vertices=[XX YY ZZ];
        
        domain_str='Australia as island (33 vertices, via data)';
       
      case 3
          
                  
        % Australia island (no Tasmania), vertices in degrees.
        % Column 1: Longitude. Column 2: Latitude.
        % Planar domain in polyshape form.
        australia_pshape=coastline_australia(0);
        vertices_degs=australia_pshape.Vertices;
        
        % Australia & Tasmania vertices in radians.
        longitudes=deg2rad(vertices_degs(:,1));
        latitudes=deg2rad(vertices_degs(:,2));
        
        % Australia & Tasmania vertices in cartesian coordinates.
        [XX,YY,ZZ] = sph2cart(longitudes,latitudes,1);
        vertices=[XX YY ZZ];
        
        domain_str='Australia as island (170 vertices, via polyshape)';
        
        
        
end










function australia_pshape=coastline_australia(flag)

%--------------------------------------------------------------------------
% Polyshape object that is a coarse approximation of Australia (including
% Tasmania).
%--------------------------------------------------------------------------
% INPUT:
% flag: 0: Australia Island 1: Australia and Tasmania
%--------------------------------------------------------------------------
% Information:
% Authors: A. Sommariva, M. Dessole and M. Vianello.
% Date: June 7, 2021.
%--------------------------------------------------------------------------

% In vertices we store, respectively, Longitude and Latitute, in degrees.
% Column 1: Longitude. Column 2: Latitude.

australia0_coord=[
    146.230028000000004 -38.697082999999999
    146.276221999999990 -39.000000000000000
    145.371638999999988 -38.539611000000001
    145.488305999999994 -38.235444000000001
    144.658360999999985 -38.311278000000001
    145.116277999999994 -38.148333000000001
    144.928305999999992 -37.842917000000000
    143.556638999999990 -38.858750000000001
    140.581666999999982 -38.032916999999998
    139.740361000000007 -37.183306000000002
    139.612110999999999 -36.156721999999995
    139.722471999999982 -36.288806000000001
    139.517555999999985 -35.961250000000000
    138.890805999999998 -35.533749999999998
    138.097860999999995 -35.626694000000001
    138.567860999999994 -34.826667000000000
    138.094193999999987 -34.135416999999997
    137.762555999999989 -35.117944000000001
    136.851693999999981 -35.285416999999995
    137.015832999999986 -34.895443999999998
    137.452944000000002 -34.908332999999999
    137.450444000000005 -34.140028000000001
    137.977916999999991 -33.553360999999995
    137.754138999999981 -32.458777999999995
    137.781000000000006 -33.000000000000000
    135.933721999999989 -34.534193999999999
    135.957499999999982 -35.007916999999999
    135.112888999999996 -34.589999999999996
    135.518749999999983 -34.614193999999998
    134.707916999999981 -33.181666999999997
    134.059610999999990 -32.914138999999999
    134.181667000000004 -32.485416999999998
    131.152471999999989 -31.464582999999998
    125.957499999999996 -32.288778000000001
    124.234138999999999 -33.019582999999997
    123.526666999999989 -33.937971999999995
    119.903360999999990 -33.934582999999996
    117.953305999999998 -35.128777999999997
    116.624139000000000 -35.057943999999999
    115.128305999999995 -34.372943999999997
    115.000416999999999 -33.528360999999997
    115.697916999999990 -33.300860999999998
    115.882943999999995 -31.961693999999998
    115.052110999999996 -30.507527999999997
    114.867055999999991 -29.114193999999998
    113.152861000000001 -26.149193999999998
    113.837916999999990 -26.591666999999998
    113.509110999999990 -25.505416999999998
    113.722082999999998 -26.200028000000000
    113.870806000000002 -25.942083000000000
    114.231221999999988 -26.315888999999999
    113.396249999999995 -24.406693999999998
    113.990416999999994 -21.872527999999999
    114.330027999999999 -22.522138999999999
    114.645832999999996 -21.837916999999997
    116.804166999999993 -20.523778000000000
    117.377471999999997 -20.777082999999998
    119.092472000000001 -19.957916999999998
    121.113305999999994 -19.539611000000001
    122.369555999999989 -18.117500000000000
    122.172916999999998 -17.263332999999999
    122.922528000000000 -16.387944000000001
    123.566721999999999 -17.627082999999999
    123.569527999999991 -17.037555999999999
    123.930805999999990 -17.267917000000001
    123.971277999999998 -16.825832999999999
    123.504582999999997 -16.662499999999998
    123.557499999999990 -16.173749999999998
    124.400027999999992 -16.565443999999999
    124.976249999999993 -16.381722000000000
    124.391306000000000 -16.339221999999999
    124.750388999999998 -15.810889000000000
    124.494999999999990 -16.002972000000000
    124.360416999999998 -15.670027999999999
    124.675027999999998 -15.254582999999998
    125.249583000000001 -15.581666999999999
    124.829555999999997 -15.157471999999999
    125.506693999999996 -15.172056000000000
    125.137889000000001 -14.745832999999999
    125.614166999999995 -14.230471999999999
    125.918360999999990 -14.677083000000000
    126.009166999999991 -13.920389000000000
    126.230027999999990 -14.250416999999999
    126.535805999999994 -13.932082999999999
    126.612527999999998 -14.250416999999999
    126.874972000000000 -13.747138999999999
    127.428332999999995 -13.942943999999999
    128.227056000000005 -14.714193999999999
    128.009972000000005 -15.513750000000000
    128.128332999999998 -15.186249999999999
    128.372139000000004 -15.496694000000000
    128.199582999999990 -15.069222000000000
    128.559138999999988 -14.766278000000000
    129.092527999999987 -14.904610999999999
    129.125860999999986 -15.281277999999999
    129.289999999999992 -14.862083000000000
    129.605832999999990 -15.206249999999999
    129.998777999999987 -14.705860999999999
    129.365417000000008 -14.339971999999999
    129.816638999999981 -13.499582999999999
    130.317971999999997 -13.373332999999999
    130.351222000000007 -12.670028000000000
    130.732082999999989 -12.728332999999999
    130.575832999999989 -12.407110999999999
    130.964971999999989 -12.671277999999999
    130.812139000000002 -12.408332999999999
    131.287527999999980 -12.045444000000000
    131.462500000000006 -12.285444000000000
    132.762944000000005 -12.154582999999999
    132.722971999999999 -11.623332999999999
    131.767888999999997 -11.325832999999999
    131.973332999999997 -11.127110999999999
    133.508332999999993 -11.879583000000000
    133.915832999999992 -11.742082999999999
    134.150889000000006 -12.174583000000000
    134.748389000000003 -11.951250000000000
    135.215388999999988 -12.300860999999999
    135.912499999999994 -11.952916999999999
    135.652888999999988 -12.206666999999999
    136.044999999999987 -12.064610999999999
    136.016694000000001 -12.498778000000000
    136.564138999999983 -11.879610999999999
    136.977971999999994 -12.348333000000000
    136.462083000000007 -12.777500000000000
    136.461667000000006 -13.253777999999999
    135.909555999999981 -13.285000000000000
    136.072111000000007 -13.667527999999999
    135.411221999999981 -14.934972000000000
    136.676666999999981 -15.926305999999999
    139.036638999999980 -16.912917000000000
    139.598360999999983 -17.538778000000001
    140.609166999999985 -17.613806000000000
    141.667027999999988 -15.040861000000000
    141.461250000000007 -13.852528000000000
    141.976832999999999 -12.594778000000000
    141.586221999999992 -12.558361000000000
    142.079611000000000 -11.985861000000000
    142.130388999999980 -10.961694000000000
    142.534166999999997 -10.688722000000000
    143.545416999999986 -12.844166999999999
    143.772917000000007 -14.399194000000000
    144.515055999999987 -14.166250000000000
    145.347888999999981 -14.949193999999999
    145.402139000000005 -16.434999999999999
    145.957888999999994 -16.897527999999998
    146.260444000000007 -18.863388999999998
    148.772943999999995 -20.236694000000000
    149.630471999999997 -22.584999999999997
    149.812500000000000 -22.382110999999998
    150.060417000000001 -22.658360999999999
    150.041666999999990 -22.125416999999999
    150.635861000000006 -22.660443999999998
    150.668306000000001 -22.347082999999998
    150.791666999999990 -23.509611000000000
    151.769555999999994 -24.020861000000000
    153.192943999999983 -25.931694000000000
    153.024583000000007 -27.294193999999997
    153.639582999999988 -28.638361000000000
    153.065416999999997 -31.058388999999998
    152.359555999999998 -32.186667000000000
    152.542944000000006 -32.444221999999996
    151.201222000000001 -33.510832999999998
    150.602943999999979 -34.865832999999995
    150.842943999999989 -35.066693999999998
    150.152972000000005 -35.700972000000000
    149.979611000000006 -37.505055999999996
    147.657471999999984 -37.850443999999996
    147.454138999999998 -38.079611000000000
    147.972082999999998 -37.893360999999999
    146.230028000000004 -38.697082999999999
    ];

if flag == 0
    australia_pshape=polyshape(australia0_coord); return;
end


tasmania_coord=100*[
    1.459990830000000  -0.433303890000000
    1.460000000000000  -0.433303610000000
    1.452553890000000  -0.426241390000000
    1.451683060000000  -0.421962500000000
    1.455553890000000  -0.423575560000000
    1.447437500000000  -0.414216940000000
    1.446837220000000  -0.406683610000000
    1.466066940000000  -0.412604170000000
    1.480775560000000  -0.407704440000000
    1.483611940000000  -0.421975000000000
    1.482066390000000  -0.419504440000000
    1.480062780000000  -0.432308610000000
    1.471966670000000  -0.427279170000000
    1.468699720000000  -0.436404440000000
    1.459990830000000  -0.433303890000000];


australia0_pshape=polyshape(australia0_coord);
tasmania_pshape=polyshape(tasmania_coord);
australia_pshape=union(australia0_pshape,tasmania_pshape);






function plot_s2(domain_type,x,pts1,pts2,title_str,R,tri_vertices,...
    tri_conn_list)

%--------------------------------------------------------------------------
% Object:
% Plotting pointsets on regions of the sphere.
%--------------------------------------------------------------------------
% Input:
% domain_type: 'sphere','spherical-rectangle','spherical-triangle',
%       'spherical-caps', 'spherical-polygon'.
%
% x     : * spherical rectangle representation / spherical representation:
%         It is a 2 x 2 matrix, containing all (theta_1,theta2) such that
%         theta_1 in (x(1,1),x(1,2)) and theta_2 in (x(2,1),x(2,2)), and
%         the points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%         * spherical triangle representation:
%         It is a 3 x 3 matrix, where the k-th row contains the cartesian
%         coordinates of the k-th vertex of the domain.
%         * spherical polygon representation:
%         It is a N x 3 matrix, where the k-th row contains the cartesian
%         coordinates of the k-th vertex of the domain.
%         
% pts1: spherical/cartesian coordinates of first set of points on the unit
%      sphere possibly belonging to the spherical rectangle/triangle
%      defined by "x".
%      The k-th row of this matrix determines the k-th point.
%         The points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%
% pts2: spherical/cartesian coordinates of first set of points on the unit
%      sphere possibly belonging to the spherical rectangle/triangle
%      defined by "x".
%      The k-th row of this matrix determines the k-th point.
%      The points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%
% title_str: string with the title of the domain
%
% R: sphere radius
%
% tri_vertices,tri_conn_list: triangulation properties (if available)
%--------------------------------------------------------------------------
% Dates:
% Written on 20/11/2020: A. Sommariva and M. Vianello;
%
% Modified on:
% 28/11/2020: A. Sommariva.
%--------------------------------------------------------------------------


% ......................... troubleshooting ...............................

if nargin < 2, x=[]; end
if nargin < 3, pts1=[]; end
if nargin < 4, pts2=[]; end
if nargin < 5, title_str=''; end
if nargin < 6, R=1; end
if nargin < 7, tri_vertices=[]; end
if nargin < 8, tri_conn_list=[]; end

if isempty(x), x=[0 pi; 0 2*pi]; end
if isempty(pts1), pts1=[]; end
if isempty(pts2), pts2=[]; end
if isempty(title_str), title_str=''; end
if isempty(R), R=1; end


% ......................... plot sphere ...................................

% ... plot transparent sphere ...
axis tight;
pts_size1=1; pts_size2=6;
% gray: [0.9000    0.9000    0.9000]
color1='black'; color2='magenta';

% ......................... plot pointset 1 ...............................
if isempty(pts1) == 0, plot_set(pts1,color1,'*',pts_size1); end

hold on;

% ......................... plot pointset 2 ...............................
if isempty(pts2) == 0, plot_set(pts2,color2,'o',pts_size2); end

switch domain_type
    case 'spherical-caps'
        % fprintf('\n \t Plot cap \n \n');
        theta_intv=x(1,:);
        funx = @(theta,phi) sin(theta).*cos(phi);
        funy = @(theta,phi) sin(theta).*sin(phi);
        funz = @(theta,phi) cos(theta);
        fsurf(funx,funy,funz,[theta_intv(1) theta_intv(2) -pi pi])
        alpha 0.1
        
    case 'spherical-triangle'
        
        % fprintf('\n \t Plot cap (sph-tri) \n \n');
        
        % .... plot cap ....
        
        theta(1)=acos(x(1,3)/norm(x(1,:)));
        theta(2)=acos(x(2,3)/norm(x(2,:)));
        theta(3)=acos(x(3,3)/norm(x(3,:)));
        theta_intv=[0 1.4*max(theta)];
        
        funx = @(theta,phi) sin(theta).*cos(phi);
        funy = @(theta,phi) sin(theta).*sin(phi);
        funz = @(theta,phi) cos(theta);
        fsurf(funx,funy,funz,[theta_intv(1) theta_intv(2) -pi pi]);
        alpha 0.1 % the lower, the more transparent the region
        
        % .... plot sph. triangle boundary ....
        A=x(1,:); B=x(2,:); C=x(3,:);
        plot_side(A,B); plot_side(B,C); plot_side(C,A);
        
        % .... hide ticks, labels, axis and background ....
        
        set(gca,'xtick',[]) % no ticks / labels
        set(gca,'xticklabel',[])
        set(gca,'ytick',[])
        set(gca,'yticklabel',[])
        set(gca,'ztick',[])
        set(gca,'zticklabel',[])
        
        set(gca,'color','none') % no background
        
        set(gca,'Visible','off') % no axis
        
    case 'spherical-polygon'
        
        % plot sphere
        [X,Y,Z] = sphere(20);
        XX=x(:,1); YY=x(:,2); ZZ=x(:,3);
        R=max(sqrt(XX.^2+YY.^2+ZZ.^2));
        plot3(R*X,R*Y,R*Z,'MarkerFaceColor','red');
        
        for k=1:size(tri_conn_list,1)
            verticesL=tri_vertices((tri_conn_list(k,:))',:);
            P1=verticesL(1,:); P2=verticesL(2,:); P3=verticesL(3,:);
            plot_side(P1,P2); plot_side(P2,P3); plot_side(P3,P1);
            hold on;
        end
        
    otherwise
        % fprintf('\n \t Plot sphere \n \n');
        [X,Y,Z] = sphere(20);
        plot3(R*X,R*Y,R*Z,'MarkerFaceColor','red');
end

hold on;




% ......................... adding title ..................................
title(title_str);
hold off;









function plot_set(pts,color_str,marker_type,marker_size)

%--------------------------------------------------------------------------
% Object:
% Plotting pointsets on regions of the sphere.
%--------------------------------------------------------------------------
% Input:
% pts: spherical or cartesian coordinates of a pointset on the
%       unit sphere.
%      The k-th row of this matrix determines the k-th point.
%      The points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%
% color_str: color of the pointset, e.g. color_str='magenta';
% marker_type: marker type of the pointset, e.g. marker_type='.';
% marker_size: marker size of the pointset, e.g. marker_size=10.
%--------------------------------------------------------------------------
% Dates:
% Written on 20/11/2020: A. Sommariva and M. Vianello;
%
% Modified on:
% 28/11/2020: A. Sommariva.
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, pts=[]; end
if nargin < 2, color_str='magenta'; end
if nargin < 3, marker_type='o'; end
if nargin < 4, marker_size=10; end

if isempty(marker_type), marker_type='*'; end
if isempty(marker_size), marker_size=10; end


% ......................... plotting points ...............................

if size(pts,2) == 2 % spherical coordinates
    theta_1=pts(:,1); theta_2=pts(:,2);
    s1=sin(theta_1); c1=cos(theta_1); s2=sin(theta_2); c2=cos(theta_2);
    xx=s1.*c2; yy=s1.*s2; zz=c1;
else % cartesian coordinates
    xx=pts(:,1); yy=pts(:,2); zz=pts(:,3);
end

plot3(xx,yy,zz,marker_type,'Color',color_str,'MarkerSize',marker_size,...
    'MarkerEdgeColor',color_str,'MarkerFaceColor',color_str);








function plot_arrow(p1,p2,str)

%--------------------------------------------------------------------------
% Object:
% Plot a line joining the row vectors "p1" and "p2" with an arrow.
%--------------------------------------------------------------------------

dp = p2-p1;
quiver3(p1(1),p1(2),p1(3),dp(1),dp(2),dp(3),0,'LineWidth',2)
text(p1(1),p1(2),str)








function plot_side(A,B)

%--------------------------------------------------------------------------
% Object:
% Plot "geodesic line" between A and B.
%--------------------------------------------------------------------------
% Input:
% A,B: row vectors of two points of the sphere to be joined.
%--------------------------------------------------------------------------

R=norm(A);
t=linspace(0,1,10);
C=[];
for k=1:length(t)
    CC=A+t(k)*(B-A); CC=CC/norm(CC); C=[C; R*CC];
end
plot3(C(:,1),C(:,2),C(:,3),'k-','LineWidth',1);






function fill_sphtri(P1,P2,P3)

%--------------------------------------------------------------------------
% Object:
% Fill regions of the sphere.
%--------------------------------------------------------------------------
% Input:
% P1,P2,P3: row vectors.
%--------------------------------------------------------------------------

R=norm(P1);
N=10000;
vals=rand(N,3);
vals=vals./sum(vals,2);
P=[];
for k=1:N
    val=vals(k,:);
    PL=val(1)*P1+val(2)*P2+val(3)*P3;
    PL=R*PL/norm(PL);
    P=[P; PL];
end

% color_str='white';
plot3(P(:,1),P(:,2),P(:,3),'y*','Color',color_str,'MarkerSize',3,...
    'MarkerEdgeColor',color_str,'MarkerFaceColor',color_str);





