
function demo_cub_sphpgon_adaptive

%--------------------------------------------------------------------------
% Demo of cubature over spherical polygons.
% 
% Adaptive cubature over spherical polygons contained in a cap whose
% polar angle is strictly inferior than π is considered.
%
% The spherical polygon is on the sphere centered in the origin, whose 
% radius is defined by the "vertices" of the polygon.
%
% A comparison between a rule with fixed algebraic degree of precision and 
% the adaptive code is given.
%--------------------------------------------------------------------------
%% Copyright (C) 2021-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date: JULY 13, 2021
%% Date: DECEMBER 29, 2025
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Define spherical polygon. Input values in "define_domain":
%                 0: Polygonal cap at North-Pole.
%                 1: Cardioid at North-Pole.
%                 2. Coarse Australia (no Tasmania, via data).
%--------------------------------------------------------------------------
domain_type=0;

%--------------------------------------------------------------------------
% Define algebraic degree of precision.
%--------------------------------------------------------------------------
n=10;

%--------------------------------------------------------------------------
% Define function to test.
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%--------------------------------------------------------------------------
function_type=5;



% .................. Numerical code starts here ........................... 

% 1. Determine rule over sph. polygon with algebraic degree of precision.

vertices=define_domain(domain_type);

fprintf('\n \t * Determine full rule: ');
tic; XW=cub_sphpgon(n,vertices); t(1)=toc; 
xF=XW(:,1); yF=XW(:,2); zF=XW(:,3); wF=XW(:,4);
fprintf(' cputime %1.1e',t(1));

% 2. Test integral quality on a function vs adaptive code.
[f,fs]=define_function(function_type,vertices);

% 3a. Integration by rule.
fnodes=feval(f,xF,yF,zF);
IF=wF'*fnodes;

% 3b. Integration by adaptive code.
fprintf('\n \t * Adaptive code: ');
tic; 
tol=10^(-12);
[I,Ierr]=cub_sphpgon_adaptive(vertices,f,tol);
t(3)=toc; fprintf(' cputime %1.1e',t(3));








% .............................. Stats ....................................
fprintf('\n \n \t ADE: %6.0f',n);
fprintf('\n \t Full rule cardinality: %6.0f',size(XW,1));
fprintf('\n \n \t Integrand: '); disp(fs);
fprintf('\n \t Cubature rule       : %1.15e',IF);
fprintf('\n \t Adaptive code       : %1.15e',I);
fprintf('\n \n \t Tolerance           : %1.3e',tol);
fprintf('\n \t Absolute error (est): %1.3e',Ierr);
fprintf('\n \n');




















function [f,fs]=define_function(function_type,parms)

%--------------------------------------------------------------------------
% Object:
% This routine, defines a function "f" to approximate and a string "fs" for
% possible messages to the user.
%--------------------------------------------------------------------------
% Input:
% function_type: determines the function to study.
% The first five functions has been used in the paper mentioned below.
%
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%         f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%         x0=centroid(1); y0=centroid(2); z0=centroid(3);
%         f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%--------------------------------------------------------------------------
% Output:
% f: defines a function "f" to approximate;
% fs: string with the content of the function "f".
%--------------------------------------------------------------------------
% Reference paper:
% A. Sommariva and M. Vianello
% Numerical hyperinterpolation overspherical triangles
%--------------------------------------------------------------------------

switch function_type
    case 1 % Fornberg
        f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
        fs='1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2';
        
    case 2
        f=@(x,y,z) cos(10*(x+y+z));
        fs='cos(10*(x+y+z))';
        
    case 3 % exp and - square north pole distance
        x0=0; y0=0; z0=1;
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2, x0=0;y0=0;z0=1;';
        
    case 4 % exp and - square centroid distance
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        ok_vertices=not(isnan(vertices(:,1))) & not(isnan(vertices(:,2)));
        vertices=vertices(ok_vertices,:);
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
    case 5 % north pole distance like
        x0=0; y0=0; z0=1;
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), x0=0; y0=0; z0=1;';
        
    case 6 % centroid distance like
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        ok_vertices=not(isnan(vertices(:,1))) & not(isnan(vertices(:,2)));
        vertices=vertices(ok_vertices,:);
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), ';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
end  % end: "define_function"









function [vertices,domain_str]=define_domain(example)

%--------------------------------------------------------------------------
% Object:
% This routine, defines the domain to analyse.
%--------------------------------------------------------------------------
% Input:
% example: determines the spherical triangle domain to be analysed. From 0
%   to 7, going from large domains to tiny ones (the higher "example", the
%   smaller the domain.
%--------------------------------------------------------------------------
% Output:
% vertices: it is a matrix whose rows are the vertices of the spherical
%           triangles.
% domain_str: string that stores the geometry of the region.
%--------------------------------------------------------------------------


switch example
    
    case 0
        % Define spherical polygon.
        factor=0.5;
        t=linspace(0,2*pi,10); t=(t(1:end-1))';
        XV=factor*cos(t); YV=factor*sin(t); ZV=sqrt(1-XV.^2-YV.^2);
        % Note: spherical polygon (first vertices not equal to last).
        vertices=[XV YV ZV];
        
        domain_str='Polygonal cap at North-Pole';
        
    case 1
        
        spec_settings=32;
        th=linspace(0,2*pi,spec_settings+1);
        %th=(th(1:end-1))';
        polygon_sides=[cos(th').*(1-cos(th')) sin(th').*(1-cos(th'))];
        polygon_sides=polygon_sides(1:end-1,:);
        XV=polygon_sides(:,1); XV=XV/2.1;
        YV=polygon_sides(:,2); YV=YV/2.1;
        ZV=sqrt(1-XV.^2-YV.^2);
        vertices=[XV YV ZV];
        
        domain_str='Polygonal cardioid at North-Pole';
        
    case 2
        
        % Australia island vertices in degrees.
        % Column 1: Longitude. Column 2: Latitude.
        vertices_degs=100*[1.462300280000000  -0.386970830000000
            1.451162780000000  -0.381483330000000
            1.396121110000000  -0.361567220000000
            1.385678610000000  -0.348266670000000
            1.374529440000000  -0.349083330000000
            1.359337220000000  -0.345341940000000
            1.340596110000000  -0.329141390000000
            1.235266670000000  -0.339379720000000
            1.150004170000000  -0.335283610000000
            1.131528610000000  -0.261491940000000
            1.142312220000000  -0.263158890000000
            1.168041670000000  -0.205237780000000
            1.221729170000000  -0.172633330000000
            1.239712780000000  -0.168258330000000
            1.243913060000000  -0.163392220000000
            1.252495830000000  -0.155816670000000
            1.259183610000000  -0.146770830000000
            1.268749720000000  -0.137471390000000
            1.283721390000000  -0.154966940000000
            1.292900000000000  -0.148620830000000
            1.303179720000000  -0.133733330000000
            1.308121390000000  -0.124083330000000
            1.317678890000000  -0.113258330000000
            1.347483890000000  -0.119512500000000
            1.360166940000000  -0.124987780000000
            1.359095560000000  -0.132850000000000
            1.395983610000000  -0.175387780000000
            1.415862220000000  -0.125583610000000
            1.437729170000000  -0.143991940000000
            1.462604440000000  -0.188633890000000
            1.500416670000000  -0.221254170000000
            1.531929440000000  -0.259316940000000
            1.525429440000000  -0.324442220000000
            1.499796110000000  -0.375050560000000
            1.462300280000000  -0.386970830000000];
        
        % Australia island vertices in radians.
        longitudes=deg2rad(vertices_degs(:,1));
        latitudes=deg2rad(vertices_degs(:,2));
        
        % Australia island vertices in cartesian coordinates.
        [XX,YY,ZZ] = sph2cart(longitudes,latitudes,1);
        vertices=[XX YY ZZ];
        
        domain_str='Australia as island (33 vertices, via data)';
        
        
        
end



