
function demo_sphrect_ade(deg,domain_type)

%--------------------------------------------------------------------------
% Object.
%--------------------------------------------------------------------------
% Testing algebraic degree of precision of "cub_sphrect" routine on some 
% domains, computing integral of the monomial basis.
%--------------------------------------------------------------------------
% Input.
%--------------------------------------------------------------------------
% deg   : algebraic degree of exactness degree of the rule.
%
% domain_type: it chooses between some preselected domain or can be your
%       own sph. rectangle in s^2 determined by extrema as a 3x2 matrix:
%           
%       1. [pi/6 pi/3; pi/8 pi/3]; % sph. Rectangle.
%       2. [0 pi/3; 0 2*pi]; % cap.
%       3. [pi/4 pi/3; pi 2*pi]; % sph. Rectangle.
%       4. [0 pi/2; 0 pi/4]; % half slice
%
%     notice that each point will be represented as
%
%           x_1=sin(theta_1)*cos(theta_2),
%           x_2=sin(theta_1)*sin(theta_2),
%           x_3=cos(theta_1)
%
%     with "theta1" in "[0,pi]" and "theta2" in "[0,2*pi]"
%--------------------------------------------------------------------------
% Routines.
%--------------------------------------------------------------------------
% 1. cub_sphrect (external to this file).
% 2. cval_powers_sincos (attached to this file).
%--------------------------------------------------------------------------
% Examples.
%--------------------------------------------------------------------------
% >> demo_sphrect_ade(10,1)
% 
%  	 Deg. rule          : 10 
%  	 Computed Integrals : 161
%  	 Points             : 132       
% 
%  	 Max abs.err.       :2.78e-17
%  	 Max rel.err        :1.85e-15 
% 
% >> demo_sphrect_ade(10,2)
% 
%  	 Deg. rule          : 10 
%  	 Computed Integrals : 161
%  	 Points             : 61        
% 
%  	 Max abs.err.       :1.33e-15
%  	 Max rel.err        :2.84e-15 
% 
% >> demo_sphrect_ade(10,3)
% 
%  	 Deg. rule          : 10 
%  	 Computed Integrals : 161
%  	 Points             : 132       
% 
%  	 Max abs.err.       :4.44e-16
%  	 Max rel.err        :2.71e-15 
% 
% >> demo_sphrect_ade(10,4)
% 
%  	 Deg. rule          : 10 
%  	 Computed Integrals : 161
%  	 Points             : 132       
% 
%  	 Max abs.err.       :5.55e-16
%  	 Max rel.err        :3.42e-15 
% 
% >> 
%--------------------------------------------------------------------------
%%
%% Copyright(C) 2013-. Mariano Gentile, Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%%          Mariano Gentile  <gentimar@hotmail.com>
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Date        : May 15, 2013
%% Last update : January 3, 2026
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, deg=10; end
if nargin < 2, domain_type=2; end
if isempty(domain_type) == 2, domain_type=2; end

% ............................ settings ...................................

% Relative error threshold (example will not be considered for integrals
% less then threshold).
threshold=10^(-15);

% ............................ examples ...................................

switch domain_type
    case 1
        x=[pi/6 pi/3; pi/8 pi/3]; % SPH. RECTANGLE.
    case 2
        x=[0 pi/3; 0 2*pi]; % CAP.
    case 3
        x=[pi/4 pi/3; pi 2*pi]; % SPH. RECTANGLE.
    case 4
        x=[0 pi/2; 0 pi/4]; % HALF SLICE
end





% ............................ main code ..................................

index=1;

flag_err_int=[]; % warning flags

[~,xw]=cub_sphrect(x,deg);

xx=xw(:,1); yy=xw(:,2); zz=xw(:,3); ww=xw(:,4);

for k1=0:deg
    for k2=0:k1
        for k3=0:deg-(k1+k2)

                % .................. DEFINE INTEGRAND  ....................

                f=@(x,y,z) (x.^k1).*(y.^k2).*(z.^k3);

                % .............. INTEGRAL VIA EXACT INTEGRATION ...........

                a=x(1,1); b=x(1,2);
                mm=k1+k2+1; nn=k3;
                I1=eval_powers_sincos(mm,nn,a,b);

                a=x(2,1); b=x(2,2);
                mm=k2; nn=k1;
                I2=eval_powers_sincos(mm,nn,a,b);

                Ie=I1*I2;

                % ........... INTEGRAL VIA NUMERICAL INTEGRATION ..........
                
                fpts=f(xx,yy,zz); I(index)=ww'*fpts;

                % ............. APPROXIMATION ERRORS ESTIMATE .............

                abserr(index)=abs(I(index)-Ie);
                if abs(Ie) > threshold
                    relerr(index)=abserr(index)/abs(Ie); flag(index)=1;
                else
                    relerr(index)=0; flag(index)=0;
                end

                err(index,:)=[abserr(index) relerr(index) flag(index)];
                seq(index,:)=[k1 k2 k3];

                index=index+1;

        end
    end
end




% ............................ statistics .................................

fprintf('\n \t Deg. rule          : %-3.0f',deg);
fprintf('\n \t Computed Integrals : %-3.0f',index-1);
fprintf('\n \t Points             : %-10.0f',size(ww,1));

fprintf('\n \n \t Max abs.err.       :%2.2e',max(abserr));
fprintf('\n \t Max rel.err        :%2.2e \n \n',max(relerr));










%--------------------------------------------------------------------------
% ATTACHED FUNCTIONS.
%--------------------------------------------------------------------------

function I=eval_powers_sincos(m,n,a,b)

I=0;
term=1;

while (n >= 2)
    Imn=f_mn(m,n,a,b);
    I=I+term*Imn;
    term=term*(n-1)/(m+n);
    n=n-2;
end

if n == 1
    Imn=((sin(b))^(m+1)-(sin(a))^(m+1))/(m+1);
else
    Imn=eval_powers_sin(m,a,b);
end

I=I+term*Imn;






function Imn=f_mn(m,n,a,b)

m1=m+1; n1=n-1;

ta=((sin(a)).^m1).*((cos(a)).^n1);
tb=((sin(b)).^m1).*((cos(b)).^n1);

Imn=(tb-ta)/(m+n);






function I=eval_powers_sin(m,a,b)

I=0;
term=1;

while (m >= 2)
    Im=g_n(m,a,b);
    I=I+term*Im;
    term=term*(m-1)/m;
    m=m-2;
end

if m == 1
    Im=-(cos(b)-cos(a));
else
    Im=b-a;
end

I=I+term*Im;






function In=g_n(n,a,b)

n1=n-1;

ta=((sin(a)).^n1).*((cos(a)));
tb=((sin(b)).^n1).*((cos(b)));

In=-(tb-ta)/n;



