
function demo_sphrect_integration(degV,domain_type,f_type)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Numerical integration of a choosen integrand on a spherical rectangle.
%
% The rule proposed by "cub_sphrect" is used for several degrees.
%
% Integrands and domains can be choosen from a list of tests (see attached
% function "integrand").
%
% A plot of the nodes of the last rule used is given.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% degV : algebraic degree of exactness degree of the rule. Can be a vector
%        of degrees.
%
% domain_type: it chooses between some preselected sph. rectangle
%       determined by extrema (see examples!) as a 2x2 matrix.
%
%         case 1
%             x=[pi/6 pi/3; 0 pi/2];    % sph. rectangle
%         case 2
%            x=[0 pi/3; 0 2*pi];        % sph. cap
%         case 3                        % non symmetric spherical collar
%             x=[-pi/8 pi/3; 0 2*pi];
%
%     Notice that each point will be represented as
%
%           x_1=sin(theta_1)*cos(theta_2),
%           x_2=sin(theta_1)*sin(theta_2),
%           x_3=cos(theta_1)
%
%     with "theta1" in "[0,pi]" and "theta2" in "[0,2*pi]"
%
% f_type: positive index, addressing a function to study.
%--------------------------------------------------------------------------
% Routines:
%--------------------------------------------------------------------------
% 1. cub_sphrect (external to this file).
% 3. cub_sphrect_adaptive (external to this file).
% 4. plot_s2 (attached to this file).
% 4. integrand (attached to this file)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >> demo_sphrect_integration
% 
%  	 domain:  1 function:  1 degree:  10 pts:    132
%  	 abs.err.: 8.67e-18 rel.err.: 6.38e-16 cpu: 7.28e-03 
% 
% >> demo_sphrect_integration([10 15],2,2)
% 
%  	 domain:  2 function:  2 degree:  10 pts:     61
%  	 abs.err.: 4.06e-03 rel.err.: 8.16e-04 cpu: 2.86e-02 
% 
%  	 domain:  2 function:  2 degree:  15 pts:    128
%  	 abs.err.: 1.38e-03 rel.err.: 2.78e-04 cpu: 2.16e-03 
% 
% >> 
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2013- Mariano Gentile, Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%%          Mariano Gentile  <gentimar@hotmail.com>
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Initial version  : May 15, 2013
%% Last update      : January 4, 2026
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, degV=10; end
if nargin < 2, domain_type=1; end
if nargin < 3, f_type=1; end




% ........................... settings ....................................

do_plot=1; % 0: DO NOT PLOT NODES. 1: PLOT NODES.

% ........................ battery of tests ...............................

[f,I,x]=integrand(domain_type,f_type);

% ................... compute integral numerically ........................

Inum=zeros(length(degV),1);

for ii=1:length(degV)
    
    % ... compute integral numerically ...
    tic;
    n=degV(ii);
    [~,xyzw]=cub_sphrect(x,n);
    fx=feval(f,xyzw(:,1),xyzw(:,2),xyzw(:,3)); w=xyzw(:,4); Inum(ii)=w'*fx;
    tt1=toc;
    
    % ... make statistics ...
    ae=abs(I-Inum(ii)); 
    re=ae*(abs(I) > 0)/abs(I+eps*(I == 0));
    
    fprintf('\n \t domain: %2.0f function: %2.0f degree: %3.0f pts: %6.0f',...
        domain_type,f_type,n,length(w));
    if abs(I) > 0
        fprintf('\n \t abs.err.: %2.2e rel.err.: %2.2e',ae,re);
    else
        fprintf('\n \t abs.err.: %2.2e',ae);
    end
    fprintf(' cpu: %2.2e \n',tt1);

end




% ........................ plot domain ....................................

if do_plot == 1
    clf; plot_s2(xyzw);
end

fprintf('\n');









function [f,I,x,s]=integrand(domain_type,f_type)


I=[];
s='';

switch domain_type
    
    case 1 % REGION 1. SPHERICAL RECTANGLE: x=[pi/6 pi/3; 0 pi/2];
        x=[pi/6 pi/3; 0 pi/2];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
                I=1.359211294541369519e-02;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
                I=2.370382551880577893e+00;
            case 3 % HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                (z-4/sqrt(18)).^2),0)).^1;
                I=2.630186422382734e-02;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
                I=1.260041859876477252e-02;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
                I=2.221882314846131135e-02;
            case 6 %TRIGONOMETRIC TEST.
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
                I=-4.684511626608869883e-02;
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2).^1;
                I=6.598105164046748072e-01;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
                I=6.544984694978748685e+00;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                (z-2/sqrt(5)).^2),0)).^2;
                I=1.790739296012628864e-03;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                (z-2/sqrt(5)).^2),0)).^3;
                I=1.817581787039426657e-04;
                
        end
        



    case 2 % REGION 2. CAP: x=[0 pi/3; 0 2*pi];
        x=[0 pi/3; 0 2*pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
                I=6.873008954228021e-02;
                
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
                I=4.970981230868087408e+00;
        end
        


    case 3 % REGION 3. NON SYMMETRIC SPH. COLLAR. x=[-pi/8 pi/3; 0 2*pi];
        x=[-pi/8 pi/3; 0 2*pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
                I=5.765892992557400398e-02;
            case 2 % POLYNOMIAL TEST.
                s='3*x.^3+4*y.^4+100*z.^16';
                f=@(x,y,z) 3*x.^3+4*y.^4+100*z.^16;
                I=1.065551074655049391e+01;
        end
        


    case 4 % REGION 4. HALF SLICE: x=[0 pi/2; 0 pi/4];
        x=[0 pi/2; 0 pi/4];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
                I=1.780704866475399384e-02;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
                I=8.677505738590557627e+00;
            case 3 % HESSE-WOMERSLEY TEST. (10^-9)
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                (z-4/sqrt(18)).^2),0)).^1;
                I=3.548144239832472235e-02;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
                I=2.078431729514684356e+00;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
                I=6.396667197458413834e-02;
            case 6 %TRIGONOMETRIC TEST. (10^-12)
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
                I=-1.320158930817089404e-01;
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1;
                I=8.980820670992835986e+00;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
                I=1.388400918174489185e+01;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                (z-2/sqrt(5)).^2),0)).^2;
                I=1.809965645127998950e-05;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                (z-2/sqrt(5)).^2),0)).^3;
                I=7.043860602145487052e-07;
                
                
        end
        
end









function plot_s2(pts)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Plotting pointsets on regions of the sphere.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% pts1: spherical/cartesian coordinates of first set of points on the unit 
%      sphere possibly belonging to the spherical rectangle/triangle 
%      defined by "x".
%      The k-th row of this matrix determines the k-th point.
%         The points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written on 20/11/2020: A. Sommariva and M. Vianello;
%
% Modified on:
% 28/11/2020 : A. Sommariva.
% 04/01/2026 : A. Sommariva
%--------------------------------------------------------------------------


% ......................... troubleshooting ...............................


if nargin < 1, pts=[]; end

if isempty(pts), pts=[]; end

% ......................... plot sphere ...................................

[X,Y,Z] = sphere(30); axis equal; grid off; axis off; hold on;

surf(X,Y,Z);
view([130 40])
% ......................... plot pointset 1 ...............................
if isempty(pts) == 0, plot_set(pts,'b','.',8); end

hold off









function plot_set(pts,color_str,marker_type,marker_size)

%--------------------------------------------------------------------------
% Object:
% Plotting pointsets on regions of the sphere.
%--------------------------------------------------------------------------
% Input:         
% pts: spherical or cartesian coordinates of a pointset on the 
%       unit sphere.
%      The k-th row of this matrix determines the k-th point.
%      The points in cartesian coordinates are
%                    x_1=sin(theta_1)*cos(theta_2),
%                    x_2=sin(theta_1)*sin(theta_2),
%                    x_3=cos(theta_1)
%
% color_str: color of the pointset, e.g. color_str='magenta';
% marker_type: marker type of the pointset, e.g. marker_type='.';
% marker_size: marker size of the pointset, e.g. marker_size=10.
%--------------------------------------------------------------------------
% Dates:
% Written on 20/11/2020: A. Sommariva and M. Vianello;
%
% Modified on:
% 28/11/2020: A. Sommariva.
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, pts=[]; end
if nargin < 2, color_str='magenta'; end
if nargin < 3, marker_type='.'; end
if nargin < 4, marker_size=10; end

if isempty(marker_type), marker_type='.'; end
if isempty(marker_size), marker_size=10; end


% ......................... plotting points ...............................

if size(pts,2) == 2 % speherical coordinates
    theta_1=pts(:,1); theta_2=pts(:,2);
    s1=sin(theta_1); c1=cos(theta_1); s2=sin(theta_2); c2=cos(theta_2);
    xx=s1.*c2; yy=s1.*s2; zz=c1;
else % cartesian coordinates
    xx=pts(:,1); yy=pts(:,2); zz=pts(:,3);
end

plot3(xx,yy,zz,marker_type,'Color',color_str,'MarkerSize',marker_size);



