
function demo_splcub_2026

%--------------------------------------------------------------------------
% OBJECT:
%--------------------------------------------------------------------------
% This demo:
% * tests the performance of our rules of degree ADE, on spline
%   curvilinear domains
% * plots domain and pointsets
%--------------------------------------------------------------------------
% EXAMPLES:
%--------------------------------------------------------------------------
% >> demo_splcub_2026
%
%  	 ade          : 6
%  	 cardinality  : 28
%  	 rule res.err.: 1.342e-15
%  	 integral     : 0.238175744716478
%  	 cputime      : 2.850e-02
%
% See figure representing the domain and cubature nodes.
%
%
% >> demo_splcub_2026
%
%     ade          : 10
%     cardinality  : 66
%     rule res.err.: 2.468e-15
%     integral     : 0.238078244740745
%     cputime      : 5.069e-02
%
% See figure representing the domain and cubature nodes.
%
%
% >>
%--------------------------------------------------------------------------
% License
%--------------------------------------------------------------------------
% Copyright (C) 2020- Alvise Sommariva, Marco Vianello.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%
% Author:  Alvise Sommariva <alvise@math.unipd.it>
%          Marco Vianello   <marcov@math.unipd.it>
%
% First version : June 3, 2020
% Latest version: January 28, 2026
%--------------------------------------------------------------------------


% ----------------------------- settings ----------------------------------

ade=6;   % algebraic degree of precision

extraction_type=1; % 1: lsqnonneg (positive weights),
% otherwise: QR (possible negative weights)

% ---------------------------- main code below ----------------------------


% --------------------------- domain definition ---------------------------

% Curvilinear vertices

vertices=[-1.0 0.0;-2.0 -1.0; -1.5 -2.0; 0.0 -1.6; 0.0 -1.0; -0.2 -0.5; ...
    -0.38 -0.75; -0.20 -0.94; -0.57 -1.28; -1.0 0.0];

% Spline orders.
%
% 1. The first 5 vertices are the knots of splines "x=x(t)", "y=y(t)" of
% order 2 that is of degree 1, i.e. linear splines.
%
% 2. From the 5-th vertex (included!) to the last one we will use a spline
% of order 4, i.e. of degree 3 (a cubic spline).
%
% The type of cubic spline is defined by the variable "SPLtypestr" that
% can be:
%
% a) 'complete'   : match endslopes (as given in VALCONDS, with default as
%                   under *default*);
%                  'not-a-knot' : make spline C^3 across first and last
%                   interior break;
%                  'periodic'   : match first and second derivatives at
%                   first data point with those at last data point;
%                  'variational': set end second derivatives equal to zero.

spline_parms=[2 5; 4 size(vertices,1)];
SPLtypestr='not-a-knot';

% ----------------------------- integrand ---------------------------------

f=@(x,y) exp(-x.^2-y.^2);


% ....................... compute algebraic rule ..........................

XV=vertices(:,1);
YV=vertices(:,2);

tic;
[xyw,res,~,~,Sx,Sy] = splcub_26(ade,XV,YV,spline_parms,SPLtypestr,...
    extraction_type);
cpus=toc;

nodes_x=xyw(:,1); nodes_y=xyw(:,2); weights=xyw(:,3);

fnodes=feval(f,nodes_x,nodes_y);

% ......................... evaluate integral .............................

I=weights'*fnodes;




% ............................ statistics .................................

fprintf('\n \t ade          : %-3.0f',ade);
fprintf('\n \t cardinality  : %-8.0f',length(nodes_x));
fprintf('\n \t rule res.err.: %-1.3e',res);
fprintf('\n \t integral     : %-1.15f',I);
fprintf('\n \t cputime      : %-1.3e',cpus);
fprintf('\n ');
fprintf('\n \t See figure representing the domain and cubature nodes.');
fprintf('\n \n');


% ........................ plot domain/nodes ..............................

plot_spl(Sx,Sy);
hold on;
plot(nodes_x,nodes_y,'o','MarkerSize',4,'MarkerEdgeColor','b',...
    'MarkerFaceColor','g');
title(['Domain and nodes of a formula with ade ',num2str(ade)]);
hold off;









%--------------------------------------------------------------------------
% Attached functions.
%--------------------------------------------------------------------------

function plot_spl(Sx,Sy)

% ........................ plot preferences ...............................
plot_axis=1;

% ........................... plot: start .................................
clf;
hold on;

if plot_axis == 0, axis off; else, axis on; end

% ........................ plot: grids .........................



% ....................... plot: parametric splines ........................

L=length(Sx);
for ii=1:L

    SxL=Sx(ii);
    SyL=Sy(ii);

    SxL_breaks=SxL.breaks;


    Nbreaks=length(SxL_breaks);

    for kk=1:Nbreaks-1

        t0=SxL_breaks(kk); t1=SxL_breaks(kk+1);

        N=10^3;
        tt=linspace(t0,t1,N);

        xx=ppval(SxL,tt);
        yy=ppval(SyL,tt);


        plot(xx,yy,'k-','LineWidth',1);
    end

end

hold off;




