
function demo_cub_torus_rect(ade,domain_type,f_type)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Numerical integration of a choosen integrand on a toroidal rectangle.
%
% The rule proposed by "cub_torus_rect" is used for several degrees.
%
% Integrands and domains can be choosen from a list of tests (see attached
% function "integrand").
%
% A plot of the nodes of the last rule used is given.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% ade : algebraic degree of exactness degree of the rule.
%
% domain_type: it chooses between some preselected toroidal rectangle
%       determined by extrema (see examples!) as a 2x2 matrix.
%
%         case 1
%             x=[pi/6 pi/3; 0 pi/2];    % toroidal rectangle
%         case 2
%            x=[0 pi/3; -pi pi];        % TOR.cap
%         case 3                        % non symmetric toroidal collar
%             x=[-pi/8 pi/3; -pi pi];
%
%       we used the toroidal coordinates:
%
%           x(theta1,theta2)=(R+r*cos(theta1)).*cos(theta2);
%           y(theta1,theta2)=(R+r*cos(theta1)).*sin(theta2);
%           z(theta1,theta2)=r*sin(theta1);
%
%       with "theta1" in "[-pi,pi]", "theta2" in "[-pi,pi]"
%
% f_type: positive index, addressing a function to study.
%--------------------------------------------------------------------------
% Routines:
%--------------------------------------------------------------------------
% 1. cub_torus_rect (external to this file).
% 2. integrand (attached to this file)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >> demo_cub_torus_rect
%
% domain:  1 function:  7 degree:  10 pts:    132
% abs.err.: 5.39e-14 rel.err.: 1.19e-13 cpu: 1.73e-03
%
% .............. On toroidal cap ..............
%
% minor radius 'r': 1.00000e+00
% major radius 'R': 2.00000e+00
% minor theta       : 5.23599e-01
% major theta       : 1.04720e+00
% minor phi         : 0.00000e+00
% major phi         : 1.57080e+00
%
% a) the angle "theta" describes the latitude.
% b) the angle "phi" describes the longitude.
%
% >>
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2026 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Initial version  : January 6, 2026
%% Last update      : January 6, 2026
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, adeV=10; end
if nargin < 2, domain_type=1; end
if nargin < 3, f_type=7; end




% ........................... settings ....................................

do_plot=1; % 0: DO NOT PLOT NODES. 1: PLOT NODES.
adeR=50;

% ........................ battery of tests ...............................

[f,I,r,R,omega]=integrand(domain_type,f_type);

if isempty(I)
    [~,xyzwR]=cub_torus_rect(r,R,omega,adeR);
    fxR=f(xyzwR(:,1),xyzwR(:,2),xyzwR(:,3)); wR=xyzwR(:,4);
    I=wR'*fxR;
end

% ................... compute integral numerically ........................

Inum=zeros(length(adeV),1);

for ii=1:length(adeV)

    % ... compute integral numerically ...
    tic;
    ade=adeV(ii);
    [~,xyzw]=cub_torus_rect(r,R,omega,ade);
    fx=feval(f,xyzw(:,1),xyzw(:,2),xyzw(:,3)); w=xyzw(:,4); Inum(ii)=w'*fx;
    tt1=toc;




    % ... make statistics ...
    ae=abs(I-Inum(ii));
    re=ae*(abs(I) > 0)/abs(I+eps*(I == 0));

    fprintf('\n \t domain: %2.0f function: %2.0f degree: %3.0f pts: %6.0f',...
        domain_type,f_type,ade,length(w));
    if abs(I) > 0
        fprintf('\n \t abs.err.: %2.2e rel.err.: %2.2e',ae,re);
    else
        fprintf('\n \t abs.err.: %2.2e',ae);
    end
    fprintf(' cpu: %2.2e \n',tt1);

end



% ............................... statistics ..............................

fprintf('\n \t .............. On toroidal cap .............. \n');

fprintf('\n \t minor radius ''r'': %1.5e',r);
fprintf('\n \t major radius ''R'': %1.5e',R);
fprintf('\n \t minor theta       : %1.5e',omega(1,1));
fprintf('\n \t major theta       : %1.5e',omega(1,2));
fprintf('\n \t minor phi         : %1.5e',omega(2,1));
fprintf('\n \t major phi         : %1.5e',omega(2,2));
fprintf('\n ');
fprintf('\n \t a) the angle "theta" describes the latitude.');
fprintf('\n \t b) the angle "phi" describes the longitude.');
fprintf('\n \n');

% ............................. plot pointset .............................

if do_plot
    figure(1);
    plot_torus(r,R,omega)
    hold on;
    plot3(xyzw(:,1),xyzw(:,2),xyzw(:,3),'ko','MarkerSize',4);
    title('Nodes of a rule on toroidal rectangle');
    hold off;
end





function plot_torus(r,R,omega)


% plot torus
[theta1,theta2] = meshgrid(linspace(0,2*pi,30));
x = (R + r*cos(theta1)).*cos(theta2);
y = (R + r*cos(theta1)).*sin(theta2);
z = r*sin(theta1);
surf(x,y,z)
hold on;
shading interp
colormap parula
axis equal

% plot rectangle
th1=linspace(omega(1,1),omega(1,2),200);
th2=linspace(omega(2,1),omega(2,2),200);
[theta1,theta2]=meshgrid(th1,th2);
theta1=theta1(:); theta2=theta2(:);
x = (R + r*cos(theta1)).*cos(theta2);
y = (R + r*cos(theta1)).*sin(theta2);
z = r*sin(theta1);
plot3(x,y,z,'o','color',[220 220 220]/256,'MarkerSize',4);
hold off




function [f,I,r,R,x,s]=integrand(domain_type,f_type)


I=[];
s='';

r=1; R=2; % torus radii

switch domain_type

    case 1 % REGION 1. toroidal RECTANGLE: x=[pi/6 pi/3; 0 pi/2];
        x=[pi/6 pi/3; 0 pi/2];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
            case 3 % HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                    (z-4/sqrt(18)).^2),0)).^1;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
            case 6 %TRIGONOMETRIC TEST.
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2).^1;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^2;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^3;

        end




    case 2 % REGION 2. CAP: x=[0 pi/3; -pi pi];
        x=[0 pi/3; -pi pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;

            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
        end



    case 3 % REGION 3. NON SYMMETRIC TOR.COLLAR. x=[-pi/8 pi/3; -pi pi];
        x=[-pi/8 pi/3; -pi pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % POLYNOMIAL TEST.
                s='3*x.^3+4*y.^4+100*z.^16';
                f=@(x,y,z) 3*x.^3+4*y.^4+100*z.^16;
        end



    case 4 % REGION 4. HALF SLICE: x=[0 pi/2; 0 pi/4];
        x=[0 pi/2; 0 pi/4];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
            case 3 % HESSE-WOMERSLEY TEST. (10^-9)
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                    (z-4/sqrt(18)).^2),0)).^1;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
            case 6 %TRIGONOMETRIC TEST. (10^-12)
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^2;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^3;


        end

end






