
function demo_torus_rect_adaptive_integration(domain_type,f_type,tol)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Numerical integration of a choosen integrand on a toroidal rectangle.
%
% The rule proposed by "cub_torus_rect_adaptive" is used with a choosen
% tolerance.
%
% Integrands and domains can be choosen from a list of tests (see attached
% function "integrand").
%
% A plot of the nodes of the last rule used is given.
%--------------------------------------------------------------------------
% Input:
%--------------------------------------------------------------------------
% domain_type: it chooses between some preselected sph. rectangle
%       determined by extrema (see examples!) as a 2x2 matrix.
%
%         case 1
%             omega=[pi/6 pi/3; 0 pi/2];    % tor. rectangle
%         case 2
%            omega=[0 pi/3; -pi pi]         % tor. cap
%         case 3
%             omega=[-pi/8 pi/3; -pi pi];   % non symmetric toroidal coll.
%         case 4
%              omega=[-pi pi; 0 pi/4];      % slice
%
%     Notice that each point will be represented as
%
%           x(theta1,theta2)=(R+r*cos(theta1)).*cos(theta2);
%           y(theta1,theta2)=(R+r*cos(theta1)).*sin(theta2);
%           z(theta1,theta2)=r*sin(theta1);
%
%     where
%
%       a) "theta1" belongs to [omega(1,1),omega(1,2)] that is
%          contained in [-pi,pi].
%       b) "theta2" belongs to [omega(2,1),omega(2,2)] that is
%          contained in [-pi,pi].
%
% f_type: positive index, addressing a function to study;
%
% tol   : vector of two components establishing the cubature error
%         tolerance
%
%            tol(1): 'AbsTol', absolute error tolerance
%            tol(2): 'RelTol', relative error tolerance
%
%         it attempts to satisfy |IH - I| <= max(AbsTol,RelTol*|IH|),
%         where "IH" is the numerical value computed by the algorithm;
%
%         if "tol" is scalar then "AbsTol=tol", "RelTol=tol".
%--------------------------------------------------------------------------
% Routines:
%--------------------------------------------------------------------------
% 1. cub_torus_rect_adaptive (external to this file).
% 2. plot_torus (attached to this file).
% 3. integrand (attached to this file)
%--------------------------------------------------------------------------
% Examples:
%--------------------------------------------------------------------------
% >>demo_torus_rect_adaptive_integration
%
%
% ............. settings ..................
%
% domain:  4
% function:  2
% f=@(x,y,z) 25*max(x+0*y+0*z,0).^3
%
% atol : 1.00000e-12  rtol : 1.00000e-12
%
% ........ errors estimation ..............
%
% abs.err.exit  : 2.626e-09
% abs.err.est   : 2.274e-12
% rel.err.est   : 8.657e-16
%
% .......... actual errors ................
%
% abs.err.   : 0.000e+00
% rel.err.   : 0.000e+00
%
% ........... integrals ................
%
% I     : 2.626391736880076e+03
% Inum  : 2.626391736880076e+03
%
% ............. cputime .................
%
% cpu: 6.79e-03

% >>
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2026 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%%          Alvise Sommariva <alvise@math.unipd.it>
%%          Marco Vianello   <marcov@math.unipd.it>
%%
%% Initial version  : January 7, 2026
%% Last update      : January 7, 2026
%--------------------------------------------------------------------------

% ......................... troubleshooting ...............................

if nargin < 1, domain_type=4; end
if nargin < 2, f_type=2; end
if nargin < 3, tol=[10^(-12) 10^(-12)]; end



% ........................ battery of tests ...............................

[f,~,r,R,omega,s]=integrand(domain_type,f_type);

% ................... compute integral numerically ........................

tic;
[Inum,Ierr]=cub_torus_rect_adaptive(r,R,omega, f,tol);
tt1=toc;

[~,xyzw]=cub_torus_rect(r,R,omega,80);
I=(xyzw(:,4))'*f(xyzw(:,1),xyzw(:,2),xyzw(:,3));

% .......................... statistics ...................................


fprintf('\n \n \t ............. settings .................. \n');
fprintf('\n \t domain: %2.0f',domain_type);
fprintf('\n \t function: %2.0f',f_type);
fprintf('\n \t f=@(x,y,z) '); disp(s)
fprintf('\n \t atol : %1.5e  rtol : %1.5e',tol(1),tol(end));

fprintf('\n \n \t ........ errors estimation .............. \n');
fprintf('\n \t abs.err.exit  : %1.3e',max(tol(1),tol(end)*abs(Inum)))
fprintf('\n \t abs.err.est   : %1.3e',Ierr);
fprintf('\n \t rel.err.est   : %1.3e',Ierr/Inum);

fprintf('\n \n \t .......... actual errors ................ \n');
if abs(I) > 0
    fprintf('\n \t abs.err.   : %1.3e',abs(I-Inum));
    fprintf('\n \t rel.err.   : %1.3e',abs(I-Inum)/abs(I));
else
    fprintf('\n \t abs.err.est   : %1.3e',abs(I-Inum));
end

fprintf('\n \n \t ........... integrals ................ \n');
fprintf('\n \t I     : %1.15e',I);
fprintf('\n \t Inum  : %1.15e',Inum);


fprintf('\n \n \t ............. cputime ................. \n');
fprintf('\n \t cpu: %2.2e \n',tt1);


% .......................... plot region ..................................

plot_torus(r,R,omega)










%--------------------------------------------------------------------------
% Subroutines
%--------------------------------------------------------------------------

function plot_torus(r,R,omega)


% plot torus
[theta1,theta2] = meshgrid(linspace(0,2*pi,30));
x = (R + r*cos(theta1)).*cos(theta2);
y = (R + r*cos(theta1)).*sin(theta2);
z = r*sin(theta1);
surf(x,y,z)
hold on;
shading interp
colormap parula
axis equal

% plot rectangle
th1=linspace(omega(1,1),omega(1,2),200);
th2=linspace(omega(2,1),omega(2,2),200);
[theta1,theta2]=meshgrid(th1,th2);
theta1=theta1(:); theta2=theta2(:);
x = (R + r*cos(theta1)).*cos(theta2);
y = (R + r*cos(theta1)).*sin(theta2);
z = r*sin(theta1);
plot3(x,y,z,'o','color',[220 220 220]/256,'MarkerSize',4);
hold off





function [f,I,r,R,omega,s]=integrand(domain_type,f_type)


I=[];
s='';

r=1; R=2; % torus radii

switch domain_type

    case 1 % REGION 1. toroidal RECTANGLE: omega=[pi/6 pi/3; 0 pi/2];
        omega=[pi/6 pi/3; 0 pi/2];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
            case 3 % HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                    (z-4/sqrt(18)).^2),0)).^1;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
            case 6 %TRIGONOMETRIC TEST.
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2).^1;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^2;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^3;

        end




    case 2 % REGION 2. CAP: omega=[0 pi/3; -pi pi];
        omega=[0 pi/3; -pi pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;

            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
        end



    case 3 % REGION 3. NON SYMMETRIC TOR.COLLAR. omega=[-pi/8 pi/3; -pi pi];
        omega=[-pi/8 pi/3; -pi pi];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % POLYNOMIAL TEST.
                s='3*x.^3+4*y.^4+100*z.^16';
                f=@(x,y,z) 3*x.^3+4*y.^4+100*z.^16;
        end



    case 4 % REGION 4. SLICE: omega=[-pi pi; 0 pi/4];
        omega=[-pi pi; 0 pi/4];
        switch f_type
            case 1 % HESSE-WOMERSLEY TEST.
                s='1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1';
                f=@(x,y,z) 1./((x-2.5).^2+(y-2.5).^2+(z-6.5).^2).^1;
            case 2 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^3';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^3;
            case 3 % HESSE-WOMERSLEY TEST. (10^-9)
                s1='(max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2';
                s2='+(z-4/sqrt(18)).^2),0)).^1';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(18)).^2+(y-1/sqrt(18)).^2+...
                    (z-4/sqrt(18)).^2),0)).^1;
            case 4 % POLYNOMIAL TEST.
                s='3*x.^40+4*y.^40+100*z.^39';
                f=@(x,y,z) 3*x.^40+4*y.^40+100*z.^39;
            case 5 % EXPONENTIAL TEST.
                s='exp(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) exp(-x.^2-100*y.^2-0.5*z.^2);
            case 6 %TRIGONOMETRIC TEST. (10^-12)
                s='sin(-x.^2-100*y.^2-0.5*z.^2)';
                f=@(x,y,z) sin(-x.^2-100*y.^2-0.5*z.^2);
            case 7 % MODIFICATION HESSE-WOMERSLEY TEST.
                s='1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1';
                f=@(x,y,z) 1./((x-1).^2+(y-0).^2+(z-0).^2+10^(-5)).^1;
            case 8 % HESSE-WOMERSLEY TEST.
                s='25*max(x+0*y+0*z,0).^1';
                f=@(x,y,z) 25*max(x+0*y+0*z,0).^1;
            case 9 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^2';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^2;
            case 10 % MODIFICATION HESSE-WOMERSLEY TEST.
                s1='(max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2';
                s2='+(z-2/sqrt(5)).^2),0)).^3';
                s=strcat(s1,s2);
                f=@(x,y,z) (max(1/4-((x-1/sqrt(5)).^2+(y-2/sqrt(5)).^2+...
                    (z-2/sqrt(5)).^2),0)).^3;


        end

end






