
function demo_wam_toroidal_rect(domain_type)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% Demo on "weakly admissible meshes" on toroidal (surface) rectangles.
%
% This routine determines such a mesh for fixed degree " deg" on specific
% domains.
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% [1] Discrete norming inequalities on sections of sphere, ball and torus,
% Journal of Inequalities and Special Functions, Volume 9, 2018,
% pp. 113-121.
% A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written in 2018, modified on January 6, 2026.
%--------------------------------------------------------------------------
%% Copyright (C) 2018-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : 2018
%% Modified on: JANUARY 6, 2026
%--------------------------------------------------------------------------

if nargin < 1, domain_type=1; end

n=10; % WAM degree
r=1; R=2; % minor and major radii of the torus


switch domain_type

    case 1 % toroidal slice: omega(1)=-pi, omega(2)=pi;
          % [omega(3),omega(4)] contained in [-pi,pi] (see [1], p.118)
        omega(1)=0; omega(2)=2*pi; omega(3)=pi/8; omega(4)=pi/4;
        title_str='WAM on toroidal slice';

    case 2 % toroidal cap: w in [0,pi/2], omega(1)=pi/2-w, omega(2)=pi/2+w
           % omega(3)=-pi; omega(4)=pi. (see [1], p.118)

        w=pi/4; 
        omega(1)=pi/2-w; omega(2)=pi/2+w; omega(3)=-pi; omega(4)=pi;
        title_str='WAM on toroidal cap';

    case 3 % toroidal zone: omega(1)=omega1, omega(2)=omega2 > omega1
           % omega(3)=-pi; omega(4)=pi. (see [1], p.118)

        omega(1)=pi/8; omega(2)=pi/3; omega(3)=-pi; omega(4)=pi;
        title_str='WAM on toroidal zone';

    case 4 % toroidal rectangle:  [omega(1),omega(2)], [omega(3),omega(4)] 
           % contained in [-pi,pi]

           omega(1)=0; omega(2)=pi/2; omega(3)=-pi/3; omega(4)=pi/8;
           title_str='WAM on toroidal rectangle';

end

wam=wam_toroidal_rect(n,r,R,omega);



% ............................... statistics ..............................
fprintf('\n \t ........... WAM on toroidal rectangle ........... \n');
fprintf('\n \t minor radius ''r''  : %1.5e',r);
fprintf('\n \t major radius ''R''  : %1.5e',R);
fprintf('\n \t minor theta       : %1.5e',omega(1));
fprintf('\n \t major theta       : %1.5e',omega(2));
fprintf('\n \t minor phi         : %1.5e',omega(3));
fprintf('\n \t major phi         : %1.5e',omega(4));
fprintf('\n ');
fprintf('\n \t degree WAM        : %-8.0f',n);
fprintf('\n \t cardinality       : %-8.0f',size(wam,1));
fprintf('\n ');
fprintf('\n \t a) the angle "theta" describes the latitude;');
fprintf('\n \t b) the angle "phi" describes the longitude.');
fprintf('\n \n');

% ............................. plot pointset .............................

figure(1);
plot_torus(r,R);
hold on; 
plot3(wam(:,1),wam(:,2),wam(:,3),'ko','MarkerSize',4);
title(title_str);
hold off;












function plot_torus(r,R)

[theta,phi] = meshgrid(linspace(0,2*pi,50));
x = (R + r*cos(theta)).*cos(phi);
y = (R + r*cos(theta)).*sin(phi);
z = r*sin(theta);
surf(x,y,z)
shading interp
colormap parula
axis equal
