
function wam=wam_toroidal_cap(n,r,R,w,m)

%--------------------------------------------------------------------------
% Object:
%--------------------------------------------------------------------------
% The routine computes the nodes of a WAM (Weakly Admissible Mesh)
% for total-degree bivariate polynomials of degree "n" on a toroidal cap.
%--------------------------------------------------------------------------
% Input: 
%--------------------------------------------------------------------------
% n  : degree of the mesh
%
% r, R : torus minor and major radii; if "R=0" then the torus degenerates
%        into a sphere;
%
% w: scalar in [0,pi/2], defining the theta angle of slice 
%           x(theta,phi)=(R+r*cos(theta)).*cos(phi);
%           y(theta,phi)=(R+r*cos(theta)).*sin(phi);
%           z(theta,phi)=r*sin(theta);
% where 
%       a) "theta" belongs to [pi/2-w, pi/2+w] that is 
%          contained in [-pi,pi].
%       b) "phi" belongs to [-pi,pi].
% (see "https://en.wikipedia.org/wiki/Toroidal_and_poloidal_coordinates");
% in other words "theta" defines the minor latitude.
%
% m   : WAM factor (m > 1).
%--------------------------------------------------------------------------
% Output:
%--------------------------------------------------------------------------
% wam: 3-column array of the WAM point coordinates 
%--------------------------------------------------------------------------
% 
%--------------------------------------------------------------------------
% Dates:
%--------------------------------------------------------------------------
% Written by Alvise Sommariva and Marco Vianello, University of Padova
% February 20, 2018
%
% Modified:
% January 6, 2026, by A. Sommariva 
%--------------------------------------------------------------------------
% Reference paper
%--------------------------------------------------------------------------
% Discrete norming inequalities on sections of sphere, ball and torus,
% Journal of Inequalities and Special Functions, Volume 9, 2018, 
% pp. 113-121.
% A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
%% Copyright (C) 2018-
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date       : February 20, 2018
%% Modified on: January 6, 2026
%--------------------------------------------------------------------------

if nargin < 5, m=2; end % "m" must be larger than 1.

mesh1=Wmap(m*n,pi/2-w,pi/2+w);
mesh2=Wmap(m*n,-pi,pi);

[thetaM,phiM]=meshgrid(mesh1,mesh2);
thetaV=thetaM(:); phiV=phiM(:);

wam=sigma(r,R,thetaV,phiV);









function val=sigma(r,R,theta,phi)

% Map defined in "Discrete norming inequalities on sections of sphere, 
% ball and torus", Journal of Inequalities and Special Functions, Volume 9,
% 2018, p.114 (formula (2.6)).

val=zeros(length(theta),3);

val(:,1)=(R+r*cos(theta)).*cos(phi);
val(:,2)=(R+r*cos(theta)).*sin(phi);
val(:,3)=r*sin(theta);









function W_eta=Wmap(deg,u,v)

% Map defined in "Discrete norming inequalities on sections of sphere, 
% ball and torus", Journal of Inequalities and Special Functions, Volume 9,
% 2018, p.115 (formula (2.13)).

Z_2eta=Zmap(2*deg,-1,1);
w=(v-u)/2;
W_eta=2*asin(sin(w/2).*Z_2eta)+(u+v)/2;









function val=Zmap(deg,a,b)

% Map defined in "Discrete norming inequalities on sections of sphere, 
% ball and torus", Journal of Inequalities and Special Functions, Volume 9,
% 2018, p.115 (formula (2.12)).

nodes=chebyshev_nodes(deg);
val=((b-a)/2)*nodes+(b+a)/2;









function nodes=chebyshev_nodes(deg)

% Classical Chebyshev nodes.

j=(0:deg)';
nodes=cos((2*j+1)*pi/(2*(deg+1)));









% function val=Xmap(deg,a,b)
% 
% % Map defined in "Discrete norming inequalities on sections of sphere, 
% % ball and torus", Journal of Inequalities and Special Functions, Volume 9,
% % 2018, p.115 (formula (2.12)).
% 
% nodes=chebyshev_lobatto_nodes(deg);
% val=((b-a)/2)*nodes+(b+a)/2;
% 
% 
% 
% 
% 
% 
% 
% 
% function nodes=chebyshev_lobatto_nodes(deg)
% 
% % Classical Chebyshev nodes.
% 
% j=(0:deg)';
% nodes=cos(j*pi/eta);