
function [pts_fek,ind,w_fek,C,P]=computing_fekete_points(pts_wam,V_wam,...
    niter,moms,extraction_method)

%--------------------------------------------------------------------------
% OBJECT.
%----------
%
% COMPUTATION OF APPROXIMATE FEKETE POINTS (AND POSSIBLY WEIGHTS OF THE 
% RELATIVE ALGEBRAIC CUBATURE RULE). 
%
%--------------------------------------------------------------------------
% INPUT.
%----------
%
% pts_wam: MATRIX WITH TWO COLUMNS (x,y) WHOSE ROWS ARE THE WAM POINTS.
%
% V_wam: VANDERMONDE BASIS AT WAM POINTS.
%
% niter: NUMBER OF TIMES THAT THE DISCRETE ORTHOGONALIZATION PROCESS TAKES 
%        PLACE. IF niter=0 THEN NO DISCRETE ORTHOGONALIZATION.
%
% moms: A COLUMN VECTOR. IN CUBATURE PROBLEMS IT CONSISTS OF THE MOMENTS OF
%    THE BASIS ON THE DOMAIN. FOR INTERPOLATION PROBLEMS IT IS ENOUGH TO
%    SET mom=ones(size(V,2),1) WHERE "V" IS THE VANDERMONDE MATRIX.
%
% a, b, c, d: THEY DETERMINE THE SMALLER SQUARE INCLUDING THE DOMAIN (AT
%    LEAST NUMERICALLY) WITH SIDES PARALLEL TO THE CARTESIAN AXIS. IT IS  
%    USED TO SCALE SOME BASIS. 
%
% extraction_method: 1: QR 2:LU.
%
%--------------------------------------------------------------------------
% OUTPUT.
%----------
%
% fek: MATRIX WITH 2 COLUMNS REPRESENTING (x,y) OF APPROXIMATE FEKETE
%    POINTS.
%
% ind: COMPONENTS OF THE WAM POINTS (nodes_x,nodes_y) THAT ARE APPROXIMATE 
%    FEKETE POINTS.
%
% w_fek: WEIGHTS OF AN ALGEBRAIC CUBATURE RULE BASED ON APPROXIMATE FEKETE
%    POINTS.
% 
% C, P: "C = P'*V_wam'" WITH C SUCH THAT 
%
%              norm(C*C'-eye(size(C*C'))) = 0
%
%   I.E. C*C'= I (THEORETICALLY BUT NOT NUMERICALLY). NOTE THAT C'*C IS NOT
%   EQUAL TO THE IDENTITY MATRIX.
%
%--------------------------------------------------------------------------
% FUNCTIONS USED IN THIS ROUTINE.
%---------------------------------
% 
% 1. find (MATLAB built-in).
% 2. qr (MATLAB built-in).
% 3. lu (MATLAB built-in).
% 4. inv (MATLAB built-in).
% 5. \  (MATLAB built-in).
% 
%--------------------------------------------------------------------------
% REFERENCES.
%-------------
%
% [1] A. SOMMARIVA and M. VIANELLO
%     "Computing approximate Fekete points by QR factorizations
%     of Vandermonde matrices".
%     Comp. Math. Appl. (to appear)
% 
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
%% Copyright (C) 2007-2009 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Federica Basaglia
%%          Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: November 15, 2009
%--------------------------------------------------------------------------


if length(moms) > 0
    m=moms;
else
    m=ones(size(V_wam,2),1);
end

B=V_wam;
P=eye(length(V_wam(1,:))); 
for it=1:niter,
    [Q,R]=qr(B,0);
    
    U=inv(R);
    B=B*U;
    P=P*U;
end;
C=B'; 

%**************************************************************************
%
% IMPORTANT: At this point "C = P'*V_wam'" (theroretically but not 
% numerically!). 
% 
% For example, in the case of our settings
%
% 	 [DOMAIN]:  8 [DEGREE]: 10 [1D WAM]:  4 [BASIS ]:  2
% 	 [ITERS.]:  2 [AF.PTS]: 66 [IN.PTS]: 1554 [WAM]:  4
% 	 [EXTRACTION METHOD]: QR
%
% we have
%
% norm(C-P'*V_wam')=1.579431464726480e-12
%
% norm(inv(P')*C-V_wam')=1.482384613615308e-11
%
% norm(C*C'-eye(size(C*C')))=1.920664854343301e-15
%
% Instead of V_wam'w=m, we intent to solve the theoretically 
% equivalent linear system (B*inv(P))'*w=m.
%
%**************************************************************************

m=P'*m;

Plu=[];

% timing(1)=cputime;

switch extraction_method
case 1
    
    % QR METHOD.
    w=C\m;

    %indexes of the nonzero weights
    ind=find(abs(w)>0);
    
    if size(C(:,ind),1) ~= size(C(:,ind),2)
        fprintf('\n \t >> [WARNING]: RANK DEFICIENCY')
    end
    
    w_fek=w(ind);
    
case 2
    % LU METHOD.
    [L,U,Plu]=lu(B);

    Plu=Plu(1:size(V_wam,2),:);
    
    ind=[];

    for index=1:size(Plu,2)
        ind_pre(index)=norm(Plu(:,index),2);
    end
    
    ind=find(ind_pre > 0);

    C_fek=C(:,ind);
    w=C_fek\m;
    
    w_fek=w;
    
end

pts_fek=pts_wam(ind,:);

% timing(2)=cputime; format short e; diff(timing)






