%--------------------------------------------------------------------------
% OBJECT.
%-------------------
%
% IN THIS DEMO, THE CODE COMPUTES APPROXIMATE FEKETE POINTS OVER BIVARIATE 
% DOMAINS.
%
%                          ...... USAGE ......
%
% 1. IF YOU INTEND TO ADD YOUR OWN WAM, ADD IT TO THE ROUTINE "wam2D.m".
%    SET A RIGHT CASE IN "switch wam_select" COMMAND AT THE BEGINNING OF 
%    THE ROUTINE "wam2D.m".
%
% 2. IF YOU INTEND TO ADD A FUNCTION, ADD IT TO "fct2D.m". IN PARTICULAR 
%    ADD A NEW CASE TO "switch function_type" IN "fct2D.m". 
%
% 3. IF YOU INTEND TO ADD A NEW POLYNOMIAL BASIS, SUITABLE FOR YOUR DOMAIN,
%    ADD IT TO "vandermonde2D.m". IN PARTICULAR ADD A NEW CASE TO "switch
%    basis_type" IN "vandermonde2D.m".
%
%--------------------------------------------------------------------------
% REFERENCE PAPERS.
%-------------------
%
% [1] L. BOS and N. LEVENBERG
%     "On the Approximate Calculation of Fekete Points: the Univariate Case"
%     ETNA.
%
% [2] L. BOS, N. LEVENBERG, J.P. CALVI, A. SOMMARIVA and M. VIANELLO
%     "Geometric Weakly Admissible Meshes, Discrete Least Squares
%      Approximations and Approximate Fekete Points". Math. Comp. (to
%      appear).
%
% [3] L.BOS, A. SOMMARIVA, M. VIANELLO. "Least-squares polynomial
%         approximation on weakly admissible meshes". Preprint.
%
% [4] J.P. CALVI and N. LEVENBERG
%     "Uniform approximation by discrete least squares polynomials".
%     J. Approx. Theory 152 (2008), 82-100.
%
% [5] A. SOMMARIVA and M. VIANELLO
%     "Gauss-Green cubature over spline curvilinear polygons".
%     J. Comp. Appl. Math.
%
% [6] A. SOMMARIVA and M. VIANELLO
%     "Computing approximate Fekete points by QR factorizations
%     of Vandermonde matrices". Comp. Math. Appl.
%
%--------------------------------------------------------------------------
% FUNCTIONS USED IN THIS ROUTINE.
%---------------------------------
%
% 1. wam2D
% 2. computing_fekete_points
% 3. vandermonde_2D
% 4. fct2D
%
% ALL THESE FUNCTIONS ARE EXTERNAL TO THIS ROUTINE.
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
%% Copyright (C) 2007-2009 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: March 01, 2010.
%--------------------------------------------------------------------------

% VANDERMONDE DEGREE.
M=10;

% 1: SQUARE.
wam_select=1;

% REFINEMENT ITERATIONS.
niter=2;

% 1: QR (AFEK). 2: LU (LEJA).
extraction_method=2;

% 1: TENSORIAL CHEBYSHEV BASIS.
basis_type=1;

%--------------------------------------------------------------------------
% [function_type]: TEST FUNCTION TO BE APPROXIMATED.
%
%                1: .75*exp(-((9*x-2).^2 + (9*y-2).^2)/4) + ...
%                    .75*exp(-((9*x+1).^2)/49 - (9*y+1)/10) + ...
%                    .5*exp(-((9*x-7).^2 + (9*y-3).^2)/4) - ...
%                    .2*exp(-(9*x-4).^2 - (9*y-7).^2);
%
%                2: ( (x-0.5).^2 +(y-0.5).^2 ).^(1/2);
%                3: (x+y).^20;
%                4: exp((x-0.5).^2+(y-0.5).^2);
%                5: exp(-100*((x-0.5).^2+(y-0.5).^2));
%                6: cos(30*(x+y));
%                7: ones(size(x));
%                8: exp(x+y);
%                9: 1./[1+16*(x.^2+y.^2)];
%                10: (x.^2+y.^2).^(3/2);
%                11: exp(-x.^2-y.^2);
%                12: exp(-[ (x.^2+y.^2) ]^(-1))
%                13: cos(x.^2+y.^2);
%                14: cos((x+y));
%                15: ((x-0.4)^2 +(y-0.4)^2 )^(1/2);
%--------------------------------------------------------------------------
function_type=11;

%--------------------------------------------------------------------------
%               ...... HERE THE MAIN PROGRAM BEGINS ......
%-------------------------------------------------------------------------

V_fek=[]; leb_const=[]; ls_norm=[]; % INITIALIZATIONS.

%--------------------------------------------------------------------------
% GENERATING POINTS OF THE WEAKLY ADMISSIBLE MESH. THE SET IS GIVEN AS
% [nodes_x,nodes_y]. WEIGHTS "weights" OF A CUBATURE RULE ARE ALSO PROVIDED.
%--------------------------------------------------------------------------

% COMPUTING WAM.
[nodes_x,nodes_y,rect]=wam2D(M,wam_select);

% WARNING: LARGE WAM.
if length(nodes_x) > 40000

    fprintf('\n \n \t >> [WARNING]: LARGE WAM (%5.0f POINTS).',...
        length(nodes_x));
    fprintf('\n \t               THE ALGORITHM MAY BE TIME CONSUMING.');

    answer=input('\n \n >> PRESS "1" TO CONTINUE, "0" TO STOP: ');
    if answer == 1
    else
        return
    end

end

pts_wam=[nodes_x nodes_y];

% VANDERMONDE AT WAM POINTS.
V_wam=vandermonde_2D(basis_type,pts_wam,M,rect);

% COMPUTING APPROXIMATE FEKETE POINTS.
[fek,ind,w_intp,C,Pmat]=computing_fekete_points(pts_wam,V_wam,...
    niter,[],extraction_method);

%--------------------------------------------------------------------------
% INTERPOLATION.
%--------------------------------------------------------------------------

% VANDERMONDE FOR INTP AT APPR. FEKETE PTS.
V_fek=vandermonde_2D(basis_type,fek,M,rect);

f_fek=fct2D(fek(:,1),fek(:,2),function_type);

% SOLVING LINEAR SYSTEM TO COMPUTE INTP. COEFFICIENTS.
coeffs_fek=V_fek\f_fek;


%--------------------------------------------------------------------------
% LEAST SQUARES.
%--------------------------------------------------------------------------

% COMPUTING LEAST SQUARES AT WAM POINTS.

% VANDERMONDE FOR INTP AT WAM POINTS.
V_ls=vandermonde_2D(basis_type,[nodes_x nodes_y],M,...
    rect);

% FUNCTION EVALUATION.
f_nodes=fct2D(nodes_x,nodes_y,function_type);

% LEAST SQUARE SOLUTION BY ITERATION (IF REQUIRED).
coeffs_ls=Pmat*((V_ls*Pmat)\f_nodes);

% -------------------------------------------------------------------------
% TESTING ERRORS.
%--------------------------------------------------------------------------

%-----------------------
% APPROXIMATION ERRORS.
%-----------------------

% TEST PTS.
[test_nodes_x,test_nodes_y,rect]=wam2D(M+1,...
    wam_select);

% VANDERMONDE MATRIX AT TEST POINTS.
V_test=vandermonde_2D(basis_type,[test_nodes_x test_nodes_y],M,rect);

% FUNCTION EVALUATION.
f_test=fct2D(test_nodes_x,test_nodes_y,function_type);

% EVALUATION OF THE INTERPOLANT AT TEST POINTS.
f_fek_test=V_test*coeffs_fek;

% EVALUATION LEAST SQUARES APPROXIMANT AT TEST POINTS.
f_ls_test=V_test*coeffs_ls;

% EVALUATION ABSOLUTE ERRORS IN INF. NORM.
abserr_inf_fek_test=norm(f_fek_test-f_test,inf);
abserr_inf_ls_test=norm(f_ls_test-f_test,inf);

% EVALUATION ABSOLUTE ERRORS IN WEIGHTED 2 NORM.
aver_abserr_2_fek_test=...
    norm(f_fek_test-f_test,2)/sqrt(length(f_fek_test));
aver_abserr_2_ls_test=norm(f_ls_test-f_test,2)/sqrt(length(f_fek_test));



%--------------------------------------------------------------------------
% STATISTICS.
%--------------------------------------------------------------------------

%-------------------
% BASIC STATISTICS.
%-------------------
fprintf('\n \t [DEGREE]: %2.0f', M);
fprintf(' [BASIS ]: %2.0f', basis_type);
fprintf('\n \t [ITERS.]: %2.0f', niter);
fprintf(' [AF.PTS]: %2.0f', size(fek,1));
fprintf(' [IN.PTS]: %2.0f', size(nodes_x,1));
fprintf(' [WAM]: %2.0f', wam_select);
if (extraction_method == 1)
    fprintf('\n \t [EXTRACTION METHOD]: QR');
else
    fprintf('\n \t [EXTRACTION METHOD]: LU');
end

%----------------------
% APPROXIMATION ERRORS.
%----------------------
fprintf('\n \n \t >> APPROXIMATION STATS. \n');
fprintf('\n \t [FUNCTION]: %2.0f',function_type);
fprintf('\n \t [ABS.ERR.APPROX.FEK.LS.(INF.NORM.)]: %2.0e',...
    abserr_inf_ls_test);

fprintf('\n \t [ABS.ERR.APPROX.FEK.INTP.(INF.NORM.)]: %2.0e',...
    abserr_inf_fek_test);

%------------------------------------------
% LEBESGUE CONSTANT AND LEAST SQUARES NORM.
%------------------------------------------

%**************************************************************************
% Here we compute the Lebesgue constant and the least square norm, by
% evaluating the Lebesgue function and the norms on wams of larger degree.
% In case these wams are too large, we try to evaluate the Lebesgue
% constant and the least square norm on small meshes.
%**************************************************************************

fprintf('\n \n \t >> LEBESGUE CONSTANTS.');
[leb_M,leb1,wam_size,ls_norm]=computing_lebesgue_constant(basis_type,M,...
    wam_select,Pmat,ind,C,V_ls);

fprintf('\n \n \t [LEB. CONST. AT APPR. FEK. PTS]');
for ii=1:length(leb1)
    fprintf('\n \t [M]: %4.0f [PTS]: %6.0f [LEB]: %2.3e',...
        leb_M(ii),wam_size(ii),leb1(ii));
end

fprintf('\n \n \t [LEAST SQUARES INF NORM]');
for ii=1:length(ls_norm)
    fprintf('\n \t [M]: %4.0f [PTS]: %6.0f [NRM]: %2.3e',...
        leb_M(ii),wam_size(ii),ls_norm(ii));
end

leb_const=max(leb1);
ls_norm=max(ls_norm);

%------------------------------------------
% CONDITIONING OF THE VANDERMONDE MATRIX
% (AFTER A POSSIBLE ORTHOGONALIZATION).
%------------------------------------------

fprintf('\n \n \t >> VANDERMONDE CONDITIONING. \n');

% spectral condition of the relevant Vandermonde matrices
vander_cond=cond(C);
extract_cond=cond((C(:,ind))');
fprintf('\n \t [VANDERMONDE COND. FULL WAM]: %2.2e',vander_cond);
fprintf('\n \t [VANDERMONDE COND. AFEK.PTS]: %2.2e',extract_cond);


fprintf('\n \n');

