
function V=vandermonde_2D(basis_type,pts,M,rect)

%--------------------------------------------------------------------------
% OBJECT:
%---------
%
% THE MAIN PURPOSE OF THIS ROUTINE IS TO COMPUTE THE VANDERMONDE MATRIX OF
% DEGREE n WITH RESPECT TO A CERTAIN BASIS OF THE SPACE OF BIVARIATE
% POLYNOMIALS (IN SOME CURVILINEAR SPLINE CASES, THE BASIS IS SCALED ON THE
% RECTANGLE rect) ON THE POINTS pts.
%
% THE BASIS IS DETERMINED BY basis_type.
%
%--------------------------------------------------------------------------
% INPUTS:
%---------
%
% basis_type : DETERMINES THE BASIS.
%
%      [basis_type = 0] TENSORIAL 3D CHEBYSHEV BASIS ON THE parallelepiped.
%
% pts: POINTS IN WHICH THE VANDERMONDE MATRIX "V" MUST BE EVALUATED.
%
% M : DEGREE OF THE VANDERMONDE BASIS.
%
% rect: REFERENCE RECTANGLE.
%
%----------
% OUTPUTS:
%----------
%
% V: VANDERMONDE MATRIX, WRT THE BASIS DETERMINED BY basis_type, EVALUATED
%    IN THE POINTS THAT ARE THE COMPONENTS OF THE VECTOR pts.
%
%--------------------------------------------------------------------------
% HOW TO ADD YOUR ON BASIS.
%---------------------------
%
%  IF YOU INTEND TO ADD A NEW POLYNOMIAL BASIS, SUITABLE FOR YOUR DOMAIN,
%  ADD IT TO "vandermonde2D.m". IN PARTICULAR ADD A NEW CASE TO "switch
%  basis_type" IN "vandermonde2D.m".
%
%--------------------------------------------------------------------------
%% Copyright (C) 2007-2009 Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Author:  Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: March 01, 2010.
%--------------------------------------------------------------------------

switch basis_type

    case 1 % TENSORIAL CHEBYSHEV VANDERMONDE.
        V = chebvand2d(M,pts,rect);
end



%--------------------------------------------------------------------------
% CHEBVAND 2D
%--------------------------------------------------------------------------

function V = chebvand2d(deg,mymesh,rect)

%--------------------------------------------------------------------------
% OBJECT.
%---------
% It computes the Chebyshev-Vandermonde matrix at a 3d mesh, in the product
% Chebyshev basis of a parallelepiped containing the mesh points.
%--------------------------------------------------------------------------
% INPUTS:
%---------
% deg: polynomial degree.
%
% mymesh: 2-columns array of mesh points coordinates.
%
% paral:   components vector such that the rectangle
%
%          [paral(1),paral(2)] x [paral(3),paral(4)]
%
%          contains the mesh.
%--------------------------------------------------------------------------
% OUTPUTS:
%---------
% V: Chebyshev-Vandermonde matrix
%--------------------------------------------------------------------------
%%
%% Copyright (C) 2010 Giordano Orzetti, Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors: Giordano Orzetti
%%          Alvise Sommariva <alvise@euler.math.unipd.it>
%%          Marco Vianello   <marcov@euler.math.unipd.it>
%%
%% Date: February 28, 2010
%%
%--------------------------------------------------------------------------

% Triples with length less or equal to "deg".
j=(0:1:deg);
[j1,j2]=meshgrid(j,j);
good=find(j1(:)+j2(:)<deg+1);
triples=[j1(good) j2(good)];

% Mapping the mesh in the cube [-1,1]^3
a=rect(1);b=rect(2);
c=rect(3);d=rect(4);
map=[(2*mymesh(:,1)-b-a)/(b-a) (2*mymesh(:,2)-d-c)/(d-c)];;

% Chebyshev-Vandermonde matrix at the mesh.
V1=cos(triples(:,1)*acos(map(:,1)'));
V2=cos(triples(:,2)*acos(map(:,2)'));
V=(V1.*V2)';



