function [fek,jvec,dbox] = dAPPROXFEKsph(deg,X,leja,dbox,domain_structure,...
    dimpoly)

%--------------------------------------------------------------------------
% Object:
% This routine extracts approximate Fekete or Leja interpolation
% points from a d-dimensional point cloud "X".
%--------------------------------------------------------------------------
% Input:
% deg: interpolation degree;
% X: d-column array of points coordinates, from which the approximate
%    Fekete or Leja points will be extracted;
% * leja: 0: computes approximate Fekete points,
%         1: computes approximate Leja points;
% * dbox: variable that defines a hyperrectangle with sides parallel to
%    the axis, containing the domain (or pointset X in the discrete case).
%    If "dbox" is not provided, it defines the smaller "hyperrectangle",
%    with sides parallel to the cartesian axes, containing the pointset
%    "X".
%    It is a matrix with dimension "2 x d", where "d" is the dimension
%    of the space in which it is embedded the domain.
%    For instance, for a 2-sphere, it is "d=3", while for a 2 dimensional
%    polygon it is "d=2".
%    As example, the set "[-1,1] x [0,1]" is described as "[-1 0; 1 1]".
% * domain_structure: structure defining the domain,
%    domain_struct.domain: string with domain name
%    domain_struct.parms: domain parameters.
% * dim_poly: dimension of polynomial space (if known in advance, useful
%   for instance on the sphere or its portions). If not available, set
%   " dim_poly='' " or do not declare the variable.
%
% Note: the variables with an asterisk "*" are not mandatory and can be
% also set as empty matrix.
%--------------------------------------------------------------------------
% Output:
% fek: d-column array of approximate Fekete or Leja points coordinates;
%      its cardinality is equal to the dimension of the space of algebraic
%      polynomials of total degree inferior or equal to "deg";
% jvec: vector of column indices, selects a polynomial basis;
% dbox: d-dim box containing X (and possibly the domain, see above);
%--------------------------------------------------------------------------
% Data:
% Written on 26/07/2020 by M.Dessole, F.Marcuzzi, M.Vianello.
% Modified on: 10/01/2021: M.Dessole, F.Marcuzzi, A.Sommariva, M.Vianello.
%--------------------------------------------------------------------------

% ........................... Function body ...............................

if nargin < 3, leja=[]; end
if nargin < 4, dbox=[]; end
if nargin < 5, domain_structure=[]; end
if nargin < 6, dimpoly=[]; end
if isempty(domain_structure), domain_structure.domain='generic'; end
if isempty(leja), leja=0; end



% ..... main code below .....

% ..... approximate Fekete or Leja points extraction from X .....


% polynomial basis orthonormalization
[~,jvec,V,~,dbox0] = dORTHVANDsph(deg,X,ones(size(X,1),1),[],[]);
% dimension of the polynomial space
dim = length(jvec);


% extracting the interpolation points
if leja == 0
    % approximate Fekete points and corresponding weights
    % by QR factorization with column pivoting of Q'
    w=V'\ones(dim,1);
    ind=find(abs(w) > 0);
    fek=X(ind,:);
else
    % approximate Leja points and corresponding weights
    % by LU factorization with row pivoting of Q
    [~,~,perm]=lu(V,'vector');
    fek=X(perm(1:dim),:);
end

if isempty(dbox) == 0, dbox=dbox0; end



