
function demo_am_01

%--------------------------------------------------------------------------
% Object:
% Demo on using "dPOLYFIT", "dPOLYVAL", "dLEB" for algebraic polynomial
% interpolation/least squares approximation on spherical triangle.
%
% The approximation points are taken from AMs on the spherical triangle.
% The new routine introduced here is "am_sphtri" that determines an
% Admissible Mesh over a spherical triangle.
%--------------------------------------------------------------------------
% Reference paper:
% "Chebyshev-Dubiner norming grids and Fekete-like interpolation on
% spherical triangles",
% A. Sommariva and M. Vianello,
% Department of Mathematics, University of Padova, Italy.
%--------------------------------------------------------------------------
% Dates:
% Written on 15/01/2021: A. Sommariva and M. Vianello.
% Modified on 15/02/2021 by A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------

clear; clf;

%--------------------------------------------------------------------------
% Function to study. The variable "function_typeV" can be:
%
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%   g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%   f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%   x0=centroid(1); y0=centroid(2); z0=centroid(3);
%   g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%   f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%   f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%   x0=centroid(1); y0=centroid(2); z0=centroid(3);
%   f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%--------------------------------------------------------------------------

function_typeV=1:6;


%--------------------------------------------------------------------------
% "pts_type_parm" can be one of the following strings:
%
%  case 1, pts_type='afek';
%  case 2, pts_type='leja';
%  case 3, pts_type='tchakaloff-near-optimal';
%  case 4, pts_type='tchakaloff-compression';
%  case 5, pts_type='standard-least-squares';
%  otherwise, pts_type='afek';
%--------------------------------------------------------------------------

pts_type_parm=1;


%--------------------------------------------------------------------------
% Degrees in numerical experiments: can be a vector.
%--------------------------------------------------------------------------

nV=1:20; % do not choose larger than 15 (time consuming!)


%--------------------------------------------------------------------------
% domain_example: determines the spherical triangle domain to be analysed.
%   From 0 to 7, going from large domains to tiny ones (i.r. the higher
%   "example", the smaller the domain.
%--------------------------------------------------------------------------

domain_example=0;


%--------------------------------------------------------------------------
% AM_factor: choose value in the interval [5,10], the larger the better,
% but also pointset cardinality is higher and may become prohibitive.
%--------------------------------------------------------------------------

AM_factor=7;





% ........................ Main code below ................................

% ....... Apply settings to define domain, pointsets, and functions .......

% ... define domain ...
[vertices,domain_struct]=define_domain(domain_example);
domain_struct.domain='spherical-triangle';


% ... define pointsets type ...
switch pts_type_parm
    case 1, pts_type='afek';
    case 2, pts_type='leja';
    case 3, pts_type='tchakaloff-near-optimal';
    case 4, pts_type='tchakaloff-compression';
    case 5, pts_type='standard-least-squares';
    otherwise, pts_type='afek';
end


% ... reference cubature set (used in L2 error norms) ...
P1=(vertices(1,:))'; P2=(vertices(2,:))'; P3=(vertices(3,:))';
[XYZWR]=cub_sphtri(40,P1,P2,P3); XR=XYZWR(:,1:3); wr=XYZWR(:,4);

% ........ Numerical approximation, varying the degree in "nV" ............
ptsH={};
for k=1:length(nV)
    n=nV(k); card_sphharm(k)=(n+1)^2;
    dimpoly=[];
    
    tic;
    
    % .......... defining initial pointset ..........
    X = am_sphtri(n,vertices,AM_factor);
    
    dbox=[min(X(:,1)) max(X(:,1));
        min(X(:,2)) max(X(:,2));
        min(X(:,3)) max(X(:,3))]';
    
    %  .......... extract interesting set (e.g. interpolation set) ........
    
    [pts,weights,jvec,dbox]=define_source_pointset(pts_type,n,X,dbox,...
        dimpoly,domain_struct);
    
    cpus(k)=toc;
    
    ptsH{end+1}=pts;
    
    % ... study each function on the pointset "pts"...
    
    for j=1:length(function_typeV)
        
        %if size(pts,1) == (n+1)^2
        function_type=function_typeV(j);
        
        % Function to approximate
        [g,gs]=define_function(function_type,vertices);
        
        % ... evaluate function to approximate ...
        g_pts=feval(g,pts(:,1),pts(:,2),pts(:,3));
        
        %  .......... determine polynomial interpolant/approximant  ...
        [coeff,R,jvec,dbox] = dPOLYFITsph(n,pts,weights,g_pts,jvec,[]);
        
        %  ..........  evaluate approximant at initial pointset  ......
        p_XR=dPOLYVALsph(n,coeff,XR,R,jvec,dbox);
        
        %  ..........  estimating interpolation error  ..........
        g_XR=feval(g,XR(:,1),XR(:,2),XR(:,3));
        
        AE_L2err(k,j)=sqrt(wr'*(p_XR-g_XR).^2);
        
        % ... estimating filtered relative error in infinity norm ...
        AE_inf(k,j)=norm(p_XR-g_XR,inf);
        RE_inf(k,j)=AE_inf(k,j)/norm(g_XR,inf);
        %         else
        %             AE_L2err(k,j)=NaN;
        %             AE_inf(k,j)=NaN;
        %             RE_inf(k,j)=NaN;
        %         end
        
    end
    
    
    
    
    % .......... parameters for statistics ..........
    [XYZW0]=cub_sphtri(2*n,P1,P2,P3); X0=XYZWR(:,1:3); w0=XYZWR(:,4);
    leb(k)=dLEBsph(n,pts,weights,X0,jvec,dbox);
    
    card_X(k)=size(X,1); card_pts(k)=size(pts,1);
    card_polyspace(k)=length(jvec);
    
    
    
    % ... iteration statistics ...
    
    if k == 1
        fprintf('\n \t .........................................................................');
        fprintf('\n \t |  n  |   leb   |   card0  |   card  |  poly |  cpus   |');
        fprintf('\n \t .........................................................................');
    end
    
    
    if card_sphharm(k)-card_pts(k) == 0
        fprintf('\n \t | %3.0f | %1.1e |   %5.0f  |   %5.0f | %5.0f | %1.1e | ',...
            nV(k),leb(k),card_X(k),card_pts(k),card_polyspace(k),cpus(k));
    else
        fprintf('\n \t | %3.0f | %1.1e |   %5.0f  |   %5.0f | %5.0f | %1.1e | *',...
            nV(k),leb(k),card_X(k),card_pts(k),card_polyspace(k),cpus(k));
    end
    
end










% ............................... Plots ...................................
%clf(1);

figure(1);
plot(nV,leb,'-o'); hold on; title('Lebesgue constant'); hold off;

% clf(2);
figure(2)
title_str='Approximation pointset';
RR=norm((vertices(1,:))); % sphere radius
plot_s2(domain_struct.domain,domain_struct.parms,[],pts,title_str,RR);

% ....... figure 2 (L2 hyp. error) .......
% clf(3);
figure(3);
title('L2 relative errors');
all_marks = {'o','v','*','d','x','s','.','^','+','>','<','p','h'};

for j=1:length(function_typeV)
    
    % ... approximating L2 norm of function "g" ...
    function_type=function_typeV(j);
    [g,gs]=define_function(function_type,vertices);
    g_X=feval(g,XR(:,1),XR(:,2),XR(:,3));
    g_L2=sqrt(wr'*(g_X).^2);
    
    % ... computing L2 hyperinterpolation relative error ...
    RE_L2err(:,j)=AE_L2err(:,j)/g_L2;
    
    % ... plotting error ...
    
    marker_type=all_marks{j};
    semilogy(nV,RE_L2err(:,j),'LineWidth',1.5,'Marker',marker_type);
    hold on;
    
end
L=length(function_typeV);
if L == 6, legend('f1','f2','f3','f4','f5','f6'); end
hold off;




%  .......... write additional information on file ........

filename0=strcat('octant_',pts_type,num2str(AM_factor));
filename=strcat(filename0,'.m');
delete(filename);
fid=fopen(filename,'w');
fprintf(fid,strcat('function  [pts,leb,AE_L2err,AE_inf,RE_inf]=',...
    filename0,'(deg)'));
fprintf(fid,'\n \n \n switch deg');

%  .......... write pointset on file ........
for jj=1:length(ptsH)
    nL=nV(jj);
    pts=ptsH{jj};
    fprintf(fid,'\n \t case %2.0f',nL);
    fprintf(fid,'\n \t pts=[');
    for kk=1:size(pts,1)
        fprintf(fid,'\n \t %1.15e  %1.15e  %1.15e ',...
            pts(kk,1),pts(kk,2),pts(kk,3));
    end
    fprintf(fid,'\n \t ]; \n \n');
end

fprintf(fid,'\n \t end \n \n');

fprintf(fid,'\n \t leb=[');
for k=1:length(leb)
    fprintf(fid,'\n \t %1.15e ',leb(k));
end
fprintf(fid,'\n \t ]; \n \n');



fprintf(fid,'\n \t AE_L2err=[');
for k=1:length(nV)
    fprintf(fid,'\n \t');
    fprintf(fid,'%1.3e ',(AE_L2err(k,:))');
end
fprintf(fid,'\n \t ]; \n \n');


fprintf(fid,'\n \t RE_L2err=[');
for k=1:length(nV)
    fprintf(fid,'\n \t');
    fprintf(fid,'%1.3e ',(RE_L2err(k,:))');
end
fprintf(fid,'\n \t ]; \n \n');


fprintf(fid,'\n \t AE_inf=[');
for k=1:length(nV)
    fprintf(fid,'\n \t');
    fprintf(fid,'%1.3e ',AE_inf(k,:));
end
fprintf(fid,'\n \t ]; \n \n');


fprintf(fid,'\n \t RE_inf=[');
for k=1:length(nV)
    fprintf(fid,'\n \t');
    fprintf(fid,'%1.3e ',RE_inf(k,:));
end
fprintf(fid,'\n \t ]; \n \n');

sfin=fclose(fid);










% ............................... Statistics ..............................

% Area (if the domain is a spherical triangle)

if strcmp(domain_struct.domain,'spherical-triangle') == 1
    area_sphtri=sum(wr);
else
    area_sphtri=NaN;
end

fprintf('\n \t .........................................................................');
domain_example_str=strcat(' * Domain:',' ',num2str(domain_example));
fprintf('\n \t'); disp(domain_example_str);
pts_type_str=strcat(' * Pointset: ',pts_type);
fprintf(' \t'); disp(pts_type_str);

fprintf('\n \t .........................................................................');
fprintf('\n \t Legend:');
fprintf('\n \t * leb   : Lebesgue constant;');
fprintf('\n \t * card0 : cardinality of initial pointset (numerical WAM);');
fprintf('\n \t * card  : cardinality of extracted pointset;');
fprintf('\n \t * poly  : dimension of space of total polynomials on the sphere;');
fprintf('\n \t * cpus  : cputime required for computing extracted pointset.');
fprintf('\n \t .........................................................................');
fprintf(' \n \t * Function analysed (see figure 3 for L2 relative errors)): \n ');
for j=1:length(function_typeV)
    
    function_type=function_typeV(j);
    
    % Function to approximate
    [g,gs]=define_function(function_type,vertices);
    function_str=strcat(num2str(function_type),': f(x,y)=@(x,y) ',gs);
    fprintf(' \t '); disp(function_str);
    
end

fprintf(' \t .........................................................................');
fprintf('\n \n');





% ..... L2 Error Statistics .....

fprintf('\n \t .........................................................................');
fprintf('\n \t For each function "f" and each degree we display the L2 relative weighted');
fprintf('\n \t hyperinterp. errors REL, evaluated w.r.t. to a high order cubature.');
fprintf('\n \t Next we compute RELinf, i.e. "maximum absolute error / norm(f,inf)".');
fprintf('\n \t .........................................................................');
for j=1:length(function_typeV)
    function_type=function_typeV(j);
    [g,gs]=define_function(function_type,vertices);
    fprintf('\n \n \n \t *** ');disp(gs);
    fprintf('\n \t .....................................');
    fprintf('\n \t |  deg  |   REL2   |  RELinf  |');
    fprintf('\n \t .....................................');
    for k=1:length(nV)
        n=nV(k);
        fprintf('\n \t |  %3.0f  | %1.2e | %1.2e |',n,RE_L2err(k,j),...
            RE_inf(k,j));
    end
    fprintf('\n \t .....................................');
end

fprintf('\n \n');









function [pts,weights,jvec,dbox]=define_source_pointset(pts_type,n,X,dbox,...
    dimpoly,domain_structure)

%--------------------------------------------------------------------------
% Object:
% This routine, defines the pointset to be extracted from "n" for
% interpolation/approximation at degree "n".
%--------------------------------------------------------------------------
% Input:
% pts_type: variable defining the property of extracted set:
%      case 1, pts_type='afek';
%      case 2, pts_type='leja';
%      case 3, pts_type='tchakaloff-near-optimal';
%      case 4, pts_type='tchakaloff-compression';
%      case 5, pts_type='standard-least-squares';
%      otherwise, pts_type='afek';
% n : interpolation or approximation degree.
% X : initial pointset, in which the k-th row describes the k-th point in
%     cartesian coordinates.
% * dbox: variable that defines a hyperrectangle with sides parallel to the
%    axis, containing the domain (or pointset "nodes" in the discrete
%    case).
%    If "dbox" is not provided, it is the smaller "hyperrectangle", with
%    sides parallel to the cartesian axes, containing the pointset "nodes".
%    It is a matrix with dimension "2 x d", where "d" is the dimension of
%    the space in which it is embedded the domain.
%    For instance, for a 2-sphere, it is "d=3", for a 2 dimensional
%    polygon it is "d=2".
%    As example, the set "[-1,1] x [0,1]" is described as
%                          "dbox=[-1 0; 1 1]".
% * dimpoly: dimension of polynomial space (may vary on domains)
% * domain_structure: structure defining the domain,
%    domain_struct.domain: string with domain name
%    domain_struct.parms: domain parameters.
%
% Note: the variables with an asterisk "*" are not mandatory and can be
% also set as empty matrix.
%--------------------------------------------------------------------------
% Output:
% pts: if pts are in R^d, it is a d-column array, containing the
%      coordinates of the extracted pointset, in which the k-th row
%      describes the k-th point in cartesian coordinates.
%      its cardinality is equal to the dimension of the space of algebraic
%      polynomials of total degree inferior or equal to "deg";
% weights: 1-column array of nonnegative weights, or nonnegative scalar in
%    case of equal weights (useful in 'tchakaloff-near-optimal' and
%    'tchakaloff-compression' setting);
% jvec: vector of column indices, selects a polynomial basis;
% dbox: variable that defines the d-dim box where to adapt the
%       product-Chebyshev basis.
%--------------------------------------------------------------------------


if nargin < 3, dbox=[]; end
if nargin < 4, dbox=[]; end
if nargin < 5, domain_structure='generic'; end


switch pts_type
    case 'afek'
        %% APPROXIMATE FEKETE INTERPOLATION POINTS
        [pts,jvec,dbox] = dAPPROXFEKsph(n,X,0,dbox,domain_structure,...
            dimpoly);
        weights=ones(length(pts(:,1)),1);
        
    case 'leja'
        %% DISCRETE LEJA INTERPOLATION POINTS
        [pts,jvec,dbox] = dAPPROXFEKsph(n,X,1,dbox,domain_structure,...
            dimpoly);
        weights=ones(length(pts(:,1)),1);
        
    case 'tchakaloff-near-optimal'
        %% TCHAKALOFF NEAR-OPTIMAL REGRESSION POINTS
        [pts,weights,~,~,dbox]=dNORD(n,X,dimpoly);
        jvec=[];
        
    case 'tchakaloff-compression'
        %% TCHKALOFF COMPRESSION OF STANDARD LEAST SQUARES
        [pts,weights,~,dbox]=dCATCH(2*n,X,dimpoly);
        jvec=[];
        
    case 'standard-least-squares'
        %% STANDARD LEAST SQUARES
        pts=X;
        weights=ones(length(X(:,1)),1);
        jvec=[];
        dbox=[];
        
        
    otherwise
        warning('Pointset not implemented, using Approx. Fekete points');
        %% APPROXIMATE FEKETE INTERPOLATION POINTS
        [pts,jvec,dbox] = dAPPROXFEK(n,X,0,dbox,dimpoly);
        weights=ones(length(fek(:,1)),1);
        
end

if isempty(dbox)
    dbox=[];
    L=size(X,2);
    for k=1:L
        mx=min(X(:,k)); Mx=max(X(:,k)); dboxL=[mx; Mx]; dbox=[dbox dboxL];
    end
end









function [f,fs]=define_function(function_type,parms)

%--------------------------------------------------------------------------
% Object:
% This routine, defines a function "f" to approximate and a string "fs" for
% possible messages to the user.
%--------------------------------------------------------------------------
% Input:
% function_type: determines the function to study.
% The first five functions has been used in the paper mentioned below.
%
% case 1, f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
% case 2, f=@(x,y,z) cos(10*(x+y+z));
% case 3  x0=0; y0=0; z0=1;
%   g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%   f=@(x,y,z) exp( - g(x,y,z) );
% case 4  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%   x0=centroid(1); y0=centroid(2); z0=centroid(3);
%   g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
%   f=@(x,y,z) exp( - g(x,y,z) );
% case 5  x0=0; y0=0; z0=1;
%   f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
% case 6  centroid=sum(vertices,1); centroid=centroid/norm(centroid);
%   x0=centroid(1); y0=centroid(2); z0=centroid(3);
%   f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
%
% parms: additional parameters given by the user, e.g. triangle vertices.
%--------------------------------------------------------------------------
% Output:
% f: defines a function "f" to approximate;
% fs: string with the content of the function "f".
%--------------------------------------------------------------------------
% Reference paper:
% A. Sommariva and M. Vianello
% Numerical hyperinterpolation over spherical triangles.
%--------------------------------------------------------------------------
% Dates:
% Written on 13/01/2021: A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------
switch function_type
    case 1 % Fornberg
        f=@(x,y,z) 1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
        fs='1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2';
        
    case 2
        f=@(x,y,z) cos(10*(x+y+z));
        fs='cos(10*(x+y+z))';
        
    case 3 % exp and - square north pole distance
        x0=0; y0=0; z0=1;
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        fs='exp(-g(x,y,z)),g=@(x,y,z) (x-x0)^2+(y-y0)^2+(z-z0)^2,x0=0;y0=0;z0=1;';
        
    case 4 % exp and - square centroid distance
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        g=@(x,y,z) (x-x0).^2 + (y-y0).^2 + (z-z0).^2;
        f=@(x,y,z) exp( - g(x,y,z) );
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2';
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='exp(-g(x,y,z)), g=@(x,y,z) (x-x0).^2+(y-y0).^2+(z-z0).^2';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
    case 5 % north pole distance like
        x0=0; y0=0; z0=1;
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), x0=0; y0=0; z0=1;';
        
    case 6 % centroid distance like
        if nargin < 1
            vertices=[0 0 1; 1 0 0; 0 1 0];
        else
            vertices=parms;
        end
        
        % ... centroid computation ...
        centroid=sum(vertices,1); centroid=centroid/norm(centroid);
        x0=centroid(1); y0=centroid(2); z0=centroid(3);
        
        % ... function ...
        f=@(x,y,z) ((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2);
        
        % ... output string ...
        x0str=num2str(x0,'%1.3e');
        y0str=num2str(y0,'%1.3e');
        z0str=num2str(z0,'%1.3e');
        fs='((x-x0).^2 + (y-y0).^2 + (z-z0).^2).^(5/2), ';
        fs=strcat(fs,'  centroid=(',x0str,',',y0str,',',z0str,')');
        
end  % end: "define_function"









function [vertices,domain_struct]=define_domain(example)

%--------------------------------------------------------------------------
% Object:
% This routine, defines the domain to analyse.
%--------------------------------------------------------------------------
% Input:
% example: determines the spherical triangle domain to be analysed. From 0
%   to 7, going from large domains to tiny ones (the higher "example", the
%   smaller the domain.
%--------------------------------------------------------------------------
% Output:
% vertices: it is a matrix whose rows are the vertices of the spherical
%     triangles.
% domain_struct: struct representing domain data.
%--------------------------------------------------------------------------

if nargin < 1
    warning('Undefined example, using unit orthant');
    example=0;
end

switch example
    case -1
        % orthant
        % LARGE SIZE (Asia+Europe+Australia)
        vertices=3*[0 0 1;
            1 0 0;
            0 1 0];
        
    case 0
        % orthant
        % LARGE SIZE (Asia+Europe+Australia)
        vertices=[0 0 1;
            1 0 0;
            0 1 0];
        
    case 1
        % vertices of the spherical-triangle, counterclockwise.
        a=0.9; b=0.5; % MEDIUM-LARGE SIZE (North-America)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 2
        % vertices of the spherical-triangle, counterclockwise.
        a=0.5; b=0.2; % MEDIUM-LARGE SIZE (Australia)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 3
        % vertices of the spherical-triangle, counterclockwise.
        a=0.1; b=0.2; % MEDIUM-SMALL SIZE (India)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 4
        % vertices of the spherical-triangle, counterclockwise.
        a=0.01; b=0.02; % SMALL SIZE (Italy)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 5
        % vertices of the spherical-triangle, counterclockwise.
        a=0.001; b=0.002; % VERY SMALL SIZE (Calabria+Campania)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 6
        % vertices of the spherical-triangle, counterclockwise.
        a=0.0001; b=0.0002; % TINY SIZE (Venice Province)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
    case 7  % medium
        theta0=pi/4; theta1=pi/3;
        vertices=[0 0 1;
            cos(theta0) 0 sin(theta0);
            0 cos(theta1) sin(theta1)];
        
    case 8 % small
        theta0=pi/2-pi/8; theta1=pi/2-pi/10;
        vertices=[0 0 1;
            cos(theta0) 0 sin(theta0);
            0 cos(theta1) sin(theta1)];
        
    case 9
        
        vertices=[ -3.189827176852499e-01 0
            3.216349554603996e-01 -4.082482904638630e-01
            -2.652237775149153e-03 4.082482904638631e-01 ];
        
        X=vertices(:,1); Y=vertices(:,2); Z=sqrt(1-X.^2-Y.^2);
        vertices=[X Y Z];
        
        
    case 10
        
        vertices=[ -3.189827176852499e-01 0
            3.216349554603996e-01 -4.082482904638630e-01
            -2.652237775149153e-03 4.082482904638631e-01 ];
        
        X=vertices(:,1); Y=vertices(:,2); Z=sqrt(1-X.^2-Y.^2);
        vertices=3*[X Y Z];
        
    otherwise
        % vertices of the spherical-triangle, counterclockwise.
        a=0.0000002; b=0.0000002; % VERY TINY SIZE (Murano Island)
        vertices=[0 0 1;
            0 sqrt(a) sqrt(1-a);
            sqrt(b) sqrt(a/2) sqrt(1-a/2-b)];
        
end


domain_struct.domain='spherical_triangle';
domain_struct.parms=vertices;








