
function demo_am_02

%--------------------------------------------------------------------------
% Object:
% This routine computes a AM on a spherical triangle and extracts AFEK
% pointset. Next it determines its main features and plots AM and AFEK.
%--------------------------------------------------------------------------
% Reference paper:
% Chebyshev-Dubiner norming grids and Fekete-like interpolation
% on spherical triangles 
% A. Sommariva and M. Vianello 
% Department of Mathematics, University of Padova, Italy
%--------------------------------------------------------------------------
% Dates:
% Written on 08/02/2021: A. Sommariva and M. Vianello.
% Modified on 12/02/20 by A. Sommariva and M. Vianello.
%--------------------------------------------------------------------------

% ............................. settings ..................................

% ... AM degree (on spherical triangle) ...
n=3;

% ... coordinates planar triangle determining the spherical counterpart ...
w_star=pi/3; z=cos(w_star); r=sin(w_star); th=[0; 2*pi/5; 4*pi/3];
Q=[r*cos(th) r*sin(th) z*ones(size(th))];



% ............................. experiment ................................

% .... determine AM ....
AM_factor=7; % AM degree factor (see )

tic; [am_sphtr,deg_am] = am_sphtri(n,Q,AM_factor); cpus(1)=toc;

% .... determine AFEK (approximate Fekete points) ....
tic; [afek_pts,w,momerr,dbox] = dCATCHsph(n,am_sphtr); cpus(2)=toc;

% .... approximate Lebesgue constant of AFEK ....
leb = dLEBsph(n,afek_pts,1,am_sphtr);

% ............................. stats .....................................
fprintf('\n \t AM degree: %4.0f',n);
fprintf('\n \t Cap angle: %1.5e',w_star);
fprintf('\n \t Subtriangles: %4.0f \n',length(deg_am));
% fprintf('\n \t actual degrees: min: %4.0f max: %4.0f',...
%     min(deg_am),max(deg_am));
fprintf('\n \t Cardinality AM  : %10.0f',size(am_sphtr,1));
fprintf('\n \t Cardinality AFEK: %10.0f',size(afek_pts,1));
fprintf('\n \t Lebesgue contant AFEK: %1.1e',leb);
fprintf('\n \t Moment error AFEK: %1.1e \n',momerr);
fprintf('\n \t Cputime: AM: %1.1e AFEK: %1.1e TOT: %1.1e \n ',...
    cpus(1),cpus(2),sum(cpus));

fprintf('\n \t .........................................................');
fprintf('\n \t Legend:');
fprintf('\n \t .........................................................');
fprintf('\n \t * Cap angle: colatitude angle of the smaller cap');
fprintf('\n \t              containing the spherical triangle;');
fprintf('\n \t * Subtriangles: the routine may subdivide the spherical');
fprintf('\n \t         triangle in smaller spherical triangles;');
fprintf('\n \t * Moment error AFEK: compression moment error in 2-norm.');
fprintf('\n \t .........................................................');
fprintf('\n \n');



% .............................. plot .....................................


clf;

% ................ plot domain ................
plot_sphtri(Q);

hold on;
% plot3(am_tri(:,1),am_tri(:,2),am_tri(:,3),'o','LineWidth',2,...
%                        'MarkerEdgeColor','k',...
%                        'MarkerFaceColor','r',...
%                        'MarkerSize',6)

% ................ plot AM ................

plot3(am_sphtr(:,1),am_sphtr(:,2),am_sphtr(:,3),'.',...
    'MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',2);

% ................ plot AFEK ................

plot3(afek_pts(:,1),afek_pts(:,2),afek_pts(:,3),'.',...
    'MarkerEdgeColor','b','MarkerFaceColor','b','MarkerSize',24);

% ................ plot circumcircle ................
[CC,rcirc,omega_star]=circumcenter(Q);
Rmat=rotation_northpole(CC); % rotation of CC to North Pole
thc=linspace(0,2*pi,1000); thc=thc';
XYZ_circNP=[rcirc*cos(thc) rcirc*sin(thc) norm(CC)*ones(size(thc))];


XYZ_circ=(Rmat'*XYZ_circNP')';


plot3(XYZ_circ(:,1),XYZ_circ(:,2),XYZ_circ(:,3),'m-',...
    'MarkerEdgeColor','m','MarkerFaceColor','m','MarkerSize',1,...
    'LineWidth',2);

hold off;











function plot_sphtri(x)

hold on;

R=1;

theta(1)=acos(x(1,3)/norm(x(1,:)));
theta(2)=acos(x(2,3)/norm(x(2,:)));
theta(3)=acos(x(3,3)/norm(x(3,:)));

theta_intv=[0 1.4*max(theta)];

funx = @(theta,phi) R*sin(theta).*cos(phi);
funy = @(theta,phi) R*sin(theta).*sin(phi);
funz = @(theta,phi) R*cos(theta);
fsurf(funx,funy,funz,[theta_intv(1) theta_intv(2) -pi pi]);
alpha 0.1 % the lower, the more transparent the region

% .... plot sph. triangle boundary ....
A=x(1,:); B=x(2,:); C=x(3,:);
plot_side(A,B); plot_side(B,C); plot_side(C,A);

% .... hide ticks, labels, axis and background ....

set(gca,'xtick',[]) % no ticks / labels
set(gca,'xticklabel',[])
set(gca,'ytick',[])
set(gca,'yticklabel',[])
set(gca,'ztick',[])
set(gca,'zticklabel',[])

set(gca,'color','none') % no background

set(gca,'Visible','off') % no axis









function plot_side(A,B)

%--------------------------------------------------------------------------
% Object:
% Plot "geodesic line" between A and B.
%--------------------------------------------------------------------------
% Input:
% A,B: row vectors of two points of the sphere to be joined.
%--------------------------------------------------------------------------

R=norm(A);
t=linspace(0,1,100);
C=[];
for k=1:length(t)
    CC=A+t(k)*(B-A); CC=CC/norm(CC); C=[C; R*CC];
end
plot3(C(:,1),C(:,2),C(:,3),'k-','LineWidth',1);









function R=rotation_northpole(CC)

% .................... rotation matrix to the north pole ..................

[az,el,r] = cart2sph(CC(1),CC(2),CC(3));
phi=az; theta=pi/2-el;
cp=cos(phi); sp=sin(phi); ct=cos(theta); st=sin(theta);
R1=[ct 0 -st; 0 1 0; st 0 ct]; R2=[cp sp 0; -sp cp 0; 0 0 1];
R=R1*R2;










function [CC,r,omega_star]=circumcenter(Q)

% https://en.wikipedia.org/wiki/Circumscribed_circle
% k-th row of Q is the k-th point

A=Q(1,:); B=Q(2,:); C=Q(3,:);
a=A-C;
b=B-C;

r=norm(a)*norm(b)*norm(a-b)/(2*norm(cross(a,b)));

t1=(norm(a))^2*b-(norm(b))^2*a;
t2=cross(a,b);
t3=norm(t2);
CC=C+cross(t1,t2)/(2*t3^2);

omega_star=asin(r);


