
function demo_figure_01A

%--------------------------------------------------------------------------
% Object:
% Radially mapped Chebyshev-Lobatto grid from the planar supporting
% triangle to a spherical triangle.
%--------------------------------------------------------------------------
% Reference:
% A. Sommariva and M. Vianello
% "Norming grids and polynomial interpolation on spherical triangles".
% See Figure 1.
%--------------------------------------------------------------------------

clear; clf;

warning off;

domain_type=1;

%--------------------------------------------------------------------------
% Degrees in numerical experiments: can be a vector.
%--------------------------------------------------------------------------

deg=10;


% ........................ Main code below ................................

% vertices: k-th represents k-th point cartesian coordinates
% Note: the triangles in our examples are at the north pole and
% "horizontal".
Q=define_triangle(domain_type);

% ... wam over a planar triangle ...
wam_triangle=wamquadrangle(deg,Q([1:3 1],1:2));

% ... wam on the planar triangle ...
wam_triangle=[wam_triangle Q(1,3)*ones(size(wam_triangle,1),1)];

% ... mesh on the spherical triangle ...
XX=wam_triangle(:,1); YY=wam_triangle(:,2); ZZ=wam_triangle(:,3);
norms_wam=sqrt(XX.^2+YY.^2+ZZ.^2);
wam_sphtriangle=[XX./norms_wam YY./norms_wam ZZ./norms_wam];

% ... lines defining the grids on the planar / spherical domain ...
[pts_tri,pts_sphtri,pts_tri00,pts_sphtri00,pts_BCsph]=...
    define_lines(Q,deg);
[pts_tri0,pts_sphtri0,pts_sphtri01,pts_ABsph,pts_ACsph]=...
    define_lines2(Q,deg);

% ........................ plot figure ....................................

hold on;

% ... plot mesh planar triangle ...
markersize_val=6;
color_dots='red';
plot3(wam_triangle(:,1),wam_triangle(:,2),wam_triangle(:,3),...
    'ko','Color',color_dots,'MarkerSize',markersize_val,...
    'MarkerEdgeColor',color_dots,'MarkerFaceColor',color_dots);

% ... plot mesh spherical triangle ...
color_dots='blue';
plot3(wam_sphtriangle(:,1),wam_sphtriangle(:,2),wam_sphtriangle(:,3),...
    'mo','Color',color_dots,'MarkerSize',markersize_val,...
    'MarkerEdgeColor',color_dots,'MarkerFaceColor',color_dots);

% ... plot lines connecting vertex A and BC ...
linewidth_val=1.8;
color_lines=[17 17 17]/255;

for s=1:(deg+1)
    
    pts_triL=pts_tri{s};
    pts_sphtriL=pts_sphtri{s};
    
    plot3(pts_triL(:,1),pts_triL(:,2),pts_triL(:,3),...
        'k:','Color',color_lines,'MarkerSize',markersize_val,...
        'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines,...
        'LineWidth',linewidth_val);
    plot3(pts_sphtriL(:,1),pts_sphtriL(:,2),pts_sphtriL(:,3),...
        'k:','Color',color_lines,'MarkerSize',markersize_val,...
        'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines,...
        'LineWidth',linewidth_val);
    
end

% ... plot lines connecting AC with AB ...

for s=1:(deg+1)
    
    pts_tri0L=pts_tri0{s};
    pts_sphtri0L=pts_sphtri0{s};
    
    plot3(pts_tri0L(:,1),pts_tri0L(:,2),pts_tri0L(:,3),...
        'k:','Color',color_lines,'MarkerSize',markersize_val,...
        'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines);
    
    plot3(pts_sphtri0L(:,1),pts_sphtri0L(:,2),pts_sphtri0L(:,3),...
        'k:','Color',color_lines,'MarkerSize',markersize_val,...
        'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines);
end

% ... plot spherical triangle ...
plot_sphtri(Q);

% ... plot circumcenter and its circle ...
[CC,r]=circumcenter(Q);
w=asin(r);
color_lines=[17 17 17]/255;
% plot3(CC(1),CC(2),Q(1,3),...
%      'mo','Color',color_lines,'MarkerSize',,...
%      'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines);

th=linspace(0,2*pi,100);
rV=[r*cos(th') r*sin(th') Q(1,3)*ones(size(th'))];
plot3(rV(:,1),rV(:,2),rV(:,3),...
    ':','Color',color_lines,'MarkerSize',markersize_val,...
    'MarkerEdgeColor',color_lines,'MarkerFaceColor',color_lines,...
        'LineWidth',linewidth_val);

%axis equal;
hold off;










function vertices=define_triangle(domain_type)

switch domain_type
    case 1
        d=0.4;
        r=sqrt(1-d^2);
        th=[0 (3/4)*pi (3/2)*pi]';
        vertices=[r*cos(th) r*sin(th) d*ones(size(th))];
        
    case 2
        d=0.8;
        r=sqrt(1-d^2);
        th=[0 (3/5)*pi (4/3)*pi]';
        vertices=[r*cos(th) r*sin(th) d*ones(size(th))];
        
    case 3
        d=0.8;
        r=sqrt(1-d^2);
        th=[0 (2/3)*pi (4/3)*pi]';
        vertices=[r*cos(th) r*sin(th) d*ones(size(th))];
        
end









function [pts_tri,pts_sphtri,pts_tri0,pts_sphtri0,pts_BCsph]=...
    define_lines(Q,deg)

% Chebyshev-Lobatto grid
j=(0:1:deg);
t=cos(j*pi/deg); t=((t+1)/2)';
tt=linspace(0,1,400); tt=tt';

A=Q(1,:);
B=Q(2,:);
C=Q(3,:);

Z0=A(:,3);

X=t*B(1)+(1-t)*C(1); Y=t*B(2)+(1-t)*C(2); Z=Z0*ones(size(X,1),1);
pts_BC=[X Y Z];
norm_pts_BC=sqrt(X.^2+Y.^2+Z.^2);
pts_BCsph=pts_BC./norm_pts_BC;

pts_tri={};
pts_sphtri={};
pts_tri0={};
pts_sphtri0={};

for k=1:length(t)
    D=pts_BC(k,:);
    
    X1=t*A(1)+(1-t)*D(1);
    Y1=t*A(2)+(1-t)*D(2);
    Z1=Z0*ones(length(t),1);
    pts_tri0{end+1}=[X1 Y1, Z1];
    
    norm_pts_tri=sqrt(X1.^2+Y1.^2+Z1.^2);
    X1sph0=X1./norm_pts_tri;
    Y1sph0=Y1./norm_pts_tri;
    Z1sph0=Z1./norm_pts_tri;
    pts_sphtri0{end+1}=[X1sph0 Y1sph0, Z1sph0];
    
    X1=tt*A(1)+(1-tt)*D(1);
    Y1=tt*A(2)+(1-tt)*D(2);
    Z1=Z0*ones(length(tt),1);
    
    pts_tri{end+1}=[X1 Y1, Z1];
    
    norm_pts_tri=sqrt(X1.^2+Y1.^2+Z1.^2);
    X1sph=X1./norm_pts_tri;
    Y1sph=Y1./norm_pts_tri;
    Z1sph=Z1./norm_pts_tri;
    pts_sphtri{end+1}=[X1sph Y1sph, Z1sph];
end










function [pts_tri,pts_sphtri,pts_sphtri01,pts_ABsph,pts_ACsph]=define_lines2(Q,deg)

%--------------------------------------------------------------------------
% Object:
%
%--------------------------------------------------------------------------
% Input:
%
%--------------------------------------------------------------------------

% Chebyshev-Lobatto grid
j=(0:1:deg);
t=cos(j*pi/deg); t=((t+1)/2)';
tt=linspace(0,1,400); tt=tt';

A=Q(1,:);
B=Q(2,:);
C=Q(3,:);

Z0=A(:,3);

X=t*A(1)+(1-t)*B(1); Y=t*A(2)+(1-t)*B(2); Z=Z0*ones(size(X,1),1);
pts_AB=[X Y Z];
pts_AB_norm=sqrt(X.^2+Y.^2+Z.^2);
pts_ABsph=pts_AB./pts_AB_norm;


X=t*A(1)+(1-t)*C(1); Y=t*A(2)+(1-t)*C(2); Z=Z0*ones(size(X,1),1);
pts_AC=[X Y Z];
pts_AC_norm=sqrt(X.^2+Y.^2+Z.^2);
pts_ACsph=pts_AC./pts_AC_norm;

pts_tri={};
pts_sphtri={};
pts_tri01={};
pts_sphtri01={};

for k=1:length(t)
    AB0=pts_AB(k,:);
    AC0=pts_AC(k,:);
    
    X1=tt*AB0(1)+(1-tt)*AC0(1);
    Y1=tt*AB0(2)+(1-tt)*AC0(2);
    Z1=Z0*ones(length(tt),1);
    
    pts_tri{end+1}=[X1 Y1, Z1];
    
    norm_pts_tri=sqrt(X1.^2+Y1.^2+Z1.^2);
    X1sph=X1./norm_pts_tri;
    Y1sph=Y1./norm_pts_tri;
    Z1sph=Z1./norm_pts_tri;
    pts_sphtri{end+1}=[X1sph Y1sph, Z1sph];
    
    
    X1=t*AB0(1)+(1-t)*AC0(1);
    Y1=t*AB0(2)+(1-t)*AC0(2);
    Z1=Z0*ones(length(t),1);
    
    pts_tri01{end+1}=[X1 Y1, Z1];
    
    norm_pts_tri=sqrt(X1.^2+Y1.^2+Z1.^2);
    X1sph=X1./norm_pts_tri;
    Y1sph=Y1./norm_pts_tri;
    Z1sph=Z1./norm_pts_tri;
    pts_sphtri01{end+1}=[X1sph Y1sph, Z1sph];
    
    
end









function plot_sphtri(x)

%--------------------------------------------------------------------------
% Object:
%
%--------------------------------------------------------------------------
% Input:
% x:
%--------------------------------------------------------------------------

hold on;

theta(1)=acos(x(1,3)/norm(x(1,:)));
theta(2)=acos(x(2,3)/norm(x(2,:)));
theta(3)=acos(x(3,3)/norm(x(3,:)));
theta_intv=[0 1.4*max(theta)];

funx = @(theta,phi) R*sin(theta).*cos(phi);
funy = @(theta,phi) R*sin(theta).*sin(phi);
funz = @(theta,phi) R*cos(theta);
fsurf(funx,funy,funz,[theta_intv(1) theta_intv(2) -pi pi]);
alpha 0.1 % the lower, the more transparent the region

% .... plot sph. triangle boundary ....
A=x(1,:); B=x(2,:); C=x(3,:);
plot_side(A,B); plot_side(B,C); plot_side(C,A);

% .... hide ticks, labels, axis and background ....

set(gca,'xtick',[]) % no ticks / labels
set(gca,'xticklabel',[])
set(gca,'ytick',[])
set(gca,'yticklabel',[])
set(gca,'ztick',[])
set(gca,'zticklabel',[])

set(gca,'color','none') % no background

set(gca,'Visible','off') % no axis









function plot_side(A,B)

%--------------------------------------------------------------------------
% Object:
% Plot "geodesic line" between A and B.
%--------------------------------------------------------------------------
% Input:
% A,B: row vectors of two points of the sphere to be joined.
%--------------------------------------------------------------------------

R=norm(A);
t=linspace(0,1,100);
C=[];
for k=1:length(t)
    CC=A+t(k)*(B-A); CC=CC/norm(CC); C=[C; R*CC];
end
plot3(C(:,1),C(:,2),C(:,3),'m-','LineWidth',3);












function [CC,r]=circumcenter(Q)

% https://en.wikipedia.org/wiki/Circumscribed_circle
% k-th row of Q is the k-th point

A=[Q(1,1:2) 0]; B=[Q(2,1:2) 0]; C=[Q(3,1:2) 0];
a=A-C;
b=B-C;

r=norm(a)*norm(b)*norm(a-b)/(2*norm(cross(a,b)));

t1=(norm(a))^2*b-(norm(b))^2*a;
t2=cross(a,b);
t3=norm(t2);
CC=C+cross(t1,t2)/(2*t3^2);

