function Quadrature_Weights=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes,Triangles,Boundary_Edge_Nodes,h,gradh,b,gradb,varargin)
%==========================================================================
%
% This function computes quadrature weights for evaluating the surface
% integral of a scalar function f(x,y,z) over the surface S.  The surface S
% may be specified implicitly as a level surface h(x,y,z)=0 (along with the
% Quadrature_Nodes and Triangles) with boundary b(x,y,z)>=0.  
%
%   Inputs: Quadrature_Nodes - A set of points located exactly on the
%           surface that can be a (Number_of_Quadrature_Nodes X 3) Array or
%           a (Number_of_Quadrature_Nodes X 2) Array
%
%           Triangles - A triangulation of the set Quadrature_Nodes.  This
%           should be an array where row k contains the indicdes in
%           Quadrature_Nodes of the vertices of triangle k
%
%           Boundary_Edge_Nodes - The set of boundary edges.  This should
%           be an array where rows 2l-1 and 2l contain the two- or
%           three-dimensional coordinates of the vertices of edge l.  If
%           this is given as an empty array, then the algorithm will try to
%           determine the boundary edges by finding those edges for which
%           both vertices have b approximately zero.
%
%           h - For the surface S defined implicitly by h(x,y,z)=0, row i
%           in the output of h should contain
%                        h(Quadrature_Nodes(i,:))
%           h should take in Quadrature_Nodes as specified above
%
%           gradh - The gradient of the function h.  Row i in the output
%           of gradh should contain
%           [dh/dx(Quadrature_Nodes(i,:)),dh/dy(Quadrature_Nodes(i,:)),dh/dz(Quadrature_Nodes(i,:)]
%           or
%           [dh/dx(Quadrature_Nodes(i,:)),dh/dy(Quadrature_Nodes(i,:))]
%           corresponding to the specification of Quadrature_Nodes. gradh 
%           should take in Quadrature_Nodes as specified above.
%
%           b - For the boundary of S defined implicitly by b(x,y,z)=0,
%           row i in the output of b should contain
%                        b(Quadrature_Nodes(i,:))
%           b should take in Quadrature_Nodes as specified above.
%
%           gradb - The gradient of the function b.  Row i in the output
%           of gradb should contain
%           [db/dx(Quadrature_Nodes(i,:)),db/dy(Quadrature_Nodes(i,:)),db/dz(Quadrature_Nodes(i,:)]
%           or
%           [db/dx(Quadrature_Nodes(i,:)),db/dy(Quadrature_Nodes(i,:))]
%           corresponding to the specification of Quadrature_Nodes. gradb 
%           should take in Quadrature_Nodes as specified above.
%   
%           options (optional) - an optional structure that is used to
%           indicate the types of surfaces and boundaries that are present.
%           There are four options that can be specified:
%                  Planar_Surface_Flag - 0 for nonplanar surface (default)
%                                        1 for planar surface
%                  Planar_Surface_Normal - the normal to the planar surface
%                                          given as a 1x3 vector.  The
%                                          default is [0 0 1]
%                  Planar_Boundary_Flag - 0 for nonplanar boundary (default)
%                                        1 for planar boundary
%                  Planar_Boundary_Normal - the normal to the planar bounary
%                                          given as a 1x3 vector.  The
%                                          default is [0 0 1]
%           Example: To specify that the boundary is planar with normal
%                    [1/2,-1,2] set
%                          options.Planar_Boundary_Flag=1;
%                          options.Planar_Boundary_Normal=[1/2,-1,2];
%                    and pass options to this function.
%
%   Output: Quadrature Weights - A set of quadrature weights-
%   (Number_of_Quadrature_Nodes X 1) vector-corresponding
%   to the set of points Quadrature_Nodes
%
% Examples of how to use this function with various options are available 
% on Matlab Central's file exchange in Bounded_Smooth_Surface_Quadrature_RBF_Test.m
%
% This implementation uses the method and default settings discussed in:
%
% J. A. Reeger and B. Fornberg "Numerical quadrature over
% smooth surfaces with boundaries".
%
% NOTE: The main loop of this method (over each triangle) can be easily
% parallelized if you have access to the parallel toolbox.  In such a case,
% change the for loop to a parfor loop.
%
%==========================================================================

warning off all

%==========================================================================
% Retrieve the options structure, if there is one, and se the options
%==========================================================================
if( nargin < 7 )
    options = [];
else
    options=varargin{1};
end
Planar_Surface_Flag = 0;
Planar_Surface_Normal = [0 0 1];
Planar_Boundary_Flag = 0;
Planar_Boundary_Normal = [0 0 1];

if( ~isempty(options) )
    if( isfield(options,'Planar_Surface_Flag') ); Planar_Surface_Flag = options.Planar_Surface_Flag; end
    if( isfield(options,'Planar_Surface_Normal') ); Planar_Surface_Normal = options.Planar_Surface_Normal; end
    if( isfield(options,'Planar_Boundary_Flag') ); Planar_Boundary_Flag = options.Planar_Boundary_Flag; end
    if( isfield(options,'Planar_Boundary_Normal') ); Planar_Boundary_Normal = options.Planar_Boundary_Normal; end
end

Arc_Length_Tol=100*eps;
odeoptions=odeset('AbsTol',10*eps,'RelTol',10*eps);
%==========================================================================

%==========================================================================
% Problem information
%==========================================================================
% Parameters that can be set
Poly_Order=7; % m in the paper, m must be at least 0 and at most 13
Number_of_Nearest_Neighbors=80; % n in the paper, must be at least
% (m+1)*(m+2)/2, the maximum value should
% be much less than N/2.

x_exponents=zeros((Poly_Order+1)*(Poly_Order+2)/2,1);
y_exponents=zeros((Poly_Order+1)*(Poly_Order+2)/2,1);
Exponent_Index=2;
for Order_Index=1:Poly_Order
    for Term_Index=0:Order_Index
        x_exponents(Exponent_Index,1)=Order_Index-Term_Index;
        y_exponents(Exponent_Index,1)=Term_Index;
        Exponent_Index=Exponent_Index+1;
    end
end

if Poly_Order>13 || Poly_Order<0
    error(['Polynomial terms of degree greater than -1 (at most 13)',...
        ' must be included.'])
end
if Number_of_Nearest_Neighbors<(((Poly_Order+1)*(Poly_Order+2))/2)
    error(['The number of nearest neighbors must be at least the number',...
        ' of bivariate polynomial terms.'])
end

% Determine some necessary parameters
Number_of_Quadrature_Nodes=size(Quadrature_Nodes,1);
Number_of_Triangles=size(Triangles,1);
%==========================================================================

%==========================================================================
% Generate Legendre-Gauss-Lobatto quadrature nodes and weights for use in
% approximating line integrals over the edges of the triangles, including
% edges that fall on the boundary.
%==========================================================================
n=10;
[Legendre_Nodes,Legendre_Weights]=Legendre_Gauss_Lobatto_Quadrature_Nodes(n,0,1);
Legendre_Weights_1D=flipud(Legendre_Weights);
Legendre_Nodes_1D=flipud(Legendre_Nodes.');
%==========================================================================

%==========================================================================
% Define the edge normals needed to determine the "cutting planes" and the
% projection point for each triangle in Triangles.
%==========================================================================
% Sort the vertices of the triangles by their index in the list of
% Quadrature_Nodes
Triangles = sort(Triangles,2);

% Define all of the edges in the triangulation (there will be a duplicate
% of each edge, since each belongs to two triangles)
Triangle_Edges = [Triangles(:,[1 2]);Triangles(:,[1 3]);Triangles(:,[2 3])];

% Assign to each edge the index of the triangle that it came from
Triangle_Numbers = repmat((1:Number_of_Triangles)',3,1);

% Sort the edges so that the duplicates of each edge are adjacent in the
% list of edges (and keep track of the triangles that the edges came from)
[Triangle_Edges,Sorted_Indices] = sortrows(Triangle_Edges);
Triangle_Numbers = Triangle_Numbers(Sorted_Indices);

% Move each triangle edge that is part of only one triangle (that is, it is
% on the boundary of the surface) to the end of the list of Triangle_Edges
% for book keeping.
Number_of_Triangle_Edges=size(Triangle_Edges,1);
if isempty(Boundary_Edge_Nodes)
    Boundary_Edge_Indices=find(abs(b(Quadrature_Nodes(Triangle_Edges(:,1),:)))<sqrt(eps) & abs(b(Quadrature_Nodes(Triangle_Edges(:,2),:)))<sqrt(eps));
else
    Boundary_Edge_Indices=[];
    for Boundary_Edge_Nodes_Index=1:size(Boundary_Edge_Nodes,1)/2
        Current_Boundary_Edge_Nodes=Boundary_Edge_Nodes(((Boundary_Edge_Nodes_Index-1)*2+1):(Boundary_Edge_Nodes_Index*2),:);
        Boundary_Edge_Indices=[Boundary_Edge_Indices;find((((Quadrature_Nodes(Triangle_Edges(:,1),1)-Current_Boundary_Edge_Nodes(1,1)).^2+(Quadrature_Nodes(Triangle_Edges(:,1),2)-Current_Boundary_Edge_Nodes(1,2)).^2)<10*eps & ((Quadrature_Nodes(Triangle_Edges(:,2),1)-Current_Boundary_Edge_Nodes(2,1)).^2+(Quadrature_Nodes(Triangle_Edges(:,2),2)-Current_Boundary_Edge_Nodes(2,2)).^2)<10*eps) | (((Quadrature_Nodes(Triangle_Edges(:,1),1)-Current_Boundary_Edge_Nodes(2,1)).^2+(Quadrature_Nodes(Triangle_Edges(:,1),2)-Current_Boundary_Edge_Nodes(2,2)).^2)<10*eps & ((Quadrature_Nodes(Triangle_Edges(:,2),1)-Current_Boundary_Edge_Nodes(1,1)).^2+(Quadrature_Nodes(Triangle_Edges(:,2),2)-Current_Boundary_Edge_Nodes(1,2)).^2)<10*eps))];
    end
end
Number_of_Boundary_Edges=size(Boundary_Edge_Indices,1);
Boundary_Edges=Triangle_Edges(Boundary_Edge_Indices,:);
Boundary_Edge_Triangle_Numbers=Triangle_Numbers(Boundary_Edge_Indices,1);
Triangle_Edges(Boundary_Edge_Indices,:)=[];
Triangle_Edges=[Triangle_Edges;Boundary_Edges];
Triangle_Numbers(Boundary_Edge_Indices,:)=[];
Triangle_Numbers=[Triangle_Numbers;Boundary_Edge_Triangle_Numbers];
Number_of_Interior_Edges=Number_of_Triangle_Edges-Number_of_Boundary_Edges;
Boundary_Edge_Flags=[zeros(Number_of_Interior_Edges,1);ones(Number_of_Boundary_Edges,1)];

if Planar_Surface_Flag==0
    %Find unit normal vectors to each triangle
    V1=Quadrature_Nodes(Triangles(:,2),:)-Quadrature_Nodes(Triangles(:,1),:);
    V2=Quadrature_Nodes(Triangles(:,3),:)-Quadrature_Nodes(Triangles(:,1),:);
    Triangle_Normals=[V1(:,2).*V2(:,3)-V2(:,2).*V1(:,3),V2(:,1).*V1(:,3)-V1(:,1).*V2(:,3),V1(:,1).*V2(:,2)-V2(:,1).*V1(:,2)];
    Triangle_Normals=Triangle_Normals./repmat(sqrt(Triangle_Normals(:,1).^2+Triangle_Normals(:,2).^2+Triangle_Normals(:,3).^2),1,3);
    
    % Find a vector normal to each edge that is the average of the normal
    % vectors of the two triangles to which the edge belongs.
    Edge_Normals=Triangle_Normals(Triangle_Numbers,:);
    Edge_Normals(1:2:(Number_of_Interior_Edges-1),:)=(Edge_Normals(1:2:(Number_of_Interior_Edges-1),:)+((sign((Edge_Normals(1:2:(Number_of_Interior_Edges-1),:).*Edge_Normals(2:2:Number_of_Interior_Edges,:))*ones(3,1)))*ones(1,3)).*Edge_Normals(2:2:Number_of_Interior_Edges,:))/2;
    Edge_Normals(1:2:(Number_of_Interior_Edges-1),:)=Edge_Normals(1:2:(Number_of_Interior_Edges-1),:)./repmat(sqrt(Edge_Normals(1:2:(Number_of_Interior_Edges-1),1).^2+Edge_Normals(1:2:(Number_of_Interior_Edges-1),2).^2+Edge_Normals(1:2:(Number_of_Interior_Edges-1),3).^2),1,3);
    Edge_Normals(2:2:Number_of_Interior_Edges,:)=Edge_Normals(1:2:(Number_of_Interior_Edges-1),:);
    if Planar_Boundary_Flag==1
        Edge_Normals(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,:)=[(Planar_Boundary_Normal(1,2).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),3)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),3))-Planar_Boundary_Normal(1,3).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),2)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),2))),...
            (Planar_Boundary_Normal(1,3).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),1)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),1))-Planar_Boundary_Normal(1,1).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),3)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),3))),...
            (Planar_Boundary_Normal(1,1).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),2)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),2))-Planar_Boundary_Normal(1,2).*(Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,2),1)-Quadrature_Nodes(Triangle_Edges(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,1),1)))];
        Edge_Normals(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,:)=Edge_Normals(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,:)./(sqrt((Edge_Normals(Number_of_Interior_Edges+1:Number_of_Triangle_Edges,:).^2)*ones(3,1))*ones(1,3));
    end
else
    Triangle_Normals=ones(Number_of_Triangles,1)*Planar_Surface_Normal;
    Edge_Normals=ones(Number_of_Triangle_Edges,1)*Planar_Surface_Normal;
end

% Find the midpoints of the triangles (in the plane containing the triangle
% vertices)
Triangle_Midpoints=(Quadrature_Nodes(Triangles(:,1),:)+Quadrature_Nodes(Triangles(:,2),:)+Quadrature_Nodes(Triangles(:,3),:))/3;
Nearest_Neighbor_Indices=knnsearch(Quadrature_Nodes,Triangle_Midpoints,'K',Number_of_Nearest_Neighbors);

% Now sort the Edge_Normals so that the edges for each triangle are grouped
% in order in the list.
[~,Sorted_Indices]=sort(Triangle_Numbers);
Triangle_Edges=Triangle_Edges(Sorted_Indices,:);
if Planar_Surface_Flag==0
    Edge_Normals=Edge_Normals(Sorted_Indices,:);
end
Boundary_Edge_Flags=Boundary_Edge_Flags(Sorted_Indices,1);
%==========================================================================

%==========================================================================
% Initialize some vectors/matrices for storage
%==========================================================================
% Initialize a vector to contain the computed quadrature weights for the
% surface at each point in Quadrature_Nodes
Quadrature_Weights=zeros(Number_of_Quadrature_Nodes,1);

% Define an array to store quadrature weights for all triangles (this is
% useful if the for loop is changed to a parfor loop)
Quadrature_Weights_Triangles=zeros(Number_of_Nearest_Neighbors,Number_of_Triangles);
%==========================================================================

%==========================================================================
% Specify the upper triangular option for linsolve in the case that the
% surface normal will be approximated.  In such a case, the QR
% decomposition is used.
%==========================================================================
opts.UT=true;
%==========================================================================

% Loop over each triangle.  Note that this loop can be easily changed to a
% parfor loop if the parallel toolbox is available
for Current_Triangle_Index=1:Number_of_Triangles
    
    %==========================================================================
    % Sort the current triangle's edges so that they vertices on the edges
    % appear in lexicographic order.  This is necessary for the next step.
    %==========================================================================
    Current_Triangle_Edges=[Triangle_Edges(3*(Current_Triangle_Index-1)+1,:);
        Triangle_Edges(3*(Current_Triangle_Index-1)+2,:);
        Triangle_Edges(3*(Current_Triangle_Index-1)+3,:)];    
    
    Current_Triangle_Edges=sort(Current_Triangle_Edges,2);
    
    [Current_Triangle_Edges,Sorted_Indices]=sortrows(Current_Triangle_Edges);
    
    if Planar_Surface_Flag==0
        Current_Triangle_Edge_Normals=[Edge_Normals(3*(Current_Triangle_Index-1)+1,:);
            Edge_Normals(3*(Current_Triangle_Index-1)+2,:);
            Edge_Normals(3*(Current_Triangle_Index-1)+3,:)];
        Current_Triangle_Edge_Normals=Current_Triangle_Edge_Normals(Sorted_Indices,:);
    end
    %==========================================================================
    
    %==========================================================================
    % Determine which, if any, of the current triangles edges fall on the
    % boundary
    %==========================================================================
    Current_Boundary_Edge_Flags=Boundary_Edge_Flags(3*(Current_Triangle_Index-1)+1:3*Current_Triangle_Index,1);
    Current_Boundary_Edge_Flags=Current_Boundary_Edge_Flags(Sorted_Indices,1);
    %==========================================================================
    
    %==========================================================================
    % Find the projection point for the current triangle.
    %==========================================================================
    A=Quadrature_Nodes(Current_Triangle_Edges(1,1),:);
    B=Quadrature_Nodes(Current_Triangle_Edges(1,2),:);
    C=Quadrature_Nodes(Current_Triangle_Edges(2,2),:);
    ABC=[A;B;A;C;B;C];
    if Planar_Surface_Flag==0
        AB=(B-A);
        BC=(C-B);
        CA=(A-C);
        nAB=Current_Triangle_Edge_Normals(1,:);
        nBC=Current_Triangle_Edge_Normals(3,:);
        nCA=Current_Triangle_Edge_Normals(2,:);
        nOAB=[nAB(:,2).*AB(:,3)-nAB(:,3).*AB(:,2),nAB(:,3).*AB(:,1)-nAB(:,1).*AB(:,3),nAB(:,1).*AB(:,2)-nAB(:,2).*AB(:,1)];
        nOBC=[nBC(:,2).*BC(:,3)-nBC(:,3).*BC(:,2),nBC(:,3).*BC(:,1)-nBC(:,1).*BC(:,3),nBC(:,1).*BC(:,2)-nBC(:,2).*BC(:,1)];
        nOCA=[nCA(:,2).*CA(:,3)-nCA(:,3).*CA(:,2),nCA(:,3).*CA(:,1)-nCA(:,1).*CA(:,3),nCA(:,1).*CA(:,2)-nCA(:,2).*CA(:,1)];
        vOA=[nOAB(:,2).*nOCA(:,3)-nOAB(:,3).*nOCA(:,2),nOAB(:,3).*nOCA(:,1)-nOAB(:,1).*nOCA(:,3),nOAB(:,1).*nOCA(:,2)-nOAB(:,2).*nOCA(:,1)];
        tA=(nOBC(:,1).*AB(:,1)+nOBC(:,2).*AB(:,2)+nOBC(:,3).*AB(:,3))./(nOBC(:,1).*vOA(:,1)+nOBC(:,2).*vOA(:,2)+nOBC(:,3).*vOA(:,3));
        Triangle_Projection_Point=A+tA.*vOA;
    end
    %==========================================================================
    
    %==========================================================================
    % Find the RBF centers nearest the current triangle.
    %==========================================================================
    % Get the indices of the nearest neighbors to the current triangle
    % midpoint
    nni=Nearest_Neighbor_Indices(Current_Triangle_Index,:);
    RBF_Centers=Quadrature_Nodes(nni,:);
    Triangle_Midpoint=Triangle_Midpoints(Current_Triangle_Index,:);
    if Planar_Surface_Flag==0
        Triangle_Normal=Triangle_Normals(Current_Triangle_Index,:);
    else
        Triangle_Normal=Planar_Surface_Normal;
    end
    %==========================================================================
    
    %==========================================================================
    % Project each of the nearest RBF centers to the plane containing the
    % current triangle.
    %==========================================================================
    % Pick out the vertices of the current triangle
    Current_Triangle_Vertices=Quadrature_Nodes(Triangles(Current_Triangle_Index,:),:);
    
    % Compute the rotation matrix that rotates the normal vector of the
    % current triangle to [0 0 1].' in order to define a two-dimensional
    % coordinate system
    if sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2)<eps
        Rotation_Matrix=eye(3);
    else
        Rotation_Matrix=[Triangle_Normal(1,3)/sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2+Triangle_Normal(1,3).^2) 0 -sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2)./(sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2+Triangle_Normal(1,3).^2));
            0 1 0;
            sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2)./(sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2+Triangle_Normal(1,3).^2)) 0 Triangle_Normal(1,3)./sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2+Triangle_Normal(1,3).^2)]*...
            [Triangle_Normal(1,1)./sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2) Triangle_Normal(1,2)./(sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2)) 0;
            -Triangle_Normal(1,2)./(sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2)) Triangle_Normal(1,1)./sqrt(Triangle_Normal(1,1).^2+Triangle_Normal(1,2).^2) 0;
            0 0 1];
    end
    
    % Project the current triangles nearest neighbors to the plane
    % containing the current triangle from the projection point then
    % shift the projected nearest neighbors by the projection point rotate
    % the points with the rotation matrix computed to point the normal to the
    % the current triangle in the direction of [0 0 1].'.  This creates a
    % two-dimensional coordinate system by taking the x- and y-components
    if Planar_Surface_Flag==0
        RBF_Centers_in_Plane=Project_Nodes_to_Plane_Non_Planar_Surface(RBF_Centers,Rotation_Matrix,Triangle_Normal,Triangle_Projection_Point,Triangle_Midpoint);
        Current_Triangle_Vertices_in_Plane=Project_Nodes_to_Plane_Non_Planar_Surface(Current_Triangle_Vertices,Rotation_Matrix,Triangle_Normal,Triangle_Projection_Point,Triangle_Midpoint);
        ABC_in_Plane=Project_Nodes_to_Plane_Non_Planar_Surface(ABC,Rotation_Matrix,Triangle_Normal,Triangle_Projection_Point,Triangle_Midpoint);
    elseif Planar_Surface_Flag==1 && norm(Planar_Surface_Normal-[0 0 1],2)~=0
        RBF_Centers_in_Plane=Project_Nodes_to_Plane_Planar_Surface(RBF_Centers,Rotation_Matrix,Triangle_Midpoint);
        Current_Triangle_Vertices_in_Plane=Project_Nodes_to_Plane_Planar_Surface(Current_Triangle_Vertices,Rotation_Matrix,Triangle_Midpoint);
        ABC_in_Plane=Project_Nodes_to_Plane_Planar_Surface(ABC,Rotation_Matrix,Triangle_Midpoint);
    elseif Planar_Surface_Flag==1 && size(Quadrature_Nodes,2)==2
        RBF_Centers_in_Plane=RBF_Centers-ones(Number_of_Nearest_Neighbors,1)*Triangle_Midpoint;
        Current_Triangle_Vertices_in_Plane=Current_Triangle_Vertices-ones(3,1)*Triangle_Midpoint;
        ABC_in_Plane=ABC-ones(6,1)*Triangle_Midpoint;
    end
    Current_Triangle_Vertices_in_Plane=[Current_Triangle_Vertices_in_Plane;Current_Triangle_Vertices_in_Plane(1:2,1:2)];
    %==========================================================================
    
    %==========================================================================
    % Compute the double integral over the triangle in the 2-coordinate plane.
    %==========================================================================
    % Reset I_RBF for each triangle
    I_RBF=zeros(size(RBF_Centers_in_Plane,1),1);
    
    if sum(Current_Boundary_Edge_Flags)==0 || Planar_Boundary_Flag==1
        
        % Compute the perpendicular vertices D, E, and F which are the
        % orthogonal projections of the RBF centers (O) onto the sides AB, BC,
        % and CA of tABC
        Perpendicular_Vertices_Mult=[((RBF_Centers_in_Plane(:,1)-Current_Triangle_Vertices_in_Plane(1,1)).*(Current_Triangle_Vertices_in_Plane(2,1)-Current_Triangle_Vertices_in_Plane(1,1))+(RBF_Centers_in_Plane(:,2)-Current_Triangle_Vertices_in_Plane(1,2)).*(Current_Triangle_Vertices_in_Plane(2,2)-Current_Triangle_Vertices_in_Plane(1,2)))/((Current_Triangle_Vertices_in_Plane(2,1)-Current_Triangle_Vertices_in_Plane(1,1)).^2+(Current_Triangle_Vertices_in_Plane(2,2)-Current_Triangle_Vertices_in_Plane(1,2)).^2),...
            ((RBF_Centers_in_Plane(:,1)-Current_Triangle_Vertices_in_Plane(2,1)).*(Current_Triangle_Vertices_in_Plane(3,1)-Current_Triangle_Vertices_in_Plane(2,1))+(RBF_Centers_in_Plane(:,2)-Current_Triangle_Vertices_in_Plane(2,2)).*(Current_Triangle_Vertices_in_Plane(3,2)-Current_Triangle_Vertices_in_Plane(2,2)))/((Current_Triangle_Vertices_in_Plane(3,1)-Current_Triangle_Vertices_in_Plane(2,1)).^2+(Current_Triangle_Vertices_in_Plane(3,2)-Current_Triangle_Vertices_in_Plane(2,2)).^2),...
            ((RBF_Centers_in_Plane(:,1)-Current_Triangle_Vertices_in_Plane(3,1)).*(Current_Triangle_Vertices_in_Plane(1,1)-Current_Triangle_Vertices_in_Plane(3,1))+(RBF_Centers_in_Plane(:,2)-Current_Triangle_Vertices_in_Plane(3,2)).*(Current_Triangle_Vertices_in_Plane(1,2)-Current_Triangle_Vertices_in_Plane(3,2)))/((Current_Triangle_Vertices_in_Plane(1,1)-Current_Triangle_Vertices_in_Plane(3,1)).^2+(Current_Triangle_Vertices_in_Plane(1,2)-Current_Triangle_Vertices_in_Plane(3,2)).^2)];
        Perpendicular_Vertices_x=[Perpendicular_Vertices_Mult(:,1)*(Current_Triangle_Vertices_in_Plane(2,1)-Current_Triangle_Vertices_in_Plane(1,1))+Current_Triangle_Vertices_in_Plane(1,1),...
            Perpendicular_Vertices_Mult(:,2)*(Current_Triangle_Vertices_in_Plane(3,1)-Current_Triangle_Vertices_in_Plane(2,1))+Current_Triangle_Vertices_in_Plane(2,1),...
            Perpendicular_Vertices_Mult(:,3)*(Current_Triangle_Vertices_in_Plane(1,1)-Current_Triangle_Vertices_in_Plane(3,1))+Current_Triangle_Vertices_in_Plane(3,1)];
        Perpendicular_Vertices_y=[Perpendicular_Vertices_Mult(:,1)*(Current_Triangle_Vertices_in_Plane(2,2)-Current_Triangle_Vertices_in_Plane(1,2))+Current_Triangle_Vertices_in_Plane(1,2),...
            Perpendicular_Vertices_Mult(:,2)*(Current_Triangle_Vertices_in_Plane(3,2)-Current_Triangle_Vertices_in_Plane(2,2))+Current_Triangle_Vertices_in_Plane(2,2),...
            Perpendicular_Vertices_Mult(:,3)*(Current_Triangle_Vertices_in_Plane(1,2)-Current_Triangle_Vertices_in_Plane(3,2))+Current_Triangle_Vertices_in_Plane(3,2)];
        
        % Compute the sign that defines the orientation of the current triangle
        % tABC
        sABC=sign((Current_Triangle_Vertices_in_Plane(1,2)-Current_Triangle_Vertices_in_Plane(2,2))*(Current_Triangle_Vertices_in_Plane(3,1)-Current_Triangle_Vertices_in_Plane(1,1))+(Current_Triangle_Vertices_in_Plane(2,1)-Current_Triangle_Vertices_in_Plane(1,1))*(Current_Triangle_Vertices_in_Plane(3,2)-Current_Triangle_Vertices_in_Plane(1,2)));
        
        % Compute the double integral of all bivariate polynomial terms up to
        % degree Poly_Order (m in the paper) using Green's theorem by computing
        % line integrals over each side of tABC
        I_poly=sABC*Polynomial_Term_Right_Triangle_Double_Integrals(Current_Triangle_Vertices_in_Plane,Poly_Order);
        
        for Side_Index=1:3
            % Compute the signs that determine the orientations of the right
            % triangles that can be summed together to determine the integral
            % over tABC
            Signs1=sABC*sign((RBF_Centers_in_Plane(:,2)-Current_Triangle_Vertices_in_Plane(Side_Index,2)).*(Perpendicular_Vertices_x(:,Side_Index)-RBF_Centers_in_Plane(:,1))+(Current_Triangle_Vertices_in_Plane(Side_Index,1)-RBF_Centers_in_Plane(:,1)).*(Perpendicular_Vertices_y(:,Side_Index)-RBF_Centers_in_Plane(:,2)));
            Signs2=sABC*sign((RBF_Centers_in_Plane(:,2)-Perpendicular_Vertices_y(:,Side_Index)).*(Current_Triangle_Vertices_in_Plane(Side_Index+1,1)-RBF_Centers_in_Plane(:,1))+(Perpendicular_Vertices_x(:,Side_Index)-RBF_Centers_in_Plane(:,1)).*(Current_Triangle_Vertices_in_Plane(Side_Index+1,2)-RBF_Centers_in_Plane(:,2)));
            
            % Compute the distance from the RBF center to its orthognal
            % projection onto the current side
            alpha=sqrt((RBF_Centers_in_Plane(:,1)-Perpendicular_Vertices_x(:,Side_Index)).^2+(RBF_Centers_in_Plane(:,2)-Perpendicular_Vertices_y(:,Side_Index)).^2);
            epsmult=0.0001;
            
            % Compute the distances from the orthogonal projection of the RBF
            % center onto this side to each of the vertices on this side
            beta1=sqrt((Perpendicular_Vertices_x(:,Side_Index)-Current_Triangle_Vertices_in_Plane(Side_Index,1)).^2+(Perpendicular_Vertices_y(:,Side_Index)-Current_Triangle_Vertices_in_Plane(Side_Index,2)).^2);
            beta2=sqrt((Perpendicular_Vertices_x(:,Side_Index)-Current_Triangle_Vertices_in_Plane(Side_Index+1,1)).^2+(Perpendicular_Vertices_y(:,Side_Index)-Current_Triangle_Vertices_in_Plane(Side_Index+1,2)).^2);
            
            % Evalutate the exact integrals of the RBF r^7 over each right
            % triangle while avoiding division by zero
            beta=beta1;
            I_RBF_TEMP=Signs1.*((alpha.*(beta.*sqrt(alpha.^2+beta.^2).*(279.*alpha.^6+326.*alpha.^4.*beta.^2+200.*alpha.^2.*beta.^4+48.*beta.^6)+105.*alpha.^8.*asinh(beta./alpha)))./3456);
            I_RBF(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1)=I_RBF(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1)+I_RBF_TEMP(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1);
            
            beta=beta2;
            I_RBF_TEMP=Signs2.*((alpha.*(beta.*sqrt(alpha.^2+beta.^2).*(279.*alpha.^6+326.*alpha.^4.*beta.^2+200.*alpha.^2.*beta.^4+48.*beta.^6)+105.*alpha.^8.*asinh(beta./alpha)))./3456);
            I_RBF(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1)=I_RBF(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1)+I_RBF_TEMP(abs(alpha)>epsmult*eps & abs(beta)>epsmult*eps,1);
        end
    else
        I_poly=zeros((Poly_Order+1)*(Poly_Order+2)/2,1);
        sABC_in_Plane=sign((ABC_in_Plane(1,2)-ABC_in_Plane(2,2))*(ABC_in_Plane(4,1)-ABC_in_Plane(1,1))+(ABC_in_Plane(2,1)-ABC_in_Plane(1,1))*(ABC_in_Plane(4,2)-ABC_in_Plane(1,2)));
        for Edge_Index=1:3
            if Current_Boundary_Edge_Flags(Edge_Index)==1
                if size(Quadrature_Nodes,2)==2
                    %==========================================================================
                    % Find the arc Length of the current curved nonplanar
                    % boundary edge
                    %==========================================================================
                    [s0,Propagation_Direction]=Determine_Arc_Length_2D(ABC,Edge_Index,Arc_Length_Tol,odeoptions,gradb);
                    %==========================================================================
                    
                    %==========================================================================
                    % Adjust the 1D quadrature nodes and weights for
                    % computing line integrals over boundary curves
                    %==========================================================================
                    Arc_Length_Quadrature_Nodes=s0*Legendre_Nodes_1D;
                    Arc_Length_Quadrature_Weights=s0*Legendre_Weights_1D;
                    %==========================================================================
                    
                    %==========================================================================
                    % Parameterize the boundary curve in arc length at the
                    % 1D quadrature nodes
                    %==========================================================================
                    if Propagation_Direction==0
                        [~,Quadrature_Nodes_on_Boundary_Curve]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradb),Arc_Length_Quadrature_Nodes,ABC(2*(Edge_Index-1)+1,:).',odeoptions);
                    elseif Propagation_Direction==1
                        [~,Quadrature_Nodes_on_Boundary_Curve]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradb),Arc_Length_Quadrature_Nodes,ABC(2*Edge_Index,:).',odeoptions);
                        Quadrature_Nodes_on_Boundary_Curve=flipud(Quadrature_Nodes_on_Boundary_Curve);
                    end
                    Quadrature_Nodes_on_Boundary_Curve_in_Plane=Quadrature_Nodes_on_Boundary_Curve-ones(size(Quadrature_Nodes_on_Boundary_Curve,1),1)*Triangle_Midpoint;
                    %==========================================================================
                    
                    %==========================================================================
                    % Compute the differentials utilizing the
                    % parameterization of the boundary curve in terms of
                    % arc length
                    %==========================================================================
                    x=Quadrature_Nodes_on_Boundary_Curve_in_Plane(:,1);
                    y=Quadrature_Nodes_on_Boundary_Curve_in_Plane(:,2);
                    xO=RBF_Centers_in_Plane(:,1);
                    yO=RBF_Centers_in_Plane(:,2);
                    gb=gradb(Quadrature_Nodes_on_Boundary_Curve);
                    dxds=([-gb(:,2),gb(:,1)]./(sqrt(gb(:,1).^2+gb(:,2).^2)*ones(1,2)));
                    dxds=(sign(dxds*(ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).')*ones(1,2)).*dxds;
                    dx=dxds(:,1).*Arc_Length_Quadrature_Weights;
                    dy=dxds(:,2).*Arc_Length_Quadrature_Weights;
                    %==========================================================================
                    
                    %==========================================================================
                    % Compute the line integral over the curved boundary
                    % using Legendre-Gauss-Lobatto quadrature
                    %==========================================================================
                    L_RBF=1./768.*(-sqrt((bsxfun(@minus,x.',xO)).^2+(bsxfun(@minus,y.',yO)).^2).*(279.*(bsxfun(@minus,x.',xO)).^6+326.*(bsxfun(@minus,x.',xO)).^4.*(bsxfun(@minus,y.',yO)).^2+200.*(bsxfun(@minus,x.',xO)).^2.*(bsxfun(@minus,y.',yO)).^4+48.*(bsxfun(@minus,y.',yO)).^6).*(bsxfun(@minus,y.',yO))-105.*(bsxfun(@minus,x.',xO)).^8.*log(bsxfun(@minus,y.',yO)+sqrt((bsxfun(@minus,x.',xO)).^2+(bsxfun(@minus,y.',yO)).^2)));
                    M_RBF=1./768.*((bsxfun(@minus,x.',xO)).*sqrt((bsxfun(@minus,x.',xO)).^2+(bsxfun(@minus,y.',yO)).^2).*(48.*(bsxfun(@minus,x.',xO)).^6+200.*(bsxfun(@minus,x.',xO)).^4.*(bsxfun(@minus,y.',yO)).^2+326.*(bsxfun(@minus,x.',xO)).^2.*(bsxfun(@minus,y.',yO)).^4+279.*(bsxfun(@minus,y.',yO)).^6)+105.*(bsxfun(@minus,y.',yO)).^8.*log(bsxfun(@minus,x.',xO)+sqrt((bsxfun(@minus,x.',xO)).^2+(bsxfun(@minus,y.',yO)).^2)));
                    L_poly=-bsxfun(@power,x.',x_exponents).*bsxfun(@power,y.',1+y_exponents)./(2*(1+y_exponents*ones(1,size(x,1))));
                    M_poly=bsxfun(@power,x.',1+x_exponents).*bsxfun(@power,y.',y_exponents)./(2*(1+x_exponents*ones(1,size(y,1))));
                    L_RBF(isnan(L_RBF) | isinf(L_RBF))=0;
                    M_RBF(isnan(M_RBF) | isinf(M_RBF))=0;
                    I_RBF_Temp=L_RBF*dx+M_RBF*dy;
                    I_poly_Temp=L_poly*dx+M_poly*dy;
                    %==========================================================================
                    
                else
                    %==========================================================================
                    % Find the arc Length of the current curved nonplanar
                    % boundary edge
                    %==========================================================================
                    [s0,Propagation_Direction]=Determine_Arc_Length_3D(ABC,Edge_Index,Arc_Length_Tol,odeoptions,gradh,gradb);
                    %==========================================================================
                    
                    %==========================================================================
                    % Adjust the 1D quadrature nodes and weights for
                    % computing line integrals over boundary curves
                    %==========================================================================
                    Arc_Length_Quadrature_Nodes=s0*Legendre_Nodes_1D;
                    Arc_Length_Quadrature_Weights=s0*Legendre_Weights_1D;
                    %==========================================================================
                    
                    %==========================================================================
                    % Parameterize the boundary curve in arc length at the
                    % 1D quadrature nodes and project to the 2D coordinate
                    % system
                    %==========================================================================
                    if Propagation_Direction==0
                        [~,Quadrature_Nodes_on_Boundary_Curve]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradh,gradb),Arc_Length_Quadrature_Nodes,ABC(2*(Edge_Index-1)+1,:).',odeoptions);
                    elseif Propagation_Direction==1
                        [~,Quadrature_Nodes_on_Boundary_Curve]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradh,gradb),Arc_Length_Quadrature_Nodes,ABC(2*Edge_Index,:).',odeoptions);
                        Quadrature_Nodes_on_Boundary_Curve=flipud(Quadrature_Nodes_on_Boundary_Curve);
                    end
                    if Planar_Surface_Flag==0
                        Quadrature_Nodes_on_Boundary_Curve_in_Plane=Project_Nodes_to_Plane_Non_Planar_Surface(Quadrature_Nodes_on_Boundary_Curve,Rotation_Matrix,Triangle_Normal,Triangle_Projection_Point,Triangle_Midpoint);
                    else
                        Quadrature_Nodes_on_Boundary_Curve_in_Plane=Project_Nodes_to_Plane_Planar_Surface(Quadrature_Nodes_on_Boundary_Curve,Rotation_Matrix,Triangle_Midpoint);
                    end
                    %==========================================================================
                    
                    %==========================================================================
                    % Compute the differential utilizing the
                    % parameterization of the boundary curve in terms of
                    % arc length
                    %==========================================================================
                    chix=Quadrature_Nodes_on_Boundary_Curve_in_Plane(:,1);
                    chiy=Quadrature_Nodes_on_Boundary_Curve_in_Plane(:,2);
                    chixO=RBF_Centers_in_Plane(:,1);
                    chiyO=RBF_Centers_in_Plane(:,2);
                    if Planar_Surface_Flag==0
                        dchi_dx=([1 0 0;0 1 0]*Rotation_Matrix*(((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*((((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.'))*[1,0,0]-Triangle_Normal(1,1)*(Triangle_Midpoint-Triangle_Projection_Point))-(((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.')*bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)-(bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*(Triangle_Midpoint-Triangle_Projection_Point))*Triangle_Normal(1,1))./((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.').^2*ones(1,3))).').';
                        dchi_dy=([1 0 0;0 1 0]*Rotation_Matrix*(((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*((((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.'))*[0,1,0]-Triangle_Normal(1,2)*(Triangle_Midpoint-Triangle_Projection_Point))-(((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.')*bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)-(bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*(Triangle_Midpoint-Triangle_Projection_Point))*Triangle_Normal(1,2))./((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.').^2*ones(1,3))).').';
                        dchi_dz=([1 0 0;0 1 0]*Rotation_Matrix*(((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*((((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.'))*[0,0,1]-Triangle_Normal(1,3)*(Triangle_Midpoint-Triangle_Projection_Point))-(((Triangle_Midpoint-Triangle_Projection_Point)*Triangle_Normal.')*bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)-(bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.')*(Triangle_Midpoint-Triangle_Projection_Point))*Triangle_Normal(1,3))./((bsxfun(@minus,Quadrature_Nodes_on_Boundary_Curve,Triangle_Projection_Point)*Triangle_Normal.').^2*ones(1,3))).').';
                    else
                        dchi_dx=([1 0 0;0 1 0]*Rotation_Matrix*[1;0;0]).';
                        dchi_dy=([1 0 0;0 1 0]*Rotation_Matrix*[0;1;0]).';
                        dchi_dz=([1 0 0;0 1 0]*Rotation_Matrix*[0;0;1]).';
                    end
                    gh=gradh(Quadrature_Nodes_on_Boundary_Curve);
                    gb=gradb(Quadrature_Nodes_on_Boundary_Curve);
                    Crossghgb=[gh(:,2).*gb(:,3)-gh(:,3).*gb(:,2),gh(:,3).*gb(:,1)-gh(:,1).*gb(:,3),gh(:,1).*gb(:,2)-gh(:,2).*gb(:,1)];
                    dxds=(Crossghgb./(sqrt(Crossghgb.^2*ones(3,1))*ones(1,3)));
                    dxds=(sign(dxds*(ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).')*ones(1,3)).*dxds;
                    dchix=(([dchi_dx(:,1),dchi_dy(:,1),dchi_dz(:,1)].*dxds)*ones(3,1)).*Arc_Length_Quadrature_Weights;
                    dchiy=(([dchi_dx(:,2),dchi_dy(:,2),dchi_dz(:,2)].*dxds)*ones(3,1)).*Arc_Length_Quadrature_Weights;
                    %==========================================================================
                    
                    %==========================================================================
                    % Compute the line integral over the curved boundary
                    % using Legendre-Gauss-Lobatto quadrature
                    %==========================================================================
                    L_RBF=1./768.*(-sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2).*(279.*(bsxfun(@minus,chix.',chixO)).^6+326.*(bsxfun(@minus,chix.',chixO)).^4.*(bsxfun(@minus,chiy.',chiyO)).^2+200.*(bsxfun(@minus,chix.',chixO)).^2.*(bsxfun(@minus,chiy.',chiyO)).^4+48.*(bsxfun(@minus,chiy.',chiyO)).^6).*(bsxfun(@minus,chiy.',chiyO))-105.*(bsxfun(@minus,chix.',chixO)).^8.*log(bsxfun(@minus,chiy.',chiyO)+sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2)));
                    M_RBF=1./768.*((bsxfun(@minus,chix.',chixO)).*sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2).*(48.*(bsxfun(@minus,chix.',chixO)).^6+200.*(bsxfun(@minus,chix.',chixO)).^4.*(bsxfun(@minus,chiy.',chiyO)).^2+326.*(bsxfun(@minus,chix.',chixO)).^2.*(bsxfun(@minus,chiy.',chiyO)).^4+279.*(bsxfun(@minus,chiy.',chiyO)).^6)+105.*(bsxfun(@minus,chiy.',chiyO)).^8.*log(bsxfun(@minus,chix.',chixO)+sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2)));
                    L_poly=-bsxfun(@power,chix.',x_exponents).*bsxfun(@power,chiy.',1+y_exponents)./(2*(1+y_exponents*ones(1,size(chix,1))));
                    M_poly=bsxfun(@power,chix.',1+x_exponents).*bsxfun(@power,chiy.',y_exponents)./(2*(1+x_exponents*ones(1,size(chiy,1))));
                    L_RBF(isnan(L_RBF) | isinf(L_RBF))=0;
                    M_RBF(isnan(M_RBF) | isinf(M_RBF))=0;
                    I_RBF_Temp=L_RBF*dchix+M_RBF*dchiy;
                    I_poly_Temp=L_poly*dchix+M_poly*dchiy;
                    %==========================================================================
                end
            else
                %==========================================================================
                % Compute the differential utilizing the
                % parameterization of the straight triangle edge
                %==========================================================================
                chix=ABC_in_Plane(2*(Edge_Index-1)+1,1)+Legendre_Nodes_1D*(ABC_in_Plane(2*Edge_Index,1)-ABC_in_Plane(2*(Edge_Index-1)+1,1));
                chiy=ABC_in_Plane(2*(Edge_Index-1)+1,2)+Legendre_Nodes_1D*(ABC_in_Plane(2*Edge_Index,2)-ABC_in_Plane(2*(Edge_Index-1)+1,2));
                chixO=RBF_Centers_in_Plane(:,1);
                chiyO=RBF_Centers_in_Plane(:,2);
                dchix=(ABC_in_Plane(2*Edge_Index,1)-ABC_in_Plane(2*(Edge_Index-1)+1,1))*Legendre_Weights_1D;
                dchiy=(ABC_in_Plane(2*Edge_Index,2)-ABC_in_Plane(2*(Edge_Index-1)+1,2))*Legendre_Weights_1D;
                %==========================================================================
                
                %==========================================================================
                % Compute the line integral over the straight boundary
                % using Legendre-Gauss-Lobatto quadrature
                %==========================================================================
                L_RBF=1./768.*(-sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2).*(279.*(bsxfun(@minus,chix.',chixO)).^6+326.*(bsxfun(@minus,chix.',chixO)).^4.*(bsxfun(@minus,chiy.',chiyO)).^2+200.*(bsxfun(@minus,chix.',chixO)).^2.*(bsxfun(@minus,chiy.',chiyO)).^4+48.*(bsxfun(@minus,chiy.',chiyO)).^6).*(bsxfun(@minus,chiy.',chiyO))-105.*(bsxfun(@minus,chix.',chixO)).^8.*log(bsxfun(@minus,chiy.',chiyO)+sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2)));
                M_RBF=1./768.*((bsxfun(@minus,chix.',chixO)).*sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2).*(48.*(bsxfun(@minus,chix.',chixO)).^6+200.*(bsxfun(@minus,chix.',chixO)).^4.*(bsxfun(@minus,chiy.',chiyO)).^2+326.*(bsxfun(@minus,chix.',chixO)).^2.*(bsxfun(@minus,chiy.',chiyO)).^4+279.*(bsxfun(@minus,chiy.',chiyO)).^6)+105.*(bsxfun(@minus,chiy.',chiyO)).^8.*log(bsxfun(@minus,chix.',chixO)+sqrt((bsxfun(@minus,chix.',chixO)).^2+(bsxfun(@minus,chiy.',chiyO)).^2)));
                L_poly=-bsxfun(@power,chix.',x_exponents).*bsxfun(@power,chiy.',1+y_exponents)./(2*(1+y_exponents*ones(1,size(chix,1))));
                M_poly=bsxfun(@power,chix.',1+x_exponents).*bsxfun(@power,chiy.',y_exponents)./(2*(1+x_exponents*ones(1,size(chiy,1))));
                L_RBF(isnan(L_RBF) | isinf(L_RBF))=0;
                M_RBF(isnan(M_RBF) | isinf(M_RBF))=0;
                I_RBF_Temp=L_RBF*dchix+M_RBF*dchiy;
                I_poly_Temp=L_poly*dchix+M_poly*dchiy;
                %==========================================================================
            end
            %==========================================================================
            % Add the line integral over the current edge
            %==========================================================================
            I_RBF=I_RBF+(-1)^(Edge_Index+1)*I_RBF_Temp;
            I_poly=I_poly+(-1)^(Edge_Index+1)*I_poly_Temp;
            %==========================================================================
        end
        I_RBF=sABC_in_Plane*I_RBF;
        I_poly=sABC_in_Plane*I_poly;
    end
    %==========================================================================
    
    %==========================================================================
    % Set up the matrices containing the RBF centers in the 2-coordinate plane
    % and define the radial variable r.  This bit of code defines a matrix
    % r2 such that
    % r2(i,j)=(xi'-xj')^2+(yi'-yj')^2;
    %==========================================================================
    IP = RBF_Centers_in_Plane * RBF_Centers_in_Plane.';
    r2 = bsxfun(@plus, diag(IP), diag(IP)') - 2 * IP;
    %==========================================================================
    
    %==========================================================================
    % Compute polynomial terms to add to the interpolation matrices.  Row i
    % of the matrix P has the form
    % [1 xi' yi' xi'^2 xi'*yi' yi'^2 xi'^3 xi'^2*yi' xi'*yi'^2 yi'^3 .....]
    %==========================================================================
    P=ones(Number_of_Nearest_Neighbors,(Poly_Order+1)*(Poly_Order+2)/2);
    for orderindex=1:Poly_Order
        for termindex=1:floor(orderindex/2)+1
            P(1:Number_of_Nearest_Neighbors,(orderindex)*(orderindex+1)/2+termindex)=RBF_Centers_in_Plane(1:Number_of_Nearest_Neighbors,1).*P(1:Number_of_Nearest_Neighbors,(orderindex-1)*(orderindex)/2+termindex);
            if termindex<orderindex/2+1
                P(1:Number_of_Nearest_Neighbors,(orderindex+1)*(orderindex+2)/2-(termindex-1))=RBF_Centers_in_Plane(1:Number_of_Nearest_Neighbors,2).*P(1:Number_of_Nearest_Neighbors,(orderindex)*(orderindex+1)/2-(termindex-1));
            end
        end
    end
    %==========================================================================
    
    %==========================================================================
    % Compute the quadrature weights in the 2-coordinate plane.  This
    % solves the linear system Atilde*w=Itilde from the paper.
    %==========================================================================
    w=[(r2.*r2.*r2.*r2.^(1/2)),P;P.',zeros(((Poly_Order+1)*(Poly_Order+2))/2)]\[I_RBF;I_poly];
    %==========================================================================
    
    %==========================================================================
    % Evaluate the gradient of h(x,y,z)=0 to determine the surface
    % normal n_S at each of the Number_of_Nearest_Neighbor points to
    % the current triangle on the surface.
    %==========================================================================
    n_S=gradh(RBF_Centers);
    Normn_S=sqrt((n_S.*n_S)*ones(3,1));
    n_S=n_S./(Normn_S*ones(1,3));
    %==========================================================================
    
    %==========================================================================
    % Scale the quadrature weights by the area element distortion of the
    % projection into the plane containing the current triangle.  Compute
    % and store the scaled quadrature weights.
    %==========================================================================
    % Compute the distortion of the quadrature weights for projection back
    % to the surface
    if Planar_Surface_Flag==0
        V=[RBF_Centers(:,1)-Triangle_Projection_Point(1,1),...
            RBF_Centers(:,2)-Triangle_Projection_Point(1,2),...
            RBF_Centers(:,3)-Triangle_Projection_Point(1,3)];
        NormV=sqrt((V.*V)*ones(3,1));
        
        rho=NormV;
        R=(((Triangle_Normal*(Triangle_Midpoint-Triangle_Projection_Point).')./((RBF_Centers-ones(Number_of_Nearest_Neighbors,1)*Triangle_Projection_Point)*Triangle_Normal.'))*ones(1,3)).*(RBF_Centers-ones(Number_of_Nearest_Neighbors,1)*Triangle_Projection_Point);
        R=sqrt(R.^2*ones(3,1));
                
        Pn=Triangle_Normal./(sqrt(Triangle_Normal.^2*ones(3,1))*ones(1,3));
        
        Area_Element_Distortions=abs(((((ones(size(V,1),1)*Pn).*(V./(NormV*ones(1,3))))*ones(3,1))./((n_S.*(V./(NormV*ones(1,3))))*ones(3,1))).*(rho./R).^2);
        w=w(1:Number_of_Nearest_Neighbors,1).*Area_Element_Distortions;
    else
        w=w(1:Number_of_Nearest_Neighbors,1);
    end
    
    
    Quadrature_Weights_Triangles(:,Current_Triangle_Index)=w;
    %==========================================================================
end

% Sum all quadrature weights corresponding to a given Quadrature node on
% the surface.  This implementation is useful when considering the use of
% parfor.
for Current_Triangle_Index=1:Number_of_Triangles
    nni=Nearest_Neighbor_Indices(Current_Triangle_Index,:);
    Quadrature_Weights(nni,1)=Quadrature_Weights(nni,1)+Quadrature_Weights_Triangles(:,Current_Triangle_Index);
end
end


function Nodes_in_Plane=Project_Nodes_to_Plane_Non_Planar_Surface(Nodes,Rotation_Matrix,Triangle_Normal,Triangle_Projection_Point,Triangle_Midpoint)
Nodes_in_Plane=Vectorized_Cross(Nodes-ones(size(Nodes,1),1)*Triangle_Midpoint,ones(size(Nodes,1),1)*(Triangle_Midpoint-Triangle_Projection_Point));
Nodes_in_Plane=Vectorized_Cross(ones(size(Nodes,1),1)*Triangle_Normal,Nodes_in_Plane);
Nodes_in_Plane=([1 0 0;0 1 0]*Rotation_Matrix*(Nodes_in_Plane./((Nodes-ones(size(Nodes,1),1)*Triangle_Projection_Point)*Triangle_Normal.')).').';
end

function Nodes_in_Plane=Project_Nodes_to_Plane_Planar_Surface(Nodes,Rotation_Matrix,Triangle_Midpoint)
Nodes_in_Plane=([1 0 0;0 1 0]*Rotation_Matrix*(Nodes-ones(size(Nodes,1),1)*Triangle_Midpoint).').';
end

function Vector=Vectorized_Cross(Vector1,Vector2)
Vector=[Vector1(:,2).*Vector2(:,3)-Vector1(:,3).*Vector2(:,2),...
    Vector1(:,3).*Vector2(:,1)-Vector1(:,1).*Vector2(:,3),...
    Vector1(:,1).*Vector2(:,2)-Vector1(:,2).*Vector2(:,1)];
end

function [s0,Propagation_Direction]=Determine_Arc_Length_2D(ABC,Edge_Index,tol,odeoptions,gradb)

s0=sqrt((ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
[s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradb),[0 s0],ABC(2*(Edge_Index-1)+1,:).',odeoptions);
F=((p(end,1:2)-ABC(2*Edge_Index,:)).^2*ones(2,1));
deltas0=1;
iter=0;
maxiter=100;
while abs(deltas0)>tol && iter<maxiter
    DF=2*(p(end,1:2)-ABC(2*Edge_Index,:))*Boundary_Curve_Explicit_ODE_2D(s(end),p(end,1:2).',ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradb);
    deltas0=-F/DF;
    s0=s0+deltas0;
    if s0<0
        iter=maxiter;
        break
    end
    [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradb),[0 s0],ABC(2*(Edge_Index-1)+1,:).',odeoptions);
    F=((p(end,1:2)-ABC(2*Edge_Index,:)).^2*ones(2,1));
    iter=iter+1;
end
Propagation_Direction=0;

% if arc length is less than zero or maxiter has been
% reached, then try to propagate in the opposite
% direction
if iter==maxiter
    s0=sqrt((ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
    [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradb),[0 s0],ABC(2*Edge_Index,:).',odeoptions);
    F=((p(end,1:2)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
    deltas0=1;
    iter=0;
    maxiter=100;
    while abs(deltas0)>tol && iter<maxiter
        DF=2*(p(end,1:2)-ABC(2*(Edge_Index-1)+1,:))*Boundary_Curve_Explicit_ODE_2D(s(end),p(end,1:2).',ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradb);
        deltas0=-F/DF;
        s0=s0+deltas0;
        [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_2D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradb),[0 s0],ABC(2*Edge_Index,:).',odeoptions);
        F=((p(end,1:2)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
        iter=iter+1;
    end
    Propagation_Direction=1;
end

end

function [s0,Propagation_Direction]=Determine_Arc_Length_3D(ABC,Edge_Index,tol,odeoptions,gradh,gradb)
s0=sqrt((ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(3,1));
[s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradh,gradb),[0 s0],ABC(2*(Edge_Index-1)+1,:).',odeoptions);
F=((p(end,1:3)-ABC(2*Edge_Index,:)).^2*ones(3,1));
deltas0=1;
iter=0;
maxiter=100;
while abs(deltas0)>tol && iter<maxiter
    DF=2*(p(end,1:3)-ABC(2*Edge_Index,:))*Boundary_Curve_Explicit_ODE_3D(s(end),p(end,1:3).',ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradh,gradb);
    deltas0=-F/DF;
    s0=s0+deltas0;
    if s0<0
        iter=maxiter;
        break
    end
    [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*(Edge_Index-1)+1,:),ABC(2*Edge_Index,:),gradh,gradb),[0 s0],ABC(2*(Edge_Index-1)+1,:).',odeoptions);
    F=((p(end,1:3)-ABC(2*Edge_Index,:)).^2*ones(3,1));
    iter=iter+1;
end
Propagation_Direction=0;

% if arc length is less than zero or maxiter has been
% reached, then try to propagate in the opposite
% direction
if iter==maxiter
    s0=sqrt((ABC(2*Edge_Index,:)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
    [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradh,gradb),[0 s0],ABC(2*Edge_Index,:).',odeoptions);
    F=((p(end,1:2)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
    deltas0=1;
    iter=0;
    maxiter=100;
    while abs(deltas0)>tol && iter<maxiter
        DF=2*(p(end,1:2)-ABC(2*(Edge_Index-1)+1,:))*Boundary_Curve_Explicit_ODE_3D(s(end),p(end,1:2).',ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradh,gradb);
        deltas0=-F/DF;
        s0=s0+deltas0;
        [s,p]=ode45(@(s,p)Boundary_Curve_Explicit_ODE_3D(s,p,ABC(2*Edge_Index,:),ABC(2*(Edge_Index-1)+1,:),gradh,gradb),[0 s0],ABC(2*Edge_Index,:).',odeoptions);
        F=((p(end,1:2)-ABC(2*(Edge_Index-1)+1,:)).^2*ones(2,1));
        iter=iter+1;
    end
    Propagation_Direction=1;
end
end

function I_poly=Polynomial_Term_Right_Triangle_Double_Integrals(Triangle_Vertices,poly_order)
% NOTE: The second index is only to highlight that this is a column vector
% being output.  The operators '.*', './', '.^' are used in place of '*', '/', '^'
% in order to simplify possible future vectorization.

I_poly=zeros((poly_order+1)*(poly_order+2)/2,1);

for Side_Index=1:3
    xV1=Triangle_Vertices(Side_Index,1);
    xV2=Triangle_Vertices(Side_Index+1,1);
    yV1=Triangle_Vertices(Side_Index,2);
    yV2=Triangle_Vertices(Side_Index+1,2);
    if poly_order>=0
        I_poly(1,1)=I_poly(1,1)+(xV1.*yV2)./2 - (xV2.*yV1)./2;
    end
    if poly_order>=1
        I_poly(2,1)=I_poly(2,1)+(yV1./12 + yV2./6).*xV1.^2 + ((xV2.*yV2)./6 - (xV2.*yV1)./6).*xV1 - ((xV2.^2.*yV1)./6 + (xV2.^2.*yV2)./12);
        I_poly(3,1)=I_poly(3,1)+((yV1.*yV2)./6 - yV1.^2./12 + yV2.^2./6).*xV1 + ((xV2.*yV2.^2)./12 - (xV2.*yV1.*yV2)./6 - (xV2.*yV1.^2)./6);
    end
    if poly_order>=2
        I_poly(4,1)=I_poly(4,1)+(yV1./12 + yV2./12).*xV1.^3 + ((xV2.*yV2)./12 - (xV2.*yV1)./12).*xV1.^2 + ((xV2.^2.*yV2)./12 - (xV2.^2.*yV1)./12).*xV1 - ((xV2.^3.*yV1)./12 + (xV2.^3.*yV2)./12);
        I_poly(5,1)=I_poly(5,1)+((xV1.*yV2 - xV2.*yV1).*(2.*xV1.*yV1 + xV1.*yV2 + xV2.*yV1 + 2.*xV2.*yV2))./24;
        I_poly(6,1)=I_poly(6,1)+((yV1.^2.*yV2)./12 - yV1.^3./12 + (yV1.*yV2.^2)./12 + yV2.^3./12).*xV1 + ((xV2.*yV2.^3)./12 - (xV2.*yV1.^2.*yV2)./12 - (xV2.*yV1.*yV2.^2)./12 - (xV2.*yV1.^3)./12);
    end
    if poly_order>=3
        I_poly(7,1)=I_poly(7,1)+((3.*yV1)./40 + yV2./20).*xV1.^4 + ((xV2.*yV2)./20 - (xV2.*yV1)./20).*xV1.^3 + ((xV2.^2.*yV2)./20 - (xV2.^2.*yV1)./20).*xV1.^2 + ((xV2.^3.*yV2)./20 - (xV2.^3.*yV1)./20).*xV1 - ((xV2.^4.*yV1)./20 + (3.*xV2.^4.*yV2)./40);
        I_poly(8,1)=I_poly(8,1)+(yV1.^2./60 + (yV1.*yV2)./20 + yV2.^2./60).*xV1.^3 + ((xV2.*yV1.*yV2)./60 - (xV2.*yV1.^2)./20 + (xV2.*yV2.^2)./30).*xV1.^2 + ((xV2.^2.*yV2.^2)./20 - (xV2.^2.*yV1.*yV2)./60 - (xV2.^2.*yV1.^2)./30).*xV1 - ((xV2.^3.*yV1.^2)./60 + (xV2.^3.*yV1.*yV2)./20 + (xV2.^3.*yV2.^2)./60);
        I_poly(9,1)=I_poly(9,1)+((yV1.^2.*yV2)./20 - yV1.^3./60 + (yV1.*yV2.^2)./30 + yV2.^3./60).*xV1.^2 + ((xV2.*yV1.*yV2.^2)./60 - (xV2.*yV1.^2.*yV2)./60 - (xV2.*yV1.^3)./20 + (xV2.*yV2.^3)./20).*xV1 + ((xV2.^2.*yV2.^3)./60 - (xV2.^2.*yV1.^2.*yV2)./30 - (xV2.^2.*yV1.*yV2.^2)./20 - (xV2.^2.*yV1.^3)./60);
        I_poly(10,1)=I_poly(10,1)+((yV1.^3.*yV2)./20 - (3.*yV1.^4)./40 + (yV1.^2.*yV2.^2)./20 + (yV1.*yV2.^3)./20 + yV2.^4./20).*xV1 + ((3.*xV2.*yV2.^4)./40 - (xV2.*yV1.^3.*yV2)./20 - (xV2.*yV1.^2.*yV2.^2)./20 - (xV2.*yV1.*yV2.^3)./20 - (xV2.*yV1.^4)./20);
    end
    if poly_order>=4
        I_poly(11,1)=I_poly(11,1)+(yV1./15 + yV2./30).*xV1.^5 + ((xV2.*yV2)./30 - (xV2.*yV1)./30).*xV1.^4 + ((xV2.^2.*yV2)./30 - (xV2.^2.*yV1)./30).*xV1.^3 + ((xV2.^3.*yV2)./30 - (xV2.^3.*yV1)./30).*xV1.^2 + ((xV2.^4.*yV2)./30 - (xV2.^4.*yV1)./30).*xV1 - ((xV2.^5.*yV1)./30 + (xV2.^5.*yV2)./15);
        I_poly(12,1)=I_poly(12,1)+(yV1.^2./48 + (yV1.*yV2)./30 + yV2.^2./120).*xV1.^4 + ((xV2.*yV1.*yV2)./60 - (xV2.*yV1.^2)./30 + (xV2.*yV2.^2)./60).*xV1.^3 + ((xV2.^2.*yV2.^2)./40 - (xV2.^2.*yV1.^2)./40).*xV1.^2 + ((xV2.^3.*yV2.^2)./30 - (xV2.^3.*yV1.*yV2)./60 - (xV2.^3.*yV1.^2)./60).*xV1 - ((xV2.^4.*yV1.^2)./120 + (xV2.^4.*yV1.*yV2)./30 + (xV2.^4.*yV2.^2)./48);
        I_poly(13,1)=I_poly(13,1)+((xV1.*yV2 - xV2.*yV1).*(6.*xV1.^2.*yV1.^2 + xV1.^2.*yV2.^2 + xV2.^2.*yV1.^2 + 6.*xV2.^2.*yV2.^2 + 3.*xV1.*xV2.*yV1.^2 + 3.*xV1.*xV2.*yV2.^2 + 3.*xV1.^2.*yV1.*yV2 + 3.*xV2.^2.*yV1.*yV2 + 4.*xV1.*xV2.*yV1.*yV2))./180;
        I_poly(14,1)=I_poly(14,1)+((yV1.^3.*yV2)./30 - yV1.^4./48 + (yV1.^2.*yV2.^2)./40 + (yV1.*yV2.^3)./60 + yV2.^4./120).*xV1.^2 + ((xV2.*yV1.*yV2.^3)./60 - (xV2.*yV1.^3.*yV2)./60 - (xV2.*yV1.^4)./30 + (xV2.*yV2.^4)./30).*xV1 + ((xV2.^2.*yV2.^4)./48 - (xV2.^2.*yV1.^3.*yV2)./60 - (xV2.^2.*yV1.^2.*yV2.^2)./40 - (xV2.^2.*yV1.*yV2.^3)./30 - (xV2.^2.*yV1.^4)./120);
        I_poly(15,1)=I_poly(15,1)+((yV1.^4.*yV2)./30 - yV1.^5./15 + (yV1.^3.*yV2.^2)./30 + (yV1.^2.*yV2.^3)./30 + (yV1.*yV2.^4)./30 + yV2.^5./30).*xV1 + ((xV2.*yV2.^5)./15 - (xV2.*yV1.^4.*yV2)./30 - (xV2.*yV1.^3.*yV2.^2)./30 - (xV2.*yV1.^2.*yV2.^3)./30 - (xV2.*yV1.*yV2.^4)./30 - (xV2.*yV1.^5)./30);
    end
    if poly_order>=5
        I_poly(16,1)=I_poly(16,1)+((5.*yV1)./84 + yV2./42).*xV1.^6 + ((xV2.*yV2)./42 - (xV2.*yV1)./42).*xV1.^5 + ((xV2.^2.*yV2)./42 - (xV2.^2.*yV1)./42).*xV1.^4 + ((xV2.^3.*yV2)./42 - (xV2.^3.*yV1)./42).*xV1.^3 + ((xV2.^4.*yV2)./42 - (xV2.^4.*yV1)./42).*xV1.^2 + ((xV2.^5.*yV2)./42 - (xV2.^5.*yV1)./42).*xV1 - ((xV2.^6.*yV1)./42 + (5.*xV2.^6.*yV2)./84);
        I_poly(17,1)=I_poly(17,1)+((3.*yV1.^2)./140 + (yV1.*yV2)./42 + yV2.^2./210).*xV1.^5 + ((xV2.*yV1.*yV2)./70 - (xV2.*yV1.^2)./42 + (xV2.*yV2.^2)./105).*xV1.^4 + ((xV2.^2.*yV1.*yV2)./210 - (2.*xV2.^2.*yV1.^2)./105 + (xV2.^2.*yV2.^2)./70).*xV1.^3 + ((2.*xV2.^3.*yV2.^2)./105 - (xV2.^3.*yV1.*yV2)./210 - (xV2.^3.*yV1.^2)./70).*xV1.^2 + ((xV2.^4.*yV2.^2)./42 - (xV2.^4.*yV1.*yV2)./70 - (xV2.^4.*yV1.^2)./105).*xV1 - ((xV2.^5.*yV1.^2)./210 + (xV2.^5.*yV1.*yV2)./42 + (3.*xV2.^5.*yV2.^2)./140);
        I_poly(18,1)=I_poly(18,1)+(yV1.^3./168 + (yV1.^2.*yV2)./42 + (yV1.*yV2.^2)./105 + yV2.^3./420).*xV1.^4 + ((xV2.*yV1.^2.*yV2)./210 - (xV2.*yV1.^3)./42 + (xV2.*yV1.*yV2.^2)./84 + (xV2.*yV2.^3)./140).*xV1.^3 + ((xV2.^2.*yV1.*yV2.^2)./140 - (xV2.^2.*yV1.^2.*yV2)./140 - (xV2.^2.*yV1.^3)./70 + (xV2.^2.*yV2.^3)./70).*xV1.^2 + ((xV2.^3.*yV2.^3)./42 - (xV2.^3.*yV1.^2.*yV2)./84 - (xV2.^3.*yV1.*yV2.^2)./210 - (xV2.^3.*yV1.^3)./140).*xV1 - ((xV2.^4.*yV1.^3)./420 + (xV2.^4.*yV1.^2.*yV2)./105 + (xV2.^4.*yV1.*yV2.^2)./42 + (xV2.^4.*yV2.^3)./168);
        I_poly(19,1)=I_poly(19,1)+((yV1.^3.*yV2)./42 - yV1.^4./168 + (yV1.^2.*yV2.^2)./70 + (yV1.*yV2.^3)./140 + yV2.^4./420).*xV1.^3 + ((xV2.*yV1.^2.*yV2.^2)./140 - (xV2.*yV1.^3.*yV2)./210 - (xV2.*yV1.^4)./42 + (xV2.*yV1.*yV2.^3)./84 + (xV2.*yV2.^4)./105).*xV1.^2 + ((xV2.^2.*yV1.*yV2.^3)./210 - (xV2.^2.*yV1.^3.*yV2)./84 - (xV2.^2.*yV1.^2.*yV2.^2)./140 - (xV2.^2.*yV1.^4)./105 + (xV2.^2.*yV2.^4)./42).*xV1 + ((xV2.^3.*yV2.^4)./168 - (xV2.^3.*yV1.^3.*yV2)./140 - (xV2.^3.*yV1.^2.*yV2.^2)./70 - (xV2.^3.*yV1.*yV2.^3)./42 - (xV2.^3.*yV1.^4)./420);
        I_poly(20,1)=I_poly(20,1)+((yV1.^4.*yV2)./42 - (3.*yV1.^5)./140 + (2.*yV1.^3.*yV2.^2)./105 + (yV1.^2.*yV2.^3)./70 + (yV1.*yV2.^4)./105 + yV2.^5./210).*xV1.^2 + ((xV2.*yV1.^2.*yV2.^3)./210 - (xV2.*yV1.^4.*yV2)./70 - (xV2.*yV1.^3.*yV2.^2)./210 - (xV2.*yV1.^5)./42 + (xV2.*yV1.*yV2.^4)./70 + (xV2.*yV2.^5)./42).*xV1 + ((3.*xV2.^2.*yV2.^5)./140 - (xV2.^2.*yV1.^4.*yV2)./105 - (xV2.^2.*yV1.^3.*yV2.^2)./70 - (2.*xV2.^2.*yV1.^2.*yV2.^3)./105 - (xV2.^2.*yV1.*yV2.^4)./42 - (xV2.^2.*yV1.^5)./210);
        I_poly(21,1)=I_poly(21,1)+((yV1.^5.*yV2)./42 - (5.*yV1.^6)./84 + (yV1.^4.*yV2.^2)./42 + (yV1.^3.*yV2.^3)./42 + (yV1.^2.*yV2.^4)./42 + (yV1.*yV2.^5)./42 + yV2.^6./42).*xV1 + ((5.*xV2.*yV2.^6)./84 - (xV2.*yV1.^5.*yV2)./42 - (xV2.*yV1.^4.*yV2.^2)./42 - (xV2.*yV1.^3.*yV2.^3)./42 - (xV2.*yV1.^2.*yV2.^4)./42 - (xV2.*yV1.*yV2.^5)./42 - (xV2.*yV1.^6)./42);
    end
    if poly_order>=6
        I_poly(22,1)=I_poly(22,1)+((3.*yV1)./56 + yV2./56).*xV1.^7 + ((xV2.*yV2)./56 - (xV2.*yV1)./56).*xV1.^6 + ((xV2.^2.*yV2)./56 - (xV2.^2.*yV1)./56).*xV1.^5 + ((xV2.^3.*yV2)./56 - (xV2.^3.*yV1)./56).*xV1.^4 + ((xV2.^4.*yV2)./56 - (xV2.^4.*yV1)./56).*xV1.^3 + ((xV2.^5.*yV2)./56 - (xV2.^5.*yV1)./56).*xV1.^2 + ((xV2.^6.*yV2)./56 - (xV2.^6.*yV1)./56).*xV1 - ((xV2.^7.*yV1)./56 + (3.*xV2.^7.*yV2)./56);
        I_poly(23,1)=I_poly(23,1)+(yV1.^2./48 + (yV1.*yV2)./56 + yV2.^2./336).*xV1.^6 + ((xV2.*yV1.*yV2)./84 - (xV2.*yV1.^2)./56 + (xV2.*yV2.^2)./168).*xV1.^5 + ((xV2.^2.*yV1.*yV2)./168 - (5.*xV2.^2.*yV1.^2)./336 + (xV2.^2.*yV2.^2)./112).*xV1.^4 + ((xV2.^3.*yV2.^2)./84 - (xV2.^3.*yV1.^2)./84).*xV1.^3 + ((5.*xV2.^4.*yV2.^2)./336 - (xV2.^4.*yV1.*yV2)./168 - (xV2.^4.*yV1.^2)./112).*xV1.^2 + ((xV2.^5.*yV2.^2)./56 - (xV2.^5.*yV1.*yV2)./84 - (xV2.^5.*yV1.^2)./168).*xV1 - ((xV2.^6.*yV1.^2)./336 + (xV2.^6.*yV1.*yV2)./56 + (xV2.^6.*yV2.^2)./48);
        I_poly(24,1)=I_poly(24,1)+(yV1.^3./120 + (yV1.^2.*yV2)./56 + (yV1.*yV2.^2)./168 + yV2.^3./840).*xV1.^5 + ((xV2.*yV1.^2.*yV2)./168 - (xV2.*yV1.^3)./56 + (xV2.*yV1.*yV2.^2)./120 + (xV2.*yV2.^3)./280).*xV1.^4 + ((xV2.^2.*yV1.*yV2.^2)./140 - (xV2.^2.*yV1.^2.*yV2)./420 - (xV2.^2.*yV1.^3)./84 + (xV2.^2.*yV2.^3)./140).*xV1.^3 + ((xV2.^3.*yV1.*yV2.^2)./420 - (xV2.^3.*yV1.^2.*yV2)./140 - (xV2.^3.*yV1.^3)./140 + (xV2.^3.*yV2.^3)./84).*xV1.^2 + ((xV2.^4.*yV2.^3)./56 - (xV2.^4.*yV1.^2.*yV2)./120 - (xV2.^4.*yV1.*yV2.^2)./168 - (xV2.^4.*yV1.^3)./280).*xV1 - ((xV2.^5.*yV1.^3)./840 + (xV2.^5.*yV1.^2.*yV2)./168 + (xV2.^5.*yV1.*yV2.^2)./56 + (xV2.^5.*yV2.^3)./120);
        I_poly(25,1)=I_poly(25,1)+((xV1.*yV2 - xV2.*yV1).*(20.*xV1.^3.*yV1.^3 + xV1.^3.*yV2.^3 + xV2.^3.*yV1.^3 + 20.*xV2.^3.*yV2.^3 + 4.*xV1.*xV2.^2.*yV1.^3 + 10.*xV1.^2.*xV2.*yV1.^3 + 10.*xV1.*xV2.^2.*yV2.^3 + 4.*xV1.^2.*xV2.*yV2.^3 + 4.*xV1.^3.*yV1.*yV2.^2 + 10.*xV1.^3.*yV1.^2.*yV2 + 10.*xV2.^3.*yV1.*yV2.^2 + 4.*xV2.^3.*yV1.^2.*yV2 + 12.*xV1.*xV2.^2.*yV1.*yV2.^2 + 9.*xV1.*xV2.^2.*yV1.^2.*yV2 + 9.*xV1.^2.*xV2.*yV1.*yV2.^2 + 12.*xV1.^2.*xV2.*yV1.^2.*yV2))./1120;
        I_poly(26,1)=I_poly(26,1)+((yV1.^4.*yV2)./56 - yV1.^5./120 + (yV1.^3.*yV2.^2)./84 + (yV1.^2.*yV2.^3)./140 + (yV1.*yV2.^4)./280 + yV2.^5./840).*xV1.^3 + ((xV2.*yV1.^3.*yV2.^2)./420 - (xV2.*yV1.^4.*yV2)./168 - (xV2.*yV1.^5)./56 + (xV2.*yV1.^2.*yV2.^3)./140 + (xV2.*yV1.*yV2.^4)./120 + (xV2.*yV2.^5)./168).*xV1.^2 + ((xV2.^2.*yV1.*yV2.^4)./168 - (xV2.^2.*yV1.^4.*yV2)./120 - (xV2.^2.*yV1.^3.*yV2.^2)./140 - (xV2.^2.*yV1.^2.*yV2.^3)./420 - (xV2.^2.*yV1.^5)./168 + (xV2.^2.*yV2.^5)./56).*xV1 + ((xV2.^3.*yV2.^5)./120 - (xV2.^3.*yV1.^4.*yV2)./280 - (xV2.^3.*yV1.^3.*yV2.^2)./140 - (xV2.^3.*yV1.^2.*yV2.^3)./84 - (xV2.^3.*yV1.*yV2.^4)./56 - (xV2.^3.*yV1.^5)./840);
        I_poly(27,1)=I_poly(27,1)+((yV1.^5.*yV2)./56 - yV1.^6./48 + (5.*yV1.^4.*yV2.^2)./336 + (yV1.^3.*yV2.^3)./84 + (yV1.^2.*yV2.^4)./112 + (yV1.*yV2.^5)./168 + yV2.^6./336).*xV1.^2 + ((xV2.*yV1.^2.*yV2.^4)./168 - (xV2.*yV1.^5.*yV2)./84 - (xV2.*yV1.^4.*yV2.^2)./168 - (xV2.*yV1.^6)./56 + (xV2.*yV1.*yV2.^5)./84 + (xV2.*yV2.^6)./56).*xV1 + ((xV2.^2.*yV2.^6)./48 - (xV2.^2.*yV1.^5.*yV2)./168 - (xV2.^2.*yV1.^4.*yV2.^2)./112 - (xV2.^2.*yV1.^3.*yV2.^3)./84 - (5.*xV2.^2.*yV1.^2.*yV2.^4)./336 - (xV2.^2.*yV1.*yV2.^5)./56 - (xV2.^2.*yV1.^6)./336);
        I_poly(28,1)=I_poly(28,1)+((yV1.^6.*yV2)./56 - (3.*yV1.^7)./56 + (yV1.^5.*yV2.^2)./56 + (yV1.^4.*yV2.^3)./56 + (yV1.^3.*yV2.^4)./56 + (yV1.^2.*yV2.^5)./56 + (yV1.*yV2.^6)./56 + yV2.^7./56).*xV1 + ((3.*xV2.*yV2.^7)./56 - (xV2.*yV1.^6.*yV2)./56 - (xV2.*yV1.^5.*yV2.^2)./56 - (xV2.*yV1.^4.*yV2.^3)./56 - (xV2.*yV1.^3.*yV2.^4)./56 - (xV2.*yV1.^2.*yV2.^5)./56 - (xV2.*yV1.*yV2.^6)./56 - (xV2.*yV1.^7)./56);
    end
    if poly_order>=7
        I_poly(29,1)=I_poly(29,1)+((7.*yV1)./144 + yV2./72).*xV1.^8 + ((xV2.*yV2)./72 - (xV2.*yV1)./72).*xV1.^7 + ((xV2.^2.*yV2)./72 - (xV2.^2.*yV1)./72).*xV1.^6 + ((xV2.^3.*yV2)./72 - (xV2.^3.*yV1)./72).*xV1.^5 + ((xV2.^4.*yV2)./72 - (xV2.^4.*yV1)./72).*xV1.^4 + ((xV2.^5.*yV2)./72 - (xV2.^5.*yV1)./72).*xV1.^3 + ((xV2.^6.*yV2)./72 - (xV2.^6.*yV1)./72).*xV1.^2 + ((xV2.^7.*yV2)./72 - (xV2.^7.*yV1)./72).*xV1 - ((xV2.^8.*yV1)./72 + (7.*xV2.^8.*yV2)./144);
        I_poly(30,1)=I_poly(30,1)+((5.*yV1.^2)./252 + (yV1.*yV2)./72 + yV2.^2./504).*xV1.^7 + ((5.*xV2.*yV1.*yV2)./504 - (xV2.*yV1.^2)./72 + (xV2.*yV2.^2)./252).*xV1.^6 + ((xV2.^2.*yV1.*yV2)./168 - (xV2.^2.*yV1.^2)./84 + (xV2.^2.*yV2.^2)./168).*xV1.^5 + ((xV2.^3.*yV1.*yV2)./504 - (5.*xV2.^3.*yV1.^2)./504 + (xV2.^3.*yV2.^2)./126).*xV1.^4 + ((5.*xV2.^4.*yV2.^2)./504 - (xV2.^4.*yV1.*yV2)./504 - (xV2.^4.*yV1.^2)./126).*xV1.^3 + ((xV2.^5.*yV2.^2)./84 - (xV2.^5.*yV1.*yV2)./168 - (xV2.^5.*yV1.^2)./168).*xV1.^2 + ((xV2.^6.*yV2.^2)./72 - (5.*xV2.^6.*yV1.*yV2)./504 - (xV2.^6.*yV1.^2)./252).*xV1 - ((xV2.^7.*yV1.^2)./504 + (xV2.^7.*yV1.*yV2)./72 + (5.*xV2.^7.*yV2.^2)./252);
        I_poly(31,1)=I_poly(31,1)+(yV1.^3./108 + (yV1.^2.*yV2)./72 + (yV1.*yV2.^2)./252 + yV2.^3./1512).*xV1.^6 + ((xV2.*yV1.^2.*yV2)./168 - (xV2.*yV1.^3)./72 + (xV2.*yV1.*yV2.^2)./168 + (xV2.*yV2.^3)./504).*xV1.^5 + ((xV2.^2.*yV1.*yV2.^2)./168 - (5.*xV2.^2.*yV1.^3)./504 + (xV2.^2.*yV2.^3)./252).*xV1.^4 + ((xV2.^3.*yV1.*yV2.^2)./252 - (xV2.^3.*yV1.^2.*yV2)./252 - (5.*xV2.^3.*yV1.^3)./756 + (5.*xV2.^3.*yV2.^3)./756).*xV1.^3 + ((5.*xV2.^4.*yV2.^3)./504 - (xV2.^4.*yV1.^2.*yV2)./168 - (xV2.^4.*yV1.^3)./252).*xV1.^2 + ((xV2.^5.*yV2.^3)./72 - (xV2.^5.*yV1.^2.*yV2)./168 - (xV2.^5.*yV1.*yV2.^2)./168 - (xV2.^5.*yV1.^3)./504).*xV1 - ((xV2.^6.*yV1.^3)./1512 + (xV2.^6.*yV1.^2.*yV2)./252 + (xV2.^6.*yV1.*yV2.^2)./72 + (xV2.^6.*yV2.^3)./108);
        I_poly(32,1)=I_poly(32,1)+(yV1.^4./360 + (yV1.^3.*yV2)./72 + (yV1.^2.*yV2.^2)./168 + (yV1.*yV2.^3)./504 + yV2.^4./2520).*xV1.^5 + ((xV2.*yV1.^3.*yV2)./504 - (xV2.*yV1.^4)./72 + (xV2.*yV1.^2.*yV2.^2)./168 + (11.*xV2.*yV1.*yV2.^3)./2520 + (xV2.*yV2.^4)./630).*xV1.^4 + ((xV2.^2.*yV1.^2.*yV2.^2)./420 - (xV2.^2.*yV1.^3.*yV2)./252 - (xV2.^2.*yV1.^4)./126 + (xV2.^2.*yV1.*yV2.^3)./180 + (xV2.^2.*yV2.^4)./252).*xV1.^3 + ((xV2.^3.*yV1.*yV2.^3)./252 - (xV2.^3.*yV1.^3.*yV2)./180 - (xV2.^3.*yV1.^2.*yV2.^2)./420 - (xV2.^3.*yV1.^4)./252 + (xV2.^3.*yV2.^4)./126).*xV1.^2 + ((xV2.^4.*yV2.^4)./72 - (11.*xV2.^4.*yV1.^3.*yV2)./2520 - (xV2.^4.*yV1.^2.*yV2.^2)./168 - (xV2.^4.*yV1.*yV2.^3)./504 - (xV2.^4.*yV1.^4)./630).*xV1 - ((xV2.^5.*yV1.^4)./2520 + (xV2.^5.*yV1.^3.*yV2)./504 + (xV2.^5.*yV1.^2.*yV2.^2)./168 + (xV2.^5.*yV1.*yV2.^3)./72 + (xV2.^5.*yV2.^4)./360);
        I_poly(33,1)=I_poly(33,1)+((yV1.^4.*yV2)./72 - yV1.^5./360 + (yV1.^3.*yV2.^2)./126 + (yV1.^2.*yV2.^3)./252 + (yV1.*yV2.^4)./630 + yV2.^5./2520).*xV1.^4 + ((xV2.*yV1.^3.*yV2.^2)./252 - (xV2.*yV1.^4.*yV2)./504 - (xV2.*yV1.^5)./72 + (xV2.*yV1.^2.*yV2.^3)./180 + (11.*xV2.*yV1.*yV2.^4)./2520 + (xV2.*yV2.^5)./504).*xV1.^3 + ((xV2.^2.*yV1.^2.*yV2.^3)./420 - (xV2.^2.*yV1.^4.*yV2)./168 - (xV2.^2.*yV1.^3.*yV2.^2)./420 - (xV2.^2.*yV1.^5)./168 + (xV2.^2.*yV1.*yV2.^4)./168 + (xV2.^2.*yV2.^5)./168).*xV1.^2 + ((xV2.^3.*yV1.*yV2.^4)./504 - (11.*xV2.^3.*yV1.^4.*yV2)./2520 - (xV2.^3.*yV1.^3.*yV2.^2)./180 - (xV2.^3.*yV1.^2.*yV2.^3)./252 - (xV2.^3.*yV1.^5)./504 + (xV2.^3.*yV2.^5)./72).*xV1 + ((xV2.^4.*yV2.^5)./360 - (xV2.^4.*yV1.^4.*yV2)./630 - (xV2.^4.*yV1.^3.*yV2.^2)./252 - (xV2.^4.*yV1.^2.*yV2.^3)./126 - (xV2.^4.*yV1.*yV2.^4)./72 - (xV2.^4.*yV1.^5)./2520);
        I_poly(34,1)=I_poly(34,1)+((yV1.^5.*yV2)./72 - yV1.^6./108 + (5.*yV1.^4.*yV2.^2)./504 + (5.*yV1.^3.*yV2.^3)./756 + (yV1.^2.*yV2.^4)./252 + (yV1.*yV2.^5)./504 + yV2.^6./1512).*xV1.^3 + ((xV2.*yV1.^3.*yV2.^3)./252 - (xV2.*yV1.^5.*yV2)./168 - (xV2.*yV1.^6)./72 + (xV2.*yV1.^2.*yV2.^4)./168 + (xV2.*yV1.*yV2.^5)./168 + (xV2.*yV2.^6)./252).*xV1.^2 + ((xV2.^2.*yV1.*yV2.^5)./168 - (xV2.^2.*yV1.^5.*yV2)./168 - (xV2.^2.*yV1.^4.*yV2.^2)./168 - (xV2.^2.*yV1.^3.*yV2.^3)./252 - (xV2.^2.*yV1.^6)./252 + (xV2.^2.*yV2.^6)./72).*xV1 + ((xV2.^3.*yV2.^6)./108 - (xV2.^3.*yV1.^5.*yV2)./504 - (xV2.^3.*yV1.^4.*yV2.^2)./252 - (5.*xV2.^3.*yV1.^3.*yV2.^3)./756 - (5.*xV2.^3.*yV1.^2.*yV2.^4)./504 - (xV2.^3.*yV1.*yV2.^5)./72 - (xV2.^3.*yV1.^6)./1512);
        I_poly(35,1)=I_poly(35,1)+((yV1.^6.*yV2)./72 - (5.*yV1.^7)./252 + (yV1.^5.*yV2.^2)./84 + (5.*yV1.^4.*yV2.^3)./504 + (yV1.^3.*yV2.^4)./126 + (yV1.^2.*yV2.^5)./168 + (yV1.*yV2.^6)./252 + yV2.^7./504).*xV1.^2 + ((xV2.*yV1.^3.*yV2.^4)./504 - (5.*xV2.*yV1.^6.*yV2)./504 - (xV2.*yV1.^5.*yV2.^2)./168 - (xV2.*yV1.^4.*yV2.^3)./504 - (xV2.*yV1.^7)./72 + (xV2.*yV1.^2.*yV2.^5)./168 + (5.*xV2.*yV1.*yV2.^6)./504 + (xV2.*yV2.^7)./72).*xV1 + ((5.*xV2.^2.*yV2.^7)./252 - (xV2.^2.*yV1.^6.*yV2)./252 - (xV2.^2.*yV1.^5.*yV2.^2)./168 - (xV2.^2.*yV1.^4.*yV2.^3)./126 - (5.*xV2.^2.*yV1.^3.*yV2.^4)./504 - (xV2.^2.*yV1.^2.*yV2.^5)./84 - (xV2.^2.*yV1.*yV2.^6)./72 - (xV2.^2.*yV1.^7)./504);
        I_poly(36,1)=I_poly(36,1)+((yV1.^7.*yV2)./72 - (7.*yV1.^8)./144 + (yV1.^6.*yV2.^2)./72 + (yV1.^5.*yV2.^3)./72 + (yV1.^4.*yV2.^4)./72 + (yV1.^3.*yV2.^5)./72 + (yV1.^2.*yV2.^6)./72 + (yV1.*yV2.^7)./72 + yV2.^8./72).*xV1 + ((7.*xV2.*yV2.^8)./144 - (xV2.*yV1.^7.*yV2)./72 - (xV2.*yV1.^6.*yV2.^2)./72 - (xV2.*yV1.^5.*yV2.^3)./72 - (xV2.*yV1.^4.*yV2.^4)./72 - (xV2.*yV1.^3.*yV2.^5)./72 - (xV2.*yV1.^2.*yV2.^6)./72 - (xV2.*yV1.*yV2.^7)./72 - (xV2.*yV1.^8)./72);
    end
    if poly_order>=8
        I_poly(37,1)=I_poly(37,1)+(4.*xV1.^9.*yV1 + xV1.^9.*yV2 - xV2.^9.*yV1 - 4.*xV2.^9.*yV2 - xV1.*xV2.^8.*yV1 - xV1.^8.*xV2.*yV1 + xV1.*xV2.^8.*yV2 + xV1.^8.*xV2.*yV2 - xV1.^2.*xV2.^7.*yV1 - xV1.^3.*xV2.^6.*yV1 - xV1.^4.*xV2.^5.*yV1 - xV1.^5.*xV2.^4.*yV1 - xV1.^6.*xV2.^3.*yV1 - xV1.^7.*xV2.^2.*yV1 + xV1.^2.*xV2.^7.*yV2 + xV1.^3.*xV2.^6.*yV2 + xV1.^4.*xV2.^5.*yV2 + xV1.^5.*xV2.^4.*yV2 + xV1.^6.*xV2.^3.*yV2 + xV1.^7.*xV2.^2.*yV2)./90;
        I_poly(38,1)=I_poly(38,1)+((3.*yV1.^2)./160 + (yV1.*yV2)./90 + yV2.^2./720).*xV1.^8 + ((xV2.*yV1.*yV2)./120 - (xV2.*yV1.^2)./90 + (xV2.*yV2.^2)./360).*xV1.^7 + ((xV2.^2.*yV1.*yV2)./180 - (7.*xV2.^2.*yV1.^2)./720 + (xV2.^2.*yV2.^2)./240).*xV1.^6 + ((xV2.^3.*yV1.*yV2)./360 - (xV2.^3.*yV1.^2)./120 + (xV2.^3.*yV2.^2)./180).*xV1.^5 + ((xV2.^4.*yV2.^2)./144 - (xV2.^4.*yV1.^2)./144).*xV1.^4 + ((xV2.^5.*yV2.^2)./120 - (xV2.^5.*yV1.*yV2)./360 - (xV2.^5.*yV1.^2)./180).*xV1.^3 + ((7.*xV2.^6.*yV2.^2)./720 - (xV2.^6.*yV1.*yV2)./180 - (xV2.^6.*yV1.^2)./240).*xV1.^2 + ((xV2.^7.*yV2.^2)./90 - (xV2.^7.*yV1.*yV2)./120 - (xV2.^7.*yV1.^2)./360).*xV1 - ((xV2.^8.*yV1.^2)./720 + (xV2.^8.*yV1.*yV2)./90 + (3.*xV2.^8.*yV2.^2)./160);
        I_poly(39,1)=I_poly(39,1)+(yV1.^3./105 + (yV1.^2.*yV2)./90 + (yV1.*yV2.^2)./360 + yV2.^3./2520).*xV1.^7 + ((xV2.*yV1.^2.*yV2)./180 - (xV2.*yV1.^3)./90 + (11.*xV2.*yV1.*yV2.^2)./2520 + (xV2.*yV2.^3)./840).*xV1.^6 + ((xV2.^2.*yV1.^2.*yV2)./840 - (xV2.^2.*yV1.^3)./120 + (xV2.^2.*yV1.*yV2.^2)./210 + (xV2.^2.*yV2.^3)./420).*xV1.^5 + ((xV2.^3.*yV1.*yV2.^2)./252 - (xV2.^3.*yV1.^2.*yV2)./504 - (xV2.^3.*yV1.^3)./168 + (xV2.^3.*yV2.^3)./252).*xV1.^4 + ((xV2.^4.*yV1.*yV2.^2)./504 - (xV2.^4.*yV1.^2.*yV2)./252 - (xV2.^4.*yV1.^3)./252 + (xV2.^4.*yV2.^3)./168).*xV1.^3 + ((xV2.^5.*yV2.^3)./120 - (xV2.^5.*yV1.^2.*yV2)./210 - (xV2.^5.*yV1.*yV2.^2)./840 - (xV2.^5.*yV1.^3)./420).*xV1.^2 + ((xV2.^6.*yV2.^3)./90 - (11.*xV2.^6.*yV1.^2.*yV2)./2520 - (xV2.^6.*yV1.*yV2.^2)./180 - (xV2.^6.*yV1.^3)./840).*xV1 - ((xV2.^7.*yV1.^3)./2520 + (xV2.^7.*yV1.^2.*yV2)./360 + (xV2.^7.*yV1.*yV2.^2)./90 + (xV2.^7.*yV2.^3)./105);
        I_poly(40,1)=I_poly(40,1)+(yV1.^4./240 + (yV1.^3.*yV2)./90 + (yV1.^2.*yV2.^2)./240 + (yV1.*yV2.^3)./840 + yV2.^4./5040).*xV1.^6 + ((xV2.*yV1.^3.*yV2)./360 - (xV2.*yV1.^4)./90 + (xV2.*yV1.^2.*yV2.^2)./210 + (xV2.*yV1.*yV2.^3)./360 + (xV2.*yV2.^4)./1260).*xV1.^5 + ((xV2.^2.*yV1.^2.*yV2.^2)./336 - (xV2.^2.*yV1.^3.*yV2)./504 - (xV2.^2.*yV1.^4)./144 + (xV2.^2.*yV1.*yV2.^3)./252 + (xV2.^2.*yV2.^4)./504).*xV1.^4 + ((xV2.^3.*yV1.*yV2.^3)./252 - (xV2.^3.*yV1.^3.*yV2)./252 - (xV2.^3.*yV1.^4)./252 + (xV2.^3.*yV2.^4)./252).*xV1.^3 + ((xV2.^4.*yV1.*yV2.^3)./504 - (xV2.^4.*yV1.^3.*yV2)./252 - (xV2.^4.*yV1.^2.*yV2.^2)./336 - (xV2.^4.*yV1.^4)./504 + (xV2.^4.*yV2.^4)./144).*xV1.^2 + ((xV2.^5.*yV2.^4)./90 - (xV2.^5.*yV1.^3.*yV2)./360 - (xV2.^5.*yV1.^2.*yV2.^2)./210 - (xV2.^5.*yV1.*yV2.^3)./360 - (xV2.^5.*yV1.^4)./1260).*xV1 - ((xV2.^6.*yV1.^4)./5040 + (xV2.^6.*yV1.^3.*yV2)./840 + (xV2.^6.*yV1.^2.*yV2.^2)./240 + (xV2.^6.*yV1.*yV2.^3)./90 + (xV2.^6.*yV2.^4)./240);
        I_poly(41,1)=I_poly(41,1)+((xV1.*yV2 - xV2.*yV1).*(70.*xV1.^4.*yV1.^4 + xV1.^4.*yV2.^4 + xV2.^4.*yV1.^4 + 70.*xV2.^4.*yV2.^4 + 5.*xV1.*xV2.^3.*yV1.^4 + 35.*xV1.^3.*xV2.*yV1.^4 + 35.*xV1.*xV2.^3.*yV2.^4 + 5.*xV1.^3.*xV2.*yV2.^4 + 5.*xV1.^4.*yV1.*yV2.^3 + 35.*xV1.^4.*yV1.^3.*yV2 + 35.*xV2.^4.*yV1.*yV2.^3 + 5.*xV2.^4.*yV1.^3.*yV2 + 15.*xV1.^2.*xV2.^2.*yV1.^4 + 15.*xV1.^2.*xV2.^2.*yV2.^4 + 15.*xV1.^4.*yV1.^2.*yV2.^2 + 15.*xV2.^4.*yV1.^2.*yV2.^2 + 40.*xV1.*xV2.^3.*yV1.*yV2.^3 + 16.*xV1.*xV2.^3.*yV1.^3.*yV2 + 16.*xV1.^3.*xV2.*yV1.*yV2.^3 + 40.*xV1.^3.*xV2.*yV1.^3.*yV2 + 30.*xV1.*xV2.^3.*yV1.^2.*yV2.^2 + 30.*xV1.^2.*xV2.^2.*yV1.*yV2.^3 + 30.*xV1.^2.*xV2.^2.*yV1.^3.*yV2 + 30.*xV1.^3.*xV2.*yV1.^2.*yV2.^2 + 36.*xV1.^2.*xV2.^2.*yV1.^2.*yV2.^2))./6300;
        I_poly(42,1)=I_poly(42,1)+((yV1.^5.*yV2)./90 - yV1.^6./240 + (yV1.^4.*yV2.^2)./144 + (yV1.^3.*yV2.^3)./252 + (yV1.^2.*yV2.^4)./504 + (yV1.*yV2.^5)./1260 + yV2.^6./5040).*xV1.^4 + ((xV2.*yV1.^4.*yV2.^2)./504 - (xV2.*yV1.^5.*yV2)./360 - (xV2.*yV1.^6)./90 + (xV2.*yV1.^3.*yV2.^3)./252 + (xV2.*yV1.^2.*yV2.^4)./252 + (xV2.*yV1.*yV2.^5)./360 + (xV2.*yV2.^6)./840).*xV1.^3 + ((xV2.^2.*yV1.^2.*yV2.^4)./336 - (xV2.^2.*yV1.^5.*yV2)./210 - (xV2.^2.*yV1.^4.*yV2.^2)./336 - (xV2.^2.*yV1.^6)./240 + (xV2.^2.*yV1.*yV2.^5)./210 + (xV2.^2.*yV2.^6)./240).*xV1.^2 + ((xV2.^3.*yV1.*yV2.^5)./360 - (xV2.^3.*yV1.^5.*yV2)./360 - (xV2.^3.*yV1.^4.*yV2.^2)./252 - (xV2.^3.*yV1.^3.*yV2.^3)./252 - (xV2.^3.*yV1.^2.*yV2.^4)./504 - (xV2.^3.*yV1.^6)./840 + (xV2.^3.*yV2.^6)./90).*xV1 + ((xV2.^4.*yV2.^6)./240 - (xV2.^4.*yV1.^5.*yV2)./1260 - (xV2.^4.*yV1.^4.*yV2.^2)./504 - (xV2.^4.*yV1.^3.*yV2.^3)./252 - (xV2.^4.*yV1.^2.*yV2.^4)./144 - (xV2.^4.*yV1.*yV2.^5)./90 - (xV2.^4.*yV1.^6)./5040);
        I_poly(43,1)=I_poly(43,1)+((yV1.^6.*yV2)./90 - yV1.^7./105 + (yV1.^5.*yV2.^2)./120 + (yV1.^4.*yV2.^3)./168 + (yV1.^3.*yV2.^4)./252 + (yV1.^2.*yV2.^5)./420 + (yV1.*yV2.^6)./840 + yV2.^7./2520).*xV1.^3 + ((xV2.*yV1.^4.*yV2.^3)./504 - (xV2.*yV1.^6.*yV2)./180 - (xV2.*yV1.^5.*yV2.^2)./840 - (xV2.*yV1.^7)./90 + (xV2.*yV1.^3.*yV2.^4)./252 + (xV2.*yV1.^2.*yV2.^5)./210 + (11.*xV2.*yV1.*yV2.^6)./2520 + (xV2.*yV2.^7)./360).*xV1.^2 + ((xV2.^2.*yV1.^2.*yV2.^5)./840 - (11.*xV2.^2.*yV1.^6.*yV2)./2520 - (xV2.^2.*yV1.^5.*yV2.^2)./210 - (xV2.^2.*yV1.^4.*yV2.^3)./252 - (xV2.^2.*yV1.^3.*yV2.^4)./504 - (xV2.^2.*yV1.^7)./360 + (xV2.^2.*yV1.*yV2.^6)./180 + (xV2.^2.*yV2.^7)./90).*xV1 + ((xV2.^3.*yV2.^7)./105 - (xV2.^3.*yV1.^6.*yV2)./840 - (xV2.^3.*yV1.^5.*yV2.^2)./420 - (xV2.^3.*yV1.^4.*yV2.^3)./252 - (xV2.^3.*yV1.^3.*yV2.^4)./168 - (xV2.^3.*yV1.^2.*yV2.^5)./120 - (xV2.^3.*yV1.*yV2.^6)./90 - (xV2.^3.*yV1.^7)./2520);
        I_poly(44,1)=I_poly(44,1)+((yV1.^7.*yV2)./90 - (3.*yV1.^8)./160 + (7.*yV1.^6.*yV2.^2)./720 + (yV1.^5.*yV2.^3)./120 + (yV1.^4.*yV2.^4)./144 + (yV1.^3.*yV2.^5)./180 + (yV1.^2.*yV2.^6)./240 + (yV1.*yV2.^7)./360 + yV2.^8./720).*xV1.^2 + ((xV2.*yV1.^3.*yV2.^5)./360 - (xV2.*yV1.^7.*yV2)./120 - (xV2.*yV1.^6.*yV2.^2)./180 - (xV2.*yV1.^5.*yV2.^3)./360 - (xV2.*yV1.^8)./90 + (xV2.*yV1.^2.*yV2.^6)./180 + (xV2.*yV1.*yV2.^7)./120 + (xV2.*yV2.^8)./90).*xV1 + ((3.*xV2.^2.*yV2.^8)./160 - (xV2.^2.*yV1.^7.*yV2)./360 - (xV2.^2.*yV1.^6.*yV2.^2)./240 - (xV2.^2.*yV1.^5.*yV2.^3)./180 - (xV2.^2.*yV1.^4.*yV2.^4)./144 - (xV2.^2.*yV1.^3.*yV2.^5)./120 - (7.*xV2.^2.*yV1.^2.*yV2.^6)./720 - (xV2.^2.*yV1.*yV2.^7)./90 - (xV2.^2.*yV1.^8)./720);
        I_poly(45,1)=I_poly(45,1)+-(4.*xV1.*yV1.^9 - xV1.*yV2.^9 + xV2.*yV1.^9 - 4.*xV2.*yV2.^9 - xV1.*yV1.*yV2.^8 - xV1.*yV1.^8.*yV2 + xV2.*yV1.*yV2.^8 + xV2.*yV1.^8.*yV2 - xV1.*yV1.^2.*yV2.^7 - xV1.*yV1.^3.*yV2.^6 - xV1.*yV1.^4.*yV2.^5 - xV1.*yV1.^5.*yV2.^4 - xV1.*yV1.^6.*yV2.^3 - xV1.*yV1.^7.*yV2.^2 + xV2.*yV1.^2.*yV2.^7 + xV2.*yV1.^3.*yV2.^6 + xV2.*yV1.^4.*yV2.^5 + xV2.*yV1.^5.*yV2.^4 + xV2.*yV1.^6.*yV2.^3 + xV2.*yV1.^7.*yV2.^2)./90;
    end
    if poly_order>=9
        I_poly(46,1)=I_poly(46,1)+((9.*yV1)./220 + yV2./110).*xV1.^10 + ((xV2.*yV2)./110 - (xV2.*yV1)./110).*xV1.^9 + ((xV2.^2.*yV2)./110 - (xV2.^2.*yV1)./110).*xV1.^8 + ((xV2.^3.*yV2)./110 - (xV2.^3.*yV1)./110).*xV1.^7 + ((xV2.^4.*yV2)./110 - (xV2.^4.*yV1)./110).*xV1.^6 + ((xV2.^5.*yV2)./110 - (xV2.^5.*yV1)./110).*xV1.^5 + ((xV2.^6.*yV2)./110 - (xV2.^6.*yV1)./110).*xV1.^4 + ((xV2.^7.*yV2)./110 - (xV2.^7.*yV1)./110).*xV1.^3 + ((xV2.^8.*yV2)./110 - (xV2.^8.*yV1)./110).*xV1.^2 + ((xV2.^9.*yV2)./110 - (xV2.^9.*yV1)./110).*xV1 - ((xV2.^10.*yV1)./110 + (9.*xV2.^10.*yV2)./220);
        I_poly(47,1)=I_poly(47,1)+((7.*yV1.^2)./396 + (yV1.*yV2)./110 + yV2.^2./990).*xV1.^9 + ((7.*xV2.*yV1.*yV2)./990 - (xV2.*yV1.^2)./110 + (xV2.*yV2.^2)./495).*xV1.^8 + ((xV2.^2.*yV1.*yV2)./198 - (4.*xV2.^2.*yV1.^2)./495 + (xV2.^2.*yV2.^2)./330).*xV1.^7 + ((xV2.^3.*yV1.*yV2)./330 - (7.*xV2.^3.*yV1.^2)./990 + (2.*xV2.^3.*yV2.^2)./495).*xV1.^6 + ((xV2.^4.*yV1.*yV2)./990 - (xV2.^4.*yV1.^2)./165 + (xV2.^4.*yV2.^2)./198).*xV1.^5 + ((xV2.^5.*yV2.^2)./165 - (xV2.^5.*yV1.*yV2)./990 - (xV2.^5.*yV1.^2)./198).*xV1.^4 + ((7.*xV2.^6.*yV2.^2)./990 - (xV2.^6.*yV1.*yV2)./330 - (2.*xV2.^6.*yV1.^2)./495).*xV1.^3 + ((4.*xV2.^7.*yV2.^2)./495 - (xV2.^7.*yV1.*yV2)./198 - (xV2.^7.*yV1.^2)./330).*xV1.^2 + ((xV2.^8.*yV2.^2)./110 - (7.*xV2.^8.*yV1.*yV2)./990 - (xV2.^8.*yV1.^2)./495).*xV1 - ((xV2.^9.*yV1.^2)./990 + (xV2.^9.*yV1.*yV2)./110 + (7.*xV2.^9.*yV2.^2)./396);
        I_poly(48,1)=I_poly(48,1)+((5.*yV1.^3)./528 + (yV1.^2.*yV2)./110 + (yV1.*yV2.^2)./495 + yV2.^3./3960).*xV1.^8 + ((xV2.*yV1.^2.*yV2)./198 - (xV2.*yV1.^3)./110 + (13.*xV2.*yV1.*yV2.^2)./3960 + (xV2.*yV2.^3)./1320).*xV1.^7 + ((7.*xV2.^2.*yV1.^2.*yV2)./3960 - (7.*xV2.^2.*yV1.^3)./990 + (xV2.^2.*yV1.*yV2.^2)./264 + (xV2.^2.*yV2.^3)./660).*xV1.^6 + ((7.*xV2.^3.*yV1.*yV2.^2)./1980 - (xV2.^3.*yV1.^2.*yV2)./1320 - (7.*xV2.^3.*yV1.^3)./1320 + (xV2.^3.*yV2.^3)./396).*xV1.^5 + ((xV2.^4.*yV1.*yV2.^2)./396 - (xV2.^4.*yV1.^2.*yV2)./396 - (xV2.^4.*yV1.^3)./264 + (xV2.^4.*yV2.^3)./264).*xV1.^4 + ((xV2.^5.*yV1.*yV2.^2)./1320 - (7.*xV2.^5.*yV1.^2.*yV2)./1980 - (xV2.^5.*yV1.^3)./396 + (7.*xV2.^5.*yV2.^3)./1320).*xV1.^3 + ((7.*xV2.^6.*yV2.^3)./990 - (xV2.^6.*yV1.^2.*yV2)./264 - (7.*xV2.^6.*yV1.*yV2.^2)./3960 - (xV2.^6.*yV1.^3)./660).*xV1.^2 + ((xV2.^7.*yV2.^3)./110 - (13.*xV2.^7.*yV1.^2.*yV2)./3960 - (xV2.^7.*yV1.*yV2.^2)./198 - (xV2.^7.*yV1.^3)./1320).*xV1 - ((xV2.^8.*yV1.^3)./3960 + (xV2.^8.*yV1.^2.*yV2)./495 + (xV2.^8.*yV1.*yV2.^2)./110 + (5.*xV2.^8.*yV2.^3)./528);
        I_poly(49,1)=I_poly(49,1)+((3.*yV1.^4)./616 + (yV1.^3.*yV2)./110 + (yV1.^2.*yV2.^2)./330 + (yV1.*yV2.^3)./1320 + yV2.^4./9240).*xV1.^7 + ((xV2.*yV1.^3.*yV2)./330 - (xV2.*yV1.^4)./110 + (xV2.*yV1.^2.*yV2.^2)./264 + (17.*xV2.*yV1.*yV2.^3)./9240 + (xV2.*yV2.^4)./2310).*xV1.^6 + ((9.*xV2.^2.*yV1.^2.*yV2.^2)./3080 - (xV2.^2.*yV1.^3.*yV2)./1320 - (xV2.^2.*yV1.^4)./165 + (13.*xV2.^2.*yV1.*yV2.^3)./4620 + (xV2.^2.*yV2.^4)./924).*xV1.^5 + ((xV2.^3.*yV1.^2.*yV2.^2)./924 - (5.*xV2.^3.*yV1.^3.*yV2)./1848 - (xV2.^3.*yV1.^4)./264 + (xV2.^3.*yV1.*yV2.^3)./308 + (xV2.^3.*yV2.^4)./462).*xV1.^4 + ((5.*xV2.^4.*yV1.*yV2.^3)./1848 - (xV2.^4.*yV1.^3.*yV2)./308 - (xV2.^4.*yV1.^2.*yV2.^2)./924 - (xV2.^4.*yV1.^4)./462 + (xV2.^4.*yV2.^4)./264).*xV1.^3 + ((xV2.^5.*yV1.*yV2.^3)./1320 - (13.*xV2.^5.*yV1.^3.*yV2)./4620 - (9.*xV2.^5.*yV1.^2.*yV2.^2)./3080 - (xV2.^5.*yV1.^4)./924 + (xV2.^5.*yV2.^4)./165).*xV1.^2 + ((xV2.^6.*yV2.^4)./110 - (17.*xV2.^6.*yV1.^3.*yV2)./9240 - (xV2.^6.*yV1.^2.*yV2.^2)./264 - (xV2.^6.*yV1.*yV2.^3)./330 - (xV2.^6.*yV1.^4)./2310).*xV1 - ((xV2.^7.*yV1.^4)./9240 + (xV2.^7.*yV1.^3.*yV2)./1320 + (xV2.^7.*yV1.^2.*yV2.^2)./330 + (xV2.^7.*yV1.*yV2.^3)./110 + (3.*xV2.^7.*yV2.^4)./616);
        I_poly(50,1)=I_poly(50,1)+(yV1.^5./660 + (yV1.^4.*yV2)./110 + (2.*yV1.^3.*yV2.^2)./495 + (yV1.^2.*yV2.^3)./660 + (yV1.*yV2.^4)./2310 + yV2.^5./13860).*xV1.^6 + ((xV2.*yV1.^4.*yV2)./990 - (xV2.*yV1.^5)./110 + (7.*xV2.*yV1.^3.*yV2.^2)./1980 + (13.*xV2.*yV1.^2.*yV2.^3)./4620 + (19.*xV2.*yV1.*yV2.^4)./13860 + (xV2.*yV2.^5)./2772).*xV1.^5 + ((xV2.^2.*yV1.^3.*yV2.^2)./924 - (xV2.^2.*yV1.^4.*yV2)./396 - (xV2.^2.*yV1.^5)./198 + (2.*xV2.^2.*yV1.^2.*yV2.^3)./693 + (xV2.^2.*yV1.*yV2.^4)./396 + (xV2.^2.*yV2.^5)./924).*xV1.^4 + ((xV2.^3.*yV1.^2.*yV2.^3)./693 - (xV2.^3.*yV1.^4.*yV2)./308 - (xV2.^3.*yV1.^3.*yV2.^2)./693 - (xV2.^3.*yV1.^5)./396 + (xV2.^3.*yV1.*yV2.^4)./308 + (xV2.^3.*yV2.^5)./396).*xV1.^3 + ((xV2.^4.*yV1.*yV2.^4)./396 - (xV2.^4.*yV1.^4.*yV2)./396 - (2.*xV2.^4.*yV1.^3.*yV2.^2)./693 - (xV2.^4.*yV1.^2.*yV2.^3)./924 - (xV2.^4.*yV1.^5)./924 + (xV2.^4.*yV2.^5)./198).*xV1.^2 + ((xV2.^5.*yV2.^5)./110 - (19.*xV2.^5.*yV1.^4.*yV2)./13860 - (13.*xV2.^5.*yV1.^3.*yV2.^2)./4620 - (7.*xV2.^5.*yV1.^2.*yV2.^3)./1980 - (xV2.^5.*yV1.*yV2.^4)./990 - (xV2.^5.*yV1.^5)./2772).*xV1 - ((xV2.^6.*yV1.^5)./13860 + (xV2.^6.*yV1.^4.*yV2)./2310 + (xV2.^6.*yV1.^3.*yV2.^2)./660 + (2.*xV2.^6.*yV1.^2.*yV2.^3)./495 + (xV2.^6.*yV1.*yV2.^4)./110 + (xV2.^6.*yV2.^5)./660);
        I_poly(51,1)=I_poly(51,1)+((yV1.^5.*yV2)./110 - yV1.^6./660 + (yV1.^4.*yV2.^2)./198 + (yV1.^3.*yV2.^3)./396 + (yV1.^2.*yV2.^4)./924 + (yV1.*yV2.^5)./2772 + yV2.^6./13860).*xV1.^5 + ((xV2.*yV1.^4.*yV2.^2)./396 - (xV2.*yV1.^5.*yV2)./990 - (xV2.*yV1.^6)./110 + (xV2.*yV1.^3.*yV2.^3)./308 + (xV2.*yV1.^2.*yV2.^4)./396 + (19.*xV2.*yV1.*yV2.^5)./13860 + (xV2.*yV2.^6)./2310).*xV1.^4 + ((xV2.^2.*yV1.^3.*yV2.^3)./693 - (7.*xV2.^2.*yV1.^5.*yV2)./1980 - (xV2.^2.*yV1.^4.*yV2.^2)./924 - (2.*xV2.^2.*yV1.^6)./495 + (2.*xV2.^2.*yV1.^2.*yV2.^4)./693 + (13.*xV2.^2.*yV1.*yV2.^5)./4620 + (xV2.^2.*yV2.^6)./660).*xV1.^3 + ((xV2.^3.*yV1.^2.*yV2.^4)./924 - (13.*xV2.^3.*yV1.^5.*yV2)./4620 - (2.*xV2.^3.*yV1.^4.*yV2.^2)./693 - (xV2.^3.*yV1.^3.*yV2.^3)./693 - (xV2.^3.*yV1.^6)./660 + (7.*xV2.^3.*yV1.*yV2.^5)./1980 + (2.*xV2.^3.*yV2.^6)./495).*xV1.^2 + ((xV2.^4.*yV1.*yV2.^5)./990 - (19.*xV2.^4.*yV1.^5.*yV2)./13860 - (xV2.^4.*yV1.^4.*yV2.^2)./396 - (xV2.^4.*yV1.^3.*yV2.^3)./308 - (xV2.^4.*yV1.^2.*yV2.^4)./396 - (xV2.^4.*yV1.^6)./2310 + (xV2.^4.*yV2.^6)./110).*xV1 + ((xV2.^5.*yV2.^6)./660 - (xV2.^5.*yV1.^5.*yV2)./2772 - (xV2.^5.*yV1.^4.*yV2.^2)./924 - (xV2.^5.*yV1.^3.*yV2.^3)./396 - (xV2.^5.*yV1.^2.*yV2.^4)./198 - (xV2.^5.*yV1.*yV2.^5)./110 - (xV2.^5.*yV1.^6)./13860);
        I_poly(52,1)=I_poly(52,1)+((yV1.^6.*yV2)./110 - (3.*yV1.^7)./616 + (yV1.^5.*yV2.^2)./165 + (yV1.^4.*yV2.^3)./264 + (yV1.^3.*yV2.^4)./462 + (yV1.^2.*yV2.^5)./924 + (yV1.*yV2.^6)./2310 + yV2.^7./9240).*xV1.^4 + ((xV2.*yV1.^5.*yV2.^2)./1320 - (xV2.*yV1.^6.*yV2)./330 - (xV2.*yV1.^7)./110 + (5.*xV2.*yV1.^4.*yV2.^3)./1848 + (xV2.*yV1.^3.*yV2.^4)./308 + (13.*xV2.*yV1.^2.*yV2.^5)./4620 + (17.*xV2.*yV1.*yV2.^6)./9240 + (xV2.*yV2.^7)./1320).*xV1.^3 + ((xV2.^2.*yV1.^3.*yV2.^4)./924 - (xV2.^2.*yV1.^6.*yV2)./264 - (9.*xV2.^2.*yV1.^5.*yV2.^2)./3080 - (xV2.^2.*yV1.^4.*yV2.^3)./924 - (xV2.^2.*yV1.^7)./330 + (9.*xV2.^2.*yV1.^2.*yV2.^5)./3080 + (xV2.^2.*yV1.*yV2.^6)./264 + (xV2.^2.*yV2.^7)./330).*xV1.^2 + ((xV2.^3.*yV1.*yV2.^6)./330 - (17.*xV2.^3.*yV1.^6.*yV2)./9240 - (13.*xV2.^3.*yV1.^5.*yV2.^2)./4620 - (xV2.^3.*yV1.^4.*yV2.^3)./308 - (5.*xV2.^3.*yV1.^3.*yV2.^4)./1848 - (xV2.^3.*yV1.^2.*yV2.^5)./1320 - (xV2.^3.*yV1.^7)./1320 + (xV2.^3.*yV2.^7)./110).*xV1 + ((3.*xV2.^4.*yV2.^7)./616 - (xV2.^4.*yV1.^6.*yV2)./2310 - (xV2.^4.*yV1.^5.*yV2.^2)./924 - (xV2.^4.*yV1.^4.*yV2.^3)./462 - (xV2.^4.*yV1.^3.*yV2.^4)./264 - (xV2.^4.*yV1.^2.*yV2.^5)./165 - (xV2.^4.*yV1.*yV2.^6)./110 - (xV2.^4.*yV1.^7)./9240);
        I_poly(53,1)=I_poly(53,1)+((yV1.^7.*yV2)./110 - (5.*yV1.^8)./528 + (7.*yV1.^6.*yV2.^2)./990 + (7.*yV1.^5.*yV2.^3)./1320 + (yV1.^4.*yV2.^4)./264 + (yV1.^3.*yV2.^5)./396 + (yV1.^2.*yV2.^6)./660 + (yV1.*yV2.^7)./1320 + yV2.^8./3960).*xV1.^3 + ((xV2.*yV1.^5.*yV2.^3)./1320 - (xV2.*yV1.^7.*yV2)./198 - (7.*xV2.*yV1.^6.*yV2.^2)./3960 - (xV2.*yV1.^8)./110 + (xV2.*yV1.^4.*yV2.^4)./396 + (7.*xV2.*yV1.^3.*yV2.^5)./1980 + (xV2.*yV1.^2.*yV2.^6)./264 + (13.*xV2.*yV1.*yV2.^7)./3960 + (xV2.*yV2.^8)./495).*xV1.^2 + ((7.*xV2.^2.*yV1.^2.*yV2.^6)./3960 - (13.*xV2.^2.*yV1.^7.*yV2)./3960 - (xV2.^2.*yV1.^6.*yV2.^2)./264 - (7.*xV2.^2.*yV1.^5.*yV2.^3)./1980 - (xV2.^2.*yV1.^4.*yV2.^4)./396 - (xV2.^2.*yV1.^3.*yV2.^5)./1320 - (xV2.^2.*yV1.^8)./495 + (xV2.^2.*yV1.*yV2.^7)./198 + (xV2.^2.*yV2.^8)./110).*xV1 + ((5.*xV2.^3.*yV2.^8)./528 - (xV2.^3.*yV1.^7.*yV2)./1320 - (xV2.^3.*yV1.^6.*yV2.^2)./660 - (xV2.^3.*yV1.^5.*yV2.^3)./396 - (xV2.^3.*yV1.^4.*yV2.^4)./264 - (7.*xV2.^3.*yV1.^3.*yV2.^5)./1320 - (7.*xV2.^3.*yV1.^2.*yV2.^6)./990 - (xV2.^3.*yV1.*yV2.^7)./110 - (xV2.^3.*yV1.^8)./3960);
        I_poly(54,1)=I_poly(54,1)+((yV1.^8.*yV2)./110 - (7.*yV1.^9)./396 + (4.*yV1.^7.*yV2.^2)./495 + (7.*yV1.^6.*yV2.^3)./990 + (yV1.^5.*yV2.^4)./165 + (yV1.^4.*yV2.^5)./198 + (2.*yV1.^3.*yV2.^6)./495 + (yV1.^2.*yV2.^7)./330 + (yV1.*yV2.^8)./495 + yV2.^9./990).*xV1.^2 + ((xV2.*yV1.^4.*yV2.^5)./990 - (7.*xV2.*yV1.^8.*yV2)./990 - (xV2.*yV1.^7.*yV2.^2)./198 - (xV2.*yV1.^6.*yV2.^3)./330 - (xV2.*yV1.^5.*yV2.^4)./990 - (xV2.*yV1.^9)./110 + (xV2.*yV1.^3.*yV2.^6)./330 + (xV2.*yV1.^2.*yV2.^7)./198 + (7.*xV2.*yV1.*yV2.^8)./990 + (xV2.*yV2.^9)./110).*xV1 + ((7.*xV2.^2.*yV2.^9)./396 - (xV2.^2.*yV1.^8.*yV2)./495 - (xV2.^2.*yV1.^7.*yV2.^2)./330 - (2.*xV2.^2.*yV1.^6.*yV2.^3)./495 - (xV2.^2.*yV1.^5.*yV2.^4)./198 - (xV2.^2.*yV1.^4.*yV2.^5)./165 - (7.*xV2.^2.*yV1.^3.*yV2.^6)./990 - (4.*xV2.^2.*yV1.^2.*yV2.^7)./495 - (xV2.^2.*yV1.*yV2.^8)./110 - (xV2.^2.*yV1.^9)./990);
        I_poly(55,1)=I_poly(55,1)+((yV1.^9.*yV2)./110 - (9.*yV1.^10)./220 + (yV1.^8.*yV2.^2)./110 + (yV1.^7.*yV2.^3)./110 + (yV1.^6.*yV2.^4)./110 + (yV1.^5.*yV2.^5)./110 + (yV1.^4.*yV2.^6)./110 + (yV1.^3.*yV2.^7)./110 + (yV1.^2.*yV2.^8)./110 + (yV1.*yV2.^9)./110 + yV2.^10./110).*xV1 + ((9.*xV2.*yV2.^10)./220 - (xV2.*yV1.^9.*yV2)./110 - (xV2.*yV1.^8.*yV2.^2)./110 - (xV2.*yV1.^7.*yV2.^3)./110 - (xV2.*yV1.^6.*yV2.^4)./110 - (xV2.*yV1.^5.*yV2.^5)./110 - (xV2.*yV1.^4.*yV2.^6)./110 - (xV2.*yV1.^3.*yV2.^7)./110 - (xV2.*yV1.^2.*yV2.^8)./110 - (xV2.*yV1.*yV2.^9)./110 - (xV2.*yV1.^10)./110);
    end
    if poly_order>=10
        I_poly(56,1)=I_poly(56,1)+(5.*xV1.^11.*yV1 + xV1.^11.*yV2 - xV2.^11.*yV1 - 5.*xV2.^11.*yV2 - xV1.*xV2.^10.*yV1 - xV1.^10.*xV2.*yV1 + xV1.*xV2.^10.*yV2 + xV1.^10.*xV2.*yV2 - xV1.^2.*xV2.^9.*yV1 - xV1.^3.*xV2.^8.*yV1 - xV1.^4.*xV2.^7.*yV1 - xV1.^5.*xV2.^6.*yV1 - xV1.^6.*xV2.^5.*yV1 - xV1.^7.*xV2.^4.*yV1 - xV1.^8.*xV2.^3.*yV1 - xV1.^9.*xV2.^2.*yV1 + xV1.^2.*xV2.^9.*yV2 + xV1.^3.*xV2.^8.*yV2 + xV1.^4.*xV2.^7.*yV2 + xV1.^5.*xV2.^6.*yV2 + xV1.^6.*xV2.^5.*yV2 + xV1.^7.*xV2.^4.*yV2 + xV1.^8.*xV2.^3.*yV2 + xV1.^9.*xV2.^2.*yV2)./132;
        I_poly(57,1)=I_poly(57,1)+(yV1.^2./60 + (yV1.*yV2)./132 + yV2.^2./1320).*xV1.^10 + ((xV2.*yV1.*yV2)./165 - (xV2.*yV1.^2)./132 + (xV2.*yV2.^2)./660).*xV1.^9 + ((xV2.^2.*yV1.*yV2)./220 - (3.*xV2.^2.*yV1.^2)./440 + (xV2.^2.*yV2.^2)./440).*xV1.^8 + ((xV2.^3.*yV1.*yV2)./330 - (xV2.^3.*yV1.^2)./165 + (xV2.^3.*yV2.^2)./330).*xV1.^7 + ((xV2.^4.*yV1.*yV2)./660 - (7.*xV2.^4.*yV1.^2)./1320 + (xV2.^4.*yV2.^2)./264).*xV1.^6 + ((xV2.^5.*yV2.^2)./220 - (xV2.^5.*yV1.^2)./220).*xV1.^5 + ((7.*xV2.^6.*yV2.^2)./1320 - (xV2.^6.*yV1.*yV2)./660 - (xV2.^6.*yV1.^2)./264).*xV1.^4 + ((xV2.^7.*yV2.^2)./165 - (xV2.^7.*yV1.*yV2)./330 - (xV2.^7.*yV1.^2)./330).*xV1.^3 + ((3.*xV2.^8.*yV2.^2)./440 - (xV2.^8.*yV1.*yV2)./220 - (xV2.^8.*yV1.^2)./440).*xV1.^2 + ((xV2.^9.*yV2.^2)./132 - (xV2.^9.*yV1.*yV2)./165 - (xV2.^9.*yV1.^2)./660).*xV1 - ((xV2.^10.*yV1.^2)./1320 + (xV2.^10.*yV1.*yV2)./132 + (xV2.^10.*yV2.^2)./60);
        I_poly(58,1)=I_poly(58,1)+(yV1.^3./108 + (yV1.^2.*yV2)./132 + (yV1.*yV2.^2)./660 + yV2.^3./5940).*xV1.^9 + ((xV2.*yV1.^2.*yV2)./220 - (xV2.*yV1.^3)./132 + (xV2.*yV1.*yV2.^2)./396 + (xV2.*yV2.^3)./1980).*xV1.^8 + ((xV2.^2.*yV1.^2.*yV2)./495 - (xV2.^2.*yV1.^3)./165 + (xV2.^2.*yV1.*yV2.^2)./330 + (xV2.^2.*yV2.^3)./990).*xV1.^7 + ((xV2.^3.*yV1.*yV2.^2)./330 - (7.*xV2.^3.*yV1.^3)./1485 + (xV2.^3.*yV2.^3)./594).*xV1.^6 + ((xV2.^4.*yV1.*yV2.^2)./396 - (xV2.^4.*yV1.^2.*yV2)./660 - (7.*xV2.^4.*yV1.^3)./1980 + (xV2.^4.*yV2.^3)./396).*xV1.^5 + ((xV2.^5.*yV1.*yV2.^2)./660 - (xV2.^5.*yV1.^2.*yV2)./396 - (xV2.^5.*yV1.^3)./396 + (7.*xV2.^5.*yV2.^3)./1980).*xV1.^4 + ((7.*xV2.^6.*yV2.^3)./1485 - (xV2.^6.*yV1.^2.*yV2)./330 - (xV2.^6.*yV1.^3)./594).*xV1.^3 + ((xV2.^7.*yV2.^3)./165 - (xV2.^7.*yV1.^2.*yV2)./330 - (xV2.^7.*yV1.*yV2.^2)./495 - (xV2.^7.*yV1.^3)./990).*xV1.^2 + ((xV2.^8.*yV2.^3)./132 - (xV2.^8.*yV1.^2.*yV2)./396 - (xV2.^8.*yV1.*yV2.^2)./220 - (xV2.^8.*yV1.^3)./1980).*xV1 - ((xV2.^9.*yV1.^3)./5940 + (xV2.^9.*yV1.^2.*yV2)./660 + (xV2.^9.*yV1.*yV2.^2)./132 + (xV2.^9.*yV2.^3)./108);
        I_poly(59,1)=I_poly(59,1)+(yV1.^4./192 + (yV1.^3.*yV2)./132 + (yV1.^2.*yV2.^2)./440 + (yV1.*yV2.^3)./1980 + yV2.^4./15840).*xV1.^8 + ((xV2.*yV1.^3.*yV2)./330 - (xV2.*yV1.^4)./132 + (xV2.*yV1.^2.*yV2.^2)./330 + (xV2.*yV1.*yV2.^3)./792 + (xV2.*yV2.^4)./3960).*xV1.^7 + ((7.*xV2.^2.*yV1.^2.*yV2.^2)./2640 - (7.*xV2.^2.*yV1.^4)./1320 + (xV2.^2.*yV1.*yV2.^3)./495 + (xV2.^2.*yV2.^4)./1584).*xV1.^6 + ((xV2.^3.*yV1.^2.*yV2.^2)./660 - (7.*xV2.^3.*yV1.^3.*yV2)./3960 - (7.*xV2.^3.*yV1.^4)./1980 + (xV2.^3.*yV1.*yV2.^3)./396 + (xV2.^3.*yV2.^4)./792).*xV1.^5 + ((xV2.^4.*yV1.*yV2.^3)./396 - (xV2.^4.*yV1.^3.*yV2)./396 - (7.*xV2.^4.*yV1.^4)./3168 + (7.*xV2.^4.*yV2.^4)./3168).*xV1.^4 + ((7.*xV2.^5.*yV1.*yV2.^3)./3960 - (xV2.^5.*yV1.^3.*yV2)./396 - (xV2.^5.*yV1.^2.*yV2.^2)./660 - (xV2.^5.*yV1.^4)./792 + (7.*xV2.^5.*yV2.^4)./1980).*xV1.^3 + ((7.*xV2.^6.*yV2.^4)./1320 - (xV2.^6.*yV1.^3.*yV2)./495 - (7.*xV2.^6.*yV1.^2.*yV2.^2)./2640 - (xV2.^6.*yV1.^4)./1584).*xV1.^2 + ((xV2.^7.*yV2.^4)./132 - (xV2.^7.*yV1.^3.*yV2)./792 - (xV2.^7.*yV1.^2.*yV2.^2)./330 - (xV2.^7.*yV1.*yV2.^3)./330 - (xV2.^7.*yV1.^4)./3960).*xV1 - ((xV2.^8.*yV1.^4)./15840 + (xV2.^8.*yV1.^3.*yV2)./1980 + (xV2.^8.*yV1.^2.*yV2.^2)./440 + (xV2.^8.*yV1.*yV2.^3)./132 + (xV2.^8.*yV2.^4)./192);
        I_poly(60,1)=I_poly(60,1)+(yV1.^5./420 + (yV1.^4.*yV2)./132 + (yV1.^3.*yV2.^2)./330 + (yV1.^2.*yV2.^3)./990 + (yV1.*yV2.^4)./3960 + yV2.^5./27720).*xV1.^7 + ((xV2.*yV1.^4.*yV2)./660 - (xV2.*yV1.^5)./132 + (xV2.*yV1.^3.*yV2.^2)./330 + (xV2.*yV1.^2.*yV2.^3)./495 + (23.*xV2.*yV1.*yV2.^4)./27720 + (xV2.*yV2.^5)./5544).*xV1.^6 + ((xV2.^2.*yV1.^3.*yV2.^2)./660 - (xV2.^2.*yV1.^4.*yV2)./660 - (xV2.^2.*yV1.^5)./220 + (xV2.^2.*yV1.^2.*yV2.^3)./420 + (xV2.^2.*yV1.*yV2.^4)./616 + (xV2.^2.*yV2.^5)./1848).*xV1.^5 + ((5.*xV2.^3.*yV1.^2.*yV2.^3)./2772 - (xV2.^3.*yV1.^4.*yV2)./396 - (xV2.^3.*yV1.^3.*yV2.^2)./2772 - (xV2.^3.*yV1.^5)./396 + (13.*xV2.^3.*yV1.*yV2.^4)./5544 + (xV2.^3.*yV2.^5)./792).*xV1.^4 + ((xV2.^4.*yV1.^2.*yV2.^3)./2772 - (13.*xV2.^4.*yV1.^4.*yV2)./5544 - (5.*xV2.^4.*yV1.^3.*yV2.^2)./2772 - (xV2.^4.*yV1.^5)./792 + (xV2.^4.*yV1.*yV2.^4)./396 + (xV2.^4.*yV2.^5)./396).*xV1.^3 + ((xV2.^5.*yV1.*yV2.^4)./660 - (xV2.^5.*yV1.^4.*yV2)./616 - (xV2.^5.*yV1.^3.*yV2.^2)./420 - (xV2.^5.*yV1.^2.*yV2.^3)./660 - (xV2.^5.*yV1.^5)./1848 + (xV2.^5.*yV2.^5)./220).*xV1.^2 + ((xV2.^6.*yV2.^5)./132 - (23.*xV2.^6.*yV1.^4.*yV2)./27720 - (xV2.^6.*yV1.^3.*yV2.^2)./495 - (xV2.^6.*yV1.^2.*yV2.^3)./330 - (xV2.^6.*yV1.*yV2.^4)./660 - (xV2.^6.*yV1.^5)./5544).*xV1 - ((xV2.^7.*yV1.^5)./27720 + (xV2.^7.*yV1.^4.*yV2)./3960 + (xV2.^7.*yV1.^3.*yV2.^2)./990 + (xV2.^7.*yV1.^2.*yV2.^3)./330 + (xV2.^7.*yV1.*yV2.^4)./132 + (xV2.^7.*yV2.^5)./420);
        I_poly(61,1)=I_poly(61,1)+((xV1.*yV2 - xV2.*yV1).*(252.*xV1.^5.*yV1.^5 + xV1.^5.*yV2.^5 + xV2.^5.*yV1.^5 + 252.*xV2.^5.*yV2.^5 + 6.*xV1.*xV2.^4.*yV1.^5 + 126.*xV1.^4.*xV2.*yV1.^5 + 126.*xV1.*xV2.^4.*yV2.^5 + 6.*xV1.^4.*xV2.*yV2.^5 + 6.*xV1.^5.*yV1.*yV2.^4 + 126.*xV1.^5.*yV1.^4.*yV2 + 126.*xV2.^5.*yV1.*yV2.^4 + 6.*xV2.^5.*yV1.^4.*yV2 + 21.*xV1.^2.*xV2.^3.*yV1.^5 + 56.*xV1.^3.*xV2.^2.*yV1.^5 + 56.*xV1.^2.*xV2.^3.*yV2.^5 + 21.*xV1.^3.*xV2.^2.*yV2.^5 + 21.*xV1.^5.*yV1.^2.*yV2.^3 + 56.*xV1.^5.*yV1.^3.*yV2.^2 + 56.*xV2.^5.*yV1.^2.*yV2.^3 + 21.*xV2.^5.*yV1.^3.*yV2.^2 + 140.*xV1.*xV2.^4.*yV1.*yV2.^4 + 25.*xV1.*xV2.^4.*yV1.^4.*yV2 + 25.*xV1.^4.*xV2.*yV1.*yV2.^4 + 140.*xV1.^4.*xV2.*yV1.^4.*yV2 + 105.*xV1.*xV2.^4.*yV1.^2.*yV2.^3 + 60.*xV1.*xV2.^4.*yV1.^3.*yV2.^2 + 105.*xV1.^2.*xV2.^3.*yV1.*yV2.^4 + 60.*xV1.^2.*xV2.^3.*yV1.^4.*yV2 + 60.*xV1.^3.*xV2.^2.*yV1.*yV2.^4 + 105.*xV1.^3.*xV2.^2.*yV1.^4.*yV2 + 60.*xV1.^4.*xV2.*yV1.^2.*yV2.^3 + 105.*xV1.^4.*xV2.*yV1.^3.*yV2.^2 + 120.*xV1.^2.*xV2.^3.*yV1.^2.*yV2.^3 + 100.*xV1.^2.*xV2.^3.*yV1.^3.*yV2.^2 + 100.*xV1.^3.*xV2.^2.*yV1.^2.*yV2.^3 + 120.*xV1.^3.*xV2.^2.*yV1.^3.*yV2.^2))./33264;
        I_poly(62,1)=I_poly(62,1)+((yV1.^6.*yV2)./132 - yV1.^7./420 + (yV1.^5.*yV2.^2)./220 + (yV1.^4.*yV2.^3)./396 + (yV1.^3.*yV2.^4)./792 + (yV1.^2.*yV2.^5)./1848 + (yV1.*yV2.^6)./5544 + yV2.^7./27720).*xV1.^5 + ((xV2.*yV1.^5.*yV2.^2)./660 - (xV2.*yV1.^6.*yV2)./660 - (xV2.*yV1.^7)./132 + (xV2.*yV1.^4.*yV2.^3)./396 + (13.*xV2.*yV1.^3.*yV2.^4)./5544 + (xV2.*yV1.^2.*yV2.^5)./616 + (23.*xV2.*yV1.*yV2.^6)./27720 + (xV2.*yV2.^7)./3960).*xV1.^4 + ((xV2.^2.*yV1.^4.*yV2.^3)./2772 - (xV2.^2.*yV1.^6.*yV2)./330 - (xV2.^2.*yV1.^5.*yV2.^2)./660 - (xV2.^2.*yV1.^7)./330 + (5.*xV2.^2.*yV1.^3.*yV2.^4)./2772 + (xV2.^2.*yV1.^2.*yV2.^5)./420 + (xV2.^2.*yV1.*yV2.^6)./495 + (xV2.^2.*yV2.^7)./990).*xV1.^3 + ((xV2.^3.*yV1.^2.*yV2.^5)./660 - (xV2.^3.*yV1.^6.*yV2)./495 - (xV2.^3.*yV1.^5.*yV2.^2)./420 - (5.*xV2.^3.*yV1.^4.*yV2.^3)./2772 - (xV2.^3.*yV1.^3.*yV2.^4)./2772 - (xV2.^3.*yV1.^7)./990 + (xV2.^3.*yV1.*yV2.^6)./330 + (xV2.^3.*yV2.^7)./330).*xV1.^2 + ((xV2.^4.*yV1.*yV2.^6)./660 - (23.*xV2.^4.*yV1.^6.*yV2)./27720 - (xV2.^4.*yV1.^5.*yV2.^2)./616 - (13.*xV2.^4.*yV1.^4.*yV2.^3)./5544 - (xV2.^4.*yV1.^3.*yV2.^4)./396 - (xV2.^4.*yV1.^2.*yV2.^5)./660 - (xV2.^4.*yV1.^7)./3960 + (xV2.^4.*yV2.^7)./132).*xV1 + ((xV2.^5.*yV2.^7)./420 - (xV2.^5.*yV1.^6.*yV2)./5544 - (xV2.^5.*yV1.^5.*yV2.^2)./1848 - (xV2.^5.*yV1.^4.*yV2.^3)./792 - (xV2.^5.*yV1.^3.*yV2.^4)./396 - (xV2.^5.*yV1.^2.*yV2.^5)./220 - (xV2.^5.*yV1.*yV2.^6)./132 - (xV2.^5.*yV1.^7)./27720);
        I_poly(63,1)=I_poly(63,1)+((yV1.^7.*yV2)./132 - yV1.^8./192 + (7.*yV1.^6.*yV2.^2)./1320 + (7.*yV1.^5.*yV2.^3)./1980 + (7.*yV1.^4.*yV2.^4)./3168 + (yV1.^3.*yV2.^5)./792 + (yV1.^2.*yV2.^6)./1584 + (yV1.*yV2.^7)./3960 + yV2.^8./15840).*xV1.^4 + ((7.*xV2.*yV1.^5.*yV2.^3)./3960 - (xV2.*yV1.^7.*yV2)./330 - (xV2.*yV1.^8)./132 + (xV2.*yV1.^4.*yV2.^4)./396 + (xV2.*yV1.^3.*yV2.^5)./396 + (xV2.*yV1.^2.*yV2.^6)./495 + (xV2.*yV1.*yV2.^7)./792 + (xV2.*yV2.^8)./1980).*xV1.^3 + ((xV2.^2.*yV1.^3.*yV2.^5)./660 - (xV2.^2.*yV1.^7.*yV2)./330 - (7.*xV2.^2.*yV1.^6.*yV2.^2)./2640 - (xV2.^2.*yV1.^5.*yV2.^3)./660 - (xV2.^2.*yV1.^8)./440 + (7.*xV2.^2.*yV1.^2.*yV2.^6)./2640 + (xV2.^2.*yV1.*yV2.^7)./330 + (xV2.^2.*yV2.^8)./440).*xV1.^2 + ((xV2.^3.*yV1.*yV2.^7)./330 - (xV2.^3.*yV1.^7.*yV2)./792 - (xV2.^3.*yV1.^6.*yV2.^2)./495 - (xV2.^3.*yV1.^5.*yV2.^3)./396 - (xV2.^3.*yV1.^4.*yV2.^4)./396 - (7.*xV2.^3.*yV1.^3.*yV2.^5)./3960 - (xV2.^3.*yV1.^8)./1980 + (xV2.^3.*yV2.^8)./132).*xV1 + ((xV2.^4.*yV2.^8)./192 - (xV2.^4.*yV1.^7.*yV2)./3960 - (xV2.^4.*yV1.^6.*yV2.^2)./1584 - (xV2.^4.*yV1.^5.*yV2.^3)./792 - (7.*xV2.^4.*yV1.^4.*yV2.^4)./3168 - (7.*xV2.^4.*yV1.^3.*yV2.^5)./1980 - (7.*xV2.^4.*yV1.^2.*yV2.^6)./1320 - (xV2.^4.*yV1.*yV2.^7)./132 - (xV2.^4.*yV1.^8)./15840);
        I_poly(64,1)=I_poly(64,1)+((yV1.^8.*yV2)./132 - yV1.^9./108 + (yV1.^7.*yV2.^2)./165 + (7.*yV1.^6.*yV2.^3)./1485 + (7.*yV1.^5.*yV2.^4)./1980 + (yV1.^4.*yV2.^5)./396 + (yV1.^3.*yV2.^6)./594 + (yV1.^2.*yV2.^7)./990 + (yV1.*yV2.^8)./1980 + yV2.^9./5940).*xV1.^3 + ((xV2.*yV1.^5.*yV2.^4)./660 - (xV2.*yV1.^8.*yV2)./220 - (xV2.*yV1.^7.*yV2.^2)./495 - (xV2.*yV1.^9)./132 + (xV2.*yV1.^4.*yV2.^5)./396 + (xV2.*yV1.^3.*yV2.^6)./330 + (xV2.*yV1.^2.*yV2.^7)./330 + (xV2.*yV1.*yV2.^8)./396 + (xV2.*yV2.^9)./660).*xV1.^2 + ((xV2.^2.*yV1.^2.*yV2.^7)./495 - (xV2.^2.*yV1.^8.*yV2)./396 - (xV2.^2.*yV1.^7.*yV2.^2)./330 - (xV2.^2.*yV1.^6.*yV2.^3)./330 - (xV2.^2.*yV1.^5.*yV2.^4)./396 - (xV2.^2.*yV1.^4.*yV2.^5)./660 - (xV2.^2.*yV1.^9)./660 + (xV2.^2.*yV1.*yV2.^8)./220 + (xV2.^2.*yV2.^9)./132).*xV1 + ((xV2.^3.*yV2.^9)./108 - (xV2.^3.*yV1.^8.*yV2)./1980 - (xV2.^3.*yV1.^7.*yV2.^2)./990 - (xV2.^3.*yV1.^6.*yV2.^3)./594 - (xV2.^3.*yV1.^5.*yV2.^4)./396 - (7.*xV2.^3.*yV1.^4.*yV2.^5)./1980 - (7.*xV2.^3.*yV1.^3.*yV2.^6)./1485 - (xV2.^3.*yV1.^2.*yV2.^7)./165 - (xV2.^3.*yV1.*yV2.^8)./132 - (xV2.^3.*yV1.^9)./5940);
        I_poly(65,1)=I_poly(65,1)+((yV1.^9.*yV2)./132 - yV1.^10./60 + (3.*yV1.^8.*yV2.^2)./440 + (yV1.^7.*yV2.^3)./165 + (7.*yV1.^6.*yV2.^4)./1320 + (yV1.^5.*yV2.^5)./220 + (yV1.^4.*yV2.^6)./264 + (yV1.^3.*yV2.^7)./330 + (yV1.^2.*yV2.^8)./440 + (yV1.*yV2.^9)./660 + yV2.^10./1320).*xV1.^2 + ((xV2.*yV1.^4.*yV2.^6)./660 - (xV2.*yV1.^9.*yV2)./165 - (xV2.*yV1.^8.*yV2.^2)./220 - (xV2.*yV1.^7.*yV2.^3)./330 - (xV2.*yV1.^6.*yV2.^4)./660 - (xV2.*yV1.^10)./132 + (xV2.*yV1.^3.*yV2.^7)./330 + (xV2.*yV1.^2.*yV2.^8)./220 + (xV2.*yV1.*yV2.^9)./165 + (xV2.*yV2.^10)./132).*xV1 + ((xV2.^2.*yV2.^10)./60 - (xV2.^2.*yV1.^9.*yV2)./660 - (xV2.^2.*yV1.^8.*yV2.^2)./440 - (xV2.^2.*yV1.^7.*yV2.^3)./330 - (xV2.^2.*yV1.^6.*yV2.^4)./264 - (xV2.^2.*yV1.^5.*yV2.^5)./220 - (7.*xV2.^2.*yV1.^4.*yV2.^6)./1320 - (xV2.^2.*yV1.^3.*yV2.^7)./165 - (3.*xV2.^2.*yV1.^2.*yV2.^8)./440 - (xV2.^2.*yV1.*yV2.^9)./132 - (xV2.^2.*yV1.^10)./1320);
        I_poly(66,1)=I_poly(66,1)+-(5.*xV1.*yV1.^11 - xV1.*yV2.^11 + xV2.*yV1.^11 - 5.*xV2.*yV2.^11 - xV1.*yV1.*yV2.^10 - xV1.*yV1.^10.*yV2 + xV2.*yV1.*yV2.^10 + xV2.*yV1.^10.*yV2 - xV1.*yV1.^2.*yV2.^9 - xV1.*yV1.^3.*yV2.^8 - xV1.*yV1.^4.*yV2.^7 - xV1.*yV1.^5.*yV2.^6 - xV1.*yV1.^6.*yV2.^5 - xV1.*yV1.^7.*yV2.^4 - xV1.*yV1.^8.*yV2.^3 - xV1.*yV1.^9.*yV2.^2 + xV2.*yV1.^2.*yV2.^9 + xV2.*yV1.^3.*yV2.^8 + xV2.*yV1.^4.*yV2.^7 + xV2.*yV1.^5.*yV2.^6 + xV2.*yV1.^6.*yV2.^5 + xV2.*yV1.^7.*yV2.^4 + xV2.*yV1.^8.*yV2.^3 + xV2.*yV1.^9.*yV2.^2)./132;
    end
    if poly_order>=11
        I_poly(67,1)=I_poly(67,1)+((11.*yV1)./312 + yV2./156).*xV1.^12 + ((xV2.*yV2)./156 - (xV2.*yV1)./156).*xV1.^11 + ((xV2.^2.*yV2)./156 - (xV2.^2.*yV1)./156).*xV1.^10 + ((xV2.^3.*yV2)./156 - (xV2.^3.*yV1)./156).*xV1.^9 + ((xV2.^4.*yV2)./156 - (xV2.^4.*yV1)./156).*xV1.^8 + ((xV2.^5.*yV2)./156 - (xV2.^5.*yV1)./156).*xV1.^7 + ((xV2.^6.*yV2)./156 - (xV2.^6.*yV1)./156).*xV1.^6 + ((xV2.^7.*yV2)./156 - (xV2.^7.*yV1)./156).*xV1.^5 + ((xV2.^8.*yV2)./156 - (xV2.^8.*yV1)./156).*xV1.^4 + ((xV2.^9.*yV2)./156 - (xV2.^9.*yV1)./156).*xV1.^3 + ((xV2.^10.*yV2)./156 - (xV2.^10.*yV1)./156).*xV1.^2 + ((xV2.^11.*yV2)./156 - (xV2.^11.*yV1)./156).*xV1 - ((xV2.^12.*yV1)./156 + (11.*xV2.^12.*yV2)./312);
        I_poly(68,1)=I_poly(68,1)+(27.*xV1.^11.*yV1.^2 + xV1.^11.*yV2.^2 - xV2.^11.*yV1.^2 - 27.*xV2.^11.*yV2.^2 + 11.*xV1.^11.*yV1.*yV2 - 11.*xV2.^11.*yV1.*yV2 - 2.*xV1.*xV2.^10.*yV1.^2 - 11.*xV1.^10.*xV2.*yV1.^2 + 11.*xV1.*xV2.^10.*yV2.^2 + 2.*xV1.^10.*xV2.*yV2.^2 - 3.*xV1.^2.*xV2.^9.*yV1.^2 - 4.*xV1.^3.*xV2.^8.*yV1.^2 - 5.*xV1.^4.*xV2.^7.*yV1.^2 - 6.*xV1.^5.*xV2.^6.*yV1.^2 - 7.*xV1.^6.*xV2.^5.*yV1.^2 - 8.*xV1.^7.*xV2.^4.*yV1.^2 - 9.*xV1.^8.*xV2.^3.*yV1.^2 - 10.*xV1.^9.*xV2.^2.*yV1.^2 + 10.*xV1.^2.*xV2.^9.*yV2.^2 + 9.*xV1.^3.*xV2.^8.*yV2.^2 + 8.*xV1.^4.*xV2.^7.*yV2.^2 + 7.*xV1.^5.*xV2.^6.*yV2.^2 + 6.*xV1.^6.*xV2.^5.*yV2.^2 + 5.*xV1.^7.*xV2.^4.*yV2.^2 + 4.*xV1.^8.*xV2.^3.*yV2.^2 + 3.*xV1.^9.*xV2.^2.*yV2.^2 - 7.*xV1.^2.*xV2.^9.*yV1.*yV2 - 5.*xV1.^3.*xV2.^8.*yV1.*yV2 - 3.*xV1.^4.*xV2.^7.*yV1.*yV2 - xV1.^5.*xV2.^6.*yV1.*yV2 + xV1.^6.*xV2.^5.*yV1.*yV2 + 3.*xV1.^7.*xV2.^4.*yV1.*yV2 + 5.*xV1.^8.*xV2.^3.*yV1.*yV2 + 7.*xV1.^9.*xV2.^2.*yV1.*yV2 - 9.*xV1.*xV2.^10.*yV1.*yV2 + 9.*xV1.^10.*xV2.*yV1.*yV2)./1716;
        I_poly(69,1)=I_poly(69,1)+(77.*xV1.^10.*yV1.^3 + xV1.^10.*yV2.^3 - xV2.^10.*yV1.^3 - 77.*xV2.^10.*yV2.^3 - 3.*xV1.*xV2.^9.*yV1.^3 - 55.*xV1.^9.*xV2.*yV1.^3 + 55.*xV1.*xV2.^9.*yV2.^3 + 3.*xV1.^9.*xV2.*yV2.^3 + 10.*xV1.^10.*yV1.*yV2.^2 + 55.*xV1.^10.*yV1.^2.*yV2 - 55.*xV2.^10.*yV1.*yV2.^2 - 10.*xV2.^10.*yV1.^2.*yV2 - 6.*xV1.^2.*xV2.^8.*yV1.^3 - 10.*xV1.^3.*xV2.^7.*yV1.^3 - 15.*xV1.^4.*xV2.^6.*yV1.^3 - 21.*xV1.^5.*xV2.^5.*yV1.^3 - 28.*xV1.^6.*xV2.^4.*yV1.^3 - 36.*xV1.^7.*xV2.^3.*yV1.^3 - 45.*xV1.^8.*xV2.^2.*yV1.^3 + 45.*xV1.^2.*xV2.^8.*yV2.^3 + 36.*xV1.^3.*xV2.^7.*yV2.^3 + 28.*xV1.^4.*xV2.^6.*yV2.^3 + 21.*xV1.^5.*xV2.^5.*yV2.^3 + 15.*xV1.^6.*xV2.^4.*yV2.^3 + 10.*xV1.^7.*xV2.^3.*yV2.^3 + 6.*xV1.^8.*xV2.^2.*yV2.^3 - 35.*xV1.*xV2.^9.*yV1.*yV2.^2 - 17.*xV1.*xV2.^9.*yV1.^2.*yV2 + 17.*xV1.^9.*xV2.*yV1.*yV2.^2 + 35.*xV1.^9.*xV2.*yV1.^2.*yV2 - 18.*xV1.^2.*xV2.^8.*yV1.*yV2.^2 - 21.*xV1.^2.*xV2.^8.*yV1.^2.*yV2 - 4.*xV1.^3.*xV2.^7.*yV1.*yV2.^2 - 22.*xV1.^3.*xV2.^7.*yV1.^2.*yV2 + 7.*xV1.^4.*xV2.^6.*yV1.*yV2.^2 - 20.*xV1.^4.*xV2.^6.*yV1.^2.*yV2 + 15.*xV1.^5.*xV2.^5.*yV1.*yV2.^2 - 15.*xV1.^5.*xV2.^5.*yV1.^2.*yV2 + 20.*xV1.^6.*xV2.^4.*yV1.*yV2.^2 - 7.*xV1.^6.*xV2.^4.*yV1.^2.*yV2 + 22.*xV1.^7.*xV2.^3.*yV1.*yV2.^2 + 4.*xV1.^7.*xV2.^3.*yV1.^2.*yV2 + 21.*xV1.^8.*xV2.^2.*yV1.*yV2.^2 + 18.*xV1.^8.*xV2.^2.*yV1.^2.*yV2)./8580;
        I_poly(70,1)=I_poly(70,1)+((5.*yV1.^4)./936 + (yV1.^3.*yV2)./156 + (yV1.^2.*yV2.^2)./572 + (yV1.*yV2.^3)./2860 + yV2.^4./25740).*xV1.^9 + ((5.*xV2.*yV1.^3.*yV2)./1716 - (xV2.*yV1.^4)./156 + (7.*xV2.*yV1.^2.*yV2.^2)./2860 + (23.*xV2.*yV1.*yV2.^3)./25740 + (xV2.*yV2.^4)./6435).*xV1.^8 + ((xV2.^2.*yV1.^3.*yV2)./2145 - (2.*xV2.^2.*yV1.^4)./429 + (xV2.^2.*yV1.^2.*yV2.^2)./429 + (19.*xV2.^2.*yV1.*yV2.^3)./12870 + (xV2.^2.*yV2.^4)./2574).*xV1.^7 + ((7.*xV2.^3.*yV1.^2.*yV2.^2)./4290 - (7.*xV2.^3.*yV1.^3.*yV2)./6435 - (7.*xV2.^3.*yV1.^4)./2145 + (5.*xV2.^3.*yV1.*yV2.^3)./2574 + (xV2.^3.*yV2.^4)./1287).*xV1.^6 + ((xV2.^4.*yV1.^2.*yV2.^2)./1716 - (49.*xV2.^4.*yV1.^3.*yV2)./25740 - (14.*xV2.^4.*yV1.^4)./6435 + (xV2.^4.*yV1.*yV2.^3)./468 + (7.*xV2.^4.*yV2.^4)./5148).*xV1.^5 + ((49.*xV2.^5.*yV1.*yV2.^3)./25740 - (xV2.^5.*yV1.^3.*yV2)./468 - (xV2.^5.*yV1.^2.*yV2.^2)./1716 - (7.*xV2.^5.*yV1.^4)./5148 + (14.*xV2.^5.*yV2.^4)./6435).*xV1.^4 + ((7.*xV2.^6.*yV1.*yV2.^3)./6435 - (5.*xV2.^6.*yV1.^3.*yV2)./2574 - (7.*xV2.^6.*yV1.^2.*yV2.^2)./4290 - (xV2.^6.*yV1.^4)./1287 + (7.*xV2.^6.*yV2.^4)./2145).*xV1.^3 + ((2.*xV2.^7.*yV2.^4)./429 - (19.*xV2.^7.*yV1.^3.*yV2)./12870 - (xV2.^7.*yV1.^2.*yV2.^2)./429 - (xV2.^7.*yV1.*yV2.^3)./2145 - (xV2.^7.*yV1.^4)./2574).*xV1.^2 + ((xV2.^8.*yV2.^4)./156 - (23.*xV2.^8.*yV1.^3.*yV2)./25740 - (7.*xV2.^8.*yV1.^2.*yV2.^2)./2860 - (5.*xV2.^8.*yV1.*yV2.^3)./1716 - (xV2.^8.*yV1.^4)./6435).*xV1 - ((xV2.^9.*yV1.^4)./25740 + (xV2.^9.*yV1.^3.*yV2)./2860 + (xV2.^9.*yV1.^2.*yV2.^2)./572 + (xV2.^9.*yV1.*yV2.^3)./156 + (5.*xV2.^9.*yV2.^4)./936);
        I_poly(71,1)=I_poly(71,1)+((3.*yV1.^5)./1040 + (yV1.^4.*yV2)./156 + (yV1.^3.*yV2.^2)./429 + (yV1.^2.*yV2.^3)./1430 + (yV1.*yV2.^4)./6435 + yV2.^5./51480).*xV1.^8 + ((xV2.*yV1.^4.*yV2)./572 - (xV2.*yV1.^5)./156 + (xV2.*yV1.^3.*yV2.^2)./390 + (19.*xV2.*yV1.^2.*yV2.^3)./12870 + (3.*xV2.*yV1.*yV2.^4)./5720 + (xV2.*yV2.^5)./10296).*xV1.^7 + ((7.*xV2.^2.*yV1.^3.*yV2.^2)./4290 - (7.*xV2.^2.*yV1.^4.*yV2)./8580 - (7.*xV2.^2.*yV1.^5)./1716 + (49.*xV2.^2.*yV1.^2.*yV2.^3)./25740 + (xV2.^2.*yV1.*yV2.^4)./936 + (xV2.^2.*yV2.^5)./3432).*xV1.^6 + ((7.*xV2.^3.*yV1.^3.*yV2.^2)./25740 - (49.*xV2.^3.*yV1.^4.*yV2)./25740 - (7.*xV2.^3.*yV1.^5)./2860 + (xV2.^3.*yV1.^2.*yV2.^3)./572 + (17.*xV2.^3.*yV1.*yV2.^4)./10296 + (7.*xV2.^3.*yV2.^5)./10296).*xV1.^5 + ((5.*xV2.^4.*yV1.^2.*yV2.^3)./5148 - (7.*xV2.^4.*yV1.^4.*yV2)./3432 - (5.*xV2.^4.*yV1.^3.*yV2.^2)./5148 - (7.*xV2.^4.*yV1.^5)./5148 + (7.*xV2.^4.*yV1.*yV2.^4)./3432 + (7.*xV2.^4.*yV2.^5)./5148).*xV1.^4 + ((49.*xV2.^5.*yV1.*yV2.^4)./25740 - (17.*xV2.^5.*yV1.^4.*yV2)./10296 - (xV2.^5.*yV1.^3.*yV2.^2)./572 - (7.*xV2.^5.*yV1.^2.*yV2.^3)./25740 - (7.*xV2.^5.*yV1.^5)./10296 + (7.*xV2.^5.*yV2.^5)./2860).*xV1.^3 + ((7.*xV2.^6.*yV1.*yV2.^4)./8580 - (xV2.^6.*yV1.^4.*yV2)./936 - (49.*xV2.^6.*yV1.^3.*yV2.^2)./25740 - (7.*xV2.^6.*yV1.^2.*yV2.^3)./4290 - (xV2.^6.*yV1.^5)./3432 + (7.*xV2.^6.*yV2.^5)./1716).*xV1.^2 + ((xV2.^7.*yV2.^5)./156 - (3.*xV2.^7.*yV1.^4.*yV2)./5720 - (19.*xV2.^7.*yV1.^3.*yV2.^2)./12870 - (xV2.^7.*yV1.^2.*yV2.^3)./390 - (xV2.^7.*yV1.*yV2.^4)./572 - (xV2.^7.*yV1.^5)./10296).*xV1 - ((xV2.^8.*yV1.^5)./51480 + (xV2.^8.*yV1.^4.*yV2)./6435 + (xV2.^8.*yV1.^3.*yV2.^2)./1430 + (xV2.^8.*yV1.^2.*yV2.^3)./429 + (xV2.^8.*yV1.*yV2.^4)./156 + (3.*xV2.^8.*yV2.^5)./1040);
        I_poly(72,1)=I_poly(72,1)+(yV1.^6./1092 + (yV1.^5.*yV2)./156 + (5.*yV1.^4.*yV2.^2)./1716 + (yV1.^3.*yV2.^3)./858 + (yV1.^2.*yV2.^4)./2574 + (yV1.*yV2.^5)./10296 + yV2.^6./72072).*xV1.^7 + ((xV2.*yV1.^5.*yV2)./1716 - (xV2.*yV1.^6)./156 + (xV2.*yV1.^4.*yV2.^2)./429 + (5.*xV2.*yV1.^3.*yV2.^3)./2574 + (xV2.*yV1.^2.*yV2.^4)./936 + (29.*xV2.*yV1.*yV2.^5)./72072 + (xV2.*yV2.^6)./12012).*xV1.^6 + ((xV2.^2.*yV1.^4.*yV2.^2)./1716 - (xV2.^2.*yV1.^5.*yV2)./572 - (xV2.^2.*yV1.^6)./286 + (xV2.^2.*yV1.^3.*yV2.^3)./572 + (5.*xV2.^2.*yV1.^2.*yV2.^4)./3003 + (23.*xV2.^2.*yV1.*yV2.^5)./24024 + (xV2.^2.*yV2.^6)./3432).*xV1.^5 + ((25.*xV2.^3.*yV1.^3.*yV2.^3)./36036 - (xV2.^3.*yV1.^5.*yV2)./468 - (5.*xV2.^3.*yV1.^4.*yV2.^2)./5148 - (xV2.^3.*yV1.^6)./572 + (125.*xV2.^3.*yV1.^2.*yV2.^4)./72072 + (17.*xV2.^3.*yV1.*yV2.^5)./10296 + (xV2.^3.*yV2.^6)./1287).*xV1.^4 + ((5.*xV2.^4.*yV1.^2.*yV2.^4)./5148 - (17.*xV2.^4.*yV1.^5.*yV2)./10296 - (125.*xV2.^4.*yV1.^4.*yV2.^2)./72072 - (25.*xV2.^4.*yV1.^3.*yV2.^3)./36036 - (xV2.^4.*yV1.^6)./1287 + (xV2.^4.*yV1.*yV2.^5)./468 + (xV2.^4.*yV2.^6)./572).*xV1.^3 + ((xV2.^5.*yV1.*yV2.^5)./572 - (23.*xV2.^5.*yV1.^5.*yV2)./24024 - (5.*xV2.^5.*yV1.^4.*yV2.^2)./3003 - (xV2.^5.*yV1.^3.*yV2.^3)./572 - (xV2.^5.*yV1.^2.*yV2.^4)./1716 - (xV2.^5.*yV1.^6)./3432 + (xV2.^5.*yV2.^6)./286).*xV1.^2 + ((xV2.^6.*yV2.^6)./156 - (29.*xV2.^6.*yV1.^5.*yV2)./72072 - (xV2.^6.*yV1.^4.*yV2.^2)./936 - (5.*xV2.^6.*yV1.^3.*yV2.^3)./2574 - (xV2.^6.*yV1.^2.*yV2.^4)./429 - (xV2.^6.*yV1.*yV2.^5)./1716 - (xV2.^6.*yV1.^6)./12012).*xV1 - ((xV2.^7.*yV1.^6)./72072 + (xV2.^7.*yV1.^5.*yV2)./10296 + (xV2.^7.*yV1.^4.*yV2.^2)./2574 + (xV2.^7.*yV1.^3.*yV2.^3)./858 + (5.*xV2.^7.*yV1.^2.*yV2.^4)./1716 + (xV2.^7.*yV1.*yV2.^5)./156 + (xV2.^7.*yV2.^6)./1092);
        I_poly(73,1)=I_poly(73,1)+((yV1.^6.*yV2)./156 - yV1.^7./1092 + (yV1.^5.*yV2.^2)./286 + (yV1.^4.*yV2.^3)./572 + (yV1.^3.*yV2.^4)./1287 + (yV1.^2.*yV2.^5)./3432 + (yV1.*yV2.^6)./12012 + yV2.^7./72072).*xV1.^6 + ((xV2.*yV1.^5.*yV2.^2)./572 - (xV2.*yV1.^6.*yV2)./1716 - (xV2.*yV1.^7)./156 + (xV2.*yV1.^4.*yV2.^3)./468 + (17.*xV2.*yV1.^3.*yV2.^4)./10296 + (23.*xV2.*yV1.^2.*yV2.^5)./24024 + (29.*xV2.*yV1.*yV2.^6)./72072 + (xV2.*yV2.^7)./10296).*xV1.^5 + ((5.*xV2.^2.*yV1.^4.*yV2.^3)./5148 - (xV2.^2.*yV1.^6.*yV2)./429 - (xV2.^2.*yV1.^5.*yV2.^2)./1716 - (5.*xV2.^2.*yV1.^7)./1716 + (125.*xV2.^2.*yV1.^3.*yV2.^4)./72072 + (5.*xV2.^2.*yV1.^2.*yV2.^5)./3003 + (xV2.^2.*yV1.*yV2.^6)./936 + (xV2.^2.*yV2.^7)./2574).*xV1.^4 + ((25.*xV2.^3.*yV1.^3.*yV2.^4)./36036 - (5.*xV2.^3.*yV1.^6.*yV2)./2574 - (xV2.^3.*yV1.^5.*yV2.^2)./572 - (25.*xV2.^3.*yV1.^4.*yV2.^3)./36036 - (xV2.^3.*yV1.^7)./858 + (xV2.^3.*yV1.^2.*yV2.^5)./572 + (5.*xV2.^3.*yV1.*yV2.^6)./2574 + (xV2.^3.*yV2.^7)./858).*xV1.^3 + ((xV2.^4.*yV1.^2.*yV2.^5)./1716 - (xV2.^4.*yV1.^6.*yV2)./936 - (5.*xV2.^4.*yV1.^5.*yV2.^2)./3003 - (125.*xV2.^4.*yV1.^4.*yV2.^3)./72072 - (5.*xV2.^4.*yV1.^3.*yV2.^4)./5148 - (xV2.^4.*yV1.^7)./2574 + (xV2.^4.*yV1.*yV2.^6)./429 + (5.*xV2.^4.*yV2.^7)./1716).*xV1.^2 + ((xV2.^5.*yV1.*yV2.^6)./1716 - (29.*xV2.^5.*yV1.^6.*yV2)./72072 - (23.*xV2.^5.*yV1.^5.*yV2.^2)./24024 - (17.*xV2.^5.*yV1.^4.*yV2.^3)./10296 - (xV2.^5.*yV1.^3.*yV2.^4)./468 - (xV2.^5.*yV1.^2.*yV2.^5)./572 - (xV2.^5.*yV1.^7)./10296 + (xV2.^5.*yV2.^7)./156).*xV1 + ((xV2.^6.*yV2.^7)./1092 - (xV2.^6.*yV1.^6.*yV2)./12012 - (xV2.^6.*yV1.^5.*yV2.^2)./3432 - (xV2.^6.*yV1.^4.*yV2.^3)./1287 - (xV2.^6.*yV1.^3.*yV2.^4)./572 - (xV2.^6.*yV1.^2.*yV2.^5)./286 - (xV2.^6.*yV1.*yV2.^6)./156 - (xV2.^6.*yV1.^7)./72072);
        I_poly(74,1)=I_poly(74,1)+((yV1.^7.*yV2)./156 - (3.*yV1.^8)./1040 + (7.*yV1.^6.*yV2.^2)./1716 + (7.*yV1.^5.*yV2.^3)./2860 + (7.*yV1.^4.*yV2.^4)./5148 + (7.*yV1.^3.*yV2.^5)./10296 + (yV1.^2.*yV2.^6)./3432 + (yV1.*yV2.^7)./10296 + yV2.^8./51480).*xV1.^5 + ((7.*xV2.*yV1.^6.*yV2.^2)./8580 - (xV2.*yV1.^7.*yV2)./572 - (xV2.*yV1.^8)./156 + (49.*xV2.*yV1.^5.*yV2.^3)./25740 + (7.*xV2.*yV1.^4.*yV2.^4)./3432 + (17.*xV2.*yV1.^3.*yV2.^5)./10296 + (xV2.*yV1.^2.*yV2.^6)./936 + (3.*xV2.*yV1.*yV2.^7)./5720 + (xV2.*yV2.^8)./6435).*xV1.^4 + ((5.*xV2.^2.*yV1.^4.*yV2.^4)./5148 - (xV2.^2.*yV1.^7.*yV2)./390 - (7.*xV2.^2.*yV1.^6.*yV2.^2)./4290 - (7.*xV2.^2.*yV1.^5.*yV2.^3)./25740 - (xV2.^2.*yV1.^8)./429 + (xV2.^2.*yV1.^3.*yV2.^5)./572 + (49.*xV2.^2.*yV1.^2.*yV2.^6)./25740 + (19.*xV2.^2.*yV1.*yV2.^7)./12870 + (xV2.^2.*yV2.^8)./1430).*xV1.^3 + ((7.*xV2.^3.*yV1.^3.*yV2.^5)./25740 - (19.*xV2.^3.*yV1.^7.*yV2)./12870 - (49.*xV2.^3.*yV1.^6.*yV2.^2)./25740 - (xV2.^3.*yV1.^5.*yV2.^3)./572 - (5.*xV2.^3.*yV1.^4.*yV2.^4)./5148 - (xV2.^3.*yV1.^8)./1430 + (7.*xV2.^3.*yV1.^2.*yV2.^6)./4290 + (xV2.^3.*yV1.*yV2.^7)./390 + (xV2.^3.*yV2.^8)./429).*xV1.^2 + ((xV2.^4.*yV1.*yV2.^7)./572 - (3.*xV2.^4.*yV1.^7.*yV2)./5720 - (xV2.^4.*yV1.^6.*yV2.^2)./936 - (17.*xV2.^4.*yV1.^5.*yV2.^3)./10296 - (7.*xV2.^4.*yV1.^4.*yV2.^4)./3432 - (49.*xV2.^4.*yV1.^3.*yV2.^5)./25740 - (7.*xV2.^4.*yV1.^2.*yV2.^6)./8580 - (xV2.^4.*yV1.^8)./6435 + (xV2.^4.*yV2.^8)./156).*xV1 + ((3.*xV2.^5.*yV2.^8)./1040 - (xV2.^5.*yV1.^7.*yV2)./10296 - (xV2.^5.*yV1.^6.*yV2.^2)./3432 - (7.*xV2.^5.*yV1.^5.*yV2.^3)./10296 - (7.*xV2.^5.*yV1.^4.*yV2.^4)./5148 - (7.*xV2.^5.*yV1.^3.*yV2.^5)./2860 - (7.*xV2.^5.*yV1.^2.*yV2.^6)./1716 - (xV2.^5.*yV1.*yV2.^7)./156 - (xV2.^5.*yV1.^8)./51480);
        I_poly(75,1)=I_poly(75,1)+((yV1.^8.*yV2)./156 - (5.*yV1.^9)./936 + (2.*yV1.^7.*yV2.^2)./429 + (7.*yV1.^6.*yV2.^3)./2145 + (14.*yV1.^5.*yV2.^4)./6435 + (7.*yV1.^4.*yV2.^5)./5148 + (yV1.^3.*yV2.^6)./1287 + (yV1.^2.*yV2.^7)./2574 + (yV1.*yV2.^8)./6435 + yV2.^9./25740).*xV1.^4 + ((7.*xV2.*yV1.^6.*yV2.^3)./6435 - (5.*xV2.*yV1.^8.*yV2)./1716 - (xV2.*yV1.^7.*yV2.^2)./2145 - (xV2.*yV1.^9)./156 + (49.*xV2.*yV1.^5.*yV2.^4)./25740 + (xV2.*yV1.^4.*yV2.^5)./468 + (5.*xV2.*yV1.^3.*yV2.^6)./2574 + (19.*xV2.*yV1.^2.*yV2.^7)./12870 + (23.*xV2.*yV1.*yV2.^8)./25740 + (xV2.*yV2.^9)./2860).*xV1.^3 + ((xV2.^2.*yV1.^4.*yV2.^5)./1716 - (7.*xV2.^2.*yV1.^8.*yV2)./2860 - (xV2.^2.*yV1.^7.*yV2.^2)./429 - (7.*xV2.^2.*yV1.^6.*yV2.^3)./4290 - (xV2.^2.*yV1.^5.*yV2.^4)./1716 - (xV2.^2.*yV1.^9)./572 + (7.*xV2.^2.*yV1.^3.*yV2.^6)./4290 + (xV2.^2.*yV1.^2.*yV2.^7)./429 + (7.*xV2.^2.*yV1.*yV2.^8)./2860 + (xV2.^2.*yV2.^9)./572).*xV1.^2 + ((xV2.^3.*yV1.^2.*yV2.^7)./2145 - (23.*xV2.^3.*yV1.^8.*yV2)./25740 - (19.*xV2.^3.*yV1.^7.*yV2.^2)./12870 - (5.*xV2.^3.*yV1.^6.*yV2.^3)./2574 - (xV2.^3.*yV1.^5.*yV2.^4)./468 - (49.*xV2.^3.*yV1.^4.*yV2.^5)./25740 - (7.*xV2.^3.*yV1.^3.*yV2.^6)./6435 - (xV2.^3.*yV1.^9)./2860 + (5.*xV2.^3.*yV1.*yV2.^8)./1716 + (xV2.^3.*yV2.^9)./156).*xV1 + ((5.*xV2.^4.*yV2.^9)./936 - (xV2.^4.*yV1.^8.*yV2)./6435 - (xV2.^4.*yV1.^7.*yV2.^2)./2574 - (xV2.^4.*yV1.^6.*yV2.^3)./1287 - (7.*xV2.^4.*yV1.^5.*yV2.^4)./5148 - (14.*xV2.^4.*yV1.^4.*yV2.^5)./6435 - (7.*xV2.^4.*yV1.^3.*yV2.^6)./2145 - (2.*xV2.^4.*yV1.^2.*yV2.^7)./429 - (xV2.^4.*yV1.*yV2.^8)./156 - (xV2.^4.*yV1.^9)./25740);
        I_poly(76,1)=I_poly(76,1)+((yV1.^9.*yV2)./156 - (7.*yV1.^10)./780 + (3.*yV1.^8.*yV2.^2)./572 + (3.*yV1.^7.*yV2.^3)./715 + (7.*yV1.^6.*yV2.^4)./2145 + (7.*yV1.^5.*yV2.^5)./2860 + (yV1.^4.*yV2.^6)./572 + (yV1.^3.*yV2.^7)./858 + (yV1.^2.*yV2.^8)./1430 + (yV1.*yV2.^9)./2860 + yV2.^10./8580).*xV1.^3 + ((7.*xV2.*yV1.^6.*yV2.^4)./8580 - (7.*xV2.*yV1.^9.*yV2)./1716 - (3.*xV2.*yV1.^8.*yV2.^2)./1430 - (xV2.*yV1.^7.*yV2.^3)./2145 - (xV2.*yV1.^10)./156 + (xV2.*yV1.^5.*yV2.^5)./572 + (xV2.*yV1.^4.*yV2.^6)./429 + (xV2.*yV1.^3.*yV2.^7)./390 + (7.*xV2.*yV1.^2.*yV2.^8)./2860 + (17.*xV2.*yV1.*yV2.^9)./8580 + (xV2.*yV2.^10)./858).*xV1.^2 + ((xV2.^2.*yV1.^3.*yV2.^7)./2145 - (17.*xV2.^2.*yV1.^9.*yV2)./8580 - (7.*xV2.^2.*yV1.^8.*yV2.^2)./2860 - (xV2.^2.*yV1.^7.*yV2.^3)./390 - (xV2.^2.*yV1.^6.*yV2.^4)./429 - (xV2.^2.*yV1.^5.*yV2.^5)./572 - (7.*xV2.^2.*yV1.^4.*yV2.^6)./8580 - (xV2.^2.*yV1.^10)./858 + (3.*xV2.^2.*yV1.^2.*yV2.^8)./1430 + (7.*xV2.^2.*yV1.*yV2.^9)./1716 + (xV2.^2.*yV2.^10)./156).*xV1 + ((7.*xV2.^3.*yV2.^10)./780 - (xV2.^3.*yV1.^9.*yV2)./2860 - (xV2.^3.*yV1.^8.*yV2.^2)./1430 - (xV2.^3.*yV1.^7.*yV2.^3)./858 - (xV2.^3.*yV1.^6.*yV2.^4)./572 - (7.*xV2.^3.*yV1.^5.*yV2.^5)./2860 - (7.*xV2.^3.*yV1.^4.*yV2.^6)./2145 - (3.*xV2.^3.*yV1.^3.*yV2.^7)./715 - (3.*xV2.^3.*yV1.^2.*yV2.^8)./572 - (xV2.^3.*yV1.*yV2.^9)./156 - (xV2.^3.*yV1.^10)./8580);
        I_poly(77,1)=I_poly(77,1)+-(27.*xV1.^2.*yV1.^11 - xV1.^2.*yV2.^11 + xV2.^2.*yV1.^11 - 27.*xV2.^2.*yV2.^11 + 11.*xV1.*xV2.*yV1.^11 - 11.*xV1.*xV2.*yV2.^11 - 2.*xV1.^2.*yV1.*yV2.^10 - 11.*xV1.^2.*yV1.^10.*yV2 + 11.*xV2.^2.*yV1.*yV2.^10 + 2.*xV2.^2.*yV1.^10.*yV2 - 3.*xV1.^2.*yV1.^2.*yV2.^9 - 4.*xV1.^2.*yV1.^3.*yV2.^8 - 5.*xV1.^2.*yV1.^4.*yV2.^7 - 6.*xV1.^2.*yV1.^5.*yV2.^6 - 7.*xV1.^2.*yV1.^6.*yV2.^5 - 8.*xV1.^2.*yV1.^7.*yV2.^4 - 9.*xV1.^2.*yV1.^8.*yV2.^3 - 10.*xV1.^2.*yV1.^9.*yV2.^2 + 10.*xV2.^2.*yV1.^2.*yV2.^9 + 9.*xV2.^2.*yV1.^3.*yV2.^8 + 8.*xV2.^2.*yV1.^4.*yV2.^7 + 7.*xV2.^2.*yV1.^5.*yV2.^6 + 6.*xV2.^2.*yV1.^6.*yV2.^5 + 5.*xV2.^2.*yV1.^7.*yV2.^4 + 4.*xV2.^2.*yV1.^8.*yV2.^3 + 3.*xV2.^2.*yV1.^9.*yV2.^2 - 7.*xV1.*xV2.*yV1.^2.*yV2.^9 - 5.*xV1.*xV2.*yV1.^3.*yV2.^8 - 3.*xV1.*xV2.*yV1.^4.*yV2.^7 - xV1.*xV2.*yV1.^5.*yV2.^6 + xV1.*xV2.*yV1.^6.*yV2.^5 + 3.*xV1.*xV2.*yV1.^7.*yV2.^4 + 5.*xV1.*xV2.*yV1.^8.*yV2.^3 + 7.*xV1.*xV2.*yV1.^9.*yV2.^2 - 9.*xV1.*xV2.*yV1.*yV2.^10 + 9.*xV1.*xV2.*yV1.^10.*yV2)./1716;
        I_poly(78,1)=I_poly(78,1)+((yV1.^11.*yV2)./156 - (11.*yV1.^12)./312 + (yV1.^10.*yV2.^2)./156 + (yV1.^9.*yV2.^3)./156 + (yV1.^8.*yV2.^4)./156 + (yV1.^7.*yV2.^5)./156 + (yV1.^6.*yV2.^6)./156 + (yV1.^5.*yV2.^7)./156 + (yV1.^4.*yV2.^8)./156 + (yV1.^3.*yV2.^9)./156 + (yV1.^2.*yV2.^10)./156 + (yV1.*yV2.^11)./156 + yV2.^12./156).*xV1 + ((11.*xV2.*yV2.^12)./312 - (xV2.*yV1.^11.*yV2)./156 - (xV2.*yV1.^10.*yV2.^2)./156 - (xV2.*yV1.^9.*yV2.^3)./156 - (xV2.*yV1.^8.*yV2.^4)./156 - (xV2.*yV1.^7.*yV2.^5)./156 - (xV2.*yV1.^6.*yV2.^6)./156 - (xV2.*yV1.^5.*yV2.^7)./156 - (xV2.*yV1.^4.*yV2.^8)./156 - (xV2.*yV1.^3.*yV2.^9)./156 - (xV2.*yV1.^2.*yV2.^10)./156 - (xV2.*yV1.*yV2.^11)./156 - (xV2.*yV1.^12)./156);
    end
    if poly_order>=12
        I_poly(79,1)=I_poly(79,1)+(6.*xV1.^13.*yV1 + xV1.^13.*yV2 - xV2.^13.*yV1 - 6.*xV2.^13.*yV2 - xV1.*xV2.^12.*yV1 - xV1.^12.*xV2.*yV1 + xV1.*xV2.^12.*yV2 + xV1.^12.*xV2.*yV2 - xV1.^2.*xV2.^11.*yV1 - xV1.^3.*xV2.^10.*yV1 - xV1.^4.*xV2.^9.*yV1 - xV1.^5.*xV2.^8.*yV1 - xV1.^6.*xV2.^7.*yV1 - xV1.^7.*xV2.^6.*yV1 - xV1.^8.*xV2.^5.*yV1 - xV1.^9.*xV2.^4.*yV1 - xV1.^10.*xV2.^3.*yV1 - xV1.^11.*xV2.^2.*yV1 + xV1.^2.*xV2.^11.*yV2 + xV1.^3.*xV2.^10.*yV2 + xV1.^4.*xV2.^9.*yV2 + xV1.^5.*xV2.^8.*yV2 + xV1.^6.*xV2.^7.*yV2 + xV1.^7.*xV2.^6.*yV2 + xV1.^8.*xV2.^5.*yV2 + xV1.^9.*xV2.^4.*yV2 + xV1.^10.*xV2.^3.*yV2 + xV1.^11.*xV2.^2.*yV2)./182;
        I_poly(80,1)=I_poly(80,1)+(65.*xV1.^12.*yV1.^2 + 2.*xV1.^12.*yV2.^2 - 2.*xV2.^12.*yV1.^2 - 65.*xV2.^12.*yV2.^2 + 24.*xV1.^12.*yV1.*yV2 - 24.*xV2.^12.*yV1.*yV2 - 4.*xV1.*xV2.^11.*yV1.^2 - 24.*xV1.^11.*xV2.*yV1.^2 + 24.*xV1.*xV2.^11.*yV2.^2 + 4.*xV1.^11.*xV2.*yV2.^2 - 6.*xV1.^2.*xV2.^10.*yV1.^2 - 8.*xV1.^3.*xV2.^9.*yV1.^2 - 10.*xV1.^4.*xV2.^8.*yV1.^2 - 12.*xV1.^5.*xV2.^7.*yV1.^2 - 14.*xV1.^6.*xV2.^6.*yV1.^2 - 16.*xV1.^7.*xV2.^5.*yV1.^2 - 18.*xV1.^8.*xV2.^4.*yV1.^2 - 20.*xV1.^9.*xV2.^3.*yV1.^2 - 22.*xV1.^10.*xV2.^2.*yV1.^2 + 22.*xV1.^2.*xV2.^10.*yV2.^2 + 20.*xV1.^3.*xV2.^9.*yV2.^2 + 18.*xV1.^4.*xV2.^8.*yV2.^2 + 16.*xV1.^5.*xV2.^7.*yV2.^2 + 14.*xV1.^6.*xV2.^6.*yV2.^2 + 12.*xV1.^7.*xV2.^5.*yV2.^2 + 10.*xV1.^8.*xV2.^4.*yV2.^2 + 8.*xV1.^9.*xV2.^3.*yV2.^2 + 6.*xV1.^10.*xV2.^2.*yV2.^2 - 16.*xV1.^2.*xV2.^10.*yV1.*yV2 - 12.*xV1.^3.*xV2.^9.*yV1.*yV2 - 8.*xV1.^4.*xV2.^8.*yV1.*yV2 - 4.*xV1.^5.*xV2.^7.*yV1.*yV2 + 4.*xV1.^7.*xV2.^5.*yV1.*yV2 + 8.*xV1.^8.*xV2.^4.*yV1.*yV2 + 12.*xV1.^9.*xV2.^3.*yV1.*yV2 + 16.*xV1.^10.*xV2.^2.*yV1.*yV2 - 20.*xV1.*xV2.^11.*yV1.*yV2 + 20.*xV1.^11.*xV2.*yV1.*yV2)./4368;
        I_poly(81,1)=I_poly(81,1)+(104.*xV1.^11.*yV1.^3 + xV1.^11.*yV2.^3 - xV2.^11.*yV1.^3 - 104.*xV2.^11.*yV2.^3 - 3.*xV1.*xV2.^10.*yV1.^3 - 66.*xV1.^10.*xV2.*yV1.^3 + 66.*xV1.*xV2.^10.*yV2.^3 + 3.*xV1.^10.*xV2.*yV2.^3 + 11.*xV1.^11.*yV1.*yV2.^2 + 66.*xV1.^11.*yV1.^2.*yV2 - 66.*xV2.^11.*yV1.*yV2.^2 - 11.*xV2.^11.*yV1.^2.*yV2 - 6.*xV1.^2.*xV2.^9.*yV1.^3 - 10.*xV1.^3.*xV2.^8.*yV1.^3 - 15.*xV1.^4.*xV2.^7.*yV1.^3 - 21.*xV1.^5.*xV2.^6.*yV1.^3 - 28.*xV1.^6.*xV2.^5.*yV1.^3 - 36.*xV1.^7.*xV2.^4.*yV1.^3 - 45.*xV1.^8.*xV2.^3.*yV1.^3 - 55.*xV1.^9.*xV2.^2.*yV1.^3 + 55.*xV1.^2.*xV2.^9.*yV2.^3 + 45.*xV1.^3.*xV2.^8.*yV2.^3 + 36.*xV1.^4.*xV2.^7.*yV2.^3 + 28.*xV1.^5.*xV2.^6.*yV2.^3 + 21.*xV1.^6.*xV2.^5.*yV2.^3 + 15.*xV1.^7.*xV2.^4.*yV2.^3 + 10.*xV1.^8.*xV2.^3.*yV2.^3 + 6.*xV1.^9.*xV2.^2.*yV2.^3 - 44.*xV1.*xV2.^10.*yV1.*yV2.^2 - 19.*xV1.*xV2.^10.*yV1.^2.*yV2 + 19.*xV1.^10.*xV2.*yV1.*yV2.^2 + 44.*xV1.^10.*xV2.*yV1.^2.*yV2 - 25.*xV1.^2.*xV2.^9.*yV1.*yV2.^2 - 24.*xV1.^2.*xV2.^9.*yV1.^2.*yV2 - 9.*xV1.^3.*xV2.^8.*yV1.*yV2.^2 - 26.*xV1.^3.*xV2.^8.*yV1.^2.*yV2 + 4.*xV1.^4.*xV2.^7.*yV1.*yV2.^2 - 25.*xV1.^4.*xV2.^7.*yV1.^2.*yV2 + 14.*xV1.^5.*xV2.^6.*yV1.*yV2.^2 - 21.*xV1.^5.*xV2.^6.*yV1.^2.*yV2 + 21.*xV1.^6.*xV2.^5.*yV1.*yV2.^2 - 14.*xV1.^6.*xV2.^5.*yV1.^2.*yV2 + 25.*xV1.^7.*xV2.^4.*yV1.*yV2.^2 - 4.*xV1.^7.*xV2.^4.*yV1.^2.*yV2 + 26.*xV1.^8.*xV2.^3.*yV1.*yV2.^2 + 9.*xV1.^8.*xV2.^3.*yV1.^2.*yV2 + 24.*xV1.^9.*xV2.^2.*yV1.*yV2.^2 + 25.*xV1.^9.*xV2.^2.*yV1.^2.*yV2)./12012;
        I_poly(82,1)=I_poly(82,1)+((3.*yV1.^4)./560 + (yV1.^3.*yV2)./182 + (yV1.^2.*yV2.^2)./728 + (yV1.*yV2.^3)./4004 + yV2.^4./40040).*xV1.^10 + ((xV2.*yV1.^3.*yV2)./364 - (xV2.*yV1.^4)./182 + (2.*xV2.*yV1.^2.*yV2.^2)./1001 + (xV2.*yV1.*yV2.^3)./1540 + (xV2.*yV2.^4)./10010).*xV1.^9 + ((3.*xV2.^2.*yV1.^3.*yV2)./4004 - (3.*xV2.^2.*yV1.^4)./728 + (81.*xV2.^2.*yV1.^2.*yV2.^2)./40040 + (xV2.^2.*yV1.*yV2.^3)./910 + (xV2.^2.*yV2.^4)./4004).*xV1.^8 + ((8.*xV2.^3.*yV1.^2.*yV2.^2)./5005 - (3.*xV2.^3.*yV1.^3.*yV2)./5005 - (3.*xV2.^3.*yV1.^4)./1001 + (3.*xV2.^3.*yV1.*yV2.^3)./2002 + (xV2.^3.*yV2.^4)./2002).*xV1.^7 + ((xV2.^4.*yV1.^2.*yV2.^2)./1144 - (xV2.^4.*yV1.^3.*yV2)./715 - (3.*xV2.^4.*yV1.^4)./1430 + (xV2.^4.*yV1.*yV2.^3)./572 + (xV2.^4.*yV2.^4)./1144).*xV1.^6 + ((xV2.^5.*yV1.*yV2.^3)./572 - (xV2.^5.*yV1.^3.*yV2)./572 - (xV2.^5.*yV1.^4)./715 + (xV2.^5.*yV2.^4)./715).*xV1.^5 + ((xV2.^6.*yV1.*yV2.^3)./715 - (xV2.^6.*yV1.^3.*yV2)./572 - (xV2.^6.*yV1.^2.*yV2.^2)./1144 - (xV2.^6.*yV1.^4)./1144 + (3.*xV2.^6.*yV2.^4)./1430).*xV1.^4 + ((3.*xV2.^7.*yV1.*yV2.^3)./5005 - (3.*xV2.^7.*yV1.^3.*yV2)./2002 - (8.*xV2.^7.*yV1.^2.*yV2.^2)./5005 - (xV2.^7.*yV1.^4)./2002 + (3.*xV2.^7.*yV2.^4)./1001).*xV1.^3 + ((3.*xV2.^8.*yV2.^4)./728 - (xV2.^8.*yV1.^3.*yV2)./910 - (81.*xV2.^8.*yV1.^2.*yV2.^2)./40040 - (3.*xV2.^8.*yV1.*yV2.^3)./4004 - (xV2.^8.*yV1.^4)./4004).*xV1.^2 + ((xV2.^9.*yV2.^4)./182 - (xV2.^9.*yV1.^3.*yV2)./1540 - (2.*xV2.^9.*yV1.^2.*yV2.^2)./1001 - (xV2.^9.*yV1.*yV2.^3)./364 - (xV2.^9.*yV1.^4)./10010).*xV1 - ((xV2.^10.*yV1.^4)./40040 + (xV2.^10.*yV1.^3.*yV2)./4004 + (xV2.^10.*yV1.^2.*yV2.^2)./728 + (xV2.^10.*yV1.*yV2.^3)./182 + (3.*xV2.^10.*yV2.^4)./560);
        I_poly(83,1)=I_poly(83,1)+(yV1.^5./315 + (yV1.^4.*yV2)./182 + (yV1.^3.*yV2.^2)./546 + (yV1.^2.*yV2.^3)./2002 + (yV1.*yV2.^4)./10010 + yV2.^5./90090).*xV1.^9 + ((xV2.*yV1.^4.*yV2)./546 - (xV2.*yV1.^5)./182 + (xV2.*yV1.^3.*yV2.^2)./462 + (xV2.*yV1.^2.*yV2.^3)./910 + (31.*xV2.*yV1.*yV2.^4)./90090 + (xV2.*yV2.^5)./18018).*xV1.^8 + ((8.*xV2.^2.*yV1.^3.*yV2.^2)./5005 - (xV2.^2.*yV1.^4.*yV2)./3003 - (xV2.^2.*yV1.^5)./273 + (68.*xV2.^2.*yV1.^2.*yV2.^3)./45045 + (xV2.^2.*yV1.*yV2.^4)./1386 + (xV2.^2.*yV2.^5)./6006).*xV1.^7 + ((4.*xV2.^3.*yV1.^3.*yV2.^2)./6435 - (xV2.^3.*yV1.^4.*yV2)./715 - (xV2.^3.*yV1.^5)./429 + (2.*xV2.^3.*yV1.^2.*yV2.^3)./1287 + (xV2.^3.*yV1.*yV2.^4)./858 + (xV2.^3.*yV2.^5)./2574).*xV1.^6 + ((xV2.^4.*yV1.^2.*yV2.^3)./858 - (xV2.^4.*yV1.^4.*yV2)./585 - (xV2.^4.*yV1.^3.*yV2.^2)./2574 - (xV2.^4.*yV1.^5)./715 + (2.*xV2.^4.*yV1.*yV2.^4)./1287 + (xV2.^4.*yV2.^5)./1287).*xV1.^5 + ((xV2.^5.*yV1.^2.*yV2.^3)./2574 - (2.*xV2.^5.*yV1.^4.*yV2)./1287 - (xV2.^5.*yV1.^3.*yV2.^2)./858 - (xV2.^5.*yV1.^5)./1287 + (xV2.^5.*yV1.*yV2.^4)./585 + (xV2.^5.*yV2.^5)./715).*xV1.^4 + ((xV2.^6.*yV1.*yV2.^4)./715 - (xV2.^6.*yV1.^4.*yV2)./858 - (2.*xV2.^6.*yV1.^3.*yV2.^2)./1287 - (4.*xV2.^6.*yV1.^2.*yV2.^3)./6435 - (xV2.^6.*yV1.^5)./2574 + (xV2.^6.*yV2.^5)./429).*xV1.^3 + ((xV2.^7.*yV1.*yV2.^4)./3003 - (xV2.^7.*yV1.^4.*yV2)./1386 - (68.*xV2.^7.*yV1.^3.*yV2.^2)./45045 - (8.*xV2.^7.*yV1.^2.*yV2.^3)./5005 - (xV2.^7.*yV1.^5)./6006 + (xV2.^7.*yV2.^5)./273).*xV1.^2 + ((xV2.^8.*yV2.^5)./182 - (31.*xV2.^8.*yV1.^4.*yV2)./90090 - (xV2.^8.*yV1.^3.*yV2.^2)./910 - (xV2.^8.*yV1.^2.*yV2.^3)./462 - (xV2.^8.*yV1.*yV2.^4)./546 - (xV2.^8.*yV1.^5)./18018).*xV1 - ((xV2.^9.*yV1.^5)./90090 + (xV2.^9.*yV1.^4.*yV2)./10010 + (xV2.^9.*yV1.^3.*yV2.^2)./2002 + (xV2.^9.*yV1.^2.*yV2.^3)./546 + (xV2.^9.*yV1.*yV2.^4)./182 + (xV2.^9.*yV2.^5)./315);
        I_poly(84,1)=I_poly(84,1)+(yV1.^6./672 + (yV1.^5.*yV2)./182 + (5.*yV1.^4.*yV2.^2)./2184 + (5.*yV1.^3.*yV2.^3)./6006 + (yV1.^2.*yV2.^4)./4004 + (yV1.*yV2.^5)./18018 + yV2.^6./144144).*xV1.^8 + ((xV2.*yV1.^5.*yV2)./1092 - (xV2.*yV1.^6)./182 + (25.*xV2.*yV1.^4.*yV2.^2)./12012 + (3.*xV2.*yV1.^3.*yV2.^3)./2002 + (xV2.*yV1.^2.*yV2.^4)./1386 + (17.*xV2.*yV1.*yV2.^5)./72072 + (xV2.*yV2.^6)./24024).*xV1.^7 + ((xV2.^2.*yV1.^4.*yV2.^2)./1144 - (xV2.^2.*yV1.^5.*yV2)./858 - (xV2.^2.*yV1.^6)./312 + (2.*xV2.^2.*yV1.^3.*yV2.^3)./1287 + (25.*xV2.^2.*yV1.^2.*yV2.^4)./20592 + (xV2.^2.*yV1.*yV2.^5)./1716 + (xV2.^2.*yV2.^6)./6864).*xV1.^6 + ((5.*xV2.^3.*yV1.^3.*yV2.^3)./5148 - (xV2.^3.*yV1.^5.*yV2)./572 - (xV2.^3.*yV1.^4.*yV2.^2)./2574 - (xV2.^3.*yV1.^6)./572 + (5.*xV2.^3.*yV1.^2.*yV2.^4)./3432 + (xV2.^3.*yV1.*yV2.^5)./936 + (xV2.^3.*yV2.^6)./2574).*xV1.^5 + ((25.*xV2.^4.*yV1.^2.*yV2.^4)./20592 - (2.*xV2.^4.*yV1.^5.*yV2)./1287 - (25.*xV2.^4.*yV1.^4.*yV2.^2)./20592 - (xV2.^4.*yV1.^6)./1144 + (2.*xV2.^4.*yV1.*yV2.^5)./1287 + (xV2.^4.*yV2.^6)./1144).*xV1.^4 + ((xV2.^5.*yV1.^2.*yV2.^4)./2574 - (xV2.^5.*yV1.^5.*yV2)./936 - (5.*xV2.^5.*yV1.^4.*yV2.^2)./3432 - (5.*xV2.^5.*yV1.^3.*yV2.^3)./5148 - (xV2.^5.*yV1.^6)./2574 + (xV2.^5.*yV1.*yV2.^5)./572 + (xV2.^5.*yV2.^6)./572).*xV1.^3 + ((xV2.^6.*yV1.*yV2.^5)./858 - (xV2.^6.*yV1.^5.*yV2)./1716 - (25.*xV2.^6.*yV1.^4.*yV2.^2)./20592 - (2.*xV2.^6.*yV1.^3.*yV2.^3)./1287 - (xV2.^6.*yV1.^2.*yV2.^4)./1144 - (xV2.^6.*yV1.^6)./6864 + (xV2.^6.*yV2.^6)./312).*xV1.^2 + ((xV2.^7.*yV2.^6)./182 - (17.*xV2.^7.*yV1.^5.*yV2)./72072 - (xV2.^7.*yV1.^4.*yV2.^2)./1386 - (3.*xV2.^7.*yV1.^3.*yV2.^3)./2002 - (25.*xV2.^7.*yV1.^2.*yV2.^4)./12012 - (xV2.^7.*yV1.*yV2.^5)./1092 - (xV2.^7.*yV1.^6)./24024).*xV1 - ((xV2.^8.*yV1.^6)./144144 + (xV2.^8.*yV1.^5.*yV2)./18018 + (xV2.^8.*yV1.^4.*yV2.^2)./4004 + (5.*xV2.^8.*yV1.^3.*yV2.^3)./6006 + (5.*xV2.^8.*yV1.^2.*yV2.^4)./2184 + (xV2.^8.*yV1.*yV2.^5)./182 + (xV2.^8.*yV2.^6)./672);
        I_poly(85,1)=I_poly(85,1)+((xV1.*yV2 - xV2.*yV1).*(924.*xV1.^6.*yV1.^6 + xV1.^6.*yV2.^6 + xV2.^6.*yV1.^6 + 924.*xV2.^6.*yV2.^6 + 7.*xV1.*xV2.^5.*yV1.^6 + 462.*xV1.^5.*xV2.*yV1.^6 + 462.*xV1.*xV2.^5.*yV2.^6 + 7.*xV1.^5.*xV2.*yV2.^6 + 7.*xV1.^6.*yV1.*yV2.^5 + 462.*xV1.^6.*yV1.^5.*yV2 + 462.*xV2.^6.*yV1.*yV2.^5 + 7.*xV2.^6.*yV1.^5.*yV2 + 28.*xV1.^2.*xV2.^4.*yV1.^6 + 84.*xV1.^3.*xV2.^3.*yV1.^6 + 210.*xV1.^4.*xV2.^2.*yV1.^6 + 210.*xV1.^2.*xV2.^4.*yV2.^6 + 84.*xV1.^3.*xV2.^3.*yV2.^6 + 28.*xV1.^4.*xV2.^2.*yV2.^6 + 28.*xV1.^6.*yV1.^2.*yV2.^4 + 84.*xV1.^6.*yV1.^3.*yV2.^3 + 210.*xV1.^6.*yV1.^4.*yV2.^2 + 210.*xV2.^6.*yV1.^2.*yV2.^4 + 84.*xV2.^6.*yV1.^3.*yV2.^3 + 28.*xV2.^6.*yV1.^4.*yV2.^2 + 504.*xV1.*xV2.^5.*yV1.*yV2.^5 + 36.*xV1.*xV2.^5.*yV1.^5.*yV2 + 36.*xV1.^5.*xV2.*yV1.*yV2.^5 + 504.*xV1.^5.*xV2.*yV1.^5.*yV2 + 378.*xV1.*xV2.^5.*yV1.^2.*yV2.^4 + 224.*xV1.*xV2.^5.*yV1.^3.*yV2.^3 + 105.*xV1.*xV2.^5.*yV1.^4.*yV2.^2 + 378.*xV1.^2.*xV2.^4.*yV1.*yV2.^5 + 105.*xV1.^2.*xV2.^4.*yV1.^5.*yV2 + 224.*xV1.^3.*xV2.^3.*yV1.*yV2.^5 + 224.*xV1.^3.*xV2.^3.*yV1.^5.*yV2 + 105.*xV1.^4.*xV2.^2.*yV1.*yV2.^5 + 378.*xV1.^4.*xV2.^2.*yV1.^5.*yV2 + 105.*xV1.^5.*xV2.*yV1.^2.*yV2.^4 + 224.*xV1.^5.*xV2.*yV1.^3.*yV2.^3 + 378.*xV1.^5.*xV2.*yV1.^4.*yV2.^2 + 420.*xV1.^2.*xV2.^4.*yV1.^2.*yV2.^4 + 350.*xV1.^2.*xV2.^4.*yV1.^3.*yV2.^3 + 225.*xV1.^2.*xV2.^4.*yV1.^4.*yV2.^2 + 350.*xV1.^3.*xV2.^3.*yV1.^2.*yV2.^4 + 400.*xV1.^3.*xV2.^3.*yV1.^3.*yV2.^3 + 350.*xV1.^3.*xV2.^3.*yV1.^4.*yV2.^2 + 225.*xV1.^4.*xV2.^2.*yV1.^2.*yV2.^4 + 350.*xV1.^4.*xV2.^2.*yV1.^3.*yV2.^3 + 420.*xV1.^4.*xV2.^2.*yV1.^4.*yV2.^2))./168168;
        I_poly(86,1)=I_poly(86,1)+((yV1.^7.*yV2)./182 - yV1.^8./672 + (yV1.^6.*yV2.^2)./312 + (yV1.^5.*yV2.^3)./572 + (yV1.^4.*yV2.^4)./1144 + (yV1.^3.*yV2.^5)./2574 + (yV1.^2.*yV2.^6)./6864 + (yV1.*yV2.^7)./24024 + yV2.^8./144144).*xV1.^6 + ((xV2.*yV1.^6.*yV2.^2)./858 - (xV2.*yV1.^7.*yV2)./1092 - (xV2.*yV1.^8)./182 + (xV2.*yV1.^5.*yV2.^3)./572 + (2.*xV2.*yV1.^4.*yV2.^4)./1287 + (xV2.*yV1.^3.*yV2.^5)./936 + (xV2.*yV1.^2.*yV2.^6)./1716 + (17.*xV2.*yV1.*yV2.^7)./72072 + (xV2.*yV2.^8)./18018).*xV1.^5 + ((xV2.^2.*yV1.^5.*yV2.^3)./2574 - (25.*xV2.^2.*yV1.^7.*yV2)./12012 - (xV2.^2.*yV1.^6.*yV2.^2)./1144 - (5.*xV2.^2.*yV1.^8)./2184 + (25.*xV2.^2.*yV1.^4.*yV2.^4)./20592 + (5.*xV2.^2.*yV1.^3.*yV2.^5)./3432 + (25.*xV2.^2.*yV1.^2.*yV2.^6)./20592 + (xV2.^2.*yV1.*yV2.^7)./1386 + (xV2.^2.*yV2.^8)./4004).*xV1.^4 + ((5.*xV2.^3.*yV1.^3.*yV2.^5)./5148 - (3.*xV2.^3.*yV1.^7.*yV2)./2002 - (2.*xV2.^3.*yV1.^6.*yV2.^2)./1287 - (5.*xV2.^3.*yV1.^5.*yV2.^3)./5148 - (5.*xV2.^3.*yV1.^8)./6006 + (2.*xV2.^3.*yV1.^2.*yV2.^6)./1287 + (3.*xV2.^3.*yV1.*yV2.^7)./2002 + (5.*xV2.^3.*yV2.^8)./6006).*xV1.^3 + ((xV2.^4.*yV1.^2.*yV2.^6)./1144 - (xV2.^4.*yV1.^7.*yV2)./1386 - (25.*xV2.^4.*yV1.^6.*yV2.^2)./20592 - (5.*xV2.^4.*yV1.^5.*yV2.^3)./3432 - (25.*xV2.^4.*yV1.^4.*yV2.^4)./20592 - (xV2.^4.*yV1.^3.*yV2.^5)./2574 - (xV2.^4.*yV1.^8)./4004 + (25.*xV2.^4.*yV1.*yV2.^7)./12012 + (5.*xV2.^4.*yV2.^8)./2184).*xV1.^2 + ((xV2.^5.*yV1.*yV2.^7)./1092 - (17.*xV2.^5.*yV1.^7.*yV2)./72072 - (xV2.^5.*yV1.^6.*yV2.^2)./1716 - (xV2.^5.*yV1.^5.*yV2.^3)./936 - (2.*xV2.^5.*yV1.^4.*yV2.^4)./1287 - (xV2.^5.*yV1.^3.*yV2.^5)./572 - (xV2.^5.*yV1.^2.*yV2.^6)./858 - (xV2.^5.*yV1.^8)./18018 + (xV2.^5.*yV2.^8)./182).*xV1 + ((xV2.^6.*yV2.^8)./672 - (xV2.^6.*yV1.^7.*yV2)./24024 - (xV2.^6.*yV1.^6.*yV2.^2)./6864 - (xV2.^6.*yV1.^5.*yV2.^3)./2574 - (xV2.^6.*yV1.^4.*yV2.^4)./1144 - (xV2.^6.*yV1.^3.*yV2.^5)./572 - (xV2.^6.*yV1.^2.*yV2.^6)./312 - (xV2.^6.*yV1.*yV2.^7)./182 - (xV2.^6.*yV1.^8)./144144);
        I_poly(87,1)=I_poly(87,1)+((yV1.^8.*yV2)./182 - yV1.^9./315 + (yV1.^7.*yV2.^2)./273 + (yV1.^6.*yV2.^3)./429 + (yV1.^5.*yV2.^4)./715 + (yV1.^4.*yV2.^5)./1287 + (yV1.^3.*yV2.^6)./2574 + (yV1.^2.*yV2.^7)./6006 + (yV1.*yV2.^8)./18018 + yV2.^9./90090).*xV1.^5 + ((xV2.*yV1.^7.*yV2.^2)./3003 - (xV2.*yV1.^8.*yV2)./546 - (xV2.*yV1.^9)./182 + (xV2.*yV1.^6.*yV2.^3)./715 + (xV2.*yV1.^5.*yV2.^4)./585 + (2.*xV2.*yV1.^4.*yV2.^5)./1287 + (xV2.*yV1.^3.*yV2.^6)./858 + (xV2.*yV1.^2.*yV2.^7)./1386 + (31.*xV2.*yV1.*yV2.^8)./90090 + (xV2.*yV2.^9)./10010).*xV1.^4 + ((xV2.^2.*yV1.^5.*yV2.^4)./2574 - (xV2.^2.*yV1.^8.*yV2)./462 - (8.*xV2.^2.*yV1.^7.*yV2.^2)./5005 - (4.*xV2.^2.*yV1.^6.*yV2.^3)./6435 - (xV2.^2.*yV1.^9)./546 + (xV2.^2.*yV1.^4.*yV2.^5)./858 + (2.*xV2.^2.*yV1.^3.*yV2.^6)./1287 + (68.*xV2.^2.*yV1.^2.*yV2.^7)./45045 + (xV2.^2.*yV1.*yV2.^8)./910 + (xV2.^2.*yV2.^9)./2002).*xV1.^3 + ((4.*xV2.^3.*yV1.^3.*yV2.^6)./6435 - (xV2.^3.*yV1.^8.*yV2)./910 - (68.*xV2.^3.*yV1.^7.*yV2.^2)./45045 - (2.*xV2.^3.*yV1.^6.*yV2.^3)./1287 - (xV2.^3.*yV1.^5.*yV2.^4)./858 - (xV2.^3.*yV1.^4.*yV2.^5)./2574 - (xV2.^3.*yV1.^9)./2002 + (8.*xV2.^3.*yV1.^2.*yV2.^7)./5005 + (xV2.^3.*yV1.*yV2.^8)./462 + (xV2.^3.*yV2.^9)./546).*xV1.^2 + ((xV2.^4.*yV1.*yV2.^8)./546 - (31.*xV2.^4.*yV1.^8.*yV2)./90090 - (xV2.^4.*yV1.^7.*yV2.^2)./1386 - (xV2.^4.*yV1.^6.*yV2.^3)./858 - (2.*xV2.^4.*yV1.^5.*yV2.^4)./1287 - (xV2.^4.*yV1.^4.*yV2.^5)./585 - (xV2.^4.*yV1.^3.*yV2.^6)./715 - (xV2.^4.*yV1.^2.*yV2.^7)./3003 - (xV2.^4.*yV1.^9)./10010 + (xV2.^4.*yV2.^9)./182).*xV1 + ((xV2.^5.*yV2.^9)./315 - (xV2.^5.*yV1.^8.*yV2)./18018 - (xV2.^5.*yV1.^7.*yV2.^2)./6006 - (xV2.^5.*yV1.^6.*yV2.^3)./2574 - (xV2.^5.*yV1.^5.*yV2.^4)./1287 - (xV2.^5.*yV1.^4.*yV2.^5)./715 - (xV2.^5.*yV1.^3.*yV2.^6)./429 - (xV2.^5.*yV1.^2.*yV2.^7)./273 - (xV2.^5.*yV1.*yV2.^8)./182 - (xV2.^5.*yV1.^9)./90090);
        I_poly(88,1)=I_poly(88,1)+((yV1.^9.*yV2)./182 - (3.*yV1.^10)./560 + (3.*yV1.^8.*yV2.^2)./728 + (3.*yV1.^7.*yV2.^3)./1001 + (3.*yV1.^6.*yV2.^4)./1430 + (yV1.^5.*yV2.^5)./715 + (yV1.^4.*yV2.^6)./1144 + (yV1.^3.*yV2.^7)./2002 + (yV1.^2.*yV2.^8)./4004 + (yV1.*yV2.^9)./10010 + yV2.^10./40040).*xV1.^4 + ((3.*xV2.*yV1.^7.*yV2.^3)./5005 - (xV2.*yV1.^9.*yV2)./364 - (3.*xV2.*yV1.^8.*yV2.^2)./4004 - (xV2.*yV1.^10)./182 + (xV2.*yV1.^6.*yV2.^4)./715 + (xV2.*yV1.^5.*yV2.^5)./572 + (xV2.*yV1.^4.*yV2.^6)./572 + (3.*xV2.*yV1.^3.*yV2.^7)./2002 + (xV2.*yV1.^2.*yV2.^8)./910 + (xV2.*yV1.*yV2.^9)./1540 + (xV2.*yV2.^10)./4004).*xV1.^3 + ((xV2.^2.*yV1.^4.*yV2.^6)./1144 - (2.*xV2.^2.*yV1.^9.*yV2)./1001 - (81.*xV2.^2.*yV1.^8.*yV2.^2)./40040 - (8.*xV2.^2.*yV1.^7.*yV2.^3)./5005 - (xV2.^2.*yV1.^6.*yV2.^4)./1144 - (xV2.^2.*yV1.^10)./728 + (8.*xV2.^2.*yV1.^3.*yV2.^7)./5005 + (81.*xV2.^2.*yV1.^2.*yV2.^8)./40040 + (2.*xV2.^2.*yV1.*yV2.^9)./1001 + (xV2.^2.*yV2.^10)./728).*xV1.^2 + ((3.*xV2.^3.*yV1.^2.*yV2.^8)./4004 - (xV2.^3.*yV1.^9.*yV2)./1540 - (xV2.^3.*yV1.^8.*yV2.^2)./910 - (3.*xV2.^3.*yV1.^7.*yV2.^3)./2002 - (xV2.^3.*yV1.^6.*yV2.^4)./572 - (xV2.^3.*yV1.^5.*yV2.^5)./572 - (xV2.^3.*yV1.^4.*yV2.^6)./715 - (3.*xV2.^3.*yV1.^3.*yV2.^7)./5005 - (xV2.^3.*yV1.^10)./4004 + (xV2.^3.*yV1.*yV2.^9)./364 + (xV2.^3.*yV2.^10)./182).*xV1 + ((3.*xV2.^4.*yV2.^10)./560 - (xV2.^4.*yV1.^9.*yV2)./10010 - (xV2.^4.*yV1.^8.*yV2.^2)./4004 - (xV2.^4.*yV1.^7.*yV2.^3)./2002 - (xV2.^4.*yV1.^6.*yV2.^4)./1144 - (xV2.^4.*yV1.^5.*yV2.^5)./715 - (3.*xV2.^4.*yV1.^4.*yV2.^6)./1430 - (3.*xV2.^4.*yV1.^3.*yV2.^7)./1001 - (3.*xV2.^4.*yV1.^2.*yV2.^8)./728 - (xV2.^4.*yV1.*yV2.^9)./182 - (xV2.^4.*yV1.^10)./40040);
        I_poly(89,1)=I_poly(89,1)+-(104.*xV1.^3.*yV1.^11 - xV1.^3.*yV2.^11 + xV2.^3.*yV1.^11 - 104.*xV2.^3.*yV2.^11 + 11.*xV1.*xV2.^2.*yV1.^11 + 66.*xV1.^2.*xV2.*yV1.^11 - 66.*xV1.*xV2.^2.*yV2.^11 - 11.*xV1.^2.*xV2.*yV2.^11 - 3.*xV1.^3.*yV1.*yV2.^10 - 66.*xV1.^3.*yV1.^10.*yV2 + 66.*xV2.^3.*yV1.*yV2.^10 + 3.*xV2.^3.*yV1.^10.*yV2 - 6.*xV1.^3.*yV1.^2.*yV2.^9 - 10.*xV1.^3.*yV1.^3.*yV2.^8 - 15.*xV1.^3.*yV1.^4.*yV2.^7 - 21.*xV1.^3.*yV1.^5.*yV2.^6 - 28.*xV1.^3.*yV1.^6.*yV2.^5 - 36.*xV1.^3.*yV1.^7.*yV2.^4 - 45.*xV1.^3.*yV1.^8.*yV2.^3 - 55.*xV1.^3.*yV1.^9.*yV2.^2 + 55.*xV2.^3.*yV1.^2.*yV2.^9 + 45.*xV2.^3.*yV1.^3.*yV2.^8 + 36.*xV2.^3.*yV1.^4.*yV2.^7 + 28.*xV2.^3.*yV1.^5.*yV2.^6 + 21.*xV2.^3.*yV1.^6.*yV2.^5 + 15.*xV2.^3.*yV1.^7.*yV2.^4 + 10.*xV2.^3.*yV1.^8.*yV2.^3 + 6.*xV2.^3.*yV1.^9.*yV2.^2 - 44.*xV1.*xV2.^2.*yV1.*yV2.^10 + 19.*xV1.*xV2.^2.*yV1.^10.*yV2 - 19.*xV1.^2.*xV2.*yV1.*yV2.^10 + 44.*xV1.^2.*xV2.*yV1.^10.*yV2 - 25.*xV1.*xV2.^2.*yV1.^2.*yV2.^9 - 9.*xV1.*xV2.^2.*yV1.^3.*yV2.^8 + 4.*xV1.*xV2.^2.*yV1.^4.*yV2.^7 + 14.*xV1.*xV2.^2.*yV1.^5.*yV2.^6 + 21.*xV1.*xV2.^2.*yV1.^6.*yV2.^5 + 25.*xV1.*xV2.^2.*yV1.^7.*yV2.^4 + 26.*xV1.*xV2.^2.*yV1.^8.*yV2.^3 + 24.*xV1.*xV2.^2.*yV1.^9.*yV2.^2 - 24.*xV1.^2.*xV2.*yV1.^2.*yV2.^9 - 26.*xV1.^2.*xV2.*yV1.^3.*yV2.^8 - 25.*xV1.^2.*xV2.*yV1.^4.*yV2.^7 - 21.*xV1.^2.*xV2.*yV1.^5.*yV2.^6 - 14.*xV1.^2.*xV2.*yV1.^6.*yV2.^5 - 4.*xV1.^2.*xV2.*yV1.^7.*yV2.^4 + 9.*xV1.^2.*xV2.*yV1.^8.*yV2.^3 + 25.*xV1.^2.*xV2.*yV1.^9.*yV2.^2)./12012;
        I_poly(90,1)=I_poly(90,1)+((yV1.^11.*yV2)./182 - (5.*yV1.^12)./336 + (11.*yV1.^10.*yV2.^2)./2184 + (5.*yV1.^9.*yV2.^3)./1092 + (3.*yV1.^8.*yV2.^4)./728 + (yV1.^7.*yV2.^5)./273 + (yV1.^6.*yV2.^6)./312 + (yV1.^5.*yV2.^7)./364 + (5.*yV1.^4.*yV2.^8)./2184 + (yV1.^3.*yV2.^9)./546 + (yV1.^2.*yV2.^10)./728 + (yV1.*yV2.^11)./1092 + yV2.^12./2184).*xV1.^2 + ((xV2.*yV1.^5.*yV2.^7)./1092 - (5.*xV2.*yV1.^11.*yV2)./1092 - (xV2.*yV1.^10.*yV2.^2)./273 - (xV2.*yV1.^9.*yV2.^3)./364 - (xV2.*yV1.^8.*yV2.^4)./546 - (xV2.*yV1.^7.*yV2.^5)./1092 - (xV2.*yV1.^12)./182 + (xV2.*yV1.^4.*yV2.^8)./546 + (xV2.*yV1.^3.*yV2.^9)./364 + (xV2.*yV1.^2.*yV2.^10)./273 + (5.*xV2.*yV1.*yV2.^11)./1092 + (xV2.*yV2.^12)./182).*xV1 + ((5.*xV2.^2.*yV2.^12)./336 - (xV2.^2.*yV1.^11.*yV2)./1092 - (xV2.^2.*yV1.^10.*yV2.^2)./728 - (xV2.^2.*yV1.^9.*yV2.^3)./546 - (5.*xV2.^2.*yV1.^8.*yV2.^4)./2184 - (xV2.^2.*yV1.^7.*yV2.^5)./364 - (xV2.^2.*yV1.^6.*yV2.^6)./312 - (xV2.^2.*yV1.^5.*yV2.^7)./273 - (3.*xV2.^2.*yV1.^4.*yV2.^8)./728 - (5.*xV2.^2.*yV1.^3.*yV2.^9)./1092 - (11.*xV2.^2.*yV1.^2.*yV2.^10)./2184 - (xV2.^2.*yV1.*yV2.^11)./182 - (xV2.^2.*yV1.^12)./2184);
        I_poly(91,1)=I_poly(91,1)+-(6.*xV1.*yV1.^13 - xV1.*yV2.^13 + xV2.*yV1.^13 - 6.*xV2.*yV2.^13 - xV1.*yV1.*yV2.^12 - xV1.*yV1.^12.*yV2 + xV2.*yV1.*yV2.^12 + xV2.*yV1.^12.*yV2 - xV1.*yV1.^2.*yV2.^11 - xV1.*yV1.^3.*yV2.^10 - xV1.*yV1.^4.*yV2.^9 - xV1.*yV1.^5.*yV2.^8 - xV1.*yV1.^6.*yV2.^7 - xV1.*yV1.^7.*yV2.^6 - xV1.*yV1.^8.*yV2.^5 - xV1.*yV1.^9.*yV2.^4 - xV1.*yV1.^10.*yV2.^3 - xV1.*yV1.^11.*yV2.^2 + xV2.*yV1.^2.*yV2.^11 + xV2.*yV1.^3.*yV2.^10 + xV2.*yV1.^4.*yV2.^9 + xV2.*yV1.^5.*yV2.^8 + xV2.*yV1.^6.*yV2.^7 + xV2.*yV1.^7.*yV2.^6 + xV2.*yV1.^8.*yV2.^5 + xV2.*yV1.^9.*yV2.^4 + xV2.*yV1.^10.*yV2.^3 + xV2.*yV1.^11.*yV2.^2)./182;
    end
    if poly_order>=13
        I_poly(92,1)=I_poly(92,1)+((13.*yV1)./420 + yV2./210).*xV1.^14 + ((xV2.*yV2)./210 - (xV2.*yV1)./210).*xV1.^13 + ((xV2.^2.*yV2)./210 - (xV2.^2.*yV1)./210).*xV1.^12 + ((xV2.^3.*yV2)./210 - (xV2.^3.*yV1)./210).*xV1.^11 + ((xV2.^4.*yV2)./210 - (xV2.^4.*yV1)./210).*xV1.^10 + ((xV2.^5.*yV2)./210 - (xV2.^5.*yV1)./210).*xV1.^9 + ((xV2.^6.*yV2)./210 - (xV2.^6.*yV1)./210).*xV1.^8 + ((xV2.^7.*yV2)./210 - (xV2.^7.*yV1)./210).*xV1.^7 + ((xV2.^8.*yV2)./210 - (xV2.^8.*yV1)./210).*xV1.^6 + ((xV2.^9.*yV2)./210 - (xV2.^9.*yV1)./210).*xV1.^5 + ((xV2.^10.*yV2)./210 - (xV2.^10.*yV1)./210).*xV1.^4 + ((xV2.^11.*yV2)./210 - (xV2.^11.*yV1)./210).*xV1.^3 + ((xV2.^12.*yV2)./210 - (xV2.^12.*yV1)./210).*xV1.^2 + ((xV2.^13.*yV2)./210 - (xV2.^13.*yV1)./210).*xV1 - ((xV2.^14.*yV1)./210 + (13.*xV2.^14.*yV2)./420);
        I_poly(93,1)=I_poly(93,1)+(77.*xV1.^13.*yV1.^2 + 2.*xV1.^13.*yV2.^2 - 2.*xV2.^13.*yV1.^2 - 77.*xV2.^13.*yV2.^2 + 26.*xV1.^13.*yV1.*yV2 - 26.*xV2.^13.*yV1.*yV2 - 4.*xV1.*xV2.^12.*yV1.^2 - 26.*xV1.^12.*xV2.*yV1.^2 + 26.*xV1.*xV2.^12.*yV2.^2 + 4.*xV1.^12.*xV2.*yV2.^2 - 6.*xV1.^2.*xV2.^11.*yV1.^2 - 8.*xV1.^3.*xV2.^10.*yV1.^2 - 10.*xV1.^4.*xV2.^9.*yV1.^2 - 12.*xV1.^5.*xV2.^8.*yV1.^2 - 14.*xV1.^6.*xV2.^7.*yV1.^2 - 16.*xV1.^7.*xV2.^6.*yV1.^2 - 18.*xV1.^8.*xV2.^5.*yV1.^2 - 20.*xV1.^9.*xV2.^4.*yV1.^2 - 22.*xV1.^10.*xV2.^3.*yV1.^2 - 24.*xV1.^11.*xV2.^2.*yV1.^2 + 24.*xV1.^2.*xV2.^11.*yV2.^2 + 22.*xV1.^3.*xV2.^10.*yV2.^2 + 20.*xV1.^4.*xV2.^9.*yV2.^2 + 18.*xV1.^5.*xV2.^8.*yV2.^2 + 16.*xV1.^6.*xV2.^7.*yV2.^2 + 14.*xV1.^7.*xV2.^6.*yV2.^2 + 12.*xV1.^8.*xV2.^5.*yV2.^2 + 10.*xV1.^9.*xV2.^4.*yV2.^2 + 8.*xV1.^10.*xV2.^3.*yV2.^2 + 6.*xV1.^11.*xV2.^2.*yV2.^2 - 18.*xV1.^2.*xV2.^11.*yV1.*yV2 - 14.*xV1.^3.*xV2.^10.*yV1.*yV2 - 10.*xV1.^4.*xV2.^9.*yV1.*yV2 - 6.*xV1.^5.*xV2.^8.*yV1.*yV2 - 2.*xV1.^6.*xV2.^7.*yV1.*yV2 + 2.*xV1.^7.*xV2.^6.*yV1.*yV2 + 6.*xV1.^8.*xV2.^5.*yV1.*yV2 + 10.*xV1.^9.*xV2.^4.*yV1.*yV2 + 14.*xV1.^10.*xV2.^3.*yV1.*yV2 + 18.*xV1.^11.*xV2.^2.*yV1.*yV2 - 22.*xV1.*xV2.^12.*yV1.*yV2 + 22.*xV1.^12.*xV2.*yV1.*yV2)./5460;
        I_poly(94,1)=I_poly(94,1)+(yV1.^3./120 + (yV1.^2.*yV2)./210 + (yV1.*yV2.^2)./1365 + yV2.^3./16380).*xV1.^12 + ((3.*xV2.*yV1.^2.*yV2)./910 - (xV2.*yV1.^3)./210 + (xV2.*yV1.*yV2.^2)./780 + (xV2.*yV2.^3)./5460).*xV1.^11 + ((11.*xV2.^2.*yV1.^2.*yV2)./5460 - (11.*xV2.^2.*yV1.^3)./2730 + (3.*xV2.^2.*yV1.*yV2.^2)./1820 + (xV2.^2.*yV2.^3)./2730).*xV1.^10 + ((xV2.^3.*yV1.^2.*yV2)./1092 - (11.*xV2.^3.*yV1.^3)./3276 + (xV2.^3.*yV1.*yV2.^2)./546 + (xV2.^3.*yV2.^3)./1638).*xV1.^9 + ((xV2.^4.*yV1.*yV2.^2)./546 - (xV2.^4.*yV1.^3)./364 + (xV2.^4.*yV2.^3)./1092).*xV1.^8 + ((3.*xV2.^5.*yV1.*yV2.^2)./1820 - (xV2.^5.*yV1.^2.*yV2)./1365 - (xV2.^5.*yV1.^3)./455 + (xV2.^5.*yV2.^3)./780).*xV1.^7 + ((xV2.^6.*yV1.*yV2.^2)./780 - (xV2.^6.*yV1.^2.*yV2)./780 - (xV2.^6.*yV1.^3)./585 + (xV2.^6.*yV2.^3)./585).*xV1.^6 + ((xV2.^7.*yV1.*yV2.^2)./1365 - (3.*xV2.^7.*yV1.^2.*yV2)./1820 - (xV2.^7.*yV1.^3)./780 + (xV2.^7.*yV2.^3)./455).*xV1.^5 + ((xV2.^8.*yV2.^3)./364 - (xV2.^8.*yV1.^2.*yV2)./546 - (xV2.^8.*yV1.^3)./1092).*xV1.^4 + ((11.*xV2.^9.*yV2.^3)./3276 - (xV2.^9.*yV1.^2.*yV2)./546 - (xV2.^9.*yV1.*yV2.^2)./1092 - (xV2.^9.*yV1.^3)./1638).*xV1.^3 + ((11.*xV2.^10.*yV2.^3)./2730 - (3.*xV2.^10.*yV1.^2.*yV2)./1820 - (11.*xV2.^10.*yV1.*yV2.^2)./5460 - (xV2.^10.*yV1.^3)./2730).*xV1.^2 + ((xV2.^11.*yV2.^3)./210 - (xV2.^11.*yV1.^2.*yV2)./780 - (3.*xV2.^11.*yV1.*yV2.^2)./910 - (xV2.^11.*yV1.^3)./5460).*xV1 - ((xV2.^12.*yV1.^3)./16380 + (xV2.^12.*yV1.^2.*yV2)./1365 + (xV2.^12.*yV1.*yV2.^2)./210 + (xV2.^12.*yV2.^3)./120);
        I_poly(95,1)=I_poly(95,1)+((7.*yV1.^4)./1320 + (yV1.^3.*yV2)./210 + (yV1.^2.*yV2.^2)./910 + (yV1.*yV2.^3)./5460 + yV2.^4./60060).*xV1.^11 + ((xV2.*yV1.^3.*yV2)./390 - (xV2.*yV1.^4)./210 + (3.*xV2.*yV1.^2.*yV2.^2)./1820 + (29.*xV2.*yV1.*yV2.^3)./60060 + (xV2.*yV2.^4)./15015).*xV1.^10 + ((xV2.^2.*yV1.^3.*yV2)./1092 - (xV2.^2.*yV1.^4)./273 + (xV2.^2.*yV1.^2.*yV2.^2)./572 + (5.*xV2.^2.*yV1.*yV2.^3)./6006 + (xV2.^2.*yV2.^4)./6006).*xV1.^9 + ((3.*xV2.^3.*yV1.^2.*yV2.^2)./2002 - (xV2.^3.*yV1.^3.*yV2)./4004 - (xV2.^3.*yV1.^4)./364 + (xV2.^3.*yV1.*yV2.^3)./858 + (xV2.^3.*yV2.^4)./3003).*xV1.^8 + ((xV2.^4.*yV1.^2.*yV2.^2)./1001 - (xV2.^4.*yV1.^3.*yV2)./1001 - (2.*xV2.^4.*yV1.^4)./1001 + (17.*xV2.^4.*yV1.*yV2.^3)./12012 + (xV2.^4.*yV2.^4)./1716).*xV1.^7 + ((xV2.^5.*yV1.^2.*yV2.^2)./2860 - (xV2.^5.*yV1.^3.*yV2)./715 - (xV2.^5.*yV1.^4)./715 + (xV2.^5.*yV1.*yV2.^3)./660 + (2.*xV2.^5.*yV2.^4)./2145).*xV1.^6 + ((xV2.^6.*yV1.*yV2.^3)./715 - (xV2.^6.*yV1.^3.*yV2)./660 - (xV2.^6.*yV1.^2.*yV2.^2)./2860 - (2.*xV2.^6.*yV1.^4)./2145 + (xV2.^6.*yV2.^4)./715).*xV1.^5 + ((xV2.^7.*yV1.*yV2.^3)./1001 - (17.*xV2.^7.*yV1.^3.*yV2)./12012 - (xV2.^7.*yV1.^2.*yV2.^2)./1001 - (xV2.^7.*yV1.^4)./1716 + (2.*xV2.^7.*yV2.^4)./1001).*xV1.^4 + ((xV2.^8.*yV1.*yV2.^3)./4004 - (xV2.^8.*yV1.^3.*yV2)./858 - (3.*xV2.^8.*yV1.^2.*yV2.^2)./2002 - (xV2.^8.*yV1.^4)./3003 + (xV2.^8.*yV2.^4)./364).*xV1.^3 + ((xV2.^9.*yV2.^4)./273 - (5.*xV2.^9.*yV1.^3.*yV2)./6006 - (xV2.^9.*yV1.^2.*yV2.^2)./572 - (xV2.^9.*yV1.*yV2.^3)./1092 - (xV2.^9.*yV1.^4)./6006).*xV1.^2 + ((xV2.^10.*yV2.^4)./210 - (29.*xV2.^10.*yV1.^3.*yV2)./60060 - (3.*xV2.^10.*yV1.^2.*yV2.^2)./1820 - (xV2.^10.*yV1.*yV2.^3)./390 - (xV2.^10.*yV1.^4)./15015).*xV1 - ((xV2.^11.*yV1.^4)./60060 + (xV2.^11.*yV1.^3.*yV2)./5460 + (xV2.^11.*yV1.^2.*yV2.^2)./910 + (xV2.^11.*yV1.*yV2.^3)./210 + (7.*xV2.^11.*yV2.^4)./1320);
        I_poly(96,1)=I_poly(96,1)+(yV1.^5./300 + (yV1.^4.*yV2)./210 + (2.*yV1.^3.*yV2.^2)./1365 + (yV1.^2.*yV2.^3)./2730 + (yV1.*yV2.^4)./15015 + yV2.^5./150150).*xV1.^10 + ((xV2.*yV1.^4.*yV2)./546 - (xV2.*yV1.^5)./210 + (xV2.*yV1.^3.*yV2.^2)./546 + (5.*xV2.*yV1.^2.*yV2.^3)./6006 + (xV2.*yV1.*yV2.^4)./4290 + (xV2.*yV2.^5)./30030).*xV1.^9 + ((3.*xV2.^2.*yV1.^3.*yV2.^2)./2002 - (3.*xV2.^2.*yV1.^5)./910 + (6.*xV2.^2.*yV1.^2.*yV2.^3)./5005 + (xV2.^2.*yV1.*yV2.^4)./2002 + (xV2.^2.*yV2.^5)./10010).*xV1.^8 + ((4.*xV2.^3.*yV1.^3.*yV2.^2)./5005 - (xV2.^3.*yV1.^4.*yV2)./1001 - (xV2.^3.*yV1.^5)./455 + (4.*xV2.^3.*yV1.^2.*yV2.^3)./3003 + (5.*xV2.^3.*yV1.*yV2.^4)./6006 + (xV2.^3.*yV2.^5)./4290).*xV1.^7 + ((xV2.^4.*yV1.^2.*yV2.^3)./858 - (xV2.^4.*yV1.^4.*yV2)./715 - (xV2.^4.*yV1.^5)./715 + (xV2.^4.*yV1.*yV2.^4)./858 + (xV2.^4.*yV2.^5)./2145).*xV1.^6 + ((xV2.^5.*yV1.^2.*yV2.^3)./1430 - (xV2.^5.*yV1.^4.*yV2)./715 - (xV2.^5.*yV1.^3.*yV2.^2)./1430 - (3.*xV2.^5.*yV1.^5)./3575 + (xV2.^5.*yV1.*yV2.^4)./715 + (3.*xV2.^5.*yV2.^5)./3575).*xV1.^5 + ((xV2.^6.*yV1.*yV2.^4)./715 - (xV2.^6.*yV1.^4.*yV2)./858 - (xV2.^6.*yV1.^3.*yV2.^2)./858 - (xV2.^6.*yV1.^5)./2145 + (xV2.^6.*yV2.^5)./715).*xV1.^4 + ((xV2.^7.*yV1.*yV2.^4)./1001 - (5.*xV2.^7.*yV1.^4.*yV2)./6006 - (4.*xV2.^7.*yV1.^3.*yV2.^2)./3003 - (4.*xV2.^7.*yV1.^2.*yV2.^3)./5005 - (xV2.^7.*yV1.^5)./4290 + (xV2.^7.*yV2.^5)./455).*xV1.^3 + ((3.*xV2.^8.*yV2.^5)./910 - (xV2.^8.*yV1.^4.*yV2)./2002 - (6.*xV2.^8.*yV1.^3.*yV2.^2)./5005 - (3.*xV2.^8.*yV1.^2.*yV2.^3)./2002 - (xV2.^8.*yV1.^5)./10010).*xV1.^2 + ((xV2.^9.*yV2.^5)./210 - (xV2.^9.*yV1.^4.*yV2)./4290 - (5.*xV2.^9.*yV1.^3.*yV2.^2)./6006 - (xV2.^9.*yV1.^2.*yV2.^3)./546 - (xV2.^9.*yV1.*yV2.^4)./546 - (xV2.^9.*yV1.^5)./30030).*xV1 - ((xV2.^10.*yV1.^5)./150150 + (xV2.^10.*yV1.^4.*yV2)./15015 + (xV2.^10.*yV1.^3.*yV2.^2)./2730 + (2.*xV2.^10.*yV1.^2.*yV2.^3)./1365 + (xV2.^10.*yV1.*yV2.^4)./210 + (xV2.^10.*yV2.^5)./300);
        I_poly(97,1)=I_poly(97,1)+(yV1.^6./540 + (yV1.^5.*yV2)./210 + (yV1.^4.*yV2.^2)./546 + (yV1.^3.*yV2.^3)./1638 + (yV1.^2.*yV2.^4)./6006 + (yV1.*yV2.^5)./30030 + yV2.^6./270270).*xV1.^9 + ((xV2.*yV1.^5.*yV2)./910 - (xV2.*yV1.^6)./210 + (xV2.*yV1.^4.*yV2.^2)./546 + (xV2.*yV1.^3.*yV2.^3)./858 + (xV2.*yV1.^2.*yV2.^4)./2002 + (xV2.*yV1.*yV2.^5)./6930 + (xV2.*yV2.^6)./45045).*xV1.^8 + ((xV2.^2.*yV1.^4.*yV2.^2)./1001 - (xV2.^2.*yV1.^5.*yV2)./1365 - (4.*xV2.^2.*yV1.^6)./1365 + (4.*xV2.^2.*yV1.^3.*yV2.^3)./3003 + (8.*xV2.^2.*yV1.^2.*yV2.^4)./9009 + (xV2.^2.*yV1.*yV2.^5)./2730 + (xV2.^2.*yV2.^6)./12870).*xV1.^7 + ((4.*xV2.^3.*yV1.^3.*yV2.^3)./3861 - (xV2.^3.*yV1.^5.*yV2)./715 - (xV2.^3.*yV1.^6)./585 + (xV2.^3.*yV1.^2.*yV2.^4)./858 + (xV2.^3.*yV1.*yV2.^5)./1430 + (4.*xV2.^3.*yV2.^6)./19305).*xV1.^6 + ((xV2.^4.*yV1.^3.*yV2.^3)./2574 - (xV2.^4.*yV1.^5.*yV2)./715 - (xV2.^4.*yV1.^4.*yV2.^2)./1287 - (2.*xV2.^4.*yV1.^6)./2145 + (xV2.^4.*yV1.^2.*yV2.^4)./858 + (7.*xV2.^4.*yV1.*yV2.^5)./6435 + (xV2.^4.*yV2.^6)./2145).*xV1.^5 + ((xV2.^5.*yV1.^2.*yV2.^4)./1287 - (7.*xV2.^5.*yV1.^5.*yV2)./6435 - (xV2.^5.*yV1.^4.*yV2.^2)./858 - (xV2.^5.*yV1.^3.*yV2.^3)./2574 - (xV2.^5.*yV1.^6)./2145 + (xV2.^5.*yV1.*yV2.^5)./715 + (2.*xV2.^5.*yV2.^6)./2145).*xV1.^4 + ((xV2.^6.*yV1.*yV2.^5)./715 - (xV2.^6.*yV1.^5.*yV2)./1430 - (xV2.^6.*yV1.^4.*yV2.^2)./858 - (4.*xV2.^6.*yV1.^3.*yV2.^3)./3861 - (4.*xV2.^6.*yV1.^6)./19305 + (xV2.^6.*yV2.^6)./585).*xV1.^3 + ((xV2.^7.*yV1.*yV2.^5)./1365 - (xV2.^7.*yV1.^5.*yV2)./2730 - (8.*xV2.^7.*yV1.^4.*yV2.^2)./9009 - (4.*xV2.^7.*yV1.^3.*yV2.^3)./3003 - (xV2.^7.*yV1.^2.*yV2.^4)./1001 - (xV2.^7.*yV1.^6)./12870 + (4.*xV2.^7.*yV2.^6)./1365).*xV1.^2 + ((xV2.^8.*yV2.^6)./210 - (xV2.^8.*yV1.^5.*yV2)./6930 - (xV2.^8.*yV1.^4.*yV2.^2)./2002 - (xV2.^8.*yV1.^3.*yV2.^3)./858 - (xV2.^8.*yV1.^2.*yV2.^4)./546 - (xV2.^8.*yV1.*yV2.^5)./910 - (xV2.^8.*yV1.^6)./45045).*xV1 - ((xV2.^9.*yV1.^6)./270270 + (xV2.^9.*yV1.^5.*yV2)./30030 + (xV2.^9.*yV1.^4.*yV2.^2)./6006 + (xV2.^9.*yV1.^3.*yV2.^3)./1638 + (xV2.^9.*yV1.^2.*yV2.^4)./546 + (xV2.^9.*yV1.*yV2.^5)./210 + (xV2.^9.*yV2.^6)./540);
        I_poly(98,1)=I_poly(98,1)+(yV1.^7./1680 + (yV1.^6.*yV2)./210 + (yV1.^5.*yV2.^2)./455 + (yV1.^4.*yV2.^3)./1092 + (yV1.^3.*yV2.^4)./3003 + (yV1.^2.*yV2.^5)./10010 + (yV1.*yV2.^6)./45045 + yV2.^7./360360).*xV1.^8 + ((xV2.*yV1.^6.*yV2)./2730 - (xV2.*yV1.^7)./210 + (3.*xV2.*yV1.^5.*yV2.^2)./1820 + (17.*xV2.*yV1.^4.*yV2.^3)./12012 + (5.*xV2.*yV1.^3.*yV2.^4)./6006 + (xV2.*yV1.^2.*yV2.^5)./2730 + (41.*xV2.*yV1.*yV2.^6)./360360 + (xV2.*yV2.^7)./51480).*xV1.^7 + ((xV2.^2.*yV1.^5.*yV2.^2)./2860 - (xV2.^2.*yV1.^6.*yV2)./780 - (xV2.^2.*yV1.^7)./390 + (xV2.^2.*yV1.^4.*yV2.^3)./858 + (xV2.^2.*yV1.^3.*yV2.^4)./858 + (xV2.^2.*yV1.^2.*yV2.^5)./1320 + (17.*xV2.^2.*yV1.*yV2.^6)./51480 + (xV2.^2.*yV2.^7)./12870).*xV1.^6 + ((xV2.^3.*yV1.^4.*yV2.^3)./2574 - (xV2.^3.*yV1.^6.*yV2)./660 - (xV2.^3.*yV1.^5.*yV2.^2)./1430 - (xV2.^3.*yV1.^7)./780 + (xV2.^3.*yV1.^3.*yV2.^4)./936 + (19.*xV2.^3.*yV1.^2.*yV2.^5)./17160 + (xV2.^3.*yV1.*yV2.^6)./1430 + (xV2.^3.*yV2.^7)./4290).*xV1.^5 + ((5.*xV2.^4.*yV1.^3.*yV2.^4)./10296 - (xV2.^4.*yV1.^6.*yV2)./858 - (xV2.^4.*yV1.^5.*yV2.^2)./858 - (5.*xV2.^4.*yV1.^4.*yV2.^3)./10296 - (xV2.^4.*yV1.^7)./1716 + (xV2.^4.*yV1.^2.*yV2.^5)./858 + (xV2.^4.*yV1.*yV2.^6)./858 + (xV2.^4.*yV2.^7)./1716).*xV1.^4 + ((xV2.^5.*yV1.^2.*yV2.^5)./1430 - (xV2.^5.*yV1.^6.*yV2)./1430 - (19.*xV2.^5.*yV1.^5.*yV2.^2)./17160 - (xV2.^5.*yV1.^4.*yV2.^3)./936 - (xV2.^5.*yV1.^3.*yV2.^4)./2574 - (xV2.^5.*yV1.^7)./4290 + (xV2.^5.*yV1.*yV2.^6)./660 + (xV2.^5.*yV2.^7)./780).*xV1.^3 + ((xV2.^6.*yV1.*yV2.^6)./780 - (17.*xV2.^6.*yV1.^6.*yV2)./51480 - (xV2.^6.*yV1.^5.*yV2.^2)./1320 - (xV2.^6.*yV1.^4.*yV2.^3)./858 - (xV2.^6.*yV1.^3.*yV2.^4)./858 - (xV2.^6.*yV1.^2.*yV2.^5)./2860 - (xV2.^6.*yV1.^7)./12870 + (xV2.^6.*yV2.^7)./390).*xV1.^2 + ((xV2.^7.*yV2.^7)./210 - (41.*xV2.^7.*yV1.^6.*yV2)./360360 - (xV2.^7.*yV1.^5.*yV2.^2)./2730 - (5.*xV2.^7.*yV1.^4.*yV2.^3)./6006 - (17.*xV2.^7.*yV1.^3.*yV2.^4)./12012 - (3.*xV2.^7.*yV1.^2.*yV2.^5)./1820 - (xV2.^7.*yV1.*yV2.^6)./2730 - (xV2.^7.*yV1.^7)./51480).*xV1 - ((xV2.^8.*yV1.^7)./360360 + (xV2.^8.*yV1.^6.*yV2)./45045 + (xV2.^8.*yV1.^5.*yV2.^2)./10010 + (xV2.^8.*yV1.^4.*yV2.^3)./3003 + (xV2.^8.*yV1.^3.*yV2.^4)./1092 + (xV2.^8.*yV1.^2.*yV2.^5)./455 + (xV2.^8.*yV1.*yV2.^6)./210 + (xV2.^8.*yV2.^7)./1680);
        I_poly(99,1)=I_poly(99,1)+((yV1.^7.*yV2)./210 - yV1.^8./1680 + (yV1.^6.*yV2.^2)./390 + (yV1.^5.*yV2.^3)./780 + (yV1.^4.*yV2.^4)./1716 + (yV1.^3.*yV2.^5)./4290 + (yV1.^2.*yV2.^6)./12870 + (yV1.*yV2.^7)./51480 + yV2.^8./360360).*xV1.^7 + ((xV2.*yV1.^6.*yV2.^2)./780 - (xV2.*yV1.^7.*yV2)./2730 - (xV2.*yV1.^8)./210 + (xV2.*yV1.^5.*yV2.^3)./660 + (xV2.*yV1.^4.*yV2.^4)./858 + (xV2.*yV1.^3.*yV2.^5)./1430 + (17.*xV2.*yV1.^2.*yV2.^6)./51480 + (41.*xV2.*yV1.*yV2.^7)./360360 + (xV2.*yV2.^8)./45045).*xV1.^6 + ((xV2.^2.*yV1.^5.*yV2.^3)./1430 - (3.*xV2.^2.*yV1.^7.*yV2)./1820 - (xV2.^2.*yV1.^6.*yV2.^2)./2860 - (xV2.^2.*yV1.^8)./455 + (xV2.^2.*yV1.^4.*yV2.^4)./858 + (19.*xV2.^2.*yV1.^3.*yV2.^5)./17160 + (xV2.^2.*yV1.^2.*yV2.^6)./1320 + (xV2.^2.*yV1.*yV2.^7)./2730 + (xV2.^2.*yV2.^8)./10010).*xV1.^5 + ((5.*xV2.^3.*yV1.^4.*yV2.^4)./10296 - (17.*xV2.^3.*yV1.^7.*yV2)./12012 - (xV2.^3.*yV1.^6.*yV2.^2)./858 - (xV2.^3.*yV1.^5.*yV2.^3)./2574 - (xV2.^3.*yV1.^8)./1092 + (xV2.^3.*yV1.^3.*yV2.^5)./936 + (xV2.^3.*yV1.^2.*yV2.^6)./858 + (5.*xV2.^3.*yV1.*yV2.^7)./6006 + (xV2.^3.*yV2.^8)./3003).*xV1.^4 + ((xV2.^4.*yV1.^3.*yV2.^5)./2574 - (5.*xV2.^4.*yV1.^7.*yV2)./6006 - (xV2.^4.*yV1.^6.*yV2.^2)./858 - (xV2.^4.*yV1.^5.*yV2.^3)./936 - (5.*xV2.^4.*yV1.^4.*yV2.^4)./10296 - (xV2.^4.*yV1.^8)./3003 + (xV2.^4.*yV1.^2.*yV2.^6)./858 + (17.*xV2.^4.*yV1.*yV2.^7)./12012 + (xV2.^4.*yV2.^8)./1092).*xV1.^3 + ((xV2.^5.*yV1.^2.*yV2.^6)./2860 - (xV2.^5.*yV1.^7.*yV2)./2730 - (xV2.^5.*yV1.^6.*yV2.^2)./1320 - (19.*xV2.^5.*yV1.^5.*yV2.^3)./17160 - (xV2.^5.*yV1.^4.*yV2.^4)./858 - (xV2.^5.*yV1.^3.*yV2.^5)./1430 - (xV2.^5.*yV1.^8)./10010 + (3.*xV2.^5.*yV1.*yV2.^7)./1820 + (xV2.^5.*yV2.^8)./455).*xV1.^2 + ((xV2.^6.*yV1.*yV2.^7)./2730 - (41.*xV2.^6.*yV1.^7.*yV2)./360360 - (17.*xV2.^6.*yV1.^6.*yV2.^2)./51480 - (xV2.^6.*yV1.^5.*yV2.^3)./1430 - (xV2.^6.*yV1.^4.*yV2.^4)./858 - (xV2.^6.*yV1.^3.*yV2.^5)./660 - (xV2.^6.*yV1.^2.*yV2.^6)./780 - (xV2.^6.*yV1.^8)./45045 + (xV2.^6.*yV2.^8)./210).*xV1 + ((xV2.^7.*yV2.^8)./1680 - (xV2.^7.*yV1.^7.*yV2)./51480 - (xV2.^7.*yV1.^6.*yV2.^2)./12870 - (xV2.^7.*yV1.^5.*yV2.^3)./4290 - (xV2.^7.*yV1.^4.*yV2.^4)./1716 - (xV2.^7.*yV1.^3.*yV2.^5)./780 - (xV2.^7.*yV1.^2.*yV2.^6)./390 - (xV2.^7.*yV1.*yV2.^7)./210 - (xV2.^7.*yV1.^8)./360360);
        I_poly(100,1)=I_poly(100,1)+((yV1.^8.*yV2)./210 - yV1.^9./540 + (4.*yV1.^7.*yV2.^2)./1365 + (yV1.^6.*yV2.^3)./585 + (2.*yV1.^5.*yV2.^4)./2145 + (yV1.^4.*yV2.^5)./2145 + (4.*yV1.^3.*yV2.^6)./19305 + (yV1.^2.*yV2.^7)./12870 + (yV1.*yV2.^8)./45045 + yV2.^9./270270).*xV1.^6 + ((xV2.*yV1.^7.*yV2.^2)./1365 - (xV2.*yV1.^8.*yV2)./910 - (xV2.*yV1.^9)./210 + (xV2.*yV1.^6.*yV2.^3)./715 + (xV2.*yV1.^5.*yV2.^4)./715 + (7.*xV2.*yV1.^4.*yV2.^5)./6435 + (xV2.*yV1.^3.*yV2.^6)./1430 + (xV2.*yV1.^2.*yV2.^7)./2730 + (xV2.*yV1.*yV2.^8)./6930 + (xV2.*yV2.^9)./30030).*xV1.^5 + ((xV2.^2.*yV1.^5.*yV2.^4)./1287 - (xV2.^2.*yV1.^8.*yV2)./546 - (xV2.^2.*yV1.^7.*yV2.^2)./1001 - (xV2.^2.*yV1.^9)./546 + (xV2.^2.*yV1.^4.*yV2.^5)./858 + (xV2.^2.*yV1.^3.*yV2.^6)./858 + (8.*xV2.^2.*yV1.^2.*yV2.^7)./9009 + (xV2.^2.*yV1.*yV2.^8)./2002 + (xV2.^2.*yV2.^9)./6006).*xV1.^4 + ((xV2.^3.*yV1.^4.*yV2.^5)./2574 - (xV2.^3.*yV1.^8.*yV2)./858 - (4.*xV2.^3.*yV1.^7.*yV2.^2)./3003 - (4.*xV2.^3.*yV1.^6.*yV2.^3)./3861 - (xV2.^3.*yV1.^5.*yV2.^4)./2574 - (xV2.^3.*yV1.^9)./1638 + (4.*xV2.^3.*yV1.^3.*yV2.^6)./3861 + (4.*xV2.^3.*yV1.^2.*yV2.^7)./3003 + (xV2.^3.*yV1.*yV2.^8)./858 + (xV2.^3.*yV2.^9)./1638).*xV1.^3 + ((xV2.^4.*yV1.^2.*yV2.^7)./1001 - (xV2.^4.*yV1.^8.*yV2)./2002 - (8.*xV2.^4.*yV1.^7.*yV2.^2)./9009 - (xV2.^4.*yV1.^6.*yV2.^3)./858 - (xV2.^4.*yV1.^5.*yV2.^4)./858 - (xV2.^4.*yV1.^4.*yV2.^5)./1287 - (xV2.^4.*yV1.^9)./6006 + (xV2.^4.*yV1.*yV2.^8)./546 + (xV2.^4.*yV2.^9)./546).*xV1.^2 + ((xV2.^5.*yV1.*yV2.^8)./910 - (xV2.^5.*yV1.^8.*yV2)./6930 - (xV2.^5.*yV1.^7.*yV2.^2)./2730 - (xV2.^5.*yV1.^6.*yV2.^3)./1430 - (7.*xV2.^5.*yV1.^5.*yV2.^4)./6435 - (xV2.^5.*yV1.^4.*yV2.^5)./715 - (xV2.^5.*yV1.^3.*yV2.^6)./715 - (xV2.^5.*yV1.^2.*yV2.^7)./1365 - (xV2.^5.*yV1.^9)./30030 + (xV2.^5.*yV2.^9)./210).*xV1 + ((xV2.^6.*yV2.^9)./540 - (xV2.^6.*yV1.^8.*yV2)./45045 - (xV2.^6.*yV1.^7.*yV2.^2)./12870 - (4.*xV2.^6.*yV1.^6.*yV2.^3)./19305 - (xV2.^6.*yV1.^5.*yV2.^4)./2145 - (2.*xV2.^6.*yV1.^4.*yV2.^5)./2145 - (xV2.^6.*yV1.^3.*yV2.^6)./585 - (4.*xV2.^6.*yV1.^2.*yV2.^7)./1365 - (xV2.^6.*yV1.*yV2.^8)./210 - (xV2.^6.*yV1.^9)./270270);
        I_poly(101,1)=I_poly(101,1)+((yV1.^9.*yV2)./210 - yV1.^10./300 + (3.*yV1.^8.*yV2.^2)./910 + (yV1.^7.*yV2.^3)./455 + (yV1.^6.*yV2.^4)./715 + (3.*yV1.^5.*yV2.^5)./3575 + (yV1.^4.*yV2.^6)./2145 + (yV1.^3.*yV2.^7)./4290 + (yV1.^2.*yV2.^8)./10010 + (yV1.*yV2.^9)./30030 + yV2.^10./150150).*xV1.^5 + ((xV2.*yV1.^7.*yV2.^3)./1001 - (xV2.*yV1.^9.*yV2)./546 - (xV2.*yV1.^10)./210 + (xV2.*yV1.^6.*yV2.^4)./715 + (xV2.*yV1.^5.*yV2.^5)./715 + (xV2.*yV1.^4.*yV2.^6)./858 + (5.*xV2.*yV1.^3.*yV2.^7)./6006 + (xV2.*yV1.^2.*yV2.^8)./2002 + (xV2.*yV1.*yV2.^9)./4290 + (xV2.*yV2.^10)./15015).*xV1.^4 + ((xV2.^2.*yV1.^5.*yV2.^5)./1430 - (xV2.^2.*yV1.^9.*yV2)./546 - (3.*xV2.^2.*yV1.^8.*yV2.^2)./2002 - (4.*xV2.^2.*yV1.^7.*yV2.^3)./5005 - (2.*xV2.^2.*yV1.^10)./1365 + (xV2.^2.*yV1.^4.*yV2.^6)./858 + (4.*xV2.^2.*yV1.^3.*yV2.^7)./3003 + (6.*xV2.^2.*yV1.^2.*yV2.^8)./5005 + (5.*xV2.^2.*yV1.*yV2.^9)./6006 + (xV2.^2.*yV2.^10)./2730).*xV1.^3 + ((4.*xV2.^3.*yV1.^3.*yV2.^7)./5005 - (5.*xV2.^3.*yV1.^9.*yV2)./6006 - (6.*xV2.^3.*yV1.^8.*yV2.^2)./5005 - (4.*xV2.^3.*yV1.^7.*yV2.^3)./3003 - (xV2.^3.*yV1.^6.*yV2.^4)./858 - (xV2.^3.*yV1.^5.*yV2.^5)./1430 - (xV2.^3.*yV1.^10)./2730 + (3.*xV2.^3.*yV1.^2.*yV2.^8)./2002 + (xV2.^3.*yV1.*yV2.^9)./546 + (2.*xV2.^3.*yV2.^10)./1365).*xV1.^2 + ((xV2.^4.*yV1.*yV2.^9)./546 - (xV2.^4.*yV1.^9.*yV2)./4290 - (xV2.^4.*yV1.^8.*yV2.^2)./2002 - (5.*xV2.^4.*yV1.^7.*yV2.^3)./6006 - (xV2.^4.*yV1.^6.*yV2.^4)./858 - (xV2.^4.*yV1.^5.*yV2.^5)./715 - (xV2.^4.*yV1.^4.*yV2.^6)./715 - (xV2.^4.*yV1.^3.*yV2.^7)./1001 - (xV2.^4.*yV1.^10)./15015 + (xV2.^4.*yV2.^10)./210).*xV1 + ((xV2.^5.*yV2.^10)./300 - (xV2.^5.*yV1.^9.*yV2)./30030 - (xV2.^5.*yV1.^8.*yV2.^2)./10010 - (xV2.^5.*yV1.^7.*yV2.^3)./4290 - (xV2.^5.*yV1.^6.*yV2.^4)./2145 - (3.*xV2.^5.*yV1.^5.*yV2.^5)./3575 - (xV2.^5.*yV1.^4.*yV2.^6)./715 - (xV2.^5.*yV1.^3.*yV2.^7)./455 - (3.*xV2.^5.*yV1.^2.*yV2.^8)./910 - (xV2.^5.*yV1.*yV2.^9)./210 - (xV2.^5.*yV1.^10)./150150);
        I_poly(102,1)=I_poly(102,1)+((yV1.^10.*yV2)./210 - (7.*yV1.^11)./1320 + (yV1.^9.*yV2.^2)./273 + (yV1.^8.*yV2.^3)./364 + (2.*yV1.^7.*yV2.^4)./1001 + (yV1.^6.*yV2.^5)./715 + (2.*yV1.^5.*yV2.^6)./2145 + (yV1.^4.*yV2.^7)./1716 + (yV1.^3.*yV2.^8)./3003 + (yV1.^2.*yV2.^9)./6006 + (yV1.*yV2.^10)./15015 + yV2.^11./60060).*xV1.^4 + ((xV2.*yV1.^8.*yV2.^3)./4004 - (xV2.*yV1.^10.*yV2)./390 - (xV2.*yV1.^9.*yV2.^2)./1092 - (xV2.*yV1.^11)./210 + (xV2.*yV1.^7.*yV2.^4)./1001 + (xV2.*yV1.^6.*yV2.^5)./715 + (xV2.*yV1.^5.*yV2.^6)./660 + (17.*xV2.*yV1.^4.*yV2.^7)./12012 + (xV2.*yV1.^3.*yV2.^8)./858 + (5.*xV2.*yV1.^2.*yV2.^9)./6006 + (29.*xV2.*yV1.*yV2.^10)./60060 + (xV2.*yV2.^11)./5460).*xV1.^3 + ((xV2.^2.*yV1.^5.*yV2.^6)./2860 - (3.*xV2.^2.*yV1.^10.*yV2)./1820 - (xV2.^2.*yV1.^9.*yV2.^2)./572 - (3.*xV2.^2.*yV1.^8.*yV2.^3)./2002 - (xV2.^2.*yV1.^7.*yV2.^4)./1001 - (xV2.^2.*yV1.^6.*yV2.^5)./2860 - (xV2.^2.*yV1.^11)./910 + (xV2.^2.*yV1.^4.*yV2.^7)./1001 + (3.*xV2.^2.*yV1.^3.*yV2.^8)./2002 + (xV2.^2.*yV1.^2.*yV2.^9)./572 + (3.*xV2.^2.*yV1.*yV2.^10)./1820 + (xV2.^2.*yV2.^11)./910).*xV1.^2 + ((xV2.^3.*yV1.^2.*yV2.^9)./1092 - (29.*xV2.^3.*yV1.^10.*yV2)./60060 - (5.*xV2.^3.*yV1.^9.*yV2.^2)./6006 - (xV2.^3.*yV1.^8.*yV2.^3)./858 - (17.*xV2.^3.*yV1.^7.*yV2.^4)./12012 - (xV2.^3.*yV1.^6.*yV2.^5)./660 - (xV2.^3.*yV1.^5.*yV2.^6)./715 - (xV2.^3.*yV1.^4.*yV2.^7)./1001 - (xV2.^3.*yV1.^3.*yV2.^8)./4004 - (xV2.^3.*yV1.^11)./5460 + (xV2.^3.*yV1.*yV2.^10)./390 + (xV2.^3.*yV2.^11)./210).*xV1 + ((7.*xV2.^4.*yV2.^11)./1320 - (xV2.^4.*yV1.^10.*yV2)./15015 - (xV2.^4.*yV1.^9.*yV2.^2)./6006 - (xV2.^4.*yV1.^8.*yV2.^3)./3003 - (xV2.^4.*yV1.^7.*yV2.^4)./1716 - (2.*xV2.^4.*yV1.^6.*yV2.^5)./2145 - (xV2.^4.*yV1.^5.*yV2.^6)./715 - (2.*xV2.^4.*yV1.^4.*yV2.^7)./1001 - (xV2.^4.*yV1.^3.*yV2.^8)./364 - (xV2.^4.*yV1.^2.*yV2.^9)./273 - (xV2.^4.*yV1.*yV2.^10)./210 - (xV2.^4.*yV1.^11)./60060);
        I_poly(103,1)=I_poly(103,1)+((yV1.^11.*yV2)./210 - yV1.^12./120 + (11.*yV1.^10.*yV2.^2)./2730 + (11.*yV1.^9.*yV2.^3)./3276 + (yV1.^8.*yV2.^4)./364 + (yV1.^7.*yV2.^5)./455 + (yV1.^6.*yV2.^6)./585 + (yV1.^5.*yV2.^7)./780 + (yV1.^4.*yV2.^8)./1092 + (yV1.^3.*yV2.^9)./1638 + (yV1.^2.*yV2.^10)./2730 + (yV1.*yV2.^11)./5460 + yV2.^12./16380).*xV1.^3 + ((xV2.*yV1.^7.*yV2.^5)./1365 - (3.*xV2.*yV1.^11.*yV2)./910 - (11.*xV2.*yV1.^10.*yV2.^2)./5460 - (xV2.*yV1.^9.*yV2.^3)./1092 - (xV2.*yV1.^12)./210 + (xV2.*yV1.^6.*yV2.^6)./780 + (3.*xV2.*yV1.^5.*yV2.^7)./1820 + (xV2.*yV1.^4.*yV2.^8)./546 + (xV2.*yV1.^3.*yV2.^9)./546 + (3.*xV2.*yV1.^2.*yV2.^10)./1820 + (xV2.*yV1.*yV2.^11)./780 + (xV2.*yV2.^12)./1365).*xV1.^2 + ((xV2.^2.*yV1.^3.*yV2.^9)./1092 - (xV2.^2.*yV1.^11.*yV2)./780 - (3.*xV2.^2.*yV1.^10.*yV2.^2)./1820 - (xV2.^2.*yV1.^9.*yV2.^3)./546 - (xV2.^2.*yV1.^8.*yV2.^4)./546 - (3.*xV2.^2.*yV1.^7.*yV2.^5)./1820 - (xV2.^2.*yV1.^6.*yV2.^6)./780 - (xV2.^2.*yV1.^5.*yV2.^7)./1365 - (xV2.^2.*yV1.^12)./1365 + (11.*xV2.^2.*yV1.^2.*yV2.^10)./5460 + (3.*xV2.^2.*yV1.*yV2.^11)./910 + (xV2.^2.*yV2.^12)./210).*xV1 + ((xV2.^3.*yV2.^12)./120 - (xV2.^3.*yV1.^11.*yV2)./5460 - (xV2.^3.*yV1.^10.*yV2.^2)./2730 - (xV2.^3.*yV1.^9.*yV2.^3)./1638 - (xV2.^3.*yV1.^8.*yV2.^4)./1092 - (xV2.^3.*yV1.^7.*yV2.^5)./780 - (xV2.^3.*yV1.^6.*yV2.^6)./585 - (xV2.^3.*yV1.^5.*yV2.^7)./455 - (xV2.^3.*yV1.^4.*yV2.^8)./364 - (11.*xV2.^3.*yV1.^3.*yV2.^9)./3276 - (11.*xV2.^3.*yV1.^2.*yV2.^10)./2730 - (xV2.^3.*yV1.*yV2.^11)./210 - (xV2.^3.*yV1.^12)./16380);
        I_poly(104,1)=I_poly(104,1)+-(77.*xV1.^2.*yV1.^13 - 2.*xV1.^2.*yV2.^13 + 2.*xV2.^2.*yV1.^13 - 77.*xV2.^2.*yV2.^13 + 26.*xV1.*xV2.*yV1.^13 - 26.*xV1.*xV2.*yV2.^13 - 4.*xV1.^2.*yV1.*yV2.^12 - 26.*xV1.^2.*yV1.^12.*yV2 + 26.*xV2.^2.*yV1.*yV2.^12 + 4.*xV2.^2.*yV1.^12.*yV2 - 6.*xV1.^2.*yV1.^2.*yV2.^11 - 8.*xV1.^2.*yV1.^3.*yV2.^10 - 10.*xV1.^2.*yV1.^4.*yV2.^9 - 12.*xV1.^2.*yV1.^5.*yV2.^8 - 14.*xV1.^2.*yV1.^6.*yV2.^7 - 16.*xV1.^2.*yV1.^7.*yV2.^6 - 18.*xV1.^2.*yV1.^8.*yV2.^5 - 20.*xV1.^2.*yV1.^9.*yV2.^4 - 22.*xV1.^2.*yV1.^10.*yV2.^3 - 24.*xV1.^2.*yV1.^11.*yV2.^2 + 24.*xV2.^2.*yV1.^2.*yV2.^11 + 22.*xV2.^2.*yV1.^3.*yV2.^10 + 20.*xV2.^2.*yV1.^4.*yV2.^9 + 18.*xV2.^2.*yV1.^5.*yV2.^8 + 16.*xV2.^2.*yV1.^6.*yV2.^7 + 14.*xV2.^2.*yV1.^7.*yV2.^6 + 12.*xV2.^2.*yV1.^8.*yV2.^5 + 10.*xV2.^2.*yV1.^9.*yV2.^4 + 8.*xV2.^2.*yV1.^10.*yV2.^3 + 6.*xV2.^2.*yV1.^11.*yV2.^2 - 18.*xV1.*xV2.*yV1.^2.*yV2.^11 - 14.*xV1.*xV2.*yV1.^3.*yV2.^10 - 10.*xV1.*xV2.*yV1.^4.*yV2.^9 - 6.*xV1.*xV2.*yV1.^5.*yV2.^8 - 2.*xV1.*xV2.*yV1.^6.*yV2.^7 + 2.*xV1.*xV2.*yV1.^7.*yV2.^6 + 6.*xV1.*xV2.*yV1.^8.*yV2.^5 + 10.*xV1.*xV2.*yV1.^9.*yV2.^4 + 14.*xV1.*xV2.*yV1.^10.*yV2.^3 + 18.*xV1.*xV2.*yV1.^11.*yV2.^2 - 22.*xV1.*xV2.*yV1.*yV2.^12 + 22.*xV1.*xV2.*yV1.^12.*yV2)./5460;
        I_poly(105,1)=I_poly(105,1)+-(13.*xV1.*yV1.^14 - 2.*xV1.*yV2.^14 + 2.*xV2.*yV1.^14 - 13.*xV2.*yV2.^14 - 2.*xV1.*yV1.*yV2.^13 - 2.*xV1.*yV1.^13.*yV2 + 2.*xV2.*yV1.*yV2.^13 + 2.*xV2.*yV1.^13.*yV2 - 2.*xV1.*yV1.^2.*yV2.^12 - 2.*xV1.*yV1.^3.*yV2.^11 - 2.*xV1.*yV1.^4.*yV2.^10 - 2.*xV1.*yV1.^5.*yV2.^9 - 2.*xV1.*yV1.^6.*yV2.^8 - 2.*xV1.*yV1.^7.*yV2.^7 - 2.*xV1.*yV1.^8.*yV2.^6 - 2.*xV1.*yV1.^9.*yV2.^5 - 2.*xV1.*yV1.^10.*yV2.^4 - 2.*xV1.*yV1.^11.*yV2.^3 - 2.*xV1.*yV1.^12.*yV2.^2 + 2.*xV2.*yV1.^2.*yV2.^12 + 2.*xV2.*yV1.^3.*yV2.^11 + 2.*xV2.*yV1.^4.*yV2.^10 + 2.*xV2.*yV1.^5.*yV2.^9 + 2.*xV2.*yV1.^6.*yV2.^8 + 2.*xV2.*yV1.^7.*yV2.^7 + 2.*xV2.*yV1.^8.*yV2.^6 + 2.*xV2.*yV1.^9.*yV2.^5 + 2.*xV2.*yV1.^10.*yV2.^4 + 2.*xV2.*yV1.^11.*yV2.^3 + 2.*xV2.*yV1.^12.*yV2.^2)./420;
    end
end
end

function sp=Boundary_Curve_Explicit_ODE_3D(~,p,p0,pf,gradh,gradb)
gh=gradh(p.');
gb=gradb(p.');
Crossx=gh(:,2).*gb(:,3)-gh(:,3).*gb(:,2);
Crossy=gh(:,3).*gb(:,1)-gh(:,1).*gb(:,3);
Crossz=gh(:,1).*gb(:,2)-gh(:,2).*gb(:,1);
NormCross=sqrt(Crossx.^2+Crossy.^2+Crossz.^2);
sp=[Crossx./NormCross;
    Crossy./NormCross;
    Crossz./NormCross];
sp=sign((pf-p0)*sp)*sp;
end

function sp=Boundary_Curve_Explicit_ODE_2D(s,p,p0,pf,gradb)
gb=gradb(p.');
NormCross=sqrt(gb(:,1).^2+gb(:,2).^2);
sp=[-gb(:,2)./NormCross;
    gb(:,1)./NormCross];
sp=sign((pf-p0)*sp)*sp;
end

function [x,w]=Legendre_Gauss_Lobatto_Quadrature_Nodes(N,a,b)

N1=N+1;

x=cos(pi*(0:N)/N)';
P=zeros(N1,N1);
xold=2;
while max(abs(x-xold))>eps
    xold=x;
    P(:,1)=1;P(:,2)=x;
    for k=2:N
        P(:,k+1)=((2*k-1)*x.*P(:,k)-(k-1)*P(:,k-1))/k;
    end
    x=xold-(x.*P(:,N1)-P(:,N))./(N1*P(:,N1));
end
w=2./(N*N1*P(:,N1).^2);
x=(x.'+1)*(b-a)/2+a;
w=(b-a)./2*w;
end


