function [Approximate_Surface_Integral_f1_Square,Approximate_Surface_Integral_f2_Square,Approximate_Surface_Integral_f3_Square,card_RBF,CPU_RBF,Quadrature_Weights_Square]=Bounded_Smooth_Surface_Quadrature_RBF_Test(NN)
%==========================================================================
%
% This function provides example calls to the function
% Bounded_Smooth_Surface_Quadrature_RBF.m.  The user can change the parameters Poly_Order and
% Number_of_Nearest_Neighbors within the Bounded_Smooth_Surface_Quadrature_RBF.m
% function.
%
% There are three possible tests indicated by the input Test_Index:
%
%       If Test_Index==1, then the test is run for three test integrands
%       and two boundaries (a unit square and circle) in the (x,y)-plane
%       (normal to [0 0 1]) that illustrate how use
%       Bounded_Smooth_Surface_Quadrature_RBF.m when the input
%       Quadrature_Nodes is already as two dimensional coordinates in the
%       plane.  That is, Quadrature_Nodes is specified as a
%       Number_of_Quadrature_Nodes X 2 vector.  The two boundaries display how
%       to use a planar and nonplanar boundary, respectively.
%
%       If Test_Index==2, then the test is run for three test integrands
%       and two boundaries (a unit square and circle) in the plane
%       normal to a random vector that illustrate how use
%       Bounded_Smooth_Surface_Quadrature_RBF.m when the boundary surface
%       is a plane, and with a planar (square) and nonplanar (circle)
%       boundary.
%
%       If Test_Index==3, then the test is run for five test integrands and
%       two boundaries.  The surface in this case is the sphere of radius
%       1/(2*sqrt(pi)), with two boundaries.  The first is a planar
%       boundary to create the hemisphere and illustrate how to use the
%       function in the presence of a nonplanar surface and planar
%       boundary.  The second is a nonplanar boundary (that creates a
%       scalloped sphere) that illustrates how to use the function in the
%       presence of a nonplanar surface and boundary.
%
% In all cases this function prints the error in the approximations
% to the command windown upon completion.
%
% The function Bounded_Smooth_Surface_Quadrature_RBF.m is the default implemenation of
% the method described in:
%
% J. A. Reeger and B. Fornberg "Numerical quadrature over
% smooth surfaces with boundaries".
%
%==========================================================================
warning off all

tic;

%==========================================================================
%
% Create a set of quadrature nodes over the unit square centered at the
% origin
%
%==========================================================================
Polygon_Vertices=[-1/2,1/2;-1/2,-1/2;1/2,-1/2;1/2,1/2;-1/2,1/2];
Points_on_Boundary_Curve=zeros(0,2);
h0=0.022;

for Polygon_Vertex_Index=1:(size(Polygon_Vertices,1)-1)
    Number_of_Points_on_Segment=ceil(sqrt((Polygon_Vertices(Polygon_Vertex_Index+1,:)-Polygon_Vertices(Polygon_Vertex_Index,:)).^2*ones(2,1))/(h0));
    Segment_Parameter=linspace(0,1,Number_of_Points_on_Segment).';
    Points_on_Boundary_Segment=(1-Segment_Parameter)*Polygon_Vertices(Polygon_Vertex_Index,:)+Segment_Parameter*Polygon_Vertices(Polygon_Vertex_Index+1,:);
    if Polygon_Vertex_Index==1
        Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment];
    else
        Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment(2:end,:)];
    end
end
Points_on_Boundary_Curve=unique(Points_on_Boundary_Curve,'rows');
p=haltonset(2);
Quadrature_Nodes_Square=[p(1:(NN-size(Points_on_Boundary_Curve,1)+1),1)-1/2,p(1:(NN-size(Points_on_Boundary_Curve,1)+1),2)-1/2];
Quadrature_Nodes_Square=[Quadrature_Nodes_Square;Points_on_Boundary_Curve];
Quadrature_Nodes_Square=unique(Quadrature_Nodes_Square,'rows');
Triangles_Square=delaunay(Quadrature_Nodes_Square);

%==========================================================================

%==========================================================================
%
% Define the level surface h(x,y,z)=0 and its gradient so that the normal
% to the surface can be computed exactly.  They have the following specifications
%
%   h - For the surface S defined implicitly by h(x,y,z)=0, row i
%   in the output of h should contain
%   h(Quadrature_Nodes(i,1:3))
%   h should take in Quadrature_Nodes as an
%   (Number_of_Quadrature_Nodes X 3) Array
%
%   gradh - The gradient of the function h.  Row i in the output
%   of gradh should contain
%   [dh/dx(Quadrature_Nodes(i,1:3)),dh/dy(Quadrature_Nodes(i,1:3)),dh/dz(Quadrature_Nodes(i,1:3)]
%   gradh should take in Quadrature_Nodes as an
%   (Number_of_Quadrature_Nodes X 3) Array
%
%==========================================================================
Plane_Normal=[0 0 1];
h=@(p) p(:,3);
gradh=@(p) [zeros(size(p,1),1),zeros(size(p,1),1),ones(size(p,1),1)];
hessianh=@(p) zeros(size(p,1),9);
%==========================================================================

%==========================================================================
%
% Define the level surface b(x,y,z)>=0 and its gradient so that the normal
% to the surface can be computed exactly.  They have the following specifications
%
%   b - For the boundary of S defined implicitly by b(x,y,z)=0,
%   row i in the output of b should contain
%         b(Quadrature_Nodes(i,1:3))
%   b should take in Quadrature_Nodes as an
%   (Number_of_Quadrature_Nodes X 3) Array
%
%   gradb - The gradient of the function b.  Row i in the output
%   of gradb should contain
%   [db/dx(Quadrature_Nodes(i,1:3)),db/dy(Quadrature_Nodes(i,1:3)),db/dz(Quadrature_Nodes(i,1:3)]
%   gradb should take in Quadrature_Nodes as an
%   (Number_of_Quadrature_Nodes X 3) Array
%
%==========================================================================
b_Square=@(p) max(abs(p),[],2)-1/2;
gradb_Square=[];
hessianb_Square=[];

%==========================================================================

%==========================================================================
%
% Generate the quadrature weights for the set of quadrature nodes generated
% above.
%
%==========================================================================
Boundary_Edge_Nodes=[];
options.Planar_Surface_Flag=1;
options.Planar_Surface_Normal=Plane_Normal;
options.Planar_Boundary_Flag=1;
% Use the approximate surface normal
Quadrature_Weights_Square=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Square,Triangles_Square,Boundary_Edge_Nodes,h,gradh,b_Square,gradb_Square,options);
CPU_RBF=toc;
options.Planar_Surface_Flag=1;
options.Planar_Surface_Normal=Plane_Normal;
options.Planar_Boundary_Flag=0;

%==========================================================================

card_RBF=length(Quadrature_Weights_Square);


%==========================================================================
%
% Compute the values of various test integrands at the quadrature nodes
% generated above.
%
%==========================================================================
F1_Square=f1_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));
F2_Square=f2_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));
F3_Square=f3_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));


%==========================================================================
%
% Compute the approximate values of the surface integrals of the test
% integrands using the quadrature weights generated above for comparison.
%
%==========================================================================
Approximate_Surface_Integral_f1_Square=F1_Square.'*Quadrature_Weights_Square;
Approximate_Surface_Integral_f2_Square=F2_Square.'*Quadrature_Weights_Square;
Approximate_Surface_Integral_f3_Square=F3_Square.'*Quadrature_Weights_Square;



function f=f1_Plane(x,y)
f=exp(-x.^2-y.^2);


function f=f2_Plane(x,y)
k=25;
f=1./(1+k*(x.^2+y.^2));


function f=f3_Plane(x,y)
k=25;
f=1./(1+k*(abs(x)+abs(y)));


