function [Approximate_Surface_Integral_f1_Square,Approximate_Surface_Integral_f2_Square,Approximate_Surface_Integral_f3_Square,card_RBF,CPU_RBF]=Bounded_Smooth_Surface_Quadrature_RBF_Test(NN,Test_Index)
%==========================================================================
%
% This function provides example calls to the function
% Bounded_Smooth_Surface_Quadrature_RBF.m.  The user can change the parameters Poly_Order and
% Number_of_Nearest_Neighbors within the Bounded_Smooth_Surface_Quadrature_RBF.m
% function.
%
% There are three possible tests indicated by the input Test_Index:
%
%       If Test_Index==1, then the test is run for three test integrands
%       and two boundaries (a unit square and circle) in the (x,y)-plane
%       (normal to [0 0 1]) that illustrate how use
%       Bounded_Smooth_Surface_Quadrature_RBF.m when the input
%       Quadrature_Nodes is already as two dimensional coordinates in the
%       plane.  That is, Quadrature_Nodes is specified as a
%       Number_of_Quadrature_Nodes X 2 vector.  The two boundaries display how
%       to use a planar and nonplanar boundary, respectively.
%
%       If Test_Index==2, then the test is run for three test integrands
%       and two boundaries (a unit square and circle) in the plane
%       normal to a random vector that illustrate how use
%       Bounded_Smooth_Surface_Quadrature_RBF.m when the boundary surface
%       is a plane, and with a planar (square) and nonplanar (circle)
%       boundary.
%
%       If Test_Index==3, then the test is run for five test integrands and
%       two boundaries.  The surface in this case is the sphere of radius
%       1/(2*sqrt(pi)), with two boundaries.  The first is a planar
%       boundary to create the hemisphere and illustrate how to use the
%       function in the presence of a nonplanar surface and planar
%       boundary.  The second is a nonplanar boundary (that creates a
%       scalloped sphere) that illustrates how to use the function in the
%       presence of a nonplanar surface and boundary.
%
% In all cases this function prints the error in the approximations
% to the command windown upon completion.
%
% The function Bounded_Smooth_Surface_Quadrature_RBF.m is the default implemenation of
% the method described in:
%
% J. A. Reeger and B. Fornberg "Numerical quadrature over
% smooth surfaces with boundaries".
%
%==========================================================================
warning off all

if nargin < 1
    NN=2500;
end

if nargin < 2
    Test_Index=1;
end

if Test_Index==1
    %==========================================================================
    %
    % Create a set of quadrature nodes over the unit square centered at the
    % origin
    %
    %==========================================================================
    Polygon_Vertices=[-1/2,1/2;-1/2,-1/2;1/2,-1/2;1/2,1/2;-1/2,1/2];
    Points_on_Boundary_Curve=zeros(0,2);
    h0=0.022;
    
    for Polygon_Vertex_Index=1:(size(Polygon_Vertices,1)-1)
        Number_of_Points_on_Segment=ceil(sqrt((Polygon_Vertices(Polygon_Vertex_Index+1,:)-Polygon_Vertices(Polygon_Vertex_Index,:)).^2*ones(2,1))/(h0));
        Segment_Parameter=linspace(0,1,Number_of_Points_on_Segment).';
        Points_on_Boundary_Segment=(1-Segment_Parameter)*Polygon_Vertices(Polygon_Vertex_Index,:)+Segment_Parameter*Polygon_Vertices(Polygon_Vertex_Index+1,:);
        if Polygon_Vertex_Index==1
            Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment];
        else
            Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment(2:end,:)];
        end
    end
    Points_on_Boundary_Curve=unique(Points_on_Boundary_Curve,'rows');
    p=haltonset(2);
    Quadrature_Nodes_Square=[p(1:(NN-size(Points_on_Boundary_Curve,1)+1),1)-1/2,p(1:(NN-size(Points_on_Boundary_Curve,1)+1),2)-1/2];
    Quadrature_Nodes_Square=[Quadrature_Nodes_Square;Points_on_Boundary_Curve];
    Quadrature_Nodes_Square=unique(Quadrature_Nodes_Square,'rows');
    Triangles_Square=delaunay(Quadrature_Nodes_Square);
    
    load('Planar_Nodes_Triangles_Cassini_lambda_0_h0_0_022.mat')
    Quadrature_Nodes_Circle=Quadrature_Nodes;
    Triangles_Circle=Triangles;
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface h(x,y,z)=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   h - For the surface S defined implicitly by h(x,y,z)=0, row i
    %   in the output of h should contain
    %   h(Quadrature_Nodes(i,1:3))
    %   h should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradh - The gradient of the function h.  Row i in the output
    %   of gradh should contain
    %   [dh/dx(Quadrature_Nodes(i,1:3)),dh/dy(Quadrature_Nodes(i,1:3)),dh/dz(Quadrature_Nodes(i,1:3)]
    %   gradh should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    Plane_Normal=[0 0 1];
    h=@(p) p(:,3);
    gradh=@(p) [zeros(size(p,1),1),zeros(size(p,1),1),ones(size(p,1),1)];
    hessianh=@(p) zeros(size(p,1),9);
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface b(x,y,z)>=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   b - For the boundary of S defined implicitly by b(x,y,z)=0,
    %   row i in the output of b should contain
    %         b(Quadrature_Nodes(i,1:3))
    %   b should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradb - The gradient of the function b.  Row i in the output
    %   of gradb should contain
    %   [db/dx(Quadrature_Nodes(i,1:3)),db/dy(Quadrature_Nodes(i,1:3)),db/dz(Quadrature_Nodes(i,1:3)]
    %   gradb should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    b_Square=@(p) max(abs(p),[],2)-1/2;
    gradb_Square=[];
    hessianb_Square=[];
    
    Sphere_Radius=1/sqrt(pi);
    b_Circle=@(p)p(:,1).^2+p(:,2).^2-Sphere_Radius.^2;
    gradb_Circle=@(p) [2.*p(:,1),2.*p(:,2)];
    hessianb_Circle=@(p) [2*ones(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),2*ones(size(p,1),1)];
    %==========================================================================
    
    %==========================================================================
    %
    % Generate the quadrature weights for the set of quadrature nodes generated
    % above.
    %
    %==========================================================================
    Boundary_Edge_Nodes=[];
    options.Planar_Surface_Flag=1;
    options.Planar_Surface_Normal=Plane_Normal;
    options.Planar_Boundary_Flag=1;
    % Use the approximate surface normal
    tic;
    Quadrature_Weights_Square=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Square,Triangles_Square,Boundary_Edge_Nodes,h,gradh,b_Square,gradb_Square,options);
    CPU_RBF=toc;
    options.Planar_Surface_Flag=1;
    options.Planar_Surface_Normal=Plane_Normal;
    options.Planar_Boundary_Flag=0;
    % Use the exact surface normal
    Quadrature_Weights_Circle=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Circle,Triangles_Circle,Boundary_Edge_Nodes,h,gradh,b_Circle,gradb_Circle,options);
    %==========================================================================
    
    card_RBF=length(Quadrature_Weights_Square);
    
    
    %==========================================================================
    %
    % Compute the values of various test integrands at the quadrature nodes
    % generated above.
    %
    %==========================================================================
    F1_Square=f1_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));
    F2_Square=f2_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));
    F3_Square=f3_Plane(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2));
    
    F1_Circle=f1_Plane(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2));
    F2_Circle=f2_Plane(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2));
    F3_Circle=f3_Plane(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2));
    %==========================================================================
    
    %==========================================================================
    %
    % Set the exact values of the surface integrals of the test integrands for
    % comparison.
    %
    %==========================================================================
    k=25;
    Exact_Surface_Integral_f1_Square=pi*erf(1/2)^2;
    Exact_Surface_Integral_f2_Square=quad2d(@(x,y) 1./(1+k*(x.^2+y.^2)),-1/2,1/2,-1/2,1/2,'AbsTol',eps,'RelTol',eps);
    Exact_Surface_Integral_f3_Square=(2.*(k.*log(4)+log(16)+2.*log(1+k)+2.*k.*log(1+k)-4.*log(2+k)-2.*k.*log(2+k)))./k.^2;
    
    Exact_Surface_Integral_f1_Circle=-pi*(exp(-Sphere_Radius^2)-1);
    Exact_Surface_Integral_f2_Circle=pi*log(1+k*Sphere_Radius^2)/k;
    Exact_Surface_Integral_f3_Circle=quad2d(@(x,y) 1./(1+k*(abs(x)+abs(y))),-Sphere_Radius,Sphere_Radius,@(x)-sqrt(Sphere_Radius.^2-x.^2),@(x) sqrt(Sphere_Radius.^2-x.^2),'AbsTol',eps,'RelTol',eps);
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the approximate values of the surface integrals of the test
    % integrands using the quadrature weights generated above for comparison.
    %
    %==========================================================================
    Approximate_Surface_Integral_f1_Square=F1_Square.'*Quadrature_Weights_Square;
    Approximate_Surface_Integral_f2_Square=F2_Square.'*Quadrature_Weights_Square;
    Approximate_Surface_Integral_f3_Square=F3_Square.'*Quadrature_Weights_Square;
    
    Approximate_Surface_Integral_f1_Circle=F1_Circle.'*Quadrature_Weights_Circle;
    Approximate_Surface_Integral_f2_Circle=F2_Circle.'*Quadrature_Weights_Circle;
    Approximate_Surface_Integral_f3_Circle=F3_Circle.'*Quadrature_Weights_Circle;
    %==========================================================================
    
%         fprintf('\n \t 1: %1.25e',Exact_Surface_Integral_f1_Square)
%     fprintf('\n \t 2: %1.25e',Exact_Surface_Integral_f2_Square)
%     fprintf('\n \t 3: %1.25e',Exact_Surface_Integral_f3_Square)
    
    %==========================================================================
    %
    % Compute the error in the approximation of the surface integrals for the
    % test integrands.
    %
    %==========================================================================
    Error_in_the_Approximate_Surface_Integral_f1_Square=abs(Exact_Surface_Integral_f1_Square-...
        Approximate_Surface_Integral_f1_Square)./abs(Approximate_Surface_Integral_f1_Square);
    Error_in_the_Approximate_Surface_Integral_f2_Square=abs(Exact_Surface_Integral_f2_Square-...
        Approximate_Surface_Integral_f2_Square)./abs(Approximate_Surface_Integral_f2_Square);
    Error_in_the_Approximate_Surface_Integral_f3_Square=abs(Exact_Surface_Integral_f3_Square-...
        Approximate_Surface_Integral_f3_Square)./abs(Approximate_Surface_Integral_f3_Square);
    
    Error_in_the_Approximate_Surface_Integral_f1_Circle=abs(Exact_Surface_Integral_f1_Circle-...
        Approximate_Surface_Integral_f1_Circle)./abs(Approximate_Surface_Integral_f1_Circle);
    Error_in_the_Approximate_Surface_Integral_f2_Circle=abs(Exact_Surface_Integral_f2_Circle-...
        Approximate_Surface_Integral_f2_Circle)./abs(Approximate_Surface_Integral_f2_Circle);
    Error_in_the_Approximate_Surface_Integral_f3_Circle=abs(Exact_Surface_Integral_f3_Circle-...
        Approximate_Surface_Integral_f3_Circle)./abs(Approximate_Surface_Integral_f3_Circle);
    %==========================================================================
    
    doprint=0;
    
    if doprint == 1
        %==========================================================================
        %
        % Print some stuff
        %
        %==========================================================================
        syms x y
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{1}(x,y)=\n')
        pretty(f1_Plane(x,y))
        fprintf('over the square with vertices (-1/2,1/2), (-1/2,-1/2), (1/2,-1/2)  \n')
        fprintf('and (1/2,1/2) is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f1_Square);
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{2}(x,y)=\n')
        pretty(f2_Plane(x,y))
        fprintf('over the square with vertices (-1/2,1/2), (-1/2,-1/2), (1/2,-1/2)  \n')
        fprintf('and (1/2,1/2) is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f2_Square);
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{3}(x,y)=\n')
        pretty(f3_Plane(x,y))
        fprintf('over the square with vertices (-1/2,1/2), (-1/2,-1/2), (1/2,-1/2)  \n')
        fprintf('and (1/2,1/2) is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f3_Square);
        fprintf('====================================================================\n')
        fprintf('\n\n\n')
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{1}(x,y)=\n')
        pretty(f1_Plane(x,y))
        fprintf('over the circle of radius 1 centered at the origin is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f1_Circle);
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{2}(x,y)=\n')
        pretty(f2_Plane(x,y))
        fprintf('over the circle of radius 1 centered at the origin is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f2_Circle);
        fprintf('====================================================================\n')
        fprintf('The relative error in the approximation of the surface integral of\n')
        fprintf('f_{3}(x,y)=\n')
        pretty(f3_Plane(x,y))
        fprintf('over the circle of radius 1 centered at the origin is %6.5e.\n',...
            Error_in_the_Approximate_Surface_Integral_f3_Circle);
        fprintf('====================================================================\n')
    end
    
    %==========================================================================
elseif Test_Index==2
    %==========================================================================
    %
    % Create a set of quadrature nodes over the unit square centered at the
    % origin
    %
    %==========================================================================
    Polygon_Vertices=[-1/2,1/2;-1/2,-1/2;1/2,-1/2;1/2,1/2;-1/2,1/2];
    Polygon_Vertices_sym=sym([-1/2,1/2;-1/2,-1/2;1/2,-1/2;1/2,1/2]);
    Points_on_Boundary_Curve=zeros(0,2);
    h0=0.022;
    
    for Polygon_Vertex_Index=1:(size(Polygon_Vertices,1)-1)
        Number_of_Points_on_Segment=ceil(sqrt((Polygon_Vertices(Polygon_Vertex_Index+1,:)-Polygon_Vertices(Polygon_Vertex_Index,:)).^2*ones(2,1))/(h0));
        Segment_Parameter=linspace(0,1,Number_of_Points_on_Segment).';
        Points_on_Boundary_Segment=(1-Segment_Parameter)*Polygon_Vertices(Polygon_Vertex_Index,:)+Segment_Parameter*Polygon_Vertices(Polygon_Vertex_Index+1,:);
        if Polygon_Vertex_Index==1
            Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment];
        else
            Points_on_Boundary_Curve=[Points_on_Boundary_Curve;Points_on_Boundary_Segment(2:end,:)];
        end
    end
    Points_on_Boundary_Curve=unique(Points_on_Boundary_Curve,'rows');
    p=haltonset(2);
    Quadrature_Nodes_Square=[p(1:(NN-size(Points_on_Boundary_Curve,1)+1),1)-1/2,p(1:(NN-size(Points_on_Boundary_Curve,1)+1),2)-1/2];
    Quadrature_Nodes_Square=[Quadrature_Nodes_Square;Points_on_Boundary_Curve];
    Quadrature_Nodes_Square=unique(Quadrature_Nodes_Square,'rows');
    Triangles_Square=delaunay(Quadrature_Nodes_Square);
    
    load('Planar_Nodes_Triangles_Cassini_lambda_0_h0_0_022.mat')
    Quadrature_Nodes_Circle=Quadrature_Nodes;
    Triangles_Circle=Triangles;
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface h(x,y,z)=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   h - For the surface S defined implicitly by h(x,y,z)=0, row i
    %   in the output of h should contain
    %   h(Quadrature_Nodes(i,1:3))
    %   h should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradh - The gradient of the function h.  Row i in the output
    %   of gradh should contain
    %   [dh/dx(Quadrature_Nodes(i,1:3)),dh/dy(Quadrature_Nodes(i,1:3)),dh/dz(Quadrature_Nodes(i,1:3)]
    %   gradh should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    Plane_Normal=(sym(randi([0 100],1,3))./sym(randi([0 100],1,3))).*sym(sign(randn(1,3)));
    Plane_Normal=simplify(Plane_Normal/norm(Plane_Normal,2));
    if sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2)<eps
        Rotation_Matrix=eye(3);
    else
        Rotation_Matrix=[Plane_Normal(1,3)/sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2+Plane_Normal(1,3).^2) 0 -sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2)./(sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2+Plane_Normal(1,3).^2));
            0 1 0;
            sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2)./(sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2+Plane_Normal(1,3).^2)) 0 Plane_Normal(1,3)./sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2+Plane_Normal(1,3).^2)]*...
            [Plane_Normal(1,1)./sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2) Plane_Normal(1,2)./(sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2)) 0;
            -Plane_Normal(1,2)./(sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2)) Plane_Normal(1,1)./sqrt(Plane_Normal(1,1).^2+Plane_Normal(1,2).^2) 0;
            0 0 1];
    end
    Plane_Normal_sym=simplify(Plane_Normal);
    Plane_Normal=double(Plane_Normal);
    Rotation_Matrix_sym=simplify(Rotation_Matrix);
    Rotation_Matrix=double(Rotation_Matrix);
    Polygon_Vertices_sym=(Rotation_Matrix*[Polygon_Vertices_sym,sym(zeros(4,1))].').';
    Quadrature_Nodes_Square=(Rotation_Matrix.'*[Quadrature_Nodes_Square,zeros(size(Quadrature_Nodes_Square,1),1)].').';
    Quadrature_Nodes_Circle=(Rotation_Matrix.'*[Quadrature_Nodes_Circle,zeros(size(Quadrature_Nodes_Circle,1),1)].').';
    h=@(p) p*Plane_Normal.';
    gradh=@(p) ones(size(p,1),1)*Plane_Normal;
    hessianh=@(p) zeros(size(p,1),9);
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface b(x,y,z)>=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   b - For the boundary of S defined implicitly by b(x,y,z)=0,
    %   row i in the output of b should contain
    %         b(Quadrature_Nodes(i,1:3))
    %   b should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradb - The gradient of the function b.  Row i in the output
    %   of gradb should contain
    %   [db/dx(Quadrature_Nodes(i,1:3)),db/dy(Quadrature_Nodes(i,1:3)),db/dz(Quadrature_Nodes(i,1:3)]
    %   gradb should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    b_Square=@(p) max(abs([p*Rotation_Matrix(1,:).',p*Rotation_Matrix(2,:).',p*Rotation_Matrix(3,:).']),[],2)-1/2;
    gradb_Square=[];
    hessianb_Square=[];
    
    Sphere_Radius=1/sqrt(pi);
    b_Circle=@(p)p(:,1).^2+p(:,2).^2+p(:,3).^2-Sphere_Radius.^2;
    gradb_Circle=@(p) [2.*p(:,1),2.*p(:,2),2.*p(:,3)];
    hessianb_Circle=@(p) [2*ones(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),2*ones(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),2*ones(size(p,1),1)];
    %==========================================================================
    
    %==========================================================================
    %
    % Generate the quadrature weights for the set of quadrature nodes generated
    % above.
    %
    %==========================================================================
    Boundary_Edge_Nodes=[];
    options.Planar_Surface_Flag=1;
    options.Planar_Surface_Normal=Plane_Normal;
    options.Planar_Boundary_Flag=1;
    % Use the approximate surface normal
    Quadrature_Weights_Square=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Square,Triangles_Square,Boundary_Edge_Nodes,h,gradh,b_Square,gradb_Square,options);
    
    options.Planar_Surface_Flag=1;
    options.Planar_Surface_Normal=Plane_Normal;
    options.Planar_Boundary_Flag=0;
    % Use the exact surface normal
    Quadrature_Weights_Circle=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Circle,Triangles_Circle,Boundary_Edge_Nodes,h,gradh,b_Circle,gradb_Circle,options);
    %==========================================================================
    
    %==========================================================================
    %
    % Adapt the planar integrands to the rotated planar surface
    %
    %==========================================================================
    f1_Plane_Rotated=@(x,y,z) f1_Plane((Rotation_Matrix(1,:)*([x,y,z].')).',(Rotation_Matrix(2,:)*([x,y,z].')).');
    f2_Plane_Rotated=@(x,y,z) f2_Plane((Rotation_Matrix(1,:)*([x,y,z].')).',(Rotation_Matrix(2,:)*([x,y,z].')).');
    f3_Plane_Rotated=@(x,y,z) f3_Plane((Rotation_Matrix(1,:)*([x,y,z].')).',(Rotation_Matrix(2,:)*([x,y,z].')).');
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the values of various test integrands at the quadrature nodes
    % generated above.
    %
    %==========================================================================
    F1_Square=f1_Plane_Rotated(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2),Quadrature_Nodes_Square(:,3));
    F2_Square=f2_Plane_Rotated(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2),Quadrature_Nodes_Square(:,3));
    F3_Square=f3_Plane_Rotated(Quadrature_Nodes_Square(:,1),Quadrature_Nodes_Square(:,2),Quadrature_Nodes_Square(:,3));
    
    F1_Circle=f1_Plane_Rotated(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2),Quadrature_Nodes_Circle(:,3));
    F2_Circle=f2_Plane_Rotated(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2),Quadrature_Nodes_Circle(:,3));
    F3_Circle=f3_Plane_Rotated(Quadrature_Nodes_Circle(:,1),Quadrature_Nodes_Circle(:,2),Quadrature_Nodes_Circle(:,3));
    %==========================================================================
    
    %==========================================================================
    %
    % Set the exact values of the surface integrals of the test integrands for
    % comparison.
    %
    %==========================================================================
    k=25;
    Exact_Surface_Integral_f1_Square=pi*erf(1/2)^2;
    Exact_Surface_Integral_f2_Square=quad2d(@(x,y) 1./(1+k*(x.^2+y.^2)),-1/2,1/2,-1/2,1/2,'AbsTol',eps,'RelTol',eps);
    Exact_Surface_Integral_f3_Square=(2.*(k.*log(4)+log(16)+2.*log(1+k)+2.*k.*log(1+k)-4.*log(2+k)-2.*k.*log(2+k)))./k.^2;
    
    
    fprintf('\n \t 1: %1.25e',Exact_Surface_Integral_f1_Square)
    fprintf('\n \t 2: %1.25e',Exact_Surface_Integral_f2_Square)
    fprintf('\n \t 3: %1.25e',Exact_Surface_Integral_f3_Square)
    
    Exact_Surface_Integral_f1_Circle=-pi*(exp(-Sphere_Radius^2)-1);
    Exact_Surface_Integral_f2_Circle=pi*log(1+k*Sphere_Radius^2)/k;
    Exact_Surface_Integral_f3_Circle=quad2d(@(x,y) 1./(1+k*(abs(x)+abs(y))),-Sphere_Radius,Sphere_Radius,@(x)-sqrt(Sphere_Radius.^2-x.^2),@(x) sqrt(Sphere_Radius.^2-x.^2),'AbsTol',eps,'RelTol',eps);
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the approximate values of the surface integrals of the test
    % integrands using the quadrature weights generated above for comparison.
    %
    %==========================================================================
    Approximate_Surface_Integral_f1_Square=F1_Square.'*Quadrature_Weights_Square;
    Approximate_Surface_Integral_f2_Square=F2_Square.'*Quadrature_Weights_Square;
    Approximate_Surface_Integral_f3_Square=F3_Square.'*Quadrature_Weights_Square;
    
    Approximate_Surface_Integral_f1_Circle=F1_Circle.'*Quadrature_Weights_Circle;
    Approximate_Surface_Integral_f2_Circle=F2_Circle.'*Quadrature_Weights_Circle;
    Approximate_Surface_Integral_f3_Circle=F3_Circle.'*Quadrature_Weights_Circle;
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the error in the approximation of the surface integrals for the
    % test integrands.
    %
    %==========================================================================
    Error_in_the_Approximate_Surface_Integral_f1_Square=abs(Exact_Surface_Integral_f1_Square-...
        Approximate_Surface_Integral_f1_Square)./abs(Approximate_Surface_Integral_f1_Square);
    Error_in_the_Approximate_Surface_Integral_f2_Square=abs(Exact_Surface_Integral_f2_Square-...
        Approximate_Surface_Integral_f2_Square)./abs(Approximate_Surface_Integral_f2_Square);
    Error_in_the_Approximate_Surface_Integral_f3_Square=abs(Exact_Surface_Integral_f3_Square-...
        Approximate_Surface_Integral_f3_Square)./abs(Approximate_Surface_Integral_f3_Square);
    
    Error_in_the_Approximate_Surface_Integral_f1_Circle=abs(Exact_Surface_Integral_f1_Circle-...
        Approximate_Surface_Integral_f1_Circle)./abs(Approximate_Surface_Integral_f1_Circle);
    Error_in_the_Approximate_Surface_Integral_f2_Circle=abs(Exact_Surface_Integral_f2_Circle-...
        Approximate_Surface_Integral_f2_Circle)./abs(Approximate_Surface_Integral_f2_Circle);
    Error_in_the_Approximate_Surface_Integral_f3_Circle=abs(Exact_Surface_Integral_f3_Circle-...
        Approximate_Surface_Integral_f3_Circle)./abs(Approximate_Surface_Integral_f3_Circle);
    %==========================================================================
    
    %==========================================================================
    %
    % Print some stuff
    %
    %==========================================================================
    f1_Plane_Rotated=@(x,y,z) f1_Plane((Rotation_Matrix_sym(1,:)*([x,y,z].')).',(Rotation_Matrix_sym(2,:)*([x,y,z].')).');
    f2_Plane_Rotated=@(x,y,z) f2_Plane((Rotation_Matrix_sym(1,:)*([x,y,z].')).',(Rotation_Matrix_sym(2,:)*([x,y,z].')).');
    f3_Plane_Rotated=@(x,y,z) f3_Plane((Rotation_Matrix_sym(1,:)*([x,y,z].')).',(Rotation_Matrix_sym(2,:)*([x,y,z].')).');
    syms x y z
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{1}(x,y,z)=\n')
    pretty(simplify(f1_Plane_Rotated(x,y,z)))
    fprintf('over the square with vertices:\n')
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(1,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(2,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(3,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(4,:)),'matrix([[','['),']])',']'))
    fprintf('in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f1_Square);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{2}(x,y,z)=\n')
    pretty(simplify(f2_Plane_Rotated(x,y,z)))
    fprintf('over the square with vertices:\n')
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(1,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(2,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(3,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(4,:)),'matrix([[','['),']])',']'))
    fprintf('in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f2_Square);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{3}(x,y,z)=\n')
    pretty(simplify(f3_Plane_Rotated(x,y,z)))
    fprintf('over the square with vertices:\n')
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(1,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(2,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(3,:)),'matrix([[','['),']])',']'))
    fprintf('    %s,\n',strrep(strrep(char(Polygon_Vertices_sym(4,:)),'matrix([[','['),']])',']'))
    fprintf('in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f3_Square);
    fprintf('====================================================================\n')
    fprintf('\n\n\n')
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{1}(x,y,z)=\n')
    pretty(simplify(f1_Plane_Rotated(x,y,z)))
    fprintf('the unit circle centered at (0,0,0) in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f1_Circle);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{2}(x,y,z)=\n')
    pretty(simplify(f2_Plane_Rotated(x,y,z)))
    fprintf('the unit circle centered at (0,0,0) in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f2_Circle);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{3}(x,y,z)=\n')
    pretty(simplify(f3_Plane_Rotated(x,y,z)))
    fprintf('the unit circle centered at (0,0,0) in the plane with normal:\n')
    fprintf('    %s \n',strrep(strrep(char(Plane_Normal_sym),'matrix([[','['),']])',']'))
    fprintf('is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f3_Circle);
    fprintf('====================================================================\n')
    %==========================================================================
elseif Test_Index==3
    %==========================================================================
    %
    % Load quadrature nodes on the surface of a sphere of area 1 using
    %
    %==========================================================================
    load('Cassini_Oval_Scalloped_Nodes_Triangles_a_0_0000_b_0_2821_freq_2_0000_amp_0_0000_shift_0_7854_sig_1_0000_h_0_0160.mat')
    Quadrature_Nodes_Planar_Boundary_Top=Quadrature_Nodes;
    Triangles_Planar_Boundary_Top=Triangles;
    load('Cassini_Oval_Scalloped_Nodes_Triangles_a_0_0000_b_0_2821_freq_2_0000_amp_0_0000_shift_0_7854_sig_m1_0000_h_0_0160.mat')
    Quadrature_Nodes_Planar_Boundary_Bottom=Quadrature_Nodes;
    Triangles_Planar_Boundary_Bottom=Triangles;
    load('Cassini_Oval_Scalloped_Nodes_Triangles_a_0_0000_b_0_2821_freq_2_0000_amp_0_0500_shift_0_7854_sig_1_0000_h_0_0160.mat')
    Quadrature_Nodes_Scalloped_Boundary_Top=Quadrature_Nodes;
    Triangles_Scalloped_Boundary_Top=Triangles;
    load('Cassini_Oval_Scalloped_Nodes_Triangles_a_0_0000_b_0_2821_freq_2_0000_amp_0_0500_shift_0_7854_sig_m1_0000_h_0_0160.mat')
    Quadrature_Nodes_Scalloped_Boundary_Bottom=Quadrature_Nodes;
    Triangles_Scalloped_Boundary_Bottom=Triangles;
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface h(x,y,z)=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   h - For the surface S defined implicitly by h(x,y,z)=0, row i
    %   in the output of h should contain
    %   h(Quadrature_Nodes(i,1:3))
    %   h should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradh - The gradient of the function h.  Row i in the output
    %   of gradh should contain
    %   [dh/dx(Quadrature_Nodes(i,1:3)),dh/dy(Quadrature_Nodes(i,1:3)),dh/dz(Quadrature_Nodes(i,1:3)]
    %   gradh should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    Sphere_Radius=1/(2*sqrt(pi));
    h=@(p)p(:,1).^2+p(:,2).^2+p(:,3).^2-Sphere_Radius.^2;
    gradh=@(p) [2.*p(:,1),2.*p(:,2),2.*p(:,3)];
    %==========================================================================
    
    %==========================================================================
    %
    % Define the level surface b(x,y,z)>=0 and its gradient so that the normal
    % to the surface can be computed exactly.  They have the following specifications
    %
    %   b - For the boundary of S defined implicitly by b(x,y,z)=0,
    %   row i in the output of b should contain
    %         b(Quadrature_Nodes(i,1:3))
    %   b should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %   gradb - The gradient of the function b.  Row i in the output
    %   of gradb should contain
    %   [db/dx(Quadrature_Nodes(i,1:3)),db/dy(Quadrature_Nodes(i,1:3)),db/dz(Quadrature_Nodes(i,1:3)]
    %   gradb should take in Quadrature_Nodes as an
    %   (Number_of_Quadrature_Nodes X 3) Array
    %
    %==========================================================================
    amp=0.05;
    
    sig=1;
    b_Planar_Boundary_Top=@(p) sig*p(:,3);
    gradb_Planar_Boundary_Top = @(p) [zeros(size(p,1),1),zeros(size(p,1),1),sig*ones(size(p,1),1)];
    hessianb_Planar_Boundary_Top=@(p) [zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1)];
    
    b_Scalloped_Boundary_Top=@(p) sig.*(p(:,3) - (2.*amp.*p(:,1).*p(:,2))./(p(:,1).^2 + p(:,2).^2));
    gradb_Scalloped_Boundary_Top=@(p) [(2.*amp.*p(:,2).*sig.*(p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^2, -(2.*amp.*p(:,1).*sig.*(p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^2, sig*ones(size(p,1),1)];
    hessianb_Scalloped_Boundary_Top=@(p) [-(4.*amp.*p(:,1).*p(:,2).*sig.*(p(:,1).^2 - 3.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, (2.*amp.*sig.*(p(:,1).^4 + p(:,2).^4 - 6.*p(:,1).^2.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, zeros(size(p,1),1), (2.*amp.*sig.*(p(:,1).^4 + p(:,2).^4 - 6.*p(:,1).^2.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, (4.*amp.*p(:,1).*p(:,2).*sig.*(3.*p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, zeros(size(p,1),1), zeros(size(p,1),1), zeros(size(p,1),1), zeros(size(p,1),1)];
    
    sig=-1;
    b_Planar_Boundary_Bottom=@(p) sig*p(:,3);
    gradb_Planar_Boundary_Bottom = @(p) [zeros(size(p,1),1),zeros(size(p,1),1),sig*ones(size(p,1),1)];
    hessianb_Planar_Boundary_Bottom=@(p) [zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1),zeros(size(p,1),1)];
    
    b_Scalloped_Boundary_Bottom=@(p) sig.*(p(:,3) - (2.*amp.*p(:,1).*p(:,2))./(p(:,1).^2 + p(:,2).^2));
    gradb_Scalloped_Boundary_Bottom=@(p) [(2.*amp.*p(:,2).*sig.*(p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^2, -(2.*amp.*p(:,1).*sig.*(p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^2, sig*ones(size(p,1),1)];
    hessianb_Scalloped_Boundary_Bottom=@(p) [-(4.*amp.*p(:,1).*p(:,2).*sig.*(p(:,1).^2 - 3.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, (2.*amp.*sig.*(p(:,1).^4 + p(:,2).^4 - 6.*p(:,1).^2.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, zeros(size(p,1),1), (2.*amp.*sig.*(p(:,1).^4 + p(:,2).^4 - 6.*p(:,1).^2.*p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, (4.*amp.*p(:,1).*p(:,2).*sig.*(3.*p(:,1).^2 - p(:,2).^2))./(p(:,1).^2 + p(:,2).^2).^3, zeros(size(p,1),1), zeros(size(p,1),1), zeros(size(p,1),1), zeros(size(p,1),1)];
    %==========================================================================
    
    %==========================================================================
    %
    % Generate the quadrature weights for the set of quadrature nodes generated
    % above.
    %
    %==========================================================================
    Boundary_Edge_Nodes=[];
    options.Planar_Boundary_Flag=0;
    % Use the approximate surface normal
    Quadrature_Weights_Scalloped_Boundary_Bottom=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Scalloped_Boundary_Bottom,Triangles_Scalloped_Boundary_Bottom,Boundary_Edge_Nodes,h,gradh,b_Scalloped_Boundary_Bottom,gradb_Scalloped_Boundary_Bottom,options);
    Quadrature_Weights_Scalloped_Boundary_Top=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Scalloped_Boundary_Top,Triangles_Scalloped_Boundary_Top,Boundary_Edge_Nodes,h,gradh,b_Scalloped_Boundary_Top,gradb_Scalloped_Boundary_Top,options);
    
    options.Planar_Boundary_Flag=1;
    options.Planar_Boundary_Normal=[0 0 1];
    % Use the exact surface normal
    Quadrature_Weights_Planar_Boundary_Bottom=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Planar_Boundary_Bottom,Triangles_Planar_Boundary_Bottom,Boundary_Edge_Nodes,h,gradh,b_Planar_Boundary_Bottom,gradb_Planar_Boundary_Bottom,options);
    Quadrature_Weights_Planar_Boundary_Top=Bounded_Smooth_Surface_Quadrature_RBF(Quadrature_Nodes_Planar_Boundary_Top,Triangles_Planar_Boundary_Top,Boundary_Edge_Nodes,h,gradh,b_Planar_Boundary_Top,gradb_Planar_Boundary_Top,options);
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the values of various test integrands at the quadrature nodes
    % generated above.
    %
    %==========================================================================
    F1_Planar_Boundary_Top=f1(Quadrature_Nodes_Planar_Boundary_Top(:,1),Quadrature_Nodes_Planar_Boundary_Top(:,2),Quadrature_Nodes_Planar_Boundary_Top(:,3));
    F2_Planar_Boundary_Top=f2(Quadrature_Nodes_Planar_Boundary_Top(:,1),Quadrature_Nodes_Planar_Boundary_Top(:,2),Quadrature_Nodes_Planar_Boundary_Top(:,3));
    F3_Planar_Boundary_Top=f3(Quadrature_Nodes_Planar_Boundary_Top(:,1),Quadrature_Nodes_Planar_Boundary_Top(:,2),Quadrature_Nodes_Planar_Boundary_Top(:,3));
    F4_Planar_Boundary_Top=f4(Quadrature_Nodes_Planar_Boundary_Top(:,1),Quadrature_Nodes_Planar_Boundary_Top(:,2),Quadrature_Nodes_Planar_Boundary_Top(:,3));
    F5_Planar_Boundary_Top=f5(Quadrature_Nodes_Planar_Boundary_Top(:,1),Quadrature_Nodes_Planar_Boundary_Top(:,2),Quadrature_Nodes_Planar_Boundary_Top(:,3));
    
    F1_Planar_Boundary_Bottom=f1(Quadrature_Nodes_Planar_Boundary_Bottom(:,1),Quadrature_Nodes_Planar_Boundary_Bottom(:,2),Quadrature_Nodes_Planar_Boundary_Bottom(:,3));
    F2_Planar_Boundary_Bottom=f2(Quadrature_Nodes_Planar_Boundary_Bottom(:,1),Quadrature_Nodes_Planar_Boundary_Bottom(:,2),Quadrature_Nodes_Planar_Boundary_Bottom(:,3));
    F3_Planar_Boundary_Bottom=f3(Quadrature_Nodes_Planar_Boundary_Bottom(:,1),Quadrature_Nodes_Planar_Boundary_Bottom(:,2),Quadrature_Nodes_Planar_Boundary_Bottom(:,3));
    F4_Planar_Boundary_Bottom=f4(Quadrature_Nodes_Planar_Boundary_Bottom(:,1),Quadrature_Nodes_Planar_Boundary_Bottom(:,2),Quadrature_Nodes_Planar_Boundary_Bottom(:,3));
    F5_Planar_Boundary_Bottom=f5(Quadrature_Nodes_Planar_Boundary_Bottom(:,1),Quadrature_Nodes_Planar_Boundary_Bottom(:,2),Quadrature_Nodes_Planar_Boundary_Bottom(:,3));
    
    F1_Scalloped_Boundary_Top=f1(Quadrature_Nodes_Scalloped_Boundary_Top(:,1),Quadrature_Nodes_Scalloped_Boundary_Top(:,2),Quadrature_Nodes_Scalloped_Boundary_Top(:,3));
    F2_Scalloped_Boundary_Top=f2(Quadrature_Nodes_Scalloped_Boundary_Top(:,1),Quadrature_Nodes_Scalloped_Boundary_Top(:,2),Quadrature_Nodes_Scalloped_Boundary_Top(:,3));
    F3_Scalloped_Boundary_Top=f3(Quadrature_Nodes_Scalloped_Boundary_Top(:,1),Quadrature_Nodes_Scalloped_Boundary_Top(:,2),Quadrature_Nodes_Scalloped_Boundary_Top(:,3));
    F4_Scalloped_Boundary_Top=f4(Quadrature_Nodes_Scalloped_Boundary_Top(:,1),Quadrature_Nodes_Scalloped_Boundary_Top(:,2),Quadrature_Nodes_Scalloped_Boundary_Top(:,3));
    F5_Scalloped_Boundary_Top=f5(Quadrature_Nodes_Scalloped_Boundary_Top(:,1),Quadrature_Nodes_Scalloped_Boundary_Top(:,2),Quadrature_Nodes_Scalloped_Boundary_Top(:,3));
    
    F1_Scalloped_Boundary_Bottom=f1(Quadrature_Nodes_Scalloped_Boundary_Bottom(:,1),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,2),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,3));
    F2_Scalloped_Boundary_Bottom=f2(Quadrature_Nodes_Scalloped_Boundary_Bottom(:,1),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,2),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,3));
    F3_Scalloped_Boundary_Bottom=f3(Quadrature_Nodes_Scalloped_Boundary_Bottom(:,1),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,2),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,3));
    F4_Scalloped_Boundary_Bottom=f4(Quadrature_Nodes_Scalloped_Boundary_Bottom(:,1),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,2),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,3));
    F5_Scalloped_Boundary_Bottom=f5(Quadrature_Nodes_Scalloped_Boundary_Bottom(:,1),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,2),Quadrature_Nodes_Scalloped_Boundary_Bottom(:,3));
    %==========================================================================
    
    %==========================================================================
    %
    % Set the exact values of the surface integrals of the test integrands for
    % comparison.
    %
    %==========================================================================
    Exact_Surface_Integral_f1=(1 + 84*pi + 560*pi^2 + 6720*pi^3)/(6720*pi^3);
    Exact_Surface_Integral_f2=quad2d(@(x,y) f2(x,y,sqrt(Sphere_Radius.^2-x.^2-y.^2)).*sqrt(-(Sphere_Radius.^2./(-Sphere_Radius.^2 + x.^2 + y.^2))),-Sphere_Radius,Sphere_Radius,@(x)-sqrt(Sphere_Radius.^2-x.^2),@(x)sqrt(Sphere_Radius.^2-x.^2),'AbsTol',10*eps,'RelTol',10*eps)+...
        quad2d(@(x,y) f2(x,y,-sqrt(Sphere_Radius.^2-x.^2-y.^2)).*sqrt(-(Sphere_Radius.^2./(-Sphere_Radius.^2 + x.^2 + y.^2))),-Sphere_Radius,Sphere_Radius,@(x)-sqrt(Sphere_Radius.^2-x.^2),@(x)sqrt(Sphere_Radius.^2-x.^2),'AbsTol',10*eps,'RelTol',10*eps);
    Exact_Surface_Integral_f3=1/9;
    Exact_Surface_Integral_f4=1/9;
    Exact_Surface_Integral_f5=1;
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the approximate values of the surface integrals of the test
    % integrands using the quadrature weights generated above for comparison.
    %
    %==========================================================================
    Approximate_Surface_Integral_f1_Planar_Boundary=F1_Planar_Boundary_Top.'*Quadrature_Weights_Planar_Boundary_Top+F1_Planar_Boundary_Bottom.'*Quadrature_Weights_Planar_Boundary_Bottom;
    Approximate_Surface_Integral_f2_Planar_Boundary=F2_Planar_Boundary_Top.'*Quadrature_Weights_Planar_Boundary_Top+F2_Planar_Boundary_Bottom.'*Quadrature_Weights_Planar_Boundary_Bottom;
    Approximate_Surface_Integral_f3_Planar_Boundary=F3_Planar_Boundary_Top.'*Quadrature_Weights_Planar_Boundary_Top+F3_Planar_Boundary_Bottom.'*Quadrature_Weights_Planar_Boundary_Bottom;
    Approximate_Surface_Integral_f4_Planar_Boundary=F4_Planar_Boundary_Top.'*Quadrature_Weights_Planar_Boundary_Top+F4_Planar_Boundary_Bottom.'*Quadrature_Weights_Planar_Boundary_Bottom;
    Approximate_Surface_Integral_f5_Planar_Boundary=F5_Planar_Boundary_Top.'*Quadrature_Weights_Planar_Boundary_Top+F5_Planar_Boundary_Bottom.'*Quadrature_Weights_Planar_Boundary_Bottom;
    
    Approximate_Surface_Integral_f1_Scalloped_Boundary=F1_Scalloped_Boundary_Top.'*Quadrature_Weights_Scalloped_Boundary_Top+F1_Scalloped_Boundary_Bottom.'*Quadrature_Weights_Scalloped_Boundary_Bottom;
    Approximate_Surface_Integral_f2_Scalloped_Boundary=F2_Scalloped_Boundary_Top.'*Quadrature_Weights_Scalloped_Boundary_Top+F2_Scalloped_Boundary_Bottom.'*Quadrature_Weights_Scalloped_Boundary_Bottom;
    Approximate_Surface_Integral_f3_Scalloped_Boundary=F3_Scalloped_Boundary_Top.'*Quadrature_Weights_Scalloped_Boundary_Top+F3_Scalloped_Boundary_Bottom.'*Quadrature_Weights_Scalloped_Boundary_Bottom;
    Approximate_Surface_Integral_f4_Scalloped_Boundary=F4_Scalloped_Boundary_Top.'*Quadrature_Weights_Scalloped_Boundary_Top+F4_Scalloped_Boundary_Bottom.'*Quadrature_Weights_Scalloped_Boundary_Bottom;
    Approximate_Surface_Integral_f5_Scalloped_Boundary=F5_Scalloped_Boundary_Top.'*Quadrature_Weights_Scalloped_Boundary_Top+F5_Scalloped_Boundary_Bottom.'*Quadrature_Weights_Scalloped_Boundary_Bottom;
    %==========================================================================
    
    %==========================================================================
    %
    % Compute the error in the approximation of the surface integrals for the
    % test integrands.
    %
    %==========================================================================
    Error_in_the_Approximate_Surface_Integral_f1_Planar_Boundary=abs(Exact_Surface_Integral_f1-...
        Approximate_Surface_Integral_f1_Planar_Boundary)./abs(Approximate_Surface_Integral_f1_Planar_Boundary);
    Error_in_the_Approximate_Surface_Integral_f2_Planar_Boundary=abs(Exact_Surface_Integral_f2-...
        Approximate_Surface_Integral_f2_Planar_Boundary)./abs(Approximate_Surface_Integral_f2_Planar_Boundary);
    Error_in_the_Approximate_Surface_Integral_f3_Planar_Boundary=abs(Exact_Surface_Integral_f3-...
        Approximate_Surface_Integral_f3_Planar_Boundary)./abs(Approximate_Surface_Integral_f3_Planar_Boundary);
    Error_in_the_Approximate_Surface_Integral_f4_Planar_Boundary=abs(Exact_Surface_Integral_f4-...
        Approximate_Surface_Integral_f4_Planar_Boundary)./abs(Approximate_Surface_Integral_f4_Planar_Boundary);
    Error_in_the_Approximate_Surface_Integral_f5_Planar_Boundary=abs(Exact_Surface_Integral_f5-...
        Approximate_Surface_Integral_f5_Planar_Boundary)./abs(Approximate_Surface_Integral_f5_Planar_Boundary);
    
    Error_in_the_Approximate_Surface_Integral_f1_Scalloped_Boundary=abs(Exact_Surface_Integral_f1-...
        Approximate_Surface_Integral_f1_Scalloped_Boundary)./abs(Approximate_Surface_Integral_f1_Scalloped_Boundary);
    Error_in_the_Approximate_Surface_Integral_f2_Scalloped_Boundary=abs(Exact_Surface_Integral_f2-...
        Approximate_Surface_Integral_f2_Scalloped_Boundary)./abs(Approximate_Surface_Integral_f2_Scalloped_Boundary);
    Error_in_the_Approximate_Surface_Integral_f3_Scalloped_Boundary=abs(Exact_Surface_Integral_f3-...
        Approximate_Surface_Integral_f3_Scalloped_Boundary)./abs(Approximate_Surface_Integral_f3_Scalloped_Boundary);
    Error_in_the_Approximate_Surface_Integral_f4_Scalloped_Boundary=abs(Exact_Surface_Integral_f4-...
        Approximate_Surface_Integral_f4_Scalloped_Boundary)./abs(Approximate_Surface_Integral_f4_Scalloped_Boundary);
    Error_in_the_Approximate_Surface_Integral_f5_Scalloped_Boundary=abs(Exact_Surface_Integral_f5-...
        Approximate_Surface_Integral_f5_Scalloped_Boundary)./abs(Approximate_Surface_Integral_f5_Scalloped_Boundary);
    %==========================================================================
    
    %==========================================================================
    %
    % Print some stuff
    %
    %==========================================================================
    syms x y z
    f1string=char(f1(x,y,z));
    f2string=char(f2(x,y,z));
    f3string=char(f3(x,y,z));
    f4string=char(f4(x,y,z));
    f5string=char(f5(x,y,z));
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{1}(x,y,z)=\n')
    pretty(f1(x,y,z))
    fprintf('over the sphere surface (area 1) with planar boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f1_Planar_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{2}(x,y,z)=\n')
    pretty(f2(x,y,z))
    fprintf('over the sphere surface (area 1) with planar boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f2_Planar_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{3}(x,y,z)=\n')
    pretty(f3(x,y,z))
    fprintf('over the sphere surface (area 1) with planar boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f3_Planar_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{4}(x,y,z)=\n')
    pretty(f4(x,y,z))
    fprintf('over the sphere surface (area 1) with planar boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f4_Planar_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{5}(x,y,z)=1\n')
    fprintf('over the sphere surface (area 1) with planar boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f5_Planar_Boundary);
    fprintf('====================================================================\n')
    fprintf('\n\n\n')
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{1}(x,y,z)=\n')
    pretty(f1(x,y,z))
    fprintf('over the sphere surface (area 1) with scalloped boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f1_Scalloped_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{2}(x,y,z)=\n')
    pretty(f2(x,y,z))
    fprintf('over the sphere surface (area 1) with scalloped boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f2_Scalloped_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{3}(x,y,z)=\n')
    pretty(f3(x,y,z))
    fprintf('over the sphere surface (area 1) with scalloped boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f3_Scalloped_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{4}(x,y,z)=\n')
    pretty(f4(x,y,z))
    fprintf('over the sphere surface (area 1) with scalloped boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f4_Scalloped_Boundary);
    fprintf('====================================================================\n')
    fprintf('The relative error in the approximation of the surface integral of\n')
    fprintf('f_{5}(x,y,z)=1\n')
    fprintf('over the sphere surface (area 1) with scalloped boundary is %6.5e.\n',...
        Error_in_the_Approximate_Surface_Integral_f5_Scalloped_Boundary);
    fprintf('====================================================================\n')
    %==========================================================================
end
end

function f=f1_Plane(x,y)
f=exp(-x.^2-y.^2);
end

function f=f2_Plane(x,y)
k=25;
f=1./(1+k*(x.^2+y.^2));
end

function f=f3_Plane(x,y)
k=25;
f=1./(1+k*(abs(x)+abs(y)));
end

function f=f1(x,y,z)
f=1+x+y.^2+x.^2.*y+x.^4+y.^5+x.^2.*y.^2.*z.^2;
end

function f=f2(x,y,z)
f=0.75.*exp(-(9.*x-2).^2./4-(9.*y-2).^2./4-(9.*z-2).^2./4)+...
    0.75.*exp(-(9.*x+1).^2./49-(9.*y+1)./10-(9.*z+1)./10)+...
    0.5.*exp(-(9.*x-7).^2./4-(9.*y-3).^2./4-(9.*z-5).^2./4)-...
    0.2.*exp(-(9.*x-4).^2-(9.*y-7).^2-(9.*z-5).^2);
end

function f=f3(x,y,z)
f=(1+tanh(-9.*x-9.*y+9.*z))./9;
end

function f=f4(x,y,z)
f=(1+sign(-9.*x-9.*y+9.*z))./9;
end

function f=f5(x,~,~)
f=ones(size(x));
end