
function [Ntri,Nrule,NruleC,momerr,cpus]=demo_cub_continents(...
    continent_parm,deg,doplot)

%--------------------------------------------------------------------------
% Object:
%
% Demo of cubature over spherical polygons, in which the domains are
% coarse approximation of continents.
%--------------------------------------------------------------------------
% Input:
%
% continent_parm   : 1. South/North America, 2. Africa, 3 or 4. Australia.
% deg              : Algebraic degree of exactness.
% doplot           : 0: no plots, 1: plots.
%--------------------------------------------------------------------------
% Output:
%
% Ntri             : Number of spherical triangles.
% Nrule            : Cardinality of the rule.
% NruleC           : Cardinality of the compressed rule.
% momerr           : Moment matching.
% cpus             : Array of cputimes
%                    cpus(1): polyshape + triangulation
%                    cpus(2): basic rule computation
%                    cpus(3): compressed rule computation (from basic rule)
%--------------------------------------------------------------------------
% Information:
% Authors: A. Sommariva and M. Vianello.
% Update: January 12, 2022.
%--------------------------------------------------------------------------
%% Copyright (C) 2021
%% Alvise Sommariva, Marco Vianello.
%%
%% This program is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published by
%% the Free Software Foundation; either version 2 of the License, or
%% (at your option) any later version.
%%
%% This program is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%% GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; if not, write to the Free Software
%% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
%%
%% Authors:
%% Alvise Sommariva, Marco Vianello.
%%
%% Date: JULY 13, 2021-
%% Significant update: January 12, 2022.
%--------------------------------------------------------------------------

% clf; clear all; close all;

if nargin < 1
    % 1. South/North America, 2. Africa, 3 or 4. Australia.
    continent_parm=3;
end

if nargin < 2
    % Algebraic degree of exactness of the cubature rule.
    deg=10;
end

if nargin < 3
    doplot=1;        % 0: no plots, 1: plots.
end
write_stats=1;   % 0: no stats, 1: stats.







%---------------------------- Main code below------------------------------

% In vertices we store, respectively, Longitude and Latitute, in degrees.
switch continent_parm
    case 1
        str='South-America and North-America';
        continent_pshape=coastline_america;

    case 2
        str='Africa';
        continent_pshape=coastline_africa;

    case 3
        str='Australia island';
        continent_pshape=coastline_australia(0);

    case 4

        % As an example, we add directly the data of Longitudes and
        % Latitutes in degrees, so that one can easily adapt the code to
        % other domains.

        str='Australia coarse (33 vertices)';
        vertices_degs=100*[
            1.462300280000000  -0.386970830000000
            1.451162780000000  -0.381483330000000
            1.396121110000000  -0.361567220000000
            1.385678610000000  -0.348266670000000
            1.374529440000000  -0.349083330000000
            1.359337220000000  -0.345341940000000
            1.340596110000000  -0.329141390000000
            1.235266670000000  -0.339379720000000
            1.150004170000000  -0.335283610000000
            1.131528610000000  -0.261491940000000
            1.142312220000000  -0.263158890000000
            1.168041670000000  -0.205237780000000
            1.221729170000000  -0.172633330000000
            1.239712780000000  -0.168258330000000
            1.243913060000000  -0.163392220000000
            1.252495830000000  -0.155816670000000
            1.259183610000000  -0.146770830000000
            1.268749720000000  -0.137471390000000
            1.283721390000000  -0.154966940000000
            1.292900000000000  -0.148620830000000
            1.303179720000000  -0.133733330000000
            1.308121390000000  -0.124083330000000
            1.317678890000000  -0.113258330000000
            1.347483890000000  -0.119512500000000
            1.360166940000000  -0.124987780000000
            1.359095560000000  -0.132850000000000
            1.395983610000000  -0.175387780000000
            1.415862220000000  -0.125583610000000
            1.437729170000000  -0.143991940000000
            1.462604440000000  -0.188633890000000
            1.500416670000000  -0.221254170000000
            1.531929440000000  -0.259316940000000
            1.525429440000000  -0.324442220000000
            1.499796110000000  -0.375050560000000
            1.462300280000000  -0.386970830000000];

        continent_pshape=polyshape(vertices_degs);


end



% fprintf(2,'\n \t Continent: '); disp(str);

% fprintf(2,'\n \t Determine triangulation: ');
tic;
% Make polyshape and triangulation
tri = triangulation(continent_pshape);
vertices=tri.Points;
cpus(1)=toc;




% Determine full cubature rule
% fprintf(2,'\n \t Determine full cubature rule: ');
tic;
xyzw=[];
for k=1:size(tri,1)

    % Local triangle
    triL=tri(k,:);

    % Vertices from degrees to radians, to cartesian coordinates.
    P1deg=vertices(triL(1),:); P1rad=deg2rad(P1deg);
    [P1x,P1y,P1z] = sph2cart(P1rad(1),P1rad(2),1); P1=[P1x P1y P1z]';

    P2deg=vertices(triL(2),:); P2rad=deg2rad(P2deg);
    [P2x,P2y,P2z] = sph2cart(P2rad(1),P2rad(2),1); P2=[P2x P2y P2z]';

    P3deg=vertices(triL(3),:); P3rad=deg2rad(P3deg);
    [P3x,P3y,P3z] = sph2cart(P3rad(1),P3rad(2),1); P3=[P3x P3y P3z]';

    % Local rule.
    [xyzwL] = cub_sphtri(deg,P1,P2,P3,0);

    % Add to storage.
    xyzw=[xyzw; xyzwL];

end
cpus(2)=toc;




% Determine Caratheodory-Tchakaloff compression.
% fprintf(2,'\n \t Determine Caratheodory-Tchakaloff compression: ');
tic;
[nodes,w,momerr,dbox] = dCATCHsph(deg,xyzw(:,1:3),xyzw(:,4));
cpus(3)=toc;

Ntri=size(tri);  Nrule=size(xyzw,1); NruleC=size(nodes,1);

if write_stats
    fprintf('\n \t .................... Stats ...............................')
    fprintf('\n \t ADE: %3.0f',deg);
    fprintf('\n \t CARD TRI: %3.0f',Ntri);
    fprintf('\n \t CARD F  : %3.0f',Nrule);
    fprintf('\n \t CARD C  : %3.0f',NruleC);
    fprintf('\n \t MOM.ERR.: %1.2e',momerr);
    fprintf('\n \t CPUTIMES: %1.2e',sum(cpus));
    fprintf('\n \n')
end




% ............................ 2D plot ....................................

% Figure 1. Plot triangulation
if doplot


    % ................. Figure 1 .................
    % 1. Plot continent.
    figure(1)
    hold on;
    plot(continent_pshape,'EdgeColor','black','LineWidth',4);
    triplot(tri);


    % 2. Plot nodes (full rule)
    XX=xyzw(:,1); YY=xyzw(:,2); ZZ=xyzw(:,3);
    [TH,PHI,R] = cart2sph(XX,YY,ZZ);
    THdeg=rad2deg(TH);
    PHIdeg=rad2deg(PHI);
    plot(THdeg,PHIdeg,'g.');


    % % 3. Plot nodes (compressed rule)
    XXc=nodes(:,1); YYc=nodes(:,2); ZZc=nodes(:,3);
    [THc,PHIc,R] = cart2sph(XXc,YYc,ZZc);
    THdegc=rad2deg(THc);
    PHIdegc=rad2deg(PHIc);
    plot(THdegc,PHIdegc,'mo','MarkerEdgeColor','k',...
        'MarkerFaceColor','k',...
        'MarkerSize',8)

    % 4. Add title.
    title_str=strcat('Cubature nodes projection, ADE= ',num2str(deg));
    title(title_str);

    hold off;






    % ............................ 3D plot ................................



    % ................. Figure 2 .................
    % * Plot Full set

    figure(2);

    % 1. Plot set.
    hold on; plot_s2('sphere',[],xyzw); hold on;

    % 2. Add title.
    title_str=strcat('Cubature nodes: full set, ADE= ',num2str(deg));
    title(title_str);

    hold off;




    % ................. Figure 3 .................
    % * Plot Compressed set and triangulation

    figure(3)
    hold on;

    % 1. Plot sphere.
    [X,Y,Z] = sphere(20);
    R=max(sqrt(XX.^2+YY.^2+ZZ.^2));
    plot3(R*X,R*Y,R*Z,'MarkerFaceColor','red');

    % 2. Plot spherical polygon triangulation.
    for k=1:size(tri,1)

        triL=tri(k,:); % local triangle
        % from degrees to radians, to cartesian

        P1deg=vertices(triL(1),:); P1rad=deg2rad(P1deg);
        [P1x,P1y,P1z] = sph2cart(P1rad(1),P1rad(2),1); P1=[P1x P1y P1z];

        P2deg=vertices(triL(2),:); P2rad=deg2rad(P2deg);
        [P2x,P2y,P2z] = sph2cart(P2rad(1),P2rad(2),1); P2=[P2x P2y P2z];

        P3deg=vertices(triL(3),:); P3rad=deg2rad(P3deg);
        [P3x,P3y,P3z] = sph2cart(P3rad(1),P3rad(2),1); P3=[P3x P3y P3z];

        plot_side(P1,P2); plot_side(P2,P3); plot_side(P3,P1);

        hold on;
    end


    % 3. Plot compressed set.
    plot_s2('sphere',[],[],nodes);


    % 4. Add title.
    title_str=...
        strcat('Triangulation and Caratheodory-Tchakaloff nodes, ADE= ',...
        num2str(deg));
    title(title_str);
    hold off;
end










function plot_side(A,B)

%--------------------------------------------------------------------------
% Object:
% Plot "geodesic line" between A and B.
%--------------------------------------------------------------------------
% Input:
% A,B: row vectors of two points of the sphere to be joined.
%--------------------------------------------------------------------------

R=norm(A);
t=linspace(0,1,10);
C=[];
for k=1:length(t)
    CC=A+t(k)*(B-A); CC=CC/norm(CC); C=[C; R*CC];
end

mycolor='black';
plot3(C(:,1),C(:,2),C(:,3),mycolor,'LineWidth',1);

