-----------------------------------------------------------------------
--          GtkAda - Ada95 binding for the Gimp Toolkit              --
--                                                                   --
--                     Copyright (C) 2003                            --
--        Emmanuel Briot, Joel Brobecker and Arnaud Charlet          --
--                                                                   --
-- This library is free software; you can redistribute it and/or     --
-- modify it under the terms of the GNU General Public               --
-- License as published by the Free Software Foundation; either      --
-- version 2 of the License, or (at your option) any later version.  --
--                                                                   --
-- This library is distributed in the hope that it will be useful,   --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of    --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details.                          --
--                                                                   --
-- You should have received a copy of the GNU General Public         --
-- License along with this library; if not, write to the             --
-- Free Software Foundation, Inc., 59 Temple Place - Suite 330,      --
-- Boston, MA 02111-1307, USA.                                       --
--                                                                   --
-- As a special exception, if other files instantiate generics from  --
-- this unit, or you link this unit with other files to produce an   --
-- execuTable, this  unit  does not  by itself cause  the resulting  --
-- execuTable to be covered by the GNU General Public License. This  --
-- exception does not however invalidate any other reasons why the   --
-- execuTable file  might be covered by the  GNU Public License.     --
-----------------------------------------------------------------------

with Glib.Object;              use Glib.Object;
with Glib.Values;              use Glib.Values;
with Gtk;                      use Gtk;
with Gtk.Handlers;             use Gtk.Handlers;
with Gtk.Enums;                use Gtk.Enums;

package body Trains_View is

   package Object_Callback is new Gtk.Handlers.Callback (GObject_Record);

   n_Train_Column    : constant := 0;
   Train_Column      : constant := 1;
   Seats_Column      : constant := 2;
   Seatsmax_Column   : constant := 3;
   Location_Column   : constant := 4;
   Foreground_Column : constant := 5;

   Time_Column : constant := 6;

   type String_Ref_T is access all String;

   type record_Train_iter is record
      Train : String_Ref_T := null;
      iter  : Gtk_Tree_Iter := Null_Iter;
   end record;

   type Trains_Known_T is array (1..100) of record_Train_iter;

   Trains_Known : Trains_Known_T;

   next_Train : Integer := Trains_Known'First;

   --  Insert a new line in the tree, as a child of Parent
   --+ (or as a root node) if Parent is Null_Iter.

   procedure Edited_Callback
     (Model  : access GObject_Record'Class;
      Params : Glib.Values.GValues);

   procedure Text_Edited_Callback
     (Model  : access GObject_Record'Class;
      Params : Glib.Values.GValues);


   function Help return String is
   begin
      return "A @bGtk_Tree_View@B is a widget used to display a Number of"
        & " lines, organized either as a simple list or into a tree"
        & " hierarchy." & ASCII.LF
        & "Each line can contain multiple resizable column, each of which"
        & " can contain pixmaps, texts, or both. The columns can be sorted"
        & " interactively by the user by clicking on the column header."
        & ASCII.LF
        & "Like the @bGtk_Text_View@B, this widget is based on the model-"
        & "view-controller paradigm: data is stored in a non graphical"
        & " object (a @bGtk_Tree_Model@B), which is then associated with"
        & " one or many view. The model provides subprograms for inserting"
        & " or removing lines, as well as for traversing an existing tree."
        & ASCII.LF
        & "Cells in the tree can be defined as ediTable, as shown in this"
        & " example: in this case, the user needs to double click on the"
        & " cell, and an entry widget is then displayed in which the text"
        & " can be modified";
   end Help;

   --------------
   -- Add_Line --
   --------------

   procedure Add_Line
     --  (Model    : access Gtk_Tree_Store_Record'Class;
     (Time : Integer;
      Train_Number : Integer;
      Train    : String;
      Seats    : Integer;
      Seatsmax : Integer;
      Location : String;
      Parent   : Gtk_Tree_Iter := Null_Iter)
   is
      Iters : Gtk_Tree_Iter;

      --  package Tree_Cb is new
      --+   Handlers.Callback (Gtk_Tree_View_Record);

   begin
      --  The implementation here is not the most efficient:
      --+ it is often easier to import yourself the function
      --+ gtk_tree_store_set with the correct set of arguments,
      --+ as shown in the example in gtk-tree_store.ads

      for I in Trains_Known'Range loop
         if Trains_Known (I).Train /= null then

            if Trains_Known (I).Train.all = Train then
               Set (Model, Trains_Known (I).iter,
                    Seats_Column, Gint (Seats));
               Set (Model, Trains_Known (I).iter,
                    Seatsmax_Column, Gint (Seatsmax));
               Set (Model, Trains_Known (I).iter,
                    Location_Column, Location);
               Set (Model, Trains_Known (I).iter,
                    Time_Column, Gint (Time));
               Draw (Tree);
               return;
            end if;
         end if;

      end loop;

      Append (Model, Iters, Parent);

      Set (Model, Iters, n_Train_Column, Gint(Train_Number));
      Set (Model, Iters, Train_Column, Train);
      Set (Model, Iters, Seats_Column, Gint(Seats));
      Set (Model, Iters, Seatsmax_Column, Gint(Seatsmax));
      Set (Model, Iters, Location_Column, Location);

      Set (Model, Iters, Time_Column, Gint(Time));

      Set (Model, Iters, Foreground_Column, "black");

      Trains_Known(next_Train).Train := new String'(Train);
      Trains_Known(next_Train).iter := Iters;

      next_Train := next_Train + 1;

      Draw(Tree);

      Iter := Iters;
   end Add_Line;

   ---------------------
   -- Edited_Callback --
   ---------------------

   procedure Edited_Callback
     (Model  : access GObject_Record'Class;
      Params : Glib.Values.GValues)
   is
      --  M             : constant Gtk_Tree_Store := Gtk_Tree_Store (Model);
      --  Path_String   : constant String := Get_String (Nth (Params, 1));
      --  Iter          : constant Gtk_Tree_Iter :=
      --  Get_Iter_From_String (M, Path_String);
      --+ Old_Value     : Boolean;
   begin
      null;
      --  Old_Value := Get_Boolean (M, Iter, Active_Column);
      --  Set (M, Iter, Active_Column, not Old_Value);
   end Edited_Callback;

   --------------------------
   -- Text_Edited_Callback --
   --------------------------

   procedure Text_Edited_Callback
     (Model  : access GObject_Record'Class;
      Params : Glib.Values.GValues)
   is
      --  M             : constant Gtk_Tree_Store := Gtk_Tree_Store (Model);
      --  Path_String   : constant String := Get_String (Nth (Params, 1));
      --  Text_Value    : constant GValue := Nth (Params, 2);
      --  Iter          : Gtk_Tree_Iter :=
      --+   Get_Iter_From_String (M, Path_String);
   begin
      null;
      --  Set_Value (M, Iter, Text_Column, Text_Value);
   end Text_Edited_Callback;

   ---------
   -- Run --
   ---------

   procedure Run (Frame : access Gtk_Scrolled_Window_Record'Class) is
      --  Model    : Gtk_Tree_Store;
      --  Tree     : Gtk_Tree_View;
      --  Scrolled : Gtk_Scrolled_Window;
      --  Col      : Gtk_Tree_View_Column;
      --  Num      : Gint;
      --  Text_Render   : Gtk_Cell_Renderer_Text;
      --  Toggle_Render : Gtk_Cell_Renderer_Toggle;
      --  Parent, Iter  : Gtk_Tree_Iter;

   begin
      --  Create the model that contains the actual data. In this example,
      --+ we create this as a tree, although it could also be a simple list.
      --+
      --+ Each data is in fact a line in the graphical widget, and can contain
      --+ multiple columns, not all of which are visible at any Time. Some can
      --+ be used as flags to indicate how the rendering should be done.

      Gtk_New (Model,
               (n_Train_Column    => GType_Int,
                Train_Column      => GType_String,
                Seats_Column      => GType_Int,
                Seatsmax_Column   => GType_Int,
                Location_Column   => GType_String,
                Time_Column       => GType_Int,
                Foreground_Column => GType_String));

      --  Create the view: it shows two columns, the first contains some text,
      --+ the second contains a toggle button. In each column, a renderer is
      --+ used to display the data graphically. In the future, it will be
      --+ possible to create your own renderers with GtkAda. For now, we simply
      --+ use two of the predefined renderers. The list of possible attributes
      --+ for these is defined in their respective packages.

      Gtk_New (Tree, Model);

      Gtk_New (Text_Render);
      Gtk_New (Toggle_Render);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "Train Nr.");
      Pack_Start (Col, Text_Render, True);
      Add_Attribute (Col, Text_Render, "text", n_Train_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "Journey & Train Type");
      Pack_Start (Col, Text_Render, True);
      Add_Attribute (Col, Text_Render, "text", Train_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "# Passengers aboard");
      Pack_Start (Col, Text_Render, False);
      Add_Attribute (Col, Text_Render, "text", Seats_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "Seats");
      Pack_Start (Col, Text_Render, False);
      Add_Attribute (Col, Text_Render, "text", Seatsmax_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "Status");
      Pack_Start (Col, Text_Render, False);
      Add_Attribute (Col, Text_Render, "text", Location_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      Gtk_New (Col);
      Num := Append_Column (Tree, Col);
      Set_Title (Col, "Time");
      Pack_Start (Col, Text_Render, False);
      Add_Attribute (Col, Text_Render, "text", Time_Column);
      Add_Attribute (Col, Text_Render, "foreground", Foreground_Column);

      --  By default, the toggle renderer doesn't react to clicks, ie the user
      --+ cannot interactively change the value of the radio button. This needs
      --+ a special callback for the "edited" signal.
      --+ The same applies for text renderers.
      --+
      --+ In both cases, the callback should be used for validation of the
      --+ input.

      Object_Callback.Object_Connect
        (Toggle_Render, "toggled", Edited_Callback'Access,
         Slot_Object => Model);
      Object_Callback.Object_Connect
        (Text_Render, "edited", Text_Edited_Callback'Access,
         Slot_Object => Model);

      --  Insert some data in the tree

      Parent := Null_Iter;

      --  Insert the view in the frame

      Gtk_New (Scrolled);
      Set_Policy (Scrolled, Policy_Always, Policy_Always);

      Add (Frame, Tree);
   end Run;

end Trains_View;
