--------------------------------------------------------
--  Copyright (C) 2006                                --
--  Alessandro Temil                                  --
--  Railway 1.0                                       --
--  Concurrent and Distributed Systems class project  --
--  Master Degree in Computer Science                 --
--  Academic year 2005/6                              --
--  Dept. of Pure and Applied Mathematics             --
--  University of Padua, Italy                        --
--                                                    --
--  This program is free software; you can            --
--  redistribute it and/or modify it under the terms  --
--  of the GNU General Public License as published    --
--  by the Free Software Foundation; either           --
--  version 2 of the License, or (at your option)     --
--  any later version.                                --
--------------------------------------------------------

with Distrailway.Central_Control;
with Distrailway.Notice_Boards;   use Distrailway.Notice_Boards;

package body Railway.Platform is

   protected body Platform_T is

      entry Train_Arrived
        (Train_Name : String_Ref_T;
         Train_Number : Range_Trains;
         Free_Seats : Integer;
         Seats : Integer;
         Destination : String_Ref_T) when True is
      begin

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Train_Number,
            Train_Name.all,
            Seats - Free_Seats,
            Seats,
            "queued for platform " & Name.all & " " & Station.all);

         Distrailway.Notice_Boards.Update
           (Integer (To_Duration (Passed)),
            Station.all,
            Name.all,
            Train_Name.all & " to " & Destination.all
            & " queued for access to platform");

         requeue Stopping;
      end Train_Arrived;

      entry Stopping
        (Train_Name : String_Ref_T;
         Train_Number : Range_Trains;
         Free_Seats : Integer;
         Seats : Integer;
         Destination : String_Ref_T) when Stopped_Train = null is
      begin

         Stopped_Train := Train_Name;
         Stopped_Train_Number := Train_Number;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Train_Number,
            Train_Name.all,
            Seats - Free_Seats,
            Seats,
            "arriving at platform " & Name.all & " in " & Station.all);

         Distrailway.Notice_Boards.Update
           (Integer (To_Duration (Passed)),
            Station.all,
            Name.all,
            Train_Name.all & " to " & Destination.all
            & " arrived at the platform");

         tFree_Seats := Free_Seats;
         tSeats := Seats;

      end Stopping;

      procedure Boarding
        (Train_Name : String_Ref_T;
         next_stage : Segment_T;
         Destination : String_Ref_T) is
      begin

         TrainBoarding := Train_Name;
         next_stage_Train := next_stage;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Stopped_Train_Number,
            Train_Name.all,
            tSeats - tFree_Seats,
            tSeats,
            "boarding passengers at platform "
            & Name.all & " in " & Station.all);

         Distrailway.Notice_Boards.Update
           (Integer (To_Duration (Passed)),
            Station.all,
            Name.all,
            Train_Name.all & " to " & Destination.all
            & " boarding passengers");

      end Boarding;

      procedure Close_Doors
        (Train_Name : String_Ref_T;
         Free_Seats : out Integer;
         Destination : String_Ref_T) is
      begin

         Stopped_Train := null;
         TrainBoarding := null;

         Free_Seats := tFree_Seats;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Stopped_Train_Number,
            Train_Name.all,
            tSeats - tFree_Seats,
            tSeats,
            "leaving platform " & Name.all & " " & Station.all);

         Distrailway.Notice_Boards.Update
           (Integer (To_Duration (Passed)),
            Station.all,
            Name.all,
            Train_Name.all & " to " & Destination.all & " departing");
      end Close_Doors;

      entry Depart(for T in Range_Trains)
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T)
      when TrainBoarding /= null and Stopped_Train_Number = T
      and tFree_Seats > 0 is

         Index_Last_Segment_current_leg : Integer;
         Next_Segment : Integer;
      begin

         Index_Last_Segment_current_leg := Legs
           (Travel_Index.all).Route'Last;
         Next_Segment := Legs
           (Travel_Index.all).Route'First;

         tFree_Seats := tFree_Seats - 1;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Stopped_Train_Number,
            Stopped_Train.all,
            tSeats - tFree_Seats,
            tSeats,
            "boarding passengers at platform "
            & Name.all & " in " & Station.all);

         Distrailway.Central_Control.Upd_Passenger
           (Integer (To_Duration (Passed)),
            Passenger_Name.all,
            "boards " & TrainBoarding.all & " at platform "
            & Name.all & " in " & Station.all);

         if(Travel_Index.all = Legs'Last - 1) then
            requeue Legs (Travel_Index.all).Route
              (Index_Last_Segment_current_leg).Departure_Platform.Arrive (T);
         else
            requeue Legs (Travel_Index.all).Route
              (Index_Last_Segment_current_leg).Departure_Platform.Change_Train (T);
         end if;

      end Depart;

      entry Arrive(for T in Range_Trains)
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T)
      when Stopped_Train /= null and Stopped_Train_Number = T is
      begin

         tFree_Seats := tFree_Seats + 1;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Passenger
           (Integer (To_Duration (Passed)),
            Passenger_Name.all,
            "alights train " & Stopped_Train.all & " at platform "
            & Name.all & " in " & Station.all);

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Stopped_Train_Number,
            Stopped_Train.all,
            tSeats - tFree_Seats,
            tSeats,
            "arriving at platform " & Name.all & " in " & Station.all);

         Travel_Index.all := Travel_Index.all + 1;

         if Legs(Travel_Index.all - 1).Train_Type = EC then
            Legs(Travel_Index.all - 1).Tickets.Release_Ticket
              (Legs,
               Travel_Index,
               Passenger_Name);
            --  null;
         end if;

      end Arrive;

      entry Change_Train(for T in Range_Trains)
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T)
      when Stopped_Train /= null
      and Stopped_Train_Number = T is

         Index_First_Segment_leg : Integer;

      begin

         tFree_Seats := tFree_Seats + 1;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Passenger
           (Integer (To_Duration (Passed)),
            Passenger_Name.all,
            "changes over at platform " & Name.all & " in " & Station.all);

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Train
           (Integer (To_Duration (Passed)),
            Stopped_Train_Number,
            Stopped_Train.all,
            tSeats - tFree_Seats,
            tSeats,
            "arriving at platform " & Name.all & " " & Station.all);

         Travel_Index.all := Travel_Index.all + 1;

         Index_First_Segment_leg := Legs
           (Travel_Index.all).Route'First;

         if Legs (Travel_Index.all - 1).Train_Type = EC then
            Legs (Travel_Index.all - 1).Tickets.Release_Ticket
              (Legs,
               Travel_Index,
               Passenger_Name);
         end if;

         requeue Legs (Travel_Index.all).Route
           (Index_First_Segment_leg).Departure_Platform.Depart
           (Legs (Travel_Index.all).Train_Number);

      end Change_Train;

   end Platform_T;

   protected body Ticket_DB_T is

      entry Book_Leg
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T) when True is

         Index_First_Segment : Integer := 0;
         R_Low : Integer;
         R_High : Integer;
      begin

         R_Low := Legs (Travel_Index.all).Route'First;
         R_High := Legs (Travel_Index.all).Route'Last;

         for Index_stage in R_Low .. (R_High - 1) loop
            if Index_stage = Legs (Travel_Index.all).Route'First
              or Legs (Travel_Index.all).
              Route (Index_stage).Stage_Number < Index_First_Segment then
               Index_First_Segment := Legs (Travel_Index.all).
                 Route (Index_stage).Stage_Number;
            end if;

         end loop;

         requeue Book_Ticket (Index_First_Segment);

      end Book_Leg;

      entry Book_Ticket (for N_Stage in Routefirst..Routelast)
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T)
      when Available_Tickets(N_Stage) > 0 is

         Next_Segment : Integer := N_Stage;
         R_Low : Integer;
         R_High : Integer;

         More_Segment : Boolean := False;
         Index_First_Segment_leg : Integer;
         book_more_leg : Boolean := False;
         Index_Train_with_lower_Number : Integer := 0;
         ix : Integer;

      begin

         ix := N_Stage;
         Available_Tickets (ix) := Available_Tickets (ix) - 1;

         R_Low := Legs (Travel_Index.all).Route'First;
         R_High := Legs (Travel_Index.all).Route'Last;

         --  selects next stage
         for Index_stage in R_Low .. (R_High - 1) loop

            if More_Segment and
              Legs (Travel_Index.all).
              Route (Index_stage).Stage_Number > N_Stage
              and Legs (Travel_Index.all).
              Route (Index_stage).Stage_Number < Next_Segment
            then
               Next_Segment :=
                 Legs (Travel_Index.all).Route(Index_stage).Stage_Number;
            end if;

            if not More_Segment and
              Legs (Travel_Index.all).
              Route (Index_stage).Stage_Number > N_Stage
            then
               More_Segment := True;
               Next_Segment := Legs
                 (Travel_Index.all).Route (Index_stage).Stage_Number;
            end if;

         end loop;

         if More_Segment then
            requeue Book_Ticket (Next_Segment);
         end if;

         --  if no more stages, next leg
         Index_Train_with_lower_Number := Travel_Index.all;

         for Legso_ix in Legs'First .. (Legs'Last - 1) loop

            if (Legs (Legso_ix).Train_Type = EC) then

               if book_more_leg
                 and Legs (Legso_ix).Train_Number >
                 Legs (Travel_Index.all).Train_Number
                 and Legs (Legso_ix).Train_Number
                 < Legs (Index_Train_with_lower_Number).Train_Number
               then
                  Index_Train_with_lower_Number := Legso_ix;
               end if;

               if not book_more_leg
                 and Legs (Legso_ix).Train_Number >
                 Legs (Travel_Index.all).Train_Number
               then
                  book_more_leg := True;
                  Index_Train_with_lower_Number := Legso_ix;
               end if;

            end if;

         end loop;

         if book_more_leg then
            Travel_Index.all := Index_Train_with_lower_Number;
            requeue Legs (Index_Train_with_lower_Number).Tickets.Book_Leg;
         end if;

         --  when no more Legs to book, Depart

         Index_First_Segment_leg := Legs(Legs'First).Route'First;
         Travel_Index.all := Legs'First;

         Passed := Clock - Start_Time;

         Distrailway.Central_Control.Upd_Passenger
           (Integer (To_Duration (Passed)),
            Passenger_Name.all,
            "is waiting for train " & Legs(Legs'First).Train_Name.all
            & " at platform "
            & Legs(Legs'First).
            Route(Index_First_Segment_leg).Departure_Platform.Name.all);

         requeue Legs (Legs'First).
           Route (Index_First_Segment_leg).
           Departure_Platform.Depart (Legs (Legs'First).Train_Number);

      end Book_Ticket;

      procedure Release_Ticket
        (Legs : Legs_Sequence_Ref_T;
         Travel_Index : Integer_Ref_T;
         Passenger_Name : String_Ref_T) is

         R_Low : Integer;
         R_High : Integer;
      begin
         R_Low := Legs (Travel_Index.all - 1).Route'First;
         R_High := Legs (Travel_Index.all - 1).Route'Last;
         for Index_Stage in R_Low .. (R_High - 1) loop
            Available_Tickets (Legs (Travel_Index.all - 1).
                                 Route (Index_Stage).Stage_Number) :=
              Available_Tickets (Legs (Travel_Index.all - 1).
                                   Route (Index_Stage).Stage_Number) + 1;
         end loop;
      end Release_Ticket;

   end Ticket_DB_T;

end Railway.Platform;
