with Ada.Text_IO;
with Time_IO;
with Ada.Real_Time;

with Railroad.Station;

package body Railroad.Train is

   -------------
   --  UTILITIES
   -------------
   --  service utilities to pretty-print the list of Travelers
   --  currently aboard the Train
   --  (the correct layout depends on Capacity)
   procedure Aboard;
   procedure Report (S : Address_T);

   procedure Aboard is
      use Ada.Text_IO;
      How_Many : Passengers_T := Passengers_T'First;
      Marker_Start : constant String := "   [";
      Marker_End   : constant String := " ]  ";

   begin
      Put (Marker_Start);
      for I in 1 .. Population loop
         if En_Route (I) then
            Put (Positive'Image (I));
            How_Many := How_Many + 1;
         end if;
      end loop;
      case How_Many is
         when 0 =>
            Put_Line (Marker_End & "        |");
         when 1 =>
            Put_Line (Marker_End & "      |");
         when 2 =>
            Put_Line (Marker_End & "    |");
         when 3 =>
            Put_Line (Marker_End & "  |");
         when 4 =>
            Put_Line (Marker_End & "|");
      when others =>
         --  never more than Capacity
         null;
      end case;
      How_Many := 0;

   end Aboard;

   procedure Report (S : Address_T) is
      use Ada.Text_IO;
      use Ada.Real_Time;
      Passed : Time_Span;

   begin
      Passed := Clock - Start_Time;
      Put (" ");
      Time_IO.Put (Interval (To_Duration (Passed)));
      Put (" " & String (Stops (S)));
      Aboard;

   end Report;

   --+---------------------------
   --  THE TRAIN PROCESS INSTANCE
   --+---------------------------
   task body The_Train is
      --  the Train is initially empty of Travelers
      Load : Passengers_T := 0;
      --  we use this variable to determine the projected
      --  time of arrival at the next station
      --  (we add a little lag to cause the train to call at the first station
      --  after the departing passengers)
      Next_Call : Ada.Real_Time.Time := Start_Time + Milliseconds (100);
      --  we assume that the travel time across Stations
      --+ are fixed and constant (are they really?)
      use Ada.Text_IO;
      use Railroad.Station;
      use Ada.Real_Time;

   begin
      Put_Line ("   TIME CALLING @ " &
                "ABOARD       | TRAVELER FROM    TO");
      Put_Line ("   ----------------------------+-------------------------");
      delay until Next_Call;
      loop
         --  the Train runs along the circular line
         for S in Address_T loop
            Report (S);
            --  when calling at a Station the Train first releases
            --  the Travelers onboard who have arrived at destination
            --  and then determines its remaining load
            Railway_Line (S).Calling (Load);
            --  subsequently the Train boards Travelers that depart
            --  from that Station as the residual capacity allows
            Railway_Line (S).Boarding;
            --  and finally the Train updates its count of onboard Travelers
            Railway_Line (S).Departing (Load);
            --  at this point the Train runs along to the next Station
            --  taking the journey time specified for that leg
            Next_Call := Next_Call + Travel_Times (S);
            delay until Next_Call;
         end loop;
      end loop;

   end The_Train;

end Railroad.Train;
